/******************************************************************************
* Copyright (c) 2018(-2021) STMicroelectronics.
* All rights reserved.
*
* This file is part of the TouchGFX 4.18.1 distribution.
*
* This software is licensed under terms that can be found in the LICENSE file in
* the root directory of this software component.
* If no LICENSE file comes with this software, it is provided AS-IS.
*
*******************************************************************************/

/**
 * @file touchgfx/widgets/canvas/PainterRGB565Bitmap.hpp
 *
 * Declares the touchgfx::PainterRGB565Bitmap class.
 */
#ifndef TOUCHGFX_PAINTERRGB565BITMAP_HPP
#define TOUCHGFX_PAINTERRGB565BITMAP_HPP

#include <touchgfx/hal/Types.hpp>
#include <touchgfx/Bitmap.hpp>
#include <touchgfx/widgets/canvas/AbstractPainterRGB565.hpp>

namespace touchgfx
{
/**
 * PainterRGB565Bitmap will take the color for a given point in the shape from a bitmap. Please
 * be aware, the the bitmap is used by the CanvasWidgetRenderer (not Shape), so any
 * rotation you might specify for a Canvas Widget (e.g. Shape) is not applied to the
 * bitmap as CWR is not aware of this rotation.
 *
 * @see AbstractPainter
 */
class PainterRGB565Bitmap : public AbstractPainterRGB565
{
public:
    /**
     * Initializes a new instance of the PainterRGB565Bitmap class.
     *
     * @param  bmp   (Optional) The bitmap, default is #BITMAP_INVALID.
     */
    PainterRGB565Bitmap(const Bitmap& bmp = Bitmap(BITMAP_INVALID))
        : AbstractPainterRGB565(),
          bitmapARGB8888Pointer(0), bitmapRGB565Pointer(0), bitmapAlphaPointer(0),
          bitmap(), bitmapRectToFrameBuffer(),
          xOffset(0), yOffset(0), isTiled(false)
    {
        setBitmap(bmp);
    }

    /**
     * Sets a bitmap to be used when drawing the CanvasWidget.
     *
     * @param  bmp The bitmap.
     */
    void setBitmap(const Bitmap& bmp);

    /**
     * Instruct the painter to tile the bitmap specified. The bitmap will be tiled both horizontally
     * and vertically.
     *
     * @param   tiled   True if tiled.
     *
     * @see setOffset
     */
    virtual void setTiled(bool tiled);

    /**
     * Sets an offset for the bitmap used. The x and y coordinates specifies how far the bitmap
     * should be moved to the right and down. This works for tiled bitmaps and non-tiled bitmaps.
     *
     * @param   x   The x coordinate.
     * @param   y   The y coordinate.
     *
     * @see setTiled
     */
    virtual void setOffset(int16_t x, int16_t y);

    virtual void render(uint8_t* ptr, int x, int xAdjust, int y, unsigned count, const uint8_t* covers);

protected:
    virtual bool renderInit();

    const uint32_t* bitmapARGB8888Pointer; ///< Pointer to the bitmap (ARGB8888)
    const uint16_t* bitmapRGB565Pointer;   ///< Pointer to the bitmap (RGB565)
    const uint8_t* bitmapAlphaPointer;     ///< Pointer to the bitmap alpha data for RGB565

    Bitmap bitmap;                ///< The bitmap to be used when painting
    Rect bitmapRectToFrameBuffer; ///< Bitmap rectangle translated to framebuffer coordinates

    int16_t xOffset; ///< The x offset of the bitmap
    int16_t yOffset; ///< The y offset of the bitmap
    bool isTiled;    ///< True if bitmap should be tiled, false if not
};

} // namespace touchgfx

#endif // TOUCHGFX_PAINTERRGB565BITMAP_HPP
