package club.bytecode.the.jda;

import club.bytecode.the.jda.api.ExceptionUI;
import club.bytecode.the.jda.api.JDANamespace;
import club.bytecode.the.jda.api.JDAPlugin;
import club.bytecode.the.jda.api.PluginLoader;
import club.bytecode.the.jda.decompilers.filter.DecompileFilters;
import club.bytecode.the.jda.decompilers.filter.DropLocalVariableTableFilter;
import club.bytecode.the.jda.decompilers.filter.IllegalAnnotationFilter;
import club.bytecode.the.jda.gui.MainViewerGUI;
import club.bytecode.the.jda.gui.fileviewer.BytecodeFoldParser;
import club.bytecode.the.jda.gui.fileviewer.BytecodeTokenizer;
import club.bytecode.the.jda.gui.fileviewer.JDAJavaTokenizer;
import club.bytecode.the.jda.gui.fileviewer.ViewerFile;
import club.bytecode.the.jda.gui.navigation.FileNavigationPane;
import club.bytecode.the.jda.settings.Settings;
import club.bytecode.the.jda.util.BytecodeUtils;
import club.bytecode.the.jda.util.GuiUtils;
import club.bytecode.the.jda.util.MiscUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.SystemUtils;
import org.fife.ui.rsyntaxtextarea.AbstractTokenMakerFactory;
import org.fife.ui.rsyntaxtextarea.TokenMakerFactory;
import org.fife.ui.rsyntaxtextarea.folding.CurlyFoldParser;
import org.fife.ui.rsyntaxtextarea.folding.FoldParserManager;
import org.objectweb.asm.tree.ClassNode;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Supplier;

public class JDA {
    public static final String version = "1.2.0";

    public static final String ISSUE_TRACKER_URL = "https://github.com/LLVM-but-worse/jda/issues";
    public static final String fs = System.getProperty("file.separator");
    public static final String nl = System.getProperty("line.separator");
    public static final File dataDir = new File(System.getProperty("user.home") + fs + ".jda");
    public static final File pluginsDir = new File(dataDir, "plugins");
    public static final File recentsFile = new File(dataDir, "recentfiles.jda");
    public static final File settingsFile = new File(dataDir, "settings.jda");
    private static final long start = System.currentTimeMillis();

    public static MainViewerGUI viewer = null;
    private static List<FileContainer> files = new ArrayList<>(); //all of JDA's loaded files/classes/etc
    private static int maxRecentFiles = 25;
    private static List<String> recentFiles = new ArrayList<>();
    public static String lastDirectory = "";
    public static List<Process> createdProcesses = new ArrayList<>();
    private static final AtomicInteger jobCount = new AtomicInteger(0);
    public static final JDANamespace namespace = JDADefaultNamespace.INSTANCE; 
    private static List<JDAPlugin> plugins = new ArrayList<>();
    
    public static Supplier<JDAPlugin> autoloadPlugin = null; // for testing purposes, a plugin to load on startup.

    /**
     * Main startup
     *
     * @param args files you want to open or CLI
     */
    public static void main(String[] args) {
        GuiUtils.setAntialiasingSettings();
        if (SystemUtils.IS_OS_LINUX) {
            GuiUtils.setWmClassName("JDA");
        }
        GuiUtils.setLookAndFeel();
        try {
            System.out.println("JDA v" + version);
            getJDADirectory();

            registerModules();
            loadPlugins();

            Settings.loadGUI();
            if (!recentsFile.exists() && !recentsFile.createNewFile())
                throw new RuntimeException("Could not create recent files file");
            recentFiles.addAll(FileUtils.readLines(recentsFile, "UTF-8"));

            viewer = new MainViewerGUI();
            JDA.onGUILoad();
            Boot.boot();
            JDA.boot(args);
        } catch (Exception e) {
            new ExceptionUI(e, "initializing");
        }
    }

    public static void loadPlugin(JDAPlugin plugin) {
        plugins.add(plugin);
        plugin.onLoad();
    }
    
    public static void unloadPlugin(JDAPlugin plugin) {
        plugin.onUnload();
        plugins.remove(plugin);
    }
    
    public static List<JDAPlugin> getLoadedPlugins() {
        return Collections.unmodifiableList(plugins);
    }

    private static void registerModules() {
        DecompileFilters.registerFilter(new IllegalAnnotationFilter());
        DecompileFilters.registerFilter(new DropLocalVariableTableFilter());
    }

    private static void loadPlugins() throws MalformedURLException {
        if (autoloadPlugin != null) {
            JDAPlugin plugin = autoloadPlugin.get();
            System.out.println("Loading statically-loaded plugin " + plugin.getName());
            loadPlugin(autoloadPlugin.get());
            System.out.println("Skipping other plugins.");
            return;
        }
        
        if (!pluginsDir.exists())
            if (!pluginsDir.mkdirs())
                throw new RuntimeException("Couldn't create plugins directory");
        else if (!pluginsDir.isDirectory())
            throw new RuntimeException("Plugins location is not a directory");

        for (File pluginFile : pluginsDir.listFiles()) {
            if (!pluginFile.getName().endsWith(".jar")) {
                System.out.println("Skipping non-jar " + pluginFile.getName());
                continue;
            }
            JDAPlugin pluginInstance = PluginLoader.tryLoadPlugin(pluginFile);
            if (pluginInstance != null)
                loadPlugin(pluginInstance);
        }
    }

    public static void onGUILoad() {
        AbstractTokenMakerFactory atmf = (AbstractTokenMakerFactory)TokenMakerFactory.getDefaultInstance();
        atmf.putMapping(BytecodeTokenizer.SYNTAX_STYLE_BYTECODE, BytecodeTokenizer.class.getName());
        atmf.putMapping(JDAJavaTokenizer.SYNTAX_STYLE_JDA_JAVA, JDAJavaTokenizer.class.getName());
        FoldParserManager.get().addFoldParserMapping(BytecodeTokenizer.SYNTAX_STYLE_BYTECODE, new BytecodeFoldParser());
        FoldParserManager.get().addFoldParserMapping(JDAJavaTokenizer.SYNTAX_STYLE_JDA_JAVA, new CurlyFoldParser());

        plugins.forEach(JDAPlugin::onGUILoad);
    }

    /**
     * The version checker thread
     */
    private static final Thread versionChecker = new Thread(() -> {});

    /**
     * Boot after all of the libraries have been loaded
     */
    public static void boot(String[] args) {
        Runtime.getRuntime().addShutdownHook(new Thread(JDA::onExit));

        resetRecentFilesMenu();

        if (Settings.DO_UPDATE_CHECK.getBool())
            versionChecker.start();

        viewer.setVisible(true);

        viewer.calledAfterLoad();

        System.out.println("Start up took " + ((System.currentTimeMillis() - start) / 1000.) + " seconds");

        if (args.length >= 1) {
            for (String s : args) {
                openFiles(new File[]{new File(s)}, true);
            }
        }
    }

    private static void onExit() {
        // unload all plugins
        plugins.forEach(JDAPlugin::onExit);
        while (!plugins.isEmpty())
            unloadPlugin(plugins.get(0));

        for (Process proc : createdProcesses)
            proc.destroy();
        try {
            FileUtils.writeLines(recentsFile, recentFiles);
        } catch (IOException e) {
            new ExceptionUI(e, "saving recent files");
        }
        if (!viewer.isMaximized)
            viewer.unmaximizedPos = viewer.getLocation();
        Settings.saveGUI();
    }
    
    /**
     * Waits for all busy-setting tasks to complete.
     */
    public static void waitForTasks() {
        while (jobCount.get() > 0) {
            GuiUtils.sleep(10L);
        }
    }

    /**
     * Sets the busy state, and toggles the spinner icon
     * Make sure to call me OUTSIDE of your worker thread for busy=true!
     * Then, you must call again busy=false once your worker thread finishes!
     * @param busy whether a background task is running
     */
    public static void setBusy(boolean busy) {
        if (busy)
            jobCount.incrementAndGet();
        else
            jobCount.decrementAndGet();
        assert (jobCount.get() >= 0);
        viewer.setIcon(jobCount.get() > 0);
    }

    public static byte[] getFileBytes(ViewerFile file) {
        if (file.container != null)
            return file.container.getFiles().get(file.name);
        else
            return null;
    }
    
    public static boolean hasFile(ViewerFile file) {
        if (file.container == null)
            return false;
        return file.container.getFiles().containsKey(file.name);
    }

    // try to get class bytes by exporting classnode, else fallback to getting the bytes from the actual file itself
    public static byte[] getClassBytes(FileContainer container, ClassNode cn) {
        byte[] result = BytecodeUtils.dumpClassToBytes(cn);
        if (result != null)
            return result;
        result = getClassFileBytes(container, cn.name);
        return result;
    }
    
    public static byte[] getClassFileBytes(FileContainer container, String className) {
        byte[] bytes = getFileBytes(new ViewerFile(container, container.findClassfile(className)));
        if (bytes == null)
            return null;
        return bytes;
    }

    public static final String HACK_PREFIX = "\0JDA-hack";

    public static File getClassFileProxy(ClassNode cn) {
        return new File('/' + HACK_PREFIX, cn.name + ".class");
    }

    public static String extractProxyClassName(String fileName) {
        return getClassFilePath(fileName.substring(fileName.indexOf(HACK_PREFIX) + HACK_PREFIX.length() + 1));
    }

    public static String getClassFilePath(String fileName) {
        return fileName.replace(File.separator, "/").substring(0, fileName.length() - ".class".length());
    }

    /**
     * Returns the last part of a fully qualified name, e.g. the class name.
     * @param fullyQualifiedName a fully qualified name like "java/lang/String"
     * @return just the name without the package name, e.g. "String"
     */
    public static String getClassName(String fullyQualifiedName) {
        return fullyQualifiedName.substring(fullyQualifiedName.lastIndexOf('/') + 1);
    }

    /**
     * Opens a file, optional if it should append to the recent files menu
     *
     * @param files       the file(s) you wish to open
     * @param recentFiles if it should append to the recent files menu
     */
    public static void openFiles(final File[] files, boolean recentFiles) {
        openFiles(files, recentFiles, null);
    }

    public static void openFiles(final File[] files, boolean recentFiles, FileNavigationPane.FileNode parent) {
        JDA.setBusy(true);
        
        if (recentFiles)
            for (File f : files)
                if (f.exists())
                    JDA.addRecentFile(f);

        FileNavigationPane fnp = MainViewerGUI.getComponent(FileNavigationPane.class);

        (new Thread(() -> {
            try {
                for (final File fileToOpen : files) {
                    final String fn = fileToOpen.getName();
                    if (!fileToOpen.exists()) {
                        showMessage("The file " + fileToOpen.getAbsolutePath() + " could not be found.");
                    } else if (fileToOpen.isDirectory()) {
                        FileNavigationPane.FileNode newNode = fnp.addTreeElement(new FileContainer(fileToOpen, new HashMap<>()), parent);
                        openFiles(fileToOpen.listFiles(), false, newNode);
                    } else if (fn.endsWith(".jar") || fn.endsWith(".zip")) {
                        try {
                            FileContainer newContainer = JarUtils.load(fileToOpen);
                            openFile(newContainer);
                            fnp.addTreeElement(newContainer, parent);
                        } catch (final Exception e) {
                            new ExceptionUI(e, "loading jar");
                        }
                    } else {
                        HashMap<String, byte[]> files1 = new HashMap<>();
                        byte[] bytes = JarUtils.getBytes(new FileInputStream(fileToOpen));
                        files1.put(fileToOpen.getName(), bytes);
                        FileContainer container = new FileContainer(fileToOpen, files1);
                        openFile(container);
                        fnp.addTreeElement(container, parent);
                    }
                }
            } catch (Exception e) {
                new ExceptionUI(e, "loading file");
            } finally {
                JDA.setBusy(false);
            }
        })).start();
    }
    
    public static void openFile(FileContainer fc) {
        JDA.files.add(fc);
        plugins.forEach((plugin -> plugin.onOpenFile(fc)));
    }
    
    public static void closeFile(FileContainer fc) {
        JDA.files.remove(fc);
        plugins.forEach(plugin -> plugin.onCloseFile(fc));
    }
    
    public static void clearFiles() {
        while (!JDA.files.isEmpty()) {
            JDA.closeFile(JDA.files.get(0));
        }
    }
    
    public static List<FileContainer> getOpenFiles() {
        return Collections.unmodifiableList(files);
    }

    /**
     *
     * @param filename the filename to search all open FileContainers for
     * @return the FileContainer which holds the specified file or null if not found
     */
    public static FileContainer findContainerForFile(String filename) {
        for (FileContainer container : files) {
            if (container.getFiles().containsKey(filename)) {
                return container;
            }
        }
        return null;
    }

    /**
     * Send a message to alert the user
     *
     * @param message the message you need to send
     */
    public static void showMessage(String message) {
        JOptionPane.showMessageDialog(viewer, message);
    }

    /**
     * Ask a yes/no dialog
     * @param question the body text of the dialog box to be created
     * @param title the window title of the dialog box to be created
     * @return true or false for yes or no
     */
    public static boolean askYesNoDialog(String question, String title) {
        JOptionPane pane = new JOptionPane(question);
        Object[] options = new String[]{"Yes", "No"};
        pane.setOptions(options);
        JDialog dialog = pane.createDialog(viewer, "JDA - " + title);
        dialog.setVisible(true);
        Object obj = pane.getValue();
        for (int k = 0; k < options.length; k++)
            if (options[k].equals(obj) && k == 0)
                return true;
        return false;
    }

    /**
     * Resets the workspace with optional user input required
     *
     * @param ask if should require user input or not
     */
    public static void resetWorkSpace(boolean ask) {
        if (ask) {
            if (!askYesNoDialog("Are you sure you want to reset the workspace?\n\rIt will also reset your file navigator and search.", "Reset Workspace"))
                return;
        }

        closeResources(false);
        viewer.resetWindows();
    }

    public static void closeResources(boolean ask) {
        if (ask) {
            if (!askYesNoDialog("Are you sure you want to close all resources?", "Close Resources"))
                return;
        }

        JDA.setBusy(true);
        clearFiles();
        viewer.closeResources();
        JDA.setBusy(false);
    }

    private static ArrayList<String> killList = new ArrayList<>();

    /**
     * Add the recent file
     *
     * @param f the recent file
     */
    public static void addRecentFile(File f) {
        for (int i = 0; i < recentFiles.size(); i++) { // remove dead strings
            String s = recentFiles.get(i);
            if (s.isEmpty() || i > maxRecentFiles)
                killList.add(s);
        }
        if (!killList.isEmpty()) {
            for (String s : killList)
                recentFiles.remove(s);
            killList.clear();
        }

        if (recentFiles.contains(f.getAbsolutePath())) // already added on the list
            recentFiles.remove(f.getAbsolutePath());
        if (recentFiles.size() >= maxRecentFiles)
            recentFiles.remove(maxRecentFiles - 1); // zero indexing

        recentFiles.add(0, f.getAbsolutePath());
        resetRecentFilesMenu();
    }

    /**
     * resets the recent files menu
     */
    public static void resetRecentFilesMenu() {
        viewer.mnRecentFiles.removeAll();
        for (String s : recentFiles)
            if (!s.isEmpty()) {
                JMenuItem m = new JMenuItem(s);
                m.addActionListener(e -> {
                    JMenuItem m1 = (JMenuItem) e.getSource();
                    openFiles(new File[]{new File(m1.getText())}, true);
                });
                viewer.mnRecentFiles.add(m);
            }
    }

    /**
     * Returns the JDA directory
     *
     * @return the static JDA directory
     */
    public static String getJDADirectory() {
        while (!dataDir.exists())
            dataDir.mkdirs();

        if (!dataDir.isHidden() && SystemUtils.IS_OS_WINDOWS)
            hideFileWindows(dataDir);

        return dataDir.getAbsolutePath();
    }

    /**
     * Runs the windows command to hide files
     *
     * @param f file you want hidden
     */
    private static void hideFileWindows(File f) {
        try {
            // Hide file by running attrib system command (on Windows)
            Runtime.getRuntime().exec("attrib +H " + f.getAbsolutePath());
        } catch (Exception e) {
            new ExceptionUI(e, "hiding file");
        }
    }

    private static boolean isCtrlDown(KeyEvent e) {
        return ((e.getModifiers() & KeyEvent.CTRL_MASK) != 0);
    }

    private static boolean isShiftDown(KeyEvent e) {
        return ((e.getModifiers() & KeyEvent.SHIFT_MASK) != 0);
    }

    /**
     * Checks the hotkeys
     */
    public static void checkHotKey(KeyEvent e) {
        if ((e.getKeyCode() == KeyEvent.VK_O) && isCtrlDown(e)) {
            openFileChooser();
        } else if ((e.getKeyCode() == KeyEvent.VK_N) && isCtrlDown(e)) {
            JDA.resetWorkSpace(true);
        } else if ((e.getKeyCode() == KeyEvent.VK_R) && isCtrlDown(e) && isShiftDown(e)) {
            viewer.reloadResources();
        } else if ((e.getKeyCode() == KeyEvent.VK_R) && isCtrlDown(e)) {
            viewer.refreshView();
        } else if ((e.getKeyCode() == KeyEvent.VK_W) && isCtrlDown(e) && isShiftDown(e)) {
            JDA.closeResources(true);
        } else if ((e.getKeyCode() == KeyEvent.VK_W) && isCtrlDown(e)) {
            if (viewer.fileViewerPane.getCurrentViewer() != null)
                viewer.fileViewerPane.tabs.remove(viewer.fileViewerPane.getCurrentViewer());
        } else if ((e.getKeyCode() == KeyEvent.VK_F) && isCtrlDown(e) && isShiftDown(e)) {
            viewer.doSearchDialog();
        }
    }

    public static void openFileChooser() {
        JFileChooser fc = new JFileChooser();
        try {
            File f = new File(JDA.lastDirectory);
            if (f.exists())
                fc.setSelectedFile(f);
        } catch (Exception e2) {
            e2.printStackTrace();
        }
        fc.setFileFilter(new JavaFileFilter());
        fc.setFileHidingEnabled(false);
        fc.setAcceptAllFileFilterUsed(false);
        int returnVal = fc.showOpenDialog(JDA.viewer);

        if (returnVal == JFileChooser.APPROVE_OPTION) {
            JDA.lastDirectory = fc.getSelectedFile().getAbsolutePath();
            try {
                JDA.setBusy(true);
                JDA.openFiles(new File[]{fc.getSelectedFile()}, true);
                JDA.setBusy(false);
            } catch (Exception e1) {
                new ExceptionUI(e1, "choosing file");
            }
        }
    }

    public static class JavaFileFilter extends FileFilter {
        @Override
        public boolean accept(File f) {
            if (f.isDirectory())
                return true;

            String extension = MiscUtils.extension(f.getAbsolutePath());
            if (extension.equals("jar") || extension.equals("zip") || extension.equals("class"))
                return true;

            return false;
        }

        @Override
        public String getDescription() {
            return "Class Files or Zip/Jar Archives";
        }
    }

    public static Function<String, List<ViewerFile>> searchCallback = JDA::search;

    public static List<ViewerFile> search(String needle) {
        List<ViewerFile> matches = new ArrayList<>();
        for (FileContainer fc : JDA.getOpenFiles()) {
            for (Map.Entry<String, byte[]> e : fc.getFiles().entrySet()) {
                if (e.getKey().endsWith(".class")) {
                    try {
                        // ClassNode cn = fc.loadClassFile(e.getKey());
                        String fileBytes = new String(e.getValue());
                        if (fileBytes.contains(needle)) {
                            matches.add(new ViewerFile(fc, e.getKey()));
                        }
                    } catch (Exception ex) {
                        ex.printStackTrace();
                    }
                }
            }
        }
        return matches;
    }
}
