## National Parks Directory: An Application Programming Interface

# Documentation

#### By Krista Rutz. Last updated on April 3, 2020.

Welcome to the National Park Directory API! Use this API to access information about National Parks in the United States and update park closure information in light of the federal social distancing recommendations, which have affected park openings in many states.

## Request Overview

Use the links in the table under the "Request" header to view more detailed information about how to format requests and specify parameters to customize results.

| Request                                       | Request Type | Route Endpoints                                            |
| :-------------------------------------------- | :----------- | :--------------------------------------------------------- |
| [National park search](#national-park-search) | GET          | `http://localhost:5000/api/nationalparks`                  |
| [Retrieve one park](#Retrieve-one-park)       | GET          | `http://localhost:5000/api/nationalparks/{NationalParkId}` |
| [Add a park](#add-a-park)                     | POST         | `http://localhost:5000/api/nationalparks`                  |
| [Edit a park](#edit-a-park)                   | PUT          | `http://localhost:5000/api/nationalparks/{NationalParkId}` |
| [Remove a park](#remove-a-park)               | DELETE       | `http://localhost:5000/api/nationalparks/{NationalParkId}` |

## About the data

Data is available in JSON format.

National park entries contain fields for the park name, the park code written in alphabet characters, the two-digit state code, the region of administration within the National Parks Service, the URL of the homepage for the park's website, and a boolean indicating closure status. The application currently contains information on 18 national parks.

#### Example Park Result

```
    {
        "nationalParkId": 17,
        "name": "Denali National Park",
        "alphaCode": "DENA",
        "stateCode": "AK",
        "region": "Alaska",
        "url": "https://www.nps.gov/dena/index.htm",
        "isOpen": false
    }
```

#### Parsing the Data

<table>
  <th>key</th>
  <th>type</th>
  <th>description</th>
  <tr>
    <td>nationalParkId</td>
    <td>int</td>
    <td>This key is autogenerated as entries are added</td>
  </tr>
  <tr>
    <td>name</td>
    <td>string</td>
    <td>The full name of the park, including the short name and the park designation</td>
  </tr>
  <tr>
    <td>alphaCode</td>
    <td>string</td>
    <td>The unique 4 character code representing the park for administrative purposes, such as url generation</td>
  </tr>
  <tr>
    <td>stateCode</td>
    <td>string</td>
    <td>The 2 character code of the US state or territory where the park is located</td>
  </tr>
  <tr>
    <td>region</td>
    <td>string</td>
    <td>One of seven administrative regions in the US NPS under which the park is located</td>
  </tr>
  <tr>
    <td>url</td>
    <td>string</td>
    <td>The url of the homepage for the park's website</td>
  </tr>
  <tr>
    <td>isOpen</td>
    <td>bool</td>
    <td>A true or false value indicating if the park is currently open during normal hours.</td>
  </tr>
</table>

<br>
<br>

# GET requests

Search for national parks, and use parameters to customize your results. Valid parameters include `name`, `alphacode`, `statecode`, `region`, `isopen`, `limit`, and `start`.

## National park search

Use this endpoint to find a list of all results, or all matching a query. Make a GET request for all parks matching a query, adjusting for pagination preferences, by adding parameters in the format of `?` + `{parameter-name}={search-term}`, separated by `&`, to the standard route format. Each query parameter is optional, and any combination of parameters can be used.

### Retrieve All

A GET request for all parks follows the following route format:

```
http://localhost:5000/api/nationalparks/
```

When seeking a list of all available park data, simply submit a GET request with **no** parameters. _Note: The default result is set to 20 parks. See the [pagination](#pagination) section if you wish to get more than the default number of results._

### Filtering Results

A GET request for all parks matching a query requires parameters in the format of `?` + `{parameter-name}={search-term}`, separated by `&`, to the standard route format. Each query parameter is optional, and any combination of parameters can be used. Null entries for string parameters will not result in errors. as shown here:.

```
http://localhost:5000/api/nationalparks/?name=&alphaCode=&stateCode=&region=&isopen=
```

The five string query parameters are `name`, `alphacode`, `statecode`, `region`, and `isopen`. See the [parsing the data](#parsing-the-data) section for more information on these fields.

- Format spaces using standard URL encoding, with `+` or `%20`.
- Queries are not case sensitive.
- Queries are not ordered, and can appear in any order.
- Special type `isOpen` can be queried as a `true` or `false` string or as `1` (for true) and `0` (for false).

<details>
  <summary>Example GET all with query parameters</summary>

```
http://localhost:5000/api/nationalparks/?name=&alphaCode=&stateCode=CA&region=
```

or

```
http://localhost:5000/api/nationalparks/?stateCode=CA
```

Body:

```
[
  {
    "nationalParkId": 12,
    "name": "Channel Islands National Park",
    "alphaCode": "CHIS",
    "stateCode": "CA",
    "region": "Pacific-West",
    "url": "https://www.nps.gov/chis/index.htm",
    "isOpen": true
  },
  {
    "nationalParkId": 16,
    "name": "Death Valley National Park",
    "alphaCode": "DEVA",
    "stateCode": "CA",
    "region": "Pacific-West",
    "url": "https://www.nps.gov/deva/index.htm",
    "isOpen": true
  }
]
```

</details>

### Pagination

The National Park Directory API returns a default of 20 results at a time, and a maximum of 1000. Use the `limit` and `start` query parameters to change the results given. The `limit` specifies how many results should be returned, and the `start` parameter indicates at which element in the set the limit should start counting.

Make a GET request for all parks matching a query, adjusting for pagination preferences, by adding parameters in the format of `?` + `{parameter-name}={search-term}`, separated by `&`, to the standard route format. Each query parameter is optional, and any combination of parameters can be used.

Null int entries WILL result in errors. To use defaults, don't include `limit` and `start` or set them equal to zero.

<details>
  <summary>Examples using default values</summary>

In this example, limit is left blank and therefore set to default (which is 20), and only start is changed.

```

http://localhost:5000/api/nationalparks/?start=3

```

Setting limit to 0 accomplishes the same thing. In this example, limit is set to default 20, and only start is changed.

```

http://localhost:5000/api/nationalparks/?limit=0&start=3

```

</details>
<details>
  <summary>Example using custom limit and start</summary>

Setting limit to 12 and start to 1 will return entry 1-12 in the parks database. Setting limit to 12 and start to 13 (or 1+limit) will return entry 13-24. This is a convenient way to paginate results with a desired limit.

```

http://localhost:5000/api/nationalparks/?limit=12&start=13

```

</details>
<details>
  <summary>Example combining pagination with queries</summary>

Each query parameter is optional, and any combination of parameters can be used, including both string queries and pagination values.

```

http://localhost:5000/api/nationalparks/?region=intermountain&limit=10

```

This will return the first 10 parks in the Intermountain region.

</details>

## Retrieve one park

A GET request for an individual park follows the following route format:

```

http://localhost:5000/api/nationalparks/{nationalParkId}

```

Use this type of request to pull a single park based on the park's `nationalParkId`. This GET request cannot be used with query parameters, as it always returns a single park (or nothing, if no parks match the given ID).

<details>
  <summary>Example GET by ID</summary>

```
http://localhost:5000/api/nationalparks/2
```

Body:

```
{
    "nationalParkId": 2,
    "name": "National Park of American Samoa",
    "alphaCode": "NPSA",
    "stateCode": "AS",
    "region": "Pacific-West",
    "url": "https://www.nps.gov/npsa/index.htm",
    "isOpen": true
}
```

</details>

<br>
<br>

# All other requests

Be very careful when adding or changing information in the database. The following request types modify the underlying data, and cannot be reversed. Requests returning a 200 OK status have successfully posted.

#### Request Format

Format POST and PUT request content as JSON objects in the request body. Include the ID of the entry for PUT and DELETE requests.

**Error Codes:** The API will throw a 400 Bad Request error if the `alphaCode` exceeds 4 characters or if the `stateCode` exceeds 2 characters.

## Add a park

A PUT request follows the following route format:

```

http://localhost:5000/api/nationalparks/

```

Specify `name`, `alphacode`, `statecode`, `region`, `url`, and `isopen`. Do not specify a `nationalParkId`, as this will be automatically generated based on the historic number of entries in the database.

_Note: The `isOpen` value may be assigned as a true/false or 1/0 value._

<details>
  <summary>Example POST</summary>

```
http://localhost:5000/api/nationalparks/
```

Body:

```
{
    "name": "National Park of American Samoa",
    "alphacode": "NPSA",
    "statecode": "AS",
    "region": "Pacific-West",
    "url": "https://www.nps.gov/npsa/index.htm",
    "isopen": true
}
```

</details>

## Edit a park

A PUT request follows the following route format:

```

http://localhost:5000/api/nationalparks/{nationalParkId}

```

Currently, this API supports PUT requests but not PATCH requests. Therefore, the body must be formatted the same way as a POST request to add a new park. All fields not included, even if they are not the field to be modified, will be overwritten as `null`.

<details>
  <summary>Example PUT</summary>

```

http://localhost:5000/api/nationalparks/6

```

Body:

```

{
"name": "Biscayne National Park",
"alphaCode": "BISC",
"stateCode": "FL",
"region": "Southeast",
"url": "https://www.nps.gov/bisc/index.htm",
"isOpen": true
}

```

</details>

## Remove a park

A DELETE request follows the following route format:

```

http://localhost:5000/api/nationalparks/{nationalParkId}

```

Submitting a DELETE request will delete the park with the matching ID.

<details>
  <summary>Example DELETE</summary>

```

http://localhost:5000/api/nationalparks/8

```

</details>
