<?php

/*
 * Regis – Static analysis as a service
 * Copyright (C) 2016-2017 Kévin Gomez <contact@kevingomez.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

declare(strict_types=1);

namespace Tests\Regis\AnalysisContext\Domain\Model\Git\Diff;

use PHPUnit\Framework\TestCase;
use Regis\AnalysisContext\Domain\Git\DiffParser;
use Regis\AnalysisContext\Domain\Model\Git\Diff\File;

class FileTest extends TestCase
{
    public function testDeletionsAreDetected()
    {
        $diffFile = new File('old-name.php', null, 'old index', 'new index', false, []);

        $this->assertTrue($diffFile->isDeletion());
        $this->assertFalse($diffFile->isRename());
        $this->assertFalse($diffFile->isModification());
        $this->assertFalse($diffFile->isCreation());
        $this->assertSame('old-name.php', $diffFile->getOldName());
        $this->assertNull($diffFile->getNewName());
        $this->assertFalse($diffFile->isBinary());
    }

    public function testRenamesAreDetected()
    {
        $diffFile = new File('old-name.php', 'new-name.php', 'old index', 'new index', false, []);

        $this->assertFalse($diffFile->isDeletion());
        $this->assertTrue($diffFile->isRename());
        $this->assertTrue($diffFile->isModification());
        $this->assertFalse($diffFile->isCreation());
    }

    public function testACreationIsNotARename()
    {
        $diffFile = new File(null, 'new-name.php', 'old index', 'new index', false, []);

        $this->assertFalse($diffFile->isDeletion());
        $this->assertFalse($diffFile->isRename());
        $this->assertFalse($diffFile->isModification());
        $this->assertTrue($diffFile->isCreation());
    }

    /**
     * @dataProvider phpFileNameProvider
     */
    public function testIsPhpWithPhpFiles(string $newFileName)
    {
        $diffFile = new File(null, $newFileName, 'old index', 'new index', false, []);

        $this->assertTrue($diffFile->isPhp());
    }

    /**
     * @dataProvider notPhpFileNameProvider
     */
    public function testIsPhpWithNonPhpFiles(string $newFileName)
    {
        $diffFile = new File(null, $newFileName, 'old index', 'new index', false, []);

        $this->assertFalse($diffFile->isPhp());
    }

    public function phpFileNameProvider()
    {
        return [
            ['foo.php'],
            ['foo-bar.php'],
            ['foo/bar.php'],
            ['foo.phps'],
        ];
    }

    public function notPhpFileNameProvider()
    {
        return [
            ['foo'],
            ['foo-bar'],
            ['bar.txt'],
            ['foo/bar.txt'],
            ['foo/bar.inc'],
        ];
    }

    /**
     * @expectedException \Regis\AnalysisContext\Domain\Model\Exception\LineNotInDiff
     */
    public function testFindPositionForLineFailsIfTheLineIsNotInTheDiff()
    {
        $diffFile = new File(null, 'new-name.php', 'old index', 'new index', false, []);

        $diffFile->findPositionForLine(42);
    }

    /**
     * @dataProvider positionProvider
     */
    public function testFindPositionForLine(string $diffContent, int $line, int $expectedPosition)
    {
        $files = (new DiffParser())->parse($diffContent);

        $this->assertSame($expectedPosition, $files[0]->findPositionForLine($line));
    }

    public function positionProvider(): array
    {
        $diff1 = 'diff --git a/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php b/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php
index eed2c89..d62fdc2 100644
--- a/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php
+++ b/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php
@@ -32,9 +32,14 @@ public function execute(AMQPMessage $msg)
 
             $this->dispatch(Event::INSPECTION_STARTED, new Event\InspectionStarted($pullRequest));
 
-            $reportSummary = $this->inspector->inspect($pullRequest);
+            try {
+                $reportSummary = $this->inspector->inspect($pullRequest);
+                $this->dispatch(Event::INSPECTION_FINISHED, new Event\InspectionFinished($pullRequest, $reportSummary));
+            } catch (\Exception $e) {
+                $this->dispatch(Event::INSPECTION_FAILED, new Event\InspectionFailed($pullRequest, $e));
+                throw $e;
+            }
 
-            $this->dispatch(Event::INSPECTION_FINISHED, new Event\InspectionFinished($pullRequest, $reportSummary));
         }
     }
';

        $diff2 = 'diff --git a/src/Regis/Application/Event.php b/src/Regis/Application/Event.php
index 1c4206f..159c0a1 100644
--- a/src/Regis/Application/Event.php
+++ b/src/Regis/Application/Event.php
@@ -12,6 +12,7 @@
 
     const INSPECTION_STARTED = \'inspection_started\';
     const INSPECTION_FINISHED = \'inspection_finished\';
+    const INSPECTION_FAILED = \'inspection_failed\';
 
     function getEventName(): string;
 }
\ No newline at end of file
';

        $diff3 = 'diff --git a/src/Regis/Bundle/WebhooksBundle/EventListener/PullRequestInspectionStatusListener.php b/src/Regis/Bundle/WebhooksBundle/EventListener/PullRequestInspectionStatusListener.php
index 88ab614..48ca702 100644
--- a/src/Regis/Bundle/WebhooksBundle/EventListener/PullRequestInspectionStatusListener.php
+++ b/src/Regis/Bundle/WebhooksBundle/EventListener/PullRequestInspectionStatusListener.php
@@ -27,8 +27,10 @@ public static function getSubscribedEvents()
         return [
             Event::PULL_REQUEST_OPENED => \'onPullRequestUpdated\',
             Event::PULL_REQUEST_SYNCED => \'onPullRequestUpdated\',
+
             Event::INSPECTION_STARTED => \'onInspectionStated\',
             Event::INSPECTION_FINISHED => \'onInspectionFinished\',
+            Event::INSPECTION_FAILED => \'onInspectionFailed\',
         ];
     }
 
@@ -63,6 +65,14 @@ public function onInspectionFinished(DomainEventWrapper $event)
         $this->setIntegrationStatus($domainEvent->getPullRequest(), $status, $message);
     }
 
+    public function onInspectionFailed(DomainEventWrapper $event)
+    {
+        /** @var Event\InspectionFailed $domainEvent */
+        $domainEvent = $event->getDomainEvent();
+
+        $this->setIntegrationStatus($domainEvent->getPullRequest(), Client::INTEGRATION_FAILURE, \'Inspection failed.\');
+    }
+
     private function setIntegrationStatus(PullRequest $pullRequest, string $status, string $description)
     {
         $this->github->setIntegrationStatus($pullRequest, $status, $description, self::STATUS_CONTEXT);
';

        $diff4 = 'diff --git a/src/Regis/Application/Model/Git/Diff/File.php b/src/Regis/Application/Model/Git/Diff/File.php
index 76fe5e2..e2499d5 100644
--- a/src/Regis/Application/Model/Git/Diff/File.php
+++ b/src/Regis/Application/Model/Git/Diff/File.php
@@ -4,6 +4,7 @@
 
 namespace Regis\Application\Model\Git\Diff;
 
+use Regis\Application\Model\Exception\LineNotInDiff;
 use Regis\Application\Model\Git\Blob;
 
 class File
@@ -66,4 +67,26 @@ public function getChanges(): array
     {
         return $this->changes;
     }
+
+    public function findPositionForLine(int $line): int
+    {
+        $changes = $this->getChanges();
+
+        $previousChangeCount = 0;
+        /** @var Change $change */
+        foreach ($changes as $change) {
+            $rangeStart = $change->getRangeNewStart() - 1;
+
+            /** @var Line $diffLine */
+            foreach ($change->getAddedLines() as $diffLine) {
+                if ($rangeStart + $diffLine->getPosition() === $line) {
+                    return $previousChangeCount + $diffLine->getPosition();
+                }
+            }
+
+            $previousChangeCount = $change->getRangeNewCount() + 1;
+        }
+
+        throw LineNotInDiff::line($line);
+    }
 }
\ No newline at end of file
';

        $diff5 = 'diff --git a/src/Regis/Application/Reporter/DuplicationGuard.php b/src/Regis/Application/Reporter/DuplicationGuard.php
index a5cf1fb..64c6816 100644
--- a/src/Regis/Application/Reporter/DuplicationGuard.php
+++ b/src/Regis/Application/Reporter/DuplicationGuard.php
@@ -21,12 +21,13 @@ public function __construct(Reporter $originalReporter, ViolationsCache $violati
 
     public function report(Model\Violation $violation, Model\Github\PullRequest $pullRequest)
     {
-        if ($this->violationsCache->has($violation, $pullRequest)) {
+        if($this->violationsCache->has($violation, $pullRequest)) {
             return;
         }
 
         $this->originalReporter->report($violation, $pullRequest);
 
         $this->violationsCache->save($violation, $pullRequest);
+
     }
-}
\ No newline at end of file
+}
';

        $diff6 = 'diff --git a/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php b/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php
index d62fdc2..cfad9e1 100644
--- a/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php
+++ b/src/Regis/Bundle/WebhooksBundle/Worker/WebhookEvent.php
@@ -27,7 +27,7 @@ public function execute(AMQPMessage $msg)
     {
         $event = unserialize($msg->body);
 
-        if ($event instanceof Event\PullRequestOpened || $event instanceof Event\PullRequestSynced) {
+        if($event instanceof Event\PullRequestOpened || $event instanceof Event\PullRequestSynced) {
             $pullRequest = $event->getPullRequest();
 
             $this->dispatch(Event::INSPECTION_STARTED, new Event\InspectionStarted($pullRequest));
@@ -41,10 +41,14 @@ public function execute(AMQPMessage $msg)
             }
 
         }
+
+        return;
+
+        $lala = 42;
     }
 
     private function dispatch(string $eventName, Event $event)
     {
         $this->dispatcher->dispatch($eventName, new DomainEventWrapper($event));
     }
-}
\ No newline at end of file
+}
';

        return [
            [$diff1, 42, 12],
            [$diff2, 15, 4],
            [$diff3, 30, 4],
            [$diff3, 33, 7],
            [$diff4, 7, 4],
            [$diff4, 90, 32],
            [$diff5, 24, 5],
            [$diff6, 30, 5],
            [$diff6, 47, 16],
        ];
    }
}
