<?php

/*
 * Regis – Static analysis as a service
 * Copyright (C) 2016-2017 Kévin Gomez <contact@kevingomez.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

declare(strict_types=1);

namespace Regis\GithubContext\Domain\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Regis\GithubContext\Domain\Model;

class Repository
{
    public const TYPE_GITHUB = 'github';

    private $id;
    private $identifier;
    private $type = self::TYPE_GITHUB;
    private $sharedSecret;
    /** @var bool */
    private $isInspectionEnabled;
    /** @var bool */
    private $isFlightModeEnabled;
    /** @var ArrayCollection */
    private $inspections;

    /** @var UserAccount */
    private $owner;

    public function newSharedSecret(string $sharedSecret): void
    {
        $this->sharedSecret = $sharedSecret;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function getOwnerUsername(): string
    {
        return $this->toIdentifier()->getOwner();
    }

    public function toIdentifier(): Model\RepositoryIdentifier
    {
        return Model\RepositoryIdentifier::fromFullName($this->identifier);
    }

    public function getIdentifier(): string
    {
        return $this->identifier;
    }

    public function getOwner(): GithubDetails
    {
        return $this->owner->getDetails();
    }

    public function getName(): string
    {
        return $this->toIdentifier()->getName();
    }

    public function getSharedSecret(): string
    {
        return $this->sharedSecret;
    }

    public function getInspections(): \Traversable
    {
        return $this->inspections;
    }

    public function isInspectionEnabled(): bool
    {
        return $this->isInspectionEnabled;
    }

    public function isFlightModeEnabled(): bool
    {
        return $this->isFlightModeEnabled;
    }

    // TODO should not be here
    public function disableInspection(): void
    {
        $this->isInspectionEnabled = false;
    }

    public function enableInspection(): void
    {
        $this->isInspectionEnabled = true;
    }
}
