var builder = WebApplication.CreateBuilder(args);
builder.Configuration.AddEnvFilesToConfiguration();
builder.Services.AddHttpContextAccessor();

builder.Host.UseSerilog(SerilogExtensions.LoggerConfiguration);

builder.Services.AddCors();
builder.Services.AddControllers().RegisterOData();

builder.Services.AddEndpointsApiExplorer();
builder.Services.AddSwaggerGen(action =>
{
    action.SwaggerDoc("v1", new OpenApiInfo { Title = "Projects API", Version = "v1" });
    action.AddSecurityDefinition(
        "bearerAuth",
        new OpenApiSecurityScheme
        {
            Type = SecuritySchemeType.Http,
            Scheme = "bearer",
            BearerFormat = "JWT",
            Description = "JWT Authorization header using the Bearer scheme.",
        });
    action.AddSecurityRequirement(new OpenApiSecurityRequirement
    {
        {
            new OpenApiSecurityScheme
            {
                Reference = new OpenApiReference { Type = ReferenceType.SecurityScheme, Id = "bearerAuth" },
            },
            Array.Empty<string>()
        },
    });
});

builder.Services.AddAuthentication("Bearer")
    .AddJwtBearer(options =>
    {
        options.Authority = Environment.GetEnvironmentVariable("IDENTITY_SERVICE_URL");
        options.Audience = "ProjectsAPI";
        options.RequireHttpsMetadata = false;

        if (builder.Environment.IsDockerComposeEnvironment())
        {
            options.TokenValidationParameters.ValidateIssuer = false;
            options.BackchannelHttpHandler = new HttpClientHandler
            {
                ServerCertificateCustomValidationCallback =
                    HttpClientHandler.DangerousAcceptAnyServerCertificateValidator,
            };
        }
    });

builder.Services.AddAuthorization(options =>
    options.AddPolicy("ApiScope", policy =>
    {
        policy.RequireAuthenticatedUser();
        policy.RequireClaim("scope", "GlobalHub.ProjectsAPI");
    }));

builder.Services.AddScoped<IAuthorizationService<Project>, ProjectAuthorizationService>();
builder.Services.AddScoped<IAuthorizationService<Tag>, TagAuthorizationService>();
builder.Services.AddScoped<IAuthorizationService<ProjectItem>, ProjectItemAuthorizationService>();
builder.Services.AddScoped<IProjectItemNotificationService, ProjectItemNotificationService>();

builder.Services.AddScoped<IFullTextIndexService<Project>, ProjectFullTextIndexService>();
builder.Services.AddScoped<IFullTextIndexService<ProjectItem>, ProjectItemFullTextIndexService>();
builder.Services.AddScoped<IProjectItemFullTextSearchIndexService, ProjectItemFullTextIndexService>();

builder.Services.AddMediatR(cfg =>
{
    cfg.RegisterServicesFromAssemblyContaining(typeof(CreateProjectRequest));
    cfg.AddOpenBehavior(typeof(ValidationBehavior<,>));
    cfg.AddOpenBehavior(typeof(DbTransactionBehavior<,>));
});

builder.Services.AddValidatorsFromAssemblyContaining(typeof(BaseTagValidator<>));

builder.Services.AddAutoMapper(config =>
{
    config.AddProfile<MappingProfile>();
});

builder.Services.RegisterInfrastructureServices();
builder.Services.RegisterRequestHandlers();

var connectionString = Environment.GetEnvironmentVariable(ConfigConstants.ProjectsDbConnectionStringEnvKey);
ArgumentException.ThrowIfNullOrEmpty(connectionString);

builder.Services.AddDbContext<ApplicationDbContext>(optionsBuilder =>
{
    optionsBuilder.UseSqlServer(connectionString);
});

await MigrateDatabase(builder.Services);

builder.Services.AddHangfireServer();
builder.Services.AddHangfire(configuration =>
{
    configuration.UseSqlServerStorage(connectionString)
        .UseColouredConsoleLogProvider()
        .UseSimpleAssemblyNameTypeSerializer()
        .UseRecommendedSerializerSettings();
});

builder.Services.AddMassTransit(massTransitConfig =>
{
    massTransitConfig.UsingRabbitMq((context, cfg) =>
    {
        var eventBusConnectionString =
            Environment.GetEnvironmentVariable(CommonConstants.EVENT_BUS_CONNECTION_STRING_KEY);
        ArgumentNullException.ThrowIfNull(eventBusConnectionString);

        cfg.Host(new Uri(eventBusConnectionString));
    });
});

var app = builder.Build();

app.UseMiddleware<ExceptionHandlingMiddleware>();
app.UseSerilogRequestLogging();

app.UseHangfireDashboard();

if (app.Environment.IsDevelopment() || app.Environment.IsDockerComposeEnvironment())
{
    IdentityModelEventSource.ShowPII = true;
    app.UseSwagger();
    app.UseSwaggerUI();
    app.UseODataRouteDebug();
}

app.UseCors(corsPolicyBuilder => corsPolicyBuilder.AllowAnyOrigin().AllowAnyHeader().AllowAnyMethod()
    .WithExposedHeaders("X-Correlation-id"));

app.UseAuthentication();
app.UseAuthorization();
app.MapControllers();

app.Run();

async Task MigrateDatabase(IServiceCollection serviceCollection)
{
    var serviceProvider = serviceCollection.BuildServiceProvider();
    var dbContext = serviceProvider.GetRequiredService<ApplicationDbContext>();

    if (dbContext.Database.GetPendingMigrations().Any())
    {
        await dbContext.Database.MigrateAsync();
        Log.Logger.Warning("Projects database has been migrated");
    }
}
