﻿# For command line usage, we provide a shell script which does the same job as the "Build" button in the Ribbon Designer and converts the XML file generated by the ribbon designer into the necessary RES and PAS files.
# 
# powershell -f .\Generate.Ribbon.Markup.pas.ps1 "..\..\Samples\High Level\02 Text Pad with Action List\Ribbon\RibbonMarkup.xml"
#
# As a second optional parameter you can supply the resource name of the ribbon in the RES-file. If omitted, the name will be APPLICATION_RIBBON.
# For more details please visit the Wiki at:  https://github.com/TurboPack/RibbonFramework/wiki/Tutorial:-Integrating-with-Delphi

# Stop on errors
$ErrorActionPreference = "Stop"

$xmlFilePath = $args[0]
if ([string]::IsNullOrEmpty($xmlFilePath))
{
    write "Please specify a ribbon XML file as first paramter. Aborting execution."
    exit 1
}

# Determine the current working directory from the given xml file path
$workingDir = ([System.IO.Path]::GetDirectoryName($xmlFilePath))
if ([string]::IsNullOrEmpty($workingDir))
{
    $workingDir = "."
}
$workingDir = $workingDir + ([System.IO.Path]::DirectorySeparatorChar)

# Prepare file paths for the files that we want to create
$pasFilePath = $workingDir + ([System.IO.Path]::GetFileNameWithoutExtension($xmlFilePath) + ".pas")
$bmlFilePath = $workingDir + ([System.IO.Path]::GetFileNameWithoutExtension($xmlFilePath) + ".bml")
$rcFilePath = $workingDir + ([System.IO.Path]::GetFileNameWithoutExtension($xmlFilePath) + ".rc")
$headerFilePath = $workingDir + ([System.IO.Path]::GetFileNameWithoutExtension($xmlFilePath) + ".h")
$resFileName = ([System.IO.Path]::GetFileNameWithoutExtension($xmlFilePath) + ".res")
$unitName = ([System.IO.Path]::GetFileNameWithoutExtension($xmlFilePath))

$ResourceName = $args[1]
if ([string]::IsNullOrEmpty($ResourceName))
{
  $ResourceName = "APPLICATION"
}

$UICCDir = $args[2]

# Checks if a file exists under a given location. If yes, the path to this file is returned. If not, we lookup several known locations and return those, if the file is found.
function FindFileInLocation($pLocation, $pFileName)
{   
    # First check if a valid path was passed via the command line
	if ($pLocation)
	{
        $lPath = Join-Path $pLocation $pFileName
        if (Test-Path $lPath)
        {
            return $lPath
        }
	}
    # Check if the file exists under %PATH%
    if (Get-Command $pFileName -ErrorAction SilentlyContinue)
    {
        return "$pFileName"
    }    
    # If not, check a few known locations for uicc.exe
    elseif (Test-Path "${env:ProgramFiles(x86)}\Microsoft SDKs\Windows\v7.1A\Bin\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Microsoft SDKs\Windows\v7.1A\Bin\$pFileName"
    }
    elseif (Test-Path "${env:ProgramFiles}\Microsoft SDKs\Windows\v7.1\Bin\$pFileName")
    {
        return "${env:ProgramFiles}\Microsoft SDKs\Windows\v7.1\Bin\$pFileName"
    }
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\8.0\bin\x86\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\8.0\bin\x86\$pFileName"
    }
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\8.1\bin\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\8.1\bin\x86\$pFileName"
    }    
# begin modification: add the path to windows 10 sdk.
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\10\bin\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\10\bin\x86\$pFileName"
    }    
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.22000.0\x86\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.22000.0\x86\$pFileName"
    }    
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.19041.0\x86\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.19041.00\x86\$pFileName"
    }    
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.17134.0\x86\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.17134.0\x86\$pFileName"
    }    
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.16299.0\x86\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.16299.0\x86\$pFileName"
    }    
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.15063.0\x86\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.15063.0\x86\$pFileName"
    }    
    elseif (Test-Path "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.14393.0\x86\$pFileName")
    {
        return "${env:ProgramFiles(x86)}\Windows Kits\10\bin\10.0.14393.0\x86\$pFileName"
    }    
# end modification.
    elseif (Test-Path "$PSScriptRoot\$pFileName")
    {
        return "$PSScriptRoot\$pFileName"
    }    
    else
    {
        # Nothing found -> exit
        write "Cannot find $pFileName. Aborting execution."
        exit 1
    }
}

# Find UICC.exe
$UICCCmd = FindFileInLocation -pLocation $UICCDir -pFileName "UICC.exe"
write-host "UICC.exe found: Using $UICCCmd"

# Use the provided xml file to Create the .bml, .h and .rc file
& $UICCCmd "/W0" "$xmlFilePath" "$bmlFilePath" "/header:$headerFilePath" "/res:$rcFilePath" "/name:$ResourceName"
If ($LASTEXITCODE -ne 0) 
{
    exit $LASTEXITCODE
}

# Find rc.exe (Use the same locations as UICC.exe)
$RCCmd = FindFileInLocation -pLocation $UICCDir -pFileName "rc.exe"
write-host "RC.exe found: Using $RCCmd"

# Create the .RES resource file
& $RCCmd "$rcFilePath"
If ($LASTEXITCODE -ne 0) 
{
    exit $LASTEXITCODE
}

# Create a new Markup .pas file that will contain the Ribbon command constants.

[System.Collections.ArrayList]$markupContent = New-Object([System.Collections.ArrayList])

$FileTopPart = @"
unit $unitName;

// *****************************************************************************
// * This is an automatically generated source file for UI Element definition  *
// * resource symbols and values. Please do not modify manually.               *
// *****************************************************************************

interface

{`$R '$resFileName'}

uses
	Generics.Collections, SysUtils, UIRibbon;

const
"@

write-host "Setting content to " + $pasFilePath
$pasFile = New-Object System.IO.StreamWriter $pasFilePath
$pasfile.WriteLine($FileTopPart)

# Get content of the header file (e.g. TreeSize.Ribbon.Markup.h).
$data = Get-Content "$headerFilePath"

foreach ($line in $data)
{
	if ($line.Contains("#define"))
	{
        $part = $line.TrimStart("#define").Split(" ")
		$appendLine = "  " + $part[1] + " = " + $part[2] + ";"
        $dummy = $markupContent.Add($appendLine);
	}
}

$pasfile.WriteLine($($markupContent -join [Environment]::NewLine))
# Add some additional predefined text.
$FileMiddlePart = @"

implementation

function RegisterRibbonElements(): TRibbonMarkupElementList;
begin
  Result := TRibbonMarkupElementList.Create('$ResourceName');
"@

$pasfile.WriteLine($FileMiddlePart)
    

# Add the mapping by using the previously generated markup content

# Initialization
$commandName = ""
$LabelTitleResourceID = -1
$LabelDescriptionResourceID = -1
$TooltipTitleResourceID = -1
$TooltipDescriptionResourceID = -1

#RegEx for resource IDs
$resourceIdRegexPattern = "\b\d{1,5}\b"

foreach ($line in $markupContent)
{
	if (!($line.Contains("RESID")))
	{       
		if (($commandName) -and ($commandID))
		{
			$appendLine = "  Result.Add(TRibbonMarkupElement.Create('$commandName', $commandId, $LabelTitleResourceID, $LabelDescriptionResourceID, $TooltipTitleResourceID, $TooltipDescriptionResourceID));"
			$pasfile.WriteLine($appendLine)
			$LabelTitleResourceID = -1
            $LabelDescriptionResourceID = -1
            $TooltipTitleResourceID = -1
            $TooltipDescriptionResourceID = -1
		}
	
		$commandName = ([regex]"\b\w+\b").match($line).groups[0].value
		$commandId = ([regex]$resourceIdRegexPattern).match($line).groups[0].value
		continue
	}
	if ($commandName -and $line.Contains($commandName))
	{
		if ($line.Contains("LabelTitle")) 
		{
			$LabelTitleResourceID = ([regex]$resourceIdRegexPattern).match($line).groups[0].value
		}
		elseif ($line.Contains("LabelDescription")) 
		{
			$LabelDescriptionResourceID = ([regex]$resourceIdRegexPattern).match($line).groups[0].value
		}
		elseif ($line.Contains("TooltipTitle")) 
		{
			$TooltipTitleResourceID = ([regex]$resourceIdRegexPattern).match($line).groups[0].value
		}
		elseif ($line.Contains("TooltipDescription")) 
		{
			$TooltipDescriptionResourceID = ([regex]$resourceIdRegexPattern).match($line).groups[0].value
		}
	}
}

if (($commandName) -and ($commandID))
{
	$appendLine = "  Result.Add(TRibbonMarkupElement.Create('$commandName', $commandId, $LabelTitleResourceID, $TooltipTitleResourceID));"    
	$pasfile.WriteLine($appendLine)
}

# Add the ending part
$FileEndPart = @"
end;
initialization

  RegisterRibbonElements();
  
end.
"@

$pasFile.WriteLine($FileEndPart)
$pasFile.Close()
write-host "Ribbon pascal markup file generation successful: '$pasFilePath'"
