import { test, expect, vi } from 'vitest'

import { testConfigFile } from '../../../test'
import type { SubscriptionSelection } from '../../lib'
import { RefetchUpdateMode } from '../../lib'
import { Cache } from '../cache'
import { rootID } from '../stuff'

const config = testConfigFile()

test('root subscribe - field change', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors',
						},
					},
				},
			},
		},
	}

	// write some data
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				favoriteColors: ['red', 'green', 'blue'],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// somehow write a user to the cache with the same id, but a different name
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'mary',
			},
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			firstName: 'mary',
			favoriteColors: ['red', 'green', 'blue'],
			id: '1',
		},
	})
})

test('root subscribe - linked object changed', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors(where: "foo")',
							nullable: true,
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				favoriteColors: ['red', 'green', 'blue'],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// somehow write a user to the cache with a different id
	cache.write({
		selection,
		data: {
			viewer: {
				id: '2',
				firstName: 'mary',
			},
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			firstName: 'mary',
			// this is a sanity-check. the cache wasn't written with that value
			favoriteColors: null,
			id: '2',
		},
	})

	// write a value to the new record
	cache.write({
		selection: {
			fields: {
				firstName: {
					type: 'String',
					visible: true,
					keyRaw: 'firstName',
				},
			},
		},
		data: {
			firstName: 'Michelle',
		},
		parent: 'User:2',
	})

	expect(set).toHaveBeenCalledTimes(2)

	// make sure that set got called with the full response
	expect(set).toHaveBeenLastCalledWith({
		viewer: {
			firstName: 'Michelle',
			id: '2',
			favoriteColors: null,
		},
	})

	// make sure we are no longer subscribing to user 1
	expect(cache._internal_unstable.subscriptions.get('User:1', 'firstName')).toHaveLength(0)
})

test("subscribing to null object doesn't explode", function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						favoriteColors: {
							nullable: true,
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors(where: "foo")',
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: null,
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// somehow write a user to the cache with a different id
	cache.write({
		selection,
		data: {
			viewer: {
				id: '2',
				firstName: 'mary',
			},
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			firstName: 'mary',
			favoriteColors: null,
			id: '2',
		},
	})
})

test('overwriting a reference with null clears its subscribers', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				nullable: true,
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors(where: "foo")',
						},
					},
				},
			},
		},
	}

	// somehow write a user to the cache with a different id
	cache.write({
		selection,
		data: {
			viewer: {
				id: '2',
				firstName: 'mary',
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: null,
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: null,
	})

	// we shouldn't be subscribing to user 3 any more
	expect(cache._internal_unstable.subscriptions.get('User:2', 'firstName')).toHaveLength(0)
})

test('overwriting a linked list with null clears its subscribers', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							nullable: true,
							selection: {
								fields: {
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// add some users that we will subscribe to
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{ id: '2', firstName: 'Jason' },
					{ id: '3', firstName: 'Nick' },
				],
			},
		},
	})

	// make sure something is subscribing to the friends field
	expect(cache._internal_unstable.subscriptions.get('User:1', 'friends')).toHaveLength(1)
	expect(cache._internal_unstable.subscriptions.get('User:2', 'firstName')).toHaveLength(1)
	expect(cache._internal_unstable.subscriptions.get('User:3', 'firstName')).toHaveLength(1)

	// write null over the list
	cache.write({
		selection: {
			fields: {
				id: {
					type: 'String',
					visible: true,
					keyRaw: 'id',
				},
				friends: selection.fields!.viewer.selection!.fields!.friends,
			},
		},
		data: {
			id: '1',
			friends: null,
		},
		parent: 'User:1',
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenNthCalledWith(2, {
		viewer: {
			id: '1',
			friends: null,
		},
	})

	// we shouldn't be subscribing to user 3 any more
	expect(cache._internal_unstable.subscriptions.get('User:2', 'firstName')).toHaveLength(0)
	expect(cache._internal_unstable.subscriptions.get('User:3', 'firstName')).toHaveLength(0)
})

test('root subscribe - linked list lost entry', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with two objects
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
						firstName: 'jane',
					},
					{
						id: '3',
						firstName: 'mary',
					},
				],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// somehow write a user to the cache with a new friends list
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
					},
				],
			},
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			id: '1',
			friends: [
				{
					firstName: 'jane',
					id: '2',
				},
			],
		},
	})

	// we shouldn't be subscribing to user 3 any more
	expect(cache._internal_unstable.subscriptions.get('User:3', 'firstName')).toHaveLength(0)
})

test("subscribing to list with null values doesn't explode", function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
						firstName: 'jane',
					},
					null,
				],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// somehow write a user to the cache with a new friends list
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
					},
				],
			},
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			id: '1',
			friends: [
				{
					firstName: 'jane',
					id: '2',
				},
			],
		},
	})
})

test('root subscribe - linked list reorder', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
						firstName: 'jane',
					},
					{
						id: '3',
						firstName: 'mary',
					},
				],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		set,
		selection,
	})

	// somehow write a user to the cache with the same id, but a different name
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '3',
					},
					{
						id: '2',
					},
				],
			},
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			id: '1',
			friends: [
				{
					id: '3',
					firstName: 'mary',
				},
				{
					id: '2',
					firstName: 'jane',
				},
			],
		},
	})

	// we should still be subscribing to both users
	expect(cache._internal_unstable.subscriptions.get('User:2', 'firstName')).toHaveLength(1)
	expect(cache._internal_unstable.subscriptions.get('User:3', 'firstName')).toHaveLength(1)
})

test('unsubscribe', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors(where: "foo")',
						},
					},
				},
			},
		},
	}

	// write some data
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				favoriteColors: ['red', 'green', 'blue'],
			},
		},
	})

	// the spec we will register/unregister
	const spec = {
		rootType: 'Query',
		selection,
		set: vi.fn(),
	}

	// subscribe to the fields
	cache.subscribe(spec)

	// make sure we  registered the subscriber
	expect(cache._internal_unstable.subscriptions.get('User:1', 'firstName')).toHaveLength(1)

	// unsubscribe
	cache.unsubscribe(spec)

	// make sure there is no more subscriber
	expect(cache._internal_unstable.subscriptions.get('User:1', 'firstName')).toHaveLength(0)
})

test('subscribe to new list nodes', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							list: {
								name: 'All_Users',
								connection: false,
								type: 'User',
							},
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		set,
		selection,
	})

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
						firstName: 'jane',
					},
				],
			},
		},
	})

	// update the user we just added
	cache.write({
		selection: {
			fields: {
				id: {
					type: 'String',
					visible: true,
					keyRaw: 'id',
				},
				firstName: {
					type: 'String',
					visible: true,
					keyRaw: 'firstName',
				},
			},
		},
		data: {
			id: '2',
			firstName: 'jane-prime',
		},
		parent: 'User:2',
	})

	// the first time set was called, a new entry was added.
	// the second time it's called, we get a new value for jane
	expect(set).toHaveBeenNthCalledWith(2, {
		viewer: {
			id: '1',
			friends: [
				{
					firstName: 'jane-prime',
					id: '2',
				},
			],
		},
	})

	// add a new user
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
						firstName: 'jane-prime',
					},
					{
						id: '3',
						firstName: 'mary',
					},
				],
			},
		},
	})

	// update the user we just added
	cache.write({
		selection: {
			fields: {
				id: {
					type: 'String',
					visible: true,
					keyRaw: 'id',
				},
				firstName: {
					type: 'String',
					visible: true,
					keyRaw: 'firstName',
				},
			},
		},
		data: {
			id: '3',
			firstName: 'mary-prime',
		},
		parent: 'User:3',
	})

	// the third time set was called, a new entry was added.
	// the fourth time it's called, we get a new value for mary
	expect(set).toHaveBeenNthCalledWith(4, {
		viewer: {
			id: '1',
			friends: [
				{
					firstName: 'jane-prime',
					id: '2',
				},
				{
					firstName: 'mary-prime',
					id: '3',
				},
			],
		},
	})
})

test('variables in query and subscription', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends(filter: $filter)',
							list: {
								name: 'All_Users',
								connection: false,
								type: 'User',
							},
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
						firstName: 'jane',
					},
					{
						id: '3',
						firstName: 'mary',
					},
				],
			},
		},
		variables: {
			filter: 'foo',
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe(
		{
			rootType: 'Query',
			selection,
			set,
			variables: () => ({ filter: 'foo' }),
		},
		{
			filter: 'foo',
		}
	)

	// make sure we have a cached value for friends(filter: "foo")
	expect(cache.list('All_Users').lists[0].key).toEqual('friends(filter: "foo")')

	// somehow write a user to the cache with a new friends list
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: [
					{
						id: '2',
					},
				],
			},
		},
		variables: {
			filter: 'foo',
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			id: '1',
			friends: [
				{
					firstName: 'jane',
					id: '2',
				},
			],
		},
	})

	// we shouldn't be subscribing to user 3 any more
	expect(cache._internal_unstable.subscriptions.get('User:3', 'firstName')).toHaveLength(0)
})

test('deleting a node removes nested subscriptions', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						__typename: {
							type: 'String',
							visible: true,
							keyRaw: '__typename',
						},
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							list: {
								name: 'All_Users',
								connection: false,
								type: 'User',
							},
							selection: {
								fields: {
									__typename: {
										type: 'String',
										visible: true,
										keyRaw: '__typename',
									},
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				friends: [
					{
						__typename: 'User',
						id: '2',
						firstName: 'jane',
					},
				],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// sanity check
	expect(cache._internal_unstable.subscriptions.get('User:2', 'firstName')).toHaveLength(1)

	// delete the parent
	cache.delete('User:1')

	// sanity check
	expect(cache._internal_unstable.subscriptions.get('User:2', 'firstName')).toHaveLength(0)
})

test('find subSelection', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						__typename: {
							type: 'String',
							visible: true,
							keyRaw: '__typename',
						},
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							list: {
								name: 'All_Users',
								connection: false,
								type: 'User',
							},
							selection: {
								fields: {
									__typename: {
										type: 'String',
										visible: true,
										keyRaw: '__typename',
									},
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				friends: [
					{
						__typename: 'User',
						id: '2',
						firstName: 'jane',
					},
				],
			},
		},
	})

	const subSelections = cache._internal_unstable.subscriptions.findSubSelections(
		rootID,
		selection,
		{},
		'User:1'
	)

	expect(subSelections).toEqual([
		{
			fields: {
				__typename: {
					type: 'String',
					visible: true,
					keyRaw: '__typename',
				},
				id: {
					type: 'ID',
					visible: true,
					keyRaw: 'id',
				},
				friends: {
					type: 'User',
					visible: true,
					keyRaw: 'friends',
					list: {
						name: 'All_Users',
						connection: false,
						type: 'User',
					},
					selection: {
						fields: {
							__typename: {
								type: 'String',
								visible: true,
								keyRaw: '__typename',
							},
							id: {
								type: 'ID',
								visible: true,
								keyRaw: 'id',
							},
							firstName: {
								type: 'String',
								visible: true,
								keyRaw: 'firstName',
							},
						},
					},
				},
			},
		},
	])
})

test('find subSelection - avoid cycles', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						__typename: {
							type: 'String',
							visible: true,
							keyRaw: '__typename',
						},
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							list: {
								name: 'All_Users',
								connection: false,
								type: 'User',
							},
							selection: {
								fields: {
									__typename: {
										type: 'String',
										visible: true,
										keyRaw: '__typename',
									},
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				friends: [
					{
						__typename: 'User',
						id: '1',
						firstName: 'jane',
					},
				],
			},
		},
	})

	const subSelections = cache._internal_unstable.subscriptions.findSubSelections(
		rootID,
		selection,
		{},
		'User:1'
	)

	expect(subSelections).toEqual([
		{
			fields: {
				__typename: {
					type: 'String',
					visible: true,
					keyRaw: '__typename',
				},
				id: {
					type: 'ID',
					visible: true,
					keyRaw: 'id',
				},
				friends: {
					type: 'User',
					visible: true,
					keyRaw: 'friends',
					list: {
						name: 'All_Users',
						connection: false,
						type: 'User',
					},
					selection: {
						fields: {
							__typename: {
								type: 'String',
								visible: true,
								keyRaw: '__typename',
							},
							id: {
								type: 'ID',
								visible: true,
								keyRaw: 'id',
							},
							firstName: {
								type: 'String',
								visible: true,
								keyRaw: 'firstName',
							},
						},
					},
				},
			},
		},
	])
})

test('find subSelection - deeply nested', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						__typename: {
							type: 'String',
							visible: true,
							keyRaw: '__typename',
						},
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							list: {
								name: 'All_Users',
								connection: false,
								type: 'User',
							},
							selection: {
								fields: {
									__typename: {
										type: 'String',
										visible: true,
										keyRaw: '__typename',
									},
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				friends: [
					{
						__typename: 'User',
						id: '2',
						firstName: 'jane',
					},
				],
			},
		},
	})

	const subSelections = cache._internal_unstable.subscriptions.findSubSelections(
		rootID,
		selection,
		{},
		'User:2'
	)

	expect(subSelections).toEqual([
		{
			fields: {
				__typename: {
					type: 'String',
					visible: true,
					keyRaw: '__typename',
				},
				id: {
					type: 'ID',
					visible: true,
					keyRaw: 'id',
				},
				firstName: {
					type: 'String',
					visible: true,
					keyRaw: 'firstName',
				},
			},
		},
	])
})

test('same record twice in a query survives one unsubscribe (reference counting)', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							list: {
								name: 'All_Users',
								connection: false,
								type: 'User',
							},
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				friends: [
					{
						id: '1',
						firstName: 'bob',
					},
				],
			},
		},
		variables: {
			filter: 'foo',
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe(
		{
			rootType: 'Query',
			selection,
			set,
		},
		{
			filter: 'foo',
		}
	)

	// make sure there is a subscriber for the user's first name
	expect(cache._internal_unstable.subscriptions.get('User:1', 'firstName')).toHaveLength(1)

	// remove the user from the list
	cache.list('All_Users').remove({ id: '1' })

	// we should still be subscribing to the user's first name
	expect(cache._internal_unstable.subscriptions.get('User:1', 'firstName')).toHaveLength(1)
})

test('embedded references', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							selection: {
								fields: {
									edges: {
										type: 'UserEdge',
										visible: true,
										keyRaw: 'edges',
										selection: {
											fields: {
												node: {
													type: 'User',
													visible: true,
													keyRaw: 'node',
													selection: {
														fields: {
															id: {
																type: 'ID',
																visible: true,
																keyRaw: 'id',
															},
															firstName: {
																type: 'String',
																visible: true,
																keyRaw: 'firstName',
															},
														},
													},
												},
											},
										},
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// write an embedded list of embedded objects holding references to an object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: {
					edges: [
						{
							node: {
								id: '2',
								firstName: 'jane',
							},
						},
						{
							node: {
								id: '3',
								firstName: 'mary',
							},
						},
					],
				},
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe(
		{
			rootType: 'Query',
			selection,
			set,
		},
		{
			filter: 'foo',
		}
	)

	// update one of the embedded references
	cache.write({
		selection: {
			fields: {
				user: {
					type: 'User',
					visible: true,
					keyRaw: 'user',
					selection: {
						fields: {
							id: {
								type: 'ID',
								visible: true,
								keyRaw: 'id',
							},
							firstName: {
								type: 'String',
								visible: true,
								keyRaw: 'firstName',
							},
						},
					},
				},
			},
		},
		data: {
			user: {
				id: '2',
				firstName: 'not-jane',
			},
		},
	})

	// make sure we got the updated data
	expect(set).toHaveBeenCalledWith({
		viewer: {
			id: '1',
			friends: {
				edges: [
					{
						node: {
							id: '2',
							firstName: 'not-jane',
						},
					},
					{
						node: {
							id: '3',
							firstName: 'mary',
						},
					},
				],
			},
		},
	})
})

test('self-referencing linked lists can be unsubscribed (avoid infinite recursion)', function () {
	// instantiate the cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
									friends: {
										type: 'User',
										visible: true,
										keyRaw: 'friends',
										selection: {
											fields: {
												id: {
													type: 'ID',
													visible: true,
													keyRaw: 'id',
												},
												firstName: {
													type: 'String',
													visible: true,
													keyRaw: 'firstName',
												},
											},
										},
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// add some data to the cache
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				friends: [
					{
						id: '1',
						firstName: 'bob',
						friends: [
							{
								id: '1',
								firstName: 'bob',
							},
						],
					},
				],
			},
		},
	})

	// subscribe to the list
	const spec = {
		set: vi.fn(),
		selection,
		rootType: 'Query',
	}
	cache.subscribe(spec)
	cache.unsubscribe(spec)

	// no one should be subscribing to User:1's first name
	expect(cache._internal_unstable.subscriptions.get('User:1', 'firstName')).toHaveLength(0)
})

test('self-referencing links can be unsubscribed (avoid infinite recursion)', function () {
	// instantiate the cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						friend: {
							type: 'User',
							visible: true,
							keyRaw: 'friend',
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									firstName: {
										type: 'String',
										visible: true,
										keyRaw: 'firstName',
									},
									friend: {
										type: 'User',
										visible: true,
										keyRaw: 'friend',
										selection: {
											fields: {
												id: {
													type: 'ID',
													visible: true,
													keyRaw: 'id',
												},
												firstName: {
													type: 'String',
													visible: true,
													keyRaw: 'firstName',
												},
												friend: {
													type: 'User',
													visible: true,
													keyRaw: 'friend',
													selection: {
														fields: {
															id: {
																type: 'ID',
																visible: true,
																keyRaw: 'id',
															},
															firstName: {
																type: 'String',
																visible: true,
																keyRaw: 'firstName',
															},
														},
													},
												},
											},
										},
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// add some data to the cache
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				friend: {
					id: '1',
					firstName: 'bob',
					friend: {
						id: '1',
						firstName: 'bob',
						friend: {
							id: '1',
							firstName: 'bob',
						},
					},
				},
			},
		},
	})

	// subscribe to the list
	const spec = {
		set: vi.fn(),
		selection,
		rootType: 'Query',
	}
	cache.subscribe(spec)
	cache.unsubscribe(spec)

	// no one should be subscribing to User:1's first name
	expect(cache._internal_unstable.subscriptions.get('User:1', 'firstName')).toHaveLength(0)
})

test('overwriting a value in an optimistic layer triggers subscribers', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors',
						},
					},
				},
			},
		},
	}

	// write some data
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				favoriteColors: ['red', 'green', 'blue'],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// create an optimistic layer on top
	const layer = cache._internal_unstable.storage.createLayer(true)

	// somehow write a user to the cache with the same id, but a different name
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'mary',
			},
		},
		layer: layer.id,
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			firstName: 'mary',
			favoriteColors: ['red', 'green', 'blue'],
			id: '1',
		},
	})
})

test('clearing a display layer updates subscribers', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors',
						},
					},
				},
			},
		},
	}

	// write some data
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
				favoriteColors: ['red', 'green', 'blue'],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// create an optimistic layer on top
	const layer = cache._internal_unstable.storage.createLayer(true)

	// somehow write a user to the cache with the same id, but a different name
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'mary',
			},
		},
		layer: layer.id,
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			firstName: 'mary',
			favoriteColors: ['red', 'green', 'blue'],
			id: '1',
		},
	})

	// clear the layer
	layer.clear()

	// write the with the same values to the layer that were previously in place
	cache.write({
		selection,
		data: {
			viewer: {
				firstName: 'mary',
				favoriteColors: ['red', 'green', 'blue'],
				id: '1',
			},
		},
		layer: layer.id,
	})

	expect(set).toHaveBeenNthCalledWith(2, {
		viewer: {
			firstName: 'mary',
			favoriteColors: ['red', 'green', 'blue'],
			id: '1',
		},
	})
})

test('optimistic layer & lists & add ok', function () {
	// instantiate a cache
	const cache = new Cache(config)

	// selection for the list
	const selectionList: SubscriptionSelection = {
		fields: {
			usersList: {
				type: 'User',
				keyRaw: 'usersList',
				directives: [
					{
						name: 'list',
						arguments: {
							name: {
								kind: 'StringValue',
								value: 'OptimisticUsersList',
							},
						},
					},
				],
				list: {
					name: 'OptimisticUsersList',
					connection: false,
					type: 'User',
				},
				selection: {
					fields: {
						name: {
							type: 'String',
							keyRaw: 'name',
							visible: true,
						},
						id: {
							type: 'ID',
							keyRaw: 'id',
							visible: true,
						},
					},
				},
				visible: true,
			},
		},
	}

	// selection for adding a user
	const selectionMutation: SubscriptionSelection = {
		fields: {
			addUser: {
				type: 'User',
				keyRaw: 'addUser',
				nullable: true,
				operations: [
					{
						action: 'insert',
						list: 'OptimisticUsersList',
						position: 'last',
					},
				],
				selection: {
					fields: {
						name: {
							type: 'String',
							keyRaw: 'name',
						},
						id: {
							type: 'ID',
							keyRaw: 'id',
							visible: true,
						},
					},
					fragments: {
						OptimisticUsersList_insert: {
							arguments: {},
						},
					},
				},
				visible: true,
			},
		},
	}

	// write initial data
	cache.write({
		selection: selectionList,
		data: {
			usersList: [
				{
					id: 'mutation-opti-list:1',
					name: 'Bruce Willis',
				},
				{
					id: 'mutation-opti-list:2',
					name: 'Samuel Jackson',
				},
			],
		},
	})

	// a function to spy on that will play the role of set
	const setOnQuery = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection: selectionList,
		set: setOnQuery,
	})

	// create an optimistic layer on top
	const layer = cache._internal_unstable.storage.createLayer(true)

	// somehow write a user to the cache with the same id, but a different name
	cache.write({
		selection: selectionMutation,
		data: {
			addUser: {
				id: '??? id ???',
				name: '...optimisticResponse... I could have guessed JYC!',
			},
		},
		layer: layer.id,
	})

	// make sure that set got called with the full response
	expect(setOnQuery).toHaveBeenCalledWith({
		usersList: [
			{
				id: 'mutation-opti-list:1',
				name: 'Bruce Willis',
			},
			{
				id: 'mutation-opti-list:2',
				name: 'Samuel Jackson',
			},
			{
				id: '??? id ???',
				name: '...optimisticResponse... I could have guessed JYC!',
			},
		],
	})

	// clear the layer
	layer.clear()

	// write the with the same values to the layer that were previously in place
	cache.write({
		selection: selectionMutation,
		data: {
			addUser: {
				id: 'mutation-opti-list:9',
				name: 'Alec',
			},
		},
		layer: layer.id,
	})

	expect(setOnQuery).toHaveBeenNthCalledWith(2, {
		usersList: [
			{
				id: 'mutation-opti-list:1',
				name: 'Bruce Willis',
			},
			{
				id: 'mutation-opti-list:2',
				name: 'Samuel Jackson',
			},
			{
				id: 'mutation-opti-list:9',
				name: 'Alec',
			},
		],
	})
})

test('optimistic layer & lists & add null (optimistic will revert)', function () {
	// instantiate a cache
	const cache = new Cache(config)

	// selection for the list
	const selectionList: SubscriptionSelection = {
		fields: {
			usersList: {
				type: 'User',
				keyRaw: 'usersList',
				directives: [
					{
						name: 'list',
						arguments: {
							name: {
								kind: 'StringValue',
								value: 'OptimisticUsersList',
							},
						},
					},
				],
				list: {
					name: 'OptimisticUsersList',
					connection: false,
					type: 'User',
				},
				selection: {
					fields: {
						name: {
							type: 'String',
							keyRaw: 'name',
							visible: true,
						},
						id: {
							type: 'ID',
							keyRaw: 'id',
							visible: true,
						},
					},
				},
				visible: true,
			},
		},
	}

	// selection for adding a user
	const selectionMutation: SubscriptionSelection = {
		fields: {
			addUser: {
				type: 'User',
				keyRaw: 'addUser',
				nullable: true,
				operations: [
					{
						action: 'insert',
						list: 'OptimisticUsersList',
						position: 'last',
					},
				],
				selection: {
					fields: {
						name: {
							type: 'String',
							keyRaw: 'name',
						},
						id: {
							type: 'ID',
							keyRaw: 'id',
							visible: true,
						},
					},
					fragments: {
						OptimisticUsersList_insert: {
							arguments: {},
						},
					},
				},
				visible: true,
			},
		},
	}

	// write initial data
	cache.write({
		selection: selectionList,
		data: {
			usersList: [
				{
					id: 'mutation-opti-list:1',
					name: 'Bruce Willis',
				},
				{
					id: 'mutation-opti-list:2',
					name: 'Samuel Jackson',
				},
			],
		},
	})

	// a function to spy on that will play the role of set
	const setOnQuery = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection: selectionList,
		set: setOnQuery,
	})

	// create an optimistic layer on top
	const layer = cache._internal_unstable.storage.createLayer(true)

	// somehow write a user to the cache with the same id, but a different name
	cache.write({
		selection: selectionMutation,
		data: {
			addUser: {
				id: '??? id ???',
				name: '...optimisticResponse... I could have guessed JYC!',
			},
		},
		layer: layer.id,
	})

	// make sure that set got called with the full response
	expect(setOnQuery).toHaveBeenCalledWith({
		usersList: [
			{
				id: 'mutation-opti-list:1',
				name: 'Bruce Willis',
			},
			{
				id: 'mutation-opti-list:2',
				name: 'Samuel Jackson',
			},
			{
				id: '??? id ???',
				name: '...optimisticResponse... I could have guessed JYC!',
			},
		],
	})

	// clear the layer
	cache.clearLayer(layer.id)

	// write the with the same values to the layer that were previously in place
	cache.write({
		selection: selectionMutation,
		data: {
			addUser: null,
		},
		layer: layer.id,
	})

	// So we have the right data in cache.
	expect(cache.read({ selection: selectionList }).data).toMatchObject({
		usersList: [
			{
				id: 'mutation-opti-list:1',
				name: 'Bruce Willis',
			},
			{
				id: 'mutation-opti-list:2',
				name: 'Samuel Jackson',
			},
		],
	})

	// Subscription should be call with the right data (element removed from the list)
	expect(setOnQuery).toHaveBeenNthCalledWith(2, {
		usersList: [
			{
				id: 'mutation-opti-list:1',
				name: 'Bruce Willis',
			},
			{
				id: 'mutation-opti-list:2',
				name: 'Samuel Jackson',
			},
		],
	})
})

test('ensure parent type is properly passed for nested lists', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			cities: {
				type: 'City',
				visible: true,
				keyRaw: 'cities',
				list: {
					name: 'City_List',
					connection: false,
					type: 'City',
				},
				updates: [RefetchUpdateMode.append],
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						name: {
							type: 'String',
							visible: true,
							keyRaw: 'name',
						},
						libraries: {
							type: 'Library',
							visible: true,
							keyRaw: 'libraries',
							list: {
								name: 'Library_List',
								connection: false,
								type: 'Library',
							},
							selection: {
								fields: {
									id: {
										type: 'ID',
										visible: true,
										keyRaw: 'id',
									},
									name: {
										type: 'String',
										visible: true,
										keyRaw: 'name',
									},
									books: {
										type: 'Book',
										visible: true,
										keyRaw: 'books',
										list: {
											name: 'Book_List',
											connection: false,
											type: 'Book',
										},
										selection: {
											fields: {
												id: {
													type: 'ID',
													visible: true,
													keyRaw: 'id',
												},
												title: {
													type: 'String',
													visible: true,
													keyRaw: 'title',
												},
											},
										},
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// add a city to the list by hand since using the list util adds type information

	cache.write({
		selection,
		data: {
			cities: [
				{
					id: '1',
					name: 'Alexandria',
					libraries: [
						{
							id: '1',
							name: 'The Library of Alexandria',
							books: [],
						},
						{
							id: '2',
							name: 'Bibliotheca Alexandrina',
							books: [],
						},
					],
				},
				{
					id: '2',
					name: 'Aalborg',
					libraries: [],
				},
			],
		},
	})

	// since there are multiple lists inside of City_List, we need to
	// specify the parentID of the city in order to add a library to City:3
	expect(() => cache.list('Library_List', '2')).not.toThrow()
	// same with Books_List for Library:2
	expect(() => cache.list('Book_List', '2')).not.toThrow()
})

test('subscribe to abstract fields of matching type', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'Node',
				visible: true,
				keyRaw: 'viewer',
				abstract: true,
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
						__typename: {
							type: 'String',
							visible: true,
							keyRaw: '__typename',
						},
					},
					abstractFields: {
						typeMap: {},
						fields: {
							User: {
								__typename: {
									type: 'String',
									visible: true,
									keyRaw: '__typename',
								},
								id: {
									type: 'ID',
									visible: true,
									keyRaw: 'id',
								},
								firstName: {
									type: 'String',
									visible: true,
									keyRaw: 'firstName',
								},
								favoriteColors: {
									type: 'String',
									visible: true,
									keyRaw: 'favoriteColors',
								},
							},
						},
					},
				},
			},
		},
	}

	// write some data
	cache.write({
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				firstName: 'bob',
				favoriteColors: [],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// somehow write a user to the cache with the same id, but a different name
	cache.write({
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				favoriteColors: ['red', 'green', 'blue'],
			},
		},
	})

	// make sure that set got called with the full response
	expect(set).toHaveBeenCalledWith({
		viewer: {
			__typename: 'User',
			firstName: 'bob',
			favoriteColors: ['red', 'green', 'blue'],
			id: '1',
		},
	})
})

test('overlapping subscriptions', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection1: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				nullable: true,
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors',
						},
					},
				},
			},
		},
	}

	const selection2: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				nullable: true,
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
					},
				},
			},
		},
	}

	// write some data
	cache.write({
		selection: selection1,
		data: {
			viewer: null,
		},
	})

	// subscribe with both subscriptions

	// a function to spy on that will play the role of set
	const set1 = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection: selection1,
		set: set1,
	})

	// a function to spy on that will play the role of set
	const set2 = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection: selection2,
		set: set2,
	})

	// write to the first selection
	cache.write({
		selection: selection1,
		data: {
			viewer: {
				id: '1',
				favoriteColors: ['test'],
			},
		},
	})

	// both sets will have been called by now

	// writing to favoriteColors should only trigger selection1
	cache.write({
		parent: 'User:1',
		selection: {
			fields: {
				id: {
					type: 'ID',
					visible: true,
					keyRaw: 'id',
				},
				favoriteColors: {
					type: 'String',
					visible: true,
					keyRaw: 'favoriteColors',
				},
			},
		},
		data: {
			id: '1',
			favoriteColors: ['test2'],
		},
	})

	expect(set2).toHaveBeenCalledOnce()

	// and writing to firstName should trigger set 2
	cache.write({
		parent: 'User:1',
		selection: {
			fields: {
				id: {
					type: 'ID',
					visible: true,
					keyRaw: 'id',
				},
				firstName: {
					type: 'String',
					visible: true,
					keyRaw: 'firstName',
				},
			},
		},
		data: {
			id: '1',
			firstName: 'test2',
		},
	})
	expect(set2).toHaveBeenCalledTimes(2)
})

test('ignore hidden fields', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				nullable: true,
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						favoriteColors: {
							type: 'String',
							keyRaw: 'favoriteColors',
						},
					},
				},
			},
		},
	}

	// write some data
	cache.write({
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				firstName: 'bob',
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// write to the favoriteColors field to make sure we didn't get an update
	cache.write({
		selection: {
			fields: {
				favoriteColors: {
					type: 'String',
					keyRaw: 'favoriteColors',
				},
			},
		},
		data: {
			favoriteColors: ['blue'],
		},
		parent: 'User:1',
	})

	// make sure we didn't call the update
	expect(set).not.toHaveBeenCalled()
})

test('clearing a layer should notify subscribers of displayed values', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				nullable: true,
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						favoriteColors: {
							type: 'String',
							visible: true,
							keyRaw: 'favoriteColors',
						},
					},
				},
			},
		},
	}

	// create an optimistic layer that we will write to
	const layer = cache._internal_unstable.storage.createLayer(true)

	// write some data
	cache.write({
		layer: layer.id,
		selection,
		data: {
			viewer: {
				__typename: 'User',
				id: '1',
				favoriteColors: ['blue'],
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection,
		set,
	})

	// clear the layer
	cache.clearLayer(layer.id)

	// make sure our callback was invoked
	expect(set).toHaveBeenCalledWith({ viewer: null })
})

test('reverting optimistic remove notifies subscribers', function () {
	// instantiate a cache
	const cache = new Cache(config)

	const user3 = {
		__typename: 'User',
		id: '3',
		firstName: 'jane',
	}

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						friends: {
							type: 'User',
							visible: true,
							keyRaw: 'friends',
							list: {
								name: 'All_Users',
								connection: true,
								type: 'User',
							},
							selection: {
								fields: {
									edges: {
										type: 'UserEdge',
										visible: true,
										keyRaw: 'edges',
										selection: {
											fields: {
												node: {
													type: 'Node',
													visible: true,
													keyRaw: 'node',
													abstract: true,
													selection: {
														fields: {
															__typename: {
																type: 'String',
																visible: true,
																keyRaw: '__typename',
															},
															id: {
																type: 'ID',
																visible: true,
																keyRaw: 'id',
															},
															firstName: {
																type: 'String',
																visible: true,
																keyRaw: 'firstName',
															},
														},
													},
												},
											},
										},
									},
								},
							},
						},
					},
				},
			},
		},
	}

	// start off associated with one object
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				friends: {
					edges: [
						{
							node: {
								__typename: 'User',
								id: '2',
								firstName: 'jane',
							},
						},
						{
							node: user3,
						},
					],
				},
			},
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		set,
		selection,
	})

	// make an optimistic layer that we will use to remove user 3 from the list
	const layer = cache._internal_unstable.storage.createLayer(true)

	// remove user 3 from the list
	cache.list('All_Users').remove(user3, {}, layer)

	// clear the layer
	cache.clearLayer(layer.id)

	// make sure our callback was invoked
	expect(set).toHaveBeenCalledWith({
		viewer: {
			id: '1',
			friends: {
				edges: [
					{
						node: {
							__typename: 'User',
							id: '2',
							firstName: 'jane',
						},
					},
					{
						node: user3,
					},
				],
			},
		},
	})
})

test.todo('can write to and resolve layers')

test.todo("resolving a layer with the same value as the most recent doesn't notify subscribers")

test('overwrite null value with list', function () {
	// instantiate the cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			friends: {
				type: 'User',
				visible: true,
				keyRaw: 'friends',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
					},
				},
			},
		},
	}

	// add some data to the cache
	cache.write({
		selection,
		data: {
			friends: null,
		},
	})

	// a function to spy on that will play the role of set
	const set = vi.fn()

	// subscribe to the fields
	cache.subscribe({
		rootType: 'Query',
		selection: selection,
		set: set,
	})

	// add some data to the cache
	cache.write({
		selection,
		data: {
			friends: [],
		},
	})

	expect(set).toHaveBeenCalledWith({
		friends: [],
	})
})

test('removing all subscribers of a field cleans up reference count object', function () {
	// instantiate the cache
	const cache = new Cache(config)

	const selection: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
					},
				},
			},
		},
	}

	// add some data to the cache
	cache.write({
		selection,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
			},
		},
	})

	// subscribe to the list
	const spec = {
		set: vi.fn(),
		selection,
		rootType: 'Query',
	}
	cache.subscribe(spec)

	// sanity check
	expect(cache._internal_unstable.subscriptions.size).toEqual(3)

	// remove the subscription
	cache.unsubscribe(spec)

	// make sure the subscribers object is empty
	expect(cache._internal_unstable.subscriptions.size).toEqual(0)
})

test('reference count garbage collection requires totally empty garbage', function () {
	// instantiate the cache
	const cache = new Cache(config)

	const selection1: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						id: {
							type: 'ID',
							visible: true,
							keyRaw: 'id',
						},
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
					},
				},
			},
		},
	}

	const selection2: SubscriptionSelection = {
		fields: {
			viewer: {
				type: 'User',
				visible: true,
				keyRaw: 'viewer',
				selection: {
					fields: {
						firstName: {
							type: 'String',
							visible: true,
							keyRaw: 'firstName',
						},
					},
				},
			},
		},
	}

	// add some data to the cache
	cache.write({
		selection: selection1,
		data: {
			viewer: {
				id: '1',
				firstName: 'bob',
			},
		},
	})

	// subscribe to selection 1
	const spec1 = {
		set: vi.fn(),
		selection: selection1,
		rootType: 'Query',
	}
	cache.subscribe(spec1)

	// subscribe to selection 2
	const spec2 = {
		set: vi.fn(),
		selection: selection2,
		rootType: 'Query',
	}
	cache.subscribe(spec2)

	// sanity check
	expect(cache._internal_unstable.subscriptions.size).toEqual(5)

	// remove the subscription from spec 1 which should clear the subscription on id, but not first name
	cache.unsubscribe(spec1)

	// make sure the subscribers object is empty
	expect(cache._internal_unstable.subscriptions.size).toEqual(2)
})
