﻿//* Design by Godra
//*
//* 21-AUG-16  v1.0
//* 
//* This is a simple rotating image control.
//*
//* Its purpose is to demonstrate the VB Net code for the following:
//*  - Limiting the control's paint region (making the control round to fit the image)
//*  - Rotation of the image around its center (arbitrary angle of rotation, -360 to 360 degree range)
//*  - Horizontal/Vertical flipping
//*  - Hardcoding an image and using it as a built-in image
//*

using System;
using System.Drawing;
using System.Windows.Forms;
using System.ComponentModel;

public class RotatingImage1 : Control
{

	//* Declare internal timer to be used for endless rotation of the image
	private readonly Timer Tmr;

	//* Declare built-in/hardcoded image (96x96 "Load Image" picture with transparent background)
	private readonly System.IO.MemoryStream PictureStream = new System.IO.MemoryStream(new byte[] {
		0x89, 0x50, 0x4e, 0x47,	0xd, 0xa, 0x1a,	0xa, 0x0, 0x0, 0x0,	0xd, 0x49, 0x48, 0x44, 0x52,
		0x0, 0x0, 0x0, 0x60, 0x0, 0x0, 0x0,	0x60, 0x8, 0x6,	0x0, 0x0, 0x0, 0xe2, 0x98, 0x77,
		0x38, 0x0, 0x0,	0x0, 0x9, 0x70,	0x48, 0x59,	0x73, 0x0, 0x0,	0xe, 0xc3, 0x0,	0x0, 0xe,
		0xc3, 0x1, 0xc7, 0x6f, 0xa8, 0x64, 0x0,	0x0, 0x3, 0x2b,	0x49, 0x44,	0x41, 0x54,	0x78, 0x9c,
		0xed, 0x9b,	0xd1, 0x95,	0xab, 0x30,	0xc, 0x5, 0xa9,	0x8b, 0x82,	0xa8, 0x87,	0x6a, 0x68,	0x86,
		0x62, 0xfc,	0x12, 0x20,	0x9, 0x1, 0x3b,	0x2b, 0x81,	0x75, 0x26,	0x79, 0x97,	0x8f, 0xf9,	0xd9,
		0x13, 0x9b,	0xc1, 0xb2,	0x6c, 0x6f,	0xac, 0x34,	0x29, 0xa5,	0xe6, 0x82,	0x3, 0x17, 0x50, 0x7,
		0x17, 0x50,	0x7, 0x17, 0x50, 0x7, 0x17,	0x50, 0x7, 0x17, 0x50, 0x7,	0x17, 0x50,	0x7, 0x17,
		0x50, 0x7, 0x17, 0x50, 0x7,	0x17, 0x50,	0x7, 0x17, 0x50, 0x7, 0x17,	0x50, 0x7, 0x17, 0x50,
		0x7, 0x17, 0x50, 0x7, 0x17,	0x50, 0x7, 0x17, 0x50, 0x7,	0x17, 0x50,	0x7, 0x17, 0x50, 0x7,
		0x17, 0x50,	0x7, 0x17, 0x50, 0x7, 0x17,	0x50, 0x7, 0x17, 0x50, 0x7,	0x17, 0x50,	0x7, 0x17,
		0x50, 0x7, 0x17, 0x50, 0x7,	0x17, 0x50,	0x7, 0x17, 0xf8, 0x1d, 0x86, 0xd4, 0xdd, 0x86, 0xab,
		0x1b, 0x52, 0xfa, 0x81,	0x0, 0x8c, 0xa9, 0x6f, 0x9b, 0xd4, 0xcc, 0xb6, 0x5f, 0x30, 0x78, 0x15,
		0x18, 0xba,	0xdb, 0x68,	0xb5, 0xa9, 0x1f, 0x7f,	0x21, 0x0, 0x63, 0x9f, 0xda, 0x5b, 0xd7, 0xed,
		0x6c, 0xcb,	0xf, 0x5e, 0x5,	0xc6, 0xbe,	0xbd, 0x8d,	0x56, 0x97,	0xe6, 0x4, 0xf8, 0xf6, 0x0,
		0x4, 0xcd, 0x16, 0x8e, 0xb8, 0x8c, 0xe,	0x11, 0x1e,	0xba, 0xe6,	0xdc, 0x6c,	0x99, 0x2, 0xd8,
		0xac, 0x70, 0xf4, 0xb5,	0x6b, 0xdb,	0xf8, 0x27,	0xc3, 0xb6,	0x8f, 0xb6,	0xd, 0xcb, 0xe8, 0xaf,
		0x9b, 0x2d,	0xfb, 0xe0,	0x2d, 0xfd,	0xb5, 0x7d,	0x1a, 0xd, 0x41, 0xb8, 0x2f, 0x15, 0xdb, 0x81,
		0xf2, 0x4c, 0x88, 0xec,	0x67, 0x97,	0x80, 0xd4,	0xde, 0x80,	0x83, 0x2, 0x30, 0x9f, 0x16, 0xe,
		0xcd, 0x96, 0xd2, 0xd2,	0x75, 0x76,	0x49, 0x9b,	0xda, 0x1b,	0x2, 0x50, 0x78, 0x4e, 0xd4, 0xfa,
		0x1f, 0x13,	0x80, 0xc3,	0xb3, 0xe5,	0x43, 0xe6,	0x2c, 0x9b,	0xba, 0xb5,	0xcf, 0x79,	0x16, 0x6f,
		0xf8, 0x33,	0x83, 0xca,	0xcf, 0x9f,	0xfa, 0x33,	0x66, 0x20,	0x1e, 0x80,	0xc3, 0xb3,	0xe5, 0xd3,
		0x20, 0x1b,	0x33, 0xe0,	0x39, 0xf0,	0x6f, 0x83,	0x68, 0x5c,	0x12, 0x8b,	0xcf, 0x9f,	0xdb, 0x47,
		0x9d, 0xe8,	0x2a, 0x77,	0xe8, 0x5b,	0xaf, 0xad,	0x83, 0x3c,	0x5, 0xf5, 0xaf, 0x3e, 0x8b, 0xed,
		0x8d, 0x4b, 0x62, 0x29,	0x73, 0x9d,	0xd9, 0x7, 0x7,	0xe0, 0xc4,	0xfa, 0x5f,	0x7c, 0x51,	0x5b,
		0x9f, 0xc5,	0x8d, 0x76,	0xea, 0xd7,	0xb0, 0x7f,	0x94, 0x2, 0x10, 0x7c, 0xa4, 0xae, 0xdb, 0xe1,
		0xa9, 0xd9,	0x92, 0xcb,	0x9e, 0x79,	0xf0, 0x2d,	0x27, 0xaa,	0x79, 0xe9,	0x7b, 0x1f,	0xa8, 0xf9,
		0x6f, 0xd6,	0x13, 0xd0,	0xfe, 0x59,	0xcf, 0xf6,	0x41, 0xeb,	0x7f, 0xf5,	0x0, 0xbc, 0x5e, 0x38,
		0x87, 0x65,	0x16, 0x2d,	0x41, 0x58,	0xb5, 0xb3,	0x7, 0x73, 0xdf, 0xf6, 0x9e, 0x35, 0xae, 0xd,
		0x74, 0x7b,	0xfe, 0xef,	0xfa, 0xf0,	0xaf, 0x54,	0x42, 0x3a, 0xbd, 0xb8,	0x2, 0xf0, 0x33, 0xe0,
		0x2, 0xea, 0xe0, 0x2, 0xea,	0xe0, 0x2, 0xea, 0xe0, 0x2,	0xea, 0xe0,	0x2, 0xea, 0xe0, 0x2,
		0xea, 0xe0,	0x2, 0xea, 0xe0, 0x2, 0xea,	0xe0, 0x2, 0xea, 0xe0, 0x2, 0xea, 0xe0,	0x2, 0xea,
		0xe0, 0x2, 0xea, 0x38, 0x3e, 0xbc, 0xff, 0xba, 0x37, 0xf2, 0x7b, 0x72, 0x15, 0xdc, 0xd,	0x22,
		0x2b, 0x4, 0x14, 0x71, 0x36, 0x38, 0x71, 0xe7, 0x7b, 0x51, 0x23, 0x0, 0xfe,	0x3b, 0xdf,	0xc7,
		0x8d, 0xd4,	0xf0, 0x76,	0x5b, 0xb6,	0x64, 0xd0,	0x72, 0x85,	0xb9, 0xbe,	0xc1, 0x32,	0xf5, 0x1b,
		0x51, 0xfd,	0xe6, 0x75,	0x78, 0x5c,	0x61, 0x1e,	0xba, 0xbd,	0x3b, 0x1a,	0x0, 0x77, 0xcd, 0xcf,
		0x6b, 0xdf, 0x78, 0xbd,	0xd8, 0x4a,	0x7c, 0x95,	0x49, 0x9e,	0xea, 0xb5,	0x3a, 0xd5,	0x6f, 0x9b,
		0x80, 0x39,	0xde, 0xed,	0x71, 0xf5,	0xba, 0xfe,	0x6c, 0xee,	0x4e, 0xba,	0x7a, 0x0, 0xdc, 0xeb,
		0x7f, 0xf6,	0x92, 0x3e,	0x5f, 0xa7,	0x73, 0x7a,	0x6f, 0x39,	0x59, 0xfd,	0xe6, 0x6b,	0x7f, 0xbc,
		0x7a, 0xe3,	0x54, 0x0, 0xdc, 0x15, 0x62, 0xd9, 0x97, 0xcd, 0xff, 0xd0, 0xc1, 0xdb, 0x37, 0x53,
		0xfd, 0xf6,	0x69, 0xf, 0xc,	0xf, 0x80, 0xbd, 0x44, 0xe4, 0x41, 0xb6, 0xa0, 0x2a, 0x1b, 0x14,
		0x7b, 0x41,	0x6f, 0x64,	0xf5, 0x9b,	0xb5, 0x7d,	0xb9, 0xf2,	0xc3, 0xbf,	0xf, 0xd8, 0x3,	0xe0,
		0xae, 0xf9,	0xf1, 0x2c,	0x35, 0xc6,	0x9f, 0xff,	0x44, 0x55,	0xbf, 0x59,	0x9f, 0x1f,	0x50, 0x25,
		0x67, 0xfe,	0xa0, 0x7f,	0x93, 0xc9,	0xf, 0xca, 0x34, 0x83, 0xb7, 0x33, 0xcd, 0x55, 0xfb, 0x9,
		0x56, 0xbf, 0x5, 0xfc, 0xf2, 0xc7, 0xfc, 0xc1, 0x9a, 0xeb, 0xff, 0xf6, 0x5,	0xac, 0x1b,	0x70,
		0x44, 0xf5, 0xdb, 0xf3,	0x38, 0x6a,	0x7a, 0xb7,	0xc7, 0xa9,	0x2e, 0x5f,	0xc2, 0x7e,	0x24, 0x30,
		0xc6, 0x99, 0xef, 0x3f, 0x6f, 0x67,	0x7, 0xb5, 0x90, 0xc2, 0xd9, 0xac, 0xf8, 0x38, 0x0,	0x6c,
		0xf5, 0xdb,	0xb1, 0x3, 0xc0, 0xc9, 0xc,	0xf8, 0x6f, 0x9, 0xae, 0x7e, 0xbe, 0x2,	0xb0, 0xe2,
		0x9e, 0x95,	0x96, 0xff,	0x49, 0xae, 0x0, 0x84, 0x90, 0xf9, 0x36, 0x17, 0x9c, 0xf9, 0x82, 0x1,
		0xf8, 0x4e, 0x70, 0x1, 0x75, 0x70, 0x1,	0x75, 0x70,	0x1, 0x75, 0x70, 0x1, 0x75,	0x70, 0x1,
		0x75, 0x70,	0x1, 0x75, 0x70, 0x1, 0x75,	0x70, 0x1, 0x75, 0x70, 0x1,	0x75, 0x70,	0x1, 0x75,
		0x70, 0x1, 0x75, 0x70, 0x1,	0x75, 0x70,	0x1, 0x75, 0x70, 0x1, 0x75,	0x70, 0x1, 0x75, 0x70,
		0x1, 0x75, 0x70, 0x1, 0x75,	0x70, 0x1, 0x75, 0x70, 0x1,	0x75, 0x70,	0x1, 0x75, 0x70, 0x1,
		0x75, 0x70,	0x1, 0x75, 0x70, 0x1, 0x75,	0x70, 0x1, 0x75, 0x70, 0x1,	0x75, 0x70,	0x1, 0x75,
		0x70, 0x1, 0x75, 0xfe, 0x1,	0x31, 0xbb,	0x32, 0x3a,	0xec, 0x5c,	0x4c, 0x60,	0x0, 0x0, 0x0,
		0x0, 0x49, 0x45, 0x4e, 0x44, 0xae, 0x42, 0x60, 0x82	});

	#region "Constructor"

	public RotatingImage1()
	{
		Resize += RotatingImage_Resize;
		DoubleClick += RotatingImage_DoubleClick;
		SetStyle(ControlStyles.OptimizedDoubleBuffer | ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint | ControlStyles.ResizeRedraw | ControlStyles.ContainerControl | ControlStyles.SupportsTransparentBackColor, true);
		base.DoubleBuffered = true;
		DoubleBuffered = true;
		BackColor = Color.Gainsboro;
		Size = new Size(136, 136);
		Tmr = new Timer { Enabled = false };
		Tmr.Tick += Tmr_Tick;
	}

	protected override void Dispose(bool disposing)
	{
		try
		{
			if (disposing)
			{
				Tmr.Tick -= Tmr_Tick;

				if (PictureStream != null)
					PictureStream.Dispose();
			}
		}
		finally
		{
			base.Dispose(disposing);
		}
	}

	#endregion

	#region "Properties"

	// Initial size of the control
	private int imgSize = 136;
	private Image img = null;
	[Browsable(true), Category("Properties"), Description("Image to show and rotate. If none is selected then the built-in image will be used."), DefaultValue("")]
	public Image RI_Image
	{
		get { return img; }
		set
		{
			if (!ReferenceEquals(img, value))
			{
				img = value;
				if (img != null)
				{
					BackColor = Color.Transparent;
					//* To account for proper image rotation, the size of the control, both width and height, will be equal to
					//* the loaded image's diagonal length measured from top-left corner to bottom-right corner
					imgSize = Convert.ToInt32(Math.Ceiling(Math.Sqrt(Math.Pow(img.Width, 2) + Math.Pow(img.Height, 2))));
				}
				else
				{
					BackColor = Color.Gainsboro;
					//* Built-in image size is 96x96 which gives approximate diagonal length of 136.
					imgSize = 136;
				}
				Size = new Size(imgSize, imgSize);
				Invalidate();
			}
		}
	}

	private float m_Angle;
	[Browsable(true), Category("Properties"), Description("Angle of rotation (valid values -360 to 360)."), DefaultValue(0f)]
	public float RI_RotationAngle
	{
		get { return m_Angle; }
		set
		{
			if (value < -360 || value > 360)
			{
				MessageBox.Show("Invalid value!");
				return;
			}
			if (m_Angle != value)
			{
				m_Angle = value;
				Invalidate();
			}
		}
	}

	public enum Direction
	{
		Clockwise = 0,
		Counterclockwise = 1
	}

	private Direction m_direction = Direction.Clockwise;
	[Browsable(true), Category("Properties"), Description("Direction of perpetual rotation (clockwise or counterclockwise)."), DefaultValue(Direction.Clockwise)]
	public Direction RI_PerpetualRotationDirection
	{
		get { return m_direction; }
		set
		{
			if (m_direction != value)
			{
				m_direction = value;
				Invalidate();
			}
		}
	}

	private bool m_flipV;
	[Browsable(true), Category("Properties"), Description("Flip control vertically."), DefaultValue(false)]
	public bool RI_FlipV
	{
		get { return m_flipV; }
		set
		{
			if (m_flipV != value)
			{
				m_flipV = value;
				Invalidate();
			}
		}
	}

	private bool m_flipH;
	[Browsable(true), Category("Properties"), Description("Flip control horizontally."), DefaultValue(false)]
	public bool RI_FlipH
	{
		get { return m_flipH; }
		set
		{
			if (m_flipH != value)
			{
				m_flipH = value;
				Invalidate();
			}
		}
	}

	private bool m_Value;
	[Browsable(true), Category("Properties"), Description("Enable endless rotation."), DefaultValue(false)]
	public bool Value
	{
		get { return m_Value; }
		set
		{
			if (m_Value != value)
			{
				m_Value = value;

				if (m_Value)
				{
					if (!Tmr.Enabled)
						Tmr.Enabled = true;
				}
				else
					Tmr.Enabled = false;

				Invalidate();
			}
		}
	}

	[Browsable(true), Category("Properties"), Description("Timer interval for endless rotation (valid values 5-10000)."), DefaultValue(100)]
	public int RI_PerpetualTimerInterval
	{
		get { return Tmr.Interval; }
		set
		{
			if (value < 5 || value > 10000)
			{
				MessageBox.Show("Invalid value!");
				return;
			}
			if (Tmr.Interval != value)
			{
				Tmr.Interval = value;
				Invalidate();
			}
		}
	}

	#endregion

	#region "Protected Metods"

	protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
	{
		base.OnPaint(e);

		//* Create a new bitmap to hold the image and allow for its manipulation before it is displayed
		Bitmap backImage;

		if (img != null)
			backImage = new Bitmap(img);
		else
			backImage = new Bitmap(PictureStream);

		//* Rotate the image by using the arbitrary angle set by user.

		//* Move the origin to the center of the control:
		e.Graphics.TranslateTransform(ClientRectangle.Width / 2f, ClientRectangle.Height / 2f);
		//* Rotate the image:
		e.Graphics.RotateTransform(-m_Angle);
		//* Move the origin back to to the upper-left corner of the control:
		e.Graphics.TranslateTransform(-ClientRectangle.Width / 2f, -ClientRectangle.Height / 2f);

		//* Limit painting region so corners do not hide other close controls.
		System.Drawing.Drawing2D.GraphicsPath CircularPath = new System.Drawing.Drawing2D.GraphicsPath();
		CircularPath.AddEllipse(ClientRectangle);
		Region = new Region(CircularPath);

		//* Pass the current image to the FlipHV sub to flip it (if H or V flip is enabled) and display it
		FlipHV(e.Graphics, backImage, m_flipV, m_flipH);

		backImage.Dispose();

		//* Reset the transformation
		e.Graphics.ResetTransform();
	}

	#endregion

	#region "Private Methods"

	private void RotatingImage_DoubleClick(object sender, EventArgs e)
	{
		Value = !Value;
	}

	private void RotatingImage_Resize(object sender, System.EventArgs e)
	{
		Width = Height;
	}

	private void Tmr_Tick(object sender, EventArgs e)
	{
		if (m_Angle == 359 || m_Angle == -359)
			m_Angle = 0;
		else
		{
			if (m_direction == Direction.Clockwise)
				m_Angle -= 1;
			else
				m_Angle += 1;
		}

		Invalidate();
	}

	//* Reference: https://msdn.microsoft.com/en-us/library/3b575a03(v=vs.110).aspx?cs-save-lang=1&cs-lang=vb#code-snippet-1
	private void FlipHV(Graphics g, Bitmap img, bool flipV, bool flipH)
	{
		//* Declare offset variables for proper scaling/positioning of the image
		float ScaleFactor = Width / imgSize;
		int Xoffset = Convert.ToInt32((Width - img.Width * ScaleFactor) / 2);
		int Yoffset = Convert.ToInt32((Height - img.Height * ScaleFactor) / 2);

		//* Original image points:   Upper-Left (Xoffset, Yoffset)
		//*                          Upper-Right (Width - Xoffset, Yoffset)
		//*                          Lower-Left (Xoffset, Height - Yoffset))
		//*
		//* Use points() array to store destination points for the above mentioned points of the original image

		//* No flipping - Destination Points are the same as original
		Point[] points = { new Point(Xoffset, Yoffset),	new Point(Width - Xoffset, Yoffset), new Point(Xoffset, Height - Yoffset) };

		//* Flip image horizontally - Destination Points: (Width - Xoffset, Yoffset); (Xoffset, Yoffset); (Width - Xoffset, Height - Yoffset)
		if (flipH)
			points = new Point[] { new Point(Width - Xoffset, Yoffset),	new Point(Xoffset, Yoffset), new Point(Width - Xoffset, Height - Yoffset) };

		//* Flip image vertically
		if (flipV)
		{
			//* Account for horizontal flip
			if (flipH) //* Destination Points: (Width - Xoffset, Height - Yoffset); (Xoffset, Height - Yoffset); (Width - Xoffset, Yoffset)
				points = new Point[] { new Point(Width - Xoffset, Height - Yoffset), new Point(Xoffset, Height - Yoffset), new Point(Width - Xoffset, Yoffset) };
			else //* Destination Points: (Xoffset, Height - Yoffset); (Width - Xoffset, Height - Yoffset); (Xoffset, Yoffset)
				points = new Point[] { new Point(Xoffset, Height - Yoffset), new Point(Width - Xoffset, Height - Yoffset), new Point(Xoffset, Yoffset) };
		}

		//* Draw image using the resulting points() array
		g.DrawImage(img, points);
	}

	#endregion

}
