/*ForcheckIDE/TreeForm.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "TreeForm.h"
#include "Environment.h"
#include "Info.h"
#include "HelpBrowser.h"

#include <QtWidgets>
#include <QFile>
#include <QMainWindow>

TreeForm::TreeForm(const QString& fileName, QMainWindow *parent) : QDialog(parent)
{
    action = new QAction(this);
    action->setText(fileName);
    action->setCheckable(true);
    connect(action, SIGNAL(triggered()), this, SLOT(show()));
    connect(action, SIGNAL(triggered()), this, SLOT(setFocus()));

    curFileName = fileName;
    myParent = parent;
    setWindowTitle(fileName);
    setAttribute(Qt::WA_DeleteOnClose);
    formCreate();
    setFocusPolicy(Qt::ClickFocus);
    update();
}

QSize TreeForm::sizeHint() const
{
    return QSize((90 * fontMetrics().width('x')),
                 80 * fontMetrics().lineSpacing());
}

void TreeForm::focusInEvent(QFocusEvent * event)
{
    emit(InFocus());
    event->accept();
}

void TreeForm::formCreate()
{
    searchEdit = new QLineEdit();
    searchEdit->setMinimumWidth(8 * fontMetrics().width('x'));
    searchButton = new QPushButton(tr("&Find"));
    searchButton->setToolTip(tr("Locate the specified program unit or procedure"));
    sortButton = new QPushButton(tr("Sort"));
    sortButton->setToolTip(tr("Sort the entries alphabetically"));
    expandAllButton = new QPushButton(tr("&Expand all"));
    expandAllButton->setToolTip(tr("Expand all nodes"));
    collapseAllButton = new QPushButton(tr("&Collapse all"));
    collapseAllButton->setToolTip(tr("Collapse all nodes"));
    helpButton = new QPushButton(tr("&Help"));
    helpButton->setToolTip(tr("Present the relevant pages of the user guide"));
//    helpButton->setShortcut(QKeySequence::HelpContents);

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget(searchEdit);
    buttonLayout->addWidget(searchButton);
    buttonLayout->addWidget(sortButton);
    buttonLayout->addWidget(expandAllButton);
    buttonLayout->addWidget(collapseAllButton);
    buttonLayout->addStretch();
    buttonLayout->addWidget(helpButton);

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addLayout(buttonLayout);
    mainLayout->addWidget(&tree);
    setLayout(mainLayout);

    tree.setColumnCount(1);
    tree.setHeaderLabels(QStringList() << tr("Tree"));
    tree.header()->setSectionResizeMode(QHeaderView::ResizeToContents);
    setGeometry(300,200,600,700);

    connect(sortButton, SIGNAL(clicked()), this, SLOT(sortClicked()));
    connect(searchButton, SIGNAL(clicked()), this, SLOT(searchClicked()));
    connect(expandAllButton, SIGNAL(clicked()), this, SLOT(expandAllClicked()));
    connect(collapseAllButton, SIGNAL(clicked()), this, SLOT(collapseAllClicked()));
    connect(helpButton, SIGNAL(clicked()), this, SLOT(helpClicked()));
}

void TreeForm::update()
{
    QFile file(curFileName);
    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        QMessageBox::warning(this, Info::TITLE,
                             tr("Cannot open file %1:\n%2.")
                             .arg(curFileName)
                             .arg(file.errorString()));
        return;
    }
    myParent->statusBar()->showMessage(tr("Reading file"));
    QApplication::setOverrideCursor(Qt::WaitCursor);
    reader.setDevice(&file);
    tree.clear();
    readFile();
    file.close();
    QApplication::restoreOverrideCursor();
    myParent->statusBar()->showMessage(QString());
}

bool TreeForm::readFile()
{
    reader.readNext();
    while (!reader.atEnd()){
        if (reader.isStartElement()){
            readTree();
        }
        else
            reader.readNext();
    }
    return true;
}

void TreeForm::readTree()
{
    reader.readNext();
    while (!reader.atEnd()){
        if (reader.isEndDocument())
            break;
        if (reader.isEndElement()){
            reader.readNext();
            break;
        }
        if (reader.isStartElement()){
            if (reader.name() == "SUB"){
                readSub(tree.invisibleRootItem());
            }
        }
        else
            reader.readNext();
    }
}

void TreeForm::readSub(QTreeWidgetItem *parent)
{
    QTreeWidgetItem *item = new QTreeWidgetItem(parent);
    item->setExpanded(true);
    QString subid = reader.attributes().value("subid").toString();
    QString subref = reader.attributes().value("subref").toString();
    reader.readNext();
    while (!reader.atEnd()){
        if (reader.isEndElement()){
            reader.readNext();
            break;
        }
        if (reader.isStartElement()){
            if (reader.name() == "SUB"){
                readSub(item);
            } else if (reader.name() == "a"){
                readRef(item, subid, subref);
            }
        }
        else
            reader.readNext();
    }
}

void TreeForm::readRef(QTreeWidgetItem *parent, const QString& subid, const QString& subref)
{
    QString subnam = reader.readElementText();
    QString ref;
    if (!subref.isEmpty()) ref = " > " + subref;
    parent->setText(0, subid + "  " + subnam + ref);
    if (reader.isEndElement())
        reader.readNext();
}

void TreeForm::sortClicked()
{
    tree.sortItems(0,Qt::AscendingOrder);
}

void TreeForm::searchClicked()
{
    QString sortstring = searchEdit->text();
    QTreeWidgetItemIterator it(&tree);
    while (*it) {
        if ((*it)->text(0).contains(sortstring,Qt::CaseInsensitive))
            (*it)->setSelected(true);
        else
            (*it)->setSelected(false);
        ++it;
    }
}

void TreeForm::expandAllClicked()
{
    tree.expandAll();
}

void TreeForm::collapseAllClicked()
{
    tree.collapseAll();
}

void TreeForm::helpClicked()
{
    myEnvironment->helpBrowser->showHelp(HelpBrowser::HelpReferenceStructureInXmlFormat);
}
