/*ForcheckIDE/ProjectFrame.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "ProjectFrame.h"
#include "Highlighter.h"
#include "Info.h"
#include "MainWindow.h"
#include "Project.h"
#include "PageSetup.h"
#include "MakefileForm.h"
#include "SelectMakefileForm.h"
#include "FileViewer.h"
#include "TreeForm.h"
#include "OptionsForm.h"
#include "Constants.h"
#include "LibrarianForm.h"
#include "CodeEditor.h"

#include <QtWidgets>
#include <QString>
#include <QStringList>
#include <QDir>
#include <QFile>
#include <QTextBlock>
#include <QTextStream>

const int ProjectFrame::MAXCOUNT = 20;

ProjectFrame::ProjectFrame(QMainWindow *parent,
                           QMdiArea *mdiarea,
                           QTabWidget *tabwidget,
                           FileViewer *progressview,
                           FileViewer *reportview,
                           FileViewer *searchview) : QTreeWidget(parent) {
    myParent = parent;
    mymdiarea = mdiarea;
    mytabwidget = tabwidget;
    myprogressview = progressview;
    myreportview = reportview;
    mysearchview = searchview;

    project = NULL;
    prevviewer = NULL;
    preveditor = NULL;

    searchInReport = false;
    gen_make = false;
    update_dependencies = false;
    analysing = false;
    building = false;
    fileSelected = false;
    viewableSelected = false;
    editableSelected = false;
    addableSelected = false;
    includableSelected = false;
    excludableSelected = false;
    multipleSelectable = false;
    canHaveFileOptionsSelected = false;
    canAddFile = false;
    printableSelected = false;

    findLabel = new QLabel("Find: ");
    replaceLabel = new QLabel("Replace: ");
    findLabel->setFixedWidth(replaceLabel->minimumSizeHint().width());
    searchComboBox = new QComboBox();
    searchComboBox->setEditable(true);
    searchComboBox->setFixedWidth(30 * fontMetrics().width('x'));
    searchComboBox->setMaxCount(MAXCOUNT);
    connect(searchComboBox->lineEdit(), SIGNAL(returnPressed()), this, SLOT(startSearchClicked()));
    replaceComboBox = new QComboBox();
    replaceComboBox->setEditable(true);
    replaceComboBox->setFixedWidth(30 * fontMetrics().width('x'));
    replaceComboBox->setMaxCount(MAXCOUNT);
    connect(replaceComboBox->lineEdit(), SIGNAL(returnPressed()), this, SLOT(replaceNextClicked()));
    searchString.clear();
    replaceString.clear();
    caseSensitiveSearch = false;
    caseCheckBox = new QCheckBox("Case sensitive", this);
    scopeLabel = new QLabel("  Scope: ");
    selectScopeComboBox = new QComboBox(this);
    selectScopeComboBox->addItem("Current file");
    selectScopeComboBox->addItem("All source and include files");

    createActions();
    createTree(parent);

    connect(progressview, SIGNAL(requestEdit(const QString&, int)),
            this, SLOT(fileEditRequest(const QString&, int)));
    connect(progressview, SIGNAL(InFocus()),
            this, SLOT(changeSubWindowFocus()));
    connect(progressview, SIGNAL(OutFocus()),
            this, SLOT(viewerOutFocus()));
    connect(myreportview, SIGNAL(requestEdit(const QString&, int)),
            this, SLOT(fileEditRequest(const QString&, int)));
    connect(myreportview, SIGNAL(InFocus()),
            this, SLOT(changeSubWindowFocus()));
    connect(myreportview, SIGNAL(OutFocus()),
            this, SLOT(viewerOutFocus()));
    connect(mysearchview, SIGNAL(requestEdit(const QString&, int)),
            this, SLOT(fileEditRequest(const QString&, int)));
    connect(mysearchview, SIGNAL(InFocus()),
            this, SLOT(changeSubWindowFocus()));
    connect(mysearchview, SIGNAL(OutFocus()),
            this, SLOT(viewerOutFocus()));


    setSelectionBehavior(QAbstractItemView::SelectItems);
    setSelectionMode(QAbstractItemView::ExtendedSelection);
    setEditTriggers(QAbstractItemView::NoEditTriggers);

//create popup menu for project files
    popUpMenu = new QMenu(parent);
    popUpMenu->addAction(fileView);
    popUpMenu->addAction(fileEdit);
    popUpMenu->addSeparator();
    popUpMenu->addAction(addFiles);
    popUpMenu->addAction(addDir);
    popUpMenu->addAction(removeFiles);
    popUpMenu->addAction(fileOptionsAction);
    popUpMenu->addSeparator();
    popUpMenu->addAction(includeFiles);
    popUpMenu->addAction(excludeFiles);
    popUpMenu->addSeparator();
    popUpMenu->addAction(filePrint);
}

void ProjectFrame::createActions(){
    //file menu
    fileNew = new QAction(QIcon(":/icons/New-16.png"),tr("&New"), this);
    fileNew->setShortcut(QKeySequence::New);
    fileNew->setStatusTip(tr("Create new text file"));
    connect(fileNew, SIGNAL(triggered()), this, SLOT(fileNewClicked()));

    fileOpen = new QAction(QIcon(":/icons/Open-16.png"),tr("&Open"), this);
    fileOpen->setShortcut(QKeySequence::Open);
    fileOpen->setStatusTip(tr("Open text file"));
    connect(fileOpen, SIGNAL(triggered()), this, SLOT(fileOpenClicked()));

    fileView = new QAction(QIcon(":/icons/Magnify-16.png"),tr("&View"), this);
    fileView->setShortcut(tr("F9"));
    fileView->setStatusTip(tr("View selected file"));
    connect(fileView, SIGNAL(triggered()), this, SLOT(fileViewClicked()));

    fileEdit = new QAction(QIcon(":/icons/Write-16.png"),tr("&Edit"), this);
    fileEdit->setShortcut(tr("F10"));
    fileEdit->setStatusTip(tr("Edit selected file"));
    connect(fileEdit, SIGNAL(triggered()), this, SLOT(fileEditClicked()));

    fileSave = new QAction(QIcon(":/icons/Save-16.png"),tr("&Save"), this);
    fileSave->setShortcut(QKeySequence::Save);
    fileSave->setStatusTip(tr("Save edited file"));
    connect(fileSave, SIGNAL(triggered()), this, SLOT(fileSaveClicked()));

    fileClose = new QAction(QIcon(":/icons/Close-16.png"),tr("&Close"), this);
    fileClose->setShortcut(QKeySequence::Close);
    fileClose->setStatusTip(tr("Close file"));
    connect(fileClose, SIGNAL(triggered()), this, SLOT(fileCloseClicked()));

    fileAbsolutePath = new QAction(tr("&Absolute path"), this);
    fileAbsolutePath->setStatusTip(tr("Display and store the absolute path of filenames "));
    fileAbsolutePath->setCheckable(true);
    fileAbsolutePath->setChecked(false);
    connect(fileAbsolutePath, SIGNAL(triggered()), this, SLOT(fileAbsolutePathClicked()));

    filePrint = new QAction(QIcon(":/icons/Printer-32.png"), tr("&Print"), this);
    filePrint->setShortcut(QKeySequence::Print);
    filePrint->setStatusTip(tr("Print current file"));
    connect(filePrint, SIGNAL(triggered()), this, SLOT(filePrintClicked()));

    //edit menu
    undoAction = new QAction(QIcon(":/icons/Undo-24.png"), tr("&Undo"), this);
    undoAction->setShortcuts(QKeySequence::Undo);
    undoAction->setStatusTip(tr("Undo the previous edit operation"));

    redoAction = new QAction(QIcon(":/icons/Redo-24.png"), tr("&Redo"), this);
    redoAction->setShortcuts(QKeySequence::Redo);
    redoAction->setStatusTip(tr("Redo the previous edit operation"));

    cutAction = new QAction(QIcon(":/icons/Cut-24.png"), tr("Cu&t"), this);
    cutAction->setShortcuts(QKeySequence::Cut);
    cutAction->setStatusTip(tr("Cut the current selection's contents to the "
                            "clipboard"));
    connect(cutAction, SIGNAL(triggered()), this, SLOT(cutClicked()));

    copyAction = new QAction(QIcon(":/icons/Copy-24.png"), tr("&Copy"), this);
    copyAction->setShortcuts(QKeySequence::Copy);
    copyAction->setStatusTip(tr("Copy the current selection's contents to the "
                             "clipboard"));
    connect(copyAction, SIGNAL(triggered()), this, SLOT(copyClicked()));

    pasteAction = new QAction(QIcon(":/icons/Paste-24.png"), tr("&Paste"), this);
    pasteAction->setShortcuts(QKeySequence::Paste);
    pasteAction->setStatusTip(tr("Paste the clipboard's contents into the current "
                              "selection"));

    selectAllAction = new QAction(tr("&Select all"), this);
    selectAllAction->setShortcuts(QKeySequence::SelectAll);
    selectAllAction->setStatusTip(tr("Select all text"));
    connect(selectAllAction, SIGNAL(triggered()), this, SLOT(selectAllClicked()));

    findReplaceAction = new QAction(tr("Find/replace"), this);
    findReplaceAction->setShortcut(QKeySequence::Find);
    findReplaceAction->setStatusTip(tr("Find and replace text in selected file"));
    connect(findReplaceAction, SIGNAL(triggered()), this, SLOT(findReplaceClicked()));

    findPrevious = new QAction(QIcon(":/icons/Arrow-left-22.png"), tr("Find previous"), this);
    findPrevious->setShortcut(QKeySequence::FindPrevious);
    findPrevious->setStatusTip(tr("Find previous occurrence of text"));
    connect(findPrevious, SIGNAL(triggered()), this, SLOT(findPreviousClicked()));

    findNext = new QAction(QIcon(":/icons/Arrow-right-22.png"), tr("Find next"), this);
    findNext->setShortcut(QKeySequence::FindNext);
    findNext->setStatusTip(tr("Find next occurrence of text"));
    connect(findNext, SIGNAL(triggered()), this, SLOT(findNextClicked()));

    replacePrevious = new QAction(QIcon(":/icons/Arrow-left-22.png"), tr("Replace and find previous"), this);
    replacePrevious->setStatusTip(tr("Replace and find previous occurrence of text"));
    connect(replacePrevious, SIGNAL(triggered()), this, SLOT(replacePreviousClicked()));

    replaceNext = new QAction(QIcon(":/icons/Arrow-right-22.png"), tr("Replace and find next"), this);
    replaceNext->setStatusTip(tr("Replace and find next occurrence of text"));
    connect(replaceNext, SIGNAL(triggered()), this, SLOT(replaceNextClicked()));

    replaceAll = new QAction(tr("Replace all"), this);
    replaceAll->setStatusTip(tr("Replace all occurrences of text"));
    connect(replaceAll, SIGNAL(triggered()), this, SLOT(replaceAllClicked()));

    findReplaceInFilesAction = new QAction(tr("Find/replace in files"), this);
    findReplaceInFilesAction->setShortcut(tr("Ctrl+Shift+F"));
    findReplaceInFilesAction->setStatusTip(tr("Find/replace in all source and include files of project"));
    connect(findReplaceInFilesAction, SIGNAL(triggered()), this, SLOT(findReplaceInFilesClicked()));

    startSearch = new QAction(tr("  Search"), this);
    startSearch->setStatusTip(tr("Search in files"));
    connect(startSearch, SIGNAL(triggered()), this, SLOT(startSearchClicked()));

    connect(selectScopeComboBox, SIGNAL(currentIndexChanged(int)),
            this, SLOT(selectScopeChanged(int)));

    gotoAction = new QAction(QIcon(":/icons/Arrow-right-32.png"), tr("&Go to line"), this);
    gotoAction->setShortcut(tr("Ctrl+L"));
    gotoAction->setStatusTip(tr("Move the cursor to the specified line"));
    connect(gotoAction, SIGNAL(triggered()), this, SLOT(gotoClicked()));

    //project menu
    newProjectAction = new QAction(QIcon(":/icons/New-16.png"), tr("&New project"), this);
    newProjectAction->setShortcut(QKeySequence::New);
    newProjectAction->setStatusTip(tr("Create a new project"));
    connect(newProjectAction, SIGNAL(triggered()), this, SLOT(newProjectClicked()));

    openProjectAction = new QAction(QIcon(":/icons/Open-16.png"), tr("&Open project"), this);
    openProjectAction ->setShortcut(QKeySequence::Open);
    openProjectAction ->setStatusTip(tr("Select and open an existing project"));
    connect(openProjectAction, SIGNAL(triggered()), this, SLOT(openProjectClicked()));

    for (int i = 0; i < Environment::MaxRecentFiles; ++i){
        recentFileActions[i] = new QAction(this);
        recentFileActions[i]->setVisible(false);
        connect(recentFileActions[i], SIGNAL(triggered()), this, SLOT(openRecentProjectClicked()));
    }

    saveProjectAction  = new QAction(QIcon(":/icons/Save-16.png"), tr("&Save project"), this);
    saveProjectAction ->setShortcut(QKeySequence::Save);
    saveProjectAction ->setStatusTip(tr("Save the current project"));
    connect(saveProjectAction, SIGNAL(triggered()), this, SLOT(saveProjectClicked()));

    saveAsProjectAction  = new QAction(tr("&Save project as"), this);
    saveAsProjectAction ->setShortcut(QKeySequence::SaveAs);
    saveAsProjectAction ->setStatusTip(tr("Save the current project"));
    connect(saveAsProjectAction, SIGNAL(triggered()), this, SLOT(saveAsProjectClicked()));

    closeProjectAction  = new QAction(QIcon(":/icons/Close-16.png"), tr("&Close project"), this);
    closeProjectAction ->setShortcut(QKeySequence::Close);
    closeProjectAction ->setStatusTip(tr("Close the current project"));
    connect(closeProjectAction, SIGNAL(triggered()), this, SLOT(closeProjectClicked()));

    addFiles = new QAction(QIcon(":/icons/AddFile.png"), tr("&Add file(s)"), this);
    addFiles->setStatusTip(tr("Add a new file of the selected category to the project"));
    connect(addFiles, SIGNAL(triggered()), this, SLOT(addFilesClicked()));

    addDir = new QAction(QIcon(":/icons/AddDir-16.png"), tr("&Add dir"), this);
    addDir->setStatusTip(tr("Add all files of the selected directory recursively to the project"));
    connect(addDir, SIGNAL(triggered()), this, SLOT(addDirClicked()));

    removeFiles = new QAction(QIcon(":/icons/RemoveFile.png"), tr("&Remove file(s)"), this);
    removeFiles->setShortcut(QKeySequence::Delete);
    removeFiles->setStatusTip(tr("Remove the selected files from the project"));
    connect(removeFiles, SIGNAL(triggered()), this, SLOT(removeFilesClicked()));

    updateIncDependencies = new QAction(QIcon(":/icons/RemoveFile-16.png"), tr("&Update include dependencies"), this);
    updateIncDependencies->setStatusTip(tr("Update the include files of the project"));
    connect(updateIncDependencies, SIGNAL(triggered()), this, SLOT(updateIncDependenciesClicked()));

    configureMakefile = new QAction(tr("&Configure and generate makefile"), this);
    configureMakefile->setStatusTip(tr("Configure and generate a makefile to build the project"));
    connect(configureMakefile, SIGNAL(triggered()), this, SLOT(configureMakefileClicked()));

    selectMakefile = new QAction(tr("&Select makefile"), this);
    selectMakefile->setStatusTip(tr("Select a makefile to build the project"));
    connect(selectMakefile, SIGNAL(triggered()), this, SLOT(selectMakefileClicked()));

    executeMakefile = new QAction(QIcon(":/icons/Make-16.png"), tr("&Execute makefile"), this);
    executeMakefile->setShortcut(tr("F7"));
    executeMakefile->setStatusTip(tr("Build the project using the selected makefile"));
    connect(executeMakefile, SIGNAL(triggered()), this, SLOT(executeMakefileClicked()));

    /*analyse menu*/
    includeFiles = new QAction(QIcon(":/icons/ForcheckPlus-16.png"), tr("&Include in analysis"), this);
    includeFiles->setShortcut(tr("Ctrl+I"));
    includeFiles->setStatusTip(tr("Include the selected files in the selective analysis"));
    connect(includeFiles, SIGNAL(triggered()), this, SLOT(includeFilesClicked()));

    excludeFiles = new QAction(QIcon(":/icons/ForcheckMinus-16.png"), tr("&Exclude from analysis"), this);
    excludeFiles->setShortcut(tr("Ctrl+X"));
    excludeFiles->setStatusTip(tr("Exclude the selected files from the selective analysis"));
    connect(excludeFiles, SIGNAL(triggered()), this, SLOT(excludeFilesClicked()));

    analyseSelected = new QAction(QIcon(":/icons/Forcheck-32.png"), tr("&Selective analysis"), this);
    analyseSelected->setShortcut(tr("F5"));
    analyseSelected->setStatusTip(tr("Analyse the files which are included in the selective analysis"));
    connect(analyseSelected, SIGNAL(triggered()), this, SLOT(analyseSelectedClicked()));

    analyseAll = new QAction(QIcon(":/icons/ForcheckAll-32.png"), tr("&Project analysis"), this);
    analyseAll->setShortcut(tr("F6"));
    analyseAll->setStatusTip(tr("Perform a complete analysis of the project"));
    connect(analyseAll, SIGNAL(triggered()), this, SLOT(analyseAllClicked()));

    killAnalysis = new QAction(tr("Abort analysis"), this);
    killAnalysis->setStatusTip(tr("Abort the analysis"));
    connect(killAnalysis, SIGNAL(triggered()), this, SLOT(killAnalysisClicked()));

    findPreviousMsg = new QAction(QIcon(":/icons/ArrowUp-16.png"), tr("&Find previous message in list or report file"), this);
    findPreviousMsg->setShortcut(tr("Alt+Up"));
    findPreviousMsg->setStatusTip(tr("Locate previous message in list or report file"));
    connect(findPreviousMsg, SIGNAL(triggered()), this, SLOT(findPreviousMsgClicked()));

    findNextMsg = new QAction(QIcon(":/icons/ArrowDown-16.png"), tr("&Find next message in list or report file"), this);
    findNextMsg->setShortcut(tr("Alt+Down"));
    findNextMsg->setStatusTip(tr("Locate next message in list or report file"));
    connect(findNextMsg, SIGNAL(triggered()), this, SLOT(findNextMsgClicked()));

    //options menu
    defaultOptionsAction = new QAction(QIcon(":/icons/Options-16.png"), tr("&Default options"), this);
    defaultOptionsAction->setStatusTip(tr("Choose the options that are applied by default"));
    connect(defaultOptionsAction, SIGNAL(triggered()), this, SLOT(defaultOptionsClicked()));

    projectOptionsAction = new QAction(QIcon(":/icons/Options-16.png"), tr("&Project options"), this);
    projectOptionsAction->setShortcut(tr("Shift+Ctrl+O"));
    projectOptionsAction->setStatusTip(tr("Select the options for the project"));
    connect(projectOptionsAction, SIGNAL(triggered()), this, SLOT(projectOptionsClicked()));

    fileOptionsAction = new QAction(QIcon(":/icons/Options-16.png"), tr("&File options"), this);
    fileOptionsAction->setShortcut(tr("Alt+Enter"));
    fileOptionsAction->setStatusTip(tr("Select the options for the file"));
    connect(fileOptionsAction, SIGNAL(triggered()), this, SLOT(fileOptionsClicked()));

    /*view menu*/
    viewSelectiveAnalysisCommandline = new QAction(QIcon(":/icons/Forcheck-32.png"), tr("&View selective analysis commandline"), this);
    connect(viewSelectiveAnalysisCommandline, SIGNAL(triggered()), this, SLOT(viewSelectiveAnalysisCommandlineClicked()));
    viewSelectiveAnalysisCommandFile = new QAction(QIcon(":/icons/Forcheck-32.png"), tr("&View selective analysis commandfile"), this);
    connect(viewSelectiveAnalysisCommandFile, SIGNAL(triggered()), this, SLOT(viewSelectiveAnalysisCommandFileClicked()));
    viewAnalysisCommandline = new QAction(QIcon(":/icons/ForcheckAll-32.png"), tr("&View project analysis commandline"), this);
    viewAnalysisCommandline->setShortcut(tr("Shift+F6"));
    connect(viewAnalysisCommandline, SIGNAL(triggered()), this, SLOT(viewAnalysisCommandlineClicked()));
    viewAnalysisCommandFile = new QAction(QIcon(":/icons/ForcheckAll-32.png"), tr("&View project analysis commandfile"), this);
    connect(viewAnalysisCommandFile, SIGNAL(triggered()), this, SLOT(viewAnalysisCommandFileClicked()));

    connect(this, SIGNAL(itemPressed(QTreeWidgetItem*,int)), this, SLOT(itemPressed(QTreeWidgetItem*,int)));
    connect(this, SIGNAL(itemClicked(QTreeWidgetItem*,int)), this, SLOT(itemClicked(QTreeWidgetItem*,int)));
    connect(this, SIGNAL(itemDoubleClicked(QTreeWidgetItem*,int)), this, SLOT(itemDoubleClicked(QTreeWidgetItem*,int)));
    connect(this, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));

    connect(mytabwidget, SIGNAL(currentChanged(int)), this, SLOT(tabChanged(int)));
}

void ProjectFrame::addFileActions1(QMenu *filemenu)
{
    filemenu->addAction(fileNew);
    filemenu->addAction(fileOpen);
    filemenu->addAction(fileView);
    filemenu->addAction(fileEdit);
    filemenu->addAction(fileSave);
    filemenu->addAction(fileClose);
    filemenu->addSeparator();
    filemenu->addAction(fileAbsolutePath);
}

void ProjectFrame::addFileActions2(QMenu *filemenu)
{
    filemenu->addAction(filePrint);
}

void ProjectFrame::addEditActions(QMenu *editmenu)
{
    editmenu->addAction(undoAction);
    editmenu->addAction(redoAction);
    editmenu->addSeparator();
    editmenu->addAction(cutAction);
    editmenu->addAction(copyAction);
    editmenu->addAction(pasteAction);
    editmenu->addSeparator();
    editmenu->addAction(selectAllAction);
    editmenu->addSeparator();
    editmenu->addAction(findReplaceAction);
    editmenu->addAction(findReplaceInFilesAction);
    editmenu->addAction(gotoAction);
}

void ProjectFrame::addProjectActions(QMenu *projectmenu)
{
    projectmenu->addAction(newProjectAction);
    projectmenu->addAction(openProjectAction);

    recentProjectSubMenu = projectmenu->addMenu(tr("Recent projects"));
    for (int i = 0; i < Environment::MaxRecentFiles; ++i){
        recentProjectSubMenu->addAction(recentFileActions[i]);
    }

    projectmenu->addAction(saveProjectAction);
    projectmenu->addAction(saveAsProjectAction);
    projectmenu->addAction(closeProjectAction);
    projectmenu->addSeparator();
    projectmenu->addAction(addFiles);
    projectmenu->addAction(addDir);
    projectmenu->addAction(removeFiles);
    projectmenu->addAction(updateIncDependencies);
    projectmenu->addSeparator();
    projectmenu->addAction(configureMakefile);
    projectmenu->addAction(selectMakefile);
    projectmenu->addAction(executeMakefile);
}

void ProjectFrame::addAnalysisActions(QMenu *analysismenu)
{
    analysismenu->addAction(includeFiles);
    analysismenu->addAction(excludeFiles);
    analysismenu->addSeparator();
    analysismenu->addAction(analyseSelected);
    analysismenu->addAction(analyseAll);
    analysismenu->addAction(killAnalysis);
    analysismenu->addSeparator();
    analysismenu->addAction(findPreviousMsg);
    analysismenu->addAction(findNextMsg);
}

void ProjectFrame::addOptionActions(QMenu *optionsmenu)
{
    optionsmenu->addAction(defaultOptionsAction);
    optionsmenu->addAction(projectOptionsAction);
    optionsmenu->addAction(fileOptionsAction);
}

void ProjectFrame::addViewActions(QMenu *viewmenu)
{
    viewmenu->addAction(viewSelectiveAnalysisCommandline);
    viewmenu->addAction(viewSelectiveAnalysisCommandFile);
    viewmenu->addAction(viewAnalysisCommandline);
    viewmenu->addAction(viewAnalysisCommandFile);
}

void ProjectFrame::addWindowActions(QMenu *windowmenu)
{
    windowMenu = windowmenu;
    windowActionGroup = new QActionGroup(this);
}

void ProjectFrame::addActionsToToolBar(QToolBar *toolbar)
{
    toolbar->addAction(newProjectAction);
    toolbar->addAction(openProjectAction);
    toolbar->addAction(saveProjectAction);
    toolbar->addAction(saveAsProjectAction);
    toolbar->addAction(closeProjectAction);
    toolbar->addSeparator();
    toolbar->addAction(addFiles);
    toolbar->addAction(addDir);
    toolbar->addAction(removeFiles);
    toolbar->addSeparator();
    toolbar->addAction(includeFiles);
    toolbar->addAction(excludeFiles);
    toolbar->addSeparator();
    toolbar->addAction(analyseSelected);
    toolbar->addAction(analyseAll);
    toolbar->addSeparator();
    toolbar->addAction(findPreviousMsg);
    toolbar->addAction(findNextMsg);
    toolbar->addSeparator();
    toolbar->addAction(fileView);
    toolbar->addAction(fileEdit);
    toolbar->addSeparator();
    toolbar->addAction(undoAction);
    toolbar->addAction(redoAction);
    toolbar->addSeparator();
    toolbar->addAction(cutAction);
    toolbar->addAction(copyAction);
    toolbar->addAction(pasteAction);
    toolbar->addAction(fileSave);
    toolbar->addSeparator();
    toolbar->addAction(executeMakefile);
}

void ProjectFrame::addActionsToSearchToolBar(QToolBar *toolbar)
{
    toolbar->addWidget(findLabel);
    toolbar->addWidget(searchComboBox);
    toolbar->addAction(findPrevious);
    toolbar->addAction(findNext);
    toolbar->addWidget(caseCheckBox);
    toolbar->addWidget(scopeLabel);
    toolbar->addWidget(selectScopeComboBox);
    toolbar->addAction(startSearch);
}

void ProjectFrame::addActionsToReplaceToolBar(QToolBar *toolbar)
{
    toolbar->addWidget(replaceLabel);
    toolbar->addWidget(replaceComboBox);
    toolbar->addAction(replacePrevious);
    toolbar->addAction(replaceNext);
    toolbar->addAction(replaceAll);
}

//========================================================================================

void ProjectFrame::updateEnabled(){
    QList<QTreeWidgetItem *> selected = selectedItems();
    bool projectpresent = project != NULL;

    int nrselected = selected.count();
    int nrfilestoanalyse = project ? project->getNrFilesInAnalysis() : 0;
    int nranfiles = project ? project->getNrFilesAnalysisAll() : 0;
    int nrsrcfiles = project ? project->getNrFiles(FileKind::Source) +
                               project->getNrFiles(FileKind::Include) : 0;

    if (nrselected == 0){
        fileSelected = false;
        viewableSelected = false;
        editableSelected = false;
        includableSelected = false;
        excludableSelected = false;
        canHaveFileOptionsSelected = false;
        printableSelected = false;
    }

    fileView->setEnabled(viewableSelected);
    fileEdit->setEnabled(editableSelected);
    filePrint->setEnabled(printableSelected);
    fileAbsolutePath->setEnabled(projectpresent);
    findReplaceInFilesAction->setEnabled(nrsrcfiles > 0);
    scopeLabel->setEnabled(projectpresent);
    selectScopeComboBox->setEnabled(projectpresent);

    newProjectAction->setEnabled(!projectpresent);
    openProjectAction->setEnabled(!projectpresent);
    recentProjectSubMenu->setEnabled(false);
    updateRecentFileActions();
    if (recentProjectSubMenu){
        recentProjectSubMenu->setEnabled(!projectpresent &&
                                         myEnvironment->recentFiles.count()>0);
    }
    saveProjectAction->setEnabled(projectpresent);
    saveAsProjectAction->setEnabled(projectpresent);
    closeProjectAction->setEnabled(projectpresent);

    addFiles->setEnabled(projectpresent && canAddFile);
    addDir->setEnabled(projectpresent && canAddFile);
    removeFiles->setEnabled(fileSelected);
    updateIncDependencies->setEnabled(nrsrcfiles > 0);
    configureMakefile->setEnabled(nrsrcfiles > 0);
    selectMakefile->setEnabled(projectpresent);
    executeMakefile->setEnabled((nrsrcfiles > 0) && !project->getMakeFilename().isEmpty());

    includeFiles->setEnabled(includableSelected);
    excludeFiles->setEnabled(excludableSelected);
    analyseSelected->setEnabled(nrfilestoanalyse > 0);
    analyseAll->setEnabled(nranfiles > 0);
    killAnalysis->setEnabled(analysing);

    defaultOptionsAction->setEnabled(!projectpresent);;
    projectOptionsAction->setEnabled(projectpresent);
    fileOptionsAction->setEnabled(fileSelected && canHaveFileOptionsSelected);

    viewSelectiveAnalysisCommandline->setEnabled(nrfilestoanalyse > 0);
    viewSelectiveAnalysisCommandFile->setEnabled(nrfilestoanalyse > 0);
    viewAnalysisCommandline->setEnabled(nranfiles > 0);
    viewAnalysisCommandFile->setEnabled(nranfiles > 0);
    }

//========================================================================================

void ProjectFrame::createTree(QWidget *parent){
    setColumnCount(2);
    setHeaderLabels(QStringList() << tr("Files") << tr("Options"));
    header()->setSectionResizeMode(QHeaderView::ResizeToContents);
}

void ProjectFrame::fillTree(){
    emptyTree();
    if (!project) return;
    QTreeWidgetItem *parent = invisibleRootItem();
    projectTreeItem = new QTreeWidgetItem(parent);
    projectTreeItem->setText(0, project->getProjectFilename());
    projectTreeItem->setIcon(0, QIcon(FileKind::getIcon(FileKind::Project)));
    projectTreeItem->setText(1, project->getProjectOptions()->toCommandString());
    ForEachProjectFileKind(fk) addToTree(projectTreeItem, fk);
    projectTreeItem->setExpanded(true);
    setFocus();
}

void ProjectFrame::emptyTree(){
    clear();
}

void ProjectFrame::addToTree(QTreeWidgetItem *parent,
                             const FileKind::ETFileKind filekind){
    if (!FileKind::canBeInProject(filekind)) return;
    QTreeWidgetItem *kinditem = new QTreeWidgetItem(parent);
    KindTreeItem[filekind] = kinditem;
    kinditem->setText(0, FileKind::fileKind2String(filekind));
    kinditem->setIcon(0, QIcon(FileKind::getIcon(filekind)));
//    kinditem->setExpanded(true);
    ProjectFiles *files = project->getProjectFiles(filekind);
    for (int i = 0; i < files->getNrFiles(); i++)
        addFileToTree(kinditem, project->getProjectFile(filekind,i), filekind, i);
    kinditem->sortChildren(0, Qt::AscendingOrder);

}

void ProjectFrame::addFileToTree(QTreeWidgetItem *parent,
                                 const QString filespec,
                                 const FileKind::ETFileKind filekind,
                                 int index){
    if (!FileKind::canBeInProject(filekind)) return;
    QTreeWidgetItem *item = new QTreeWidgetItem(parent);
    if (fileAbsolutePath->isChecked())
        item->setText(0,filespec);
    else{
        QDir cwd(myEnvironment->getWorkingDir());
        QString filename = cwd.relativeFilePath(filespec);
        item->setText(0,filename);
    }
    if (project->getAnalyse(filekind, index))
        item->setIcon(0, QIcon(":/icons/ForcheckPlus-16.png"));
    Options *opts = project->getFileOptions(filekind, index);
    if (opts && (filekind != FileKind::LibOutput))
        item->setText(1,opts->toCommandString());
}

void ProjectFrame::itemPressed(QTreeWidgetItem *item, int column)
{
    if (column == 0){
        Qt::MouseButtons buttons = QApplication::mouseButtons();
        bool rightbutton = buttons & Qt::RightButton;
        itemSelected(item, rightbutton);
        if (buttons.testFlag(Qt::RightButton)){
            popUpMenu->exec(QCursor::pos());
        }
    }
}

void ProjectFrame::itemClicked(QTreeWidgetItem *item, int column)
{
    if (column == 0){
        Qt::MouseButtons buttons = QApplication::mouseButtons();
        bool rightbutton = buttons & Qt::RightButton;
        itemSelected(item, rightbutton);
        if (rightbutton)
            popUpMenu->exec(QCursor::pos());
    }
}

void ProjectFrame::itemDoubleClicked(QTreeWidgetItem *item, int column)
{
    fileViewClicked();
}

void ProjectFrame::itemSelectionChanged()
{
    updateEnabled();
}

void ProjectFrame::itemSelected(QTreeWidgetItem *item, bool rightbutton)
{
    FileKind::ETFileKind filekind = FileKind::string2FileKind(item->text(0));

//only items of the same kind can be selected together
    QList<QTreeWidgetItem *> list = selectedItems();
    for (int i=0; i < list.count(); ++i){
        if (FileKind::string2FileKind(list.at(i)->text(0)) != filekind)
            list.at(i)->setSelected(false);
    }

    fileSelected = false;
    viewableSelected = false;
    editableSelected = false;
    includableSelected = false;
    excludableSelected = false;
    canHaveFileOptionsSelected = false;
    multipleSelectable = FileKind::isMultipleSelectableFiles(filekind);
    canAddFile = multipleSelectable || (project->getNrFiles(filekind) == 0);
    printableSelected = false;
    if (filekind == FileKind::Unknown){
//specific file selected
        int index;
        locateFile(item, filekind, index);
        fileSelected = index >= 0;
        viewableSelected = FileKind::isViewable(filekind);
        editableSelected = FileKind::isEditable(filekind);
        if (index >= 0) excludableSelected = project->getAnalyse(filekind, index);
        includableSelected = FileKind::isIncludable(filekind) && !excludableSelected;
        canHaveFileOptionsSelected = FileKind::canHaveFileOptions(filekind);
        multipleSelectable = false;
        canAddFile = false;
        printableSelected = FileKind::isPrintable(filekind);
    }
    updateEnabled();
}

FileKind::ETFileKind ProjectFrame::getSelectedFileCategory()
{
    QList<QTreeWidgetItem *> selected = selectedItems();
    if (selected.count() != 1){
        QMessageBox::information(this, Info::TITLE,
                                 tr("Select a single category"));
        return FileKind::Unknown;
    }
    return FileKind::string2FileKind(selected[0]->text(0));
}


void ProjectFrame::locateFile(QTreeWidgetItem *item, FileKind::ETFileKind &filekind, int &index)
{
    QTreeWidgetItem *parent = item->parent();
    if (parent){
        filekind = FileKind::string2FileKind(parent->text(0));
        QFileInfo info(item->text(0));
        QString filespec = info.absoluteFilePath();
        index = project->findFile(filespec);
    }
    else {
        filekind = FileKind::Project;
        index = 0;
    }
}

void ProjectFrame::getSelectedFilename(QString& filename, FileKind::ETFileKind& filekind)
{
    filename.clear();
    filekind = FileKind::Unknown;
    QList<QTreeWidgetItem *> selected = selectedItems();
    if (selected.count() <= 0)
        return;
    else if (selected.count() > 1){
        QMessageBox::information(this, Info::TITLE,
                                 tr("Select a single file"));
        return;
    };
    int index;
    locateFile(selected[0], filekind, index);
    if (index >=0) {
        filename = project->getProjectFile(filekind, index);
    }
}

//========================================================================================

void ProjectFrame::getCurrentFileAndKind(QString& filename, FileKind::ETFileKind& filekind){
    filename = QString();
    filekind = FileKind::Unknown;
    QMdiSubWindow *subwindow = mymdiarea->currentSubWindow();
    if (searchInReport){
        filename = myreportview->getFileName();
    }
    else if (subwindow)
        filename = subwindow->windowFilePath();
    else if (project)
        getSelectedFilename(filename, filekind);

    if (!filename.isEmpty()){
        QString ext = FileKind::fileExt(filename);
        filekind = FileKind::ext2FileKind(ext);
    }
}

bool ProjectFrame::fileExists(const QString &filename)
{
    QFile file(filename);
    if (file.exists())
        return true;
    QMessageBox::information(this, Info::TITLE,
                             filename + tr("\nFile does not exists"));
    return false;
}

void ProjectFrame::fileNewClicked()
{
    editFile("", 0);
}

void ProjectFrame::fileOpenClicked()
{
    QString filename = QFileDialog::getOpenFileName(this,
                                                    tr("Open file"),
                                                    QDir::currentPath(),
                                                    tr("all files (*.*)"));
    if (filename.isEmpty())
        return;
    editFile(filename, 0);
}

void ProjectFrame::fileViewClicked()
{
    QString filename;
    FileKind::ETFileKind filekind;
    getSelectedFilename(filename, filekind);
    if (!fileExists(filename)) return;

    if (filekind == FileKind::ReferenceStructure ||
        filekind == FileKind::ModDep){
        if (findSubWindow(filename)) return;
        treeView = new TreeForm(filename, myParent);
        connect(treeView, SIGNAL(InFocus()),
                this, SLOT(changeSubWindowFocus()));
        windowMenu->addAction(treeView->windowMenuAction());
        windowActionGroup->addAction(treeView->windowMenuAction());
        QMdiSubWindow *subwindow = mymdiarea->addSubWindow(treeView);
        subwindow->setWindowFilePath(filename);
        subwindow->show();
    }
    else if (filekind == FileKind::LibOutput ||
             filekind == FileKind::LibRef){
        if (findSubWindow(filename)) return;
        librarianForm = new LibrarianForm(filename, this);
        connect(librarianForm, SIGNAL(InFocus()),
                this, SLOT(changeSubWindowFocus()));
        windowMenu->addAction(librarianForm->windowMenuAction());
        windowActionGroup->addAction(librarianForm->windowMenuAction());
        QMdiSubWindow *subwindow = mymdiarea->addSubWindow(librarianForm);
        subwindow->setWindowFilePath(filename);
        librarianForm->exec();
    }
    else if (filekind == FileKind::Report){
        if (myEnvironment->getEditorAsViewer()){
            fileEditClicked();
        }
        else {
            myreportview->clear();
            myreportview->loadFile(filename);
        }
        myreportview->setFocus();
        mytabwidget->setCurrentIndex(1);
    }
    else if (filekind == FileKind::Listing){
        if (findSubWindow(filename)) return;
        if (myEnvironment->getEditorAsViewer()){
            fileEditClicked();
        }
        else {
            listView = new FileViewer(myParent);
            listView->loadFile(filename);
            addViewer(listView, filename);
        }
    }
    else if (FileKind::isEditable(filekind)){
        fileEditClicked();
    }
    changeSubWindowFocus();
}

CodeEditor *ProjectFrame::activeEditor()
{
    QMdiSubWindow *subwindow = mymdiarea->currentSubWindow();
    if (subwindow)
        return qobject_cast<CodeEditor *>(subwindow->widget());
    return 0;
}

FileViewer *ProjectFrame::activeViewer()
{
    QMdiSubWindow *subwindow = mymdiarea->currentSubWindow();
    if (subwindow)
        return qobject_cast<FileViewer *>(subwindow->widget());
    return 0;
}

QMdiSubWindow *ProjectFrame::findSubWindow(const QString& fileName)
{
    QFileInfo fileinfo1(fileName);
    QList<QMdiSubWindow *> subwindowlist = mymdiarea->subWindowList();
    foreach (QMdiSubWindow *subwindow, subwindowlist){
        QFileInfo fileinfo2(subwindow->windowFilePath());
        if (fileinfo1.absoluteFilePath().compare(fileinfo2.absoluteFilePath(),Qt::CaseInsensitive) == 0){
            mymdiarea->setActiveSubWindow(subwindow);
            return subwindow;
        }
    }
    return 0;
}

void ProjectFrame::tabChanged(int index)
{
    if (index == 0) {
        myprogressview->setFocus();
    }
    else if (index == 1) {
        myreportview->setFocus();
    }
    else {
        mysearchview->setFocus();
    }
    changeSubWindowFocus();
}

void ProjectFrame::changeSubWindowFocus()
{
    bool projectpresent = project != NULL;

    fileSave->setEnabled(false);
    fileClose->setEnabled(false);

    undoAction->setEnabled(false);
    redoAction->setEnabled(false);
    cutAction->setEnabled(false);
    copyAction->setEnabled(false);
    pasteAction->setEnabled(false);
    selectAllAction->setEnabled(false);

    selectScopeComboBox->setCurrentIndex(1);
    gotoAction->setEnabled(false);

    findPreviousMsg->setEnabled(false);
    findNextMsg->setEnabled(false);

    if (prevviewer){
        disconnect(prevviewer, SIGNAL(copyAvailable(bool)),
                   copyAction, SLOT(setEnabled(bool)));
    }
    prevviewer = NULL;
    if (preveditor){
        disconnect(preveditor, SIGNAL(undoAvailable(bool)),
                   undoAction, SLOT(setEnabled(bool)));
        disconnect(undoAction, SIGNAL(triggered()),
                   preveditor, SLOT(undo()));
        disconnect(preveditor, SIGNAL(redoAvailable(bool)),
                   redoAction, SLOT(setEnabled(bool)));
        disconnect(redoAction, SIGNAL(triggered()),
                   preveditor, SLOT(redo()));
        disconnect(preveditor, SIGNAL(copyAvailable(bool)),
                   cutAction, SLOT(setEnabled(bool)));
        disconnect(preveditor, SIGNAL(copyAvailable(bool)),
                   copyAction, SLOT(setEnabled(bool)));
    }
    preveditor = NULL;

    if (!projectpresent) return;

    searchInReport = (mytabwidget->currentWidget()==myreportview) && myreportview->hasFocus();
    QMdiSubWindow *subwindow = mymdiarea->activeSubWindow();
    QString filename;
    FileKind::ETFileKind filekind;
    getCurrentFileAndKind(filename, filekind);
    if (filekind == FileKind::PreSource) filekind = FileKind:: Source;

    if (searchInReport){
        findReplaceAction->setEnabled(true);
        setFindEnabled(true);
        selectScopeComboBox->setCurrentIndex(0);
        findPreviousMsg->setEnabled(true);
        findNextMsg->setEnabled(true);
    }
    else {
        if (subwindow) {
        if (filekind == FileKind::ReferenceStructure){
            qobject_cast<TreeForm *>(subwindow->widget())->windowMenuAction()->setChecked(true);
        }
        else if (filekind == FileKind::LibOutput ||
                 filekind == FileKind::LibRef)
            qobject_cast<LibrarianForm *>(subwindow->widget())->windowMenuAction()->setChecked(true);
        else if (filekind == FileKind::Listing ||
                 filekind == FileKind::Report){
            FileViewer *listview = qobject_cast<FileViewer *>(subwindow->widget());
            if (listview){
                listview->windowMenuAction()->setChecked(true);
                fileClose->setEnabled(true);
                selectAllAction->setEnabled(true);
                findReplaceAction->setEnabled(true);
                setFindEnabled(true);
                setReplaceEnabled(false);
                selectScopeComboBox->setCurrentIndex(0);
                findPreviousMsg->setEnabled(true);
                findNextMsg->setEnabled(true);
                connect(listview, SIGNAL(copyAvailable(bool)),
                        copyAction, SLOT(setEnabled(bool)));
            }
        }
        else {
            CodeEditor *editor = qobject_cast<CodeEditor *>(subwindow->widget());
            if (editor){
                editor->windowMenuAction()->setChecked(true);
                fileSave->setEnabled(true);
                fileClose->setEnabled(true);
                selectAllAction->setEnabled(true);
                findReplaceAction->setEnabled(true);
                setFindEnabled(true);
                setReplaceEnabled(true);
                selectScopeComboBox->setCurrentIndex(0);
                connect(editor, SIGNAL(undoAvailable(bool)),
                        undoAction, SLOT(setEnabled(bool)));
                connect(undoAction, SIGNAL(triggered()),
                        editor, SLOT(undo()));
                connect(editor, SIGNAL(redoAvailable(bool)),
                        redoAction, SLOT(setEnabled(bool)));
                connect(redoAction, SIGNAL(triggered()),
                        editor, SLOT(redo()));
                connect(editor, SIGNAL(copyAvailable(bool)),
                        cutAction, SLOT(setEnabled(bool)));
                connect(editor, SIGNAL(copyAvailable(bool)),
                        copyAction, SLOT(setEnabled(bool)));

                cutAction->setEnabled(editor->textCursor().hasSelection());
                copyAction->setEnabled(editor->textCursor().hasSelection());
                pasteAction->setEnabled(true);
                undoAction->setEnabled(editor->document()->isUndoAvailable());
                redoAction->setEnabled(editor->document()->isRedoAvailable());
                gotoAction->setEnabled(true);
            }
        }
        }
    }

    if (project){
//deselect selections
        QList<QTreeWidgetItem *> list = selectedItems();
        for (int i=0; i < list.count(); ++i){
            list.at(i)->setSelected(false);
        }
//select current
        if (filekind != FileKind::Unknown){
            QTreeWidgetItem* kinditem = KindTreeItem[filekind];
            if (kinditem){
                for (int ind = 0; ind < kinditem->childCount(); ind++ ){
                    QTreeWidgetItem* item = kinditem->child(ind);
                    QString text = item->text(0);
                    QFileInfo info(text);
                    text = info.absoluteFilePath();
                    if (text == filename) item->setSelected(true);
                }
            }
            printableSelected = FileKind::isPrintable(filekind);
            filePrint->setEnabled(printableSelected);
        }
    }
}

//========================================================================================

void ProjectFrame::addViewer(FileViewer *listview, const QString& fileName)
{
    connect(listview, SIGNAL(InFocus()),
            this, SLOT(changeSubWindowFocus()));
    connect(listview, SIGNAL(OutFocus()),
            this, SLOT(viewerOutFocus()));
    windowMenu->addAction(listview->windowMenuAction());
    windowActionGroup->addAction(listview->windowMenuAction());
    QMdiSubWindow *subwindow = mymdiarea->addSubWindow(listview);
    subwindow->setWindowFilePath(fileName);
    subwindow->show();
}

void ProjectFrame::viewerOutFocus()
{
    prevviewer = activeViewer();
}

CodeEditor *ProjectFrame::findEditor(const QString& fileName)
{
    QMdiSubWindow *subwindow = findSubWindow(fileName);
    if (subwindow)
        return qobject_cast<CodeEditor *>(subwindow->widget());
    return 0;
}

void ProjectFrame::fileEditClicked()
{
    QString filename;
    FileKind::ETFileKind filekind;
    getSelectedFilename(filename, filekind);
    if (!fileExists(filename)) return;
    if (myEnvironment->getEditor().isEmpty())
        editFile(filename, 0);
    else
        myEnvironment->execEditor(filename);
}

void ProjectFrame::fileEditRequest(const QString& fileName, int linNum)
{
    editFile(fileName, linNum);
}

void ProjectFrame::editFile(const QString& fileName, int linNum)
{
    CodeEditor *editor = findEditor(fileName);
    if (!editor){
        QString filename = fileName;
        editor = new CodeEditor(myParent);
        QString fileext = FileKind::fileExt(filename);
        FileKind::ETFileKind filekind = FileKind::ext2FileKind(fileext);
        if (filekind == FileKind::PreSource) filekind = FileKind::Source;
        Highlighter *highlighter;
        if (FileKind::isHighlightable(filekind)) {
            bool isff = FileKind::isDefaultFfFile(filename);
            int index = project->findFile(filename);
            if (index >= 0 && FileKind::canBeInProject(filekind)) {
                Options *opts = project->getFileOptions(filekind, index);
                if (opts) isff = opts->getBoolEffective(optFF);
            }
            highlighter = new Highlighter(editor->document(), isff);
        }
        if (filename.isEmpty())
            filename = editor->newFileName();
        else
            if (!editor->loadFile(filename)){
                if (FileKind::isHighlightable(filekind))
                  delete highlighter;
                delete editor;
                return;
            }
        addEditor(editor, filename);
    }
    if (linNum > 0)
        editor->gotoLine(linNum);
}

void ProjectFrame::addEditor(CodeEditor *editor, const QString& fileName)
{
    connect(editor, SIGNAL(fileSaved(const QString &)),
            this, SLOT(fileSaved(const QString &)));
    connect(editor, SIGNAL(closed()),
            this, SLOT(closeEdit()));
    connect(editor, SIGNAL(InFocus()),
            this, SLOT(changeSubWindowFocus()));
    connect(editor, SIGNAL(OutFocus()),
            this, SLOT(editorOutFocus()));

    windowMenu->addAction(editor->windowMenuAction());
    windowActionGroup->addAction(editor->windowMenuAction());
    QMdiSubWindow *subwindow = mymdiarea->addSubWindow(editor);
    subwindow->setWindowFilePath(fileName);
    subwindow->show();
}

void ProjectFrame::editorOutFocus()
{
    preveditor = activeEditor();
}

void ProjectFrame::closeEdit()
{
    if (activeEditor()) {
        activeEditor()->close();
        updateEnabled();
    }
}

void ProjectFrame::cutClicked()
{
    if (activeEditor())
        activeEditor()->cut();
}

void ProjectFrame::copyClicked()
{
    if (activeViewer())
        activeViewer()->copy();
    if (activeEditor())
        activeEditor()->copy();
}

void ProjectFrame::pasteClicked()
{
    if (activeEditor())
        activeEditor()->paste();
}

void ProjectFrame::selectAllClicked()
{
    if (activeViewer())
        activeViewer()->selectAll();
    if (activeEditor())
        activeEditor()->selectAll();
}

void ProjectFrame::findReplaceClicked()
{
    QString text;
    if ((mytabwidget->currentWidget()==myreportview) && myreportview->hasFocus()){
        setFindEnabled(true);
        setReplaceEnabled(false);
        text = myreportview->textCursor().selectedText();
        searchInReport = true;
    }
    else if (activeViewer()){
        setFindEnabled(true);
        setReplaceEnabled(false);
        text = activeViewer()->textCursor().selectedText();
        searchInReport = false;
    }
    else if (activeEditor()){
        setFindEnabled(true);
        setReplaceEnabled(true);
        text = activeEditor()->textCursor().selectedText();
        searchInReport = false;
    }
    searchComboBox->setEditText(text);
    updateSearchComboBox();
    searchComboBox->setFocus();
}

void ProjectFrame::findPreviousClicked()
{
    caseSensitiveSearch = caseCheckBox->isChecked();
    searchString = searchComboBox->currentText();
    updateSearchComboBox();
if (searchInReport && myreportview){
        if (caseSensitiveSearch)
            myreportview->find(searchString, QTextDocument::FindCaseSensitively | QTextDocument::FindBackward);
        else
            myreportview->find(searchString, QTextDocument::FindBackward);
    }
    else if (activeViewer()){
        if (caseSensitiveSearch)
            activeViewer()->find(searchString, QTextDocument::FindCaseSensitively | QTextDocument::FindBackward);
        else
            activeViewer()->find(searchString, QTextDocument::FindBackward);
    }
    else if (activeEditor()){
        if (caseSensitiveSearch)
            activeEditor()->find(searchString, QTextDocument::FindCaseSensitively | QTextDocument::FindBackward);
        else
            activeEditor()->find(searchString, QTextDocument::FindBackward);
    }
}

void ProjectFrame::findNextClicked()
{
    caseSensitiveSearch = caseCheckBox->isChecked();
    searchString = searchComboBox->currentText();
    updateSearchComboBox();
    if (searchInReport && myreportview){
        if (caseSensitiveSearch)
            myreportview->find(searchString, QTextDocument::FindCaseSensitively);
        else
            myreportview->find(searchString);
    }
    else if (activeViewer()){
        if (caseSensitiveSearch)
            activeViewer()->find(searchString, QTextDocument::FindCaseSensitively);
        else
            activeViewer()->find(searchString);
    }
    else if (activeEditor()){
        if (caseSensitiveSearch)
            activeEditor()->find(searchString, QTextDocument::FindCaseSensitively);
        else
            activeEditor()->find(searchString);
    }
}

void ProjectFrame::replacePreviousClicked()
{
    if (activeEditor()){
        searchString = searchComboBox->currentText();
        updateSearchComboBox();
        caseSensitiveSearch = caseCheckBox->isChecked();
        replaceString = replaceComboBox->currentText();
        updateReplaceComboBox();
        activeEditor()->textCursor().clearSelection();
        activeEditor()->textCursor().insertText(replaceString);
        if (caseSensitiveSearch)
            activeEditor()->find(searchString, QTextDocument::FindCaseSensitively | QTextDocument::FindBackward);
        else
            activeEditor()->find(searchString, QTextDocument::FindBackward);
    }
}

void ProjectFrame::replaceNextClicked()
{
    if (activeEditor()){
        searchString = searchComboBox->currentText();
        updateSearchComboBox();
        caseSensitiveSearch = caseCheckBox->isChecked();
        replaceString = replaceComboBox->currentText();
        updateReplaceComboBox();
        activeEditor()->textCursor().clearSelection();
        activeEditor()->textCursor().insertText(replaceString);
        if (caseSensitiveSearch)
            activeEditor()->find(searchString, QTextDocument::FindCaseSensitively);
        else
            activeEditor()->find(searchString);
    }
}

void ProjectFrame::findReplaceInFilesClicked()
{
    setFindEnabled(true);
    setReplaceEnabled(true);
    selectScopeComboBox->setCurrentIndex(1);
    selectScopeComboBox->setFocus();
}

void ProjectFrame::selectScopeChanged(int index)
{
    bool projectpresent = project != NULL;
    switch(index){
    case 0:
        setFindEnabled(projectpresent);
        setReplaceEnabled(activeEditor());
        break;
    case 1:
        setFindEnabled(projectpresent);
        setReplaceEnabled(projectpresent);
        break;
    }
}

void ProjectFrame::updateSearchComboBox()
{
    if ((searchComboBox->count() < MAXCOUNT) &&
        !searchComboBox->currentText().isEmpty() &&
        searchComboBox->findText(searchComboBox->currentText()) == -1)
        searchComboBox->insertItem(-1,searchComboBox->currentText());
}

void ProjectFrame::updateReplaceComboBox()
{
    if ((replaceComboBox->count() < MAXCOUNT) &&
        !replaceComboBox->currentText().isEmpty() &&
        replaceComboBox->findText(replaceComboBox->currentText()) == -1)
        replaceComboBox->insertItem(-1,replaceComboBox->currentText());
}

void ProjectFrame::startSearchClicked()
{
    if (selectScopeComboBox->currentIndex() == 0)
        findNextClicked();
    else
        doFindReplaceInFiles(false);
}

void ProjectFrame::replaceAllClicked()
{
    if (selectScopeComboBox->currentIndex() == 0)
        doReplaceInWindow();
    else
        doFindReplaceInFiles(true);
}

void ProjectFrame::startReplaceInFilesClicked()
{
    doFindReplaceInFiles(true);
}

void ProjectFrame::doReplaceInWindow()
{
    if (activeEditor()){
        searchString = searchComboBox->currentText();
        updateSearchComboBox();
        caseSensitiveSearch = caseCheckBox->isChecked();
        replaceString = replaceComboBox->currentText();
        updateReplaceComboBox();
        activeEditor()->moveCursor(QTextCursor::Start);
        bool found = false;
        do {
            if (activeEditor()->textCursor().hasSelection()){
                activeEditor()->textCursor().clearSelection();
                activeEditor()->textCursor().insertText(replaceString);
            }
            if (caseSensitiveSearch)
                found = activeEditor()->find(searchString, QTextDocument::FindCaseSensitively);
            else
                found = activeEditor()->find(searchString);
        } while(found);
    }
}

void ProjectFrame::doFindReplaceInFiles(bool replace)
    {
    if (!project) return;
    if (isModified()) {
        if (myEnvironment->getAutosaveFilesSearch())
            saveAll();
        else
        if (QMessageBox::question(this, Info::TITLE,
                                  tr("Some files have been modified. \nSave files before searching?"),
                                  QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)
            saveAll();
    }
    searchString = searchComboBox->currentText();
    updateSearchComboBox();
    caseSensitiveSearch = caseCheckBox->isChecked();
    replaceString = replaceComboBox->currentText();
    updateReplaceComboBox();
    int nrocc = 0;
    int nrfiles = 0;
    int filenum = -1;
    mysearchview->clear();
    mytabwidget->setCurrentWidget(mysearchview);
    changeSubWindowFocus();
    connect(mysearchview, SIGNAL(requestEdit(const QString&, int)),
            this, SLOT(fileEditRequest(const QString&, int)));
    FileViewer *view = new FileViewer(myParent);
    ForEachProjectFileKind(fk){
        if (FileKind::isEditable(fk)){
            ProjectFiles *files = project->getProjectFiles(fk);
            for (int i = 0; i < files->getNrFiles(); i++){
                QString filename = project->getFilename(fk, i);
                bool modified = false;
                if (replace) {
//replace in active editor
                    QMdiSubWindow *window = findSubWindow(filename);
                    if (window){
                        window->setFocus();
                        doReplaceInWindow();
                        if (activeEditor()){
                            modified = activeEditor()->document()->isModified();
                           if (modified)
                               activeEditor()->saveFile();
                        }
                    }
                }
//find/replace in file
                if (!modified){
                    if (!view->loadFile(filename)) return;
                    bool found = false;
                    do {
                        if (replace)
                        {
                            if (view->textCursor().hasSelection()){
                                view->textCursor().clearSelection();
                                view->textCursor().insertText(replaceString);
                            }
                        }
                        if (caseSensitiveSearch)
                            found = view->find(searchString, QTextDocument::FindCaseSensitively);
                        else
                            found = view->find(searchString);
                        if (found){
                            int linnum = view->textCursor().blockNumber() + 1;
                            mysearchview->appendText(tr("(file: %1, line: %2)")
                                                   .arg(filename)
                                                   .arg(linnum));
                            mysearchview->appendText(view->textCursor().block().text());
                            nrocc = nrocc + 1;
                            if (i != filenum) nrfiles = nrfiles + 1;
                            filenum = i;
                        }
                    } while(found);
                    if (view->document()->isModified()){
                        view->saveFile();

//                     mysearchview->appendText(tr("file saved."));
                    }
                }
            }
        }
    }
    mysearchview->appendText(tr("\n%1 occurrences in %2 file(s).")
                             .arg(nrocc)
                             .arg(nrfiles));
    view->close();
}

void ProjectFrame::setFindEnabled(bool enable)
{
    findLabel->setEnabled(enable);
    searchComboBox->setEnabled(enable);
    findPrevious->setEnabled(enable);
    findNext->setEnabled(enable);
    caseCheckBox->setEnabled(enable);
    startSearch->setEnabled(enable);
}

void ProjectFrame::setReplaceEnabled(bool enable)
{
    replaceLabel->setEnabled(enable);
    replaceComboBox->setEnabled(enable);
    replacePrevious->setEnabled(enable);
    replaceNext->setEnabled(enable);
    replaceAll->setEnabled(enable);
}

void ProjectFrame::gotoClicked()
{
    if (activeEditor()){
        bool ok;
        int linnum = QInputDialog::getInt(this, Info::TITLE,
                                    tr("Enter line number"),
                                    1, 1, activeEditor()->blockCount(), 1, &ok);
        if (ok)
            activeEditor()->gotoLine(linnum);
    }
}

void ProjectFrame::fileSaved(const QString &filename)
{
    if (!project) return;
    FileKind::ETFileKind filekind = FileKind::ext2FileKind(FileKind::fileExt(filename));
    if (!FileKind::isEditable(filekind)) return;
    if (project->findFile(filename) >= 0) return;
    if (QMessageBox::question(this, Info::TITLE,
                              filename +
                              tr("\nAdd file to project?"),
                              QMessageBox::Yes,
                              QMessageBox::No) == QMessageBox::Yes)
    {
        if (filekind == FileKind::Unknown) filekind = FileKind::Source;
        int index = project->addProjectFile(filekind, filename);
        addFileToTree(KindTreeItem[filekind],
                      filename,
                      filekind,
                      index);
    }
}

void ProjectFrame::fileSaveClicked()
{
    if (activeEditor())
        activeEditor()->saveFile();
}

void ProjectFrame::fileCloseClicked()
{
    QMdiSubWindow *subwindow = mymdiarea->activeSubWindow();
    if (subwindow) subwindow->close();
}

void ProjectFrame::closeAllFiles()
{
    QList<QMdiSubWindow *> subwindowlist = mymdiarea->subWindowList();
    foreach (QMdiSubWindow *subwindow, subwindowlist){
        QString filename = subwindow->windowFilePath();
        FileKind::ETFileKind filekind = FileKind::None;
        if (project) filekind = project->getFileKind(filename);
        if (FileKind::isEditable(filekind)){
            CodeEditor *editor = qobject_cast<CodeEditor *>(subwindow->widget());
            if (editor) editor->closeFile();
        }
    }
}

bool ProjectFrame::isModified()
{
    QList<QMdiSubWindow *> subwindowlist = mymdiarea->subWindowList();
    foreach (QMdiSubWindow *subwindow, subwindowlist){
        QString filename = subwindow->windowFilePath();
        FileKind::ETFileKind filekind = FileKind::None;
        if (project) filekind = project->getFileKind(filename);
        if (FileKind::isEditable(filekind)){
            CodeEditor *editor = qobject_cast<CodeEditor *>(subwindow->widget());
            if (editor) {
                if (editor->document()->isModified()) return true;
            }
        }
    }
    return false;
}

void ProjectFrame::saveAll()
{
    QList<QMdiSubWindow *> subwindowlist = mymdiarea->subWindowList();
    foreach (QMdiSubWindow *subwindow, subwindowlist){
        QString filename = subwindow->windowFilePath();
        FileKind::ETFileKind filekind = FileKind::None;
        if (project) filekind = project->getFileKind(filename);
        if (FileKind::isEditable(filekind)){
            CodeEditor *editor = qobject_cast<CodeEditor *>(subwindow->widget());
            if (editor) {
                editor->saveFile();
            }
        }
    }
}

void ProjectFrame::verifyProject()
{
    ForEachProjectFileKind(fk){
        if (FileKind::isMultipleSelectableFiles(fk)){
            ProjectFiles *files = project->getProjectFiles(fk);
            for (int i = 0; i < files->getNrFiles(); i++){
                QString name = project->getProjectFile(fk,i);
                QFile file(name);
                if (!file.exists()) {
                    int ret = QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                             name + tr("\nFile does not exist. Remove from project?"),
                                             QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
                    if (ret == QMessageBox::Cancel){
                        return;
                    }
                    if (ret == QMessageBox::Yes){
                        project->removeProjectFile(fk,i);
                        fillTree();
                    }
                }
            }
        }
    }
}

void ProjectFrame::fileAbsolutePathClicked()
{
    fillTree();
}

void ProjectFrame::filePrintClicked()
{
    QString filename;
    FileKind::ETFileKind filekind;
    getCurrentFileAndKind(filename, filekind);

    if (project && filekind==FileKind::Listing) {
        if ((project->LastUsedInAnalysis.PW > 0 &&
             project->getProjectOptions()->getIntEffective(optPW) != project->LastUsedInAnalysis.PW) ||
            (project->LastUsedInAnalysis.PL > 0 &&
             project->getProjectOptions()->getIntEffective(optPL) != project->LastUsedInAnalysis.PL)) {
            if (QMessageBox::question(this, Info::TITLE,
                                      tr("Page setup has been changed since file has been created.\n"
                                         "Run the analysis again to update layout.\n"
                                         "Continue printing anyway?")) == QMessageBox::Ok)
                return;
            }
    }
    QFile file(filename);
    if (!file.open(QIODevice::ReadOnly | QFile::Text)){
        QMessageBox::information(this, Info::TITLE,
                                 tr("Could not open '%1'").arg(filename));
        return;
    }
    myEnvironment->printFile(file);
    file.close();
}

void ProjectFrame::newProjectClicked()
{
    createProject();
    saveAsProjectClicked();
    QString filename = project->getProjectFilename();
    if (filename.isEmpty()){
        closeProjectClicked();
        return;
    }
    QFileInfo info(filename);
    myEnvironment->setWorkingDir(info.absolutePath());
    QString projectpath = info.absolutePath() + "/" + info.baseName();

    QString listfilename = projectpath + myEnvironment->getListingExt();
    QString reportfilename = projectpath + myEnvironment->getReportExt();
    QString refstructfilename = projectpath + "_refstruct" + myEnvironment->getRefstructExt();
    QString moddepfilename = projectpath + "_moddep" + myEnvironment->getModDepExt();

    project->addProjectFile(FileKind::Listing, listfilename);
    project->addProjectFile(FileKind::Report, reportfilename);
    project->addProjectFile(FileKind::ReferenceStructure, refstructfilename);
    project->addProjectFile(FileKind::ModDep, moddepfilename);

    project->save();
    fillTree();
    updateEnabled();
}

void ProjectFrame::openProjectClicked()
{
    QString projectfile = QFileDialog::getOpenFileName(0,
                                                       tr("Open Project File"),
                                                       QDir::currentPath(),
                                                       tr("Forcheck Project Files (*") +
                                                          myEnvironment->getProjectExt() +
                                                          ")");
    openProject(projectfile);
    updateEnabled();
}

void ProjectFrame::openRecentProjectClicked()
{
    QAction *action = qobject_cast<QAction *>(sender());
    if (action){
        QString projectfile = action->text();
        openProject(projectfile);
        updateEnabled();
    }
}

void ProjectFrame::removeRecentProjectClicked()
{
    QAction *action = qobject_cast<QAction *>(sender());
    if (action){
        QString projectfile = action->text();
    }
}

void ProjectFrame::saveProjectClicked()
{
    if (!project) return;
    verifyProject();
    if (!project->getProjectFilename().isEmpty())
        project->save();
    else
        saveAsProjectClicked();
}

void ProjectFrame::saveAsProjectClicked()
{
    if (!project) return;
    verifyProject();
    QString filename = project->getProjectFilename();
    QString filetypes = tr("Forcheck Project Files (*") +
                        myEnvironment->getProjectExt() +
                        ")";
    QString header;
    if (filename.isEmpty()) {
        header = tr("New project file");
        filename = myEnvironment->getFckUserDataDir();
    }
    else
        header = tr("Save project file");
    QString projectfile = QFileDialog::getSaveFileName(0,
                                                       header,
                                                       filename,
                                                       filetypes);
    project->saveAs(projectfile);
    fillTree();
    updateTitle(projectfile);
}

void ProjectFrame::closeProjectClicked()
{
    closeProject();
}

void ProjectFrame::closeProject()
{
    if (!project) return;
    closeAllFiles();
    mymdiarea->closeAllSubWindows();
    if (project->isChanged()) {
        if (myEnvironment->getAutosaveProject()) {
            verifyProject();
            project->save();
        }
        else {
            QMessageBox::StandardButton ret;
            ret = QMessageBox::question(this, Info::TITLE,
                                        tr("The project has been modified.\n"
                                           "Save changes?"),
                                        QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
            if (ret == QMessageBox::Save){
                verifyProject();
                project->save();
            }
            else if (ret == QMessageBox::Cancel)
                return;
        }
    }
    emptyTree();
    delete project;
    project = NULL;
    updateTitle();
    updateEnabled();
    findReplaceAction->setEnabled(false);
    setFindEnabled(false);
    setReplaceEnabled(false);
    myprogressview->clear();
    myreportview->clear();
    mysearchview->clear();
}

bool ProjectFrame::projectChanged()
{
    if (project)
        return project->isChanged();
    else
        return false;
}

void ProjectFrame::createProject()
{
    if (project)
        closeProjectClicked();
    if (project)
        return;
    project = new Project();
}

void ProjectFrame::openProject(const QString &projectfilename)
{
    if (projectfilename.isEmpty())
        return;
    myParent->statusBar()->showMessage(tr("Loading project"));
    createProject();
    if (!project->load(projectfilename)) {
        closeProjectClicked();
        myParent->statusBar()->showMessage(QString());
        return;
    };
    updateEnabled();
    fillTree();
    updateTitle(projectfilename);
    findReplaceAction->setEnabled(true);
    setFindEnabled(true);
    setReplaceEnabled(true);
    myParent->statusBar()->showMessage(QString());
    if (myEnvironment->getVerProjectFiles()) verifyProject();
    if (myEnvironment->getVerIncludeDep()) updateIncDependenciesClicked();
}

void ProjectFrame::saveProject(){
    project->save();
}

void ProjectFrame::updateRecentFileActions()
{
    QMutableStringListIterator i(myEnvironment->recentFiles);
    while (i.hasNext()){
        if (!QFile::exists(i.next()))
            i.remove();
    }
    for (int j = 0; j < Environment::MaxRecentFiles; ++j){
        if (j < myEnvironment->recentFiles.count()){
            recentFileActions[j]->setText(myEnvironment->recentFiles[j]);
            recentFileActions[j]->setData(myEnvironment->recentFiles[j]);
            recentFileActions[j]->setVisible(true);
        }
        else
            recentFileActions[j]->setVisible(false);
    }
}

void ProjectFrame::updateTitle(QString projectname)
{
    if (!projectname.isEmpty()){
        myEnvironment->recentFiles.removeAll(projectname);
        myEnvironment->recentFiles.prepend(projectname);
        updateRecentFileActions();
    }
    myParent->setWindowTitle(QString(Info::TITLE) +
                             QString(Info::VERSION) +
                             "  " +
                             projectname);
}

void ProjectFrame::addFilesClicked()
{
    QList<QTreeWidgetItem *> selected = selectedItems();
    FileKind::ETFileKind filekind = getSelectedFileCategory();
    if (filekind == FileKind::Unknown)
        return;
    QStringList filenames = openFileDialog(filekind);
    QStringList::Iterator it = filenames.begin();
    int n = 0;
    while(it != filenames.end()) {
        int index = project->addProjectFile(filekind, *it);
        if (index >=0)
            addFileToTree(selected[0],
                          *it,
                          filekind,
                          index);
        ++n;
        ++it;
    }
    if (n > 0) updateEnabled();
}

void ProjectFrame::addDirClicked()
{
    QList<QTreeWidgetItem *> selected = selectedItems();
    FileKind::ETFileKind filekind = getSelectedFileCategory();
    if (filekind == FileKind::Unknown)
        return;
    QString dirname = openDirDialog(filekind);

    addFilesFromDir(dirname);
    updateEnabled();
}

void ProjectFrame::addFilesFromDir(const QString &path)
{
    QList<QTreeWidgetItem *> selected = selectedItems();
    FileKind::ETFileKind filekind = getSelectedFileCategory();
    QDir dir(path);
    QStringList filenames = dir.entryList(QDir::Files);
    QStringList::Iterator it = filenames.begin();
    QString fileext;
    QString filespec;


    while(it != filenames.end()) {
        fileext = FileKind::fileExt(*it);
        if (FileKind::ext2FileKind(fileext) == filekind) {
            filespec = path + QDir::separator() + *it;
            int index = project->addProjectFile(filekind, filespec);
            if (index >=0)
                addFileToTree(selected[0],
                              filespec,
                              filekind,
                              index);
        }
        ++it;
    }
    foreach (QString subDir, dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot))
        addFilesFromDir(path + QDir::separator() + subDir);
}

QString ProjectFrame::getFilter(const FileKind::ETFileKind filekind)
{
    QString filter;
    switch (filekind){
    case FileKind::Source:
        filter = FileKind::fileKind2String(filekind) + tr(" files") +
                 " (*" + myEnvironment->getExtension(filekind) + ");;" +
                 tr("all fortran source files (*.f *.for *.ftn *.f90 *.f95 *.fpp *.F *.F90 *.F95);;") +
                 tr("all files (*.*)");
        break;
    case FileKind::CSource:
        filter = FileKind::fileKind2String(filekind) + tr(" files") +
                 " (*" + myEnvironment->getExtension(filekind) + ");;" +
                 tr("all C source files (*.c *.cpp);;") +
                 tr("all files (*.*)");
        break;
    case FileKind::Include:
        filter = FileKind::fileKind2String(filekind) + tr(" files") +
                 " (*" + myEnvironment->getExtension(filekind) + ");;" +
                 tr("all files (*.*)");
        break;
    default:
        filter = FileKind::fileKind2String(filekind) + tr(" files") +
                 " (*" + myEnvironment->getExtension(filekind) + ");;" +
                 tr("all files (*.*)");
        break;
    }
    return filter;
}

QStringList ProjectFrame::openFileDialog(const FileKind::ETFileKind filekind)
{
    QString filter = getFilter(filekind);
    QString directory = QString();
    QStringList filenames;

    if (FileKind::isOutputFile(filekind)){
        QString filename = QFileDialog::getSaveFileName(this,
                                                        tr("Enter ") + FileKind::fileKind2String(filekind) + tr(" file"),
                                                        directory,
                                                        filter);
        if (!filename.isEmpty()) filenames.append(filename);
    }
    else
        filenames = QFileDialog::getOpenFileNames(this,
                                                  tr("Select ") + FileKind::fileKind2String(filekind) + tr(" files"),
                                                  directory,
                                                  filter);
    return filenames;
}

QString ProjectFrame::openDirDialog(const FileKind::ETFileKind filekind)
{
    QString directory = QString();
    QString dirname = QFileDialog::getExistingDirectory(this,
                                                        tr("Enter all ") + FileKind::fileKind2String(filekind) + tr(" files recursively from directory"),
                                                        directory,
                                                        QFileDialog::ShowDirsOnly
                                                        | QFileDialog::DontResolveSymlinks);
    return dirname;
}

void ProjectFrame::removeFilesClicked()
{
    QList<QTreeWidgetItem *> selected = selectedItems();
    for (int i = 0; i < selected.count(); i++) {
        QTreeWidgetItem *item = selected[i];
        FileKind::ETFileKind filekind;
        int index;
        locateFile(item, filekind, index);
        if (filekind != FileKind::Project && index >=0) {
            project->removeProjectFile(filekind, index);
            QTreeWidgetItem *parent = item->parent();
            parent->removeChild(item);
        }
    }
    updateEnabled();
}

void ProjectFrame::includeFilesClicked()
{
    QList<QTreeWidgetItem *> selected = selectedItems();
    for (int i = 0; i < selected.count(); i++) {
        setIncluded(selected[i], true);
    }
    updateEnabled();
}

void ProjectFrame::excludeFilesClicked()
{
    QList<QTreeWidgetItem *> selected = selectedItems();
    for (int i = 0; i < selected.count(); i++) {
        setIncluded(selected[i], false);
    }
    updateEnabled();
}

void ProjectFrame::setIncluded(QTreeWidgetItem *item, bool analyse)
{
    FileKind::ETFileKind filekind;
    int index;
    locateFile(item, filekind, index);
    if (index >= 0 && FileKind::isIncludable(filekind)){
        project->setAnalyse(filekind, index, analyse);
        if (analyse)
            item->setIcon(0, QIcon(":/icons/ForcheckPlus-16.png"));
        else
            item->setIcon(0, QIcon(QString()));
    }
    item->setSelected(false);
}

void ProjectFrame::updateIncDependenciesClicked()
{
    Version FCKversion = myEnvironment->getFCKversion();
    if (FCKversion.isKnown() && FCKversion.minor < 2){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Search for include dependencies is only available from Forcheck V14.2."),
                             QMessageBox::Ok);
        return;
    }
    gen_make = false;
    update_dependencies = true;
    analyse(false, true, false);
}

void ProjectFrame::analyseSelectedClicked()
{
    gen_make = false;
    update_dependencies = false;
    analyse(false, false, true);
}

void ProjectFrame::analyseAllClicked()
{
    gen_make = false;
    update_dependencies = false;
    analyse(false, false, false);
}

void ProjectFrame::killAnalysisClicked()
{
    myEnvironment->process->kill();
    analysisFinished(0, QProcess::CrashExit);
}

void ProjectFrame::analyse(const bool make, const bool dependencies, const bool selective)
{
    if (isModified()) {
        if (myEnvironment->getAutosaveFilesAnalysis())
            saveAll();
        else
        if (QMessageBox::question(this, Info::TITLE,
                                  tr("Some files have been modified. \nSave files before analysing?"),
                                  QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)
            saveAll();
    }
    analysing = true;
    analyseSelected->setEnabled(false);
    analyseAll->setEnabled(false);
    killAnalysis->setEnabled(true);
    myprogressview->clear();
    myprogressview->setFocus();
    mytabwidget->setCurrentWidget(myprogressview);
    myreportview->clear();
    changeSubWindowFocus();

    QString commandFile = myEnvironment->getCommandFilename();
    QFile  commandfile(commandFile);
    commandfile.remove();
    bool res;
    if (make) {
        myParent->statusBar()->showMessage(tr("Configuring Make file"));
        depFilename = project->getMakeFilename();
        res = project->makeCommandFileDep(commandfile, optMK, depFilename);
    }
    else if (dependencies) {
        myParent->statusBar()->showMessage(tr("Determining Dependencies"));
        depFilename = myEnvironment->getDepFilename();
        QFile depfile(depFilename);
        depfile.remove();
        res = project->makeCommandFileDep( commandfile, optID, depFilename);
    }
    else {
        myParent->statusBar()->showMessage(tr("Analysing"));
        reportFilename = project->getFilename(FileKind::Report, 0);
        if (reportFilename.isEmpty())
            reportFilename = myEnvironment->getDefaultReportFilename();
        QFile reportfile(reportFilename);
        reportfile.remove();
        res = project->makeCommandFileAn(commandfile,!selective, reportFilename);
    }
    if (!res){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not generate the command file for the analyser."),
                             QMessageBox::Ok);
        analysisFinished(0,QProcess::CrashExit);
        return;
    }

    connect(myEnvironment->process, SIGNAL(finished(int, QProcess::ExitStatus)),
            this, SLOT(analysisFinished(int, QProcess::ExitStatus)));
    connect(myEnvironment->process, SIGNAL(readyRead()),
            this, SLOT(showAnalysisProgress()));

    QStringList args;
//#if defined(Q_OS_WIN)
    args.append('@' + myEnvironment->getCommandFilename());
//#else
//    if (make)
//      args.append(project->makeCommandArgDep(optMK, depFilename));
//    else if (dependencies)
//      args.append(project->makeCommandArgDep(optID, depFilename));
//    else
//      args.append(project->makeCommandArgAn(!selective, reportFilename));
//#endif
    int result = myEnvironment->execForcheck(args);
    if (result == -1){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not start the analyser."),
                             QMessageBox::Ok);
        analysisFinished(result,QProcess::CrashExit);
    }
}

void ProjectFrame::showAnalysisProgress()
{
    if (gen_make) {
        myParent->statusBar()->showMessage(tr("Configuring Make file"));
    }
    else if (update_dependencies) {
        myParent->statusBar()->showMessage(tr("Determining Dependencies"));
    }
    else {
        myParent->statusBar()->showMessage(tr("Analysing"));
    }
    myprogressview->appendText(myEnvironment->process->readAll());
}

void ProjectFrame::analysisFinished(int exitcode, QProcess::ExitStatus exitStatus)
{
    disconnect(myEnvironment->process, SIGNAL(finished(int, QProcess::ExitStatus)),
               this, SLOT(analysisFinished(int, QProcess::ExitStatus)));
    disconnect(myEnvironment->process, SIGNAL(readyRead()),
               this, SLOT(showAnalysisProgress()));
    if (exitStatus == QProcess::CrashExit)
        myParent->statusBar()->showMessage(tr("Forcheck was aborted or crashed"));
    else if (exitcode == 4){
        myParent->statusBar()->showMessage(tr("Forcheck completed with informative messages"));
    }
    else if (exitcode == 4){
        myParent->statusBar()->showMessage(tr("Forcheck completed with warnings"));
    }
    else if (exitcode == 8){
        myParent->statusBar()->showMessage(tr("Forcheck completed with error messages"));
    }
    else if (exitcode == 16){
        myParent->statusBar()->showMessage(tr("Forcheck completed with fatal error"));
    }
    else {
        project->compressUpdateLibrary();
        myParent->statusBar()->showMessage(QString());
    }
    analysing = false;
    updateEnabled();

//has view file been modified?
    QList<QMdiSubWindow *> subwindowlist = mymdiarea->subWindowList();
    foreach (QMdiSubWindow *subwindow, subwindowlist){
        QString filename = subwindow->windowFilePath();
        FileKind::ETFileKind filekind = project->getFileKind(filename);
        if (FileKind::isViewable(filekind)){
            if (QMessageBox::question(this, Info::TITLE,
                                      filename + tr(". File has been modified. Reload?"),
                                      QMessageBox::Yes,
                                      QMessageBox::No) == QMessageBox::Yes){
                if (filekind == FileKind::ReferenceStructure){
                    treeView->update();
                }
                else if (filekind == FileKind::LibOutput){
                    librarianForm->update();
                }
                else if (filekind == FileKind::Listing){
                    if (!qobject_cast<FileViewer *>(subwindow->widget())->loadFile(filename)) return;
                }
            };
        }
    }
    if (update_dependencies){
        if (exitStatus == QProcess::NormalExit && exitcode < 16)
            doUpdateIncDependencies();
        else
            QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                 tr("Project include files are not updated."),
                                 QMessageBox::Ok);
    }
    else if (gen_make){
        if (exitStatus == QProcess::NormalExit && exitcode < 16);
        else
            QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                 tr("Make file was not sucessfull generated."),
                                 QMessageBox::Ok);
    }
    else {
        myreportview->clear();
        myreportview->loadFile(reportFilename);
        connect(myreportview, SIGNAL(requestEdit(const QString&, int)),
                this, SLOT(fileEditRequest(const QString&, int)));
    }
}

void ProjectFrame::doUpdateIncDependencies()
{
    gen_make = false;
    update_dependencies = false;
    QFile file(depFilename);
    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        QMessageBox::warning(this, Info::TITLE,
                             tr("Could not open dependencies file\n%1.\n%2.")
                             .arg(depFilename)
                             .arg(file.errorString()));
        return;
    }
    QTextStream in(&file);
    QStringList incfiles;
    while (!in.atEnd()) {
        QString filename = in.readLine();
#if defined(Q_OS_WIN)
        filename = filename.toUpper();
#endif
        incfiles.append(filename);
    }
    ProjectFiles *files = project->getProjectFiles(FileKind::Include);
    bool uptodate = true;
    bool skipq = false;
    int ans = QMessageBox::No;
    for (int index = 0; index < files->getNrFiles(); index++){
        QFileInfo info(project->getFilename(FileKind::Include, index));
        QString filename = info.absoluteFilePath();
#if defined(Q_OS_WIN)
        filename = filename.toUpper();
#endif
        if (!incfiles.contains(QDir::toNativeSeparators(filename))){
            if (!skipq)
                ans = QMessageBox::question(this, Info::TITLE,
                                            filename + tr(". Include file not used. Remove from project?"),
                                            QMessageBox::Yes,
                                            QMessageBox::YesAll,
                                            QMessageBox::No);
            skipq = ans==QMessageBox::YesAll;
            if (ans==QMessageBox::Yes || skipq){
                project->removeProjectFile(FileKind::Include, index);
                fillTree();
            }
            else
                uptodate = false;
        }
    }
    skipq = false;
    ans = QMessageBox::No;
    for (int i = 0; i < incfiles.size(); ++i){
        int index = project->findFile(incfiles.at(i), FileKind::Include);
        if (index < 0){
            if (!skipq)
                ans = QMessageBox::question(this, Info::TITLE,
                                            incfiles.at(i) + tr(". Include file not in project. Add to project?"),
                                            QMessageBox::Yes |
                                            QMessageBox::YesToAll |
                                            QMessageBox::No |
                                            QMessageBox::Cancel);
            skipq = ans==QMessageBox::YesToAll;
            if (ans==QMessageBox::Yes || skipq){
                QFileInfo info(incfiles.at(i));
                QString filename = info.filePath();
                project->addProjectFile(FileKind::Include, filename);
                fillTree();
            }
            else
                if (ans==QMessageBox::Cancel){
                uptodate = false;
                break;
            }
            else
                uptodate = false;
        }
    }
    if (uptodate)
        QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                 tr("Project include files are up to date."),
                                 QMessageBox::Ok);
}

void ProjectFrame::findPreviousMsgClicked()
{
    searchString = "**[";
    if (searchInReport && myreportview){
        myreportview->find(searchString, QTextDocument::FindBackward);
    }
    else if (activeViewer()){
        activeViewer()->find(searchString, QTextDocument::FindBackward);
    }
}

void ProjectFrame::findNextMsgClicked()
{
    searchString = "**[";
    if (searchInReport && myreportview){
        myreportview->find(searchString);
    }
    else if (activeViewer()){
        activeViewer()->find(searchString);
    }
}

void ProjectFrame::configureMakefileClicked()
{
    bool gen;
    makefileForm = new MakefileForm(project, &gen, this);
    makefileForm->exec();
    if (gen){
      executeMakefile->setEnabled(false);
      project->genMakefileLeader();
      gen_make = true;
      update_dependencies = false;
      analyse(true, false, false);
    }
}

void ProjectFrame::selectMakefileClicked()
{
    selectMakefileForm = new SelectMakefileForm(project, this);
    selectMakefileForm->exec();
    updateEnabled();
}

void ProjectFrame::executeMakefileClicked()
{
    QString makefilename = project->getMakeFilename();
    QFile makefile(makefilename);
    if (!makefile.exists()){
        QMessageBox::information(this, Info::TITLE,
                                 makefilename + tr("\nMakefile does not exists"));
        return;
    }
    if (isModified()) {
        if (myEnvironment->getAutosaveFilesMake())
            saveAll();
        else
            if (QMessageBox::question(this, Info::TITLE,
                                      tr("Some files have been modified.\n"
                                         "Save files before make?"),
                                      QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)
                saveAll();
    }
    if (project->makeChanged()) {
        QMessageBox::StandardButton ret;
        ret = QMessageBox::question(this, Info::TITLE,
                                    tr("The project has been modified.\n"
                                       "Update makefile?"),
                                    QMessageBox::Ok | QMessageBox::Discard | QMessageBox::Cancel);
        if (ret == QMessageBox::Ok) {
            project->genMakefileLeader();
            gen_make = true;
            update_dependencies = false;
            analyse(true, false, false);
            project->setMakeChanged(false);
        }
        else
            if (ret == QMessageBox::Discard) {
                project->setMakeChanged(false);
            }
        else
            return;
    }
    QString objdirname = project->getObjDir(project->getMakeFileIndex());
    QDir objdir = QDir(objdirname);
    if (!objdir.exists()) {
        if (QMessageBox::question(this, Info::TITLE,
                                  tr("Directory ")+objdirname+tr(" does not exists.\n"
                                                                 "OK to create?"),
                                  QMessageBox::Yes | QMessageBox::No) !=
                QMessageBox::Yes) return;
        objdir.mkpath(objdirname);
    }
    QString targetdirname = project->getTargetDir(project->getMakeFileIndex());
    QDir targetdir = QDir(targetdirname);
    if (!targetdir.exists()) {
        if (QMessageBox::question(this, Info::TITLE,
                                  tr("Directory ")+targetdirname+tr(" does not exists.\n"
                                                                    "OK to create?"),
                                  QMessageBox::Yes | QMessageBox::No) !=
                QMessageBox::Yes) return;
        targetdir.mkpath(targetdirname);
    }
    executeMakefile->setEnabled(false);
    myParent->statusBar()->showMessage(tr("Building project"));
    myprogressview->clear();
    myprogressview->setFocus();
    changeSubWindowFocus();
    mytabwidget->setCurrentWidget(myprogressview);
    connect(myEnvironment->process, SIGNAL(finished(int, QProcess::ExitStatus)),
            this, SLOT(buildFinished(int, QProcess::ExitStatus)));
    connect(myEnvironment->process, SIGNAL(readyRead()),
            this, SLOT(showBuildProgress()));
    building = true;
    int result = myEnvironment->execMake(makefilename);
    if (result == -1){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not start make."));
        buildFinished(result,QProcess::CrashExit);
    }
}

void ProjectFrame::showBuildProgress()
{
    if (gen_make) {
        myParent->statusBar()->showMessage(tr("Configuring Make file"));
    }
    else if (building) {
        myParent->statusBar()->showMessage(tr("Building project"));
    }
    myprogressview->appendText(myEnvironment->process->readAll());
}

void ProjectFrame::buildFinished(int exitcode, QProcess::ExitStatus exitStatus)
{
    disconnect(myEnvironment->process, SIGNAL(finished(int, QProcess::ExitStatus)),
               this, SLOT(buildFinished(int, QProcess::ExitStatus)));
    disconnect(myEnvironment->process, SIGNAL(readyRead()),
               this, SLOT(showBuildProgress()));
    updateEnabled();
    building = false;
    myParent->statusBar()->showMessage(QString());
}

void ProjectFrame::defaultOptionsClicked()
{
    optionsForm = new OptionsForm(this);
    optionsForm->DefaultOptions(myEnvironment->getDefaultOptions(),
                                myEnvironment->getDefaultOptionsFilename());
    optionsForm->exec();
    myEnvironment->saveDefaultOptions();
    delete optionsForm;
}

void ProjectFrame::projectOptionsClicked()
{
    optionsForm = new OptionsForm(this);
    optionsForm->ProjectOptions(project->getProjectOptions(),
                                project->getProjectFilename());
    optionsForm->exec();
    delete optionsForm;
    fillTree();
    if (project->getProjectOptions()->isChanged()){
        project->setMakeChanged(true);
    }
}

void ProjectFrame::fileOptionsClicked()
{
    QList<QTreeWidgetItem *> list = selectedItems();
    FileKind::ETFileKind filekind;
    int index;
    locateFile(list[0], filekind, index);
    if (filekind == FileKind::Source){
        optionsForm = new OptionsForm(this);
        optionsForm->SourceFileOptions(project->getFileOptions(filekind, index),
                                       project->getProjectOptions(),
                                       project->getFilename(filekind, index));
    }
    else if (filekind == FileKind::LibRef){
        optionsForm = new OptionsForm(this);
        optionsForm->LibraryFileOptions(project->getFileOptions(filekind, index),
                                        project->getProjectOptions(),
                                        project->getFilename(filekind, index));
    }
    else
        return;
    optionsForm->exec();
    int ind;
    Options *popt = project->getFileOptions(filekind, index);
    for (int i=0; i<list.count(); ++i){
        locateFile(list[i], filekind, ind);
        Options *popt1 = project->getFileOptions(filekind, ind);
        *popt1 = *popt;
    }
    delete optionsForm;
    fillTree();
    updateEnabled();
}

void ProjectFrame::viewSelectiveAnalysisCommandlineClicked()
{
    QString arg;
    arg.append('@' + myEnvironment->getCommandFilename());
    myprogressview->putText(Constants::FckExecutable + " " + arg);
    mytabwidget->setCurrentWidget(myprogressview);
}

void ProjectFrame::viewAnalysisCommandlineClicked()
{
    QString arg;
    arg.append('@' + myEnvironment->getCommandFilename());
    myprogressview->putText(Constants::FckExecutable + " " + arg);
    mytabwidget->setCurrentWidget(myprogressview);
}

void ProjectFrame::viewSelectiveAnalysisCommandFileClicked()
{
    QFile commandfile(myEnvironment->getCommandFilename());
    commandfile.remove();
    reportFilename = project->getFilename(FileKind::Report, 0);
    if (reportFilename.isEmpty())
      reportFilename = myEnvironment->getDefaultReportFilename();
    if (!project->makeCommandFileAn(commandfile,false, reportFilename))
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not generate the command file."),
                             QMessageBox::Ok);
    if (!commandfile.open(QIODevice::ReadOnly | QIODevice::Text))
        return;
    QTextStream stream;
    stream.setDevice(&commandfile);
    myprogressview->putText(stream.readAll());
    mytabwidget->setCurrentWidget(myprogressview);
    commandfile.close();
}

void ProjectFrame::viewAnalysisCommandFileClicked()
{
    QFile commandfile(myEnvironment->getCommandFilename());
    commandfile.remove();
    reportFilename = project->getFilename(FileKind::Report, 0);
    if (reportFilename.isEmpty())
      reportFilename = myEnvironment->getDefaultReportFilename();
    if (!project->makeCommandFileAn(commandfile,true,reportFilename))
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not generate the command file."),
                             QMessageBox::Ok);
    if (!commandfile.open(QIODevice::ReadOnly | QIODevice::Text))
        return;
    QTextStream stream;
    stream.setDevice(&commandfile);
    myprogressview->putText(stream.readAll());
    mytabwidget->setCurrentWidget(myprogressview);
    commandfile.close();
}
