/*ForcheckIDE/ProjectFiles.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "ProjectFiles.h"
#include "FileKinds.h"
#include "IniFile.h"
#include "Environment.h"
#include "Info.h"

#include <QtWidgets>
#include <QString>
#include <QList>
#include <QStandardItem>
#include <QFileInfo>
#include <QDir>

ProjectFiles::ProjectFiles(Options *projectoptions,
                           const FileKind::ETFileKind filekind)
{
        projectOptions = projectoptions;
        kind = filekind;

        nr_files_in_analysis = 0;
        changed = false;
        fileEntries = new QList<FileEntry *>;
}

ProjectFiles::~ProjectFiles()
{
        for (int i = 0; i< fileEntries->count(); i++){
            delete fileEntries->at(i);
        }
        delete fileEntries;
}

int ProjectFiles::getNrFiles() const
{
    if (fileEntries == 0)
        return 0;
    else {
        return fileEntries->count();
    }
}

int ProjectFiles::findFile(const QString& filespec) const
{
         QFileInfo fileinfo(filespec);
         for (int i = 0; i< fileEntries->count(); i++){
             QString filename1 = QDir::toNativeSeparators(fileinfo.absoluteFilePath());
             QString filename2 = QDir::toNativeSeparators(fileEntries->at(i)->filename);
#if defined(Q_OS_WIN)
             if (filename2.compare(&filename1,Qt::CaseInsensitive) == 0)
#elif defined(Q_OS_LINUX)
             if (filename2.compare(&filename1,Qt::CaseSensitive) == 0)
#endif
                 return i;
         }
         return -1;
}

int ProjectFiles::addFile(const QString& filespec)
{
        if (findFile(filespec) >= 0) return -1;
        FileEntry *fileentry = new FileEntry(this);
        QFileInfo info(filespec);
        fileentry->filename = info.absoluteFilePath();
        fileentry->fileOptions = NULL;
        switch (kind){
        case FileKind::Source:
            fileentry->fileOptions = new Options(projectOptions, OptionsKind::OptionsKindSource);
            break;
        case FileKind::LibRef:
            fileentry->fileOptions = new Options(projectOptions, OptionsKind::OptionsKindRefLib);
            break;
        case FileKind::LibOutput:
            fileentry->fileOptions = new Options(projectOptions, OptionsKind::OptionsKindOutLib);
            break;
        default:
            break;
        }
        fileEntries->append(fileentry);
        int index = fileEntries->count() - 1;
        return index;
}

void ProjectFiles::renameFileAt(int index, const QString& newfilespec)
{
        setFilename(index,newfilespec);
}

QString ProjectFiles::getFilename(int index) const
{
        if (fileEntries->isEmpty() || index < 0 || index > fileEntries->count())
            return QString();
        else
            return fileEntries->at(index)->filename;
}

QString ProjectFiles::getRelFilename(int index) const
{
        if (fileEntries->isEmpty() || index < 0 || index > fileEntries->count())
            return QString();
        else{
            QDir cwd(myEnvironment->getWorkingDir());
            return cwd.relativeFilePath(fileEntries->at(index)->filename);
        }
}

void ProjectFiles::setFilename(int index, const QString& newfilespec)
{
    if (fileEntries->isEmpty() || index < 0 || index > fileEntries->count())
        return;
    else {
        fileEntries->at(index)->filename = newfilespec;
    }
}

Options *ProjectFiles::getFileOptions(int index)
{
    if (index < 0)
        return 0;
    else {
        return fileEntries->at(index)->fileOptions;
    }
}

const Options *ProjectFiles::getFileOptions(int index) const
{
    if (fileEntries == 0 || index < 0)
        return 0;
    else {
        return fileEntries->at(index)->fileOptions;
    }
}

bool ProjectFiles::isChanged() const
{
        if (changed)
                return true;
        for (int i = 0; i < fileEntries->count(); ++i){
            if (getFileOptions(i))
                if (getFileOptions(i)->isChanged())
                    return true;
        }
        return false;
}

void ProjectFiles::setChanged(bool change)
{
        if (change) {
                changed = true;
                return;
        }
        changed = false;
        for (int i = 0; i < fileEntries->count(); ++i)
            if (getFileOptions(i))
                getFileOptions(i)->clearChanged();
}

void ProjectFiles::removeFileAt(int index)
{
        if (index < 0 || index >= fileEntries->count()) return;
        fileEntries->removeAt(index);
        changed = true;
}

bool ProjectFiles::analyseFileAt(int index) const
// Check if file is to be analysed
{
        return getFileEntry(index)->analyseFile;
}

void ProjectFiles::setAnalyse(int index, bool analyse)
{
        if (index < 0 || index >= fileEntries->count()) return;
        FileEntry *fileEntry = getFileEntry(index);
        if (analyse == fileEntry->analyseFile)
                return;
        fileEntry->analyseFile = analyse;
        if (analyse)
                ++nr_files_in_analysis;
        else
                --nr_files_in_analysis;
        changed = true;
}

void ProjectFiles::setOption(int index, const QString& OptionName, TriStateBool value)
{
        if (index < 0 || index >= fileEntries->count()) return;
        FileEntry *fileEntry = getFileEntry(index);
        Options *opts = fileEntry->fileOptions;
        if (opts)
            opts->setBool(OptionName, value);
}

QString const sectionSplit = " file - ";

void ProjectFiles::save(IniFile &conf)
{
    for (int i = 0; i < fileEntries->count(); ++i)
        ((FileEntry *) fileEntries->at(i))->save(conf,getFilename(i));
}

bool ProjectFiles::load(IniFile &conf)
{
    foreach (QString section, conf.childGroups()){
        FileKind::ETFileKind filekind =
            FileKind::string2FileKind(section.section(sectionSplit, 0, 0));
        if (filekind != kind) continue;
        QString sectionfilename = section.section(sectionSplit, 1);
        conf.beginGroup(section);
          QString filename = conf.value("FileName").toString();
          if (sectionfilename != filename) {
              if (QMessageBox::warning(QApplication::activeWindow(),
                                       Info::TITLE,
                                       "section: " + sectionfilename + "\n" +
                                       "file: " + filename + "\n" +
                                       QObject::tr("Mismatch of filename and section header in project file."),
                                       QMessageBox::Ignore | QMessageBox::Abort) ==
                  QMessageBox::Abort) return false;
          }
          int index = addFile(filename);
          if (index < 0) {
              if (QMessageBox::warning(QApplication::activeWindow(),
                                       Info::TITLE,
                                       "section: " + sectionfilename + "\n" +
                                       "file: " + filename + "\n" +
                                       QObject::tr("Could not add file to project."),
                                       QMessageBox::Ignore | QMessageBox::Abort) ==
                  QMessageBox::Abort) return false;
          }
          else{
              Options *opts = getFileEntry(index)->fileOptions;
              if (opts) opts->load(conf);
              getFileEntry(index)->analyseFile = conf.value("IncludedForAnalysis").toInt() == 1;
              if (getFileEntry(index)->analyseFile)
                  ++nr_files_in_analysis;
          }
        conf.endGroup();
    }
    return true;
}

//=======================================================================================================
ProjectFiles::FileEntry::FileEntry(ProjectFiles* parent)
{
        Parent = parent;
        filename.clear();
        fileOptions = NULL;
        analyseFile = false;
        absolutePathname = false;
}

ProjectFiles::FileEntry::FileEntry(const FileEntry &original)
{
        Parent = original.Parent;
        filename = original.filename;
        fileOptions = NULL;
        if (original.fileOptions){
            fileOptions = new Options();
            *fileOptions = *original.fileOptions;
        }
        analyseFile = original.analyseFile;
        absolutePathname = original.absolutePathname;
}

ProjectFiles::FileEntry::~FileEntry()
{
        if (fileOptions)
            delete fileOptions;
        fileOptions = NULL;
}

ProjectFiles::FileEntry *ProjectFiles::getFileEntry(int index)
{
        if (index < 0 || index >= fileEntries->count())
                return NULL;
        return fileEntries->at(index);
}

const ProjectFiles::FileEntry *ProjectFiles::getFileEntry(int index) const
{
        if (index < 0 || index >= fileEntries->count()) {
                return NULL;
        }
        return fileEntries->at(index);
}

void ProjectFiles::FileEntry::save(IniFile& conf, const QString& filespec)
{
    QDir cwd(myEnvironment->getWorkingDir());
    QString filename = cwd.relativeFilePath(filespec);
    QString section = FileKind::fileKind2String(Parent->kind);
    section.append(sectionSplit);
    section.append(filename);

    conf.beginGroup(section);
        conf.setValue("FileName", filename);
        conf.setValue("IncludedForAnalysis", analyseFile);
        if (fileOptions) fileOptions->save(conf);
    conf.endGroup();
}
