/*ForcheckIDE/Project.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "Project.h"
#include "Environment.h"
#include "Info.h"
#include "FileKinds.h"
#include "Constants.h"
#include "Version.h"
#include "IniFile.h"
#include "LibrarianActions.h"

#include <QtWidgets>
#include <QFile>
#include <QDir>
#include <QTextStream>
#include <QProcess>

Project::Project()
{      
        // create project options
        myProjectOptions = new Options(myEnvironment->getDefaultOptions(),
                                       OptionsKind::OptionsKindProject);
        LastUsedInAnalysis.PW = 0;
        LastUsedInAnalysis.PL = 0;
        makefileIndex = 0;

        // create list for each filekind
        ForEachProjectFileKind(fk)
                myProjectFiles[fk]  = new ProjectFiles(myProjectOptions, fk);

        title.clear();
        changed = false;
        datetimePreferences[0] = myEnvironment->getdatetimePreferences();
        datetimePreferences[1] = myEnvironment->getdatetimePreferences();
}

Project::~Project()
{
        ForEachProjectFileKind(fk)
            if (myProjectFiles[fk])
                delete myProjectFiles[fk];

        if (myProjectOptions)
            delete myProjectOptions;
}

void Project::save()
{
        if (projectFilename.length() == 0)
                return;

        QFileInfo info(projectFilename);
        projectFilename = info.absoluteFilePath();
        projectDirectory = info.absolutePath();
        myEnvironment->setWorkingDir(projectDirectory);

        // Create a backup of the project file
        // This also ensures the new project file will start empty
        QString backup_projectfilename = projectFilename + "~";
        QDir dir(backup_projectfilename);
        dir.remove(backup_projectfilename);
        dir.rename(projectFilename,backup_projectfilename);

        // Create and initialize projectfile
        IniFile projectsettings(projectFilename);
        if (!projectsettings.create()){
            QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                 projectFilename+tr(". Could not create project file."),
                                 QMessageBox::Ok);
            return;
        };
        Version versionInfo(Info::VERSION);
        projectsettings.beginGroup("Main");
            projectsettings.setValue("ident", "FORCHECK project file");
            projectsettings.setValue("version", versionInfo.getMajorVersion());
            projectsettings.setValue("release", versionInfo.getMinorVersion());
            projectsettings.setValue("Title", title);
            projectsettings.setValue("LastUsed_PW",LastUsedInAnalysis.PW);
            projectsettings.setValue("LastUsed_PL",LastUsedInAnalysis.PL);
        projectsettings.endGroup();

        projectsettings.beginGroup("Make");
            projectsettings.setValue("makefileIndex", makefileIndex);
            projectsettings.setValue("makefile_Debug", makefileName[0]);
            projectsettings.setValue("makefileChanged_Debug", make_changed[0]);
            projectsettings.setValue("datetimePreferences_Debug", datetimePreferences[0].toString());
            projectsettings.setValue("compiler_Debug", compiler[0]);
            projectsettings.setValue("compilerOptions_Debug", compilerOptions[0]);
            projectsettings.setValue("cCompiler_Debug", cCompiler[0]);
            projectsettings.setValue("cCompilerOptions_Debug", cCompilerOptions[0]);
            projectsettings.setValue("objDir_Debug", objDir[0]);
            projectsettings.setValue("target_Debug", target[0]);
            projectsettings.setValue("targetDir_Debug", targetDir[0]);
            projectsettings.setValue("linkerOptions_Debug", linkerOptions[0]);
            projectsettings.setValue("makefile_Release", makefileName[1]);
            projectsettings.setValue("makefileChanged_Release", make_changed[1]);
            projectsettings.setValue("datetimePreferences_Release", datetimePreferences[1].toString());
            projectsettings.setValue("compiler_Release", compiler[1]);
            projectsettings.setValue("compilerOptions_Release", compilerOptions[1]);
            projectsettings.setValue("cCompiler_Release", cCompiler[1]);
            projectsettings.setValue("cCompilerOptions_Release", cCompilerOptions[1]);
            projectsettings.setValue("objDir_Release", objDir[1]);
            projectsettings.setValue("target_Release", target[1]);
            projectsettings.setValue("targetDir_Release", targetDir[1]);
            projectsettings.setValue("linkerOptions_Release", linkerOptions[1]);
        projectsettings.endGroup();

        projectsettings.beginGroup("Project Options");
            myProjectOptions->save(projectsettings);
        projectsettings.endGroup();
        ForEachProjectFileKind(fk)
            myProjectFiles[fk]->save(projectsettings);

        changed = false;
        projectsettings.close();
}

void Project::saveAs(const QString &name)
{
    projectFilename = name;
    save();
    setMakeChanged(true);
}

bool Project::load(const QString &name)
{
        changed = false;
        QFile file(name);
        if (!file.exists()) {
            if (QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                tr("Project does not exist. Create new project?"),
                QMessageBox::Ok | QMessageBox::Cancel) ==
                QMessageBox::Ok)
                    return true;
	        else
                    return false;
        }
        projectDirectory = FileKind::fileDir(name);
        myEnvironment->setWorkingDir(projectDirectory);

        IniFile projectsettings(name);
        if (!projectsettings.open()){
            QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                 name+tr(". Could not open project file."),
                                 QMessageBox::Ok);
            return false;
        };
        projectsettings.beginGroup("Main");
        if (projectsettings.value("ident").toString() != "FORCHECK project file") {
                QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                     name+tr(". File is not a FORCHECK project file."),
                                     QMessageBox::Ok);
                return false;
            }
            Version versionInfo(Info::VERSION);
            QString s = projectsettings.value("version").toString();
            if (s == "13") {
                if (QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                         name+tr(". Projectfile is version 13. Update to ")+versionInfo.getMajorVersion()+"?",
                                         QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes)
                      changed = true;
                    else
                      return false;
            }
            else if (s != versionInfo.getMajorVersion()) {
                QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                     name+tr(". Project is not version ")+versionInfo.getMajorVersion(),
                                     QMessageBox::Ok);
                return false;
            }
            title = projectsettings.value("Title").toString();
            LastUsedInAnalysis.PW = projectsettings.value("LastUsed_PW").toInt();
            LastUsedInAnalysis.PL = projectsettings.value("LastUsed_PL").toInt();
        projectsettings.endGroup();

        projectsettings.beginGroup("Make");
            makefileIndex = projectsettings.value("makefileIndex").toInt();
            makefileName[0] = projectsettings.value("makefile_Debug").toString();
            make_changed[0] = projectsettings.value("makefileChanged_Debug").toBool();
            datetimePreferences[0] = QDateTime :: fromString(projectsettings.value("datetimePreferences_Debug").toString());
            compiler[0] = projectsettings.value("compiler_Debug").toString();
            compilerOptions[0] = projectsettings.value("compilerOptions_Debug").toString();
            cCompiler[0] = projectsettings.value("cCompiler_Debug").toString();
            cCompilerOptions[0] = projectsettings.value("cCompilerOptions_Debug").toString();
            objDir[0] = projectsettings.value("objDir_Debug").toString();
            target[0] = projectsettings.value("target_Debug").toString();
            targetDir[0] = projectsettings.value("targetDir_Debug").toString();
            linkerOptions[0] = projectsettings.value("linkerOptions_Debug").toString();
            makefileName[1] = projectsettings.value("makefile_Release").toString();
            make_changed[1] = projectsettings.value("makefileChanged_Release").toBool();
            datetimePreferences[1] = QDateTime :: fromString(projectsettings.value("datetimePreferences_Release").toString());
            compiler[1] = projectsettings.value("compiler_Release").toString();
            compilerOptions[1] = projectsettings.value("compilerOptions_Release").toString();
            cCompiler[1] = projectsettings.value("cCompiler_Release").toString();
            cCompilerOptions[1] = projectsettings.value("cCompilerOptions_Release").toString();
            objDir[1] = projectsettings.value("objDir_Release").toString();
            target[1] = projectsettings.value("target_Release").toString();
            targetDir[1] = projectsettings.value("targetDir_Release").toString();
            linkerOptions[1] = projectsettings.value("linkerOptions_Release").toString();
        projectsettings.endGroup();

        projectsettings.beginGroup("Project Options");
            myProjectOptions->load(projectsettings);
        projectsettings.endGroup();

        ForEachProjectFileKind(fk)
            if (!myProjectFiles[fk]->load(projectsettings))
                return false;
        projectFilename = name;
        projectsettings.close();
        return true;
}

ProjectFiles* Project::getProjectFiles(const FileKind::ETFileKind filekind)
{
    if (!FileKind::canBeInProject(filekind))
            return 0;
    else {
        return myProjectFiles[filekind];
    }
}

QString Project::getProjectFile(const FileKind::ETFileKind filekind, const int index)
{ 
    if (!FileKind::canBeInProject(filekind))
        return QString();
    return myProjectFiles[filekind]->getFilename(index);
}

int Project::addProjectFile(const FileKind::ETFileKind filekind, const QString& filename)
{
    if (!FileKind::canBeInProject(filekind))
        return -1;
    if (findFile(filename) >= 0)
        return -1;

    int index = myProjectFiles[filekind]->addFile(filename);
    if (index < 0) return index;

// set default cpp option for files with specific extensions
    if (filekind==FileKind::Source &&
        FileKind::isFppFile(filename) &&
        getProjectOptions()->getBoolEffective(optCPP) != TSB_true)
            myProjectFiles[filekind]->setOption(index, optCPP, TSB_true);

// set default ff option for files with specific extensions
    if (filekind==FileKind::Source &&
        FileKind::isDefaultFfFile(filename) &&
        getProjectOptions()->getBoolEffective(optFF) != TSB_true)
            myProjectFiles[filekind]->setOption(index, optFF, TSB_true);

    changed = true;
    if (filekind==FileKind::Source || filekind==FileKind::CSource) setMakeChanged(true);
    return index;
}

bool Project::removeProjectFile(const FileKind::ETFileKind filekind, const int index)
{
    if (!FileKind::canBeInProject(filekind))
        return false;
    myProjectFiles[filekind]->removeFileAt(index);
    changed = true;
    if (filekind==FileKind::Source) setMakeChanged(true);
    return true;
}

void Project::setMakeFileIndex(const int& index)
{
    if (index != makefileIndex){
        makefileIndex = index;
        changed = true;
    }
}

void Project::setMakeChanged(bool c)
{
    if (c) {
        if (!make_changed[0] || !make_changed[1]) changed = true;
        make_changed[0] = true;
        make_changed[1] = true;
    }
    else {
        if (makeChanged()) changed = true;
        make_changed[makefileIndex] = false;
        datetimePreferences[makefileIndex] = myEnvironment->getdatetimePreferences();
    }
}

bool Project::makeChanged() const
{
    QDateTime d1 = datetimePreferences[makefileIndex];
    QDateTime d2 = myEnvironment->getdatetimePreferences();
    bool outdated = (datetimePreferences[makefileIndex] < myEnvironment->getdatetimePreferences());
    return make_changed[makefileIndex] || outdated;
}

void Project::setMakeFilename(const QString& filename)
{
    if (filename != makefileName[makefileIndex]){
        makefileName[makefileIndex] = filename;
        changed = true;
    }
}

void Project::setCompiler(const QString& name)
{
    if (name != compiler[makefileIndex]){
        compiler[makefileIndex] = name;
        changed = true;
    }
}

void Project::setCompilerOptions(const QString& options)
{
    if (options != compilerOptions[makefileIndex]){
        compilerOptions[makefileIndex] = options;
        changed = true;
    }
}

void Project::setCCompiler(const QString& name)
{
    if (name != cCompiler[makefileIndex]){
        cCompiler[makefileIndex] = name;
        changed = true;
    }
}

void Project::setCCompilerOptions(const QString& options)
{
    if (options != cCompilerOptions[makefileIndex]){
        cCompilerOptions[makefileIndex] = options;
        changed = true;
    }
}

void Project::setObjDir(const QString& dir)
{
    if (dir != objDir[makefileIndex]){
        objDir[makefileIndex] = dir;
        changed = true;
    }
}

void Project::setTarget(const QString& tar)
{
    if (tar != target[makefileIndex]){
        target[makefileIndex] = tar;
        changed = true;
    }
}

void Project::setTargetDir(const QString& dir)
{
    if (dir != targetDir[makefileIndex]){
        targetDir[makefileIndex] = dir;
        changed = true;
    }
}

void Project::setLinkerOptions(const QString& options)
{
    if (options != linkerOptions[makefileIndex]){
        linkerOptions[makefileIndex] = options;
        changed = true;
    }
}


void Project::genMakefileLeader()
{
    if (makefileName[makefileIndex].length() == 0)
        return;

    // Create a backup of the makefile
    QString backup_makefilename = makefileName[makefileIndex] + "~";
    QDir dir(backup_makefilename);
    dir.remove(backup_makefilename);
    dir.rename(makefileName[makefileIndex],backup_makefilename);

    // Create makefile
    file.setFileName(makefileName[makefileIndex]);
    if (!file.open(QIODevice::WriteOnly | QIODevice::Text)){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             makefileName[makefileIndex]+tr(". Could not create makefile."),
                             QMessageBox::Ok);
        return;
    };
    QTextStream stream;
    stream.setDevice(&file);
    stream << ".SUFFIXES:" << endl;
    stream << "COMP=" << compiler[makefileIndex] << endl;
    stream << "OPTSCOMP=" << compilerOptions[makefileIndex];
    if (!myEnvironment->IPnames->isEmpty())
      stream << " -I" << myEnvironment->IPnames->join(" -I");
    if (myProjectOptions->getList(optIP)->size() > 0)
      stream << " -I" << myProjectOptions->getList(optIP)->join(" -I");
    stream << endl;
    stream << "CC=" << cCompiler[makefileIndex] << endl;
    stream << "OPTSCC=" << cCompilerOptions[makefileIndex] << endl;
    stream << "DEFS=";
    if (myProjectOptions->getList(optDF)->size() > 0)
      stream << "-D" << myProjectOptions->getList(optDF)->join(" -D");
    stream << endl;
    stream << "OBJPATH=" << objDir[makefileIndex] << endl;
    stream << "TARGETDIR=" << targetDir[makefileIndex] << endl;
    if (targetDir[makefileIndex] == "")
        stream << "TARGET= " << target[makefileIndex] << endl;
    else
        stream << "TARGET= $(TARGETDIR)/" << target[makefileIndex] << endl;
    stream << "OBJS= \\" << endl;
// write object filename(s)
    int n = 0;
    int ntot = myProjectFiles[FileKind::Source]->getNrFiles() +
               myProjectFiles[FileKind::CSource]->getNrFiles();
    for (int i = 0; i < myProjectFiles[FileKind::Source]->getNrFiles(); i++){
        QFileInfo fi(getFilename(FileKind::Source,i));
        QFileInfo fi1(fi.canonicalFilePath());
        stream << "$(OBJPATH)/" << fi1.baseName() << ".o";
        n++;
        if (n < ntot)
            stream << " \\";
        stream << endl;
    }
    for (int i = 0; i < myProjectFiles[FileKind::CSource]->getNrFiles(); i++){
        QFileInfo fi(getFilename(FileKind::CSource,i));
        stream << "$(OBJPATH)/" << fi.baseName() << ".o";
        n++;
        if (n < ntot)
            stream << " \\";
        stream << endl;
    }
    stream << endl;
    stream << "$(TARGET): $(OBJS)" << endl;
    stream << QChar(9) << "$(COMP) " << linkerOptions[makefileIndex] << " -o $(TARGET) $(OBJS)" << endl;
    stream << endl;

    for (int i = 0; i < myProjectFiles[FileKind::CSource]->getNrFiles(); i++){
        QFileInfo fi(getFilename(FileKind::CSource,i));
#if defined(Q_OS_WIN)
        stream << "$(OBJPATH)/" << fi.baseName() << ".o: " << fi.filePath() << endl;
        stream << QChar(9) << "$(CC) " << "$(OPTSCC) " << fi.filePath() << " -o $(OBJPATH)/" << fi.baseName() << ".o" << endl;
#else
        stream << "$(OBJPATH)/" << fi.baseName() << ".o: " << fi.filePath() << endl;
        stream << QChar(9) << "$(CC) " << "$(OPTSCC) " << fi.filePath() << " -o $(OBJPATH)/" << fi.baseName() << ".o" << endl;
#endif
        stream << endl;
    }

    stream.flush();
    file.close();
    setMakeChanged(false);
}

bool Project::setAnalyse(const FileKind::ETFileKind filekind, const int index, bool analyse)
{
    if (!FileKind::isIncludable(filekind))
        return false;
    myProjectFiles[filekind]->setAnalyse(index, analyse);
    return true;
}

bool Project::getAnalyse(const FileKind::ETFileKind filekind, const int index)
{
    if (!FileKind::isIncludable(filekind))
        return false;
    return myProjectFiles[filekind]->analyseFileAt(index);
}

int Project::findFile(const QString &path)
{
    ForEachProjectFileKind(fk) {
        int index = myProjectFiles[fk]->findFile(path);
        if (index >= 0) return index;
    }
    return -1;
}

int Project::findFile(const QString &path, const FileKind::ETFileKind filekind)
{
    ForEachProjectFileKind(fk) {
        if (fk == filekind)
            return myProjectFiles[fk]->findFile(path);
    }
    return -1;
}

FileKind::ETFileKind Project::getFileKind(const QString &path)
{
    ForEachProjectFileKind(fk) {
        if (myProjectFiles[fk]->findFile(path) >= 0) return fk;
    }
    return FileKind::Unknown;
}

QString Project::getFilename(const FileKind::ETFileKind filekind, const int index) const
{
    return myProjectFiles[filekind]->getFilename(index);
}

QString Project::getRelFilename(const FileKind::ETFileKind filekind, const int index) const
{
    return myProjectFiles[filekind]->getRelFilename(index);
}

Options* Project::getFileOptions(const FileKind::ETFileKind filekind, const int index)
{
    return myProjectFiles[filekind]->getFileOptions(index);
}

int Project::getNrFilesInAnalysis(const FileKind::ETFileKind filekind)
// returns the number of files in this project for selective analysis
{
    return myProjectFiles[filekind]->getNrFilesInAnalysis();
}

int Project::getNrFilesInAnalysis()
// returns the number of files in this project for selective analysis
{
    int count = 0;
    ForEachProjectFileKind(fk){
        if (FileKind::canBeAnalysed(fk))
            count += myProjectFiles[fk]->getNrFilesInAnalysis();
    }
    return count;
}

int Project::getNrFilesAnalysisAll()
// returns the number of files in this project for complete analysis
{
    int count = 0;
    ForEachProjectFileKind(fk){
        if (FileKind::canBeAnalysed(fk))
            count += myProjectFiles[fk]->getNrFiles();
    }
    return count;
}

int Project::getNrFiles()
// returns the number of files in this project
{
    int count = 0;
    ForEachProjectFileKind(fk)
        count += getNrFiles(fk);
    return count;
}

int Project::getNrFiles(const FileKind::ETFileKind filekind)
{
    if (filekind <= FileKind::Project || filekind >= FileKind::Count)
        return 0;
    return myProjectFiles[filekind]->getNrFiles();
}

bool Project::makeCommandFileAn(QFile &commandfile, const bool fullanalysis, const QString& reportfilename)
{
    if (!commandfile.open(QIODevice::WriteOnly | QIODevice::Text))
        return false;
    QTextStream stream;
    stream.setDevice(&commandfile);

    LastUsedInAnalysis.PW = myProjectOptions->getIntEffective(optPW);
    LastUsedInAnalysis.PL = myProjectOptions->getIntEffective(optPL);

    // if update/create library already exists it will be updated
    if (getNrFiles(FileKind::LibOutput) > 0) {
        QFile file(getFilename(FileKind::LibOutput,0));
        if (myEnvironment->updateLibrary && file.exists()) {
            getFileOptions(FileKind::LibOutput,0)->setUpdateLibOption();
        }
        else {
            file.remove();
            getFileOptions(FileKind::LibOutput,0)->setCreateLibOption();
        }
    }

    // make the analysis complete or selective
    myEnvironment->getDefaultOptions()->setSelective(!fullanalysis);
    myProjectOptions->setSelective(!fullanalysis);

    // list file
    if ((fullanalysis && myProjectFiles[FileKind::Listing]->getNrFiles() > 0) ||
        (getNrFilesInAnalysis(FileKind::Listing) > 0)) {
        if (optLI != ""){
            stream << PREopt;
            stream << optLI;
            stream << DELopt;
        }
        stream << QDir::toNativeSeparators(getRelFilename(FileKind::Listing,0));
    }

    // report file
        stream << PREopt;
        stream << optRP;
        stream << DELopt;
        //        stream << QDir::toNativeSeparators(getRelFilename(FileKind::Report,0));
        QDir cwd(myEnvironment->getWorkingDir());
        QString repfil = cwd.relativeFilePath(reportfilename);
        stream << QDir::toNativeSeparators(repfil);

        // reference structure file
        if ((fullanalysis && myProjectFiles[FileKind::ReferenceStructure]->getNrFiles() > 0) ||
            (getNrFilesInAnalysis(FileKind::ReferenceStructure) > 0)) {
            stream << PREopt;
            stream << optRSF;
            stream << DELopt;
            stream << QDir::toNativeSeparators(getRelFilename(FileKind::ReferenceStructure,0));
        }

        // module dependencies file
        if ((fullanalysis && myProjectFiles[FileKind::ModDep]->getNrFiles() > 0) ||
            (getNrFilesInAnalysis(FileKind::ModDep) > 0)) {
            stream << PREopt;
            stream << optMDF;
            stream << DELopt;
            stream << QDir::toNativeSeparators(getRelFilename(FileKind::ModDep,0));
        }

    stream << myProjectOptions->toCommandString();
    stream << endl;

    // write source filename(s) + local source file options
    int n = 0;
    for (int i = 0; i < myProjectFiles[FileKind::Source]->getNrFiles(); i++){
        if (fullanalysis || myProjectFiles[FileKind::Source]->analyseFileAt(i)){
#if defined(Q_OS_WIN)
            stream << QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
            stream << getFileOptions(FileKind::Source,i)->toCommandString();
#elif defined(Q_OS_LINUX)
            stream << getFileOptions(FileKind::Source,i)->toCommandString();
            stream << DELopt;
            stream << QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
#endif
            n++;
            if ((fullanalysis && (n < myProjectFiles[FileKind::Source]->getNrFiles())) ||
                (n < getNrFilesInAnalysis(FileKind::Source)))
               stream << CNTopt;
            stream << endl;
        }
    }
    if (n == 0)
        stream << endl;
    n = 0;
    bool sepinserted = true;

    // write create library filename + local library file options
    if ((fullanalysis && (getNrFiles(FileKind::LibOutput) > 0)) ||
                         (getNrFilesInAnalysis(FileKind::LibOutput) > 0))
    {
#if defined(Q_OS_WIN)
        stream << QDir::toNativeSeparators(getRelFilename(FileKind::LibOutput,0));
        stream << getFileOptions(FileKind::LibOutput,0)->toCommandString();
#elif defined(Q_OS_X11)
        stream << getFileOptions(FileKind::LibOutput,0)->toCommandString();
        stream << DELopt;
        stream << QDir::toNativeSeparators(getRelFilename(FileKind::LibOutput,0));
#endif
        n++;
        sepinserted = false;
    }

    // write reference library filenames + local library file options
    if ((fullanalysis && (getNrFiles(FileKind::LibRef) > 0)) ||
                         (getNrFilesInAnalysis(FileKind::LibRef) > 0))
    {
        if (!sepinserted){
            stream << CNTopt;
            stream << endl;
        }
        sepinserted = false;
        for (int i = 0; i < getNrFiles(FileKind::LibRef); i++){
#if defined(Q_OS_WIN)
            stream << QDir::toNativeSeparators(getRelFilename(FileKind::LibRef,i));
            stream << getFileOptions(FileKind::LibRef,i)->toCommandString();
#elif defined(Q_OS_X11)
            stream << getFileOptions(FileKind::LibRef,0)->toCommandString();
            stream << DELopt;
            stream << QDir::toNativeSeparators(getRelFilename(FileKind::LibRef,0));
#endif
            n++;
            if (i != (getNrFiles(FileKind::LibRef) - 1))
                stream << CNTopt;
            stream << endl;
        }
    }
    if (n == 0)
        stream << endl;

    stream.flush();
    commandfile.close();

    return true;
}

QString Project::makeCommandArgAn(const bool fullanalysis, const QString& reportfilename)
{
    LastUsedInAnalysis.PW = myProjectOptions->getIntEffective(optPW);
    LastUsedInAnalysis.PL = myProjectOptions->getIntEffective(optPL);

    // if update/create library already exists it will be updated
    if (getNrFiles(FileKind::LibOutput) > 0) {
        QFile file(getFilename(FileKind::LibOutput,0));
        if (myEnvironment->updateLibrary && file.exists()) {
            getFileOptions(FileKind::LibOutput,0)->setUpdateLibOption();
        }
        else {
            file.remove();
            getFileOptions(FileKind::LibOutput,0)->setCreateLibOption();
        }
    }

    // make the analysis complete or selective
    myEnvironment->getDefaultOptions()->setSelective(!fullanalysis);
    myProjectOptions->setSelective(!fullanalysis);

    QString command_arg;

    // list file
    if ((fullanalysis && myProjectFiles[FileKind::Listing]->getNrFiles() > 0) ||
        (getNrFilesInAnalysis(FileKind::Listing) > 0)) {
        if (optLI != ""){
            command_arg += PREopt;
            command_arg += optLI;
            command_arg += DELopt;
        }
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::Listing,0));
    }

    // report file
    if (myProjectFiles[FileKind::Report]->getNrFiles() == 0) {
        command_arg += PREopt;
        command_arg += optRP;
        command_arg += DELopt;
        command_arg += QDir::toNativeSeparators(myEnvironment->getDefaultReportFilename());
    }
    else {
        command_arg += PREopt;
        command_arg += optRP;
        command_arg += DELopt;
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::Report,0));
    }

    // reference structure file
    if ((fullanalysis && myProjectFiles[FileKind::ReferenceStructure]->getNrFiles() > 0) ||
        (getNrFilesInAnalysis(FileKind::ReferenceStructure) > 0)) {
        command_arg += PREopt;
        command_arg += optRSF;
        command_arg += DELopt;
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::ReferenceStructure,0));
    }

    // module dependencies file
    if ((fullanalysis && myProjectFiles[FileKind::ModDep]->getNrFiles() > 0) ||
        (getNrFilesInAnalysis(FileKind::ModDep) > 0)) {
        command_arg += PREopt;
        command_arg += optMDF;
        command_arg += DELopt;
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::ModDep,0));
    }

    command_arg += myProjectOptions->toCommandString();

    command_arg += NXTopt;

    // write source filename(s) + local source file options
    int n = 0;
    for (int i = 0; i < myProjectFiles[FileKind::Source]->getNrFiles(); i++){
        if (fullanalysis || myProjectFiles[FileKind::Source]->analyseFileAt(i)){
#if defined(Q_OS_WIN)
            command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
            command_arg += getFileOptions(FileKind::Source,i)->toCommandString();
#elif defined(Q_OS_X11)
            command_arg += getFileOptions(FileKind::Source,i)->toCommandString();
            command_arg += DELopt;
            command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
#endif
            n++;
            if ((fullanalysis && (n < myProjectFiles[FileKind::Source]->getNrFiles())) ||
                (n < getNrFilesInAnalysis(FileKind::Source)))
                command_arg += CNTopt;
        }
    }
    bool sepinserted = false;

    // write create library filename + local library file options
    if ((fullanalysis && (getNrFiles(FileKind::LibOutput) > 0)) ||
                         (getNrFilesInAnalysis(FileKind::LibOutput) > 0))
    {
        command_arg += NXTopt;
        sepinserted = true;
#if defined(Q_OS_WIN)
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::LibOutput,0));
        command_arg += getFileOptions(FileKind::LibOutput,0)->toCommandString();
#elif defined(Q_OS_X11)
        command_arg += getFileOptions(FileKind::LibOutput,0)->toCommandString();
        command_arg += DELopt;
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::LibOutput,0));
#endif
    }

    // write reference library filenames + local library file options
    if ((fullanalysis && (getNrFiles(FileKind::LibRef) > 0)) ||
                         (getNrFilesInAnalysis(FileKind::LibRef) > 0))
    {
        if (sepinserted)
            command_arg += CNTopt;
        else
            command_arg += NXTopt;
        sepinserted = true;
        for (int i = 0; i < getNrFiles(FileKind::LibRef); i++){
#if defined(Q_OS_WIN)
            command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::LibRef,i));
            command_arg += getFileOptions(FileKind::LibRef,i)->toCommandString();
#elif defined(Q_OS_X11)
            command_arg += getFileOptions(FileKind::LibRef,0)->toCommandString();
            command_arg += DELopt;
            command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::LibRef,0));
#endif
            if (i != (getNrFiles(FileKind::LibRef) - 1))
                command_arg += CNTopt;
        }
    }
    if (!sepinserted)
        command_arg += SEPopt;

    return command_arg;
}

bool Project::makeCommandFileDep(QFile &commandfile, const QString& opt, const QString& depfilename)
{
    if (!commandfile.open(QIODevice::WriteOnly | QIODevice::Text))
        return false;
    QTextStream stream;
    stream.setDevice(&commandfile);

    stream << PREopt;
    stream << opt;
    stream << DELopt;
    stream << QDir::toNativeSeparators(depfilename);

    stream << myProjectOptions->toCommandString();

    stream << endl;

    // write source filename(s) + local source file options
    int n = 0;
    for (int i = 0; i < myProjectFiles[FileKind::Source]->getNrFiles(); i++){
#if defined(Q_OS_WIN)
        stream << QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
        stream << getFileOptions(FileKind::Source,i)->toCommandString();
#elif defined(Q_OS_LINUX)
        stream << getFileOptions(FileKind::Source,i)->toCommandString();
        stream << DELopt;
        stream << QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
#endif
        n++;
        if (n < myProjectFiles[FileKind::Source]->getNrFiles())
           stream << CNTopt;
        stream << endl;
    }
    stream << endl;

    stream.flush();
    commandfile.close();

    return true;
}

QString Project::makeCommandArgDep(const QString& opt, const QString& depfilename)
{
    QString command_arg;

    // dep file
    command_arg += PREopt;
    command_arg += opt;
    command_arg += DELopt;
    command_arg += QDir::toNativeSeparators(depfilename);

    command_arg += myProjectOptions->toCommandString();

    command_arg += NXTopt;

    // write source filename(s) + local source file options
    int n = 0;
    for (int i = 0; i < myProjectFiles[FileKind::Source]->getNrFiles(); i++){
#if defined(Q_OS_WIN)
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
        command_arg += getFileOptions(FileKind::Source,i)->toCommandString();
#elif defined(Q_OS_X11)
        command_arg += getFileOptions(FileKind::Source,i)->toCommandString();
        command_arg += DELopt;
        command_arg += QDir::toNativeSeparators(getRelFilename(FileKind::Source,i));
#endif
        n++;
        if (n < getNrFilesInAnalysis(FileKind::Source))
            command_arg += CNTopt;
    }
    command_arg += SEPopt;

    return command_arg;
}

int Project::compressUpdateLibrary()
{
    int status = 0;
    if (myEnvironment->updateLibrary &&
        myEnvironment->compressLibrary &&
        getNrFiles(FileKind::LibOutput) == 1){
            LibrarianActions *fcklibact = new LibrarianActions(getRelFilename(FileKind::LibOutput,0));
            status = fcklibact->compress();
            if (status != LIB_ERR_OK) {
                fcklibact->showErrorMsg();
            }
            delete fcklibact;
    }
    return status;
}

bool Project::isChanged() const {

    if (changed)
        return true;
    if (myProjectOptions->isChanged())
        return true;
    ForEachProjectFileKind(fk)
        if (myProjectFiles[fk]->isChanged())
            return true;
    return false;
}

void Project::setChanged(bool c)
{
    changed = c;
    if (!c)
        ForEachProjectFileKind(fk)
            myProjectFiles[fk]->setChanged(false);
}
