/*ForcheckIDE/PreferencesForm.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "PreferencesForm.h"
#include "Constants.h"
#include "FileKinds.h"
#include "Environment.h"
#include "Info.h"
#include "HelpBrowser.h"
#include "CnfOptions.h"
#include "FileExistsValidator.h"

#include <QtWidgets>
#include <QDir>

static QString output_streams[NR_OUTPUT_STREAMS] = {
        "stdout",
        "listing",
        "report"
};

static QString output_cat[NR_OUTPUT_CAT] = {
        "message summary",
        "metrics",
        "table usage"
};

PreferencesForm::PreferencesForm(QWidget *parent) : QDialog(parent)
{
    setWindowTitle("Preferences");

    tabWidget = new QTabWidget();

    createTabs();

    tabWidget->addTab(directoriesTab, tr("Directories"));
    tabWidget->addTab(extensionsTab, tr("Extensions"));
    tabWidget->addTab(emulationTab, tr("Compiler emulation"));
    tabWidget->addTab(editorTab, tr("Editor"));
    tabWidget->addTab(buildTab, tr("Build"));
    tabWidget->addTab(messagesTab, tr("Messages"));
    tabWidget->addTab(messageReportTab, tr("Message reporting"));
    tabWidget->addTab(numberingTab, tr("Numbering"));
    tabWidget->addTab(dateTimeFormatTab, tr("Date/time"));
    tabWidget->addTab(outputTab, tr("Output"));
    tabWidget->addTab(librariesTab, tr("Libraries"));
    connect(tabWidget, SIGNAL(currentChanged(int)), this, SLOT(tabClicked(int)));

    helpButton = new QPushButton(tr("&Help"));
    helpButton->setToolTip(tr("Present the relevant page of the user guide"));
    helpButton->setFixedWidth(80);
    helpButton->setShortcut(QKeySequence::HelpContents);
    okButton = new QPushButton(tr("&OK"));
    okButton->setToolTip(tr("Accept the specified preferences and close the window"));
    okButton->setFixedWidth(80);
    applyButton = new QPushButton(tr("&Apply"));
    applyButton->setToolTip(tr("Accept the specified preferences"));
    applyButton->setFixedWidth(80);
    cancelButton = new QPushButton(tr("&Cancel"));
    cancelButton->setToolTip(tr("Cancel the specified preferences and close the window"));
    cancelButton->setFixedWidth(80);

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget(helpButton);
    buttonLayout->addWidget(okButton);
    buttonLayout->addWidget(applyButton);
    buttonLayout->addWidget(cancelButton);

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(tabWidget);
    mainLayout->addLayout(buttonLayout);
    setLayout(mainLayout);

    connect(helpButton, SIGNAL(clicked()), this, SLOT(helpClicked()));
    connect(okButton, SIGNAL(clicked()), this, SLOT(okClicked()));
    connect(applyButton, SIGNAL(clicked()), this, SLOT(applyClicked()));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(close()));

    myCnfOptions = NULL;
    initialize();
}

PreferencesForm::~PreferencesForm()
{
    if (myCnfOptions) delete myCnfOptions;
    delete helpButton;
    delete okButton;
    delete applyButton;
    delete cancelButton;
    delete tabWidget;
}

void PreferencesForm::createTabs()
{
    createIncludeDirectoriesTab();
    createExtensionsTab();
    createEmulationTab();
    createEditorTab();
    createBuildTab();
    createMessagesTab();
    createReportMessageTab();
    createNumberingTab();
    createDateTimeFormatTab();
    createOutputTab();
    createLibrariesTab();
}

void PreferencesForm::initialize()
{
    initializeIncludeDirectories();
    initializeExtensions();
    initializeEmulation();
    initializeEditor();
    initializeBuild();
    initializeMessages();
    initializeReportMessage();
    initializeCnfOptions();
    initializeLibraries();
}

void PreferencesForm::applyChanges()
{
    applyChangesIncludeDirectories();
    applyChangesExtensions();
    applyChangesEmulation();
    applyChangesEditor();
    applyChangesBuild();
    applyChangesMessages();
    applyChangesReportMessage();
    applyChangesCnfOptions();
    applyChangesLibraries();
    QWidget *widget = tabWidget->currentWidget();
    if ((widget == directoriesTab) || (widget == buildTab))
        myEnvironment->setDateTimePreferences();
    myEnvironment->saveIdeSettings();
}

void PreferencesForm::tabClicked(int index)
{
    switch (index) {
    case 0:
        break;
    case 1:
        break;
    case 2:
        break;
    case 3:
        break;
    case 4:
        break;
    case 5:
        searchEdit->setFocus();
        break;
    case 6:
        break;
    case 7:
        break;
    case 8:
        break;
    case 9:
        break;
    case 10:
        break;
    }
}

void PreferencesForm::createIncludeDirectoriesTab()
{
    directoriesTab = new QWidget;
    QGroupBox *directoriesGroupBox = new QGroupBox(tr("Absolute include directories"));
    directoriesListBox = new QListWidget();
    connect(directoriesListBox, SIGNAL(itemClicked(QListWidgetItem *)), this, SLOT(itemDirClicked()));
    addButton = new QPushButton(tr("&Add..."));
    addButton->setToolTip(tr("Select a directory to add to the search paths"));
    connect(addButton, SIGNAL(clicked()), this, SLOT(addDirClicked()));
    removeButton = new QPushButton(tr("&Remove"));
    removeButton->setToolTip(tr("Remove the selected paths from the search paths"));
    connect(removeButton, SIGNAL(clicked()), this, SLOT(removeDirClicked()));
    clearButton = new QPushButton(tr("&Clear"));
    clearButton->setToolTip(tr("Clear the list of search paths"));
    connect(clearButton, SIGNAL(clicked()), this, SLOT(clearDirClicked()));
    QLabel *memo = new QLabel(this);
    memo->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    memo->setText(tr("Here you can specify the absolute paths to search for include files.\n") +
                  tr("You can specify releative search paths in the project options."));

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addWidget(addButton);
    vbox->addWidget(removeButton);
    vbox->addWidget(clearButton);
    QHBoxLayout *hbox = new QHBoxLayout;
    hbox->addWidget(directoriesListBox);
    hbox->addLayout(vbox);
    directoriesGroupBox->setLayout(hbox);
    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(directoriesGroupBox);
    vboxt->addStretch();
    vboxt->addWidget(memo);
    directoriesTab->setLayout(vboxt);
}

void PreferencesForm::initializeIncludeDirectories()
{
    directoriesListBox->clear();
    directoriesListBox->addItems(*myEnvironment->IPnames);
    removeButton->setEnabled(false);
    if (directoriesListBox->count() == 0)
        clearButton->setEnabled(false);
}

void PreferencesForm::addDirClicked()
{
    QString dir = QFileDialog::getExistingDirectory(this,
                                                    tr("Select a directory to search for include files"),
                                                    QDir::currentPath());
    if (!dir.isEmpty()){
        *myEnvironment->IPnames << dir;
        directoriesListBox->addItem(dir);
        clearButton->setEnabled(true);
    }
    removeButton->setEnabled(false);
}

void PreferencesForm::itemDirClicked()
{
    removeButton->setEnabled(true);
}

void PreferencesForm::removeDirClicked()
{
    myEnvironment->IPnames->removeAt(directoriesListBox->currentRow());
    directoriesListBox->takeItem(directoriesListBox->currentRow());
    removeButton->setEnabled(false);
    if (directoriesListBox->count() == 0)
        clearButton->setEnabled(false);
}

void PreferencesForm::clearDirClicked()
{
    myEnvironment->IPnames->removeAt(directoriesListBox->currentRow());
    directoriesListBox->clear();
    removeButton->setEnabled(false);
    clearButton->setEnabled(false);
}

void PreferencesForm::applyChangesIncludeDirectories()
{
}

void PreferencesForm::createExtensionsTab()
{
    extensionsTab = new QWidget;
    QGroupBox *extensionsGroupBox = new QGroupBox(tr("Default filename extensions"));
    int width = 10 * fontMetrics().width('x');

    QLabel *projectLabel = new QLabel(tr("Project file:"));
    projectExtEdit = new QLineEdit();
    projectExtEdit->setFixedWidth(width);

    QLabel *makeLabel = new QLabel(tr("Make file:"));
    makeExtEdit = new QLineEdit();
    makeExtEdit->setFixedWidth(width);

    QLabel *sourceLabel = new QLabel(tr("Source file:"));
    sourceExtEdit = new QLineEdit();
    sourceExtEdit->setFixedWidth(width);

    QLabel *presourceLabel = new QLabel(tr("Pre-source file:"));
    presourceExtEdit = new QLineEdit();
    presourceExtEdit->setFixedWidth(width);

    QLabel *cSourceLabel = new QLabel(tr("C-source file:"));
    cSourceExtEdit = new QLineEdit();
    cSourceExtEdit->setFixedWidth(width);

    QLabel *includeLabel = new QLabel(tr("Include file:"));
    includeExtEdit = new QLineEdit();
    includeExtEdit->setFixedWidth(width);

    QLabel *libraryLabel = new QLabel(tr("Library file:"));
    libraryExtEdit = new QLineEdit();
    libraryExtEdit->setFixedWidth(width);

    QLabel *listingLabel = new QLabel(tr("Listing file:"));
    listingExtEdit = new QLineEdit();
    listingExtEdit->setFixedWidth(width);

    QLabel *reportLabel = new QLabel(tr("Report file:"));
    reportExtEdit = new QLineEdit();
    reportExtEdit->setFixedWidth(width);

    QLabel *refstructLabel = new QLabel(tr("Reference structure file:"));
    refstructExtEdit = new QLineEdit();
    refstructExtEdit->setFixedWidth(width);

    QLabel *moddepLabel = new QLabel(tr("Module dependencies file:"));
    moddepExtEdit = new QLineEdit();
    moddepExtEdit->setFixedWidth(width);

    QPushButton *defaultButton = new QPushButton(tr("&Default"));
    defaultButton->setToolTip(tr("Set each extension to the default value"));
    connect(defaultButton, SIGNAL(clicked()), this, SLOT(defaultClicked()));

    QHBoxLayout *projectHbox = new QHBoxLayout;
    projectHbox->addWidget(projectLabel);
    projectHbox->addStretch();
    projectHbox->addWidget(projectExtEdit);

    QHBoxLayout *makeHbox = new QHBoxLayout;
    makeHbox->addWidget(makeLabel);
    makeHbox->addStretch();
    makeHbox->addWidget(makeExtEdit);

    QHBoxLayout *sourceHbox = new QHBoxLayout;
    sourceHbox->addWidget(sourceLabel);
    sourceHbox->addStretch();
    sourceHbox->addWidget(sourceExtEdit);

    QHBoxLayout *presourceHbox = new QHBoxLayout;
    presourceHbox->addWidget(presourceLabel);
    presourceHbox->addStretch();
    presourceHbox->addWidget(presourceExtEdit);

    QHBoxLayout *cSourceHbox = new QHBoxLayout;
    cSourceHbox->addWidget(cSourceLabel);
    cSourceHbox->addStretch();
    cSourceHbox->addWidget(cSourceExtEdit);

    QHBoxLayout *includeHbox = new QHBoxLayout;
    includeHbox->addWidget(includeLabel);
    includeHbox->addStretch();
    includeHbox->addWidget(includeExtEdit);

    QHBoxLayout *libraryHbox = new QHBoxLayout;
    libraryHbox->addWidget(libraryLabel);
    libraryHbox->addStretch();
    libraryHbox->addWidget(libraryExtEdit);

    QHBoxLayout *listingHbox = new QHBoxLayout;
    listingHbox->addWidget(listingLabel);
    listingHbox->addStretch();
    listingHbox->addWidget(listingExtEdit);

    QHBoxLayout *reportHbox = new QHBoxLayout;
    reportHbox->addWidget(reportLabel);
    reportHbox->addStretch();
    reportHbox->addWidget(reportExtEdit);

    QHBoxLayout *refstructHbox = new QHBoxLayout;
    refstructHbox->addWidget(refstructLabel);
    refstructHbox->addStretch();
    refstructHbox->addWidget(refstructExtEdit);

    QHBoxLayout *moddepHbox = new QHBoxLayout;
    moddepHbox->addWidget(moddepLabel);
    moddepHbox->addStretch();
    moddepHbox->addWidget(moddepExtEdit);

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addLayout(projectHbox);
    vbox->addLayout(makeHbox);
    vbox->addLayout(sourceHbox);
    vbox->addLayout(presourceHbox);
    vbox->addLayout(cSourceHbox);
    vbox->addLayout(includeHbox);
    vbox->addLayout(libraryHbox);
    vbox->addLayout(listingHbox);
    vbox->addLayout(reportHbox);
    vbox->addLayout(refstructHbox);
    vbox->addLayout(moddepHbox);
    vbox->addStretch(1);

    extensionsGroupBox->setLayout(vbox);

    QHBoxLayout *defaultHbox = new QHBoxLayout;
    defaultHbox->addWidget(defaultButton);
    defaultHbox->addStretch();

    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(extensionsGroupBox);
    vboxt->addLayout(defaultHbox);
    extensionsTab->setLayout(vboxt);
}

void PreferencesForm::initializeExtensions()
{
    projectExtEdit->setText(myEnvironment->getProjectExt());
    makeExtEdit->setText(myEnvironment->getMakeExt());
    sourceExtEdit->setText(myEnvironment->getSourceExt());
    presourceExtEdit->setText(myEnvironment->getPreSourceExt());
    cSourceExtEdit->setText(myEnvironment->getCSourceExt());
    includeExtEdit->setText(myEnvironment->getIncludeExt());
    libraryExtEdit->setText(myEnvironment->getLibraryExt());
    listingExtEdit->setText(myEnvironment->getListingExt());
    reportExtEdit->setText(myEnvironment->getReportExt());
    refstructExtEdit->setText(myEnvironment->getRefstructExt());
    moddepExtEdit->setText(myEnvironment->getModDepExt());
}

void PreferencesForm::defaultClicked()
{
    projectExtEdit->setText(Constants::DefaultExt_Project);
    makeExtEdit->setText(Constants::DefaultExt_Make);
    sourceExtEdit->setText(Constants::DefaultExt_Source);
    presourceExtEdit->setText(Constants::DefaultExt_PreSource);
    cSourceExtEdit->setText(Constants::DefaultExt_CSource);
    includeExtEdit->setText(Constants::DefaultExt_Include);
    libraryExtEdit->setText(Constants::DefaultExt_Library);
    listingExtEdit->setText(Constants::DefaultExt_Listing);
    reportExtEdit->setText(Constants::DefaultExt_Report);
    refstructExtEdit->setText(Constants::DefaultExt_Refstruct);
    moddepExtEdit->setText(Constants::DefaultExt_ModDep);
}

void PreferencesForm::applyChangesExtensions()
{
    myEnvironment->setExtension(FileKind::Project,projectExtEdit->text());
    myEnvironment->setExtension(FileKind::Make,makeExtEdit->text());
    myEnvironment->setExtension(FileKind::Source,sourceExtEdit->text());
    myEnvironment->setExtension(FileKind::PreSource,presourceExtEdit->text());
    myEnvironment->setExtension(FileKind::CSource,cSourceExtEdit->text());
    myEnvironment->setExtension(FileKind::Include,includeExtEdit->text());
    myEnvironment->setExtension(FileKind::LibRef,libraryExtEdit->text());
    myEnvironment->setExtension(FileKind::Listing,listingExtEdit->text());
    myEnvironment->setExtension(FileKind::Report,reportExtEdit->text());
    myEnvironment->setExtension(FileKind::ReferenceStructure,refstructExtEdit->text());
    myEnvironment->setExtension(FileKind::ModDep,moddepExtEdit->text());
}

//======================================================================================================

void PreferencesForm::createEmulationTab()
{
    emulationTab = new QWidget;
    QGroupBox *emulationGroupBox = new QGroupBox(tr("Compiler emulation"));

    QLabel *label1 = new QLabel("Compiler emulation file:");
    emulationFileEdit = new QLineEdit();
    emulationFileEdit->setFixedWidth(400);
    validator = new FileExistsValidator();
    emulationFileEdit->setValidator(validator);
    QPushButton *p = new QPushButton("...");
    p->setToolTip(tr("Browse for a compiler emulation file"));
    p->setFixedWidth(25);
    connect(p, SIGNAL(clicked()), this, SLOT(selectEmulationFileClicked()));
    QLabel *label2 = new QLabel("Available compiler emulations:");
    emulationListBox = new QListWidget();
    connect(emulationListBox, SIGNAL(itemClicked(QListWidgetItem *)),
            this, SLOT(emulationListBoxItemClicked(QListWidgetItem *)));

    QHBoxLayout *hbox = new QHBoxLayout;
    hbox->addWidget(emulationFileEdit);
    hbox->addStretch();
    hbox->addWidget(p);

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addWidget(label1);
    vbox->addLayout(hbox);
    vbox->addWidget(label2);
    vbox->addWidget(emulationListBox);
    vbox->addStretch();
    emulationGroupBox->setLayout(vbox);

    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(emulationGroupBox);
    emulationTab->setLayout(vboxt);
}

void PreferencesForm::initializeEmulation()
{
// present emulation file in use
    emulationFileEdit->setText(myEnvironment->getEmulationFilename());
// fill combobox with names of emulated compilers
    QDir dir(myEnvironment->getFckDataDir());
    fileList = new QStringList(dir.entryList(QStringList("*.cnf"), QDir::Files, QDir::Name));
    emulationListBox->clear();
    QString emulation;
    bool initfmt;
    int nr = 0;
    for (int i = 0; i < fileList->size(); ++i) {
        myEnvironment->getEmulation(myEnvironment->getFckDataDir() + fileList->at(i), emulation, initfmt);
        if (!emulation.isEmpty()){
            emulationListBox->addItem(emulation);
            fileIndex[nr] = i;
            ++nr;
        }
    }
}

void PreferencesForm::selectEmulationFileClicked()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Select File"),
                                                    myEnvironment->getFckDataDir(),
                                                    tr("Compiler emulation files (*.cnf)"));
    if (filename.isEmpty()) return;
    emulationFileEdit->setText(filename);
    QString emulation;
    bool initfmt;
    myEnvironment->getEmulation(filename, emulation, initfmt);
    if (emulation.isEmpty()) return;
    QList<QListWidgetItem *> items = emulationListBox->findItems(emulation, Qt::MatchExactly);
    if (items.isEmpty()) return;
    if (items.at(0))
        emulationListBox->setCurrentItem(items.at(0));
}

void PreferencesForm::emulationListBoxItemClicked(QListWidgetItem *item)
{
    int i = emulationListBox->row(item);
    emulationFileEdit->setText(myEnvironment->getFckDataDir() + fileList->at(fileIndex[i]));
}

void PreferencesForm::applyChangesEmulation()
{
    myEnvironment->setEmulationFilename(emulationFileEdit->text());
}

//======================================================================================================

void PreferencesForm::createEditorTab()
{
    editorTab = new QWidget;
    QGroupBox *editorGroupBox = new QGroupBox(tr("Editor setup"));
    QLabel *editorLabel = new QLabel(tr("Editor:"));
    editorEdit = new QLineEdit();
    QPushButton *defaultButton = new QPushButton(tr("&Default"));
    defaultButton->setToolTip(tr("Reset the editor and commandline to the default"));
    connect(defaultButton, SIGNAL(clicked()), this, SLOT(defaultEditorClicked()));
    QPushButton *browseButton = new QPushButton(tr("&Browse..."));
    browseButton->setToolTip(tr("Browse for an editor"));
    connect(browseButton, SIGNAL(clicked()), this, SLOT(browseClicked()));
    viewerCheck = new QCheckBox(tr("&Use this editor also as file viewer"));
    editorCommandlineLabel = new QLabel(tr("Editor commandline:"));
    editorCommandlineEdit = new QLineEdit();
    QLabel *memo = new QLabel(this);
    memo->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    memo->setText(tr("Here you can specify the parameters on the commandline of your editor.\n") +
                  tr("This is useful if you want to edit a source file at the line which contains errors.\n\n") +
                  tr("%f	filename\n") +
                  tr("%n	line number\n") +
                  tr("%t	warning/error message\n") +
                  tr("%%	a % character\n\n") +
                  tr("Examples:\n") +
                  tr("Notepad++:	-n%n -lfortran %f\n") +
                  tr("UltraEdit32:	\"%f\"/%n\n") +
                  tr("WinEdt:	-c\"%t\" \"[Open('%f');SelLine(%n,7)]\""));

    QHBoxLayout *hbox = new QHBoxLayout;
    hbox->addWidget(defaultButton);
    hbox->addStretch();
    hbox->addWidget(browseButton);

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addWidget(editorLabel);
    vbox->addWidget(editorEdit);
    vbox->addLayout(hbox);
    vbox->addWidget(viewerCheck);
    vbox->addWidget(editorCommandlineLabel);
    vbox->addWidget(editorCommandlineEdit);
    vbox->addStretch();
    vbox->addWidget(memo);
    editorGroupBox->setLayout(vbox);

    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(editorGroupBox);
    editorTab->setLayout(vboxt);
    editorEdit->setFocus();
}

void PreferencesForm::resetEditor()
{
    editorCommandlineLabel->setEnabled(false);
    editorCommandlineEdit->setEnabled(false);
    viewerCheck->setChecked(false);
    viewerCheck->setEnabled(false);
}

void PreferencesForm::initializeEditor()
{
    editorEdit->setText(myEnvironment->getEditor());
    if (editorEdit->text().isEmpty()) {
        resetEditor();
    }
    else {
        editorCommandlineLabel->setEnabled(true);
        editorCommandlineEdit->setText(myEnvironment->getEditorCommandline());
        editorCommandlineEdit->setEnabled(true);
        viewerCheck->setChecked(myEnvironment->getEditorAsViewer());
        viewerCheck->setEnabled(true);
    }
}

void PreferencesForm::defaultEditorClicked()
{
    editorEdit->setText(Constants::DefaultEditorFileName);
    if (editorEdit->text().isEmpty()) {
        resetEditor();
    }
    else {
        editorCommandlineEdit->setText(Constants::DefaultEditorCommandLine);
        editorCommandlineEdit->setEnabled(true);
        viewerCheck->setChecked(false);
        viewerCheck->setEnabled(true);
    }
}

void PreferencesForm::browseClicked()
{
    QString fileName = QFileDialog::getOpenFileName(this,
                                                    tr("Locate Editor"),
                                                    QDir::rootPath(),
                                                    tr("Executable (*") +
                                                       myEnvironment->getBinExt() +
                                                       ")");
    if (!fileName.isEmpty()) {
        editorEdit->setText(fileName);
        editorCommandlineLabel->setEnabled(true);
        editorCommandlineEdit->setEnabled(true);
        viewerCheck->setEnabled(true);
    }
}

void PreferencesForm::applyChangesEditor()
{
    myEnvironment->setEditor(editorEdit->text());
    myEnvironment->setEditorCommandline(editorCommandlineEdit->text());
    myEnvironment->setEditorAsViewer(viewerCheck->isChecked());
}

//======================================================================================================

void PreferencesForm::createBuildTab()
{
    buildTab = new QWidget;
    QGroupBox *buildGroupBox = new QGroupBox(tr("Build setup"));
    int width = 60 * fontMetrics().width('x');
    int width1 = 55 * fontMetrics().width('x');

    QLabel *compilerLabel = new QLabel(tr("Fortran compiler:"));
    compilerEdit = new QLineEdit();
    compilerEdit->setFixedWidth(width);

    QLabel *compilerOptionsLabel = new QLabel(tr("Compiler options (release):"));
    compilerOptionsEdit = new QLineEdit();
    compilerOptionsEdit->setFixedWidth(width);

    QLabel *compilerDebugOptionsLabel = new QLabel(tr("Compiler options (debug):"));
    compilerDebugOptionsEdit = new QLineEdit();
    compilerDebugOptionsEdit->setFixedWidth(width);

    QLabel *cCompilerLabel = new QLabel(tr("C compiler:"));
    cCompilerEdit = new QLineEdit();
    cCompilerEdit->setFixedWidth(width);

    QLabel *cCompilerOptionsLabel = new QLabel(tr("C compiler options:"));
    cCompilerOptionsEdit = new QLineEdit();
    cCompilerOptionsEdit->setFixedWidth(width);

    QLabel *linkerOptionsLabel = new QLabel(tr("Linker options:"));
    linkerOptionsEdit = new QLineEdit();
    linkerOptionsEdit->setFixedWidth(width);

    QLabel *buildUtilityLabel = new QLabel(tr("Build (make) utility:"));
    buildUtilityEdit = new QLineEdit();
    buildUtilityEdit->setFixedWidth(width1);

    QPushButton *p = new QPushButton("...");
    p->setToolTip(tr("Browse for a build (make) utility"));
    p->setFixedWidth(25);
    connect(p, SIGNAL(clicked()), this, SLOT(selectBuildUtilityClicked()));

    QLabel *memo = new QLabel(this);
    memo->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    memo->setText(tr("Here you can specify some default settings to compose the makefile and the utility to build the project, e.g. make.\n\n") +
                  tr("The makefile specified for the project is applied as the -f option of the make command.\n\n") +
                  tr("If no make utility is specified the makefile is executed as a script or batch file."));

    QHBoxLayout *hboxCompiler = new QHBoxLayout;
    hboxCompiler->addWidget(compilerLabel);
    hboxCompiler->addStretch();
    hboxCompiler->addWidget(compilerEdit);

    QHBoxLayout *hboxCompilerOptions = new QHBoxLayout;
    hboxCompilerOptions->addWidget(compilerOptionsLabel);
    hboxCompilerOptions->addStretch();
    hboxCompilerOptions->addWidget(compilerOptionsEdit);

    QHBoxLayout *hboxCompilerDebugOptions = new QHBoxLayout;
    hboxCompilerDebugOptions->addWidget(compilerDebugOptionsLabel);
    hboxCompilerDebugOptions->addStretch();
    hboxCompilerDebugOptions->addWidget(compilerDebugOptionsEdit);

    QHBoxLayout *hboxcCompiler = new QHBoxLayout;
    hboxcCompiler->addWidget(cCompilerLabel);
    hboxcCompiler->addStretch();
    hboxcCompiler->addWidget(cCompilerEdit);

    QHBoxLayout *hboxcCompilerOptions = new QHBoxLayout;
    hboxcCompilerOptions->addWidget(cCompilerOptionsLabel);
    hboxcCompilerOptions->addStretch();
    hboxcCompilerOptions->addWidget(cCompilerOptionsEdit);

    QHBoxLayout *hboxLinkerOptions = new QHBoxLayout;
    hboxLinkerOptions->addWidget(linkerOptionsLabel);
    hboxLinkerOptions->addStretch();
    hboxLinkerOptions->addWidget(linkerOptionsEdit);

    QHBoxLayout *hboxBuildUtility = new QHBoxLayout;
    hboxBuildUtility->addWidget(buildUtilityLabel);
    hboxBuildUtility->addStretch();
    hboxBuildUtility->addWidget(buildUtilityEdit);
    hboxBuildUtility->addWidget(p);

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addLayout(hboxCompiler);
    vbox->addLayout(hboxCompilerOptions);
    vbox->addLayout(hboxCompilerDebugOptions);
    vbox->addLayout(hboxcCompiler);
    vbox->addLayout(hboxcCompilerOptions);
    vbox->addLayout(hboxLinkerOptions);
    vbox->addLayout(hboxBuildUtility);
    vbox->addStretch();
    vbox->addWidget(memo);
    buildGroupBox->setLayout(vbox);

    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(buildGroupBox);
    buildTab->setLayout(vboxt);
    buildUtilityEdit->setFocus();
}

void PreferencesForm::initializeBuild()
{
    compilerEdit->setText(myEnvironment->getCompiler());
    compilerOptionsEdit->setText(myEnvironment->getCompilerOptions());
    compilerDebugOptionsEdit->setText(myEnvironment->getCompilerDebugOptions());
    cCompilerEdit->setText(myEnvironment->getcCompiler());
    cCompilerOptionsEdit->setText(myEnvironment->getcCompilerOptions());
    linkerOptionsEdit->setText(myEnvironment->getLinkerOptions());
    buildUtilityEdit->setText(myEnvironment->getBuildUtility());
}

void PreferencesForm::selectBuildUtilityClicked()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Select build utility"),
                                                    "",
                                                    tr("Build utility (*.exe)"));
    if (filename.isEmpty()) return;
    buildUtilityEdit->setText(filename);
}

void PreferencesForm::applyChangesBuild()
{
    myEnvironment->setCompiler(compilerEdit->text());
    myEnvironment->setCompilerOptions(compilerOptionsEdit->text());
    myEnvironment->setCompilerDebugOptions(compilerDebugOptionsEdit->text());
    myEnvironment->setcCompiler(cCompilerEdit->text());
    myEnvironment->setcCompilerOptions(cCompilerOptionsEdit->text());
    myEnvironment->setLinkerOptions(linkerOptionsEdit->text());
    myEnvironment->setBuildUtility(buildUtilityEdit->text());
}

//======================================================================================================

void PreferencesForm::createMessagesTab()
{
    messagesTab = new QWidget;
    levelGroupBox = new QGroupBox(tr("Level"));
    messagesListBox = new QListWidget;
    connect(messagesListBox, SIGNAL(itemClicked(QListWidgetItem *)),
            this, SLOT(messageSelected(QListWidgetItem *)));
    defaultLevelButton = new QRadioButton(tr("default"));
    connect(defaultLevelButton, SIGNAL(clicked(bool)),
            this, SLOT(defaultLevelButtonClicked(bool)));
    errorLevelButton = new QRadioButton(tr("E"));
    connect(errorLevelButton, SIGNAL(clicked(bool)),
            this, SLOT(errorLevelButtonClicked(bool)));
    warningLevelButton = new QRadioButton(tr("W"));
    connect(warningLevelButton, SIGNAL(clicked(bool)),
            this, SLOT(warningLevelButtonClicked(bool)));
    informationLevelButton = new QRadioButton(tr("I"));
    connect(informationLevelButton, SIGNAL(clicked(bool)),
            this, SLOT(informationLevelButtonClicked(bool)));
    overflowLevelButton = new QRadioButton(tr("O"));
    connect(overflowLevelButton, SIGNAL(clicked(bool)),
            this, SLOT(overflowLevelButtonClicked(bool)));
    suppressLevelButton = new QRadioButton(tr("suppress"));
    connect(suppressLevelButton, SIGNAL(clicked(bool)),
            this, SLOT(suppressLevelButtonClicked(bool)));
    suppressAllCheck = new QCheckBox(tr("Suppress all except the redefined messages"));
    connect(suppressAllCheck, SIGNAL(clicked(bool)),
            this, SLOT(suppressAllClicked(bool)));
    QLabel *label = new QLabel(tr("Search message: number or text"));
    searchEdit = new QLineEdit();
    connect(searchEdit, SIGNAL(returnPressed()),
            this, SLOT(searchEditEntered()));
    showRedefinedCheck = new QCheckBox(tr("List redefined messages"));
    showRedefinedCheck->setChecked(false);
    connect(showRedefinedCheck, SIGNAL(clicked()),
            this, SLOT(showRedefinedClicked()));

    QHBoxLayout *hbox1 = new QHBoxLayout;
    hbox1->addWidget(defaultLevelButton);
    hbox1->addWidget(errorLevelButton);
    hbox1->addWidget(warningLevelButton);
    hbox1->addWidget(informationLevelButton);
    hbox1->addWidget(overflowLevelButton);
    hbox1->addWidget(suppressLevelButton);
    levelGroupBox->setLayout(hbox1);
    QHBoxLayout *hbox2 = new QHBoxLayout;
    hbox2->addWidget(label);
    hbox2->addWidget(searchEdit);
    hbox2->addWidget(showRedefinedCheck);
    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(messagesListBox);
    vboxt->addWidget(levelGroupBox);
    vboxt->addWidget(suppressAllCheck);
    vboxt->addLayout(hbox2);
    messagesTab->setLayout(vboxt);
}

void PreferencesForm::initializeMessages()
{
    if (!myCnfOptions)
        myCnfOptions = new CnfOptions(myEnvironment->getFckConfigurationFilename(),
                                      myEnvironment->getMsgFilename());
    messagesListBox->clear();
    rowSelected = -1;
    msgSelected = -1;
    searchRow = 0;
    levelGroupBox->setEnabled(false);
    char msg[LERM+1];
    QString line;
    for (int i=0; i<myCnfOptions->numMsg; ++i) {
        if (myCnfOptions->msgs[i].Level == ' ')
            continue;
        if (showRedefinedCheck->isChecked() && (!myCnfOptions->newLevel[i]))
            continue;
        line.setNum(i+1);
        line.append(" ");
        if (myCnfOptions->newLevel[i])
             line.append(myCnfOptions->levelFlags[myCnfOptions->newLevel[i]]);
        else
             line.append(myCnfOptions->msgs[i].Level);
        strncpy(msg,myCnfOptions->msgs[i].Message,LERM);
        msg[LERM]='\0';
        line.append(" ");
        line.append(msg);
        messagesListBox->addItem(line);
    }
    suppressAllCheck->setChecked(myCnfOptions->getSuppressAll());
}

void PreferencesForm::messageSelected(QListWidgetItem *item)
{
    messagesListBox->setCurrentItem(item);
    rowSelected = messagesListBox->currentRow();
    msgSelected = rowToMsgnum(rowSelected);
    presentMessageInfo(msgSelected);
    searchRow = rowSelected;
}

int PreferencesForm::rowToMsgnum(const int row)
{
    QListWidgetItem *item = messagesListBox->item(row);
    QString line = item->text();
    QString num = line.section(" ",0,0);
    return num.toInt() - 1;
}

void PreferencesForm::presentMessageInfo(int msgnum)
{
    levelGroupBox->setEnabled(true);
    switch (myCnfOptions->newLevel[msgnum]) {
    case 0:
        defaultLevelButton->setChecked(true);
        break;
    case 1:
        errorLevelButton->setChecked(true);
        break;
    case 2:
        warningLevelButton->setChecked(true);
        break;
    case 3:
        informationLevelButton->setChecked(true);
        break;
    case 4:
        overflowLevelButton->setChecked(true);
        break;
    case 5:
        suppressLevelButton->setChecked(true);
        break;
    }
}

void PreferencesForm::defaultLevelButtonClicked(bool checked)
{
    if (msgSelected >=0 ) {
        myCnfOptions->newLevel[msgSelected] = 0;
        initializeMessages();
    }
}

void PreferencesForm::errorLevelButtonClicked(bool checked)
{
    if (msgSelected >=0 ) {
        myCnfOptions->newLevel[msgSelected] = 1;
        initializeMessages();
    }
}

void PreferencesForm::warningLevelButtonClicked(bool checked)
{
    if (msgSelected >=0 ) {
        myCnfOptions->newLevel[msgSelected] = 2;
        initializeMessages();
    }
}

void PreferencesForm::informationLevelButtonClicked(bool checked)
{
    if (msgSelected >=0 ) {
        myCnfOptions->newLevel[msgSelected] = 3;
        initializeMessages();
    }
}

void PreferencesForm::overflowLevelButtonClicked(bool checked)
{
    if (msgSelected >=0 ) {
        myCnfOptions->newLevel[msgSelected] = 4;
        initializeMessages();
    }
}

void PreferencesForm::suppressLevelButtonClicked(bool checked)
{
    if (msgSelected >=0 ) {
        myCnfOptions->newLevel[msgSelected] = 5;
        initializeMessages();
    }
}

void PreferencesForm::suppressAllClicked(bool checked)
{
    myCnfOptions->setChanged(checked != myCnfOptions->getSuppressAll());
    myCnfOptions->setSuppressAll(checked);
}

void PreferencesForm::searchEditEntered()
{
    QListWidgetItem *item;
    int msgnum = searchEdit->text().toInt() - 1;
    if (msgnum >= 0 && msgnum < myCnfOptions->numMsg){
        for (int row=0; row<messagesListBox->count(); ++row) {
            if (rowToMsgnum(row) == msgnum){
                item = messagesListBox->item(row);
                messageSelected(item);
                return;
            }
        }
    }
    QString line;
    for (int row=searchRow; row<messagesListBox->count(); ++row) {
        item = messagesListBox->item(row);
        line = item->text();
        if (line.section(" ",2).contains(searchEdit->text(),Qt::CaseInsensitive)){
            messageSelected(item);
            return;
        }
    }
    QMessageBox::warning(this, Info::TITLE,
                         tr("Not found"),
                         QMessageBox::Ok);
}

void PreferencesForm::showRedefinedClicked()
{
    initializeMessages();
}

void PreferencesForm::applyChangesMessages()
{
    myCnfOptions->writeToFile();
}

//======================================================================================================

void PreferencesForm::createReportMessageTab()
{
    messageReportTab = new QWidget;
    QGroupBox *messageReportGroupBox = new QGroupBox(tr("Reporting messages"));

    QLabel *maxmsgLabel = new QLabel(tr("Maximum number of messages presented in a statement or list:"));
    maxMsgEdit = new QLineEdit;
    maxMsgEdit->setFixedWidth(50);
    QValidator *validator = new QIntValidator(1, 1000, this);
    maxMsgEdit->setValidator(validator);

    printSourceCheck = new QCheckBox(tr("Show source statement"));
    connect(printSourceCheck, SIGNAL(stateChanged(int)),
            this, SLOT(setPrintSourceCheckEnabled(int)));
    printLinStmNumCheck = new QCheckBox(tr("Show line/statement number of source statement"));

    QLabel *messageFormatLabel = new QLabel(tr("File/line format:"));
    fileLineFormatComboBox = new QComboBox();
    fileLineFormatComboBox->setEditable(true);
    fileLineFormatComboBox->addItem("('(file: ',a,', line: ',i0,')')");
    fileLineFormatComboBox->addItem("(a,':',i0,':')");

    QHBoxLayout *hbox1 = new QHBoxLayout;
    hbox1->addWidget(maxmsgLabel);
    hbox1->addStretch();
    hbox1->addWidget(maxMsgEdit);

    QHBoxLayout *hbox2 = new QHBoxLayout;
    hbox2->addWidget(messageFormatLabel);
    hbox2->addStretch();
    hbox2->addWidget(fileLineFormatComboBox);

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addLayout(hbox1);
    vbox->addWidget(printSourceCheck);
    vbox->addWidget(printLinStmNumCheck);
    vbox->addLayout(hbox2);

    messageReportGroupBox->setLayout(vbox);
    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(messageReportGroupBox);
    messageReportTab->setLayout(vboxt);
}

void PreferencesForm::initializeReportMessage()
{
    if (!myCnfOptions)
        myCnfOptions = new CnfOptions(myEnvironment->getFckConfigurationFilename(),
                                      myEnvironment->getMsgFilename());
    maxMsgEdit->setText(myCnfOptions->getMaxMsg());
    printSourceCheck->setChecked(myCnfOptions->getPrintSourceStm());
    printLinStmNumCheck->setChecked(myCnfOptions->getPrintLinStmNum());
    int ind = fileLineFormatComboBox->findText(myCnfOptions->getFileLineFormat());
    if (ind >= 0)
        fileLineFormatComboBox->setCurrentIndex(ind);
    else
        fileLineFormatComboBox->setEditText(myCnfOptions->getFileLineFormat());
    setPrintSourceCheckEnabled(printSourceCheck->checkState());
}

void PreferencesForm::setPrintSourceCheckEnabled(int state)
{
    printLinStmNumCheck->setEnabled(printSourceCheck->isChecked());
}

void PreferencesForm::applyChangesReportMessage()
{
    myCnfOptions->setChanged(
            fileLineFormatComboBox->currentText()!=myCnfOptions->getFileLineFormat() ||
            printSourceCheck->isChecked()!=myCnfOptions->getPrintSourceStm() ||
            printLinStmNumCheck->isChecked()!=myCnfOptions->getPrintLinStmNum());
    myCnfOptions->setMaxMsg(maxMsgEdit->text());
    myCnfOptions->setPrintSourceStm(printSourceCheck->isChecked());
    myCnfOptions->setPrintLinStmNum(printLinStmNumCheck->isChecked());
    myCnfOptions->setFileLineFormat(fileLineFormatComboBox->currentText());
    myCnfOptions->writeToFile();
}
//======================================================================================================

void PreferencesForm::createNumberingTab()
{
    numberingTab = new QWidget;
    QGroupBox *sourceNumberingGroupBox = new QGroupBox(tr("Source line/statement numbering"));
    linesButton = new QRadioButton(tr("&Lines"));
    statementsButton = new QRadioButton(tr("&Statements"));

    QGroupBox *restartNumberingGroupBox = new QGroupBox(tr("Restart numbering"));
    subButton = new QRadioButton(tr("Each &subprogram"));
    fileButton = new QRadioButton(tr("Each source input &file"));

    QGroupBox *includeNumberingGroupBox = new QGroupBox(tr("Included line/statement numbering"));
    hierButton = new QRadioButton(tr("&Hierarchical"));
    seqButton = new QRadioButton(tr("&Sequential"));

    QVBoxLayout *vbox1 = new QVBoxLayout;
    vbox1->addWidget(linesButton);
    vbox1->addWidget(statementsButton);
    vbox1->addStretch(1);
    sourceNumberingGroupBox->setLayout(vbox1);

    QVBoxLayout *vbox2 = new QVBoxLayout;
    vbox2->addWidget(subButton);
    vbox2->addWidget(fileButton);
    vbox2->addStretch(1);
    restartNumberingGroupBox->setLayout(vbox2);

    QVBoxLayout *vbox3 = new QVBoxLayout;
    vbox3->addWidget(hierButton);
    vbox3->addWidget(seqButton);
    vbox3->addStretch(1);
    includeNumberingGroupBox->setLayout(vbox3);

    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(sourceNumberingGroupBox);
    vboxt->addWidget(restartNumberingGroupBox);
    vboxt->addWidget(includeNumberingGroupBox);
    numberingTab->setLayout(vboxt);
}

//======================================================================================================

void PreferencesForm::createDateTimeFormatTab()
{
    dateTimeFormatTab = new QWidget;
    QGroupBox *dateTimeFormatGroupBox = new QGroupBox(tr("Date and time format"));

    QLabel *dateFormatLabel = new QLabel(tr("Date format:"));
    dateFormatComboBox = new QComboBox();
    dateFormatComboBox->addItem("yyyy-mm-dd");
    dateFormatComboBox->addItem("MMM-dd-yy");
    dateFormatComboBox->addItem("dd-mm-yyyy");
    dateFormatComboBox->addItem("mm-dd-yyyy");
    dateFormatComboBox->addItem("d-M-yyyy");
    dateFormatComboBox->addItem("xx xx xx");

    QLabel *timeFormatLabel = new QLabel(tr("Time format:"));
    timeFormatComboBox = new QComboBox();
    timeFormatComboBox->addItem("h:mm:ss");
    timeFormatComboBox->addItem("hh:mm:ss");
    timeFormatComboBox->addItem("xx xx xx");

    QLabel *memo = new QLabel(this);
    memo->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    memo->setText(tr("You can choose from a number of date and time formats.\n\n") +
                  tr("If you select the xx formats for date and time, date and time are suppressed in the listing."));

    QHBoxLayout *hbox1 = new QHBoxLayout;
    hbox1->addWidget(dateFormatLabel);
    hbox1->addWidget(dateFormatComboBox);

    QHBoxLayout *hbox2 = new QHBoxLayout;
    hbox2->addWidget(timeFormatLabel);
    hbox2->addWidget(timeFormatComboBox);

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addLayout(hbox1);
    vbox->addLayout(hbox2);
    vbox->addStretch(1);
    vbox->addWidget(memo);
    dateTimeFormatGroupBox->setLayout(vbox);

    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(dateTimeFormatGroupBox);
    dateTimeFormatTab->setLayout(vboxt);
}

//======================================================================================================

void PreferencesForm::createOutputTab()
{
    outputTab = new QWidget;
    QGroupBox *outputGroupBox = new QGroupBox(tr("Output preferences"));

    QVBoxLayout *vbox = new QVBoxLayout;
// header
    QLabel *emptyLabel = new QLabel(tr(" "));
    QLabel *col_label[NR_OUTPUT_STREAMS];

    QHBoxLayout *hbox1 = new QHBoxLayout;
    hbox1->addWidget(emptyLabel);
    for (int k = 0; k < NR_OUTPUT_STREAMS; ++k){
        col_label[k] = new QLabel(output_streams[k]);
        hbox1->addWidget(col_label[k]);
    }
    vbox->addLayout(hbox1);

//rows
    QHBoxLayout *hbox2[NR_OUTPUT_CAT];
    QLabel *row_label[NR_OUTPUT_CAT];
    for (int i = 0; i < NR_OUTPUT_CAT; ++i){
        hbox2[i] = new QHBoxLayout;
        row_label[i] = new QLabel(output_cat[i]);
        hbox2[i]->addWidget(row_label[i]);
        for (int k = 0; k < NR_OUTPUT_STREAMS; ++k){
            outputCheck[k][i] = new QCheckBox();
            hbox2[i]->addWidget(outputCheck[k][i]);
        }
        vbox->addLayout(hbox2[i]);
    }

    QLabel *memo = new QLabel(this);
    memo->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    memo->setText(tr("For each output stream you can select which output categories you like to view or store.\n"));

    vbox->addStretch(1);
    vbox->addWidget(memo);
    outputGroupBox->setLayout(vbox);

    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(outputGroupBox);
    outputTab->setLayout(vboxt);
}

//======================================================================================================

void PreferencesForm::initializeCnfOptions()
{
    if (!myCnfOptions)
        myCnfOptions = new CnfOptions(myEnvironment->getFckConfigurationFilename(),
                                      myEnvironment->getMsgFilename());
    linesButton->setChecked(myCnfOptions->getCountModeOption(0));
    statementsButton->setChecked(myCnfOptions->getCountModeOption(1));
    subButton->setChecked(myCnfOptions->getCountModeOption(2));
    fileButton->setChecked(myCnfOptions->getCountModeOption(3));
    hierButton->setChecked(myCnfOptions->getCountModeOption(4));
    seqButton->setChecked(myCnfOptions->getCountModeOption(5));

    dateFormatComboBox->
        setCurrentIndex(dateFormatComboBox->findText(myCnfOptions->getDateFormat()));
    timeFormatComboBox->
        setCurrentIndex(timeFormatComboBox->findText(myCnfOptions->getTimeFormat()));

    for (int k = 0; k < NR_OUTPUT_STREAMS; ++k)
        for (int i = 0; i < NR_OUTPUT_CAT; ++i)
        outputCheck[k][i]->setChecked(myCnfOptions->getOutputOption(k*NR_OUTPUT_CAT+i));
}

void PreferencesForm::applyChangesCnfOptions()
{
    myCnfOptions->setChanged(
        linesButton->isChecked()!=myCnfOptions->getCountModeOption(0) ||
        statementsButton->isChecked()!=myCnfOptions->getCountModeOption(1) ||
        subButton->isChecked()!=myCnfOptions->getCountModeOption(2) ||
        fileButton->isChecked()!=myCnfOptions->getCountModeOption(3) ||
        hierButton->isChecked()!=myCnfOptions->getCountModeOption(4) ||
        seqButton->isChecked()!=myCnfOptions->getCountModeOption(5) ||
        dateFormatComboBox->currentText()!=myCnfOptions->getDateFormat());
    myCnfOptions->setCountModeOption(0,linesButton->isChecked());
    myCnfOptions->setCountModeOption(1,statementsButton->isChecked());
    myCnfOptions->setCountModeOption(2,subButton->isChecked());
    myCnfOptions->setCountModeOption(3,fileButton->isChecked());
    myCnfOptions->setCountModeOption(4,hierButton->isChecked());
    myCnfOptions->setCountModeOption(5,seqButton->isChecked());

    myCnfOptions->setDateFormat(dateFormatComboBox->currentText());
    myCnfOptions->setTimeFormat(timeFormatComboBox->currentText());

    for (int k = 0; k < NR_OUTPUT_STREAMS; ++k)
        for (int i = 0; i < NR_OUTPUT_CAT; ++i){
            myCnfOptions->setChanged(outputCheck[k][i]->isChecked()!=myCnfOptions->getOutputOption(k*NR_OUTPUT_CAT+i));
            myCnfOptions->setOutputOption(k*NR_OUTPUT_CAT+i,outputCheck[k][i]->isChecked());
        }

    myCnfOptions->writeToFile();
}

//======================================================================================================

void PreferencesForm::createLibrariesTab()
{
    librariesTab = new QWidget;
    QGroupBox *librariesGroupBox = new QGroupBox(tr("Library settings"));
    updateCheck = new QCheckBox(tr("Update existing library file instead of creating a new one"));
    compressCheck = new QCheckBox(tr("Automatically compress updated library file after the analysis"));

    QVBoxLayout *vbox = new QVBoxLayout;
    vbox->addWidget(updateCheck);
    vbox->addWidget(compressCheck);
    vbox->addStretch(1);
    librariesGroupBox->setLayout(vbox);
    QVBoxLayout *vboxt = new QVBoxLayout;
    vboxt->addWidget(librariesGroupBox);
    librariesTab->setLayout(vboxt);
}

void PreferencesForm::initializeLibraries()
{
    updateCheck->setChecked(myEnvironment->getLibrariesPreferencesUpdate());
    compressCheck->setChecked(myEnvironment->getLibrariesPreferencesCompress());
}

void PreferencesForm::applyChangesLibraries()
{
    myEnvironment->setLibrariesPreferences(updateCheck->isChecked(),compressCheck->isChecked());
}

//======================================================================================================

void PreferencesForm::okClicked()
{
    applyChanges();
    close();
}

void PreferencesForm::applyClicked()
{
    applyChanges();
}

void PreferencesForm::helpClicked()
{
    HelpBrowser::HelpEntry helpentry;
    QWidget *widget = tabWidget->currentWidget();
    if (widget == directoriesTab)
        helpentry = HelpBrowser::HelpDirectories;
    else if (widget == extensionsTab)
        helpentry = HelpBrowser::HelpDefaultFileExtensions;
    else if (widget == emulationTab)
        helpentry = HelpBrowser::HelpCompilersSupported;
    else if (widget == editorTab)
        helpentry = HelpBrowser::HelpEditor;
    else if (widget == buildTab)
        helpentry = HelpBrowser::HelpBuild;
    else if (widget == messagesTab)
        helpentry = HelpBrowser::HelpRedefinitionAndSuppressionOfMessages;
    else if (widget == messageReportTab)
        helpentry = HelpBrowser::HelpMessageFormat;
    else if (widget == numberingTab)
        helpentry = HelpBrowser::HelpLineOrStatementNumbering;
    else if (widget == dateTimeFormatTab)
        helpentry = HelpBrowser::HelpDateFormat;
    else if (widget == librariesTab)
        helpentry = HelpBrowser::HelpUsingForcheckLibrariesInTheIDE;
    else
        helpentry = HelpBrowser::HelpMain;

    myEnvironment->helpBrowser->showHelp(helpentry);
}
