/*ForcheckIDE/PageSetup.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "PageSetup.h"

#include <QPrinterInfo>
#include <QPrinter>
#include <QRectF>
#include <QFont>
#include <QFontMetrics>
#include <QList>
#include <QPageSetupDialog>
#include <QPrintDialog>
#include <QFile>
#include <QTextDocument>
#include <QSettings>

PageSetup::PageSetup()
{
    printer = NULL;
    printer_name.clear();
    font_name = DEFAULT_FONT_NAME;
    font_size = DEFAULT_FONT_SIZE;
    margin_left = DEFAULT_MARGIN_LEFT;
    margin_right = DEFAULT_MARGIN_RIGHT;
    margin_top = DEFAULT_MARGIN_TOP;
    margin_bottom = DEFAULT_MARGIN_BOTTOM;
    wrap_long_lines = DEFAULT_WRAP_LONG_LINES;
    orientation = DEFAULT_ORIENTATION;
    paper_size = DEFAULT_PAPER_SIZE;
    charactersPerLine = 0;
    linesPerPage = 0;
}

void PageSetup::save(QSettings& ideConfSettings)
{
    ideConfSettings.setValue("Page_Setup/Printer", printer_name);
    ideConfSettings.setValue("Page_Setup/Font_name", font_name);
    ideConfSettings.setValue("Page_Setup/Font_size", font_size);
    ideConfSettings.setValue("Page_Setup/Margin_left", margin_left);
    ideConfSettings.setValue("Page_Setup/Margin_right", margin_right);
    ideConfSettings.setValue("Page_Setup/Margin_top", margin_top);
    ideConfSettings.setValue("Page_Setup/Margin_bottom", margin_bottom);
    ideConfSettings.setValue("Page_Setup/Wrap_long_lines", wrap_long_lines);
    ideConfSettings.setValue("Page_Setup/Orientation", orientation);
    ideConfSettings.setValue("Page_Setup/Paper_size", paper_size);
}

void PageSetup::restore(QSettings& ideConfSettings)
{
    printer_name = ideConfSettings.value("Page_Setup/Printer", "").toString();
    font_name = ideConfSettings.value("Page_Setup/Font_name", DEFAULT_FONT_NAME).toString();
    font_size = ideConfSettings.value("Page_Setup/Font_size", DEFAULT_FONT_SIZE).toInt();
    margin_left = ideConfSettings.value("Page_Setup/Margin_left", DEFAULT_MARGIN_LEFT).toInt();
    margin_right = ideConfSettings.value("Page_Setup/Margin_right", DEFAULT_MARGIN_RIGHT).toInt();
    margin_top = ideConfSettings.value("Page_Setup/Margin_top", DEFAULT_MARGIN_TOP).toInt();
    margin_bottom = ideConfSettings.value("Page_Setup/Margin_bottom", DEFAULT_MARGIN_BOTTOM).toInt();
    wrap_long_lines = ideConfSettings.value("Page_Setup/Wrap_long_lines", DEFAULT_WRAP_LONG_LINES).toBool();
    orientation = ideConfSettings.value("Page_Setup/Orientation", DEFAULT_ORIENTATION).toInt();
    paper_size = ideConfSettings.value("Page_Setup/Paper_size", DEFAULT_PAPER_SIZE).toInt();
}

void PageSetup::setupPrinter()
{
    QPrinterInfo printerinfo;
    printerinfo = QPrinterInfo::defaultPrinter();
    if (!printer_name.isEmpty()){
        QList<QPrinterInfo> list = QPrinterInfo::availablePrinters();
        int i;
        for (i = 0; i < list.size(); ++i)
            if (list.at(i).printerName() == printer_name) {
                printerinfo = list.at(i);
                break;
            }
    }
    if (!printerinfo.isNull()){
        printer_name = printerinfo.printerName();
        printer = new QPrinter(printerinfo);
        //set printer characteristics
        if (orientation == 2)
            printer->setOrientation(QPrinter::Landscape);
        qreal left, top, right, bottom;
        printer->getPageMargins(&left,&top,&right,&bottom,QPrinter::Millimeter);
        if (qreal(margin_left) > left) left = qreal(margin_left);
        if (qreal(margin_top) > top) top = qreal(margin_top);
        if (qreal(margin_right) > right) right = qreal(margin_right);
        if (qreal(margin_bottom) > bottom) bottom = qreal(margin_bottom);
        printer->setPageMargins(left, top, right, bottom, QPrinter::Millimeter);
        switch (paper_size) {
        case 1:
            printer->setPaperSize(QPrinter::Letter);
            break;
        case 9:
            printer->setPaperSize(QPrinter::A4);
            break;
        }
    }
}

//determine charactersPerLine and linesPerPage
void PageSetup::getPageSize()
{
    if (printer)
    {
        QRectF pagerect = printer->pageRect(QPrinter::Millimeter);
        QRectF paperrect = printer->paperRect(QPrinter::Millimeter);

        double printingwidth = paperrect.width() - qreal(margin_left - margin_right);
        if (printingwidth > pagerect.width())
            printingwidth = pagerect.width();

        double printingheight = paperrect.height() - qreal(margin_top - margin_bottom);
        if (printingheight > pagerect.height())
            printingheight = pagerect.height();

        QFont myfont(font_name, font_size);
        QFontMetrics metrics(myfont, printer);
        double charwidth = double(metrics.maxWidth()) / (double(printer->resolution())/25.4);
        double lineheight = double(metrics.lineSpacing()) / (double(printer->resolution())/25.4);
        charactersPerLine = int(printingwidth / charwidth);
        linesPerPage = int(printingheight / lineheight);
    }
}

void PageSetup::setupPage()
{
    QPageSetupDialog pageDialog(printer);
    if (pageDialog.exec()){
        if (printer)
        {
            QPrinterInfo info(*printer);
            printer_name = info.printerName();
            if (printer->orientation() == QPrinter::Portrait)
                orientation = 1;
            else
                orientation = 2;
            qreal left, top, right, bottom;
            printer->getPageMargins(&left,&top,&right,&bottom,QPrinter::Millimeter);
            margin_left = int(left+0.5);
            margin_top = int(top+0.5);
            margin_right = int(right+0.5);
            margin_bottom = int(bottom+0.5);
            switch (printer->paperSize()) {
            case QPrinter::Letter:
                paper_size = 1;
                break;
            case QPrinter::A4:
                paper_size = 9;
                break;
            default:
                paper_size = 9;
                break;
            }
        }
    }

}

void PageSetup::printFile(QFile &file)
{
    QPrintDialog printDialog(printer);
    if (printDialog.exec()){
        QString text=file.readAll();
        QTextDocument document(text);
        QFont font(font_name, font_size);
        document.setDefaultFont(font);
        document.print(printer);
    };
}
