/*ForcheckIDE/OptionsForm.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "OptionsForm.h"
#include "HelpBrowser.h"
#include "Environment.h"
#include "ListDialog.h"
#include "LibListDialog.h"
#include "FileKinds.h"

#include <QtWidgets>
#include <QStringList>
#include <QIntValidator>

OptionsForm::OptionsForm(QWidget *parent) : QDialog(parent)
{
    currentFilename.clear();
    currentOptionKind = OptionsKind::OptionsKindNone;
    currentOptions = 0;
    projOpts = 0;
    formCreate();
}

void OptionsForm::formCreate()
{
    ForEachOptionsGroupOnForm(optiongroup) {
        optionsGroups[optiongroup].optionGroupBox = new QGroupBox(OptionsGroupNames[optiongroup]);
        fillGroupBox(optiongroup);
    }

    helpButton = new QPushButton(tr("Help"));
    helpButton->setToolTip(tr("Present the options pages of the user guide"));
    helpButton->setFixedWidth(80);
    helpButton->setShortcut(QKeySequence::HelpContents);
    okButton = new QPushButton(tr("Ok"));
    okButton->setToolTip(tr("Accept the specified options and close the window"));
    okButton->setFixedWidth(80);
    cancelButton = new QPushButton(tr("Cancel"));
    cancelButton->setToolTip(tr("Cancel the specified options and close the window"));
    cancelButton->setFixedWidth(80);
    defaultButton = new QPushButton(tr("Default"));
    defaultButton->setToolTip(tr("Set all options to the defaults"));
    defaultButton->setFixedWidth(80);
    connect(helpButton, SIGNAL(clicked()), this, SLOT(helpButtonClicked()));
    connect(okButton, SIGNAL(clicked()), this, SLOT(okButtonClicked()));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(cancelButtonClicked()));
    connect(defaultButton, SIGNAL(clicked()), this, SLOT(defaultButtonClicked()));

    QVBoxLayout *column1Layout = new QVBoxLayout;
    column1Layout->addWidget(optionsGroups[OptionsGroup::ProgramUnitOptions].optionGroupBox);
    column1Layout->addWidget(optionsGroups[OptionsGroup::IntegerSizeOptions].optionGroupBox);

    QVBoxLayout *column2Layout = new QVBoxLayout;
    column2Layout->addWidget(optionsGroups[OptionsGroup::StandardConformanceOptions].optionGroupBox);
    column2Layout->addWidget(optionsGroups[OptionsGroup::ListingOptions].optionGroupBox);

    QVBoxLayout *column3Layout = new QVBoxLayout;
    column3Layout->addWidget(optionsGroups[OptionsGroup::GlobalAnalysisOptions].optionGroupBox);
    column3Layout->addWidget(optionsGroups[OptionsGroup::MiscOptions].optionGroupBox);
    column3Layout->addWidget(optionsGroups[OptionsGroup::RefLibOptions].optionGroupBox);

    QHBoxLayout *groupBoxLayout = new QHBoxLayout;
    groupBoxLayout->addLayout(column1Layout);
    groupBoxLayout->addLayout(column2Layout);
    groupBoxLayout->addLayout(column3Layout);

    QHBoxLayout *buttonsLayout = new QHBoxLayout;
    buttonsLayout->addWidget(helpButton);
    buttonsLayout->addWidget(okButton);
    buttonsLayout->addWidget(cancelButton);
    buttonsLayout->addWidget(defaultButton);

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addLayout(groupBoxLayout);
    mainLayout->addLayout(buttonsLayout);

    setLayout(mainLayout);
}

void OptionsForm::fillGroupBox(int optiongroup)
{
    const MTOptDef *options = OptionsGroupDefs[optiongroup];
    int numoptions = OptionsGroupDefsCount[optiongroup];

    typedef QLabel *pQLabel;
    optionsGroups[optiongroup].labels = new pQLabel[numoptions];
    typedef QLineEdit *pQLineEdit;
    optionsGroups[optiongroup].editfields = new pQLineEdit[numoptions];
    typedef QAbstractButton *pQAbstractButton;
    optionsGroups[optiongroup].buttons = new pQAbstractButton[numoptions];
    typedef QPushButton *pQPushButton;
    optionsGroups[optiongroup].listButtons = new pQPushButton[numoptions];
    typedef QStringList *pQStringList;
    optionsGroups[optiongroup].valuesList = new pQStringList[numoptions];

    buttonGroup[optiongroup] = new QButtonGroup;
    buttonGroup[optiongroup]->setExclusive(OptionsGroupRadioButtons[optiongroup]);
    connect(buttonGroup[optiongroup], SIGNAL(buttonClicked(QAbstractButton*)), this,
            SLOT(buttonClicked(QAbstractButton*)));
    listButtonGroup[optiongroup] = new QButtonGroup;
    listButtonGroup[optiongroup]->setExclusive(false);
    connect(listButtonGroup[optiongroup], SIGNAL(buttonClicked(QAbstractButton*)), this,
            SLOT(listButtonClicked(QAbstractButton*)));

    QVBoxLayout *vbox = new QVBoxLayout;
    for (int i = 0; i < numoptions; ++i){
        QHBoxLayout *buttonLayout = new QHBoxLayout;
        if (OptionsGroupDefs[optiongroup][i].Type == OINT){
            pQLineEdit *p = optionsGroups[optiongroup].editfields;
            p[i] = new QLineEdit;
            p[i]->setFixedWidth(50);
            QValidator *validator = new QIntValidator(options[i].minInt, options[i].maxInt, this);
            p[i]->setValidator(validator);
            pQLabel *pl = optionsGroups[optiongroup].labels;
            pl[i] = new QLabel(options[i].caption);
            buttonLayout->addWidget(pl[i]);
            buttonLayout->addStretch();
            buttonLayout->addWidget(p[i]);
        }
        else {
            if (OptionsGroupRadioButtons[optiongroup]){
                pQAbstractButton *p1 = optionsGroups[optiongroup].buttons;
                p1[i] = new QRadioButton(options[i].caption);
                buttonLayout->addWidget(p1[i]);
            }
            else {
                pQAbstractButton *p1 = optionsGroups[optiongroup].buttons;
                p1[i] = new QCheckBox(options[i].caption);
                buttonLayout->addWidget(p1[i]);
            }

            if (OptionsGroupDefs[optiongroup][i].Type == OLIST ||
                OptionsGroupDefs[optiongroup][i].Type == OSYMBLIST ||
                OptionsGroupDefs[optiongroup][i].Type == ODIRLIST ||
                OptionsGroupDefs[optiongroup][i].Type == OLIBLIST){
                pQAbstractButton *p1 = optionsGroups[optiongroup].buttons;
                buttonGroup[optiongroup]->addButton(p1[i]);
                pQPushButton *p = optionsGroups[optiongroup].listButtons;
                p[i] = new QPushButton("...");
                p[i]->setFixedWidth(25);
                buttonLayout->addStretch();
                buttonLayout->addWidget(p[i]);
                listButtonGroup[optiongroup]->addButton(p[i]);
            }
        }
        vbox->addLayout(buttonLayout);
    }
    optionsGroups[optiongroup].optionGroupBox->setLayout(vbox);
}

void OptionsForm::optionsToForm(const Options *opts)
{
    ForEachOptionsGroupOnForm(optiongroup) {
        int numoptions = OptionsGroupDefsCount[optiongroup];
        for (int i = 0; i < numoptions; ++i){
            Options::Option *effopt = opts->find(optiongroup,i)->getEffectiveOption();
            if (OptionsGroupDefs[optiongroup][i].Type == OINT){
                optionsGroups[optiongroup].labels[i]->
                    setEnabled(OptionsGroupDefs[optiongroup][i].Enabled[currentOptionKind]);
                optionsGroups[optiongroup].editfields[i]->
                    setEnabled(OptionsGroupDefs[optiongroup][i].Enabled[currentOptionKind]);
                int val = effopt->getInt();
                optionsGroups[optiongroup].editfields[i]->setText(QString::number(val));
            }
            else {
                optionsGroups[optiongroup].buttons[i]->
                    setEnabled(OptionsGroupDefs[optiongroup][i].Enabled[currentOptionKind]);
                optionsGroups[optiongroup].buttons[i]->
                    setChecked(effopt->getBool());
                if (OptionsGroupDefs[optiongroup][i].Type == OLIST ||
                    OptionsGroupDefs[optiongroup][i].Type == OSYMBLIST ||
                    OptionsGroupDefs[optiongroup][i].Type == ODIRLIST ||
                    OptionsGroupDefs[optiongroup][i].Type == OLIBLIST){
                    optionsGroups[optiongroup].listButtons[i]->
                        setEnabled(OptionsGroupDefs[optiongroup][i].Enabled[currentOptionKind] &&
                                   optionsGroups[optiongroup].buttons[i]->isChecked());
                    optionsGroups[optiongroup].valuesList[i] =
                        effopt->getList();
                }
            }
        }
    }
    cancelButton->setFocus();
}

void OptionsForm::optionsFromForm(Options *opts)
{
    ForEachOptionsGroupOnForm(optiongroup) {
        int numoptions = OptionsGroupDefsCount[optiongroup];
        bool checked;
        for (int i = 0; i < numoptions; ++i){
            Options::Option *opt = opts->find(optiongroup,i);
            Options::Option *effopt = opts->find(optiongroup,i)->getEffectiveOption();
            Options::Option *effparentopt = opts->parent->find(optiongroup,i)->getEffectiveOption();
            if (OptionsGroupDefs[optiongroup][i].Type == OINT){
                int val = optionsGroups[optiongroup].editfields[i]->text().toInt();
                if (val > 0 && val != effopt->getInt())
                    opt->setChanged(true);
                if (val == 0 || val == effparentopt->getInt()){
                    opt->setBool(TSB_default);
                    opt->setInt(0);
                }
                else {
                    opt->setBool(TSB_true);
                    opt->setInt(val);
                }
            }
            else {
                checked = optionsGroups[optiongroup].buttons[i]->isChecked();
                if (checked &&
                    (OptionsGroupDefs[optiongroup][i].Type == OSYMBLIST ||
                     OptionsGroupDefs[optiongroup][i].Type == ODIRLIST) &&
                    optionsGroups[optiongroup].valuesList[i]->count() == 0)
                    checked = false;
                QStringList *list = opt->getList();
                QStringList *efflist = effopt->getList();
                QStringList *effparentlist = effparentopt->getList();
                if ((effopt->getBool() != checked) ||
                    (checked && (*list != *efflist)))
                    opt->setChanged(true);
                if ((effparentopt->getBool() == checked) &&
                    (!checked || list->isEmpty() || (*list == *effparentlist)))
                    opt->setBool(TSB_default);
                else{
                    if (checked) {
                        opt->setBool(TSB_true);
                        if (OptionsGroupDefs[optiongroup][i].Type == OLIST ||
                            OptionsGroupDefs[optiongroup][i].Type == OSYMBLIST ||
                            OptionsGroupDefs[optiongroup][i].Type == ODIRLIST ||
                            OptionsGroupDefs[optiongroup][i].Type == OLIBLIST){
                                opt->setList(optionsGroups[optiongroup].valuesList[i]);
                            }
                    }
                    else
                        opt->setBool(TSB_false);
                }
            }
        }
    }
}

void OptionsForm::DefaultOptions(Options *opt, const QString& filename)
{
    currentFilename = filename;
    currentOptionKind = OptionsKind::OptionsKindDefault;
    currentOptions = opt;
    if (filename.isEmpty()) currentFilename = "<Unnamed>";
    optionsToForm(opt);
    setWindowTitle(tr("Default options - ") + currentFilename);
}

void OptionsForm::ProjectOptions(Options *opt, const QString& filename)
{
    currentFilename = filename;
    currentOptionKind = OptionsKind::OptionsKindProject;
    currentOptions = opt;
    if (filename.isEmpty()) currentFilename = "<Unnamed>";
    optionsToForm(opt);
    setWindowTitle(tr("Project options - ") + currentFilename);
}

void OptionsForm::SourceFileOptions(Options *opt, Options *projopts, const QString& filename)
{
    currentFilename = filename;
    currentOptionKind = OptionsKind::OptionsKindSource;
    currentOptions = opt;
    if (filename.isEmpty()) currentFilename = "<Unnamed>";
    projOpts = projopts;
    optionsToForm(opt);
    setWindowTitle(tr("Source file options - ") + currentFilename);
}

void OptionsForm::LibraryFileOptions(Options *opt, Options *projopts, const QString& filename)
{
    currentFilename = filename;
    currentOptionKind = OptionsKind::OptionsKindRefLib;
    currentOptions = opt;
    if (filename.isEmpty()) currentFilename = "<Unnamed>";
    projOpts = projopts;
    optionsToForm(opt);
    setWindowTitle(tr("Library file options - ") + currentFilename);
}

void OptionsForm::buttonClicked(QAbstractButton *button)
{
    OptionsGroup::ETOptionsGroup optiongroup;
    int i;
    findOpt(button, optiongroup, i);

    if (optiongroup >=0 && i >=0){
        optionsGroups[optiongroup].listButtons[i]->
            setEnabled(optionsGroups[optiongroup].buttons[i]->isChecked());
    };
}

void OptionsForm::listButtonClicked(QAbstractButton *listbutton)
{
    OptionsGroup::ETOptionsGroup optiongroup;
    int optionindex;
    findListOpt(listbutton, optiongroup, optionindex);
    const MTOptDef *options = OptionsGroupDefs[optiongroup];

    ListDialog *listDialog;
    LibListDialog *libListDialog;
    switch (OptionsGroupDefs[optiongroup][optionindex].Type){
    case OLIST:
        listDialog = new ListDialog(0, ListDialog::ListAllNone, HelpBrowser::HelpListingOptions);
        break;
    case OSYMBLIST:
        listDialog = new ListDialog(0, ListDialog::ListSymb, HelpBrowser::HelpMiscellaneousOptions);
        break;
    case ODIRLIST:
        listDialog = new ListDialog(0, ListDialog::ListDir, HelpBrowser::HelpMiscellaneousOptions);
        break;
    case OLIBLIST:
        libListDialog = new LibListDialog(0);
        libListDialog->setWindowTitle("Include specific program units");
        libListDialog->setLists(currentFilename, optionsGroups[optiongroup].valuesList[optionindex]);
        if (libListDialog->exec() == QDialog::Accepted)
            optionsGroups[optiongroup].valuesList[optionindex] = libListDialog->getLists();
        delete libListDialog;
        return;
    default:
        return;
    }

    listDialog->setWindowTitle(options[optionindex].caption);
    listDialog->setLists(optionsGroups[optiongroup].valuesList[optionindex]);
    if (listDialog->exec() == QDialog::Accepted)
        optionsGroups[optiongroup].valuesList[optionindex] = listDialog->getLists();
    delete listDialog;
}

void OptionsForm::findOpt(QAbstractButton *button, OptionsGroup::ETOptionsGroup &optiongroup, int &i)
{
    ForEachOptionsGroupOnForm(group){
        int numoptions = OptionsGroupDefsCount[group];
        for (i = 0; i < numoptions; ++i){
            if (optionsGroups[group].buttons[i] == button){
                optiongroup = group;
                return;
            }
        }
    }
    optiongroup = OptionsGroup::OptionsGroupNone;
    i = -1;
}

void OptionsForm::findListOpt(QAbstractButton *listbutton, OptionsGroup::ETOptionsGroup &optiongroup, int &i)
{
    ForEachOptionsGroupOnForm(group){
        int numoptions = OptionsGroupDefsCount[group];
        for (i = 0; i < numoptions; ++i){
            if (optionsGroups[group].listButtons[i] == listbutton){
                optiongroup = group;
                return;
            }
        }
    }
    optiongroup = OptionsGroup::OptionsGroupNone;
    i = -1;
}

void OptionsForm::helpButtonClicked()
{
    myEnvironment->helpBrowser->showHelp(HelpBrowser::HelpOptions);
}

void OptionsForm::okButtonClicked()
{
    if (currentOptions){
        optionsFromForm(currentOptions);
    }
    close();
}

void OptionsForm::defaultButtonClicked()
{
// Get options from higher in the hierarchy
    Options *poptions;
    switch (currentOptionKind) {
    case OptionsKind::OptionsKindSource:
        // source <- project
        poptions = new Options(projOpts, OptionsKind::OptionsKindSource);
        *poptions = *projOpts;
        if (FileKind::isFppFile(currentFilename))
            poptions->setBool(optCPP, TSB_true);
        if (FileKind::isDefaultFfFile(currentFilename))
            poptions->setBool(optFF, TSB_true);
        optionsToForm(poptions);
        delete poptions;
        break;
    case OptionsKind::OptionsKindRefLib:
        // reflib <- project
        optionsToForm(projOpts);
        break;
    case OptionsKind::OptionsKindProject:
        // project <- default
        optionsToForm(myEnvironment->getDefaultOptions());
        break;
    case OptionsKind::OptionsKindDefault:
        // default <- factory default
        optionsToForm(myEnvironment->getFactoryOptions());
        break;
    default:
        return;
    }
}

void OptionsForm::cancelButtonClicked()
{
    close();
}
