/*ForcheckIDE/Options.h*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#ifndef OPTIONS_H
#define OPTIONS_H

#include <QString>

class IniFile;
class QStringList;

enum TriStateBool {
    TSB_false,
    TSB_true,
    TSB_default};

class OptionsKind {
public:
        enum ETOptionsKind {
                OptionsKindNone = -1,
                OptionsKindFirst = 0,
                OptionsKindSource = OptionsKindFirst,
                OptionsKindCSource,
                OptionsKindInclude,
                OptionsKindRefLib,
                OptionsKindOutLib,
                OptionsKindListing,
                OptionsKindReport,
                OptionsKindRefStruct,
                OptionsKindModDep,
                OptionsKindProject,
                OptionsKindDefault,
                OptionsKindFactory,
                OptionsKindCount
        };
        static ETOptionsKind int2OptionsKind(int kindint)
        {
        #define TESTOPTIONSKIND(optionskind) if (optionskind == kindint) return optionskind;
                TESTOPTIONSKIND(OptionsKindSource)
                TESTOPTIONSKIND(OptionsKindCSource)
                TESTOPTIONSKIND(OptionsKindInclude)
                TESTOPTIONSKIND(OptionsKindRefLib)
                TESTOPTIONSKIND(OptionsKindOutLib)
                TESTOPTIONSKIND(OptionsKindListing)
                TESTOPTIONSKIND(OptionsKindReport)
                TESTOPTIONSKIND(OptionsKindRefStruct)
                TESTOPTIONSKIND(OptionsKindModDep)
                TESTOPTIONSKIND(OptionsKindProject)
                TESTOPTIONSKIND(OptionsKindDefault)
                TESTOPTIONSKIND(OptionsKindFactory)
                return OptionsKindNone;
        #undef TESTOPTIONSKIND
        }
        static ETOptionsKind nextOptionsKind(ETOptionsKind kind)
        {
            return int2OptionsKind(kind+1);
        }

#define ForEachOptionsKind(ok) \
        for (OptionsKind::ETOptionsKind ok = OptionsKind::KindFirst; ok != OptionsKind::KindNone; ok = OptionsKind::nextOptionsKind(ok))
};

inline bool IsOptionKindFile(OptionsKind::ETOptionsKind optionkind)
{
        return optionkind == OptionsKind::OptionsKindSource
            || optionkind == OptionsKind::OptionsKindRefLib
            || optionkind == OptionsKind::OptionsKindOutLib
            || optionkind == OptionsKind::OptionsKindListing
        ;

}

class OptionsGroup {
public:
    enum ETOptionsGroup {
            OptionsGroupNone = -1,
            OptionsGroupFirst = 0,
            ProgramUnitOptions = OptionsGroupFirst,
            StandardConformanceOptions,
            IntegerSizeOptions,
            ListingOptions,
            GlobalAnalysisOptions,
            MiscOptions,
            RefLibOptions,
            OutLibOptions,
            OptionsGroupCount,
            OptionsGroupCountNoLib = MiscOptions + 1,
            OptionsGroupCountOnForm = RefLibOptions + 1
    };
    static ETOptionsGroup int2OptionsGroup(int groupint)
    {
    #define TESTOPTIONSGROUP(optionsgroup) if (optionsgroup == groupint) return optionsgroup;
            TESTOPTIONSGROUP(ProgramUnitOptions)
            TESTOPTIONSGROUP(StandardConformanceOptions)
            TESTOPTIONSGROUP(IntegerSizeOptions)
            TESTOPTIONSGROUP(ListingOptions)
            TESTOPTIONSGROUP(GlobalAnalysisOptions)
            TESTOPTIONSGROUP(MiscOptions)
            TESTOPTIONSGROUP(RefLibOptions)
            TESTOPTIONSGROUP(OutLibOptions)
            return OptionsGroupNone;
    #undef TESTOPTIONSGROUP
    }
    static ETOptionsGroup nextOptionsGroup(ETOptionsGroup group)
    {
        return int2OptionsGroup(group+1);
    }

#define ForEachOptionsGroup(group) \
    for (OptionsGroup::ETOptionsGroup group = OptionsGroup::OptionsGroupFirst; group != OptionsGroup::OptionsGroupNone; group = OptionsGroup::nextOptionsGroup(group))

#define ForEachOptionsGroupNoLib(group) \
    for (OptionsGroup::ETOptionsGroup group = OptionsGroup::OptionsGroupFirst; group != OptionsGroup::OptionsGroupCountNoLib; group = OptionsGroup::nextOptionsGroup(group))

#define ForEachOptionsGroupOnForm(group) \
    for (OptionsGroup::ETOptionsGroup group = OptionsGroup::OptionsGroupFirst; group != OptionsGroup::OptionsGroupCountOnForm; group = OptionsGroup::nextOptionsGroup(group))

#define EmptySetOptionsGroups SetOptionsGroups()
};

extern const QString OptionsGroupNames[];
extern const bool OptionsGroupRadioButtons[];

enum OptionType {
        OBOOL, OINT, OLIST, OSYMBLIST, ODIRLIST, OLIBLIST
};

struct MTOptDef {
        const QString OptionName;
        const bool Enabled[OptionsKind::OptionsKindCount];
        const OptionType Type;
        const TriStateBool factoryValue;
        const int minInt;
        const int maxInt;
        const int factoryInt;
        const QString caption;
};

#if defined(Q_OS_WIN)
const QString PREopt= "/";
const QString NEGopt= "NO";
const QString DELopt = "=";
const QString SEPopt = ";";
const QString CNTopt = "+";
const QString NXTopt = ",";
const QString VALSEPopt = ";";

const QString optAC = "AC";
const QString optAQI = "AQI";
const QString optCN = "CN";
const QString optCPP = "CPP";
const QString optDC = "DC";
const QString optDE = "DE";
const QString optDP = "DP";
const QString optEX = "EX";
const QString optFF = "FF";
const QString optINTENT = "INTENT";
const QString optINTR = "INTR";
const QString optOB = "OB";
const QString optR8 = "R8";
const QString optRE = "RE";
const QString optSV = "SAVE";
const QString optSF = "SF";

const QString optNOST = "NOST";
const QString optST = "ST";
const QString optF77 = "F77";
const QString optF90 = "F90";
const QString optF95 = "F95";
const QString optF03 = "F03";
const QString optF08 = "F08";

const QString optI2 = "I2";
const QString optI4 = "I4";
const QString optI8 = "I8";

const QString optSB = "SB";
const QString optSS = "SS";
const QString optSH = "SH";
const QString optSI = "SI";
const QString optSR = "SRS";
const QString optSMD = "SMD";
const QString optSP = "SP";
const QString optSC = "SC";
const QString optSMT = "SMT";
const QString optSMV = "SMV";
const QString optPW = "PW";
const QString optPL = "PL";

const QString optAP = "AP";
const QString optCO = "CO";
const QString optAR = "AR";

const QString optRI = "RI";
const QString optTR = "TR";
const QString optINF = "INF";
const QString optWA = "WA";
const QString optLG = "LG";
const QString optIP = "IP";
const QString optDF = "DF";

const QString optIL = "IL";
const QString optCR = "CR";
const QString optUP = "UP";

const QString optBA = "BA";
const QString optMK = "MK";
const QString optID = "ID";
const QString optLI = "";
const QString optRP = "RP";
const QString optRSF = "RSF";
const QString optMDF = "MDF";
const QString optLI1 = "LI";
#elif defined(Q_OS_LINUX)
const QString PREopt= " -";
const QString NEGopt= "n";
const QString DELopt = " ";
const QString SEPopt = " ";
const QString CNTopt = " ";
const QString NXTopt = " ";
const QString VALSEPopt = ":";

const QString optAC = "allc";
const QString optAQI = "aqintf";
const QString optCN = "cntl";
const QString optCPP = "cpp";
const QString optDC = "decl";
const QString optDE = "cond";
const QString optDP = "dp";
const QString optEX = "ext";
const QString optFF = "ff";
const QString optOB = "obs";
const QString optR8 = "r8";
const QString optRE = "relax";
const QString optINTENT = "intent";
const QString optINTR = "intr";
const QString optSF = "spec";
const QString optSV = "save";

const QString optNOST = "nstandard";
const QString optST = "standard";
const QString optF77 = "f77";
const QString optF90 = "f90";
const QString optF95 = "f95";
const QString optF03 = "f03";
const QString optF08 = "f08";

const QString optI2 = "i2";
const QString optI4 = "i4";
const QString optI8 = "i8";

const QString optSB = "shsub";
const QString optSS = "shsrc";
const QString optSH = "shinc";
const QString optSI = "shsngl";
const QString optSR = "shref";
const QString optSMD = "shmoddep";
const QString optSP = "shprg";
const QString optSC = "shcom";
const QString optSMT = "shmodtyp";
const QString optSMV = "shmodvar";
const QString optPW = "pwid";
const QString optPL = "plen";

const QString optAP = "anprg";
const QString optCO = "ancmpl";
const QString optAR = "anref";

const QString optRI = "rigor";
const QString optTR = "trunc";
const QString optINF = "inf";
const QString optWA = "warn";
const QString optLG = "log";
const QString optIP = "I";
const QString optDF = "define";

const QString optIL = "incl";
const QString optCR = "cre";
const QString optUP = "upd";

const QString optBA = "batch";
const QString optMK = "M";
const QString optID = "idep";
const QString optLI = "l";
const QString optRP = "report";
const QString optRSF = "refstruct";
const QString optMDF = "moddep";
const QString optLI1 = "l";
#endif

#define SRCT true
#define SRCF false
#define CSRCT true
#define CSRCF false
#define INCT true
#define INCF false
#define OUTLIBT true
#define OUTLIBF false
#define REFLIBT true
#define REFLIBF false
#define LSTT true
#define LSTF false
#define REPT true
#define REPF false
#define REFT true
#define REFF false
#define MODF false
#define PRJT true
#define PRJF false
#define DEFT true
#define DEFF false
#define FACT true
#define FACF false

extern const MTOptDef *OptionsGroupDefs[OptionsGroup::OptionsGroupCount];
extern const int OptionsGroupDefsCount[OptionsGroup::OptionsGroupCount];

class Options
{
public:

//==========================================================================
class Option
{
public:
        Option(Options *parentopt,
               const OptionsGroup::ETOptionsGroup group,
               int index,
               const MTOptDef *dataentry);
        Option(const Option& opt);
        ~Option();
        void save(IniFile& conf);
        void load(IniFile& conf);
        
        Option& operator=(const Option& opt);
        
        Option* getEffectiveOption() const;
        int getInt() const;
        int getIntEffective() const;
        void setInt(int value);
        TriStateBool getBoolEffective() const;
        TriStateBool getBool() const;
        void setBool(TriStateBool value, bool change=true);
        void setBool(bool value, bool change=true);
        QStringList *getList() const;
        void setList(QStringList* list);
        QString toCommandString(const bool &selective) const;
        bool isChanged() const {return changed;}
        void setChanged(bool change) {changed = change;}

        struct {
                TriStateBool TSB;
                int Int;
                QStringList *List;
        } Value;
        OptionsGroup::ETOptionsGroup Group;
        const MTOptDef *OptDef;
        int Index;

private:
        void create();
        void assign(const Option& opt);

        Options *Parent;
        bool changed;
};
//==========================================================================
        typedef Option *pOption;

        Options(Options *parentopts = 0,
                const OptionsKind::ETOptionsKind optionkind = OptionsKind::OptionsKindNone);
        Options(const Options& opts);
        ~Options();

        Options& operator=(const Options& opts);

        static Options *getEmptyOptions();
        static int getNumOptions();

        bool isChanged() const;
        void clearChanged();

        void save(IniFile &conf);
        void load(IniFile &conf);

        TriStateBool getBoolEffective(const QString& OptionName) const;
        int getInt(const QString& OptionName) const;
        int getIntEffective(const QString& OptionName) const;
        void setInt(const QString& OptionName, int value);
        void setBool(const QString& OptionName, bool value, bool change = true);
        QStringList* getList(const QString& OptionName) const;
        void setList(const QString& OptionName, QStringList* List);

        void setSelective(bool sel) { selective = sel; }
        bool isSelective() { return selective; }

        void setUpdateLibOption();
        void setCreateLibOption();

        Option *find(int optiongroup, int optionindex);
        const Option *find(int optiongroup, int optionindex) const;

        QString toCommandString(const QString& optionName) const;
        QString toCommandString() const;

        Options *parent;
        OptionsKind::ETOptionsKind kind;

private:
        void create();
        void assign(const Options& opts);
        int find(const QString& optionName) const;
        QString toCommandStringGroup(const int &optionsGroup) const;

        pOption *theOptions;
        bool selective;
};

#endif
