/*ForcheckIDE/Options.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "Options.h"
#include "IniFile.h"

#include <QStringList>
#include <QDir>

const QString OptionsGroupNames[] = {
    "Program-unit analysis",
    "Standard conformance",
    "Default integer/logical size",
    "Listing",
    "Global analysis",
    "Miscellaneous",
    "Reference library",
    "Create/update Library"};

const bool OptionsGroupRadioButtons[] = {
    false,
    true,
    true,
    false,
    false,
    false,
    false,
    false};

static const MTOptDef ProgramUnitOptionsDefs[] = {
{       optAC,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Analyze all columns" },
{       optAQI, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Use previously acquired interfaces" },
{       optCN,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OINT,  TSB_false, 19,999,19 ,"Max # continuation lines" },
{       optCPP, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Enable cpp preprocessing" },
{       optDC,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Flag implicitly typed entities" },
{       optDE,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Process D-lines" },
{       optEX,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Flag undeclared external subprograms" },
{       optFF,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Free source form" },
{       optOB,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Flag obsolescent syntax" },
{       optRE,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Relax type checking" },
{       optINTENT,{SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Flag dummy arguments without INTENT" },
{       optINTR,{SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Flag undeclared intrinsics" },
{       optSF,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Flag specific intrinsic functions" },
{       optR8,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Default REAL(8)" },
{       optDP,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Default DOUBLE PRECISION" },
{       optSV,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Save all variables by default" }
};

static const MTOptDef StandardConformanceOptionsDefs[] = {
{       optNOST,{SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Allow syntax extensions" },
{       optST,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Conform to Fortran standard in effect" },
{       optF77, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Fortran 77 conformance" },
{       optF90, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Fortran 90 conformance" },
{       optF95, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Fortran 95 conformance" },
{       optF03, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Fortran 2003 conformance" },
{       optF08, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Fortran 2008 conformance" }
};

static const MTOptDef IntegerSizeOptionsDefs[] = {
{       optI2,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "2 bytes" },
{       optI4,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "4 bytes" },
{       optI8,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "8 bytes" }
};

static const MTOptDef ListingOptionsDefs[] = {
{       optSB,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Program unit cross-reference" },
{       optSS,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "List source lines" },
{       optSH,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "List included lines" },
{       optSI,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "List unreferenced entities" },
{       optSR,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OLIST, TSB_true,  0,0,0, "Reference structure" },
{       optSMD, {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OLIST, TSB_true,  0,0,0, "Module dependencies" },
{       optSP,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Program cross-reference" },
{       optSC,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OLIST, TSB_false, 0,0,0, "Cross-reference of common-block objects" },
{       optSMT, {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OLIST, TSB_false, 0,0,0, "Cross-reference of module derived types" },
{       optSMV, {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTT,REPF,REFF,MODF,PRJT,DEFT,FACF}, OLIST, TSB_false, 0,0,0, "Cross-reference of module data" },
{       optPW,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OINT,  TSB_false, 60,256,100,"Page width" },
{       optPL,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OINT,  TSB_false, 20,999,62,"Page length" }
};

static const MTOptDef GlobalAnalysisOptionsDefs[] = {
{       optAP,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Verify program consistency" },
{       optCO,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Flag unreferenced global entities" },
{       optAR,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Analyze reference structure" }
};

static const MTOptDef MiscOptionsDefs[] = {
{       optRI,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Rigorous syntax analysis" },
{       optTR,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Truncate names to 6 chars" },
{       optINF, {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Show informative messages" },
{       optWA,  {SRCT,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_true,  0,0,0, "Show warnings" },
{       optLG,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OBOOL, TSB_false, 0,0,0, "Log internal table usage" },
{       optIP,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, ODIRLIST, TSB_false, 0,0,0, "Search paths for include files" },
{       optDF,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBF,LSTF,REPF,REFF,MODF,PRJT,DEFT,FACF}, OSYMBLIST, TSB_false, 0,0,0, "Define meta-symbols for conditional compilation" }
};

static const MTOptDef ReferenceLibraryOptionsDefs[] = {
{       optIL,  {SRCF,CSRCF,INCF,REFLIBT,OUTLIBF,LSTF,REPF,REFF,MODF,PRJF,DEFF,FACF}, OLIBLIST, TSB_false, 0,0,0, "Include all or specific program units" }
};

static const MTOptDef OutputLibraryOptionsDefs[] = {
{       optCR,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBT,LSTF,REPF,REFF,MODF,PRJF,DEFF,FACF}, OBOOL, TSB_false, 0,0,0, "Create new library" },
{       optUP,  {SRCF,CSRCF,INCF,REFLIBF,OUTLIBT,LSTF,REPF,REFF,MODF,PRJF,DEFF,FACF}, OBOOL, TSB_false, 0,0,0, "Update library" },
};

const MTOptDef *OptionsGroupDefs[OptionsGroup::OptionsGroupCount] = {
        ProgramUnitOptionsDefs,
        StandardConformanceOptionsDefs,
        IntegerSizeOptionsDefs,
        ListingOptionsDefs,
        GlobalAnalysisOptionsDefs,
        MiscOptionsDefs,
        ReferenceLibraryOptionsDefs,
        OutputLibraryOptionsDefs
};

const int OptionsGroupDefsCount[OptionsGroup::OptionsGroupCount] = {
        sizeof(ProgramUnitOptionsDefs)/sizeof(MTOptDef),
        sizeof(StandardConformanceOptionsDefs)/sizeof(MTOptDef),
        sizeof(IntegerSizeOptionsDefs)/sizeof(MTOptDef),
        sizeof(ListingOptionsDefs)/sizeof(MTOptDef),
        sizeof(GlobalAnalysisOptionsDefs)/sizeof(MTOptDef),
        sizeof(MiscOptionsDefs)/sizeof(MTOptDef),
        sizeof(ReferenceLibraryOptionsDefs)/sizeof(MTOptDef),
        sizeof(OutputLibraryOptionsDefs)/sizeof(MTOptDef)
};

//==========================================================================

Options::Option::Option(Options *parentopt,
                        const OptionsGroup::ETOptionsGroup group,
                        int index,
                        const MTOptDef *optdef)
{
        Parent = parentopt;
        Group = group;
        Index = index;
        OptDef = optdef;
        create();
}

Options::Option::Option(const Option& opt)
{
        create();
        if (&opt != this)
            assign(opt);
}

Options::Option::~Option()
{
        delete Value.List;
}

void Options::Option::create()
{
        Value.TSB = TSB_default;
        Value.Int = 0;
        Value.List = new QStringList();
        changed = false;
}

Options::Option& Options::Option::operator=(const Options::Option& opt)
{
        if (&opt != this)
                assign(opt);
        return *this;
}

void Options::Option::assign(const Option& opt)
{
        Parent = opt.Parent;
        Group = opt.Group;
        Index = opt.Index;
        OptDef = opt.OptDef;
        Value.TSB = opt.Value.TSB;
        Value.Int = opt.Value.Int;
        *Value.List = *opt.Value.List;
}

void Options::Option::save(IniFile& conf)
{
        changed = false;
        if (Value.TSB == TSB_default) return;

        switch (OptDef->Type) {
        case OBOOL:
            conf.setValue(OptDef->OptionName, Value.TSB==TSB_true);
            break;
        case OINT:
            if (Value.TSB == TSB_true)
                conf.setValue(OptDef->OptionName, Value.Int);
            break;
        case OLIST:
        case OSYMBLIST:
        case ODIRLIST:
        case OLIBLIST:
            if (Value.TSB == TSB_true)
                conf.setValue(OptDef->OptionName, Value.List->join(";"));
            else
                conf.setValue(OptDef->OptionName, "<none>");
            break;
        }
}

void Options::Option::load(IniFile& conf)
{
        changed = false;
        switch (OptDef->Type) {
        case OBOOL:
            if (conf.contains(OptDef->OptionName))
                Value.TSB = conf.value(OptDef->OptionName).toInt() ? TSB_true : TSB_false;
            else {
                Value.TSB = TSB_default;
                Value.List->clear();
            }
            break;
        case OINT:
            if (conf.contains(OptDef->OptionName)) {
                Value.TSB = TSB_true;
                Value.Int = conf.value(OptDef->OptionName,0).toInt();
                if (Value.Int <= 0) {
                    Value.TSB = TSB_default;
                    Value.Int = 0;
                }
            }
            else {
                Value.TSB = TSB_default;
                Value.Int = 0;
            }
            break;
        case OLIST:
        case OSYMBLIST:
        case ODIRLIST:
        case OLIBLIST:
            if (conf.contains(OptDef->OptionName)) {
                if (conf.value(OptDef->OptionName).toString() == "<none>")
                    Value.TSB = TSB_false;
                else {
                    Value.TSB = TSB_true;
                    QString list = conf.value(OptDef->OptionName).toString();
                    if (!list.isEmpty())
                        Value.List->append(list.split(";"));
                }
            }
            else {
                Value.TSB = TSB_default;
                Value.List->clear();
            }
            break;
        }
}

Options::Option *Options::Option::getEffectiveOption() const
{
        TriStateBool value = Value.TSB;
        if (value != TSB_default) return Parent->find(Group, Index);
        Options *opts = Parent;         //this options set
        while (opts->parent){
            opts = opts->parent;
            value = opts->find(Group, Index)->Value.TSB;
            if (value != TSB_default) return opts->find(Group, Index);
        };
        return Parent->find(Group, Index);
}

int Options::Option::getInt() const
{
        return Value.Int;
}

int Options::Option::getIntEffective() const
{
    if (Value.Int > 0)
        return Value.Int;
    Options *opts = Parent;         //this options set
    while (opts->parent){
        opts = opts->parent;
        Option *opt = opts->find(Group, Index);
        if (opt->Value.Int > 0)
            return opt->Value.Int;
    };
    return 0;
}

void Options::Option::setInt(int value)
{
        if (Value.Int != value)
                changed = true;
        Value.Int = value;
}

TriStateBool Options::Option::getBoolEffective() const
{
    if (Value.TSB != TSB_default)
        return Value.TSB;
    Options *opts = Parent;         //this options set
    while (opts->parent){
        opts = opts->parent;
        Option *opt = opts->find(Group, Index);
        if (opt->Value.TSB != TSB_default)
            return opt->Value.TSB;
    };
    return TSB_default;
}

TriStateBool Options::Option::getBool() const
{
        return Value.TSB;
}

void Options::Option::setBool(TriStateBool value, bool change)
{
        if (Value.TSB != value)
                if (change) changed = true;
        Value.TSB = value;
}

void Options::Option::setBool(bool value, bool change)
{
        if (value){
            if (change && (Value.TSB != TSB_true))
                changed = true;
            Value.TSB = TSB_true;
        }
        else{
            if (change && (Value.TSB != TSB_false))
                changed = true;
            Value.TSB = TSB_false;
        }
}

QStringList * Options::Option::getList() const
{
        return Value.List;
}

void Options::Option::setList(QStringList* List)
{
        if (*Value.List != *List) changed = true;
        *Value.List=*List;
}

QString Options::Option::toCommandString(const bool &selective) const
{
        if (IsOptionKindFile(Parent->kind) && Value.TSB == TSB_default)
            return QString();
        if (QString(OptDef->OptionName).startsWith(NEGopt))
            return QString();

        QString string = PREopt;
        if (selective && QString(OptDef->OptionName) == optCO) {
            string += NEGopt;
            string += QString(OptDef->OptionName);
            return string;
        }
        else if (Value.TSB == TSB_default)
            return QString();

        switch (OptDef->Type) {
        case OBOOL:
            if (Value.TSB == TSB_false)
                string += NEGopt;
            string += QString(OptDef->OptionName);
            break;
        case OINT:
            if (getInt() == 0)
                return QString();
            string += QString(OptDef->OptionName) + DELopt + QString("%1").arg(getInt());
            break;
        case OLIST:
        case OSYMBLIST:
        case ODIRLIST:
        case OLIBLIST:
            if (Value.TSB == TSB_false || Value.List->join(SEPopt) == "<none>")
                string += NEGopt;
            string += QString(OptDef->OptionName);
            if (Value.TSB == TSB_false ||
                Value.List->join(SEPopt) == "<all>" ||
                Value.List->join(SEPopt) == "<none>")
                break;
            if (Value.List->count() > 0) {
                string += DELopt;
                QString list = Value.List->join(VALSEPopt);
                if (OptDef->Type == ODIRLIST)
                    list = QDir::toNativeSeparators(list);
                string += list;
            }
            break;
        }
        return string;
}

//==========================================================================

Options::Options(Options *parentopts, const OptionsKind::ETOptionsKind optionkind)
{
        parent = parentopts;
        kind = optionkind;
        selective = false;
        create();
}

Options::Options(const Options& opts)
{
        create();
        if (&opts != this)
            assign(opts);
}

Options::~Options()
{
    if (theOptions) {
        for (int i = 0; i < getNumOptions(); ++i)
                delete theOptions[i];
        delete[] theOptions;
    }
    theOptions = NULL;
}

void Options::create()
{
        theOptions = new pOption[getNumOptions()];
        int i = 0;
        ForEachOptionsGroup(optiongroup) {
                for (int optionindex = 0; optionindex < OptionsGroupDefsCount[optiongroup]; ++optionindex) {
                        theOptions[i] = new Option(this,optiongroup,optionindex,&OptionsGroupDefs[optiongroup][optionindex]);
                        ++i;
                }
        }
        if (kind == OptionsKind::OptionsKindFactory){
            for (int i = 0; i < getNumOptions(); ++i) {
                theOptions[i]->Value.TSB = theOptions[i]->OptDef->factoryValue;
                theOptions[i]->Value.Int = theOptions[i]->OptDef->factoryInt;
                theOptions[i]->Value.List->clear();
            }
        }
}

Options& Options::operator=(const Options& opts)
{
        if (&opts != this)
                assign(opts);
        return *this;
}

void Options::assign(const Options& opts)
{
        parent = opts.parent;
        kind = opts.kind;
        selective = opts.selective;
        for (int i = 0; i < getNumOptions(); ++i) {
                Option *p = opts.theOptions[i];
                *theOptions[i] = *p;
                theOptions[i]->setChanged(false);
        }
}

int Options::getNumOptions()
{
        int NumOptions = 0;
        ForEachOptionsGroup(optiongroup)
                NumOptions += OptionsGroupDefsCount[optiongroup];
        return NumOptions;
}

Options* Options::getEmptyOptions()
{
        static Options EmptyOptions;
        return &EmptyOptions;
}

bool Options::isChanged() const
{
        for (int i = 0; i < getNumOptions(); ++i)
            if (theOptions[i]->isChanged())
                return true;
        return false;
}

void Options::clearChanged()
{
        for (int i = 0; i < getNumOptions(); ++i)
            theOptions[i]->setChanged(false);
}

void Options::save(IniFile& conf)
{
        for (int i = 0; i < getNumOptions(); ++i) {
                theOptions[i]->save(conf);
        }
}

void Options::load(IniFile& conf)
{
    for (int i = 0; i < getNumOptions(); ++i){
        if (theOptions[i]->OptDef->Enabled[kind])
            theOptions[i]->load(conf);
    }
}

int Options::find(const QString& OptionName) const
{
        for (int i = 0; i < getNumOptions(); ++i)
            if (QString::compare(theOptions[i]->OptDef->OptionName,OptionName) == 0)
                return i;
        return -1;
}

Options::Option *Options::find(int optiongroup, int optionindex)
{
        for (int i = 0; i < getNumOptions(); ++i)
                if (theOptions[i]->Group == optiongroup && theOptions[i]->Index == optionindex)
                        return theOptions[i];
        return NULL;
}

const Options::Option *Options::find(int optiongroup, int optionindex) const
{
        for (int i = 0; i < getNumOptions(); ++i){
                if (theOptions[i]->Group == optiongroup && theOptions[i]->Index == optionindex){
                        return theOptions[i];}
                }
        return NULL;
}

TriStateBool Options::getBoolEffective(const QString& OptionName) const
{
        int i = find(OptionName);
        if (i < 0) return TSB_default;
        return theOptions[i]->getBoolEffective();
}


int Options::getInt(const QString& OptionName) const
{
        int i = find(OptionName);
        if (i < 0) return 0;
        return theOptions[i]->getInt();
}

int Options::getIntEffective(const QString& OptionName) const
{
        int i = find(OptionName);
        if (i < 0) return 0;
        return theOptions[i]->getIntEffective();
}

void Options::setInt(const QString& optionName, int value)
{
        int i = find(optionName);
        if (i < 0) return;
        theOptions[i]->setInt(value);
}

void Options::setBool(const QString& optionName, bool value, bool change)
{
        int i = find(optionName);
        if (i < 0) return;
        theOptions[i]->setBool(value, change);
}

QStringList* Options::getList(const QString& optionName) const
{
        int i = find(optionName);
        if (i < 0)
            return NULL;
        return theOptions[i]->getList();
}

void Options::setList(const QString& optionName, QStringList* List)
{
        int i = find(optionName);
        if (i < 0) return;
        theOptions[i]->setList(List);
}

void Options::setUpdateLibOption()
{
        setBool(optCR, TSB_default, false);
        setBool(optUP, TSB_true, false);
}

void Options::setCreateLibOption()
{
        setBool(optCR, TSB_true, false);
        setBool(optUP, TSB_default, false);
}

QString Options::toCommandString(const QString& optionName) const
{
        if (optionName.isEmpty())
            return QString();
        int i = find(optionName);
        if (i < 0)
            return QString();
        return theOptions[i]->toCommandString(selective);
}

QString Options::toCommandString() const
{
        QString string = QString();
        ForEachOptionsGroup(optiongroup){
            string += toCommandStringGroup(optiongroup);
        }
        return string;
}

QString Options::toCommandStringGroup(const int &optionsgroup) const
{
        QString string = QString();
        for (int i = 0; i < getNumOptions(); ++i)
            if (theOptions[i]->Group == optionsgroup){
                string += theOptions[i]->toCommandString(selective);
            }
        return string;
}
