/*ForcheckIDE/NewRelease.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "NewRelease.h"
#include "MainWindow.h"
#include "Environment.h"
#include "Info.h"
#include "Version.h"
#include "IniFile.h"

#include <QtWidgets>
#include <QFile>
#include <QUrl>
#include <QTimer>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>

NewRelease::NewRelease(QMainWindow *parent) :
    QObject(parent)
{
    myParent = parent;
}

void NewRelease::checkForNewRelease()
{
    if (myParent)
        myParent->statusBar()->showMessage(tr("Checking for updates"));
    httpAborted = false;

    manager = new QNetworkAccessManager;
    timer = new QTimer;
    connect(timer, SIGNAL(timeout()), this, SLOT(httpTimeout()));
    timer->setSingleShot(true);
    timer->start(4000);

    connect(manager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(downloadFinished(QNetworkReply*)));
    QUrl url(Info::VERSION_URL);
    QNetworkRequest request(url);
    QNetworkReply *reply = manager->get(request);
}

void NewRelease::httpTimeout()
{
    httpAborted = true;
    delete timer;
    delete manager;
    if (myParent)
        myParent->statusBar()->showMessage(QString());
    QMessageBox::warning(myParent, Info::TITLE,
                         tr("Version info file could not be retrieved (time out)."),
                         QMessageBox::Ok);
}

void NewRelease::downloadFinished(QNetworkReply *reply)
{
    if (httpAborted) return;
    timer->stop();
    delete timer;
    if (reply->error()) {
        QMessageBox::warning(myParent, Info::TITLE,
                             reply->errorString()+tr(". Could not retrieve version info file."),
                             QMessageBox::Ok);
    } else {
        saveVersionFile(reply);
        compareVersions();
        versionfile->remove();
        delete versionfile;
    }
    if (myParent)
        myParent->statusBar()->showMessage(QString());
}

void NewRelease::saveVersionFile(QIODevice *data)
{
    QString versionfilename = QDir::tempPath() + "/versioninfo.txt";
    versionfile = new QFile(versionfilename);
    versionfile->close();       //just in case
    versionfile->remove();      //just in case
    if (!versionfile->open(QIODevice::WriteOnly)){
        QMessageBox::warning(myParent, Info::TITLE,
                             versionfilename + tr(". Could not create temporary version info file."),
                             QMessageBox::Ok);
        return;
    }
    versionfile->write(data->readAll());
    versionfile->close();
}

void NewRelease::compareVersions()
{
    if (!versionfile->exists()){
        QMessageBox::warning(myParent, Info::TITLE,
                             versionfile->fileName()+tr(". Temporary version info file not found."),
                             QMessageBox::Ok);
        return;
    };

    IniFile latestVersionSettings(versionfile->fileName());
    if (!latestVersionSettings.open()){
        QMessageBox::warning(myParent, Info::TITLE,
                             versionfile->fileName()+tr(". Could not open temporary version info file."),
                             QMessageBox::Ok);
        return;
    };

// For the IDE

    QString keyIDE = Info::IMPLEMENTATION;
    keyIDE += "_IDE";
    keyIDE += "_" + QString::number(QSysInfo::WordSize);
    latestVersionSettings.beginGroup("version");
      QString strLatestIDEVersion = latestVersionSettings.value(keyIDE, "Error").toString();
      if (strLatestIDEVersion == "Error"){
          QMessageBox::warning(myParent, Info::TITLE,
                               tr("Unable to retrieve latest version information of the IDE."),
                               QMessageBox::Ok);
          return;
      }
    latestVersionSettings.endGroup();

    Version latestideversion = Version(strLatestIDEVersion);
    Version IDEversion = Version(Info::VERSION);
    //compare IDE versions and download if needed
    if (IDEversion.isNewerRelease(latestideversion)){
      if (QMessageBox::information(myParent, Info::TITLE,
                                   tr("A new release of the IDE is available. Click OK to download."),
                                   QMessageBox::Ok, QMessageBox::Cancel) == QMessageBox::Ok) {
        latestVersionSettings.beginGroup("url");
        QString strDownloadURL = latestVersionSettings.value(keyIDE, "Error").toString();
        latestVersionSettings.endGroup();
        if (strDownloadURL == "Error"){
          QMessageBox::warning(myParent, Info::TITLE,
                               tr("Unable to retrieve download url for IDE."),
                               QMessageBox::Ok);
        }
        else {
          myEnvironment->wwwBrowser(strDownloadURL);
#if defined(Q_OS_WIN)
          return;
#endif
        }
      }
    }
    else{
      QMessageBox::information(myParent, Info::TITLE,
                               tr("You have the latest release of the IDE installed."),
                               QMessageBox::Ok);
    }

// For the analyser

    QString key(Info::IMPLEMENTATION);
    key += "_" + QString::number(QSysInfo::WordSize);
    latestVersionSettings.beginGroup("version");
      QString strLatestFCKVersion = latestVersionSettings.value(key, "Error").toString();
      if (strLatestFCKVersion == "Error"){
          QMessageBox::warning(myParent, Info::TITLE,
                               tr("Unable to retrieve latest version information of the analyser."),
                               QMessageBox::Ok);
          return;
      }
    latestVersionSettings.endGroup();

    Version latestfckversion = Version(strLatestFCKVersion);
    Version FCKversion = myEnvironment->getFCKversion();
    if (!FCKversion.isKnown()){
        latestVersionSettings.close();
        return;
    }
    //get exp_license and exp_maintenance
    QString userid, licensecat, serial;
    QDate exp_license, exp_maintenance;
    pwdValid = myEnvironment->getForcheckPWDInfo(userid,
                                                 licensecat,
                                                 serial,
                                                 exp_license,
                                                 exp_maintenance);
    //compare Forcheck versions and download if needed
    if (FCKversion.isNewerRelease(latestfckversion)){
      if (FCKversion.isSameMinorVersion(latestfckversion) || (exp_maintenance >= QDate::currentDate())){
        if (QMessageBox::information(myParent, Info::TITLE,
                                     tr("A free new release of the analyser is available. Click OK to download."),
                                     QMessageBox::Ok, QMessageBox::Cancel) == QMessageBox::Ok) {
          latestVersionSettings.beginGroup("url");
          QString strDownloadURL = latestVersionSettings.value(key, "Error").toString();
          latestVersionSettings.endGroup();
          if (strDownloadURL == "Error"){
            QMessageBox::warning(myParent, Info::TITLE,
                                 tr("Unable to retrieve download url for the analyser."),
                                 QMessageBox::Ok);
          }
          else {
            myEnvironment->wwwBrowser(strDownloadURL);
          }
        }
      }
      else{
        QMessageBox::information(myParent, Info::TITLE,
                                 tr("A new release of the analyser is available. Please order the update."),
                                 QMessageBox::Ok);
      }
    }
    else{
      QMessageBox::information(myParent, Info::TITLE,
                               tr("You have the latest version of the analyser installed."),
                               QMessageBox::Ok);
    }

    latestVersionSettings.close();
}
