/*ForcheckIDE/MainWindow.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include <QtWidgets>
#include <QApplication>
#include <QFile>
#include <QUrl>
#include <QFont>
#include <QFontMetrics>
#include <QSysInfo>

#include "MainWindow.h"

#include "Info.h"
#include "Version.h"
#include "Environment.h"
#include "IDESettingsForm.h"
#include "IniFile.h"
#include "ProjectFrame.h"
#include "OptionsForm.h"
#include "PreferencesForm.h"
#include "LibrarianForm.h"
#include "HelpBrowser.h"
#include "FileViewer.h"
#include "NewRelease.h"

MainWindow::MainWindow()
{
    setWindowTitle(Info::TITLE +
                   Info::VERSION);
    setWindowIcon(QIcon(":/icons/Forcheck-32.png"));

    mdiArea = new QMdiArea;
    connect(mdiArea, SIGNAL(subWindowActivated(QMdiSubWindow*)),
            this, SLOT(updateActions()));

    searchToolBar = new QToolBar();
    replaceToolBar = new QToolBar();

    progressView = new FileViewer(this, true);
    reportView = new FileViewer(this, true);
    searchView = new FileViewer(this, true);

    tabWidget = new QTabWidget(this);
    tabWidget->setTabPosition(QTabWidget::South);
    tabWidget->addTab(progressView, tr("Progress"));
    tabWidget->addTab(reportView, tr("Report"));
    tabWidget->addTab(searchView, tr("Search"));

    projectFrame = new ProjectFrame(this, mdiArea, tabWidget, progressView, reportView, searchView);
    projectFrame->addActionsToSearchToolBar(searchToolBar);
    projectFrame->addActionsToReplaceToolBar(replaceToolBar);

    QVBoxLayout *mainlayout = new QVBoxLayout;
    mainlayout->addWidget(mdiArea);
    mainlayout->addWidget(searchToolBar);
    mainlayout->addWidget(replaceToolBar);
    QWidget *mainwidget = new QWidget;
    mainwidget->setLayout(mainlayout);

    rightSplitter = new QSplitter(Qt::Vertical);
    rightSplitter->addWidget(mainwidget);
    rightSplitter->addWidget(tabWidget);
    rightSplitter->setStretchFactor(0,0.5);

    mainSplitter = new QSplitter(Qt::Horizontal);
    mainSplitter->addWidget(projectFrame);
    mainSplitter->addWidget(rightSplitter);
    mainSplitter->setStretchFactor(1,1);
    setCentralWidget(mainSplitter);

    createActions();
    createMenus();
    createToolBars();
    updateActions();

    myEnvironment = new Environment();       //global variable!
    newrelease = new NewRelease(this);
    QByteArray geometry, state, rightsplitterstate, mainsplitterstate;
    myEnvironment->restoreStatus(geometry,
                                 state,
                                 rightsplitterstate,
                                 mainsplitterstate);
    restoreGeometry(geometry);
    restoreState(state);
    rightSplitter->restoreState(rightsplitterstate);
    mainSplitter->restoreState(mainsplitterstate);

    myEnvironment->restoreIdeSettings();

    showToolbar->setChecked(toolBar->isVisible());
    myEnvironment->setPageOptions();
    if (!qApp->arguments().isEmpty()) {
        if (qApp->arguments().size() > 1)
            projectFrame->openProject(qApp->arguments().at(1));
    }
    projectFrame->updateEnabled();

    statusMessage = new QLabel("Ready");
    statusBar()->addWidget(statusMessage);

    if (!QFileInfo(myEnvironment->getIdeConfigurationFilename()).exists()) helpQuickStartClicked();
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    if (projectFrame->isAnalysing() ||
        projectFrame->isBuilding()){

        int r = QMessageBox::warning(this, Info::TITLE,
                                     tr("Analysis or build is in progress.\n"
                                        "Exit nevertheless?"),
                                     QMessageBox::Yes | QMessageBox::No);
        if (r != QMessageBox::Yes){
            event->ignore();
            return;
        }
    }

    projectFrame->closeProject();

    myEnvironment->saveStatus(saveGeometry(),
                              saveState(),
                              rightSplitter->saveState(),
                              mainSplitter->saveState());
    myEnvironment->helpBrowser->close();
    event->accept();
}

void MainWindow::createActions()
{
    /*file menu*/
    pageSetup = new QAction(QIcon(":/icons/PageSetup.png"), tr("&Page Setup"), this);
    pageSetup->setStatusTip(tr("Set up printer"));
    connect(pageSetup, SIGNAL(triggered()), this, SLOT(pageSetupClicked()));
    fileExit = new QAction(QIcon(":/icons/Exit-16.png"), tr("E&xit"), this);
    fileExit->setShortcut(tr("Ctrl+Q"));
    fileExit->setStatusTip(tr("Exit the Forcheck IDE"));
    connect(fileExit, SIGNAL(triggered()), this, SLOT(close()));

    /*options menu*/
    preferences = new QAction(QIcon(":/icons/Evironment-16.png"), tr("&Preferences"), this);
    preferences->setShortcut(tr("Shift+Ctrl+P"));
    preferences->setStatusTip(tr("Set various preferences"));
    connect(preferences, SIGNAL(triggered()), this, SLOT(preferencesClicked()));
    IDESettings = new QAction(tr("&IDE settings"), this);
    IDESettings->setStatusTip(tr("Set some IDE preferences"));
    connect(IDESettings, SIGNAL(triggered()), this, SLOT(IDESettingsClicked()));

    /*tools menu*/
    librarianSelectFile = new QAction(tr("&Librarian: select file..."), this);
    librarianSelectFile->setStatusTip(tr("Activate the librarian and select a Forcheck library"));
    connect(librarianSelectFile, SIGNAL(triggered()), this, SLOT(librarianClicked()));
    interfaceBuilderSelectFile = new QAction(tr("&Interface builder: select file..."), this);
    interfaceBuilderSelectFile->setStatusTip(tr("Select a Forcheck library to build the interfaces"));
    connect(interfaceBuilderSelectFile, SIGNAL(triggered()), this, SLOT(interfaceBuilderClicked()));

    /*view menu*/
    showToolbar = new QAction(tr("Show toolbar"), this);
    showToolbar->setCheckable(true);
    showToolbar->setChecked(true);
    showToolbar->setStatusTip(tr("Choose to display or hide the toolbar"));
    connect(showToolbar, SIGNAL(triggered()), this, SLOT(showToolbarClicked()));

    /*window menu*/
    closeAct = new QAction(tr("Cl&ose"), this);
    closeAct->setShortcut(QKeySequence::Close);
    closeAct->setStatusTip(tr("Close the active window"));
    connect(closeAct, SIGNAL(triggered()),
            mdiArea, SLOT(closeActiveSubWindow()));
    closeAllAct = new QAction(tr("Close &All"), this);
    closeAllAct->setStatusTip(tr("Close all the windows"));
    connect(closeAllAct, SIGNAL(triggered()),
            mdiArea, SLOT(closeAllSubWindows()));
    tileAct = new QAction(tr("&Tile"), this);
    tileAct->setStatusTip(tr("Tile the windows"));
    connect(tileAct, SIGNAL(triggered()), mdiArea, SLOT(tileSubWindows()));
    cascadeAct = new QAction(tr("&Cascade"), this);
    cascadeAct->setStatusTip(tr("Cascade the windows"));
    connect(cascadeAct, SIGNAL(triggered()), mdiArea, SLOT(cascadeSubWindows()));
    nextAct = new QAction(tr("Ne&xt"), this);
    nextAct->setShortcuts(QKeySequence::NextChild);
    nextAct->setStatusTip(tr("Move the focus to the next window"));
    connect(nextAct, SIGNAL(triggered()),
            mdiArea, SLOT(activateNextSubWindow()));
    previousAct = new QAction(tr("Pre&vious"), this);
    previousAct->setShortcuts(QKeySequence::PreviousChild);
    previousAct->setStatusTip(tr("Move the focus to the previous window"));
    connect(previousAct, SIGNAL(triggered()),
            mdiArea, SLOT(activatePreviousSubWindow()));
    separatorAct = new QAction(this);
    separatorAct->setSeparator(true);

    /*help menu*/
    helpMain = new QAction(QIcon(":/icons/Help-16.png"), tr("&Help"), this);
    helpMain->setStatusTip(tr("Present the user guide"));
    helpMain->setShortcut(QKeySequence::HelpContents);
    connect(helpMain, SIGNAL(triggered()), this, SLOT(mainHelpClicked()));
    helpQuickStart = new QAction(QIcon(":/icons/Help-16.png"), tr("&QuickStart"), this);
    helpQuickStart->setStatusTip(tr("Provide help for first-time users of the IDE"));
    connect(helpQuickStart, SIGNAL(triggered()), this, SLOT(helpQuickStartClicked()));
    helpUsingTheIDE = new QAction(QIcon(":/icons/Help-16.png"), tr("&Using the IDE"), this);
    helpUsingTheIDE->setStatusTip(tr("Present the IDE section of the user guide"));
    connect(helpUsingTheIDE, SIGNAL(triggered()), this, SLOT(helpIDEClicked()));
    helpTutorial = new QAction(QIcon(":/icons/Help-16.png"), tr("&Tutorial"), this);
    helpTutorial->setStatusTip(tr("Present the Tutorial section of the user guide"));
    connect(helpTutorial, SIGNAL(triggered()), this, SLOT(helpTutorialClicked()));
    helpIndex = new QAction(QIcon(":/icons/Help-16.png"), tr("&Index"), this);
    helpIndex->setStatusTip(tr("Present the index of the user guide"));
    connect(helpIndex, SIGNAL(triggered()), this, SLOT(helpIndexClicked()));
    helpGlossary = new QAction(QIcon(":/icons/Help-16.png"), tr("&Glossary"), this);
    helpGlossary->setStatusTip(tr("Present the glossary of the user guide"));
    connect(helpGlossary, SIGNAL(triggered()), this, SLOT(helpGlossaryClicked()));
    gotoHomepage = new QAction(QIcon(":/icons/Home-16.png"), tr("&Forcheck's homepage"), this);
    gotoHomepage->setStatusTip(tr("Visit Forcheck's Homepage"));
    connect(gotoHomepage, SIGNAL(triggered()), this, SLOT(homepageClicked()));
    frequentlyAskedQuestions = new QAction(tr("&Frequently asked questions"), this);
    frequentlyAskedQuestions->setStatusTip(tr("View frequently asked questions with answers"));
    connect(frequentlyAskedQuestions, SIGNAL(triggered()), this, SLOT(frequentlyAskedQuestionsClicked()));
    registerForcheck = new QAction(tr("&Register Forcheck"), this);
    registerForcheck->setStatusTip(tr("Register Forcheck online"));
    connect(registerForcheck, SIGNAL(triggered()), this, SLOT(registerClicked()));
    sendEmail = new QAction(tr("&Send email"), this); //goede icon niet gevonden
    sendEmail->setStatusTip(tr("Send email to the Forcheck developers"));
    connect(sendEmail, SIGNAL(triggered()), this, SLOT(sendEMailClicked()));
    checkForUpdates = new QAction(tr("&Check for updates"), this);
    checkForUpdates->setStatusTip(tr("Look if an update is available"));
    connect(checkForUpdates, SIGNAL(triggered()), this, SLOT(checkForUpdatesClicked()));
    aboutQt = new QAction(QIcon(":/icons/Information-16.png"), tr("&AboutQt"), this);
    aboutQt->setStatusTip(tr("Present information about Qt"));
    connect(aboutQt, SIGNAL(triggered()), this, SLOT(aboutQtClicked()));
    about = new QAction(QIcon(":/icons/Information-16.png"), tr("&About Forcheck IDE"), this);
    about->setStatusTip(tr("Present information about the Forcheck IDE"));
    connect(about, SIGNAL(triggered()), this, SLOT(aboutClicked()));
}

void MainWindow::createMenus()
{
    /*file menu*/
    fileMenu = menuBar()->addMenu(tr("&File"));
    projectFrame->addFileActions1(fileMenu);
    fileMenu->addSeparator();
    fileMenu->addAction(pageSetup);
    projectFrame->addFileActions2(fileMenu);
    fileMenu->addSeparator();
    fileMenu->addAction(fileExit);

    /*edit menu*/
    editMenu = menuBar()->addMenu(tr("&Edit"));
    projectFrame->addEditActions(editMenu);

    /*project menu*/
    projectMenu = menuBar()->addMenu(tr("&Project"));
    projectFrame->addProjectActions(projectMenu);

    /*analysis menu*/
    analysisMenu = menuBar()->addMenu(tr("&Analysis"));
    projectFrame->addAnalysisActions(analysisMenu);

    /*options menu*/
    optionsMenu = menuBar()->addMenu(tr("&Options"));
    projectFrame->addOptionActions(optionsMenu);
    optionsMenu->addSeparator();
    optionsMenu->addAction(preferences);
    optionsMenu->addAction(IDESettings);

    /*tools menu*/
    toolsMenu = menuBar()->addMenu(tr("&Tools"));
    toolsMenu->addAction(librarianSelectFile);
    toolsMenu->addAction(interfaceBuilderSelectFile);

    /*view menu*/
    viewMenu = menuBar()->addMenu(tr("&View"));
    viewMenu->addAction(showToolbar);
    viewMenu->addSeparator();
    projectFrame->addViewActions(viewMenu);

    /*window menu*/
    windowMenu = menuBar()->addMenu(tr("&Windows"));
    windowMenu->addAction(closeAct);
    windowMenu->addAction(closeAllAct);
    windowMenu->addSeparator();
    windowMenu->addAction(tileAct);
    windowMenu->addAction(cascadeAct);
    windowMenu->addSeparator();
    windowMenu->addAction(nextAct);
    windowMenu->addAction(previousAct);
    windowMenu->addAction(separatorAct);
    projectFrame->addWindowActions(windowMenu);

    /*help menu*/
    helpMenu = menuBar()->addMenu(tr("&Help"));
    helpMenu->addAction(helpMain);
    helpMenu->addAction(helpQuickStart);
    helpMenu->addAction(helpUsingTheIDE);
    helpMenu->addAction(helpTutorial);
    helpMenu->addAction(helpIndex);
    helpMenu->addAction(helpGlossary);
    helpMenu->addSeparator();
    helpMenu->addAction(gotoHomepage);
    helpMenu->addAction(frequentlyAskedQuestions);
    helpMenu->addAction(registerForcheck);
    helpMenu->addAction(sendEmail);
    helpMenu->addAction(checkForUpdates);
    helpMenu->addSeparator();
    helpMenu->addAction(aboutQt);
    helpMenu->addAction(about);
}

void MainWindow::createToolBars()
{
    toolBar = addToolBar(tr("Toolbar"));
    toolBar->setObjectName("toolBar");
    toolBar->addSeparator();
    projectFrame->addActionsToToolBar(toolBar);
    toolBar->addSeparator();
    toolBar->addAction(helpMain);
    toolBar->addSeparator();
}

void MainWindow::updateActions()
{
    bool haschild = mdiArea->activeSubWindow() != 0;
    closeAct->setEnabled(haschild);
    closeAllAct->setEnabled(haschild);
    tileAct->setEnabled(haschild);
    cascadeAct->setEnabled(haschild);
    nextAct->setEnabled(haschild);
    previousAct->setEnabled(haschild);
    separatorAct->setVisible(haschild);
    projectFrame->changeSubWindowFocus();
}

void MainWindow::pageSetupClicked()
{
    myEnvironment->setupPage();
}

void MainWindow::IDESettingsClicked()
{
    ideSettingsForm = new IDESettingsForm(this);
    ideSettingsForm->exec();
}

void MainWindow::preferencesClicked()
{
    preferencesForm = new PreferencesForm(this);
    preferencesForm->exec();
    delete preferencesForm;
}

void MainWindow::librarianClicked()
{
    QString libName = QFileDialog::getOpenFileName(this,
                                                   tr("Select Forcheck Library File"),
                                                   QDir::currentPath(),
                                                   tr("Forcheck Library Files (*") +
                                                       myEnvironment->getLibraryExt() +
                                                       ")");
    if (!libName.isEmpty()){

        librarianForm = new LibrarianForm(libName,this);
        windowMenu->addAction(librarianForm->windowMenuAction());
        projectFrame->windowActionGroup->addAction(librarianForm->windowMenuAction());
        QMdiSubWindow *subWindow = mdiArea->addSubWindow(librarianForm);
        subWindow->setWindowFilePath(libName);
        librarianForm->exec();
    };
}

void MainWindow::interfaceBuilderClicked()
{
    QString libName = QFileDialog::getOpenFileName(this,
                                                   tr("Select Forcheck Library File"),
                                                   QDir::currentPath(),
                                                   tr("Forcheck Library Files (*") +
                                                      myEnvironment->getLibraryExt() +
                                                      ")");
    if (!libName.isEmpty()){
        QFileInfo file(libName);
        QString suggestedName = file.path() + "/" + file.baseName() + myEnvironment->getSourceExt();
        QString outName = QFileDialog::getSaveFileName(this,
                                                       tr("Save Interface File"),
                                                       suggestedName,
                                                       tr("Source Files (*") +
                                                          myEnvironment->getSourceExt() +
                                                          ")");
        if (!outName.isEmpty()){
            QString progress;
            progressView->clear();
            myEnvironment->execInterfaceBuilder(libName, outName, progress);
            progressView->putText(progress);
        }
    };
}

void MainWindow::showToolbarClicked()
{
    toolBar->setVisible(showToolbar->isChecked());
}

void MainWindow::mainHelpClicked()
{
    HelpBrowser::HelpEntry helpentry;
    helpentry = HelpBrowser::HelpMain;
    QMdiSubWindow *subwindow = mdiArea->currentSubWindow();
    if (subwindow){
        QString filename = subwindow->windowFilePath();
        FileKind::ETFileKind filekind = FileKind::Unknown;
        if (!filename.isEmpty()){
            QString ext = FileKind::fileExt(filename);
            filekind = FileKind::ext2FileKind(ext);
            if (filekind == FileKind::ReferenceStructure){
                helpentry = HelpBrowser::HelpReferenceStructure;
            }
            else if (filekind == FileKind::ModDep){
                helpentry = HelpBrowser::HelpModDep;
            }
            else if (filekind == FileKind::LibOutput ||
                     filekind == FileKind::LibRef){
                helpentry = HelpBrowser::HelpMaintainingLibraryFilesFromTheIDE;
            }
        }
    }
    myEnvironment->helpBrowser->showHelp(helpentry);
}

void MainWindow::helpQuickStartClicked()
{
    introView = new FileViewer(this);
    QString filename = myEnvironment->getFckIDEDocDir() + "QUICKSTART";
    introView->loadFile(filename);
    windowMenu->addAction(introView->windowMenuAction());
    projectFrame->windowActionGroup->addAction(introView->windowMenuAction());
    QMdiSubWindow *subWindow = mdiArea->addSubWindow(introView);
    subWindow->setWindowFilePath(filename);
    subWindow->show();
}

void MainWindow::helpIDEClicked()
{
    myEnvironment->helpBrowser->showHelp(HelpBrowser::HelpUsingTheIDE);
}

void MainWindow::helpTutorialClicked()
{
    myEnvironment->helpBrowser->showHelp(HelpBrowser::HelpTutorial);
}

void MainWindow::helpIndexClicked()
{
    myEnvironment->helpBrowser->showHelp(HelpBrowser::HelpIndex);
}

void MainWindow::helpGlossaryClicked()
{
    myEnvironment->helpBrowser->showHelp(HelpBrowser::HelpGlossary);
}

void MainWindow::homepageClicked()
{
    myEnvironment->wwwBrowser(Info::HOMEPAGE_URL);
}

void MainWindow::frequentlyAskedQuestionsClicked()
{
    myEnvironment->wwwBrowser(Info::FAQ_URL);
}

void MainWindow::registerClicked()
{
    myEnvironment->wwwBrowser(Info::REGISTRATION_URL);
}

void MainWindow::sendEMailClicked()
{
    myEnvironment->sendEmail(Info::E_MAIL);
}

void MainWindow::checkForUpdatesClicked()
{
    newrelease->checkForNewRelease();
}


void MainWindow::aboutQtClicked()
{
    QMessageBox::aboutQt(this, Info::TITLE);
}

void MainWindow::aboutClicked()
{
    QString userid, license, serial;
    QDate exp_license, exp_maintenance;
    pwdValid = myEnvironment->getForcheckPWDInfo(userid,
                                                 license,
                                                 serial,
                                                 exp_license,
                                                 exp_maintenance);
    QString text(tr("<h2>Forcheck IDE</h2>"
                    "<p>Version: "));
    text.append(Info::VERSION);
    text.append(tr("</p>"
                   "<p>Copyright &copy; 2016 Forcheck b.v.</p>"
                   "<p>Distribution under the GNU General Public License, version 3.</p>"
                   "<p>Developed using:"
                   "<br>Qt5 application framework, Digia plc."
                   "<br>Poppler PDF library.</p>"));
    if (pwdValid){
        text.append(tr("<p>The Forcheck analyser is licensed to:"
                       "<br>"));
        text.append(userid);
        text.append("<br>");
        text.append(license);
        text.append(tr("<br>Serial number: "));
        text.append(serial);
        text.append(tr("<br>License expires: "));
        text.append(exp_license.toString());
        text.append(tr("<br>Maintenance expires: "));
        text.append(exp_maintenance.toString());
        text.append("</p>");
    }
    else
        text.append(tr("<p>You are not licensed to use the Forcheck analyser.</p>"));

    text.append("<p>");
    text.append(Info::Address);
    text.append("<br>");
    text.append(Info::E_MAIL);
    text.append("<br>");
    text.append(Info::HOMEPAGE_URL);
    text.append("</p>");

    QMessageBox::about(this,
                       tr("About Forcheck IDE"),
                       text);
}
