/*ForcheckIDE/ListDialog.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "ListDialog.h"
#include "Environment.h"
#include "HelpBrowser.h"
#include "Info.h"

#include <QtWidgets>
#include <QString>
#include <QStringList>

const char* const ALLSTRING = "<all>";

ListDialog::ListDialog(QWidget *parent, ListType listtype, HelpBrowser::HelpEntry helpentry) : QDialog(parent)
{
    listType = listtype;
    helpEntry = helpentry;

    lineEdit = new QLineEdit();
    listBox = new QListWidget();
    listBox ->setSelectionMode(QAbstractItemView::ExtendedSelection);

    addButton = new QPushButton(tr("&Add"));
    addButton->setToolTip(tr("Add the entered line to the list"));
    editButton = new QPushButton(tr("&Edit"));
    editButton->setToolTip(tr("Edit the selected line"));
    removeButton = new QPushButton(tr("&Remove"));
    removeButton->setToolTip(tr("Remove the selected lines from the list"));
    clearButton = new QPushButton(tr("&Clear"));
    clearButton->setToolTip(tr("Remove all lines from the list"));
    allButton = new QPushButton(tr("All"));
    allButton->setToolTip(tr("No list, show all items"));

    abspathCheckBox = new QCheckBox("add an absolute path");

    okButton = new QPushButton(tr("&OK"));
    okButton->setToolTip(tr("Accept the list and close the window"));
    okButton->setFixedWidth(80);
    cancelButton = new QPushButton(tr("&Cancel"));
    cancelButton->setToolTip(tr("Ignore the changes and close the window"));
    cancelButton->setFixedWidth(80);
    helpButton = new QPushButton(tr("&Help"));
    helpButton->setToolTip(tr("Present the relevant pages of the user guide"));
    helpButton->setFixedWidth(80);
    helpButton->setShortcut(QKeySequence::HelpContents);

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget(helpButton);
    buttonLayout->addWidget(okButton);
    buttonLayout->addWidget(cancelButton);

    QVBoxLayout *vbox1 = new QVBoxLayout;
    vbox1->addWidget(lineEdit);
    vbox1->addWidget(listBox);
    if (listType == ListDir) {
        vbox1->addWidget(abspathCheckBox);
    }
    vbox1->addLayout(buttonLayout);

    QVBoxLayout *vbox2 = new QVBoxLayout;
    vbox2->addWidget(addButton);
    vbox2->addWidget(editButton);
    vbox2->addWidget(removeButton);
    vbox2->addWidget(clearButton);
    vbox2->addWidget(allButton);
    vbox2->addStretch();

    QHBoxLayout *mainLayout = new QHBoxLayout;
    mainLayout->addLayout(vbox1);
    mainLayout->addLayout(vbox2);
    setLayout(mainLayout);

    connect(lineEdit, SIGNAL(textEdited(const QString&)), this, SLOT(lineEditted(const QString&)));
    connect(lineEdit, SIGNAL(returnPressed()), this, SLOT(addClicked()));
    connect(listBox, SIGNAL(itemClicked(QListWidgetItem*)), this,
            SLOT(listItemSelected(QListWidgetItem*)));
    connect(addButton, SIGNAL(clicked()), this, SLOT(addClicked()));
    connect(editButton, SIGNAL(clicked()), this, SLOT(editClicked()));
    connect(removeButton, SIGNAL(clicked()), this, SLOT(removeClicked()));
    connect(clearButton, SIGNAL(clicked()), this, SLOT(clearClicked()));
    connect(allButton, SIGNAL(clicked()), this, SLOT(allClicked()));
    connect(okButton, SIGNAL(clicked()), this, SLOT(accept()));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(reject()));
    connect(helpButton, SIGNAL(clicked()), this, SLOT(helpClicked()));

    removeButton->setEnabled(false);
    if (listType == ListSymb) {
        allButton->setVisible(false);
    }
    else if (listType == ListDir) {
        lineEdit->setVisible(false);
        addButton->setText("Add...");
        addButton->setToolTip(tr("Browse and add a directory to the list"));
        editButton->setVisible(false);
        allButton->setVisible(false);
    }
}

ListDialog::~ListDialog()
{
}

void ListDialog::update()
{
    if (listType == ListAllNone) {
        lineEdit->clear();
        lineEdit->setEnabled(true);
        lineEdit->setFocus();
        addButton->setEnabled(false);
        editButton->setEnabled(false);
    }
    else if (listType == ListSymb) {
        lineEdit->clear();
        lineEdit->setEnabled(true);
        lineEdit->setFocus();
        addButton->setEnabled(false);
        editButton->setEnabled(false);
    }
    else if (listType == ListDir) {
        addButton->setEnabled(true);
        addButton->setFocus();
    }
    removeButton->setEnabled(false);

    if (listType == ListAllNone) {
        if (listBox->count() == 0) {
            clearButton->setEnabled(false);
            allButton->setEnabled(true);
        }
        else if (listBox->item(0)->text() == ALLSTRING){
            lineEdit->setEnabled(false);
            clearButton->setEnabled(true);
            allButton->setEnabled(false);
        }
        else {
            clearButton->setEnabled(true);
            allButton->setEnabled(true);
        }
    }
    else {
        if (listBox->count() == 0) {
            clearButton->setEnabled(false);
        }
        else {
            clearButton->setEnabled(true);
        }
    }
}

void ListDialog::setLists(QStringList *items)
{
    listBox->addItems(*items);
    update();
}

QStringList* ListDialog::getLists()
{
    QStringList *list = new QStringList;
    for (int i = 0; i < listBox->count(); i++) {
        list->append(listBox->item(i)->text());
    }
    return list;
}

void ListDialog::lineEditted(const QString &line)
{
    addButton->setEnabled(!line.isEmpty());
}

void ListDialog::listItemSelected(QListWidgetItem *item)
{
    if (editButton->isVisible() && listBox->item(0)->text() != ALLSTRING)
        editButton->setEnabled(true);
    removeButton->setEnabled(true);
}

void ListDialog::addClicked()
    {
    QString line;
    if (listType == ListDir){
        line = QFileDialog::getExistingDirectory(this,
                                                 tr("Select a directory to search"),
                                                 QDir::currentPath());
        if (!abspathCheckBox->isChecked()){
            QDir cwd(myEnvironment->getWorkingDir());
            line = cwd.relativeFilePath(line);
        }
    }
    else
        line = lineEdit->text();
    QList<QListWidgetItem*> list = listBox->findItems(line,Qt::MatchFixedString);
    if (list.count() > 0) {
        QMessageBox::warning(this, Info::TITLE,
                             tr("Already in list"),
                             QMessageBox::Ok);
        return;
    }
    listBox->addItem(line);
    update();
}

void ListDialog::editClicked()
{
    QString line = listBox->takeItem(listBox->currentRow())->text();
    update();
    lineEdit->setText(line);
}

void ListDialog::removeClicked()
{
    QList<QListWidgetItem*> list = listBox->selectedItems();
    for (int i=0; i < list.size(); i++)
      listBox->takeItem(listBox->row(list[i]));
    update();
}

void ListDialog::allClicked()
{
    clearClicked();
    listBox->addItem(ALLSTRING);
    update();
}

void ListDialog::clearClicked()
{
    if (listBox->count() > 1) {
        if (QMessageBox::information(this, Info::TITLE,
                                     tr("All items in list will be removed"),
                                     QMessageBox::Ok) != QMessageBox::Ok)
            return;
    }
    listBox->clear();
    update();
}

void ListDialog::helpClicked()
{
    myEnvironment->helpBrowser->showHelp(helpEntry);
}
