/*ForcheckIDE/LibrarianForm.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "LibrarianForm.h"
#include "LibrarianActions.h"
#include "Info.h"
#include "Environment.h"
#include "HelpBrowser.h"

#include <QtWidgets>
#include <QStringList>
#include <QList>

LibrarianForm::LibrarianForm(const QString& fileName, QWidget *parent) : QDialog(parent)
{
    setAttribute(Qt::WA_DeleteOnClose);
    curFileName = fileName;
    setWindowTitle("Forcheck Librarian");

    action = new QAction(this);
    action->setCheckable(true);
    action->setText(fileName);
    connect(action, SIGNAL(triggered()), this, SLOT(show()));
    connect(action, SIGNAL(triggered()), this, SLOT(setFocus()));

    formCreate();
    setFocusPolicy(Qt::ClickFocus);
    librarianActions = new LibrarianActions(fileName);
    update();

    statusBar->showMessage(tr("Ready"));
}

void LibrarianForm::formCreate()
{

    filenameLabel = new QLabel(curFileName);
    listWidget = new QListWidget(this);
    listWidget->setSelectionMode(QAbstractItemView::MultiSelection);

    removeButton = new QPushButton(tr("&Remove"));
    removeButton->setToolTip(tr("Remove the selected program unit entries from the library"));
    compressButton = new QPushButton(tr("&Compress"));
    compressButton->setToolTip(tr("Copy the library file and delete all removed program unit entries"));
    helpButton = new QPushButton(tr("&Help"));
    helpButton->setToolTip(tr("Present the relevant page of the user guide"));
    statusBar = new QStatusBar();

    QVBoxLayout *buttonLayout = new QVBoxLayout;
    buttonLayout->addWidget(removeButton);
    buttonLayout->addWidget(compressButton);
    buttonLayout->addStretch();
    buttonLayout->addWidget(helpButton);
    QHBoxLayout *subLayout = new QHBoxLayout;
    subLayout->addWidget(listWidget);
    subLayout->addLayout(buttonLayout);
    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(filenameLabel);
    mainLayout->addLayout(subLayout);
    mainLayout->addWidget(statusBar);
    setLayout(mainLayout);

    connect(removeButton, SIGNAL(clicked()), this, SLOT(removeClicked()));
    connect(compressButton, SIGNAL(clicked()), this, SLOT(compressClicked()));
    connect(helpButton, SIGNAL(clicked()), this, SLOT(helpClicked()));
    connect(listWidget, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));
}

void LibrarianForm::update()
{
    QStringList list;
    if (librarianActions->makeList(list) != LIB_ERR_OK)
        librarianActions->showErrorMsg();
    listWidget->clear();
    listWidget->addItems(list);
    updateEnabled();
}

QSize LibrarianForm::sizeHint() const
{
    return QSize(55 * fontMetrics().width('x'),
                 50 * fontMetrics().lineSpacing());
}

void LibrarianForm::focusInEvent(QFocusEvent * event)
{
    emit(InFocus());
    event->accept();
}

void LibrarianForm::updateEnabled(){
    QList<QListWidgetItem *> selected = listWidget->selectedItems();
    removeButton->setEnabled(selected.count() > 0);
}

void LibrarianForm::itemSelectionChanged()
{
    updateEnabled();
}

void LibrarianForm::removeClicked()
{
    if (QMessageBox::question(this, Info::TITLE,
                              tr("Remove all selected entries from library?"),
                              QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel,
                              QMessageBox::Yes) != QMessageBox::Yes)
        return;
    QList<QListWidgetItem *> removeList = listWidget->selectedItems();
    QListIterator<QListWidgetItem *> i(removeList);
    QListWidgetItem *item;
    QString sub;
    if (librarianActions->open(true) != LIB_ERR_OK){
        librarianActions->showErrorMsg();
        return;
    };
    while (i.hasNext()){
        item = i.next();
        sub = item->data(Qt::DisplayRole).toString();
        if (librarianActions->remove(sub) != LIB_ERR_OK){
            librarianActions->showErrorMsg();
            QMessageBox::warning(this, Info::TITLE,
                                 sub+tr(". Unable to remove entry from library"),
                                 QMessageBox::Ok);
        }
    };
    librarianActions->close();
    QStringList list;
    if (librarianActions->makeList(list) != LIB_ERR_OK)
        librarianActions->showErrorMsg();
    listWidget->clear();
    listWidget->addItems(list);
    updateEnabled();
}

void LibrarianForm::compressClicked()
{
    statusBar->showMessage(tr("Compressing"));
    if (librarianActions->compress()!= LIB_ERR_OK)
        librarianActions->showErrorMsg();
    updateEnabled();
    statusBar->showMessage(tr("Ready"));
}

void LibrarianForm::helpClicked()
{
    myEnvironment->helpBrowser->showHelp(HelpBrowser::HelpMaintainingLibraryFilesFromTheIDE);
}
