/*ForcheckIDE/LibrarianActions.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "LibrarianActions.h"
#include "Info.h"

#include <QtWidgets>
#include <QFile>
#include <QString>
#include <QIODevice>

#define RECREAD(f,a) \
       if (f->read((char*)&(a), sizeof(a)) != sizeof(a))\
                return LIB_ERR_READ;
#define RECREADL(f,a,l) \
       if (f->read((char*)&(a), l) != l)\
                return LIB_ERR_READ;
#define RECWRITE(f,a) \
       if (f->write((char*)&(a), sizeof(a)) != sizeof(a))\
                return LIB_ERR_WRITE;
#define RECWRITEL(f,a,l) \
       if (f->write((char*)&(a), l) != l)\
                return LIB_ERR_WRITE;

LibrarianActions::LibrarianActions(const QString& filename)
{
        libfile = new QFile(filename);
        last_status = LIB_ERR_OK;
}

LibrarianActions::~LibrarianActions()
{
        if (libfile)
                delete libfile;
}

int LibrarianActions::open(const bool readwrite)
{
        last_status = LIB_ERR_NOT_EXISTS;
        if (!libfile->exists())
                return last_status;
        last_status = LIB_ERR_OPEN;
        if (readwrite){
            if (!libfile->open(QIODevice::ReadWrite))
                return last_status;
        }
        else {
            if (!libfile->open(QIODevice::ReadOnly))
                return last_status;
        }
        last_status = LIB_ERR_READ;
        // read first header record
        RECREAD(libfile,Fckvers)
        QString msg = QString(Fckvers);
        last_status = LIB_ERR_FCK;
        if (Fckvers[0] != 'V' || Fckvers[3] != '.')
                return last_status;
        RECREAD(libfile,Credatetime)
        RECREAD(libfile,Updatetime)
        RECREAD(libfile,Lbvers)
        last_status = LIB_ERR_VERSION;
        if (Lbvers[1] != '0' || Lbvers[2] != '2')
                return LIB_ERR_VERSION;
        // read second header record
        last_status = LIB_ERR_READ;
        if (readHeaderRecord(libfile) != LIB_ERR_OK)
            return last_status;
        // verify header data
        last_status = LIB_ERR_CONSIST;
        if (LstIrc < StIrc || EndIrc < StIrc || NumIrc < 0 || Lsbrc <= StIrc)
                return last_status;
        last_status = LIB_ERR_OK;
        return last_status;
}

void LibrarianActions::close()
{
    libfile->close();
}

int LibrarianActions::makeList(QStringList &list)
{
        int recpos, nxtirc, subirc, sbrc, lbsq;
        short lnam2;
        char tag, name[MXSYML+1];

        last_status = open();
        if (last_status != LIB_ERR_OK) return last_status;
        last_status = LIB_ERR_READ;
        nxtirc = StIrc;
        for (int is=0; is<NumIrc && nxtirc>0; is++) {
                subirc = nxtirc;
                recpos = (subirc-1) * ISBRCL;
                if (!libfile->seek(recpos))
                        return LIB_ERR_READ;
                RECREAD(libfile,tag);
                if (tag != REC_INDEX)
                        return last_status;
                RECREAD(libfile,nxtirc)
                RECREAD(libfile,sbrc)
                RECREAD(libfile,lbsq)
                RECREAD(libfile,lnam2)
                RECREADL(libfile,name,MXSYMC);
                if (lnam2 > MXSYMC) {
                        subirc = subirc + 1;
                        recpos = (subirc-1) * ISBRCL;
                        if (!libfile->seek(recpos))
                                return last_status;
                        RECREAD(libfile,tag);
                        if (tag != REC_NAME_CONT)
                            return last_status;
                        RECREADL(libfile,name[MXSYMC],MXSYML-MXSYMC);
                }
                name[lnam2] = 0;
                if (*name) // An empty library has NumIrc == 1 ???? resulting in an empty name
                        list << QString(name);
        }
        close();
        last_status = LIB_ERR_OK;
        return last_status;
}

int LibrarianActions::remove(const QString &remname)
{
        int recpos;
        int nxtirc, subirc, sbrc, lbsq;
        int prvirc, prvsbrc, prvlbs;
        short lnam2, prvlnam2;
        char tag, name[MXSYML+1];
        char prvnam[MXSYML];

        nxtirc = StIrc;
        prvirc = StIrc;
        prvsbrc = 0;
        prvlbs = 0;
        prvlnam2 = 0;
        for (int i=0; i<MXSYML; i++)
                prvnam[i] = ' ';

        for (int is = 0; is<NumIrc && nxtirc>0; is++) {
                subirc = nxtirc;
                recpos = (subirc-1) * ISBRCL;
                last_status = LIB_ERR_READ;
                if (!libfile->seek(recpos))
                        return LIB_ERR_READ;
                RECREAD(libfile,tag)
                if (tag != REC_INDEX)
                        return LIB_ERR_READ;
                RECREAD(libfile,nxtirc)
                RECREAD(libfile,sbrc)
                RECREAD(libfile,lbsq)
                RECREAD(libfile,lnam2)
                RECREADL(libfile,name,MXSYMC)
                if (lnam2 > MXSYMC) {
                        subirc = subirc + 1;
                        recpos = (subirc-1) * ISBRCL;
                        if (!libfile->seek(recpos))
                                return LIB_ERR_READ;
                        RECREAD(libfile,tag);
                        if (tag != REC_NAME_CONT)
                            return LIB_ERR_READ;
                        RECREADL(libfile,name[MXSYMC],MXSYML-MXSYMC)
                }
                name[lnam2] = 0;
                if (name == remname) {
                        // remove entry
                        if (is==0 && nxtirc>0)
                                StIrc = nxtirc;
                        else {
                                recpos = (prvirc-1) * ISBRCL;
                                last_status = LIB_ERR_WRITE;
                                if (!libfile->seek(recpos))
                                        return LIB_ERR_WRITE;
                                tag = REC_INDEX;
                                RECWRITE(libfile,tag)
                                RECWRITE(libfile,nxtirc)
                                RECWRITE(libfile,prvsbrc)
                                RECWRITE(libfile,prvlbs)
                                RECWRITE(libfile,prvlnam2)
                                RECWRITEL(libfile,prvnam,MXSYMC)
                                if (prvlnam2 > MXSYMC) {
                                        prvirc = prvirc + 1;
                                        recpos = (prvirc-1) * ISBRCL;
                                        if (!libfile->seek(recpos))
                                                return LIB_ERR_READ;
                                        tag = REC_NAME_CONT;
                                        RECWRITE(libfile,tag)
                                        RECWRITEL(libfile,prvnam[MXSYMC],MXSYML-MXSYMC);
                                }
                                last_status = LIB_ERR_OK;
                                return last_status;
                        }
                        if (is>0 || nxtirc>0) NumIrc = NumIrc - 1;
                        if (nxtirc<=0) LstIrc = prvirc;
                        // update header record
                        last_status = writeHeaderRecord(libfile);
                        return last_status;
                }
                else {
                        prvirc = subirc;
                        prvsbrc = sbrc;
                        prvlbs = lbsq;
                        prvlnam2 = lnam2;
                        strncpy(prvnam, name, MXSYML);
                }
        }
        last_status = LIB_ERR_FOUND;
        return last_status;
}

int LibrarianActions::compress()
{
        QFile newlibfile("fcklib.tmp");
        // delete the previous temporary file, if still present
        last_status = LIB_ERR_DELOLD;
        if (newlibfile.exists()) {
                if (!newlibfile.remove())
                        return last_status;
        }
        // create the new temporary file
        last_status = LIB_ERR_CREATE;
        if (!newlibfile.open(QIODevice::WriteOnly))
                return last_status;

        last_status = open();
        if (last_status != LIB_ERR_OK){
            cleanup(&newlibfile);
            return last_status;
        }

        last_status = cmpr1(&newlibfile);
        close();
        if (last_status != LIB_ERR_OK){
            cleanup(&newlibfile);
            return last_status;
        }

        // delete the original library file
        last_status = LIB_ERR_DELETE;
        if (!libfile->remove())
                return last_status;
        // rename the new file to the old
        last_status = LIB_ERR_MOVE;
        if (!newlibfile.rename(libfile->fileName()))
                return last_status;
        last_status = LIB_ERR_OK;
        return last_status;
}

void LibrarianActions::cleanup(QFile *file)
{
        file->close();
        file->remove();
}

int LibrarianActions::cmpr1(QFile *newlibfile)
{
        int status;
        
        // write empty index
        status = fill(newlibfile, 1, EndIrc);
        if (status != LIB_ERR_OK)
                return status;

        // create first header record
        newlibfile->seek(0);

        RECWRITE(newlibfile,Lbvers)
        RECWRITE(newlibfile,Credatetime)
        RECWRITE(newlibfile,Updatetime)
        RECWRITE(newlibfile,Lbvers)

        // copy library
        status = cmpr2(newlibfile);
        if (status != LIB_ERR_OK)
                return status;
        // write second header record
        return writeHeaderRecord(newlibfile);
}

int LibrarianActions::cmpr2(QFile *newlibfile)
// go through index.
// copy program unit chunks
// copy index entries
{
        int recpos, status;
        int nxtirc, subirc, sbrc, lbsq;
        int nwirc, nwsbrc, nircrec;
        short lnam2;
        char tag, name[MXSYML];

// copy index

        nircrec = 0;
        nxtirc = StIrc;
        LstIrc = StIrc;

        for (int is=0; is<NumIrc && nxtirc>0; is++) {
                // read index record
                subirc = nxtirc;
                recpos = (subirc-1) * ISBRCL;
                if (!libfile->seek(recpos))
                        return LIB_ERR_READ;
                RECREAD(libfile,tag)
                if (tag != REC_INDEX)
                        return LIB_ERR_READ;
                RECREAD(libfile,nxtirc)
                RECREAD(libfile,sbrc)
                RECREAD(libfile,lbsq)
                RECREAD(libfile,lnam2)
                for (int i=0; i < MXSYML; i++) name[i] = ' ';
                RECREADL(libfile,name,MXSYMC)
                nircrec = nircrec + 1;
                if (lnam2 > MXSYMC) {
                        subirc = subirc + 1;
                        recpos = (subirc-1) * ISBRCL;
                        if (!libfile->seek(recpos))
                                return LIB_ERR_READ;
                        RECREAD(libfile,tag)
                        if (tag != REC_NAME_CONT)
                                return LIB_ERR_READ;
                        RECREADL(libfile,name[MXSYMC],MXSYML-MXSYMC)
                        nircrec = nircrec + 1;
                }

                // create new index record
                if (nxtirc > 0) {
                        nwirc = LstIrc + 1;
                        if (lnam2 > MXSYMC)
                                nwirc = nwirc + 1;
                }
                else
                        nwirc = 0;
                nwsbrc = 0;

                recpos = (LstIrc-1) * ISBRCL;
                if (!newlibfile->seek(recpos))
                        return LIB_ERR_WRITE;
                tag = REC_INDEX;
                RECWRITE(newlibfile,tag)
                RECWRITE(newlibfile,nwirc)
                RECWRITE(newlibfile,nwsbrc)
                RECWRITE(newlibfile,lbsq)
                RECWRITE(newlibfile,lnam2)
                RECWRITEL(newlibfile,name,MXSYMC)
                if (lnam2 > MXSYMC) {
                        LstIrc = LstIrc + 1;
                        recpos = (LstIrc-1) * ISBRCL;
                        if (!newlibfile->seek(recpos))
                                return LIB_ERR_WRITE;
                        tag = REC_NAME_CONT;
                        RECWRITE(newlibfile,tag)
                        RECWRITEL(newlibfile,name[MXSYMC],MXSYML-MXSYMC)
                }

                if (nwirc > 0) LstIrc = nwirc;

                }

// copy subprogram info

                EndIrc = StIrc - 1 + nircrec + 100;
                Lsbrc = EndIrc;

                nxtirc = StIrc;
                LstIrc = StIrc;

                for (int is=0; is<NumIrc && nxtirc>0; is++) {
                        // read index record
                        subirc = nxtirc;
                        recpos = (subirc-1) * ISBRCL;
                        if (!libfile->seek(recpos))
                                return LIB_ERR_READ;
                        RECREAD(libfile,tag)
                        if (tag != REC_INDEX)
                                return LIB_ERR_READ;
                        RECREAD(libfile,nxtirc)
                        RECREAD(libfile,sbrc)
                        RECREAD(libfile,lbsq)
                        RECREAD(libfile,lnam2)
                        for (int i=0; i < MXSYML; i++) name[i] = ' ';
                        RECREADL(libfile,name,MXSYMC)

                        // create new index record
                        if (nxtirc > 0) {
                                nwirc = LstIrc + 1;
                                if (lnam2 > MXSYMC)
                                        nwirc = nwirc + 1;
                        }
                        else
                                nwirc = 0;
                        if (sbrc > 0)
                                nwsbrc = Lsbrc + 1;
                        else
                                nwsbrc = 0;

                        // update new index record
                        recpos = (LstIrc-1) * ISBRCL;
                        if (!newlibfile->seek(recpos))
                                return LIB_ERR_WRITE;
                        tag = REC_INDEX;
                        RECWRITE(newlibfile,tag)
                        RECWRITE(newlibfile,nwirc)
                        RECWRITE(newlibfile,nwsbrc)
                        RECWRITE(newlibfile,lbsq)
                        RECWRITE(newlibfile,lnam2)
                        RECWRITEL(newlibfile,name,MXSYMC)

                        if (nwirc > 0) LstIrc = nwirc;

                        if (sbrc > 0) {
                                status = copsub(newlibfile,sbrc);
                                if (status != LIB_ERR_OK)
                                        return status;
                        }
        }
         return LIB_ERR_OK;
}

int LibrarianActions::copsub(QFile *newlibfile, int sbrc)
// copy program unit chunk
{
        int recnum, recpos, numrec;
        char tag, rec[ISBRCL];

        recnum = sbrc;

        // read number of records of this chunk
        recpos = (recnum-1) * ISBRCL;
        if (!libfile->seek(recpos))
                return LIB_ERR_READ;
        RECREAD(libfile,tag)
        RECREAD(libfile,numrec)

        for (int i=0; i<numrec; i++) {
                // read record
                recpos = (recnum-1) * ISBRCL;
                if (!libfile->seek(recpos))
                        return LIB_ERR_READ;
                RECREAD(libfile,rec)
                // write record
                Lsbrc = Lsbrc + 1;
                recpos = (Lsbrc-1) * ISBRCL;
                if (!newlibfile->seek(recpos))
                        return LIB_ERR_WRITE;
                RECWRITE(newlibfile,rec)
                recnum = recnum + 1;
        }
        return LIB_ERR_OK;
}

int LibrarianActions::readHeaderRecord(QFile *file)
{
        if (!libfile->seek(1*ISBRCL))
                return LIB_ERR_READ;
        RECREAD(file,LbSeq)
        RECREAD(file,StIrc)
        RECREAD(file,LstIrc)
        RECREAD(file,EndIrc)
        RECREAD(file,NumIrc)
        RECREAD(file,Lsbrc)
        return LIB_ERR_OK;
}

int LibrarianActions::writeHeaderRecord(QFile *file)
{
        if (!file->seek(1*ISBRCL))
                return LIB_ERR_WRITE;
        RECWRITE(file,LbSeq)
        RECWRITE(file,StIrc)
        RECWRITE(file,LstIrc)
        RECWRITE(file,EndIrc)
        RECWRITE(file,NumIrc)
        RECWRITE(file,Lsbrc)
        return LIB_ERR_OK;
}

int LibrarianActions::fill(QFile *newlibfile, int fromrec, int torec)
// fill up file gap with empty records
{
        int recnum, recpos;
        char rec[ISBRCL];

        for (recnum=fromrec; recnum<=torec; recnum++) {
                recpos = (recnum-1) * ISBRCL;
                if (!newlibfile->seek(recpos))
                        return LIB_ERR_WRITE;
                RECWRITE(newlibfile,rec)
        }
        return LIB_ERR_OK;
}

void LibrarianActions::showErrorMsg()
{
	QString msg;
        switch (last_status){
        case LIB_ERR_OPEN:
                msg = QString(tr("Could not open library file."));
                break;
        case LIB_ERR_NOT_EXISTS:
                msg = QString(tr("Library file does not exist."));
                break;
        case LIB_ERR_FCK:
                msg = QString(tr("No Forcheck library file"));
                break;
        case LIB_ERR_VERSION:
                msg = QString(tr("Incompatible library file"));
                break;
        case LIB_ERR_CONSIST:
                msg = QString(tr("Inconsistent library file"));
                break;
        case LIB_ERR_CREATE:
                msg = QString(tr("Unable to create new library"));
                break;
        case LIB_ERR_READ:
                msg = QString(tr("Read error on library file"));
                break;
        case LIB_ERR_WRITE:
                msg = QString(tr("Write error on library file"));
                break;
        case LIB_ERR_FOUND:
                msg = QString(tr("Libary entry not found"));
                break;
        case LIB_ERR_DELOLD:
                msg = QString(tr("Unable to delete temporary library file"));
                break;
        case LIB_ERR_DELETE:
                msg = QString(tr("Unable to delete the original library"));
                break;
        case LIB_ERR_MOVE:
                msg = QString(tr("Unable to restore the compressed library"));
                break;
        case LIB_ERR_OK:
                msg = QString(tr("Ok"));
                break;
        default:
                msg = QString(tr("Unable to perform library action"));
                break;
        }
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             msg,
                             QMessageBox::Ok);
}

