/*ForcheckIDE/LibListDialog.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "LibListDialog.h"
#include "LibrarianActions.h"

#include <QtWidgets>
#include <QStringList>

LibListDialog::LibListDialog(QWidget *parent) : QDialog(parent)
{
    header1 = new QLabel("Program units in library");
    header2 = new QLabel("Include in analysis");

    listBox1 = new QListWidget();
    listBox2 = new QListWidget();
    listBox1->setSelectionMode(QAbstractItemView::ExtendedSelection);
    listBox2->setSelectionMode(QAbstractItemView::ExtendedSelection);

    includeButton = new QPushButton(">");
    includeButton->setFixedWidth(25);
    includeAllButton = new QPushButton(">>");
    includeAllButton->setFixedWidth(25);
    excludeButton = new QPushButton("<");
    excludeButton->setFixedWidth(25);
    excludeAllButton = new QPushButton("<<");
    excludeAllButton->setFixedWidth(25);

    okButton = new QPushButton(tr("&OK"));
    cancelButton = new QPushButton(tr("&Cancel"));
    helpButton = new QPushButton(tr("&Help"));
    helpButton->setShortcut(QKeySequence::HelpContents);

    QVBoxLayout *vbox1 = new QVBoxLayout;
    vbox1->addWidget(header1);
    vbox1->addWidget(listBox1);

    QVBoxLayout *vbox2 = new QVBoxLayout;
    vbox2->addStretch();
    vbox2->addWidget(includeButton);
    vbox2->addWidget(includeAllButton);
    vbox2->addWidget(excludeButton);
    vbox2->addWidget(excludeAllButton);
    vbox2->addStretch();

    QVBoxLayout *vbox3 = new QVBoxLayout;
    vbox3->addWidget(header2);
    vbox3->addWidget(listBox2);

    QHBoxLayout *hbox = new QHBoxLayout;
    hbox->addLayout(vbox1);
    hbox->addLayout(vbox2);
    hbox->addLayout(vbox3);

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addWidget(okButton);
    buttonLayout->addWidget(cancelButton);
    buttonLayout->addWidget(helpButton);
    buttonLayout->addStretch();

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addLayout(hbox);
    mainLayout->addLayout(buttonLayout);
    setLayout(mainLayout);

    connect(listBox1, SIGNAL(itemClicked(QListWidgetItem*)), this,
            SLOT(listItemSelected1(QListWidgetItem*)));
    connect(listBox2, SIGNAL(itemClicked(QListWidgetItem*)), this,
            SLOT(listItemSelected2(QListWidgetItem*)));
    connect(includeButton, SIGNAL(clicked()), this, SLOT(includeClicked()));
    connect(includeAllButton, SIGNAL(clicked()), this, SLOT(includeAllClicked()));
    connect(excludeButton, SIGNAL(clicked()), this, SLOT(excludeClicked()));
    connect(excludeAllButton, SIGNAL(clicked()), this, SLOT(excludeAllClicked()));
    connect(okButton, SIGNAL(clicked()), this, SLOT(accept()));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(reject()));
}

LibListDialog::~LibListDialog()
{
    delete header1;
    delete header2;

    delete listBox1;
    delete listBox2;

    delete includeButton;
    delete includeAllButton;
    delete excludeButton;
    delete excludeAllButton;

    delete okButton;
    delete cancelButton;
    delete helpButton;
}

void LibListDialog::update()
{
    listBox1->sortItems();
    listBox2->sortItems();
    includeButton->setEnabled(false);
    includeAllButton->setEnabled(listBox1->count() > 0);
    excludeButton->setEnabled(false);
    excludeAllButton->setEnabled(listBox2->count() > 0);
}

void LibListDialog::setLists(const QString& filename, QStringList *items)
{
    LibrarianActions libActions(filename);
    QStringList list;
    if (libActions.makeList(list) != LIB_ERR_OK)
        libActions.showErrorMsg();
    listBox1->addItems(list);
    listBox2->addItems(*items);
    update();
}

QStringList* LibListDialog::getLists()
{
    QStringList *list = new QStringList;
    for (int i = 0; i < listBox1->count(); i++) {
        list->append(listBox2->item(i)->text());
    }
    return list;
}

void LibListDialog::listItemSelected1(QListWidgetItem *item)
{
    includeButton->setEnabled(true);
}

void LibListDialog::listItemSelected2(QListWidgetItem *item)
{
    excludeButton->setEnabled(true);
}

void LibListDialog::includeClicked()
{
    QList<QListWidgetItem *> list1 = listBox1->selectedItems();
    QList<QListWidgetItem *> list2;
    QListIterator<QListWidgetItem *> i(list1);
    QListWidgetItem *item;
    QString sub;
    while (i.hasNext()){
        item = i.next();
        sub = item->data(Qt::DisplayRole).toString();
        list2 = listBox2->findItems(sub, Qt::MatchFixedString);
        if (list2.count() == 0){
            listBox2->addItem(sub);
        }
    }
    update();
}

void LibListDialog::includeAllClicked()
{
    QList<QListWidgetItem *> list2;
    QListWidgetItem *item;
    QString sub;
    for (int row=0; row < listBox1->count(); row++){
        item = listBox1->item(row);
        sub = item->data(Qt::DisplayRole).toString();
        list2 = listBox2->findItems(sub, Qt::MatchFixedString);
        if (list2.count() == 0){
            listBox2->addItem(sub);
        }
    }
    update();
}

void LibListDialog::excludeClicked()
{
    QList<QListWidgetItem *> list = listBox2->selectedItems();
    QListIterator<QListWidgetItem *> i(list);
    QListWidgetItem *item;
    while (i.hasNext()){
        item = i.next();
        listBox2->takeItem(listBox2->row(item));
    }
    update();
}

void LibListDialog::excludeAllClicked()
{
    listBox2->clear();
    update();
}
