/*ForcheckIDE/IniFile.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "IniFile.h"

IniFile::IniFile(const QString &filename)
{
    file.setFileName(filename);
    writing = false;
    endGroup();
}

IniFile::~IniFile()
{
    close();
}

bool IniFile::create()
{
    if (!file.open(QIODevice::WriteOnly | QIODevice::Text))
        return false;
    stream.setDevice(&file);
    empty = true;
    writing = true;
    return true;
}

bool IniFile::exists()
{
    return file.exists();
}

bool IniFile::open()
{
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
        return false;
    stream.setDevice(&file);
    QString line;
    do {
        line = stream.readLine();
        if (line != QString()) empty = false;
        content.append(line);
    } while (!stream.atEnd());
    writing = false;
    return true;
}

bool IniFile::isEmpty()
{
    return empty;
}

bool IniFile::close()
{
    content.clear();
    stream.flush();
    file.close();
    endGroup();
    return true;
}

QStringList IniFile::childGroups()
{
    QStringList result;
    for (int index = 0; index < content.count(); ++index){
        if (content[index].left(1) == "["){
            QString group = content[index];
            group.remove('[');
            group.remove(']');
            result.append(group);
        }
    }
    return result;
}

bool IniFile::beginGroup(const QString &group)
    {
    QString templ = "[" + group + "]";
    if (writing)
        stream << templ << endl;
    else {
        startIndex = 0;
        endIndex = content.count() - 1;
        startIndex = locate(templ);
        if (startIndex < 0) return false;
        startIndex = startIndex + 1;
        int index = locate("[");
        if (index >= 0) endIndex = index - 1;
    }
    return true;
}

void IniFile::endGroup()
{
    if (writing)
        stream << endl;
    startIndex = -1;
    endIndex = -1;
}

bool IniFile::contains(const QString &key)
{
    QString templ = key + "=";
    int index = locate(templ);
    return index >= 0;
}

int IniFile::locate(const QString &templ){
    int index = -1;
    if (startIndex < 0 || endIndex < 0) return index;
    for (int i = startIndex; i <= endIndex; ++i){
        QString t = content[i];
        if (t.startsWith(templ,Qt::CaseInsensitive)){
            index = i;
            break;
        }
    }
    return index;
}

void IniFile::setValue(const QString &key, const int &value)
{
    stream << key << "=" << value << endl;
}

void IniFile::setValue(const QString &key, const QString &value)
{
    stream << key << "=" << value << endl;
}

QVariant IniFile::value(const QString &key, const QVariant &defaultValue)
{
    QString templ = key + "=";
    int index = locate(templ);
    if (index < 0) return defaultValue;
    QString line = content.at(index);
    line.remove(templ,Qt::CaseInsensitive);
    return QVariant(line);
}
