/*ForcheckIDE/HelpBrowser.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "HelpBrowser.h"
#include "Info.h"
#include "PdfToc.h"

#include "poppler-link.h"

#include <QtWidgets>
#include <QString>
#include <QSettings>
//---------------------------------------------------------------------------
typedef struct {
        const char *KeyName;
        const char *Destination;
} THelpList;
//---------------------------------------------------------------------------
const THelpList HelpList[] = {
#if defined(Q_OS_WIN)
#include "win_ide.inc"
#elif defined(Q_OS_LINUX)
#include "unix_ide.inc"
#endif
};
//---------------------------------------------------------------------------
const int HelpListSize = sizeof(HelpList) / sizeof(*HelpList);
//---------------------------------------------------------------------------
const HelpBrowser::HelpEntryKey HelpBrowser::helpEntryKeyList[] = {
        { HelpMain, "" },
        { HelpUsingTheIDE, "Using the IDE" },
        { HelpTutorial, "Tutorial" },
        { HelpIndex, "Index" },
        { HelpGlossary, "Glossary" },
        { HelpIDESettings, "Using the IDE" },
        { HelpLineOrStatementNumbering, "Line or statement numbering" },
        { HelpDirectories, "Directories" },
        { HelpRedefinitionAndSuppressionOfMessages, "Redefinition and suppression of messages" },
        { HelpMessageFormat, "Reporting messages" },
        { HelpMessageSummary, "Message summary" },
        { HelpDateFormat, "Date format" },
        { HelpDefaultFileExtensions, "Default file name extensions" },
        { HelpCompilersSupported, "Compiler Emulation" },
        { HelpEditor, "Editor" },
        { HelpBuild, "Build setup" },
        { HelpTheUsageOfIncludeFiles, "The usage of include files" },
        { HelpOptions, "Options" },
        { HelpProgramUnitAnalysisOptions, "Program-unit analysis options" },
        { HelpGlobalAnalysisOptions, "Global analysis options" },
        { HelpListingOptions, "Listing options" },
        { HelpLibraryOptions, "Library options" },
        { HelpMiscellaneousOptions, "Miscellaneous options" },
        { HelpUsingForcheckLibrariesInTheIDE, "Using FORCHECK libraries in the IDE" },
        { HelpMaintainingLibraryFilesFromTheIDE, "Maintaining library files from the IDE" },
        { HelpReferenceStructure, "Reference structure" },
        { HelpReferenceStructureInXmlFormat, "Reference structure in XML format" },
        { HelpModDep, "Display of module dependencies" },
        { HelpModDepInXmlFormat, "Display of module dependencies in XML format" }
};
const int HelpBrowser::helpEntryKeyListSize = sizeof(HelpBrowser::helpEntryKeyList) /
                                              sizeof(HelpBrowser::HelpEntryKey);
//---------------------------------------------------------------------------

HelpBrowser::HelpBrowser(const QString& path, QWidget *parent) :
        QWidget(parent)
{
    if (path.isEmpty())
        return;
    helpFile = new QFile(path);
    pdfWidget = new PdfWidget(this);
    pdfTocTree = new PdfTocTree(this);
    formCreate();
    loaded = false;
    setWindowTitle("Forcheck IDE help " + path);
    setWindowIcon(QIcon(":/icons/Forcheck-32.png"));
}

bool HelpBrowser::load()
{
    bool result = pdfWidget->setDocument(helpFile->fileName());
    if (result){
        pageSpinBox->setMinimum(1);
        pageSpinBox->setMaximum(pdfWidget->document()->numPages());
        pageSpinBox->setValue(1);
        pdfTocTree->fillInfo(pdfWidget->document());
        tocArea->setWidget(pdfTocTree);
        scrollArea->setWidget(pdfWidget);
    }
    return result;
}

void HelpBrowser::formCreate()
{
    QLabel *pagelabel = new QLabel(tr("Page: "));
    pageSpinBox = new QSpinBox;
    connect(pageSpinBox, SIGNAL(valueChanged(int)),
            pdfWidget, SLOT(setPage(int)));
    connect(pdfWidget, SIGNAL(pageChanged(int)),
            pageSpinBox, SLOT(setValue(int)));

    firstPage = new QAction(QIcon(":/icons/Arrow-left-double-22.png"), tr("&Top"), this);
    firstPage->setShortcut(QKeySequence::MoveToStartOfDocument);
    firstPage->setStatusTip(tr("Show first page"));
    connect(firstPage, SIGNAL(triggered()), pdfWidget, SLOT(firstPage()));

    previousPage = new QAction(QIcon(":/icons/Arrow-left-22.png"), tr("&Previous page"), this);
    previousPage->setShortcut(QKeySequence::MoveToPreviousPage);
    previousPage->setStatusTip(tr("Show previous page"));
    connect(previousPage, SIGNAL(triggered()), pdfWidget, SLOT(previousPage()));

    nextPage = new QAction(QIcon(":/icons/Arrow-right-22.png"), tr("&Next page"), this);
    nextPage->setShortcut(QKeySequence::MoveToNextPage);
    nextPage->setStatusTip(tr("Show next page"));
    connect(nextPage, SIGNAL(triggered()), pdfWidget, SLOT(nextPage()));

    lastPage = new QAction(QIcon(":/icons/Arrow-right-double-22.png"), tr("&Bottom"), this);
    lastPage->setShortcut(QKeySequence::MoveToEndOfDocument);
    lastPage->setStatusTip(tr("Show last page"));
    connect(lastPage, SIGNAL(triggered()), pdfWidget, SLOT(lastPage()));

    QLabel *searchlabel = new QLabel(tr("Search for: "));
    searchLineEdit = new QLineEdit();
    searchLineEdit->setFixedWidth(30 * fontMetrics().width('x'));
    connect(searchLineEdit, SIGNAL(returnPressed()), this, SLOT(findNextClicked()));

    findPrevious = new QAction(QIcon(":/icons/Arrow-left-22.png"), tr("Find previous"), this);
    findPrevious->setShortcut(QKeySequence::FindPrevious);
    findPrevious->setStatusTip(tr("Find previous occurrence of text"));
    connect(findPrevious, SIGNAL(triggered()), this, SLOT(findPreviousClicked()));

    findNext = new QAction(QIcon(":/icons/Arrow-right-22.png"), tr("&Find next"), this);
    findNext->setShortcut(QKeySequence::FindNext);
    findNext->setStatusTip(tr("Find next occurrence of text"));
    connect(findNext, SIGNAL(triggered()), this, SLOT(findNextClicked()));

    QLabel *scalelabel = new QLabel(tr("Scale document: "));
    QComboBox *scaleComboBox = new QComboBox();
    scaleComboBox->insertItems(0, QStringList()
     << "25%"
     << "50%"
     << "75%"
     << "100%"
     << "125%"
     << "150%"
     << "200%"
     << "300%"
     << "400%"
    );
    scaleComboBox->setCurrentIndex(3);
    scaleFactors << 0.25 << 0.5 << 0.75 << 1. << 1.25 << 1.5 << 2. << 3. << 4.;
    connect(scaleComboBox, SIGNAL(currentIndexChanged(int)),
            this, SLOT(scaleDocument(int)));

    QHBoxLayout *hbox = new QHBoxLayout;
    QToolBar *toolbar = new QToolBar();
    hbox->addWidget(toolbar);
    toolbar->addWidget(pagelabel);
    toolbar->addWidget(pageSpinBox);
    toolbar->addAction(firstPage);
    toolbar->addAction(previousPage);
    toolbar->addAction(nextPage);
    toolbar->addAction(lastPage);
    toolbar->addSeparator();
    toolbar->addWidget(searchlabel);
    toolbar->addWidget(searchLineEdit);
    toolbar->addAction(findPrevious);
    toolbar->addAction(findNext);
    toolbar->addSeparator();
    toolbar->addWidget(scalelabel);
    toolbar->addWidget(scaleComboBox);

    tocArea = new QScrollArea;
    tocArea->setWidgetResizable(true);

    scrollArea = new QScrollArea;
    scrollArea->setWidgetResizable(true);

    splitter = new QSplitter(Qt::Horizontal);
    splitter->addWidget(tocArea);
    splitter->addWidget(scrollArea);
    splitter->setStretchFactor(1,1);

    QVBoxLayout *mainlayout = new QVBoxLayout;
    mainlayout->addLayout(hbox);
    mainlayout->addWidget(splitter);
    setLayout(mainlayout);

    resize(1100,1100);
    searchLineEdit->setFocus();
}

void HelpBrowser::save(QSettings &settings)
{
    settings.setValue("HelpBrowser/Geometry", saveGeometry());
    settings.setValue("HelpBrowser/Splitter",splitter->saveState());
}

void HelpBrowser::restore(QSettings &settings)
{
    restoreGeometry(settings.value("HelpBrowser/Geometry").toByteArray());
    splitter->restoreState(settings.value("HelpBrowser/Splitter").toByteArray());
}

void HelpBrowser::keyPressEvent(QKeyEvent *event)
{
    switch (event->key())
    {
    case Qt::Key_Home:
        pdfWidget->setPage(1);
        break;
    case Qt::Key_End:
        pdfWidget->setPage(pdfWidget->document()->numPages());
        break;
    case Qt::Key_PageUp:
        pdfWidget->nextPage();
        break;
    case Qt::Key_PageDown:
        pdfWidget->previousPage();
        break;
    default:
        QWidget::keyPressEvent(event);
    }
}

void HelpBrowser::tocDoubleClicked(QString& itemtext)
{
    showHelp(itemtext);
}

bool HelpBrowser::helpAvailable()
{
    return helpFile->exists();
}

int HelpBrowser::getKey(const HelpEntry helpentry)
{
    for (int i = 0; i < helpEntryKeyListSize; ++i) {
        if (helpEntryKeyList[i].key == helpentry) {
            return i;
        }
    }
    return -1;
}

int HelpBrowser::findDestination(const QString& key)
{
    for (int i = 0; i < HelpListSize; ++i) {
        if (QString(HelpList[i].KeyName) == key) {
            return i;
        }
    }
    return -1;
}

void HelpBrowser::showHelp(const HelpBrowser::HelpEntry entry)
{
    if (!helpFile->exists()){
        QMessageBox::warning(this, Info::TITLE,
                             helpFile->fileName() + tr("\nHelp file does not exists."),
                             QMessageBox::Ok);
        return;
    }
    QString keyname = QString();
    int index = getKey(entry);
    if (index >= 0)
        keyname = helpEntryKeyList[index].keyName;
    showHelp(keyname);
}


void HelpBrowser::showHelp(const QString& keyname)
{
    QString destination = QString();
    int index = findDestination(keyname);
    if (index >= 0)
        destination = HelpList[index].Destination;

    if (!keyname.isEmpty() && destination.isEmpty()) {
        QMessageBox::warning(this, Info::TITLE,
                             keyname + tr(". Cannot find help topic."),
                             QMessageBox::Ok);
        return;
    }
    if (!loaded)
        loaded = load();
    if (!loaded){
        QMessageBox::warning(this, Info::TITLE,
                             helpFile->fileName() + tr(". Could not present help file."),
                             QMessageBox::Ok);
        return;
    }
    Poppler::Document *doc = pdfWidget->document();
    if (doc){
        int pagenumber = 1;
        if (!destination.isEmpty()){
            Poppler::LinkDestination *linkdestination = doc->linkDestination(destination);
            if (linkdestination){
                pagenumber = linkdestination->pageNumber();
                delete linkdestination;
            }
        }
        if (pagenumber > 0) pdfWidget->setPage(pagenumber);
        setWindowModality(Qt::WindowModal);
        show();
    }
}

void HelpBrowser::findPreviousClicked()
{
    if (searchLineEdit->text().isEmpty()) return;
    QRectF location = pdfWidget->searchBackwards(searchLineEdit->text());
    QPoint target = pdfWidget->matrix().mapRect(location).center().toPoint();
    scrollArea->ensureVisible(target.x(), target.y());
}

void HelpBrowser::findNextClicked()
{
    if (searchLineEdit->text().isEmpty()) return;
    QRectF location = pdfWidget->searchForwards(searchLineEdit->text());
    QPoint target = pdfWidget->matrix().mapRect(location).center().toPoint();
    scrollArea->ensureVisible(target.x(), target.y());
}

void HelpBrowser::scaleDocument(int index)
{
    pdfWidget->setScale(scaleFactors[index]);
}
