/*ForcheckIDE/FileViewer.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "FileViewer.h"
#include "Environment.h"
#include "Info.h"

#include <QtWidgets>
#include <QFile>
#include <QTextDocument>
#include <QTextStream>
#include <QTextBlock>

FileViewer::FileViewer(QMainWindow *parent, bool small) : QPlainTextEdit(parent)
{
    myparent = parent;
    mysmall = small;
    curFileName = QString();
    create();
//    setReadOnly(true);        //if set: no cursor, Home, End
    setAttribute(Qt::WA_DeleteOnClose);
}

void FileViewer::create()
{
    showAction = new QAction(this);
    showAction->setCheckable(true);
    connect(showAction, SIGNAL(triggered()), this, SLOT(show()));
    connect(showAction, SIGNAL(triggered()), this, SLOT(setFocus()));

    setLineWrapMode(QPlainTextEdit::NoWrap);
    setFont(QFont("Courier New",9));
}

QSize FileViewer::sizeHint() const
{
    if (mysmall) {

        return QSize(90 * fontMetrics().width('x'),
                     20 * fontMetrics().lineSpacing()); // this determines the initial size of the tabwidget!
    }
    else {
        return QSize(90 * fontMetrics().width('x'),
                     80 * fontMetrics().lineSpacing()); // this determines the initial size of the tabwidget!
    }
}

void FileViewer::focusInEvent(QFocusEvent * event)
{
    emit(InFocus());
    event->accept();
}

void FileViewer::focusOutEvent(QFocusEvent * event)
{
    emit(OutFocus());
    event->accept();
}

void FileViewer::clear()
{
    curFileName = QString();
    QPlainTextEdit::clear();
}

bool FileViewer::loadFile(const QString& fileName)
{
    QFile file(fileName);
    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        QMessageBox::warning(this, Info::TITLE,
                             tr("Cannot open file %1:\n%2.")
                             .arg(fileName)
                             .arg(file.errorString()));
        return false;
    }
    myparent->statusBar()->showMessage(tr("Reading file"));
    QApplication::setOverrideCursor(Qt::WaitCursor);
    setWindowTitle(file.fileName());
    showAction -> setText(fileName);
    curFileName = fileName;

    QTextStream in(&file);
    setPlainText(in.readAll());

    file.close();
    QApplication::restoreOverrideCursor();
    myparent->statusBar()->showMessage(QString());
    return true;
}

bool FileViewer::saveFile()
{
    QFile file(curFileName);
    if (!file.open(QFile::WriteOnly | QFile::Text)) {
        QMessageBox::warning(this, Info::TITLE,
                             tr("Cannot write file %1:\n%2.")
                             .arg(curFileName)
                             .arg(file.errorString()));
        return false;
    }

    myparent->statusBar()->showMessage(tr("Saving file"));
    QTextStream out(&file);
    QApplication::setOverrideCursor(Qt::WaitCursor);
    out << toPlainText();
    QApplication::restoreOverrideCursor();

    myparent->statusBar()->showMessage(QString());
    return true;
}

void FileViewer::putText(const QString& text)
{
    setPlainText(text);
    moveCursor(QTextCursor::End);
    show();
}

void FileViewer::appendText(const QString& text)
{
    appendPlainText(text);
    moveCursor(QTextCursor::End);
}

void FileViewer::mouseDoubleClickEvent(QMouseEvent *event )
{
    QPoint pos = event->pos();
    QTextCursor cursor = cursorForPosition(pos);
    cursor.movePosition(QTextCursor::StartOfLine);
    cursor.select(QTextCursor::LineUnderCursor);
    QString line = cursor.selectedText();
    if (!line.contains(":")){
// move one line up
        cursor.movePosition(QTextCursor::Up);
        cursor.select(QTextCursor::LineUnderCursor);
        line = cursor.selectedText();
        if (!line.contains(":")){
// move another line up
            cursor.movePosition(QTextCursor::Up);
            cursor.select(QTextCursor::LineUnderCursor);
            line = cursor.selectedText();
            if (!line.contains(":")) return;
        }
    }
    highlightCurrentLine();
    QString filename;
    int linnum;
    parseLine(line, filename, linnum);
    if (!filename.isEmpty()){
        if (myEnvironment->getEditor().isEmpty())
            emit requestEdit(filename, linnum);
        else
            myEnvironment->execEditor(filename, linnum);
    }
}

void FileViewer::parseLine(const QString& line, QString& filename, int& linnum)
{
    filename.clear();
    linnum = 1;

    int ib, ie, n;
    n = line.length();
    ib = line.indexOf("file: ");
// get filename
    if (ib >=0) {
        ib = ib + 6;
        filename = line.right(n - ib);
        ie = filename.indexOf(",");
        if (ie > 0)
            filename = filename.left(ie);
        else {
            ie = filename.indexOf(")");
            if (ie > 0)
                filename = filename.left(ie);
        }
        filename.trimmed();
// get line number
        ib = line.indexOf("line: ");
        if (ib >= 0){
            ib = ib + 6;
            QString ls = line.right(n - ib);
            ie = ls.indexOf(")");
            if (ie >= 0){
                ls = ls.left(ie);
                linnum = ls.toInt();
            }
            else {
                ie = ls.indexOf(",");
                if (ie >= 0){
                    ls = ls.left(ie);
                    linnum = ls.toInt();
                }
            }
        }
    }
// gnu id?
    else if (line.endsWith(":")) {
        ie = line.indexOf(":");
        if (ie > 0){
            filename = line.left(ie);
            ib = filename.lastIndexOf(" ");
            if (ib >= 0){
                filename.remove(0,ib+1);
            }
            filename.trimmed();
            ib = ie + 1;
            QString ls = line.right(n - ib);
            ie = ls.indexOf(".");
            if (ie >= 0)
              ls = ls.left(ie);
            else {
                ie = ls.indexOf(":");
                if (ie >= 0)
                  ls = ls.left(ie);
            }
            linnum = ls.toInt();
        }
    }

}

void FileViewer::highlightCurrentLine()
{
    QList<QTextEdit::ExtraSelection> extraSelections;

    if (!isReadOnly()) {
        QTextEdit::ExtraSelection selection;

        QColor lineColor = QColor(Qt::yellow).lighter(160);

        selection.format.setBackground(lineColor);
        selection.format.setProperty(QTextFormat::FullWidthSelection, true);
        selection.cursor = textCursor();
        selection.cursor.clearSelection();
        extraSelections.append(selection);
    }

    setExtraSelections(extraSelections);
}
