/*ForcheckIDE/FileKinds.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "FileKinds.h"

#include "Constants.h"
#include "Environment.h"

QString FileKind::fileKind2String(const ETFileKind filekind)
{
        switch (filekind) {
        case FileKind::Project:
                return "project";
        case FileKind::Source:
                return "fortran source";
        case FileKind::PreSource:
                return "fortran pre-source";
        case FileKind::CSource:
                return "C source";
        case FileKind::Include:
                return "fortran include";
        case FileKind::LibOutput:
                return "create/update library";
        case FileKind::LibRef:
                return "reference library";
        case FileKind::Listing:
                return "listing";
        case FileKind::Report:
                return "report";
        case FileKind::ReferenceStructure:
                return "reference structure";
        case FileKind::ModDep:
                return "module dependencies";
        case FileKind::Any:
                return "any";
        case FileKind::None:
                return "none";
        case FileKind::Unknown:
                return "unknown";
        default:
                return "really unknown";
        }
}

FileKind::ETFileKind FileKind::string2FileKind(const QString filekindstring)
{
#define TESTFILEKIND(filekind) if (fileKind2String(filekind).toLower() == filekindstring.toLower()) return filekind;
        TESTFILEKIND(Project)
        TESTFILEKIND(Make)
        TESTFILEKIND(Source)
        TESTFILEKIND(PreSource)
        TESTFILEKIND(CSource)
        TESTFILEKIND(Include)
        TESTFILEKIND(LibOutput)
        TESTFILEKIND(LibRef)
        TESTFILEKIND(Listing)
        TESTFILEKIND(Report)
        TESTFILEKIND(ReferenceStructure)
        TESTFILEKIND(ModDep)
        TESTFILEKIND(None)
        return Unknown;
#undef TESTFILEKIND
}

FileKind::ETFileKind FileKind::int2FileKind(const int filekindint)
{
#define TESTFILEKIND(filekind) if (filekind == filekindint) return filekind;
        TESTFILEKIND(Project)
        TESTFILEKIND(Make)
        TESTFILEKIND(Source)
        TESTFILEKIND(PreSource)
        TESTFILEKIND(CSource)
        TESTFILEKIND(Include)
        TESTFILEKIND(LibOutput)
        TESTFILEKIND(LibRef)
        TESTFILEKIND(Listing)
        TESTFILEKIND(Report)
        TESTFILEKIND(ReferenceStructure)
        TESTFILEKIND(ModDep)
        TESTFILEKIND(None)
        return Unknown;
#undef TESTFILEKIND
}

FileKind::ETFileKind FileKind::ext2FileKind(const QString &fileext)
{
    if (QString::compare(fileext, myEnvironment->getProjectExt(), Qt::CaseInsensitive) == 0)
        return Project;
    if (QString::compare(fileext, myEnvironment->getMakeExt(), Qt::CaseInsensitive) == 0)
        return Make;
    if (QString::compare(fileext, myEnvironment->getSourceExt(), Qt::CaseInsensitive) == 0)
        return Source;
    if (QString::compare(fileext, myEnvironment->getPreSourceExt(), Qt::CaseInsensitive) == 0)
        return PreSource;
    if (QString::compare(fileext, myEnvironment->getCSourceExt(), Qt::CaseInsensitive) == 0)
        return CSource;
    if (QString::compare(fileext, myEnvironment->getIncludeExt(), Qt::CaseInsensitive) == 0)
        return Include;
    if (QString::compare(fileext, myEnvironment->getLibraryExt(), Qt::CaseInsensitive) == 0)
        return LibRef;
    if (QString::compare(fileext, myEnvironment->getListingExt(), Qt::CaseInsensitive) == 0)
        return Listing;
    if (QString::compare(fileext, myEnvironment->getReportExt(), Qt::CaseInsensitive) == 0)
        return Report;
    if (QString::compare(fileext, myEnvironment->getRefstructExt(), Qt::CaseInsensitive) == 0)
        return ReferenceStructure;
    if (QString::compare(fileext, myEnvironment->getModDepExt(), Qt::CaseInsensitive) == 0)
        return ModDep;
    if (fileext.isEmpty())
        return None;
    return Unknown;
}

bool FileKind::canBeInProject(const ETFileKind filekind)
{
        return (filekind >= Source) && (filekind < Count);
}

OptionsKind::ETOptionsKind FileKind::getOptionKind(const ETFileKind filekind)
{
        switch (filekind)
        {
        case Project:
                return OptionsKind::OptionsKindProject;
        case Source:
                return OptionsKind::OptionsKindSource;
        case PreSource:
                return OptionsKind::OptionsKindSource;
        case Include:
                return OptionsKind::OptionsKindInclude;
        case LibOutput:
                return OptionsKind::OptionsKindOutLib;
        case LibRef:
                return OptionsKind::OptionsKindRefLib;
        case Report:
                return OptionsKind::OptionsKindReport;
        case ReferenceStructure:
                return OptionsKind::OptionsKindRefStruct;
        case ModDep:
                return OptionsKind::OptionsKindModDep;
        case Listing:
                return OptionsKind::OptionsKindListing;
        default:
                return OptionsKind::OptionsKindNone;
        }
}

QString FileKind::getIcon(const ETFileKind kind) {
        switch (kind) {
        case FileKind::Project:
            return ":/icons/Project-16.png";
        case FileKind::Make:
            return ":/icons/Make-16.png";
        case FileKind::Source:
            return ":/icons/Source-16.png";
        case FileKind::PreSource:
            return ":/icons/Source-16.png";
        case FileKind::CSource:
            return ":/icons/Source-16.png";
        case FileKind::Include:
            return ":/icons/Source-16.png";
        case FileKind::LibOutput:
            return ":/icons/LibraryCreate-16.png";
        case FileKind::LibRef:
            return ":/icons/Library-16.png";
        case FileKind::Listing:
            return ":/icons/Listing-16.png";
        case FileKind::Report:
            return ":/icons/Report-16.png";
        case FileKind::ReferenceStructure:
            return ":/icons/ReferenceStructure-16.png";
        case FileKind::ModDep:
            return ":/icons/ReferenceStructure-16.png";
        default:
            return QString();
        }
}

QString FileKind::fileDir(const QString &filename)
{
        int ind = filename.lastIndexOf("\\");
        if (ind < 0)
            ind = filename.lastIndexOf("/");
        if (ind)
            return filename.left(ind);
        else
            return filename;
}

QString FileKind::fileExt(const QString &filename)
{
    return '.' + filename.section('.', -1);
}

bool FileKind::isFppFile(const QString &filename)
{
#if defined(Q_OS_WIN)
    return fileExt(filename).toLower() == myEnvironment->getPreSourceExt().toLower();
#else
    return fileExt(filename) == myEnvironment->getPreSourceExt();
#endif
}

bool FileKind::isDefaultFfFile(const QString &filename)
{
    return (fileExt(filename).toLower() == ".f90") ||
           (fileExt(filename).toLower() == ".f95") ||
           (fileExt(filename).toLower() == ".f03") ||
           (fileExt(filename).toLower() == ".f2003") ||
           (fileExt(filename).toLower() == ".f08") ||
           (fileExt(filename).toLower() == ".f2008");
}
