/*ForcheckIDE/Environment.h*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#ifndef ENVIRONMENT_H
#define ENVIRONMENT_H

#include "Constants.h"
#include "Version.h"
#include "FileKinds.h"

#include <QDateTime>
#include <QObject>
#include <QString>
#include <QStringList>

class QDate;
class QPoint;
class QSize;
class QByteArray;
class QFile;
class QProcess;
class PageSetup;
class Options;
class HelpBrowser;
class Environment : public QObject
{
      Q_OBJECT

      public:
             //Procedures
             Environment();
             ~Environment();
             void saveStatus(const QByteArray &geometry,
                             const QByteArray &state,
                             const QByteArray &rightsplitterstate,
                             const QByteArray &mainsplitterstate);
             void restoreStatus(QByteArray &geometry,
                                QByteArray &state,
                                QByteArray &rightsplitterstate,
                                QByteArray &mainsplitterstate);
             void saveIdeSettings();
             void restoreIdeSettings();
             bool getForcheckPWDInfo(QString &userid,
                                     QString &licensecat,
                                     QString &serial,
                                     QDate &exp_license,
                                     QDate &exp_maintenance);
             Version getFCKversion();
             QString getTempDirDefault();
             QString getFckDataDir() const {return fckDataDir;}
             QString getFckUserDataDir() const {return fckUserDataDir;}
             QString getFckIDEDocDir() const {return fckIDEDocDir;}
             QString getWorkingDir() const {return workingDir;}
             void loadDefaultOptions();
             void saveDefaultOptions();
             void setupPage();
             void printFile(QFile& file);

             QString getProjectExt() const      {return Extensions.Ext_project;}
             QString getMakeExt() const         {return Extensions.Ext_make;}
             QString getSourceExt() const       {return Extensions.Ext_source;}
             QString getPreSourceExt() const    {return Extensions.Ext_presource;}
             QString getCSourceExt() const      {return Extensions.Ext_csource;}
             QString getIncludeExt() const      {return Extensions.Ext_include;}
             QString getLibraryExt() const      {return Extensions.Ext_library;}
             QString getListingExt() const      {return Extensions.Ext_listing;}
             QString getReportExt()    const    {return Extensions.Ext_report;}
             QString getRefstructExt() const    {return Extensions.Ext_refstruct;}
             QString getModDepExt() const       {return Extensions.Ext_moddep;}
             QString getBinExt() const          {return Constants::DefaultExt_Bin;}
             QString getExtension(FileKind::ETFileKind filekind) const;
             QString addDot(const QString& ext);

             bool getLibrariesPreferencesUpdate() const {return updateLibrary;}
             bool getLibrariesPreferencesCompress() const {return compressLibrary;}

             QString getFckConfigurationFilename() const {return fckconfigurationfilename;} // Get the name of the fck configuration file
             QString getIdeConfigurationFilename() const {return ideconfigurationfilename;} // Get the name of the ide configuration file
             QString getEmulationFilename() const {return emulationfilename;} // Get the name of the compiler emulation file
             QString getDefaultOptionsFilename() const {return defaultoptionsfilename;} // Get the name of the default options file
             QString getDepFilename() const {return depfilename;} // Get the name of the dependencies file
             QString getDefaultReportFilename() const {return defaultreportfilename;} // Get the name of the default report file
             QString getCommandFilename() const {return commandfilename;} // Get the name of the command file
             QString getMsgFilename() const {return msgfilename;} // Get the name of the forcheck messages file
             QString getEditor() const {return editor;}
             QString getEditorCommandline() const {return editorCommandline;}
             bool getEditorAsViewer() const {return editorAsViewer;}

             QString getCompiler() const {return compiler;}
             QString getCompilerOptions() const {return compilerOptions;}
             QString getCompilerDebugOptions() const {return compilerDebugOptions;}
             QString getcCompiler() const {return cCompiler;}
             QString getcCompilerOptions() const {return cCompilerOptions;}
             QString getLinkerOptions() const {return linkerOptions;}
             QString getBuildUtility() const {return buildUtility;}       

             bool getVerProjectFiles() const {return verProjectFiles;}
             bool getVerIncludeDep() const {return verIncludeDep;}
             bool getAutosaveFilesSearch() const {return autosaveFilesSearch;}
             bool getAutosaveFilesAnalysis() const {return autosaveFilesAnalysis;}
             bool getAutosaveFilesMake() const {return autosaveFilesMake;}
             bool getAutosaveProject() const {return autosaveProject;}

             QString *commandFilename();          // Get the name for the command file to run forcheck
             void getEmulation(const QString& filename, QString& emulation, bool& initfmt);

             Options *getFactoryOptions() { return myFactoryOptions; }
             Options *getDefaultOptions() { return myDefaultOptions; }

             QDateTime getdatetimePreferences() const {return datetimePreferences;}

             void setWorkingDir(const QString& cwd);
             void setPageOptions();
             void setEmulationFilename(const QString& filename);
             void setExtension(FileKind::ETFileKind filekind, const QString& ext);
             void setEditor(const QString& filename);
             void setEditorCommandline(const QString& commandline);
             void setEditorAsViewer(const bool editorasviewer);

             void setCompiler(const QString&);
             void setCompilerOptions(const QString&);
             void setCompilerDebugOptions(const QString&);
             void setcCompiler(const QString&);
             void setcCompilerOptions(const QString&);
             void setLinkerOptions(const QString&);
             void setBuildUtility(const QString&);
             void setLibrariesPreferences(const bool& update, const bool& compress);

             void setVerProjectFiles(const bool b) {verProjectFiles = b;}
             void setVerIncludeDep(const bool b) {verIncludeDep = b;}
             void setAutosaveFilesSearch(const bool b) {autosaveFilesSearch = b;}
             void setAutosaveFilesAnalysis(const bool b) {autosaveFilesAnalysis = b;}
             void setAutosaveFilesMake(const bool b) {autosaveFilesMake = b;}
             void setAutosaveProject(const bool b) {autosaveProject = b;}

             void setDateTimePreferences() {datetimePreferences = QDateTime::currentDateTime();}

             int execForcheck(const QStringList& args);
             int execMake(const QString& makefilename);
             int execInterfaceBuilder(const QString& libname,
                                      const QString& outputfilename,
                                      QString& report);
             void execEditor(const QString& filename, int linnum = -1);
             void helpMain();
             void wwwBrowser(const QString& url);
             void sendEmail(const QString& emailAddress);

             //Variables
             enum {MaxRecentFiles = 10};
             QStringList recentFiles;
             QStringList *IPnames;
             bool updateLibrary;
             bool compressLibrary;
             HelpBrowser *helpBrowser;
             QProcess *process;

      private:
             //Variables
             PageSetup *pageSetup;
             QString fckBinDir;                 // Installation directory for binaries
             QString fckDataDir;                // Shared data directory for support files
             QString fckDocDir;                 // Shared data directory for documentation files
             QString fckIDEDataDir;             // Shared data directory for support files of the IDE
             QString fckIDEDocDir;              // Shared data directory for documentation files of the IDE
             QString fckUserDataDir;            // User data directory
             QString fckConfigDir;              // User configuration directory
             QString workingDir;                // Current working directory
             QString fckExecutable;             // Forcheck executable
             QString interfExecutable;          // Interface builder executable
             QString fckconfigurationfilename;  // configuration filename
             QString ideconfigurationfilename;  // configuration filename
             QString defaultoptionsfilename;    // default options file
             QString emulationfilename;         // compiler emulation file
             QString msgfilename;               // messages file
             QString depfilename;               // dependencies file
             QString defaultreportfilename;     // default report file
             QString commandfilename;           // command file
             QString editor;                    // filename of the editor
             QString editorCommandline;         // editors commandline template
             bool editorAsViewer;               // use the editor as file viewer
             QString compiler;                  // default compiler
             QString compilerOptions;           // default compiler options
             QString compilerDebugOptions;      // default compiler debug options
             QString cCompiler;                 // default C compiler
             QString cCompilerOptions;          // default C compiler options
             QString linkerOptions;             // default linker options for makefile
             QString buildUtility;              // utility to build the project
             QDateTime datetimePreferences;     // date/time of last change of preferences relevant for make

             //IDESettings
             bool verProjectFiles;              // verify project files at open project
             bool verIncludeDep;                // verify include dependencies at open project
             bool autosaveFilesSearch;          // save all open files at global search
             bool autosaveFilesAnalysis;        // save all open files at analysis
             bool autosaveFilesMake;            // save all open files at make
             bool autosaveProject;              // save project at close project

             Options *myFactoryOptions;
             Options *myDefaultOptions;

             struct {
                     QString Ext_project;     // project
                     QString Ext_make;        // make
                     QString Ext_source;      // source
                     QString Ext_presource;   // pre-source
                     QString Ext_csource;     // C-source
                     QString Ext_include;     // include
                     QString Ext_library;     // library
                     QString Ext_listing;     // listing
                     QString Ext_report;      // report
                     QString Ext_refstruct;   // reference structure
                     QString Ext_moddep;      // module dependencies
             } Extensions;
};

extern Environment *myEnvironment;

#endif
