/*ForcheckIDE/Environment.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "Environment.h"
#include "PageSetup.h"
#include "Options.h"
#include "IniFile.h"
#include "Info.h"
#include "HelpBrowser.h"

#include <QtWidgets>
#include <QDate>
#include <QSettings>
#include <QDir>
#include <QUrl>
#include <QFile>
#include <QIODevice>
#include <QTextStream>
#include <QDesktopServices>
#include <QProcess>

#if defined(Q_OS_WIN)
#include <Shlobj.h>
#endif

Environment *myEnvironment;

Environment::Environment()
{
    process = new QProcess();
    setWorkingDir(QDir::currentPath());
//get installation directory of the analyser
#if defined(Q_OS_WIN)
    Version versionInfo = Version(Info::VERSION);
    QString fckdir;
    QSettings settings("HKEY_LOCAL_MACHINE\\Software\\Forcheck\\Forcheck for WIndows",
                       QSettings::NativeFormat);
    settings.beginGroup(versionInfo.getMajorVersion());
        fckdir = settings.value("FCKDIR",
                                "C:\\Program Files\\Forcheck").toString();
    settings.endGroup();
    fckBinDir = fckdir + "\\bin\\";
    fckDataDir = fckdir + "\\share\\";
    fckDocDir = fckdir + "\\doc\\";
    fckIDEDocDir = fckdir + "\\ForcheckIDE\\doc\\";

    // get APPDATA directory
    char szPath[256];
    HRESULT hr = SHGetFolderPathA(NULL, CSIDL_APPDATA, NULL, 0, szPath);
    QString appdir = QString(szPath);
    fckConfigDir = appdir + "\\Forcheck\\";
    QDir appDir(appdir);
    if (!appDir.exists("Forcheck"))
        appDir.mkdir("Forcheck");
    // get LOCALAPPDATA directory
    hr = SHGetFolderPathA(NULL, CSIDL_LOCAL_APPDATA, NULL, 0, szPath);
    QString localappdir = QString(szPath);
    fckUserDataDir = localappdir + "\\Forcheck\\";
    QDir localAppDir(localappdir);
    if (!localAppDir.exists("Forcheck"))
        localAppDir.mkdir("Forcheck");
#elif defined(Q_OS_LINUX)
    QString fckdir = getenv("FCKDIR");
    if (fckdir == NULL) fckdir = "/usr/local";
    fckBinDir = fckdir + "/bin/";
    fckDataDir = fckdir + "/share/forcheck/";
    fckDocDir = fckDataDir + "doc/";
    QString fckidedir = getenv("FCKIDEDIR");
    if (fckidedir == NULL) fckidedir = "/usr/local";
    fckIDEDataDir = fckidedir + "/share/forcheckIDE/";
    fckIDEDocDir = fckIDEDataDir + "doc/";

    // get CONFIG and APP DATA directory
    QString appdir = getenv("XDG_CONFIG_HOME");
    if (appdir == NULL)
        appdir = QDir::homePath() + "/.config";
    fckConfigDir = appdir + "/forcheck/";
    QDir appDir(appdir);
    if (!appDir.exists("forcheck"))
        appDir.mkdir("forcheck");
    fckUserDataDir = getenv("XDG_RUNTIME_DIR");
    if (fckUserDataDir == NULL)
        fckUserDataDir = ".";
    fckUserDataDir = fckUserDataDir + "/";
#endif

    fckExecutable = fckBinDir + Constants::FckExecutable;
    interfExecutable = fckBinDir + Constants::InterfExecutable;
    emulationfilename = fckDataDir + Constants::DefaultEmulationFilename;
    msgfilename = fckDataDir + Constants::FckMsgFilename;

    depfilename = fckUserDataDir + Constants::DepFilename;
    defaultreportfilename = fckUserDataDir + Constants::DefaultReportFilename;
    commandfilename = fckUserDataDir + Constants::CommandFilename;

    fckconfigurationfilename = fckConfigDir + Constants::FckConfigurationFilename;
    ideconfigurationfilename = fckConfigDir + Constants::IDEConfigurationFilename;
    defaultoptionsfilename = fckConfigDir + Constants::DefaultOptionsFilename;

    datetimePreferences = QDateTime::currentDateTime();

    //create helpbrowser object and set help file
    QString helpfilename = fckDocDir + Constants::HelpFilename;
    helpBrowser = new HelpBrowser(helpfilename);

    pageSetup = new PageSetup();
    myFactoryOptions = new Options(0, OptionsKind::OptionsKindFactory);
    myDefaultOptions = new Options(myFactoryOptions, OptionsKind::OptionsKindDefault);
    IPnames = new QStringList();
    loadDefaultOptions();
}

Environment::~Environment()
{
    delete process;
    delete IPnames;
    delete myDefaultOptions;
    delete myFactoryOptions;
    delete pageSetup;
    delete helpBrowser;
}

void Environment::setWorkingDir(const QString& cwd)
{
    workingDir = cwd;
    QDir::setCurrent(cwd);
    process->setWorkingDirectory(cwd);
}

void Environment::saveStatus(const QByteArray &geometry,
                             const QByteArray &state,
                             const QByteArray &rightsplitterstate,
                             const QByteArray &mainsplitterstate)
{
    QSettings settings;
    settings.setValue("MainWindow/Geometry", geometry);
    settings.setValue("MainWindow/RecentFiles", recentFiles);
    settings.setValue("MainWindow/Properties", state);
    settings.setValue("MainWindow/RightSplitter", rightsplitterstate);
    settings.setValue("MainWindow/MainSplitter", mainsplitterstate);
    //store page setup
    pageSetup->save(settings);
    helpBrowser->save(settings);
}

void Environment::restoreStatus(QByteArray &geometry,
                                QByteArray &state,
                                QByteArray &rightsplitterstate,
                                QByteArray &mainsplitterstate)
{
    QSettings settings;
    //get geometry
    geometry = settings.value("MainWindow/Geometry").toByteArray();
    recentFiles = settings.value("MainWindow/RecentFiles").toStringList();
    state = settings.value("MainWindow/Properties").toByteArray();
    rightsplitterstate = settings.value("MainWindow/RightSplitter").toByteArray();
    mainsplitterstate = settings.value("MainWindow/MainSplitter").toByteArray();
    //get page setup
    pageSetup->restore(settings);
    pageSetup->setupPrinter();
    helpBrowser->restore(settings);
}

void Environment::saveIdeSettings()
{
    IniFile ideConfSettings(ideconfigurationfilename);
    if (!ideConfSettings.create()){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             ideconfigurationfilename+tr(". Could not create IDE configuration file."),
                             QMessageBox::Ok);
    };
//store datetime stamp
    ideConfSettings.beginGroup("General");
      ideConfSettings.setValue("DateTime", datetimePreferences.toString());
    ideConfSettings.endGroup();
//store Include paths
    ideConfSettings.beginGroup("Directories");
      ideConfSettings.setValue("INCLUDE", IPnames->join(";"));
    ideConfSettings.endGroup();
//store emulationfile
    QString emulstring = emulationfilename;
// if fck configuration file present then concatenate its name
    if (QFile(fckconfigurationfilename).exists()) {
            emulstring += ';';
            emulstring += fckconfigurationfilename;
    }
    ideConfSettings.beginGroup("Files");
      ideConfSettings.setValue("FCKCNF", emulstring);
    ideConfSettings.endGroup();
//store filename extensions
      ideConfSettings.beginGroup("Extensions");
      ideConfSettings.setValue("PRJ_EXT", Extensions.Ext_project);
      ideConfSettings.setValue("MAK_EXT", Extensions.Ext_make);
      ideConfSettings.setValue("SRC_EXT", Extensions.Ext_source);
      ideConfSettings.setValue("PRESRC_EXT", Extensions.Ext_presource);
      ideConfSettings.setValue("CSRC_EXT", Extensions.Ext_csource);
      ideConfSettings.setValue("INC_EXT", Extensions.Ext_include);
      ideConfSettings.setValue("LIB_EXT", Extensions.Ext_library);
      ideConfSettings.setValue("LST_EXT", Extensions.Ext_listing);
      ideConfSettings.setValue("REP_EXT", Extensions.Ext_report);
      ideConfSettings.setValue("REFSTRUCT_EXT", Extensions.Ext_refstruct);
      ideConfSettings.setValue("MODDEP_EXT", Extensions.Ext_moddep);
    ideConfSettings.endGroup();
//store editor
    ideConfSettings.beginGroup("Main");
      ideConfSettings.setValue("EDITOR", editor);
      ideConfSettings.setValue("EDITORCMDLN", editorCommandline);
      ideConfSettings.setValue("EDITORASVIEWER", editorAsViewer);
      ideConfSettings.setValue("BUILDUTILITY", buildUtility);           //to be downwards compatible
    ideConfSettings.endGroup();
//store IDESettings
    ideConfSettings.beginGroup("IDESettings");
      ideConfSettings.setValue("VERPROJECTFILES", verProjectFiles);
      ideConfSettings.setValue("VERINCLUDEDEP", verIncludeDep);
      ideConfSettings.setValue("AUTOSAVEFILESSEARCH", autosaveFilesSearch);
      ideConfSettings.setValue("AUTOSAVEFILESANALYSIS", autosaveFilesAnalysis);
      ideConfSettings.setValue("AUTOSAVEFILESMAKE", autosaveFilesMake);
      ideConfSettings.setValue("AUTOSAVEPROJECT", autosaveProject);
    ideConfSettings.endGroup();
//store build info
    ideConfSettings.beginGroup("Build");
      ideConfSettings.setValue("COMPILER", compiler);
      ideConfSettings.setValue("COMPILER_OPTIONS", compilerOptions);
      ideConfSettings.setValue("COMPILER_DEBUG_OPTIONS", compilerDebugOptions);
      ideConfSettings.setValue("C_COMPILER", cCompiler);
      ideConfSettings.setValue("C_COMPILER_OPTIONS", cCompilerOptions);
      ideConfSettings.setValue("LINKER_OPTIONS", linkerOptions);
      ideConfSettings.setValue("BUILDUTILITY", buildUtility);
    ideConfSettings.endGroup();
//store library settings
    ideConfSettings.beginGroup("Miscellaneous");
      ideConfSettings.setValue("UpdateForcheckLibraryFile", updateLibrary);
      ideConfSettings.setValue("CompressAfterAnalysis", compressLibrary);
    ideConfSettings.endGroup();

    ideConfSettings.close();
}

void Environment::restoreIdeSettings()
{
    IniFile ideConfSettings(ideconfigurationfilename);
    if (!ideConfSettings.open()){
//        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
//                             ideconfigurationfilename+tr(". Could not open IDE configuration file."),
//                             QMessageBox::Ok);
        Extensions.Ext_project = Constants::DefaultExt_Project;
        Extensions.Ext_make = Constants::DefaultExt_Make;
        Extensions.Ext_source = Constants::DefaultExt_Source;
        Extensions.Ext_presource = Constants::DefaultExt_PreSource;
        Extensions.Ext_csource = Constants::DefaultExt_CSource;
        Extensions.Ext_include = Constants::DefaultExt_Include;
        Extensions.Ext_library = Constants::DefaultExt_Library;
        Extensions.Ext_listing = Constants::DefaultExt_Listing;
        Extensions.Ext_report = Constants::DefaultExt_Report;
        Extensions.Ext_refstruct = Constants::DefaultExt_Refstruct;
        Extensions.Ext_moddep = Constants::DefaultExt_ModDep;

        editor = Constants::DefaultEditorFileName;
        editorCommandline = Constants::DefaultEditorCommandLine;
        editorAsViewer = false;
        buildUtility = Constants::DefaultBuildUtility;

        compiler = Constants::DefaultCompiler;
        compilerOptions = Constants::DefaultCompilerOptions;
        compilerDebugOptions = Constants::DefaultCompilerDebugOptions;
        cCompiler = Constants::DefaultcCompiler;
        cCompilerOptions = Constants::DefaultcCompilerOptions;
        linkerOptions = Constants::DefaultLinkerOptions;
        buildUtility = Constants::DefaultBuildUtility;

        return;
    };
//get datetime stamp
    ideConfSettings.beginGroup("General");
      datetimePreferences = QDateTime :: fromString(ideConfSettings.value("DateTime").toString());
    ideConfSettings.endGroup();
//get Include paths
    ideConfSettings.beginGroup("Directories");
      QString tmp = ideConfSettings.value("INCLUDE").toString();
      if (!tmp.isEmpty()) IPnames->append(tmp.split(";"));
    ideConfSettings.endGroup();
//get emulationfile
    ideConfSettings.beginGroup("Files");
      QStringList cnf = ideConfSettings.value("FCKCNF").toString().split(";");
      emulationfilename = cnf[0];
      if (cnf.size() > 1) fckconfigurationfilename = cnf[1];
    ideConfSettings.endGroup();
//get filename extensions
    ideConfSettings.beginGroup("Extensions");
      Extensions.Ext_project = addDot(ideConfSettings.value("PRJ_EXT",Constants::DefaultExt_Project).toString());
      Extensions.Ext_make = addDot(ideConfSettings.value("MAK_EXT",Constants::DefaultExt_Make).toString());
      Extensions.Ext_source = addDot(ideConfSettings.value("SRC_EXT",Constants::DefaultExt_Source).toString());
      Extensions.Ext_presource = addDot(ideConfSettings.value("PRESRC_EXT",Constants::DefaultExt_PreSource).toString());
      Extensions.Ext_csource = addDot(ideConfSettings.value("CSRC_EXT",Constants::DefaultExt_CSource).toString());
      Extensions.Ext_include = addDot(ideConfSettings.value("SRC_INC",Constants::DefaultExt_Include).toString());
      Extensions.Ext_library = addDot(ideConfSettings.value("LIB_EXT",Constants::DefaultExt_Library).toString());
      Extensions.Ext_listing = addDot(ideConfSettings.value("LST_EXT",Constants::DefaultExt_Listing).toString());
      Extensions.Ext_report = addDot(ideConfSettings.value("REP_EXT",Constants::DefaultExt_Report).toString());
      Extensions.Ext_refstruct = addDot(ideConfSettings.value("REFSTRUCT_EXT",Constants::DefaultExt_Refstruct).toString());
      Extensions.Ext_moddep = addDot(ideConfSettings.value("MODDEP_EXT",Constants::DefaultExt_ModDep).toString());
    ideConfSettings.endGroup();
//get editor
      ideConfSettings.beginGroup("Main");
        editor = ideConfSettings.value("EDITOR",Constants::DefaultEditorFileName).toString();
        editorCommandline = ideConfSettings.value("EDITORCMDLN",Constants::DefaultEditorCommandLine).toString();
        editorAsViewer = ideConfSettings.value("EDITORASVIEWER",false).toBool();
        buildUtility = ideConfSettings.value("BUILDUTILITY",Constants::DefaultBuildUtility).toString();     //to be upwards compatible
      ideConfSettings.endGroup();
//get IDESettings
      ideConfSettings.beginGroup("IDESettings");
        verProjectFiles = ideConfSettings.value("VERPROJECTFILES",false).toBool();
        verIncludeDep = ideConfSettings.value("VERINCLUDEDEP",false).toBool();
        autosaveFilesSearch = ideConfSettings.value("AUTOSAVEFILESSEARCH",false).toBool();
        autosaveFilesAnalysis = ideConfSettings.value("AUTOSAVEFILESANALYSIS",false).toBool();
        autosaveFilesMake = ideConfSettings.value("AUTOSAVEFILESMAKE",false).toBool();
        autosaveProject = ideConfSettings.value("AUTOSAVEPROJECT",false).toBool();
      ideConfSettings.endGroup();
//get build settings
      ideConfSettings.beginGroup("Build");
        compiler = ideConfSettings.value("COMPILER",Constants::DefaultCompiler).toString();
        compilerOptions = ideConfSettings.value("COMPILER_OPTIONS",Constants::DefaultCompilerOptions).toString();
        compilerDebugOptions = ideConfSettings.value("COMPILER_DEBUG_OPTIONS",Constants::DefaultCompilerDebugOptions).toString();
        cCompiler = ideConfSettings.value("C_COMPILER",Constants::DefaultcCompiler).toString();
        cCompilerOptions = ideConfSettings.value("C_COMPILER_OPTIONS",Constants::DefaultcCompilerOptions).toString();
        linkerOptions = ideConfSettings.value("LINKER_OPTIONS",Constants::DefaultLinkerOptions).toString();
        buildUtility = ideConfSettings.value("BUILDUTILITY",Constants::DefaultBuildUtility).toString();
      ideConfSettings.endGroup();
//get library settings
    ideConfSettings.beginGroup("Miscellaneous");
      updateLibrary = ideConfSettings.value("UpdateForcheckLibraryFile",true).toBool();
      compressLibrary = ideConfSettings.value("CompressAfterAnalysis",false).toBool();
    ideConfSettings.endGroup();

    ideConfSettings.close();
}

QString Environment::addDot(const QString& ext)
{
    if (ext.isEmpty() || ext.left(1)==".")
        return ext;
    return "." + ext;
}

void Environment::saveDefaultOptions()
{
    if (myDefaultOptions->isChanged()){
        Version versionInfo(Info::VERSION);
        IniFile defOptionsSettings(defaultoptionsfilename);
        if (!defOptionsSettings.create()){
            QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                 defaultoptionsfilename+tr(". Could not create default options file."),
                                 QMessageBox::Ok);
            return;
        };
        defOptionsSettings.beginGroup("Main");
            defOptionsSettings.setValue("ident", "FORCHECK default options file");
            defOptionsSettings.setValue("version", versionInfo.getMajorVersion());
            defOptionsSettings.setValue("release", versionInfo.getMinorVersion());
        defOptionsSettings.endGroup();
        defOptionsSettings.beginGroup("Default Options");
          myDefaultOptions->save(defOptionsSettings);
        defOptionsSettings.endGroup();
        defOptionsSettings.close();
    }
}

void Environment::loadDefaultOptions()
{
    IniFile defOptionsSettings(defaultoptionsfilename);
    if (!defOptionsSettings.exists()){
        return;
    }
    if (!defOptionsSettings.open()){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             defaultoptionsfilename+tr(". Could not open default options file."),
                             QMessageBox::Ok);
        return;
    };
    if (defOptionsSettings.isEmpty()) {
        //        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
        //                             defaultoptionsfilename+tr(". Empty default options file."),
        //                             QMessageBox::Ok);
        return;
    };
    if (!defOptionsSettings.beginGroup("Main")){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             defaultoptionsfilename+tr(". Incorrect default options file."),
                             QMessageBox::Ok);
        return;
    }
        if (defOptionsSettings.value("ident").toString() != "FORCHECK default options file"){
                QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                     defaultoptionsfilename+tr(". Incorrect default options file."),
                                     QMessageBox::Ok);
                return;
        }
    defOptionsSettings.endGroup();
    defOptionsSettings.beginGroup("Default Options");
        myDefaultOptions->load(defOptionsSettings);
    defOptionsSettings.endGroup();
    defOptionsSettings.close();
}

void Environment::setPageOptions()
{
    if (pageSetup->charactersPerLine > 0)
        myFactoryOptions->setInt(optPW, pageSetup->charactersPerLine);
    if (pageSetup->linesPerPage > 0)
        myFactoryOptions->setInt(optPL, pageSetup->linesPerPage);
}

void Environment::setupPage()
{
    pageSetup->setupPage();
    pageSetup->getPageSize();
    setPageOptions();
}

void Environment::printFile(QFile& file)
{
    pageSetup->printFile(file);
}

void Environment::setEmulationFilename(const QString& filename)
{
    emulationfilename = filename;
}

void Environment::setExtension(FileKind::ETFileKind filekind, const QString &ext)
{
    switch (filekind) {
    case FileKind::Project:
        Extensions.Ext_project = ext;
        break;
    case FileKind::Make:
        Extensions.Ext_make = ext;
        break;
    case FileKind::Source:
        Extensions.Ext_source = ext;
        break;
    case FileKind::PreSource:
        Extensions.Ext_presource = ext;
        break;
    case FileKind::CSource:
        Extensions.Ext_csource = ext;
        break;
    case FileKind::Include:
        Extensions.Ext_include = ext;
        break;
    case FileKind::LibRef:
    case FileKind::LibOutput:
        Extensions.Ext_library = ext;
        break;
    case FileKind::Listing:
        Extensions.Ext_listing = ext;
        break;
    case FileKind::Report:
        Extensions.Ext_report = ext;
        break;
    case FileKind::ReferenceStructure:
        Extensions.Ext_refstruct = ext;
        break;
    case FileKind::ModDep:
        Extensions.Ext_moddep = ext;
        break;
    default:
        return;
    }
}

QString Environment::getExtension(FileKind::ETFileKind filekind) const
{
    switch (filekind) {
    case FileKind::Project:
        return Extensions.Ext_project;
    case FileKind::Make:
        return Extensions.Ext_make;
    case FileKind::Source:
        return Extensions.Ext_source;
    case FileKind::PreSource:
        return Extensions.Ext_presource;
    case FileKind::CSource:
        return Extensions.Ext_csource;
    case FileKind::Include:
        return Extensions.Ext_include;
    case FileKind::LibRef:
    case FileKind::LibOutput:
        return Extensions.Ext_library;
    case FileKind::Listing:
        return Extensions.Ext_listing;
    case FileKind::Report:
        return Extensions.Ext_report;
    case FileKind::ReferenceStructure:
        return Extensions.Ext_refstruct;
    case FileKind::ModDep:
        return Extensions.Ext_moddep;
    default:
        return QString();
    }
}

void Environment::setEditor(const QString& filename)
{
    editor = filename;
}

void Environment::setEditorCommandline(const QString& commandline)
{
    editorCommandline = commandline;
}

void Environment::setEditorAsViewer(const bool editorasviewer)
{
    editorAsViewer = editorasviewer;
}

void Environment::setCompiler(const QString& name)
{
    compiler = name;
}

void Environment::setCompilerOptions(const QString& options)
{
    compilerOptions = options;
}

void Environment::setCompilerDebugOptions(const QString& options)
{
    compilerDebugOptions = options;
}

void Environment::setcCompiler(const QString& name)
{
    cCompiler = name;
}

void Environment::setcCompilerOptions(const QString& options)
{
    cCompilerOptions = options;
}

void Environment::setLinkerOptions(const QString& options)
{
    linkerOptions = options;
}

void Environment::setBuildUtility(const QString& filename)
{
    buildUtility = filename;
}

void Environment::setLibrariesPreferences(const bool &update, const bool &compress)
{
    updateLibrary = update;
    compressLibrary = compress;
}

bool Environment::getForcheckPWDInfo(QString &userid,
                                     QString &licensecat,
                                     QString &serial,
                                     QDate &exp_license,
                                     QDate &exp_maintenance)
{
        userid.clear();
        licensecat.clear();
        serial = "?";
        exp_license = QDate();
        exp_maintenance = QDate();

        QString passwordFilename = fckDataDir + Constants::FckPwdFilename;
        QFile passwordFile(passwordFilename);
        if (!passwordFile.open(QIODevice::ReadOnly)) {
                QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                         passwordFilename+tr(". No password file found."),
                                         QMessageBox::Ok);
                return false;
        };
        QTextStream in(&passwordFile);
//read userid
        userid = in.readLine();
//read platform name
        QString tmp = in.readLine();
//        int ide = tmp.indexOf("(+IDE)");
//        if (ide==0) ide = tmp.indexOf("Multi");
//read serial number, number of users
        tmp = in.readLine();
        serial = tmp.section(" ",0,0);
     //read number of users
        QString nusers = tmp.section(" ",2,2);
        if (nusers.indexOf("-") > -1)
              licensecat = "site license";
//read license expiration date
        tmp = in.readLine();
        if (tmp.section(" ",1,1)=="00" ) tmp.replace(6,1,"1");
        if (tmp.section(" ",2,2)=="00") tmp.replace(9,1,"1");
        exp_license = QDate::fromString(tmp,"yyyy MM dd");
//read maintenance expiration date
        tmp = in.readLine();
        exp_maintenance = QDate::fromString(tmp,"yyyy MM dd");
        if (exp_license < QDate::currentDate()){
                QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                     tr("Your Forcheck license has expired."),
                                     QMessageBox::Ok);
                return false;
        }
        return true;
}

Version Environment::getFCKversion() {
    QFile reportFile(defaultreportfilename);
    reportFile.close();
    reportFile.remove();
    Version fckVersion = Version();
    QStringList args(PREopt + optRP + DELopt +
                     QDir::toNativeSeparators(defaultreportfilename) +
                     SEPopt);
    int result = execForcheck(args);
    if (result == -1){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             fckExecutable+tr(". Could not start the analyser."),
                             QMessageBox::Ok);
        return fckVersion;
    };
    if (!process->waitForFinished(5000)){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             fckExecutable+tr(". The analyser did not complete."),
                             QMessageBox::Ok);
        return fckVersion;
    };
    if (!reportFile.exists()){
//try on temp directory for compatibility with Forcheck <= V14.1.04
        reportFile.setFileName(QDir::tempPath() + "/fck.rep");
        if (!reportFile.exists()){
            QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                 reportFile.fileName()+tr(". Report file not found."),
                                 QMessageBox::Ok);
            return fckVersion;
        };
    };
    if (!reportFile.open(QIODevice::ReadOnly)) {
            QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                                 reportFile.fileName()+tr(". Open error on report file."),
                                 QMessageBox::Ok);
        return fckVersion;
    };
    QTextStream in(&reportFile);
    QString tmp = in.readLine();
    int p = tmp.indexOf("V");
    if (p > -1) {
       tmp.remove(0,p+1);
       fckVersion = Version(tmp);
    }
    reportFile.remove();
    return fckVersion;
}

int Environment::execForcheck(const QStringList& args){
    QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
    QString defopt = PREopt;
    defopt += optBA;
    defopt += myDefaultOptions->toCommandString();
    if (myDefaultOptions->getInt(optPW) == 0){
        defopt += myFactoryOptions->toCommandString(optPW);
    }
    if (myDefaultOptions->getInt(optPL) == 0){
        defopt += myFactoryOptions->toCommandString(optPL);
    }
    env.insert("FCKOPT", defopt);
    env.insert("INCLUDE", IPnames->join(";"));
    process->setProcessEnvironment(env);
    process->setProcessChannelMode(QProcess::MergedChannels);
    process->start(fckExecutable, args);
    if (!process->waitForStarted(2000))
        return -1;
    return 0;
};

int Environment::execMake(const QString& makefilename){
    QStringList args;
    QString command = buildUtility;
    if (command.isEmpty())
        command = QDir::toNativeSeparators(makefilename);
    else
        args.append("-f" + QDir::toNativeSeparators(makefilename));
//    QDir::setCurrent(FileKind::fileDir(makefilename));
    QDir::setCurrent(workingDir);
    process->setProcessChannelMode(QProcess::MergedChannels);
    process->start(command, args);
    if (!process->waitForStarted(2000))
        return -1;
    return 0;
};

int Environment::execInterfaceBuilder(const QString& libname,
                                      const QString& outputfilename,
                                      QString& report)
{
#if defined(Q_OS_WIN)
    QStringList args(QDir::toNativeSeparators(libname) +
                     PREopt + optBA + PREopt + optLI1 + DELopt +
                     QDir::toNativeSeparators(outputfilename));
#elif defined(Q_OS_LINUX)
    QStringList args(PREopt + optBA + DELopt +
                     QDir::toNativeSeparators(libname) +
                     NXTopt +
                     QDir::toNativeSeparators(outputfilename));
#endif
    process->start(interfExecutable, args);
    if (!process->waitForStarted(2000)){
        QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                 tr("Could not start the interface builder"),
                                 QMessageBox::Ok);
        return -1;
    }
    if (!process->waitForFinished()){
        QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                 tr("Error executing the interface builder"),
                                 QMessageBox::Ok);
        return -1;
    }
    int status = process->exitStatus();
    if (status != QProcess::NormalExit)
        QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                 tr("Error executing the interface builder"),
                                 QMessageBox::Ok);
    report = QString(process->readAll());
    return status;
}

void Environment::execEditor(const QString& filename, int linnum)
{
    if (filename.size() == 0) {
        QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                 tr("No file selected"),
                                 QMessageBox::Ok);
            return;
    }
    QFile file(filename);
    if (!file.exists()) {
        QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                 filename + tr("\nFile does not exists"),
                                 QMessageBox::Ok);
        return;
    }
    if (linnum < 1)
        linnum = 1;

    QString line;
    QStringList args;
    char c;
    if (editorCommandline.size() > 0){
        for (int i=0; i < editorCommandline.size(); ++i) {
//            c = editorCommandline[i].toAscii();
            c = editorCommandline[i].unicode();
            if ((i < editorCommandline.size()-1) && (c == '%')) {
//                c = editorCommandline[++i].toAscii();
                c = editorCommandline[++i].unicode();
                switch (c) {
                case 'f':
                    line += filename;
                    break;
                case 'n':
                    line += QString::number(linnum);
                    break;
                case '%':
                    line += "%";
                    break;
                }
            }
            else if (c == ' ') {
                args << line;
                line.clear();
            }
            else
                line += c;
        }
        if (line.size() > 0) args << line;
    }
    else
        args << filename;

    if (!QProcess::startDetached(editor, args))
        QMessageBox::information(QApplication::activeWindow(), Info::TITLE,
                                 tr("Error executing the editor"),
                                 QMessageBox::Ok);
    return;
}

QString Environment::getTempDirDefault()
{
    return QDir::toNativeSeparators(QDir::tempPath())+QDir::separator();
}

void Environment::wwwBrowser(const QString& surl)
{
    QDesktopServices desktop;
    QUrl url(surl);
    desktop.openUrl(url);
}

void Environment::sendEmail(const QString& emailAddress)
{
    QDesktopServices desktop;
    QUrl url(emailAddress);
    desktop.openUrl(url);
}

void Environment::getEmulation(const QString& filename, QString& emulation, bool& initfmt)
{
    emulation = QString();
    initfmt = false;
    QString line;
    QFile file(filename);
    if (!file.open(QIODevice::ReadOnly))
        return;
    QTextStream in(&file);
    line = in.readLine();
    if (line.left(1) == "["){
        line = in.readLine();
        initfmt = true;
    }
    if (line.left(2) != "'V")
        return;
    line = in.readLine();
    file.close();
    emulation = line.section("'", 5, 5);
}
