/*ForcheckIDE/CodeEditor.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/
#include "CodeEditor.h"
#include "Info.h"

#include <QtWidgets>
#include <QFile>
#include <QTextStream>

CodeEditor::CodeEditor(QMainWindow *parent) : QPlainTextEdit(parent)
{
    myparent = parent;
    action = new QAction(this);
    action->setCheckable(true);
    connect(action, SIGNAL(triggered()), this, SLOT(show()));
    connect(action, SIGNAL(triggered()), this, SLOT(setFocus()));

    connect(document(), SIGNAL(contentsChanged()),
            this, SLOT(documentWasModified()));

    setLineWrapMode(QPlainTextEdit::NoWrap);
    setCurrentFileName("");
    isUntitled = true;
    lineNumberArea = new LineNumberArea(this);

    connect(this, SIGNAL(blockCountChanged(int)), this, SLOT(updateLineNumberAreaWidth(int)));
    connect(this, SIGNAL(updateRequest(const QRect &, int)), this, SLOT(updateLineNumberArea(const QRect &, int)));
    connect(this, SIGNAL(cursorPositionChanged()), this, SLOT(highlightCurrentLine()));

    QFont font = QFont("Courier New");
    font.setPixelSize(14);
    setFont(font);
    QFontMetrics metrics(font);
    setTabStopWidth(metrics.width("a")*8);

    updateLineNumberAreaWidth(0);
    setWindowTitle("[*]");
    setAttribute(Qt::WA_DeleteOnClose);
}

QSize CodeEditor::sizeHint() const
{
    return QSize(80 * fontMetrics().width('x') + lineNumberArea->sizeHint().width(),
                 80 * fontMetrics().lineSpacing());
}

void CodeEditor::closeEvent(QCloseEvent *event)
{
    if (isOkToContinue()){
        emit(closed());
        event->accept();
    } else {
        event->ignore();
    }
}

void CodeEditor::focusInEvent(QFocusEvent * event)
{
    emit(InFocus());
    event->accept();
}

void CodeEditor::focusOutEvent(QFocusEvent * event)
{
    emit(OutFocus());
    event->accept();
}

bool CodeEditor::isOkToContinue()
{
    if (document()->isModified()){
        QMessageBox::StandardButton ret;
        ret = QMessageBox::question(this, Info::TITLE,
                                    curFileName +
                                    tr("\nThe file has been modified.\n"
                                       "Save changes?"),
                                    QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
        if (ret == QMessageBox::Save)
            return saveFile();
        else if (ret == QMessageBox::Cancel)
            return false;
    }
    return true;
}

bool CodeEditor::loadFile(const QString& fileName)
{
    QFile file(fileName);
    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        QMessageBox::warning(this, Info::TITLE,
                             tr("Cannot open file %1:\n%2.")
                             .arg(fileName)
                             .arg(file.errorString()));
        return false;
    }
    myparent->statusBar()->showMessage(tr("Reading file"));
    QApplication::setOverrideCursor(Qt::WaitCursor);
    setWindowTitle(file.fileName());

    QTextStream in(&file);
    setPlainText(in.readAll());

    file.close();
    setCurrentFileName(fileName);
    QApplication::restoreOverrideCursor();
    myparent->statusBar()->showMessage(QString());
    return true;
}

bool CodeEditor::closeFile()
{
    if (isOkToContinue())
        return close();
    return false;
}

bool CodeEditor::saveFile()
{
    QString filename = curFileName;
    if (isUntitled){
        filename = QFileDialog::getSaveFileName(this);
        if (filename.isEmpty())
            return false;
    }
    QFile file(filename);
    if (!file.open(QFile::WriteOnly | QFile::Text)) {
        QMessageBox::warning(this, Info::TITLE,
                             tr("Cannot write file %1:\n%2.")
                             .arg(curFileName)
                             .arg(file.errorString()));
        return false;
    }

    myparent->statusBar()->showMessage(tr("Saving file"));
    QTextStream out(&file);
    QApplication::setOverrideCursor(Qt::WaitCursor);
    out << toPlainText();
    QApplication::restoreOverrideCursor();

    setCurrentFileName(filename);
    emit fileSaved(filename);
    myparent->statusBar()->showMessage(QString());
    return true;
}

QString CodeEditor::newFileName()
{
    static int documentNumber = 1;

    curFileName = tr("unnamed%1.txt").arg(documentNumber);
    setWindowTitle(curFileName + "[*]");
    action->setText(curFileName);
    ++documentNumber;
    return curFileName;
}

void CodeEditor::setCurrentFileName(const QString &fileName)
{
    action -> setText(fileName);
    curFileName = fileName;
    isUntitled = false;
    document()->setModified(false);
    setWindowTitle(curFileName);
    setWindowModified(false);
}

void CodeEditor::gotoLine(int linnum)
{
    if (linnum > 0){
        QTextBlock block = document()->findBlockByNumber(linnum-1);
        QTextCursor cursor(block);
        setTextCursor(cursor);
        ensureCursorVisible();
        highlightCurrentLine();
    }
}

void CodeEditor::documentWasModified()
{
    setWindowTitle(curFileName + "[*]");
    setWindowModified(true);
}

int CodeEditor::lineNumberAreaWidth()
{
    int digits = 1;
    int max = qMax(1, blockCount());
    while (max >= 10) {
        max /= 10;
        ++digits;
    }

    int space = 3 + fontMetrics().width(QLatin1Char('9')) * digits;
    return space;
}

void CodeEditor::updateLineNumberAreaWidth(int /* newBlockCount */)
{
    setViewportMargins(lineNumberAreaWidth(), 0, 0, 0);
}

void CodeEditor::updateLineNumberArea(const QRect &rect, int dy)
{
    if (dy)
        lineNumberArea->scroll(0, dy);
    else
        lineNumberArea->update(0, rect.y(), lineNumberArea->width(), rect.height());

    if (rect.contains(viewport()->rect()))
        updateLineNumberAreaWidth(0);
}

void CodeEditor::resizeEvent(QResizeEvent *e)
{
    QPlainTextEdit::resizeEvent(e);

    QRect cr = contentsRect();
    lineNumberArea->setGeometry(QRect(cr.left(), cr.top(), lineNumberAreaWidth(), cr.height()));
}

void CodeEditor::highlightCurrentLine()
{
    QList<QTextEdit::ExtraSelection> extraSelections;

    if (!isReadOnly()) {
        QTextEdit::ExtraSelection selection;

        QColor lineColor = QColor(Qt::yellow).lighter(160);

        selection.format.setBackground(lineColor);
        selection.format.setProperty(QTextFormat::FullWidthSelection, true);
        selection.cursor = textCursor();
        selection.cursor.clearSelection();
        extraSelections.append(selection);
    }

    setExtraSelections(extraSelections);
}

void CodeEditor::lineNumberAreaPressEvent(QMouseEvent *event)
{
    QTextBlock block = firstVisibleBlock();
    int blocknumber = block.blockNumber();
    int top = (int) blockBoundingGeometry(block).translated(contentOffset()).top();
    blocknumber = (event->y() - top) / (int) blockBoundingRect(block).height() +
                  blocknumber;
    block = document()->findBlockByNumber(blocknumber);
    QTextCursor cursor(block);
    cursor.select(QTextCursor::LineUnderCursor);
    cursor.movePosition(QTextCursor::NextBlock, QTextCursor::KeepAnchor);
    setTextCursor(cursor);
    ensureCursorVisible();
}

void CodeEditor::lineNumberAreaPaintEvent(QPaintEvent *event)
{
    QPainter painter(lineNumberArea);
    painter.fillRect(event->rect(), Qt::lightGray);

    QTextBlock block = firstVisibleBlock();
    int blockNumber = block.blockNumber();
    int top = (int) blockBoundingGeometry(block).translated(contentOffset()).top();
    int bottom = top + (int) blockBoundingRect(block).height();

    while (block.isValid() && top <= event->rect().bottom()) {
        if (block.isVisible() && bottom >= event->rect().top()) {
            QString number = QString::number(blockNumber + 1);
            painter.setPen(Qt::black);
            painter.drawText(0, top, lineNumberArea->width(), fontMetrics().height(),
                             Qt::AlignRight, number);
        }

        block = block.next();
        top = bottom;
        bottom = top + (int) blockBoundingRect(block).height();
        ++blockNumber;
    }
}
