/*ForcheckIDE/CnfOptions.cpp*/

/****************************************************************************

    Copyright 2016 Erik Kruyt, Forcheck b.v.

    This file is part of forcheckIDE.

    forcheckIDE is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    forcheckIDE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with forcheckIDE.  If not, see <http://www.gnu.org/licenses/>.

****************************************************************************/

#include "CnfOptions.h"
#include "Info.h"
#include "Environment.h"

#include <QtWidgets>
#include <QFile>
#include <QTextStream>
#include <QIODevice>
#include <QFileInfo>

static int count_mode_defaults[NR_COUNT_MODE_OPTIONS]{
        true,
        false,
        false,
        false,
        true,
        false
};

static int output_defaults[NR_OUTPUT_OPTIONS]{
        true,
        false,
        false,
        true,
        true,
        false,
        true,
        true,
        true
};

static QString count_mode_names[NR_COUNT_MODE_OPTIONS] = {
        "line",
        "statement",
        "new_in_sub",
        "new_in_file",
        "new_in_include",
        "continue_in_include"
};

static QString output_names[NR_OUTPUT_OPTIONS] = {
        "stdout_msgsum",
        "stdout_metrics",
        "stdout_usage",
        "listing_msgsum",
        "listing_metrics",
        "listing_usage",
        "report_msgsum",
        "report_metrics",
        "report_usage"
};

const char CnfOptions::levelFlags[] = "dEWIO ";

CnfOptions::CnfOptions(const QString &fckcnffilename, const QString &msgfilename)
{
    readMsgFile(msgfilename);
    if (numMsg > 0) {
        newLevel = new int[numMsg];
    }
    init();
    cnffile = new QFile(fckcnffilename);
    readFromFile();
}

void CnfOptions::init()
{
    changed = false;
    suppressAll = false;
    maxMsg = "6";
    printSourceStm = true;
    printLinStmNum = true;
    fileLineFmt = "('(file: ',a,', line: ',i0,')')";

    for (int i = 0; i < numMsg; ++i)
        newLevel[i] = 0;

    for (int i = 0; i < NR_COUNT_MODE_OPTIONS; ++i)
        count_mode_options[i] = count_mode_defaults[i];

//    QLocale locale;
//    fckDateFormat = locale.dateFormat(QLocale::ShortFormat);
    fckDateFormat = "yyyy-mm-dd";
    fckTimeFormat = "h:mm:ss";

    for (int i = 0; i < NR_OUTPUT_OPTIONS; ++i)
        output_options[i] = output_defaults[i];
}

CnfOptions::~CnfOptions()
{
    if (numMsg > 0){
        delete[] msgs;
        delete[] newLevel;
    }
    delete cnffile;
}

void CnfOptions::readFromFile()
{
    if (!cnffile->exists()) return;
    if (!cnffile->open(QIODevice::ReadOnly|QIODevice::Text)){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not read preferences from file"),
                             QMessageBox::Ok);
        return;
    }

    char rec[1024];
    char del;
    int c;
    do {
        c = cnffile->readLine(rec, sizeof(rec)-1);
        if (c <= 0) break;
        char *p;
        p = strchr(rec, '\n');
        if (p) *p = 0;

        p = strchr(rec, '=');
        if (p) {
            *p = 0;
            char *keyword = rec;
            char *value = p+1;
            del = *value;
            if (del == '\'' || del == '\"') {
                ++value;
                p = strchr(value,del);
                if (p) *p = 0;
            }
            if (strlen(value) < 1)
                continue;

            if (strcasecmp(keyword, "suppress") == 0) {
                suppressAll = (QString::compare(value, "all") == 0);
            }
            else if (strcasecmp(keyword, "max_msg") == 0) {
                maxMsg = value;
            }
            else if (strcasecmp(keyword, "source_stm") == 0) {
                printSourceStm = (QString::compare(value, "yes") == 0);
            }
            else if (strcasecmp(keyword, "source_linstm_number") == 0) {
                printLinStmNum = (QString::compare(value, "yes") == 0);
            }
            else if (strcasecmp(keyword, "file_line_format") == 0) {
                fileLineFmt = value;
            }
            else if (strcasecmp(keyword, "count_mode") == 0) {
                for (int i = 0; i < NR_COUNT_MODE_OPTIONS; i++)
                    if (QString::compare(value, count_mode_names[i]) == 0) {
                        count_mode_options[i] = true;
                        break;
                    }
            }
            else if (strcasecmp(keyword, "date_format") == 0) {
                fckDateFormat = value;
            }
            else if (strcasecmp(keyword, "time_format") == 0) {
                fckTimeFormat = value;
            }
            else
                for (int i = 0; i <NR_OUTPUT_OPTIONS; i++)
                    if (QString::compare(keyword, output_names[i]) == 0) {
                        output_options[i] = strcasecmp(value, "'true'") == 0;
                        break;
                    }
        }
        else {
            p = strchr(rec, ' ');
            if (p && p[1] == '\'') {
                *p = 0;
                int num = QString(rec).toInt();
                if (num < 1 || num > numMsg)
                    continue;
                const char *pLevelFlag = strchr(levelFlags,p[2]);
                if (!pLevelFlag)
                    continue;
                int level = pLevelFlag - CnfOptions::levelFlags;
                newLevel[num-1] = level;
            }
        }
    }
    while (true);
    cnffile->close();
}

void CnfOptions::writeToFile()
{
    for (int i = 0; i < numMsg; ++i)
        if (newLevel[i]) changed = true;
    if (!changed) return;

    if (!cnffile->open(QIODevice::WriteOnly|QIODevice::Text)){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not write preferences to file"),
                             QMessageBox::Ok);
        return;
    }
    QString emulation;
    bool initfmt;
    myEnvironment->getEmulation(myEnvironment->getEmulationFilename(), emulation, initfmt);

    changed = false;
    QTextStream os(cnffile);

    if (initfmt)
        os << "[MESSAGES]" << endl;
    if (suppressAll){
        os << "suppress='all'" << endl;
    }
    for (int i = 0; i < numMsg; ++i) {
        if (newLevel[i])
            os << (i+1) << " '" << CnfOptions::levelFlags[newLevel[i]] << "'" << endl;
    }
    if (initfmt)
        os << "[VARIOUS]" << endl;
    else
        os << "0 'delimiter of redefined messages'" << endl;

    if (maxMsg != "6" && maxMsg != "") {
        os << "max_msg="
           << maxMsg
           << endl;
    }
    if (printSourceStm) {
      os << "source_stm='yes'"
         << endl;
    }
    else {
        os << "source_stm='no'"
           << endl;
    }
    if (printLinStmNum){
        os << "source_linstm_number='yes'"
           << endl;
    }
    else {
        os << "source_linstm_number='no'"
           << endl;
    }

    os << "file_line_format=\""
       << fileLineFmt.toUtf8()
       << "\""
       << endl;

    for (int i = 0; i < NR_COUNT_MODE_OPTIONS; ++i) {
        if (count_mode_options[i]) {
                os << "count_mode='"
                   << count_mode_names[i]
                   << "'"
                   << endl;
        }
    }
    os << "date_format='"
//       << fckDateFormat.toAscii()
       << fckDateFormat.toUtf8()
       << "'"
       << endl;

    os << "time_format='"
//       << fckTimeFormat.toAscii()
       << fckTimeFormat.toUtf8()
       << "'"
       << endl;

    if (initfmt)
        os << "[OUTPUT]" << endl;
    else
        os << "0 'delimiter of various section'" << endl;

    for (int i = 0; i < NR_OUTPUT_OPTIONS; ++i) {
        if (output_options[i] != output_defaults[i]) {
            if (output_options[i]) {
                os << output_names[i]
                   << "='true'"
                   << endl;
            }
            else {
                 os << output_names[i]
                    << "='false'"
                    << endl;
            }
        }
    }

    cnffile->close();
}

void CnfOptions::readMsgFile(const QString &msgfilename)
{
    numMsg = 0;
    QFile *msgfile = new QFile(msgfilename);
    if (!msgfile->open(QIODevice::ReadOnly)){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Could not open messages file"),
                             QMessageBox::Ok);
        return;
    }
    QFileInfo info(msgfilename);
    numMsg = info.size() / sizeof(MSG_STRUCT);
    int numb = numMsg * sizeof(MSG_STRUCT);
    msgs = new MSG_STRUCT[numMsg];
    if (msgfile->read((char*)msgs, numb) != numb){
        QMessageBox::warning(QApplication::activeWindow(), Info::TITLE,
                             tr("Read error on messages file"),
                             QMessageBox::Ok);
        numMsg = 0;
    }
    msgfile->close();
}
