/* gallagher_util.c - Utilities for parsing Gallagher cards (New Zealand).
 * Author: Nick Mooney (nick@mooney.nz)
 * 
 * Reference: https://github.com/megabug/gallagher-research
*/

#include "gallagher_util.h"

#define GALLAGHER_CREDENTIAL_SECTOR 15

/* The Gallagher obfuscation algorithm is a 256-byte substitution table. The below array is generated from
 * https://github.com/megabug/gallagher-research/blob/master/formats/cardholder/substitution-table.bin.
*/
const uint8_t GALLAGHER_DECODE_TABLE[256] = {
    0x2f, 0x6e, 0xdd, 0xdf, 0x1d, 0xf,  0xb0, 0x76, 0xad, 0xaf, 0x7f, 0xbb, 0x77, 0x85, 0x11,
    0x6d, 0xf4, 0xd2, 0x84, 0x42, 0xeb, 0xf7, 0x34, 0x55, 0x4a, 0x3a, 0x10, 0x71, 0xe7, 0xa1,
    0x62, 0x1a, 0x3e, 0x4c, 0x14, 0xd3, 0x5e, 0xb2, 0x7d, 0x56, 0xbc, 0x27, 0x82, 0x60, 0xe3,
    0xae, 0x1f, 0x9b, 0xaa, 0x2b, 0x95, 0x49, 0x73, 0xe1, 0x92, 0x79, 0x91, 0x38, 0x6c, 0x19,
    0xe,  0xa9, 0xe2, 0x8d, 0x66, 0xc7, 0x5a, 0xf5, 0x1c, 0x80, 0x99, 0xbe, 0x4e, 0x41, 0xf0,
    0xe8, 0xa6, 0x20, 0xab, 0x87, 0xc8, 0x1e, 0xa0, 0x59, 0x7b, 0xc,  0xc3, 0x3c, 0x61, 0xcc,
    0x40, 0x9e, 0x6,  0x52, 0x1b, 0x32, 0x8c, 0x12, 0x93, 0xbf, 0xef, 0x3b, 0x25, 0xd,  0xc2,
    0x88, 0xd1, 0xe0, 0x7,  0x2d, 0x70, 0xc6, 0x29, 0x6a, 0x4d, 0x47, 0x26, 0xa3, 0xe4, 0x8b,
    0xf6, 0x97, 0x2c, 0x5d, 0x3d, 0xd7, 0x96, 0x28, 0x2,  0x8,  0x30, 0xa7, 0x22, 0xc9, 0x65,
    0xf8, 0xb7, 0xb4, 0x8a, 0xca, 0xb9, 0xf2, 0xd0, 0x17, 0xff, 0x46, 0xfb, 0x9a, 0xba, 0x8f,
    0xb6, 0x69, 0x68, 0x8e, 0x21, 0x6f, 0xc4, 0xcb, 0xb3, 0xce, 0x51, 0xd4, 0x81, 0x0,  0x2e,
    0x9c, 0x74, 0x63, 0x45, 0xd9, 0x16, 0x35, 0x5f, 0xed, 0x78, 0x9f, 0x1,  0x48, 0x4,  0xc1,
    0x33, 0xd6, 0x4f, 0x94, 0xde, 0x31, 0x9d, 0xa,  0xac, 0x18, 0x4b, 0xcd, 0x98, 0xb8, 0x37,
    0xa2, 0x83, 0xec, 0x3,  0xd8, 0xda, 0xe5, 0x7a, 0x6b, 0x53, 0xd5, 0x15, 0xa4, 0x43, 0xe9,
    0x90, 0x67, 0x58, 0xc0, 0xa5, 0xfa, 0x2a, 0xb1, 0x75, 0x50, 0x39, 0x5c, 0xe6, 0xdc, 0x89,
    0xfc, 0xcf, 0xfe, 0xf9, 0x57, 0x54, 0x64, 0xa8, 0xee, 0x23, 0xb,  0xf1, 0xea, 0xfd, 0xdb,
    0xbd, 0x9,  0xb5, 0x5b, 0x5,  0x86, 0x13, 0xf3, 0x24, 0xc5, 0x3f, 0x44, 0x72, 0x7c, 0x7e,
    0x36};

// The second block of a Gallagher credential sector is the literal
// "www.cardax.com  " (note two padding spaces)
const uint8_t GALLAGHER_CARDAX_ASCII[MF_CLASSIC_BLOCK_SIZE] =
    {'w', 'w', 'w', '.', 'c', 'a', 'r', 'd', 'a', 'x', '.', 'c', 'o', 'm', ' ', ' '};

/* Precondition: cardholder_data_obfuscated points to at least 8 safe-to-read bytes of memory.
*/
void gallagher_deobfuscate_and_parse_credential(
    GallagherCredential* credential,
    const uint8_t* cardholder_data_obfuscated) {
    uint8_t cardholder_data_deobfuscated[8];
    for(int i = 0; i < 8; i++) {
        cardholder_data_deobfuscated[i] = GALLAGHER_DECODE_TABLE[cardholder_data_obfuscated[i]];
    }

    // Pull out values from the deobfuscated data
    credential->region = (cardholder_data_deobfuscated[3] >> 1) & 0x0F;
    credential->facility = ((uint16_t)(cardholder_data_deobfuscated[5] & 0x0F) << 12) +
                           ((uint16_t)cardholder_data_deobfuscated[1] << 4) +
                           (((uint16_t)cardholder_data_deobfuscated[7] >> 4) & 0x0F);
    credential->card = ((uint32_t)cardholder_data_deobfuscated[0] << 16) +
                       ((uint32_t)(cardholder_data_deobfuscated[4] & 0x1F) << 11) +
                       ((uint32_t)cardholder_data_deobfuscated[2] << 3) +
                       (((uint32_t)cardholder_data_deobfuscated[3] >> 5) & 0x07);
    credential->issue = cardholder_data_deobfuscated[7] & 0x0F;
}