import yaml
import requests


from datetime import datetime


from pathlib import Path


def _get_metadata(url):
    data = {}
    if url.startswith("https://huggingface.co/"):
        # examples:
        #  - datasets/bigcode/the-stack
        #  - models/Phind/Phind-CodeLlama-34B-v2
        #  - WizardLM/WizardCoder-Python-34B-V1.0 (assumed to be a model)
        repo_id = url[23:]
        if not any([repo_id.startswith("datasets/"), repo_id.startswith("models/")]):
            repo_id = f"models/{repo_id}"
        response = requests.get(f"https://huggingface.co/api/{repo_id}")
        response.raise_for_status()
        json_data = response.json()
        data["stars"] = json_data["likes"]
    elif url.startswith("https://github.com/"):
        github_api_url = "https://api.github.com/repos"
        repo_id = url[19:]
        response = requests.get(f"{github_api_url}/{repo_id}")
        response.raise_for_status()
        json_data = response.json()
        data["stars"] = json_data["stargazers_count"]
        data["created"] = datetime.strptime(
            json_data["created_at"], "%Y-%m-%dT%H:%M:%SZ"
        ).isoformat()
    else:
        raise ValueError(f"Unknown repo url: {url}")

    data["updated"] = datetime.now().isoformat()
    return data


def save_metadata(file_path: str, dest: str) -> None:
    with open(file_path, "r") as f:
        data = yaml.load(f, Loader=yaml.SafeLoader)

    metamap = {}
    if Path(dest).exists():
        with open(dest, "r") as f:
            metamap = yaml.load(f, Loader=yaml.SafeLoader)

    def _update_metadata(link):
        try:
            print(f"Attempting to get stars for {link}")
            metamap[link] = _get_metadata(link)
            print(f"  Succeeded: {metamap[link]}")
        except Exception as e:
            print(f"  Error: {e}")

    always_update = False
    max_days = 1

    # Update the existing entries, if it's been more than max_days
    for link in metamap:
        d = datetime.fromisoformat(metamap[link]["updated"])
        if always_update or (datetime.now() - d).days > max_days:
            _update_metadata(link)

    # If any new
    for category in data:
        for item in data[category]:
            if item["link"] not in metamap:
                print("New entry: ", item["link"])
                _update_metadata(item["link"])

    with open(dest, "w") as f:
        # add comment to the top of the file, so that it's clear it's autogenerated
        f.write("# Autogenerated by update_stars.py\n")
        yaml.dump(metamap, f, default_flow_style=False)


if __name__ == "__main__":
    save_metadata("data/data.yaml", "data/metadata.yaml")
