-------------------------------- MODULE raft --------------------------------

\* Raft consensus protocol specification based on the TLA+ spec from
\* https://github.com/ongardie/raft.tla.
\*
\* We used limited buffer sized network channels to improve model checking 
\* perofmance. Also leader election part of Raft is not live by design.
\* These two can cause deadlock, so don't check deadlocks in model checking.
\*
\* Properties are defined based on the presented properties in the original 
\* Raft paper. See figure 3 in https://raft.github.io/raft.pdf.

EXTENDS Naturals, Sequences, TLC, FiniteSets

CONSTANT ExploreFail
CONSTANT Debug

CONSTANT NumServers

CONSTANT BufferSize

CONSTANT MaxTerm
CONSTANT MaxCommitIndex

CONSTANT MaxNodeFail

CONSTANT LogConcat
CONSTANT LogPop

CONSTANT LeaderTimeoutReset

Nil == 0

RECURSIVE MinAcc(_, _), MaxAcc(_, _)

Min(s) ==
    LET e == CHOOSE e \in s : TRUE
    IN MinAcc(s \ { e }, e)

MinAcc(s, e1) ==
    IF s = {} THEN e1
    ELSE
        LET e2 == CHOOSE e2 \in s : TRUE
        IN MinAcc(s \ { e2 }, IF e2 < e1 THEN e2 ELSE e1)

Max(s) ==
    LET e == CHOOSE e \in s : TRUE
    IN MaxAcc(s \ { e }, e)

MaxAcc(s, e1) ==
    IF s = {} THEN e1
    ELSE
        LET e2 == CHOOSE e2 \in s : TRUE
        IN MaxAcc(s \ { e2 }, IF e2 > e1 THEN e2 ELSE e1)

RECURSIVE FindMaxAgreeIndexRec(_, _, _, _)

isQuorum(s) == Cardinality(s) * 2 > NumServers
ServerSet   == 1..NumServers

FindMaxAgreeIndex(logLocal, i, matchIndex) ==
    FindMaxAgreeIndexRec(logLocal, i, matchIndex, Len(logLocal))

FindMaxAgreeIndexRec(logLocal, i, matchIndex, index) ==
    IF index = 0 THEN Nil
    ELSE
        IF isQuorum({i} \cup {k \in ServerSet : matchIndex[k] >= index})
        THEN index
        ELSE FindMaxAgreeIndexRec(logLocal, i, matchIndex, index - 1)

(********************

--mpcal raft {
    define {
        Follower  == "follower"
        Candidate == "candidate"
        Leader    == "leader"

        RequestVoteRequest    == "rvq"
        RequestVoteResponse   == "rvp"
        AppendEntriesRequest  == "apq"
        AppendEntriesResponse == "app"

        ProposeMessage == "prm"
        AcceptMessage  == "acm"

        Key1   == "key1"
        Key2   == "key2"
        Value1 == "value1"

        LastTerm(xlog) == IF Len(xlog) = 0 THEN 0 ELSE xlog[Len(xlog)].term

        ServerNetListenerSet                == (0*NumServers+1)..(1*NumServers)
        ServerPropChListenerSet             == (1*NumServers+1)..(2*NumServers) 
        ServerRequestVoteSet                == (2*NumServers+1)..(3*NumServers)
        ServerAppendEntriesSet              == (3*NumServers+1)..(4*NumServers)
        ServerAdvanceCommitIndexSet         == (4*NumServers+1)..(5*NumServers)
        ServerBecomeLeaderSet               == (5*NumServers+1)..(6*NumServers)
        ServerFollowerAdvanceCommitIndexSet == (6*NumServers+1)..(7*NumServers)

        ServerCrasherSet == IF ExploreFail 
                            THEN (7*NumServers+1)..(7*NumServers+MaxNodeFail)
                            ELSE {}

        NodeSet   == ServerSet

        KeySet == {}
    }

    macro checkFail(selfId, netEnabled) {
        if (ExploreFail) {
            if (\lnot netEnabled[selfId]) {
                await FALSE;
            };
        };
    }

    macro debug(toPrint) {
        if (Debug) {
            print toPrint;
        };
    }

    macro Send(net, dest, fd, m) {
        either {
            net[dest] := m;
        } or {
            await fd[dest];
        };
    }

    macro UpdateTerm(i, m, currentTerm, state, votedFor, leader) {
        if (m.mterm > currentTerm[i]) {
            currentTerm[i] := m.mterm;
            state[i]       := Follower;
            votedFor[i]    := Nil;
            leader[i]      := Nil;
        };
    }

    mapping macro ReliableFIFOLink {
        read {
            assert $variable.enabled;
            await Len($variable.queue) > 0;
            with (readMsg = Head($variable.queue)) {
                $variable := [queue |-> Tail($variable.queue), enabled |-> $variable.enabled];
                yield readMsg;
            };
        }

        write {
            await $variable.enabled;
            await Len($variable.queue) < BufferSize;
            yield [queue |-> Append($variable.queue, $value), enabled |-> $variable.enabled];
        }
    }

    mapping macro NetworkToggle {
        read { yield $variable.enabled; }

        write {
            yield [queue |-> $variable.queue, enabled |-> $value];
        }
    }

    mapping macro PerfectFD {
        read {
            yield $variable;
        }

        write { yield $value; }
    }

    mapping macro NetworkBufferLength {
        read {
            yield Len($variable.queue);
        }

        write {
            assert FALSE;
            yield $value;
        }
    }

    mapping macro Channel {
        read {
            await Len($variable) > 0;
            with (res = Head($variable)) {
                $variable := Tail($variable);
                yield res;
            };
        }

        write {
            await Len($variable) < BufferSize;
            yield Append($variable, $value);
        }
    }

    mapping macro PersistentLog {
        read {
            yield $variable;
        }

        write {
            if ($value.cmd = LogConcat) {
                yield $variable \o $value.entries;
            } else if ($value.cmd = LogPop) {
                yield SubSeq($variable, 1, Len($variable) - $value.cnt);
            };
        }
    }

    archetype AServerNetListener(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
    variables
        idx = 1,
        m;
    {
    serverLoop:
        while (TRUE) {
            checkFail(srvId, netEnabled);

            m := net[self];
            assert m.mdest = self;

        handleMsg:
            checkFail(srvId, netEnabled);

            if (m.mtype = RequestVoteRequest) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* HandleRequestVoteRequest
                with (
                    i = self, j = m.msource,
                    logOK = \/ m.mlastLogTerm > LastTerm(log[i])
                            \/ /\ m.mlastLogTerm = LastTerm(log[i])
                               /\ m.mlastLogIndex >= Len(log[i]),
                    grant = /\ m.mterm = currentTerm[i]
                            /\ logOK
                            /\ votedFor[self] \in {Nil, j}
                ) {
                    assert m.mterm <= currentTerm[i];
                    if (grant) {
                        votedFor[i] := j;
                    };
                    Send(net, j, fd, [
                        mtype        |-> RequestVoteResponse,
                        mterm        |-> currentTerm[i],
                        mvoteGranted |-> grant,
                        msource      |-> i,
                        mdest        |-> j
                    ]);

                    debug(<<"HandleRequestVoteRequest", i, j, currentTerm[i], grant>>);
                };
            } else if (m.mtype = RequestVoteResponse) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* DropStaleResponse
                if (m.mterm < currentTerm[self]) {
                    \* goto serverLoop;
                    skip;
                } else {
                    \* HandleRequestVoteResponse
                    with (i = self, j = m.msource) {
                        assert m.mterm = currentTerm[i];
                        votesResponded[i] := votesResponded[i] \cup {j};
                        if (m.mvoteGranted) {
                            votesGranted[i] := votesGranted[i] \cup {j};
                            if (
                                /\ state[i] = Candidate
                                /\ isQuorum(votesGranted[i])
                            ) {
                                becomeLeaderCh[i] := TRUE;
                            };
                        }; 
                    };
                };
            } else if (m.mtype = AppendEntriesRequest) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* HandleAppendEntriesRequest
                with (
                    i = self, j = m.msource,
                    logOK = \/ m.mprevLogIndex = 0
                            \/ /\ m.mprevLogIndex > 0
                               /\ m.mprevLogIndex <= Len(log[i])
                               /\ m.mprevLogTerm = log[i][m.mprevLogIndex].term
                ) {
                    assert m.mterm <= currentTerm[i];

                    if (m.mterm = currentTerm[i]) {
                        leader[i] := m.msource;
                        leaderTimeout := LeaderTimeoutReset;
                    };

                    \* return to follower state
                    if (
                        /\ m.mterm = currentTerm[i]
                        /\ state[i] = Candidate
                    ) {
                        state[i] := Follower;
                    };

                    \* reject request
                    if (
                        \/ m.mterm < currentTerm[i]
                        \/ /\ m.mterm = currentTerm[i]
                           /\ state[i] = Follower
                           /\ \lnot logOK
                    ) {
                        Send(net, j, fd, [
                            mtype       |-> AppendEntriesResponse,
                            mterm       |-> currentTerm[i],
                            msuccess    |-> FALSE,
                            mmatchIndex |-> 0,
                            msource     |-> i,
                            mdest       |-> j
                        ]);
                    }
                    \* accept request
                    else {
                        assert (
                            /\ m.mterm = currentTerm[i]
                            /\ state[i] = Follower
                            /\ logOK
                        );
                        with (index = m.mprevLogIndex + 1) {
                            plog[i] := [
                                cmd |-> LogPop, 
                                cnt |-> Len(log[i]) - m.mprevLogIndex
                            ];
                            log[i]  := SubSeq(log[i], 1, m.mprevLogIndex);

                            plog[i] := [
                                cmd     |-> LogConcat, 
                                entries |-> m.mentries
                            ];
                            log[i]  := log[i] \o m.mentries;

                            assert m.mcommitIndex <= Len(log[i]);

                            fAdvCommitIdxCh[i] := m;
                            Send(net, j, fd, [
                                mtype       |-> AppendEntriesResponse,
                                mterm       |-> currentTerm[i],
                                msuccess    |-> TRUE,
                                mmatchIndex |-> m.mprevLogIndex + Len(m.mentries),
                                msource     |-> i,
                                mdest       |-> j
                            ]); 
                        };
                    };
                };
            } else if (m.mtype = AppendEntriesResponse) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* DropStaleResponse
                if (m.mterm < currentTerm[self]) {
                    \* goto serverLoop;
                    skip;
                } else {
                    \* HandleAppendEntriesResponse
                    with (i = self, j = m.msource) {
                        assert m.mterm = currentTerm[i];
                        if (m.msuccess) {
                            nextIndex[i] := [nextIndex[i] EXCEPT ![j] = m.mmatchIndex + 1];
                            matchIndex[i] := [matchIndex[i] EXCEPT ![j] = m.mmatchIndex];
                        } else {
                            nextIndex[i] := [nextIndex[i] EXCEPT ![j] = Max({nextIndex[i][j]-1, 1})];
                        };
                    };
                };
            } else if (m.mtype = ProposeMessage) {
                \* HandleProposeMessage

                with (i = self) {
                    debug(<<"HandleProposeMessage", i, currentTerm[i], state[i], leader[i]>>);

                    if (state[i] = Leader) {
                        with (entry = [term |-> currentTerm[i],
                                       cmd  |-> m.mcmd]
                        ) {
                            log[i]  := Append(log[i], entry);
                            plog[i] := [cmd |-> LogConcat, entries |-> <<entry>>];

                            \* commented out for perf optimization
                            \* uncommented for model checking performance
                            appendEntriesCh[i] := TRUE;
                        };
                    } else if (leader[i] /= Nil) {
                        with (j = leader[i]) {
                            Send(net, j, fd, [
                                mtype   |-> ProposeMessage,
                                mcmd    |-> m.mcmd,
                                msource |-> i,
                                mdest   |-> j
                            ]);
                        };
                    };
                };
            };
        };
    }

    archetype AServerPropChListener(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
    variables
        idx = 1,
        m;
    {
    serverLoop:
        while (TRUE) {
            checkFail(srvId, netEnabled);

            with (i = srvId) {
                m := propCh[i];
                debug(<<"ReceiveProposeMessage", i, currentTerm[i], state[i], leader[i], m>>);
            };

        handleMsg:
            checkFail(srvId, netEnabled);

            \* HandleProposeMessage
            assert m.mtype = ProposeMessage;

            with (i = srvId) {
                debug(<<"HandleProposeMessage", i, currentTerm[i], state[i], leader[i]>>);

                if (state[i] = Leader) {
                    with (entry = [term |-> currentTerm[i],
                                    cmd  |-> m.mcmd]
                    ) {
                        log[i]  := Append(log[i], entry);
                        plog[i] := [cmd |-> LogConcat, entries |-> <<entry>>];

                        \* commented out for perf optimization
                        \* uncommented for model checking performance
                        appendEntriesCh[i] := TRUE;
                    };
                } else if (leader[i] /= Nil) {
                    with (j = leader[i]) {
                        Send(net, j, fd, [
                            mtype   |-> ProposeMessage,
                            mcmd    |-> m.mcmd,
                            msource |-> i,
                            mdest   |-> j
                        ]);
                    };
                };
            };
        };
    }

    archetype AServerRequestVote(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
    variable idx = 1;
    {
    serverRequestVoteLoop:
        while (TRUE) {
            checkFail(srvId, netEnabled);

            \* Server times out and starts a new election.
            await leaderTimeout;
            await netLen[srvId] = 0;

            await state[srvId] \in {Follower, Candidate};

            with (i = srvId) {
                state[i]          := Candidate;
                currentTerm[i]    := currentTerm[i] + 1;
                votedFor[i]       := i;
                votesResponded[i] := {i};
                votesGranted[i]   := {i};
                leader[i]         := Nil;

                debug(<<"ServerTimeout", i, currentTerm[i]>>);
            };

            idx := 1;
        requestVoteLoop:
            while (idx <= NumServers) {
                checkFail(srvId, netEnabled);

                \* RequestVote
                if (idx /= srvId) {
                    Send(net, idx, fd, [
                        mtype         |-> RequestVoteRequest,
                        mterm         |-> currentTerm[srvId],
                        mlastLogTerm  |-> LastTerm(log[srvId]),
                        mlastLogIndex |-> Len(log[srvId]),
                        msource       |-> srvId,
                        mdest         |-> idx
                    ]);
                };
                idx := idx + 1;
            };
        }
    }

    archetype AServerAppendEntries(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
    variable idx;
    {
    serverAppendEntriesLoop:
        while (appendEntriesCh[srvId]) {
            checkFail(srvId, netEnabled);

            await state[srvId] = Leader;
            idx := 1;

        appendEntriesLoop:
            \* AppendEntries
            while (
                /\ state[srvId] = Leader
                /\ idx <= NumServers
            ) {
                checkFail(srvId, netEnabled);

                if (idx /= srvId) {
                    with (
                        prevLogIndex = nextIndex[srvId][idx] - 1,
                        prevLogTerm  = IF prevLogIndex > 0
                                       THEN log[srvId][prevLogIndex].term
                                       ELSE 0,
                        \* lastEntry    = Min({Len(log[srvId]), nextIndex[srvId][idx]}),
                        entries      = SubSeq(log[srvId], nextIndex[srvId][idx], Len(log[srvId]))
                    ) {
                        Send(net, idx, fd, [
                            mtype         |-> AppendEntriesRequest,
                            mterm         |-> currentTerm[srvId],
                            mprevLogIndex |-> prevLogIndex,
                            mprevLogTerm  |-> prevLogTerm,
                            mentries      |-> entries,
                            mcommitIndex  |-> commitIndex[srvId],
                            msource       |-> srvId,
                            mdest         |-> idx
                        ]);
                    };
                };
                idx := idx + 1;
            };
        };
    }

    archetype AServerAdvanceCommitIndex(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
    variable newCommitIndex = 0,
    {
    serverAdvanceCommitIndexLoop:
        while (TRUE) {
            checkFail(srvId, netEnabled);

            await state[srvId] = Leader;

            \* AdvanceCommitIndex
            with (
                i = srvId,

                \* explores a much smaller space than the previous version,
                \* proportional to disagreement not the log size
                maxAgreeIndex = FindMaxAgreeIndex(log[i], i, matchIndex[i]),

                nCommitIndex =
                    IF /\ maxAgreeIndex /= Nil
                       /\ log[i][maxAgreeIndex].term = currentTerm[i]
                    THEN maxAgreeIndex
                    ELSE commitIndex[i]
            ) {
                newCommitIndex := nCommitIndex;
                assert newCommitIndex >= commitIndex[i];
            };

        applyLoop:
            while (commitIndex[srvId] < newCommitIndex) {
                checkFail(srvId, netEnabled);

                commitIndex[srvId] := commitIndex[srvId] + 1;
                with (
                    i = srvId,
                    k = commitIndex[i],
                    entry = log[i][k],
                    cmd   = entry.cmd
                ) {
                    debug(<<"ServerAccept", i, cmd>>);
                    acctCh[i] := [
                        mtype |-> AcceptMessage,
                        mcmd  |-> cmd
                    ];
                };
            };
        };
    }

    archetype AServerBecomeLeader(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
    {
    serverBecomeLeaderLoop:
        while (becomeLeaderCh[srvId]) {
            checkFail(srvId, netEnabled);

            \* BecomeLeader
            await state[srvId] = Candidate;
            await isQuorum(votesGranted[srvId]);

            with (i = srvId) {
                state[i]      := Leader;
                nextIndex[i]  := [j \in ServerSet |-> Len(log[i]) + 1];
                matchIndex[i] := [j \in ServerSet |-> 0];
                leader[i]     := i;

                appendEntriesCh[i] := TRUE;
                debug(<<"BecomeLeader", i, currentTerm[i], state[i], leader[i]>>);
            };
        };
    }

    archetype AServerFollowerAdvanceCommitIndex(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
    variable m;
    {
    serverFollowerAdvanceCommitIndexLoop:
        while (TRUE) {
            checkFail(srvId, netEnabled);

            m := fAdvCommitIdxCh[srvId];

        acctLoop:
            while (commitIndex[srvId] < m.mcommitIndex) {
                checkFail(srvId, netEnabled);

                commitIndex[srvId] := commitIndex[srvId] + 1;
                with (
                    i = srvId,
                    k = commitIndex[i],
                    entry = log[i][k],
                    cmd   = entry.cmd
                ) {
                    debug(<<"ServerAccept", i, cmd>>);
                    acctCh[i] := [
                        mtype |-> AcceptMessage,
                        mcmd  |-> cmd
                    ];
                };
            };
        };
    }

    archetype AServerCrasher(srvId, ref netEnabled[_], ref fd[_]) {
    serverCrash:
        netEnabled[srvId] := FALSE;
    fdUpdate:
        fd[srvId] := TRUE;
    }

    archetype AProposer(ref propCh[_]) {
    sndReq:
        propCh[1] := [
            mtype |-> ProposeMessage,
            mcmd  |-> [data |-> "hello"]
        ];
    }

    variables
        network = [i \in NodeSet |-> [queue |-> << >>, enabled |-> TRUE]];
        fd      = [i \in ServerSet |-> FALSE];

        state       = [i \in ServerSet |-> Follower];
        \* state       = [i \in ServerSet |-> IF i = 1 THEN Leader ELSE Follower];
        currentTerm = [i \in ServerSet |-> 1];

        commitIndex = [i \in ServerSet |-> 0];
        nextIndex   = [i \in ServerSet |-> [j \in ServerSet |-> 1]];
        matchIndex  = [i \in ServerSet |-> [j \in ServerSet |-> 0]];

        log  = [i \in ServerSet |-> <<>>];
        plog = [i \in ServerSet |-> <<>>];

        votedFor       = [i \in ServerSet |-> Nil];
        votesResponded = [i \in ServerSet |-> {}];
        votesGranted   = [i \in ServerSet |-> {}];

        leader = [i \in ServerSet |-> Nil];
        \* leader = [i \in ServerSet |-> 1];

        propCh = [i \in ServerSet |-> <<>>];
        acctCh = [i \in ServerSet |-> <<>>];

        leaderTimeout   = TRUE;
        appendEntriesCh = [i \in ServerSet |-> <<>>];
        becomeLeaderCh  = [i \in ServerSet |-> 
            IF NumServers > 1 
            THEN <<>>
            ELSE <<TRUE>>
        ];
        fAdvCommitIdxCh = [i \in ServerSet |-> <<>>];

        netListenerSrvId        = [i \in ServerNetListenerSet                |-> i - 0*NumServers];
        propChListenerSrvId     = [i \in ServerPropChListenerSet             |-> i - 1*NumServers];
        requestVoteSrvId        = [i \in ServerRequestVoteSet                |-> i - 2*NumServers];
        appendEntriesSrvId      = [i \in ServerAppendEntriesSet              |-> i - 3*NumServers];
        advanceCommitIndexSrvId = [i \in ServerAdvanceCommitIndexSet         |-> i - 4*NumServers];
        becomeLeaderSrvId       = [i \in ServerBecomeLeaderSet               |-> i - 5*NumServers];
        fAdvCommitSrvId         = [i \in ServerFollowerAdvanceCommitIndexSet |-> i - 6*NumServers];
        crasherSrvId            = [i \in ServerCrasherSet                    |-> i - 7*NumServers];

    fair process (s0 \in ServerNetListenerSet) == instance AServerNetListener(
        netListenerSrvId[s0],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
        mapping @2[_]  via ReliableFIFOLink
        mapping @3[_]  via NetworkBufferLength
        mapping @4[_]  via NetworkToggle
        mapping @5[_]  via PerfectFD
        mapping @9[_]  via PersistentLog
        mapping @17[_] via Channel
        mapping @18[_] via Channel
        mapping @20[_] via Channel
        mapping @21[_] via Channel
        mapping @22[_] via Channel;

    fair process (s1 \in ServerPropChListenerSet) == instance AServerPropChListener(
        propChListenerSrvId[s1],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
        mapping @2[_]  via ReliableFIFOLink
        mapping @3[_]  via NetworkBufferLength
        mapping @4[_]  via NetworkToggle
        mapping @5[_]  via PerfectFD
        mapping @9[_]  via PersistentLog
        mapping @17[_] via Channel
        mapping @18[_] via Channel
        mapping @20[_] via Channel
        mapping @21[_] via Channel
        mapping @22[_] via Channel;

    fair process (s2 \in ServerRequestVoteSet) == instance AServerRequestVote(
        requestVoteSrvId[s2],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
        mapping @2[_]  via ReliableFIFOLink
        mapping @3[_]  via NetworkBufferLength
        mapping @4[_]  via NetworkToggle
        mapping @5[_]  via PerfectFD
        mapping @9[_]  via PersistentLog
        mapping @17[_] via Channel
        mapping @18[_] via Channel
        mapping @20[_] via Channel
        mapping @21[_] via Channel
        mapping @22[_] via Channel;

    fair process (s3 \in ServerAppendEntriesSet) == instance AServerAppendEntries(
        appendEntriesSrvId[s3],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
        mapping @2[_]  via ReliableFIFOLink
        mapping @3[_]  via NetworkBufferLength
        mapping @4[_]  via NetworkToggle
        mapping @5[_]  via PerfectFD
        mapping @9[_]  via PersistentLog
        mapping @17[_] via Channel
        mapping @18[_] via Channel
        mapping @20[_] via Channel
        mapping @21[_] via Channel
        mapping @22[_] via Channel;

    fair process (s4 \in ServerAdvanceCommitIndexSet) == instance AServerAdvanceCommitIndex(
        advanceCommitIndexSrvId[s4],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
        mapping @2[_]  via ReliableFIFOLink
        mapping @3[_]  via NetworkBufferLength
        mapping @4[_]  via NetworkToggle
        mapping @5[_]  via PerfectFD
        mapping @9[_]  via PersistentLog
        mapping @17[_] via Channel
        mapping @18[_] via Channel
        mapping @20[_] via Channel
        mapping @21[_] via Channel
        mapping @22[_] via Channel;

    fair process (s5 \in ServerBecomeLeaderSet) == instance AServerBecomeLeader(
        becomeLeaderSrvId[s5],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
        mapping @2[_]  via ReliableFIFOLink
        mapping @3[_]  via NetworkBufferLength
        mapping @4[_]  via NetworkToggle
        mapping @5[_]  via PerfectFD
        mapping @9[_]  via PersistentLog
        mapping @17[_] via Channel
        mapping @18[_] via Channel
        mapping @20[_] via Channel
        mapping @21[_] via Channel
        mapping @22[_] via Channel;

    fair process(s6 \in ServerFollowerAdvanceCommitIndexSet) == instance AServerFollowerAdvanceCommitIndex(
        fAdvCommitSrvId[s6],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref propCh[_], ref acctCh[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_], ref fAdvCommitIdxCh[_]
    )
        mapping @2[_]  via ReliableFIFOLink
        mapping @3[_]  via NetworkBufferLength
        mapping @4[_]  via NetworkToggle
        mapping @5[_]  via PerfectFD
        mapping @9[_]  via PersistentLog
        mapping @17[_] via Channel
        mapping @18[_] via Channel
        mapping @20[_] via Channel
        mapping @21[_] via Channel
        mapping @22[_] via Channel;

    fair process (crasher \in ServerCrasherSet) == instance AServerCrasher(
        crasherSrvId[crasher],
        ref network[_], ref fd[_]
    )
        mapping @2[_] via NetworkToggle
        mapping @3[_] via PerfectFD;
    
    fair process (proposer = 0) == instance AProposer(ref propCh[_])
        mapping @1[_] via Channel;
}

\* BEGIN PLUSCAL TRANSLATION
--algorithm raft {
  variables network = [i \in NodeSet |-> [queue |-> <<>>, enabled |-> TRUE]]; fd = [i \in ServerSet |-> FALSE]; state = [i \in ServerSet |-> Follower]; currentTerm = [i \in ServerSet |-> 1]; commitIndex = [i \in ServerSet |-> 0]; nextIndex = [i \in ServerSet |-> [j \in ServerSet |-> 1]]; matchIndex = [i \in ServerSet |-> [j \in ServerSet |-> 0]]; log = [i \in ServerSet |-> <<>>]; plog = [i \in ServerSet |-> <<>>]; votedFor = [i \in ServerSet |-> Nil]; votesResponded = [i \in ServerSet |-> {}]; votesGranted = [i \in ServerSet |-> {}]; leader = [i \in ServerSet |-> Nil]; propCh = [i \in ServerSet |-> <<>>]; acctCh = [i \in ServerSet |-> <<>>]; leaderTimeout = TRUE; appendEntriesCh = [i \in ServerSet |-> <<>>]; becomeLeaderCh = [i \in ServerSet |-> IF (NumServers) > (1) THEN <<>> ELSE <<TRUE>>]; fAdvCommitIdxCh = [i \in ServerSet |-> <<>>]; netListenerSrvId = [i \in ServerNetListenerSet |-> (i) - ((0) * (NumServers))]; propChListenerSrvId = [i \in ServerPropChListenerSet |-> (i) - ((1) * (NumServers))]; requestVoteSrvId = [i \in ServerRequestVoteSet |-> (i) - ((2) * (NumServers))]; appendEntriesSrvId = [i \in ServerAppendEntriesSet |-> (i) - ((3) * (NumServers))]; advanceCommitIndexSrvId = [i \in ServerAdvanceCommitIndexSet |-> (i) - ((4) * (NumServers))]; becomeLeaderSrvId = [i \in ServerBecomeLeaderSet |-> (i) - ((5) * (NumServers))]; fAdvCommitSrvId = [i \in ServerFollowerAdvanceCommitIndexSet |-> (i) - ((6) * (NumServers))]; crasherSrvId = [i \in ServerCrasherSet |-> (i) - ((7) * (NumServers))];
  define{
    Follower == "follower"
    Candidate == "candidate"
    Leader == "leader"
    RequestVoteRequest == "rvq"
    RequestVoteResponse == "rvp"
    AppendEntriesRequest == "apq"
    AppendEntriesResponse == "app"
    ProposeMessage == "prm"
    AcceptMessage == "acm"
    Key1 == "key1"
    Key2 == "key2"
    Value1 == "value1"
    LastTerm(xlog) == IF (Len(xlog)) = (0) THEN 0 ELSE ((xlog)[Len(xlog)]).term
    ServerNetListenerSet == (((0) * (NumServers)) + (1)) .. ((1) * (NumServers))
    ServerPropChListenerSet == (((1) * (NumServers)) + (1)) .. ((2) * (NumServers))
    ServerRequestVoteSet == (((2) * (NumServers)) + (1)) .. ((3) * (NumServers))
    ServerAppendEntriesSet == (((3) * (NumServers)) + (1)) .. ((4) * (NumServers))
    ServerAdvanceCommitIndexSet == (((4) * (NumServers)) + (1)) .. ((5) * (NumServers))
    ServerBecomeLeaderSet == (((5) * (NumServers)) + (1)) .. ((6) * (NumServers))
    ServerFollowerAdvanceCommitIndexSet == (((6) * (NumServers)) + (1)) .. ((7) * (NumServers))
    ServerCrasherSet == IF ExploreFail THEN (((7) * (NumServers)) + (1)) .. (((7) * (NumServers)) + (MaxNodeFail)) ELSE {}
    NodeSet == ServerSet
    KeySet == {}
  }
  
  fair process (s0 \in ServerNetListenerSet)
    variables idx = 1; m; srvId = (netListenerSrvId)[self];
  {
    serverLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network14 = ((network)[srvId]).enabled) {
            if (~ (yielded_network14)) {
              await FALSE;
              assert ((network)[self]).enabled;
              await (Len(((network)[self]).queue)) > (0);
              with (readMsg00 = Head(((network)[self]).queue)) {
                network := [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]];
                with (yielded_network00 = readMsg00) {
                  m := yielded_network00;
                  assert ((m).mdest) = (self);
                  goto handleMsg;
                };
              };
            } else {
              assert ((network)[self]).enabled;
              await (Len(((network)[self]).queue)) > (0);
              with (readMsg01 = Head(((network)[self]).queue)) {
                network := [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]];
                with (yielded_network01 = readMsg01) {
                  m := yielded_network01;
                  assert ((m).mdest) = (self);
                  goto handleMsg;
                };
              };
            };
          };
        } else {
          assert ((network)[self]).enabled;
          await (Len(((network)[self]).queue)) > (0);
          with (readMsg02 = Head(((network)[self]).queue)) {
            network := [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]];
            with (yielded_network02 = readMsg02) {
              m := yielded_network02;
              assert ((m).mdest) = (self);
              goto handleMsg;
            };
          };
        };
      } else {
        goto Done;
      };
    handleMsg:
      if (ExploreFail) {
        with (yielded_network15 = ((network)[srvId]).enabled) {
          if (~ (yielded_network15)) {
            await FALSE;
            if (((m).mtype) = (RequestVoteRequest)) {
              if (((m).mterm) > ((currentTerm)[self])) {
                currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                state := [state EXCEPT ![self] = Follower];
                with (votedFor3 = [votedFor EXCEPT ![self] = Nil]) {
                  leader := [leader EXCEPT ![self] = Nil];
                  with (
                    i = self, 
                    j = (m).msource, 
                    logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                    grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor3)[self]) \in ({Nil, j}))
                  ) {
                    assert ((m).mterm) <= ((currentTerm)[i]);
                    if (grant) {
                      votedFor := [votedFor3 EXCEPT ![i] = j];
                      either {
                        with (value20 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value20), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd6 = (fd)[j]) {
                          await yielded_fd6;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      either {
                        with (value21 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value21), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor3;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor3;
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd7 = (fd)[j]) {
                          await yielded_fd7;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor3;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor3;
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                };
              } else {
                with (
                  i = self, 
                  j = (m).msource, 
                  logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                  grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j}))
                ) {
                  assert ((m).mterm) <= ((currentTerm)[i]);
                  if (grant) {
                    votedFor := [votedFor EXCEPT ![i] = j];
                    either {
                      with (value22 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value22), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd8 = (fd)[j]) {
                        await yielded_fd8;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    either {
                      with (value23 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value23), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd9 = (fd)[j]) {
                        await yielded_fd9;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  };
                };
              };
            } else {
              if (((m).mtype) = (RequestVoteResponse)) {
                if (((m).mterm) > ((currentTerm)[self])) {
                  currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                  state := [state EXCEPT ![self] = Follower];
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  leader := [leader EXCEPT ![self] = Nil];
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value00 = TRUE) {
                            await (Len((becomeLeaderCh)[i])) < (BufferSize);
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value00)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value01 = TRUE) {
                            await (Len((becomeLeaderCh)[i])) < (BufferSize);
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value01)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                };
              } else {
                if (((m).mtype) = (AppendEntriesRequest)) {
                  if (((m).mterm) > ((currentTerm)[self])) {
                    currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                    with (state3 = [state EXCEPT ![self] = Follower]) {
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      with (
                        leader3 = [leader EXCEPT ![self] = Nil], 
                        i = self, 
                        j = (m).msource, 
                        logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                      ) {
                        assert ((m).mterm) <= ((currentTerm)[i]);
                        if (((m).mterm) = ((currentTerm)[i])) {
                          leader := [leader3 EXCEPT ![i] = (m).msource];
                          leaderTimeout := LeaderTimeoutReset;
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Candidate))) {
                            state := [state3 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value110 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value110), enabled |-> ((network)[j]).enabled]];
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd00 = (fd)[j]) {
                                  await yielded_fd00;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value24 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value24).cmd) = (LogConcat)) {
                                  with (
                                    plog24 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value24).entries)], 
                                    log24 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value30 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value30).cmd) = (LogConcat)) {
                                      plog := [plog24 EXCEPT ![i] = ((plog24)[i]) \o ((value30).entries)];
                                      log := [log24 EXCEPT ![i] = ((log24)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value40 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value40)];
                                        either {
                                          with (value50 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value50), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd10 = (fd)[j]) {
                                            await yielded_fd10;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value30).cmd) = (LogPop)) {
                                        plog := [plog24 EXCEPT ![i] = SubSeq((plog24)[i], 1, (Len((plog24)[i])) - ((value30).cnt))];
                                        log := [log24 EXCEPT ![i] = ((log24)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value41 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value41)];
                                          either {
                                            with (value51 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value51), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd11 = (fd)[j]) {
                                              await yielded_fd11;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log24 EXCEPT ![i] = ((log24)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value42 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value42)];
                                          either {
                                            with (value52 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value52), enabled |-> ((network)[j]).enabled]];
                                              plog := plog24;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd12 = (fd)[j]) {
                                              await yielded_fd12;
                                              plog := plog24;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value24).cmd) = (LogPop)) {
                                    with (
                                      plog25 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value24).cnt))], 
                                      log25 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value31 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value31).cmd) = (LogConcat)) {
                                        plog := [plog25 EXCEPT ![i] = ((plog25)[i]) \o ((value31).entries)];
                                        log := [log25 EXCEPT ![i] = ((log25)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value43 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value43)];
                                          either {
                                            with (value53 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value53), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd13 = (fd)[j]) {
                                              await yielded_fd13;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value31).cmd) = (LogPop)) {
                                          plog := [plog25 EXCEPT ![i] = SubSeq((plog25)[i], 1, (Len((plog25)[i])) - ((value31).cnt))];
                                          log := [log25 EXCEPT ![i] = ((log25)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value44 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value44)];
                                            either {
                                              with (value54 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value54), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd14 = (fd)[j]) {
                                                await yielded_fd14;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log25 EXCEPT ![i] = ((log25)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value45 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value45)];
                                            either {
                                              with (value55 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value55), enabled |-> ((network)[j]).enabled]];
                                                plog := plog25;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd15 = (fd)[j]) {
                                                await yielded_fd15;
                                                plog := plog25;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log26 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value32 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value32).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value32).entries)];
                                        log := [log26 EXCEPT ![i] = ((log26)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value46 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value46)];
                                          either {
                                            with (value56 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value56), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd16 = (fd)[j]) {
                                              await yielded_fd16;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value32).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value32).cnt))];
                                          log := [log26 EXCEPT ![i] = ((log26)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value47 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value47)];
                                            either {
                                              with (value57 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value57), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd17 = (fd)[j]) {
                                                await yielded_fd17;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log26 EXCEPT ![i] = ((log26)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value48 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value48)];
                                            either {
                                              with (value58 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value58), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd18 = (fd)[j]) {
                                                await yielded_fd18;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value111 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value111), enabled |-> ((network)[j]).enabled]];
                                  state := state3;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd01 = (fd)[j]) {
                                  await yielded_fd01;
                                  state := state3;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value25 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value25).cmd) = (LogConcat)) {
                                  with (
                                    plog26 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value25).entries)], 
                                    log27 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value33 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value33).cmd) = (LogConcat)) {
                                      plog := [plog26 EXCEPT ![i] = ((plog26)[i]) \o ((value33).entries)];
                                      log := [log27 EXCEPT ![i] = ((log27)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value49 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value49)];
                                        either {
                                          with (value59 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value59), enabled |-> ((network)[j]).enabled]];
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd19 = (fd)[j]) {
                                            await yielded_fd19;
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value33).cmd) = (LogPop)) {
                                        plog := [plog26 EXCEPT ![i] = SubSeq((plog26)[i], 1, (Len((plog26)[i])) - ((value33).cnt))];
                                        log := [log27 EXCEPT ![i] = ((log27)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value410 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value410)];
                                          either {
                                            with (value510 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value510), enabled |-> ((network)[j]).enabled]];
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd110 = (fd)[j]) {
                                              await yielded_fd110;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log27 EXCEPT ![i] = ((log27)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value411 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value411)];
                                          either {
                                            with (value511 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value511), enabled |-> ((network)[j]).enabled]];
                                              plog := plog26;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd111 = (fd)[j]) {
                                              await yielded_fd111;
                                              plog := plog26;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value25).cmd) = (LogPop)) {
                                    with (
                                      plog27 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value25).cnt))], 
                                      log28 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value34 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value34).cmd) = (LogConcat)) {
                                        plog := [plog27 EXCEPT ![i] = ((plog27)[i]) \o ((value34).entries)];
                                        log := [log28 EXCEPT ![i] = ((log28)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value412 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value412)];
                                          either {
                                            with (value512 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value512), enabled |-> ((network)[j]).enabled]];
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd112 = (fd)[j]) {
                                              await yielded_fd112;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value34).cmd) = (LogPop)) {
                                          plog := [plog27 EXCEPT ![i] = SubSeq((plog27)[i], 1, (Len((plog27)[i])) - ((value34).cnt))];
                                          log := [log28 EXCEPT ![i] = ((log28)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value413 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value413)];
                                            either {
                                              with (value513 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value513), enabled |-> ((network)[j]).enabled]];
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd113 = (fd)[j]) {
                                                await yielded_fd113;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log28 EXCEPT ![i] = ((log28)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value414 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value414)];
                                            either {
                                              with (value514 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value514), enabled |-> ((network)[j]).enabled]];
                                                plog := plog27;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd114 = (fd)[j]) {
                                                await yielded_fd114;
                                                plog := plog27;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log29 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value35 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value35).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value35).entries)];
                                        log := [log29 EXCEPT ![i] = ((log29)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value415 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value415)];
                                          either {
                                            with (value515 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value515), enabled |-> ((network)[j]).enabled]];
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd115 = (fd)[j]) {
                                              await yielded_fd115;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value35).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value35).cnt))];
                                          log := [log29 EXCEPT ![i] = ((log29)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value416 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value416)];
                                            either {
                                              with (value516 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value516), enabled |-> ((network)[j]).enabled]];
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd116 = (fd)[j]) {
                                                await yielded_fd116;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log29 EXCEPT ![i] = ((log29)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value417 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value417)];
                                            either {
                                              with (value517 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value517), enabled |-> ((network)[j]).enabled]];
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd117 = (fd)[j]) {
                                                await yielded_fd117;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Candidate))) {
                            state := [state3 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value112 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value112), enabled |-> ((network)[j]).enabled]];
                                  leader := leader3;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd02 = (fd)[j]) {
                                  await yielded_fd02;
                                  leader := leader3;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value26 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value26).cmd) = (LogConcat)) {
                                  with (
                                    plog28 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value26).entries)], 
                                    log30 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value36 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value36).cmd) = (LogConcat)) {
                                      plog := [plog28 EXCEPT ![i] = ((plog28)[i]) \o ((value36).entries)];
                                      log := [log30 EXCEPT ![i] = ((log30)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value418 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value418)];
                                        either {
                                          with (value518 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value518), enabled |-> ((network)[j]).enabled]];
                                            leader := leader3;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd118 = (fd)[j]) {
                                            await yielded_fd118;
                                            leader := leader3;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value36).cmd) = (LogPop)) {
                                        plog := [plog28 EXCEPT ![i] = SubSeq((plog28)[i], 1, (Len((plog28)[i])) - ((value36).cnt))];
                                        log := [log30 EXCEPT ![i] = ((log30)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value419 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value419)];
                                          either {
                                            with (value519 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value519), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd119 = (fd)[j]) {
                                              await yielded_fd119;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log30 EXCEPT ![i] = ((log30)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value420 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value420)];
                                          either {
                                            with (value520 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value520), enabled |-> ((network)[j]).enabled]];
                                              plog := plog28;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd120 = (fd)[j]) {
                                              await yielded_fd120;
                                              plog := plog28;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value26).cmd) = (LogPop)) {
                                    with (
                                      plog29 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value26).cnt))], 
                                      log31 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value37 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value37).cmd) = (LogConcat)) {
                                        plog := [plog29 EXCEPT ![i] = ((plog29)[i]) \o ((value37).entries)];
                                        log := [log31 EXCEPT ![i] = ((log31)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value421 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value421)];
                                          either {
                                            with (value521 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value521), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd121 = (fd)[j]) {
                                              await yielded_fd121;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value37).cmd) = (LogPop)) {
                                          plog := [plog29 EXCEPT ![i] = SubSeq((plog29)[i], 1, (Len((plog29)[i])) - ((value37).cnt))];
                                          log := [log31 EXCEPT ![i] = ((log31)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value422 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value422)];
                                            either {
                                              with (value522 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value522), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd122 = (fd)[j]) {
                                                await yielded_fd122;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log31 EXCEPT ![i] = ((log31)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value423 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value423)];
                                            either {
                                              with (value523 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value523), enabled |-> ((network)[j]).enabled]];
                                                plog := plog29;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd123 = (fd)[j]) {
                                                await yielded_fd123;
                                                plog := plog29;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log32 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value38 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value38).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value38).entries)];
                                        log := [log32 EXCEPT ![i] = ((log32)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value424 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value424)];
                                          either {
                                            with (value524 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value524), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd124 = (fd)[j]) {
                                              await yielded_fd124;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value38).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value38).cnt))];
                                          log := [log32 EXCEPT ![i] = ((log32)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value425 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value425)];
                                            either {
                                              with (value525 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value525), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd125 = (fd)[j]) {
                                                await yielded_fd125;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log32 EXCEPT ![i] = ((log32)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value426 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value426)];
                                            either {
                                              with (value526 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value526), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd126 = (fd)[j]) {
                                                await yielded_fd126;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value113 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value113), enabled |-> ((network)[j]).enabled]];
                                  leader := leader3;
                                  state := state3;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd03 = (fd)[j]) {
                                  await yielded_fd03;
                                  leader := leader3;
                                  state := state3;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value27 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value27).cmd) = (LogConcat)) {
                                  with (
                                    plog30 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value27).entries)], 
                                    log33 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value39 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value39).cmd) = (LogConcat)) {
                                      plog := [plog30 EXCEPT ![i] = ((plog30)[i]) \o ((value39).entries)];
                                      log := [log33 EXCEPT ![i] = ((log33)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value427 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value427)];
                                        either {
                                          with (value527 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value527), enabled |-> ((network)[j]).enabled]];
                                            leader := leader3;
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd127 = (fd)[j]) {
                                            await yielded_fd127;
                                            leader := leader3;
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value39).cmd) = (LogPop)) {
                                        plog := [plog30 EXCEPT ![i] = SubSeq((plog30)[i], 1, (Len((plog30)[i])) - ((value39).cnt))];
                                        log := [log33 EXCEPT ![i] = ((log33)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value428 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value428)];
                                          either {
                                            with (value528 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value528), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd128 = (fd)[j]) {
                                              await yielded_fd128;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log33 EXCEPT ![i] = ((log33)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value429 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value429)];
                                          either {
                                            with (value529 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value529), enabled |-> ((network)[j]).enabled]];
                                              plog := plog30;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd129 = (fd)[j]) {
                                              await yielded_fd129;
                                              plog := plog30;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value27).cmd) = (LogPop)) {
                                    with (
                                      plog31 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value27).cnt))], 
                                      log34 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value310 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value310).cmd) = (LogConcat)) {
                                        plog := [plog31 EXCEPT ![i] = ((plog31)[i]) \o ((value310).entries)];
                                        log := [log34 EXCEPT ![i] = ((log34)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value430 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value430)];
                                          either {
                                            with (value530 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value530), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd130 = (fd)[j]) {
                                              await yielded_fd130;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value310).cmd) = (LogPop)) {
                                          plog := [plog31 EXCEPT ![i] = SubSeq((plog31)[i], 1, (Len((plog31)[i])) - ((value310).cnt))];
                                          log := [log34 EXCEPT ![i] = ((log34)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value431 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value431)];
                                            either {
                                              with (value531 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value531), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd131 = (fd)[j]) {
                                                await yielded_fd131;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log34 EXCEPT ![i] = ((log34)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value432 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value432)];
                                            either {
                                              with (value532 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value532), enabled |-> ((network)[j]).enabled]];
                                                plog := plog31;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd132 = (fd)[j]) {
                                                await yielded_fd132;
                                                plog := plog31;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log35 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value311 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value311).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value311).entries)];
                                        log := [log35 EXCEPT ![i] = ((log35)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value433 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value433)];
                                          either {
                                            with (value533 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value533), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd133 = (fd)[j]) {
                                              await yielded_fd133;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value311).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value311).cnt))];
                                          log := [log35 EXCEPT ![i] = ((log35)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value434 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value434)];
                                            either {
                                              with (value534 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value534), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd134 = (fd)[j]) {
                                                await yielded_fd134;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log35 EXCEPT ![i] = ((log35)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value435 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value435)];
                                            either {
                                              with (value535 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value535), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd135 = (fd)[j]) {
                                                await yielded_fd135;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  } else {
                    with (
                      i = self, 
                      j = (m).msource, 
                      logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                    ) {
                      assert ((m).mterm) <= ((currentTerm)[i]);
                      if (((m).mterm) = ((currentTerm)[i])) {
                        leader := [leader EXCEPT ![i] = (m).msource];
                        leaderTimeout := LeaderTimeoutReset;
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value114 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value114), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd04 = (fd)[j]) {
                                await yielded_fd04;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value28 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value28).cmd) = (LogConcat)) {
                                with (
                                  plog32 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value28).entries)], 
                                  log36 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value312 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value312).cmd) = (LogConcat)) {
                                    plog := [plog32 EXCEPT ![i] = ((plog32)[i]) \o ((value312).entries)];
                                    log := [log36 EXCEPT ![i] = ((log36)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value436 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value436)];
                                      either {
                                        with (value536 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value536), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd136 = (fd)[j]) {
                                          await yielded_fd136;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value312).cmd) = (LogPop)) {
                                      plog := [plog32 EXCEPT ![i] = SubSeq((plog32)[i], 1, (Len((plog32)[i])) - ((value312).cnt))];
                                      log := [log36 EXCEPT ![i] = ((log36)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value437 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value437)];
                                        either {
                                          with (value537 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value537), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd137 = (fd)[j]) {
                                            await yielded_fd137;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log36 EXCEPT ![i] = ((log36)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value438 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value438)];
                                        either {
                                          with (value538 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value538), enabled |-> ((network)[j]).enabled]];
                                            plog := plog32;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd138 = (fd)[j]) {
                                            await yielded_fd138;
                                            plog := plog32;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value28).cmd) = (LogPop)) {
                                  with (
                                    plog33 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value28).cnt))], 
                                    log37 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value313 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value313).cmd) = (LogConcat)) {
                                      plog := [plog33 EXCEPT ![i] = ((plog33)[i]) \o ((value313).entries)];
                                      log := [log37 EXCEPT ![i] = ((log37)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value439 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value439)];
                                        either {
                                          with (value539 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value539), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd139 = (fd)[j]) {
                                            await yielded_fd139;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value313).cmd) = (LogPop)) {
                                        plog := [plog33 EXCEPT ![i] = SubSeq((plog33)[i], 1, (Len((plog33)[i])) - ((value313).cnt))];
                                        log := [log37 EXCEPT ![i] = ((log37)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value440 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value440)];
                                          either {
                                            with (value540 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value540), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd140 = (fd)[j]) {
                                              await yielded_fd140;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log37 EXCEPT ![i] = ((log37)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value441 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value441)];
                                          either {
                                            with (value541 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value541), enabled |-> ((network)[j]).enabled]];
                                              plog := plog33;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd141 = (fd)[j]) {
                                              await yielded_fd141;
                                              plog := plog33;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log38 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value314 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value314).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value314).entries)];
                                      log := [log38 EXCEPT ![i] = ((log38)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value442 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value442)];
                                        either {
                                          with (value542 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value542), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd142 = (fd)[j]) {
                                            await yielded_fd142;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value314).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value314).cnt))];
                                        log := [log38 EXCEPT ![i] = ((log38)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value443 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value443)];
                                          either {
                                            with (value543 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value543), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd143 = (fd)[j]) {
                                              await yielded_fd143;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log38 EXCEPT ![i] = ((log38)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value444 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value444)];
                                          either {
                                            with (value544 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value544), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd144 = (fd)[j]) {
                                              await yielded_fd144;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value115 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value115), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd05 = (fd)[j]) {
                                await yielded_fd05;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value29 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value29).cmd) = (LogConcat)) {
                                with (
                                  plog34 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value29).entries)], 
                                  log39 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value315 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value315).cmd) = (LogConcat)) {
                                    plog := [plog34 EXCEPT ![i] = ((plog34)[i]) \o ((value315).entries)];
                                    log := [log39 EXCEPT ![i] = ((log39)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value445 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value445)];
                                      either {
                                        with (value545 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value545), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd145 = (fd)[j]) {
                                          await yielded_fd145;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value315).cmd) = (LogPop)) {
                                      plog := [plog34 EXCEPT ![i] = SubSeq((plog34)[i], 1, (Len((plog34)[i])) - ((value315).cnt))];
                                      log := [log39 EXCEPT ![i] = ((log39)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value446 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value446)];
                                        either {
                                          with (value546 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value546), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd146 = (fd)[j]) {
                                            await yielded_fd146;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log39 EXCEPT ![i] = ((log39)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value447 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value447)];
                                        either {
                                          with (value547 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value547), enabled |-> ((network)[j]).enabled]];
                                            plog := plog34;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd147 = (fd)[j]) {
                                            await yielded_fd147;
                                            plog := plog34;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value29).cmd) = (LogPop)) {
                                  with (
                                    plog35 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value29).cnt))], 
                                    log40 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value316 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value316).cmd) = (LogConcat)) {
                                      plog := [plog35 EXCEPT ![i] = ((plog35)[i]) \o ((value316).entries)];
                                      log := [log40 EXCEPT ![i] = ((log40)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value448 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value448)];
                                        either {
                                          with (value548 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value548), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd148 = (fd)[j]) {
                                            await yielded_fd148;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value316).cmd) = (LogPop)) {
                                        plog := [plog35 EXCEPT ![i] = SubSeq((plog35)[i], 1, (Len((plog35)[i])) - ((value316).cnt))];
                                        log := [log40 EXCEPT ![i] = ((log40)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value449 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value449)];
                                          either {
                                            with (value549 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value549), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd149 = (fd)[j]) {
                                              await yielded_fd149;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log40 EXCEPT ![i] = ((log40)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value450 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value450)];
                                          either {
                                            with (value550 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value550), enabled |-> ((network)[j]).enabled]];
                                              plog := plog35;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd150 = (fd)[j]) {
                                              await yielded_fd150;
                                              plog := plog35;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log41 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value317 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value317).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value317).entries)];
                                      log := [log41 EXCEPT ![i] = ((log41)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value451 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value451)];
                                        either {
                                          with (value551 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value551), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd151 = (fd)[j]) {
                                            await yielded_fd151;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value317).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value317).cnt))];
                                        log := [log41 EXCEPT ![i] = ((log41)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value452 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value452)];
                                          either {
                                            with (value552 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value552), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd152 = (fd)[j]) {
                                              await yielded_fd152;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log41 EXCEPT ![i] = ((log41)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value453 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value453)];
                                          either {
                                            with (value553 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value553), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd153 = (fd)[j]) {
                                              await yielded_fd153;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value116 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value116), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd06 = (fd)[j]) {
                                await yielded_fd06;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value210 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value210).cmd) = (LogConcat)) {
                                with (
                                  plog36 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value210).entries)], 
                                  log42 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value318 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value318).cmd) = (LogConcat)) {
                                    plog := [plog36 EXCEPT ![i] = ((plog36)[i]) \o ((value318).entries)];
                                    log := [log42 EXCEPT ![i] = ((log42)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value454 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value454)];
                                      either {
                                        with (value554 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value554), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd154 = (fd)[j]) {
                                          await yielded_fd154;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value318).cmd) = (LogPop)) {
                                      plog := [plog36 EXCEPT ![i] = SubSeq((plog36)[i], 1, (Len((plog36)[i])) - ((value318).cnt))];
                                      log := [log42 EXCEPT ![i] = ((log42)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value455 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value455)];
                                        either {
                                          with (value555 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value555), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd155 = (fd)[j]) {
                                            await yielded_fd155;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log42 EXCEPT ![i] = ((log42)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value456 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value456)];
                                        either {
                                          with (value556 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value556), enabled |-> ((network)[j]).enabled]];
                                            plog := plog36;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd156 = (fd)[j]) {
                                            await yielded_fd156;
                                            plog := plog36;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value210).cmd) = (LogPop)) {
                                  with (
                                    plog37 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value210).cnt))], 
                                    log43 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value319 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value319).cmd) = (LogConcat)) {
                                      plog := [plog37 EXCEPT ![i] = ((plog37)[i]) \o ((value319).entries)];
                                      log := [log43 EXCEPT ![i] = ((log43)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value457 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value457)];
                                        either {
                                          with (value557 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value557), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd157 = (fd)[j]) {
                                            await yielded_fd157;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value319).cmd) = (LogPop)) {
                                        plog := [plog37 EXCEPT ![i] = SubSeq((plog37)[i], 1, (Len((plog37)[i])) - ((value319).cnt))];
                                        log := [log43 EXCEPT ![i] = ((log43)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value458 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value458)];
                                          either {
                                            with (value558 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value558), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd158 = (fd)[j]) {
                                              await yielded_fd158;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log43 EXCEPT ![i] = ((log43)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value459 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value459)];
                                          either {
                                            with (value559 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value559), enabled |-> ((network)[j]).enabled]];
                                              plog := plog37;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd159 = (fd)[j]) {
                                              await yielded_fd159;
                                              plog := plog37;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log44 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value320 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value320).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value320).entries)];
                                      log := [log44 EXCEPT ![i] = ((log44)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value460 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value460)];
                                        either {
                                          with (value560 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value560), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd160 = (fd)[j]) {
                                            await yielded_fd160;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value320).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value320).cnt))];
                                        log := [log44 EXCEPT ![i] = ((log44)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value461 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value461)];
                                          either {
                                            with (value561 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value561), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd161 = (fd)[j]) {
                                              await yielded_fd161;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log44 EXCEPT ![i] = ((log44)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value462 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value462)];
                                          either {
                                            with (value562 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value562), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd162 = (fd)[j]) {
                                              await yielded_fd162;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value117 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value117), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd07 = (fd)[j]) {
                                await yielded_fd07;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value211 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value211).cmd) = (LogConcat)) {
                                with (
                                  plog38 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value211).entries)], 
                                  log45 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value321 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value321).cmd) = (LogConcat)) {
                                    plog := [plog38 EXCEPT ![i] = ((plog38)[i]) \o ((value321).entries)];
                                    log := [log45 EXCEPT ![i] = ((log45)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value463 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value463)];
                                      either {
                                        with (value563 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value563), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd163 = (fd)[j]) {
                                          await yielded_fd163;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value321).cmd) = (LogPop)) {
                                      plog := [plog38 EXCEPT ![i] = SubSeq((plog38)[i], 1, (Len((plog38)[i])) - ((value321).cnt))];
                                      log := [log45 EXCEPT ![i] = ((log45)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value464 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value464)];
                                        either {
                                          with (value564 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value564), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd164 = (fd)[j]) {
                                            await yielded_fd164;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log45 EXCEPT ![i] = ((log45)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value465 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value465)];
                                        either {
                                          with (value565 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value565), enabled |-> ((network)[j]).enabled]];
                                            plog := plog38;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd165 = (fd)[j]) {
                                            await yielded_fd165;
                                            plog := plog38;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value211).cmd) = (LogPop)) {
                                  with (
                                    plog39 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value211).cnt))], 
                                    log46 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value322 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value322).cmd) = (LogConcat)) {
                                      plog := [plog39 EXCEPT ![i] = ((plog39)[i]) \o ((value322).entries)];
                                      log := [log46 EXCEPT ![i] = ((log46)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value466 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value466)];
                                        either {
                                          with (value566 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value566), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd166 = (fd)[j]) {
                                            await yielded_fd166;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value322).cmd) = (LogPop)) {
                                        plog := [plog39 EXCEPT ![i] = SubSeq((plog39)[i], 1, (Len((plog39)[i])) - ((value322).cnt))];
                                        log := [log46 EXCEPT ![i] = ((log46)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value467 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value467)];
                                          either {
                                            with (value567 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value567), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd167 = (fd)[j]) {
                                              await yielded_fd167;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log46 EXCEPT ![i] = ((log46)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value468 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value468)];
                                          either {
                                            with (value568 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value568), enabled |-> ((network)[j]).enabled]];
                                              plog := plog39;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd168 = (fd)[j]) {
                                              await yielded_fd168;
                                              plog := plog39;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log47 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value323 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value323).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value323).entries)];
                                      log := [log47 EXCEPT ![i] = ((log47)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value469 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value469)];
                                        either {
                                          with (value569 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value569), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd169 = (fd)[j]) {
                                            await yielded_fd169;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value323).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value323).cnt))];
                                        log := [log47 EXCEPT ![i] = ((log47)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value470 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value470)];
                                          either {
                                            with (value570 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value570), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd170 = (fd)[j]) {
                                              await yielded_fd170;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log47 EXCEPT ![i] = ((log47)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value471 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value471)];
                                          either {
                                            with (value571 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value571), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd171 = (fd)[j]) {
                                              await yielded_fd171;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((m).mtype) = (AppendEntriesResponse)) {
                    if (((m).mterm) > ((currentTerm)[self])) {
                      currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                      state := [state EXCEPT ![self] = Follower];
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      leader := [leader EXCEPT ![self] = Nil];
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  } else {
                    if (((m).mtype) = (ProposeMessage)) {
                      with (i = self) {
                        if (Debug) {
                          print <<"HandleProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                          if (((state)[i]) = (Leader)) {
                            with (entry = [term |-> (currentTerm)[i], cmd |-> (m).mcmd]) {
                              log := [log EXCEPT ![i] = Append((log)[i], entry)];
                              with (value60 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                                if (((value60).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value60).entries)];
                                  with (value70 = TRUE) {
                                    await (Len((appendEntriesCh)[i])) < (BufferSize);
                                    appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value70)];
                                    goto serverLoop;
                                  };
                                } else {
                                  if (((value60).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value60).cnt))];
                                    with (value71 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value71)];
                                      goto serverLoop;
                                    };
                                  } else {
                                    with (value72 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value72)];
                                      goto serverLoop;
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((leader)[i]) # (Nil)) {
                              with (j = (leader)[i]) {
                                either {
                                  with (value80 = [mtype |-> ProposeMessage, mcmd |-> (m).mcmd, msource |-> i, mdest |-> j]) {
                                    await ((network)[j]).enabled;
                                    await (Len(((network)[j]).queue)) < (BufferSize);
                                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value80), enabled |-> ((network)[j]).enabled]];
                                    goto serverLoop;
                                  };
                                } or {
                                  with (yielded_fd20 = (fd)[j]) {
                                    await yielded_fd20;
                                    goto serverLoop;
                                  };
                                };
                              };
                            } else {
                              goto serverLoop;
                            };
                          };
                        } else {
                          if (((state)[i]) = (Leader)) {
                            with (entry = [term |-> (currentTerm)[i], cmd |-> (m).mcmd]) {
                              log := [log EXCEPT ![i] = Append((log)[i], entry)];
                              with (value61 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                                if (((value61).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value61).entries)];
                                  with (value73 = TRUE) {
                                    await (Len((appendEntriesCh)[i])) < (BufferSize);
                                    appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value73)];
                                    goto serverLoop;
                                  };
                                } else {
                                  if (((value61).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value61).cnt))];
                                    with (value74 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value74)];
                                      goto serverLoop;
                                    };
                                  } else {
                                    with (value75 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value75)];
                                      goto serverLoop;
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((leader)[i]) # (Nil)) {
                              with (j = (leader)[i]) {
                                either {
                                  with (value81 = [mtype |-> ProposeMessage, mcmd |-> (m).mcmd, msource |-> i, mdest |-> j]) {
                                    await ((network)[j]).enabled;
                                    await (Len(((network)[j]).queue)) < (BufferSize);
                                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value81), enabled |-> ((network)[j]).enabled]];
                                    goto serverLoop;
                                  };
                                } or {
                                  with (yielded_fd21 = (fd)[j]) {
                                    await yielded_fd21;
                                    goto serverLoop;
                                  };
                                };
                              };
                            } else {
                              goto serverLoop;
                            };
                          };
                        };
                      };
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              };
            };
          } else {
            if (((m).mtype) = (RequestVoteRequest)) {
              if (((m).mterm) > ((currentTerm)[self])) {
                currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                state := [state EXCEPT ![self] = Follower];
                with (votedFor4 = [votedFor EXCEPT ![self] = Nil]) {
                  leader := [leader EXCEPT ![self] = Nil];
                  with (
                    i = self, 
                    j = (m).msource, 
                    logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                    grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor4)[self]) \in ({Nil, j}))
                  ) {
                    assert ((m).mterm) <= ((currentTerm)[i]);
                    if (grant) {
                      votedFor := [votedFor4 EXCEPT ![i] = j];
                      either {
                        with (value62 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value62), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd22 = (fd)[j]) {
                          await yielded_fd22;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      either {
                        with (value63 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value63), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor4;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor4;
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd23 = (fd)[j]) {
                          await yielded_fd23;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor4;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor4;
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                };
              } else {
                with (
                  i = self, 
                  j = (m).msource, 
                  logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                  grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j}))
                ) {
                  assert ((m).mterm) <= ((currentTerm)[i]);
                  if (grant) {
                    votedFor := [votedFor EXCEPT ![i] = j];
                    either {
                      with (value64 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value64), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd24 = (fd)[j]) {
                        await yielded_fd24;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    either {
                      with (value65 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value65), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd25 = (fd)[j]) {
                        await yielded_fd25;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  };
                };
              };
            } else {
              if (((m).mtype) = (RequestVoteResponse)) {
                if (((m).mterm) > ((currentTerm)[self])) {
                  currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                  state := [state EXCEPT ![self] = Follower];
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  leader := [leader EXCEPT ![self] = Nil];
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value02 = TRUE) {
                            await (Len((becomeLeaderCh)[i])) < (BufferSize);
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value02)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value03 = TRUE) {
                            await (Len((becomeLeaderCh)[i])) < (BufferSize);
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value03)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                };
              } else {
                if (((m).mtype) = (AppendEntriesRequest)) {
                  if (((m).mterm) > ((currentTerm)[self])) {
                    currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                    with (state4 = [state EXCEPT ![self] = Follower]) {
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      with (
                        leader4 = [leader EXCEPT ![self] = Nil], 
                        i = self, 
                        j = (m).msource, 
                        logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                      ) {
                        assert ((m).mterm) <= ((currentTerm)[i]);
                        if (((m).mterm) = ((currentTerm)[i])) {
                          leader := [leader4 EXCEPT ![i] = (m).msource];
                          leaderTimeout := LeaderTimeoutReset;
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Candidate))) {
                            state := [state4 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value118 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value118), enabled |-> ((network)[j]).enabled]];
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd08 = (fd)[j]) {
                                  await yielded_fd08;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value212 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value212).cmd) = (LogConcat)) {
                                  with (
                                    plog40 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value212).entries)], 
                                    log48 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value324 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value324).cmd) = (LogConcat)) {
                                      plog := [plog40 EXCEPT ![i] = ((plog40)[i]) \o ((value324).entries)];
                                      log := [log48 EXCEPT ![i] = ((log48)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value472 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value472)];
                                        either {
                                          with (value572 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value572), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd172 = (fd)[j]) {
                                            await yielded_fd172;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value324).cmd) = (LogPop)) {
                                        plog := [plog40 EXCEPT ![i] = SubSeq((plog40)[i], 1, (Len((plog40)[i])) - ((value324).cnt))];
                                        log := [log48 EXCEPT ![i] = ((log48)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value473 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value473)];
                                          either {
                                            with (value573 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value573), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd173 = (fd)[j]) {
                                              await yielded_fd173;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log48 EXCEPT ![i] = ((log48)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value474 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value474)];
                                          either {
                                            with (value574 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value574), enabled |-> ((network)[j]).enabled]];
                                              plog := plog40;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd174 = (fd)[j]) {
                                              await yielded_fd174;
                                              plog := plog40;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value212).cmd) = (LogPop)) {
                                    with (
                                      plog41 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value212).cnt))], 
                                      log49 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value325 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value325).cmd) = (LogConcat)) {
                                        plog := [plog41 EXCEPT ![i] = ((plog41)[i]) \o ((value325).entries)];
                                        log := [log49 EXCEPT ![i] = ((log49)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value475 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value475)];
                                          either {
                                            with (value575 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value575), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd175 = (fd)[j]) {
                                              await yielded_fd175;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value325).cmd) = (LogPop)) {
                                          plog := [plog41 EXCEPT ![i] = SubSeq((plog41)[i], 1, (Len((plog41)[i])) - ((value325).cnt))];
                                          log := [log49 EXCEPT ![i] = ((log49)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value476 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value476)];
                                            either {
                                              with (value576 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value576), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd176 = (fd)[j]) {
                                                await yielded_fd176;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log49 EXCEPT ![i] = ((log49)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value477 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value477)];
                                            either {
                                              with (value577 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value577), enabled |-> ((network)[j]).enabled]];
                                                plog := plog41;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd177 = (fd)[j]) {
                                                await yielded_fd177;
                                                plog := plog41;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log50 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value326 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value326).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value326).entries)];
                                        log := [log50 EXCEPT ![i] = ((log50)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value478 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value478)];
                                          either {
                                            with (value578 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value578), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd178 = (fd)[j]) {
                                              await yielded_fd178;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value326).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value326).cnt))];
                                          log := [log50 EXCEPT ![i] = ((log50)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value479 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value479)];
                                            either {
                                              with (value579 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value579), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd179 = (fd)[j]) {
                                                await yielded_fd179;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log50 EXCEPT ![i] = ((log50)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value480 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value480)];
                                            either {
                                              with (value580 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value580), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd180 = (fd)[j]) {
                                                await yielded_fd180;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value119 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value119), enabled |-> ((network)[j]).enabled]];
                                  state := state4;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd09 = (fd)[j]) {
                                  await yielded_fd09;
                                  state := state4;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value213 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value213).cmd) = (LogConcat)) {
                                  with (
                                    plog42 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value213).entries)], 
                                    log51 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value327 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value327).cmd) = (LogConcat)) {
                                      plog := [plog42 EXCEPT ![i] = ((plog42)[i]) \o ((value327).entries)];
                                      log := [log51 EXCEPT ![i] = ((log51)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value481 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value481)];
                                        either {
                                          with (value581 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value581), enabled |-> ((network)[j]).enabled]];
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd181 = (fd)[j]) {
                                            await yielded_fd181;
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value327).cmd) = (LogPop)) {
                                        plog := [plog42 EXCEPT ![i] = SubSeq((plog42)[i], 1, (Len((plog42)[i])) - ((value327).cnt))];
                                        log := [log51 EXCEPT ![i] = ((log51)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value482 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value482)];
                                          either {
                                            with (value582 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value582), enabled |-> ((network)[j]).enabled]];
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd182 = (fd)[j]) {
                                              await yielded_fd182;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log51 EXCEPT ![i] = ((log51)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value483 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value483)];
                                          either {
                                            with (value583 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value583), enabled |-> ((network)[j]).enabled]];
                                              plog := plog42;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd183 = (fd)[j]) {
                                              await yielded_fd183;
                                              plog := plog42;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value213).cmd) = (LogPop)) {
                                    with (
                                      plog43 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value213).cnt))], 
                                      log52 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value328 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value328).cmd) = (LogConcat)) {
                                        plog := [plog43 EXCEPT ![i] = ((plog43)[i]) \o ((value328).entries)];
                                        log := [log52 EXCEPT ![i] = ((log52)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value484 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value484)];
                                          either {
                                            with (value584 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value584), enabled |-> ((network)[j]).enabled]];
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd184 = (fd)[j]) {
                                              await yielded_fd184;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value328).cmd) = (LogPop)) {
                                          plog := [plog43 EXCEPT ![i] = SubSeq((plog43)[i], 1, (Len((plog43)[i])) - ((value328).cnt))];
                                          log := [log52 EXCEPT ![i] = ((log52)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value485 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value485)];
                                            either {
                                              with (value585 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value585), enabled |-> ((network)[j]).enabled]];
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd185 = (fd)[j]) {
                                                await yielded_fd185;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log52 EXCEPT ![i] = ((log52)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value486 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value486)];
                                            either {
                                              with (value586 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value586), enabled |-> ((network)[j]).enabled]];
                                                plog := plog43;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd186 = (fd)[j]) {
                                                await yielded_fd186;
                                                plog := plog43;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log53 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value329 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value329).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value329).entries)];
                                        log := [log53 EXCEPT ![i] = ((log53)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value487 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value487)];
                                          either {
                                            with (value587 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value587), enabled |-> ((network)[j]).enabled]];
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd187 = (fd)[j]) {
                                              await yielded_fd187;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value329).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value329).cnt))];
                                          log := [log53 EXCEPT ![i] = ((log53)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value488 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value488)];
                                            either {
                                              with (value588 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value588), enabled |-> ((network)[j]).enabled]];
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd188 = (fd)[j]) {
                                                await yielded_fd188;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log53 EXCEPT ![i] = ((log53)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value489 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value489)];
                                            either {
                                              with (value589 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value589), enabled |-> ((network)[j]).enabled]];
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd189 = (fd)[j]) {
                                                await yielded_fd189;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Candidate))) {
                            state := [state4 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value120 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value120), enabled |-> ((network)[j]).enabled]];
                                  leader := leader4;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd010 = (fd)[j]) {
                                  await yielded_fd010;
                                  leader := leader4;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value214 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value214).cmd) = (LogConcat)) {
                                  with (
                                    plog44 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value214).entries)], 
                                    log54 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value330 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value330).cmd) = (LogConcat)) {
                                      plog := [plog44 EXCEPT ![i] = ((plog44)[i]) \o ((value330).entries)];
                                      log := [log54 EXCEPT ![i] = ((log54)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value490 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value490)];
                                        either {
                                          with (value590 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value590), enabled |-> ((network)[j]).enabled]];
                                            leader := leader4;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd190 = (fd)[j]) {
                                            await yielded_fd190;
                                            leader := leader4;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value330).cmd) = (LogPop)) {
                                        plog := [plog44 EXCEPT ![i] = SubSeq((plog44)[i], 1, (Len((plog44)[i])) - ((value330).cnt))];
                                        log := [log54 EXCEPT ![i] = ((log54)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value491 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value491)];
                                          either {
                                            with (value591 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value591), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd191 = (fd)[j]) {
                                              await yielded_fd191;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log54 EXCEPT ![i] = ((log54)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value492 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value492)];
                                          either {
                                            with (value592 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value592), enabled |-> ((network)[j]).enabled]];
                                              plog := plog44;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd192 = (fd)[j]) {
                                              await yielded_fd192;
                                              plog := plog44;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value214).cmd) = (LogPop)) {
                                    with (
                                      plog45 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value214).cnt))], 
                                      log55 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value331 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value331).cmd) = (LogConcat)) {
                                        plog := [plog45 EXCEPT ![i] = ((plog45)[i]) \o ((value331).entries)];
                                        log := [log55 EXCEPT ![i] = ((log55)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value493 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value493)];
                                          either {
                                            with (value593 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value593), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd193 = (fd)[j]) {
                                              await yielded_fd193;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value331).cmd) = (LogPop)) {
                                          plog := [plog45 EXCEPT ![i] = SubSeq((plog45)[i], 1, (Len((plog45)[i])) - ((value331).cnt))];
                                          log := [log55 EXCEPT ![i] = ((log55)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value494 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value494)];
                                            either {
                                              with (value594 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value594), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd194 = (fd)[j]) {
                                                await yielded_fd194;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log55 EXCEPT ![i] = ((log55)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value495 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value495)];
                                            either {
                                              with (value595 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value595), enabled |-> ((network)[j]).enabled]];
                                                plog := plog45;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd195 = (fd)[j]) {
                                                await yielded_fd195;
                                                plog := plog45;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log56 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value332 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value332).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value332).entries)];
                                        log := [log56 EXCEPT ![i] = ((log56)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value496 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value496)];
                                          either {
                                            with (value596 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value596), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd196 = (fd)[j]) {
                                              await yielded_fd196;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value332).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value332).cnt))];
                                          log := [log56 EXCEPT ![i] = ((log56)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value497 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value497)];
                                            either {
                                              with (value597 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value597), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd197 = (fd)[j]) {
                                                await yielded_fd197;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log56 EXCEPT ![i] = ((log56)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value498 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value498)];
                                            either {
                                              with (value598 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value598), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd198 = (fd)[j]) {
                                                await yielded_fd198;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value121 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value121), enabled |-> ((network)[j]).enabled]];
                                  leader := leader4;
                                  state := state4;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd011 = (fd)[j]) {
                                  await yielded_fd011;
                                  leader := leader4;
                                  state := state4;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value215 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value215).cmd) = (LogConcat)) {
                                  with (
                                    plog46 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value215).entries)], 
                                    log57 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value333 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value333).cmd) = (LogConcat)) {
                                      plog := [plog46 EXCEPT ![i] = ((plog46)[i]) \o ((value333).entries)];
                                      log := [log57 EXCEPT ![i] = ((log57)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value499 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value499)];
                                        either {
                                          with (value599 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value599), enabled |-> ((network)[j]).enabled]];
                                            leader := leader4;
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd199 = (fd)[j]) {
                                            await yielded_fd199;
                                            leader := leader4;
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value333).cmd) = (LogPop)) {
                                        plog := [plog46 EXCEPT ![i] = SubSeq((plog46)[i], 1, (Len((plog46)[i])) - ((value333).cnt))];
                                        log := [log57 EXCEPT ![i] = ((log57)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4100 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4100)];
                                          either {
                                            with (value5100 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5100), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1100 = (fd)[j]) {
                                              await yielded_fd1100;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log57 EXCEPT ![i] = ((log57)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4101 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4101)];
                                          either {
                                            with (value5101 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5101), enabled |-> ((network)[j]).enabled]];
                                              plog := plog46;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1101 = (fd)[j]) {
                                              await yielded_fd1101;
                                              plog := plog46;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value215).cmd) = (LogPop)) {
                                    with (
                                      plog47 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value215).cnt))], 
                                      log58 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value334 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value334).cmd) = (LogConcat)) {
                                        plog := [plog47 EXCEPT ![i] = ((plog47)[i]) \o ((value334).entries)];
                                        log := [log58 EXCEPT ![i] = ((log58)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4102 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4102)];
                                          either {
                                            with (value5102 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5102), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1102 = (fd)[j]) {
                                              await yielded_fd1102;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value334).cmd) = (LogPop)) {
                                          plog := [plog47 EXCEPT ![i] = SubSeq((plog47)[i], 1, (Len((plog47)[i])) - ((value334).cnt))];
                                          log := [log58 EXCEPT ![i] = ((log58)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value4103 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4103)];
                                            either {
                                              with (value5103 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5103), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1103 = (fd)[j]) {
                                                await yielded_fd1103;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log58 EXCEPT ![i] = ((log58)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value4104 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4104)];
                                            either {
                                              with (value5104 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5104), enabled |-> ((network)[j]).enabled]];
                                                plog := plog47;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1104 = (fd)[j]) {
                                                await yielded_fd1104;
                                                plog := plog47;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log59 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value335 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value335).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value335).entries)];
                                        log := [log59 EXCEPT ![i] = ((log59)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4105 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4105)];
                                          either {
                                            with (value5105 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5105), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1105 = (fd)[j]) {
                                              await yielded_fd1105;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value335).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value335).cnt))];
                                          log := [log59 EXCEPT ![i] = ((log59)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value4106 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4106)];
                                            either {
                                              with (value5106 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5106), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1106 = (fd)[j]) {
                                                await yielded_fd1106;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log59 EXCEPT ![i] = ((log59)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (value4107 = m) {
                                            await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4107)];
                                            either {
                                              with (value5107 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5107), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1107 = (fd)[j]) {
                                                await yielded_fd1107;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  } else {
                    with (
                      i = self, 
                      j = (m).msource, 
                      logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                    ) {
                      assert ((m).mterm) <= ((currentTerm)[i]);
                      if (((m).mterm) = ((currentTerm)[i])) {
                        leader := [leader EXCEPT ![i] = (m).msource];
                        leaderTimeout := LeaderTimeoutReset;
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value122 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value122), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd012 = (fd)[j]) {
                                await yielded_fd012;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value216 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value216).cmd) = (LogConcat)) {
                                with (
                                  plog48 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value216).entries)], 
                                  log60 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value336 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value336).cmd) = (LogConcat)) {
                                    plog := [plog48 EXCEPT ![i] = ((plog48)[i]) \o ((value336).entries)];
                                    log := [log60 EXCEPT ![i] = ((log60)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4108 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4108)];
                                      either {
                                        with (value5108 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5108), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1108 = (fd)[j]) {
                                          await yielded_fd1108;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value336).cmd) = (LogPop)) {
                                      plog := [plog48 EXCEPT ![i] = SubSeq((plog48)[i], 1, (Len((plog48)[i])) - ((value336).cnt))];
                                      log := [log60 EXCEPT ![i] = ((log60)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4109 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4109)];
                                        either {
                                          with (value5109 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5109), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1109 = (fd)[j]) {
                                            await yielded_fd1109;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log60 EXCEPT ![i] = ((log60)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4110 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4110)];
                                        either {
                                          with (value5110 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5110), enabled |-> ((network)[j]).enabled]];
                                            plog := plog48;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1110 = (fd)[j]) {
                                            await yielded_fd1110;
                                            plog := plog48;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value216).cmd) = (LogPop)) {
                                  with (
                                    plog49 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value216).cnt))], 
                                    log61 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value337 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value337).cmd) = (LogConcat)) {
                                      plog := [plog49 EXCEPT ![i] = ((plog49)[i]) \o ((value337).entries)];
                                      log := [log61 EXCEPT ![i] = ((log61)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4111 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4111)];
                                        either {
                                          with (value5111 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5111), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1111 = (fd)[j]) {
                                            await yielded_fd1111;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value337).cmd) = (LogPop)) {
                                        plog := [plog49 EXCEPT ![i] = SubSeq((plog49)[i], 1, (Len((plog49)[i])) - ((value337).cnt))];
                                        log := [log61 EXCEPT ![i] = ((log61)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4112 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4112)];
                                          either {
                                            with (value5112 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5112), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1112 = (fd)[j]) {
                                              await yielded_fd1112;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log61 EXCEPT ![i] = ((log61)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4113 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4113)];
                                          either {
                                            with (value5113 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5113), enabled |-> ((network)[j]).enabled]];
                                              plog := plog49;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1113 = (fd)[j]) {
                                              await yielded_fd1113;
                                              plog := plog49;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log62 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value338 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value338).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value338).entries)];
                                      log := [log62 EXCEPT ![i] = ((log62)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4114 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4114)];
                                        either {
                                          with (value5114 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5114), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1114 = (fd)[j]) {
                                            await yielded_fd1114;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value338).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value338).cnt))];
                                        log := [log62 EXCEPT ![i] = ((log62)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4115 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4115)];
                                          either {
                                            with (value5115 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5115), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1115 = (fd)[j]) {
                                              await yielded_fd1115;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log62 EXCEPT ![i] = ((log62)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4116 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4116)];
                                          either {
                                            with (value5116 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5116), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1116 = (fd)[j]) {
                                              await yielded_fd1116;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value123 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value123), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd013 = (fd)[j]) {
                                await yielded_fd013;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value217 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value217).cmd) = (LogConcat)) {
                                with (
                                  plog50 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value217).entries)], 
                                  log63 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value339 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value339).cmd) = (LogConcat)) {
                                    plog := [plog50 EXCEPT ![i] = ((plog50)[i]) \o ((value339).entries)];
                                    log := [log63 EXCEPT ![i] = ((log63)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4117 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4117)];
                                      either {
                                        with (value5117 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5117), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1117 = (fd)[j]) {
                                          await yielded_fd1117;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value339).cmd) = (LogPop)) {
                                      plog := [plog50 EXCEPT ![i] = SubSeq((plog50)[i], 1, (Len((plog50)[i])) - ((value339).cnt))];
                                      log := [log63 EXCEPT ![i] = ((log63)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4118 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4118)];
                                        either {
                                          with (value5118 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5118), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1118 = (fd)[j]) {
                                            await yielded_fd1118;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log63 EXCEPT ![i] = ((log63)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4119 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4119)];
                                        either {
                                          with (value5119 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5119), enabled |-> ((network)[j]).enabled]];
                                            plog := plog50;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1119 = (fd)[j]) {
                                            await yielded_fd1119;
                                            plog := plog50;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value217).cmd) = (LogPop)) {
                                  with (
                                    plog51 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value217).cnt))], 
                                    log64 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value340 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value340).cmd) = (LogConcat)) {
                                      plog := [plog51 EXCEPT ![i] = ((plog51)[i]) \o ((value340).entries)];
                                      log := [log64 EXCEPT ![i] = ((log64)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4120 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4120)];
                                        either {
                                          with (value5120 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5120), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1120 = (fd)[j]) {
                                            await yielded_fd1120;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value340).cmd) = (LogPop)) {
                                        plog := [plog51 EXCEPT ![i] = SubSeq((plog51)[i], 1, (Len((plog51)[i])) - ((value340).cnt))];
                                        log := [log64 EXCEPT ![i] = ((log64)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4121 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4121)];
                                          either {
                                            with (value5121 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5121), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1121 = (fd)[j]) {
                                              await yielded_fd1121;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log64 EXCEPT ![i] = ((log64)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4122 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4122)];
                                          either {
                                            with (value5122 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5122), enabled |-> ((network)[j]).enabled]];
                                              plog := plog51;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1122 = (fd)[j]) {
                                              await yielded_fd1122;
                                              plog := plog51;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log65 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value341 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value341).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value341).entries)];
                                      log := [log65 EXCEPT ![i] = ((log65)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4123 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4123)];
                                        either {
                                          with (value5123 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5123), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1123 = (fd)[j]) {
                                            await yielded_fd1123;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value341).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value341).cnt))];
                                        log := [log65 EXCEPT ![i] = ((log65)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4124 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4124)];
                                          either {
                                            with (value5124 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5124), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1124 = (fd)[j]) {
                                              await yielded_fd1124;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log65 EXCEPT ![i] = ((log65)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4125 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4125)];
                                          either {
                                            with (value5125 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5125), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1125 = (fd)[j]) {
                                              await yielded_fd1125;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value124 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value124), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd014 = (fd)[j]) {
                                await yielded_fd014;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value218 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value218).cmd) = (LogConcat)) {
                                with (
                                  plog52 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value218).entries)], 
                                  log66 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value342 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value342).cmd) = (LogConcat)) {
                                    plog := [plog52 EXCEPT ![i] = ((plog52)[i]) \o ((value342).entries)];
                                    log := [log66 EXCEPT ![i] = ((log66)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4126 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4126)];
                                      either {
                                        with (value5126 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5126), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1126 = (fd)[j]) {
                                          await yielded_fd1126;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value342).cmd) = (LogPop)) {
                                      plog := [plog52 EXCEPT ![i] = SubSeq((plog52)[i], 1, (Len((plog52)[i])) - ((value342).cnt))];
                                      log := [log66 EXCEPT ![i] = ((log66)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4127 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4127)];
                                        either {
                                          with (value5127 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5127), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1127 = (fd)[j]) {
                                            await yielded_fd1127;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log66 EXCEPT ![i] = ((log66)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4128 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4128)];
                                        either {
                                          with (value5128 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5128), enabled |-> ((network)[j]).enabled]];
                                            plog := plog52;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1128 = (fd)[j]) {
                                            await yielded_fd1128;
                                            plog := plog52;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value218).cmd) = (LogPop)) {
                                  with (
                                    plog53 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value218).cnt))], 
                                    log67 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value343 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value343).cmd) = (LogConcat)) {
                                      plog := [plog53 EXCEPT ![i] = ((plog53)[i]) \o ((value343).entries)];
                                      log := [log67 EXCEPT ![i] = ((log67)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4129 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4129)];
                                        either {
                                          with (value5129 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5129), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1129 = (fd)[j]) {
                                            await yielded_fd1129;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value343).cmd) = (LogPop)) {
                                        plog := [plog53 EXCEPT ![i] = SubSeq((plog53)[i], 1, (Len((plog53)[i])) - ((value343).cnt))];
                                        log := [log67 EXCEPT ![i] = ((log67)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4130 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4130)];
                                          either {
                                            with (value5130 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5130), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1130 = (fd)[j]) {
                                              await yielded_fd1130;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log67 EXCEPT ![i] = ((log67)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4131 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4131)];
                                          either {
                                            with (value5131 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5131), enabled |-> ((network)[j]).enabled]];
                                              plog := plog53;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1131 = (fd)[j]) {
                                              await yielded_fd1131;
                                              plog := plog53;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log68 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value344 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value344).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value344).entries)];
                                      log := [log68 EXCEPT ![i] = ((log68)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4132 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4132)];
                                        either {
                                          with (value5132 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5132), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1132 = (fd)[j]) {
                                            await yielded_fd1132;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value344).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value344).cnt))];
                                        log := [log68 EXCEPT ![i] = ((log68)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4133 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4133)];
                                          either {
                                            with (value5133 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5133), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1133 = (fd)[j]) {
                                              await yielded_fd1133;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log68 EXCEPT ![i] = ((log68)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4134 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4134)];
                                          either {
                                            with (value5134 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5134), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1134 = (fd)[j]) {
                                              await yielded_fd1134;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value125 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value125), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd015 = (fd)[j]) {
                                await yielded_fd015;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value219 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value219).cmd) = (LogConcat)) {
                                with (
                                  plog54 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value219).entries)], 
                                  log69 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value345 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value345).cmd) = (LogConcat)) {
                                    plog := [plog54 EXCEPT ![i] = ((plog54)[i]) \o ((value345).entries)];
                                    log := [log69 EXCEPT ![i] = ((log69)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4135 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4135)];
                                      either {
                                        with (value5135 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5135), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1135 = (fd)[j]) {
                                          await yielded_fd1135;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value345).cmd) = (LogPop)) {
                                      plog := [plog54 EXCEPT ![i] = SubSeq((plog54)[i], 1, (Len((plog54)[i])) - ((value345).cnt))];
                                      log := [log69 EXCEPT ![i] = ((log69)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4136 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4136)];
                                        either {
                                          with (value5136 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5136), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1136 = (fd)[j]) {
                                            await yielded_fd1136;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log69 EXCEPT ![i] = ((log69)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4137 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4137)];
                                        either {
                                          with (value5137 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5137), enabled |-> ((network)[j]).enabled]];
                                            plog := plog54;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1137 = (fd)[j]) {
                                            await yielded_fd1137;
                                            plog := plog54;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value219).cmd) = (LogPop)) {
                                  with (
                                    plog55 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value219).cnt))], 
                                    log70 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value346 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value346).cmd) = (LogConcat)) {
                                      plog := [plog55 EXCEPT ![i] = ((plog55)[i]) \o ((value346).entries)];
                                      log := [log70 EXCEPT ![i] = ((log70)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4138 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4138)];
                                        either {
                                          with (value5138 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5138), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1138 = (fd)[j]) {
                                            await yielded_fd1138;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value346).cmd) = (LogPop)) {
                                        plog := [plog55 EXCEPT ![i] = SubSeq((plog55)[i], 1, (Len((plog55)[i])) - ((value346).cnt))];
                                        log := [log70 EXCEPT ![i] = ((log70)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4139 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4139)];
                                          either {
                                            with (value5139 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5139), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1139 = (fd)[j]) {
                                              await yielded_fd1139;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log70 EXCEPT ![i] = ((log70)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4140 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4140)];
                                          either {
                                            with (value5140 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5140), enabled |-> ((network)[j]).enabled]];
                                              plog := plog55;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1140 = (fd)[j]) {
                                              await yielded_fd1140;
                                              plog := plog55;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log71 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value347 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value347).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value347).entries)];
                                      log := [log71 EXCEPT ![i] = ((log71)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4141 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4141)];
                                        either {
                                          with (value5141 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5141), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1141 = (fd)[j]) {
                                            await yielded_fd1141;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value347).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value347).cnt))];
                                        log := [log71 EXCEPT ![i] = ((log71)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4142 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4142)];
                                          either {
                                            with (value5142 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5142), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1142 = (fd)[j]) {
                                              await yielded_fd1142;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log71 EXCEPT ![i] = ((log71)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (value4143 = m) {
                                          await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                          fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4143)];
                                          either {
                                            with (value5143 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5143), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1143 = (fd)[j]) {
                                              await yielded_fd1143;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((m).mtype) = (AppendEntriesResponse)) {
                    if (((m).mterm) > ((currentTerm)[self])) {
                      currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                      state := [state EXCEPT ![self] = Follower];
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      leader := [leader EXCEPT ![self] = Nil];
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  } else {
                    if (((m).mtype) = (ProposeMessage)) {
                      with (i = self) {
                        if (Debug) {
                          print <<"HandleProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                          if (((state)[i]) = (Leader)) {
                            with (entry = [term |-> (currentTerm)[i], cmd |-> (m).mcmd]) {
                              log := [log EXCEPT ![i] = Append((log)[i], entry)];
                              with (value66 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                                if (((value66).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value66).entries)];
                                  with (value76 = TRUE) {
                                    await (Len((appendEntriesCh)[i])) < (BufferSize);
                                    appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value76)];
                                    goto serverLoop;
                                  };
                                } else {
                                  if (((value66).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value66).cnt))];
                                    with (value77 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value77)];
                                      goto serverLoop;
                                    };
                                  } else {
                                    with (value78 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value78)];
                                      goto serverLoop;
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((leader)[i]) # (Nil)) {
                              with (j = (leader)[i]) {
                                either {
                                  with (value82 = [mtype |-> ProposeMessage, mcmd |-> (m).mcmd, msource |-> i, mdest |-> j]) {
                                    await ((network)[j]).enabled;
                                    await (Len(((network)[j]).queue)) < (BufferSize);
                                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value82), enabled |-> ((network)[j]).enabled]];
                                    goto serverLoop;
                                  };
                                } or {
                                  with (yielded_fd26 = (fd)[j]) {
                                    await yielded_fd26;
                                    goto serverLoop;
                                  };
                                };
                              };
                            } else {
                              goto serverLoop;
                            };
                          };
                        } else {
                          if (((state)[i]) = (Leader)) {
                            with (entry = [term |-> (currentTerm)[i], cmd |-> (m).mcmd]) {
                              log := [log EXCEPT ![i] = Append((log)[i], entry)];
                              with (value67 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                                if (((value67).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value67).entries)];
                                  with (value79 = TRUE) {
                                    await (Len((appendEntriesCh)[i])) < (BufferSize);
                                    appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value79)];
                                    goto serverLoop;
                                  };
                                } else {
                                  if (((value67).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value67).cnt))];
                                    with (value710 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value710)];
                                      goto serverLoop;
                                    };
                                  } else {
                                    with (value711 = TRUE) {
                                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value711)];
                                      goto serverLoop;
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((leader)[i]) # (Nil)) {
                              with (j = (leader)[i]) {
                                either {
                                  with (value83 = [mtype |-> ProposeMessage, mcmd |-> (m).mcmd, msource |-> i, mdest |-> j]) {
                                    await ((network)[j]).enabled;
                                    await (Len(((network)[j]).queue)) < (BufferSize);
                                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value83), enabled |-> ((network)[j]).enabled]];
                                    goto serverLoop;
                                  };
                                } or {
                                  with (yielded_fd27 = (fd)[j]) {
                                    await yielded_fd27;
                                    goto serverLoop;
                                  };
                                };
                              };
                            } else {
                              goto serverLoop;
                            };
                          };
                        };
                      };
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              };
            };
          };
        };
      } else {
        if (((m).mtype) = (RequestVoteRequest)) {
          if (((m).mterm) > ((currentTerm)[self])) {
            currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
            state := [state EXCEPT ![self] = Follower];
            with (votedFor5 = [votedFor EXCEPT ![self] = Nil]) {
              leader := [leader EXCEPT ![self] = Nil];
              with (
                i = self, 
                j = (m).msource, 
                logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor5)[self]) \in ({Nil, j}))
              ) {
                assert ((m).mterm) <= ((currentTerm)[i]);
                if (grant) {
                  votedFor := [votedFor5 EXCEPT ![i] = j];
                  either {
                    with (value68 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                      await ((network)[j]).enabled;
                      await (Len(((network)[j]).queue)) < (BufferSize);
                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value68), enabled |-> ((network)[j]).enabled]];
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        goto serverLoop;
                      } else {
                        goto serverLoop;
                      };
                    };
                  } or {
                    with (yielded_fd28 = (fd)[j]) {
                      await yielded_fd28;
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        goto serverLoop;
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  either {
                    with (value69 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                      await ((network)[j]).enabled;
                      await (Len(((network)[j]).queue)) < (BufferSize);
                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value69), enabled |-> ((network)[j]).enabled]];
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        votedFor := votedFor5;
                        goto serverLoop;
                      } else {
                        votedFor := votedFor5;
                        goto serverLoop;
                      };
                    };
                  } or {
                    with (yielded_fd29 = (fd)[j]) {
                      await yielded_fd29;
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        votedFor := votedFor5;
                        goto serverLoop;
                      } else {
                        votedFor := votedFor5;
                        goto serverLoop;
                      };
                    };
                  };
                };
              };
            };
          } else {
            with (
              i = self, 
              j = (m).msource, 
              logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
              grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j}))
            ) {
              assert ((m).mterm) <= ((currentTerm)[i]);
              if (grant) {
                votedFor := [votedFor EXCEPT ![i] = j];
                either {
                  with (value84 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                    await ((network)[j]).enabled;
                    await (Len(((network)[j]).queue)) < (BufferSize);
                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value84), enabled |-> ((network)[j]).enabled]];
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                } or {
                  with (yielded_fd30 = (fd)[j]) {
                    await yielded_fd30;
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              } else {
                either {
                  with (value85 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                    await ((network)[j]).enabled;
                    await (Len(((network)[j]).queue)) < (BufferSize);
                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value85), enabled |-> ((network)[j]).enabled]];
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                } or {
                  with (yielded_fd31 = (fd)[j]) {
                    await yielded_fd31;
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              };
            };
          };
        } else {
          if (((m).mtype) = (RequestVoteResponse)) {
            if (((m).mterm) > ((currentTerm)[self])) {
              currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
              state := [state EXCEPT ![self] = Follower];
              votedFor := [votedFor EXCEPT ![self] = Nil];
              leader := [leader EXCEPT ![self] = Nil];
              if (((m).mterm) < ((currentTerm)[self])) {
                skip;
                goto serverLoop;
              } else {
                with (
                  i = self, 
                  j = (m).msource
                ) {
                  assert ((m).mterm) = ((currentTerm)[i]);
                  votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                  if ((m).mvoteGranted) {
                    votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                    if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                      with (value04 = TRUE) {
                        await (Len((becomeLeaderCh)[i])) < (BufferSize);
                        becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value04)];
                        goto serverLoop;
                      };
                    } else {
                      goto serverLoop;
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              };
            } else {
              if (((m).mterm) < ((currentTerm)[self])) {
                skip;
                goto serverLoop;
              } else {
                with (
                  i = self, 
                  j = (m).msource
                ) {
                  assert ((m).mterm) = ((currentTerm)[i]);
                  votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                  if ((m).mvoteGranted) {
                    votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                    if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                      with (value05 = TRUE) {
                        await (Len((becomeLeaderCh)[i])) < (BufferSize);
                        becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value05)];
                        goto serverLoop;
                      };
                    } else {
                      goto serverLoop;
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              };
            };
          } else {
            if (((m).mtype) = (AppendEntriesRequest)) {
              if (((m).mterm) > ((currentTerm)[self])) {
                currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                with (state5 = [state EXCEPT ![self] = Follower]) {
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  with (
                    leader5 = [leader EXCEPT ![self] = Nil], 
                    i = self, 
                    j = (m).msource, 
                    logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                  ) {
                    assert ((m).mterm) <= ((currentTerm)[i]);
                    if (((m).mterm) = ((currentTerm)[i])) {
                      leader := [leader5 EXCEPT ![i] = (m).msource];
                      leaderTimeout := LeaderTimeoutReset;
                      if ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Candidate))) {
                        state := [state5 EXCEPT ![i] = Follower];
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value126 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value126), enabled |-> ((network)[j]).enabled]];
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd016 = (fd)[j]) {
                              await yielded_fd016;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value220 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value220).cmd) = (LogConcat)) {
                              with (
                                plog56 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value220).entries)], 
                                log72 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value348 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value348).cmd) = (LogConcat)) {
                                  plog := [plog56 EXCEPT ![i] = ((plog56)[i]) \o ((value348).entries)];
                                  log := [log72 EXCEPT ![i] = ((log72)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4144 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4144)];
                                    either {
                                      with (value5144 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5144), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1144 = (fd)[j]) {
                                        await yielded_fd1144;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value348).cmd) = (LogPop)) {
                                    plog := [plog56 EXCEPT ![i] = SubSeq((plog56)[i], 1, (Len((plog56)[i])) - ((value348).cnt))];
                                    log := [log72 EXCEPT ![i] = ((log72)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4145 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4145)];
                                      either {
                                        with (value5145 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5145), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1145 = (fd)[j]) {
                                          await yielded_fd1145;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log72 EXCEPT ![i] = ((log72)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4146 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4146)];
                                      either {
                                        with (value5146 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5146), enabled |-> ((network)[j]).enabled]];
                                          plog := plog56;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1146 = (fd)[j]) {
                                          await yielded_fd1146;
                                          plog := plog56;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value220).cmd) = (LogPop)) {
                                with (
                                  plog57 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value220).cnt))], 
                                  log73 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value349 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value349).cmd) = (LogConcat)) {
                                    plog := [plog57 EXCEPT ![i] = ((plog57)[i]) \o ((value349).entries)];
                                    log := [log73 EXCEPT ![i] = ((log73)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4147 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4147)];
                                      either {
                                        with (value5147 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5147), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1147 = (fd)[j]) {
                                          await yielded_fd1147;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value349).cmd) = (LogPop)) {
                                      plog := [plog57 EXCEPT ![i] = SubSeq((plog57)[i], 1, (Len((plog57)[i])) - ((value349).cnt))];
                                      log := [log73 EXCEPT ![i] = ((log73)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4148 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4148)];
                                        either {
                                          with (value5148 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5148), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1148 = (fd)[j]) {
                                            await yielded_fd1148;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log73 EXCEPT ![i] = ((log73)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4149 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4149)];
                                        either {
                                          with (value5149 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5149), enabled |-> ((network)[j]).enabled]];
                                            plog := plog57;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1149 = (fd)[j]) {
                                            await yielded_fd1149;
                                            plog := plog57;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log74 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value350 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value350).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value350).entries)];
                                    log := [log74 EXCEPT ![i] = ((log74)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4150 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4150)];
                                      either {
                                        with (value5150 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5150), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1150 = (fd)[j]) {
                                          await yielded_fd1150;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value350).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value350).cnt))];
                                      log := [log74 EXCEPT ![i] = ((log74)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4151 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4151)];
                                        either {
                                          with (value5151 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5151), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1151 = (fd)[j]) {
                                            await yielded_fd1151;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log74 EXCEPT ![i] = ((log74)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4152 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4152)];
                                        either {
                                          with (value5152 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5152), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1152 = (fd)[j]) {
                                            await yielded_fd1152;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value127 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value127), enabled |-> ((network)[j]).enabled]];
                              state := state5;
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd017 = (fd)[j]) {
                              await yielded_fd017;
                              state := state5;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value221 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value221).cmd) = (LogConcat)) {
                              with (
                                plog58 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value221).entries)], 
                                log75 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value351 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value351).cmd) = (LogConcat)) {
                                  plog := [plog58 EXCEPT ![i] = ((plog58)[i]) \o ((value351).entries)];
                                  log := [log75 EXCEPT ![i] = ((log75)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4153 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4153)];
                                    either {
                                      with (value5153 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5153), enabled |-> ((network)[j]).enabled]];
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1153 = (fd)[j]) {
                                        await yielded_fd1153;
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value351).cmd) = (LogPop)) {
                                    plog := [plog58 EXCEPT ![i] = SubSeq((plog58)[i], 1, (Len((plog58)[i])) - ((value351).cnt))];
                                    log := [log75 EXCEPT ![i] = ((log75)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4154 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4154)];
                                      either {
                                        with (value5154 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5154), enabled |-> ((network)[j]).enabled]];
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1154 = (fd)[j]) {
                                          await yielded_fd1154;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log75 EXCEPT ![i] = ((log75)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4155 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4155)];
                                      either {
                                        with (value5155 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5155), enabled |-> ((network)[j]).enabled]];
                                          plog := plog58;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1155 = (fd)[j]) {
                                          await yielded_fd1155;
                                          plog := plog58;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value221).cmd) = (LogPop)) {
                                with (
                                  plog59 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value221).cnt))], 
                                  log76 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value352 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value352).cmd) = (LogConcat)) {
                                    plog := [plog59 EXCEPT ![i] = ((plog59)[i]) \o ((value352).entries)];
                                    log := [log76 EXCEPT ![i] = ((log76)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4156 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4156)];
                                      either {
                                        with (value5156 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5156), enabled |-> ((network)[j]).enabled]];
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1156 = (fd)[j]) {
                                          await yielded_fd1156;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value352).cmd) = (LogPop)) {
                                      plog := [plog59 EXCEPT ![i] = SubSeq((plog59)[i], 1, (Len((plog59)[i])) - ((value352).cnt))];
                                      log := [log76 EXCEPT ![i] = ((log76)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4157 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4157)];
                                        either {
                                          with (value5157 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5157), enabled |-> ((network)[j]).enabled]];
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1157 = (fd)[j]) {
                                            await yielded_fd1157;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log76 EXCEPT ![i] = ((log76)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4158 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4158)];
                                        either {
                                          with (value5158 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5158), enabled |-> ((network)[j]).enabled]];
                                            plog := plog59;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1158 = (fd)[j]) {
                                            await yielded_fd1158;
                                            plog := plog59;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log77 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value353 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value353).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value353).entries)];
                                    log := [log77 EXCEPT ![i] = ((log77)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4159 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4159)];
                                      either {
                                        with (value5159 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5159), enabled |-> ((network)[j]).enabled]];
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1159 = (fd)[j]) {
                                          await yielded_fd1159;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value353).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value353).cnt))];
                                      log := [log77 EXCEPT ![i] = ((log77)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4160 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4160)];
                                        either {
                                          with (value5160 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5160), enabled |-> ((network)[j]).enabled]];
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1160 = (fd)[j]) {
                                            await yielded_fd1160;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log77 EXCEPT ![i] = ((log77)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4161 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4161)];
                                        either {
                                          with (value5161 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5161), enabled |-> ((network)[j]).enabled]];
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1161 = (fd)[j]) {
                                            await yielded_fd1161;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    } else {
                      if ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Candidate))) {
                        state := [state5 EXCEPT ![i] = Follower];
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value128 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value128), enabled |-> ((network)[j]).enabled]];
                              leader := leader5;
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd018 = (fd)[j]) {
                              await yielded_fd018;
                              leader := leader5;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value222 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value222).cmd) = (LogConcat)) {
                              with (
                                plog60 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value222).entries)], 
                                log78 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value354 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value354).cmd) = (LogConcat)) {
                                  plog := [plog60 EXCEPT ![i] = ((plog60)[i]) \o ((value354).entries)];
                                  log := [log78 EXCEPT ![i] = ((log78)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4162 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4162)];
                                    either {
                                      with (value5162 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5162), enabled |-> ((network)[j]).enabled]];
                                        leader := leader5;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1162 = (fd)[j]) {
                                        await yielded_fd1162;
                                        leader := leader5;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value354).cmd) = (LogPop)) {
                                    plog := [plog60 EXCEPT ![i] = SubSeq((plog60)[i], 1, (Len((plog60)[i])) - ((value354).cnt))];
                                    log := [log78 EXCEPT ![i] = ((log78)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4163 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4163)];
                                      either {
                                        with (value5163 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5163), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1163 = (fd)[j]) {
                                          await yielded_fd1163;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log78 EXCEPT ![i] = ((log78)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4164 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4164)];
                                      either {
                                        with (value5164 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5164), enabled |-> ((network)[j]).enabled]];
                                          plog := plog60;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1164 = (fd)[j]) {
                                          await yielded_fd1164;
                                          plog := plog60;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value222).cmd) = (LogPop)) {
                                with (
                                  plog61 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value222).cnt))], 
                                  log79 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value355 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value355).cmd) = (LogConcat)) {
                                    plog := [plog61 EXCEPT ![i] = ((plog61)[i]) \o ((value355).entries)];
                                    log := [log79 EXCEPT ![i] = ((log79)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4165 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4165)];
                                      either {
                                        with (value5165 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5165), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1165 = (fd)[j]) {
                                          await yielded_fd1165;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value355).cmd) = (LogPop)) {
                                      plog := [plog61 EXCEPT ![i] = SubSeq((plog61)[i], 1, (Len((plog61)[i])) - ((value355).cnt))];
                                      log := [log79 EXCEPT ![i] = ((log79)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4166 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4166)];
                                        either {
                                          with (value5166 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5166), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1166 = (fd)[j]) {
                                            await yielded_fd1166;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log79 EXCEPT ![i] = ((log79)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4167 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4167)];
                                        either {
                                          with (value5167 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5167), enabled |-> ((network)[j]).enabled]];
                                            plog := plog61;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1167 = (fd)[j]) {
                                            await yielded_fd1167;
                                            plog := plog61;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log80 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value356 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value356).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value356).entries)];
                                    log := [log80 EXCEPT ![i] = ((log80)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4168 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4168)];
                                      either {
                                        with (value5168 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5168), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1168 = (fd)[j]) {
                                          await yielded_fd1168;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value356).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value356).cnt))];
                                      log := [log80 EXCEPT ![i] = ((log80)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4169 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4169)];
                                        either {
                                          with (value5169 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5169), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1169 = (fd)[j]) {
                                            await yielded_fd1169;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log80 EXCEPT ![i] = ((log80)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4170 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4170)];
                                        either {
                                          with (value5170 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5170), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1170 = (fd)[j]) {
                                            await yielded_fd1170;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value129 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value129), enabled |-> ((network)[j]).enabled]];
                              leader := leader5;
                              state := state5;
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd019 = (fd)[j]) {
                              await yielded_fd019;
                              leader := leader5;
                              state := state5;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value223 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value223).cmd) = (LogConcat)) {
                              with (
                                plog62 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value223).entries)], 
                                log81 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value357 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value357).cmd) = (LogConcat)) {
                                  plog := [plog62 EXCEPT ![i] = ((plog62)[i]) \o ((value357).entries)];
                                  log := [log81 EXCEPT ![i] = ((log81)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4171 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4171)];
                                    either {
                                      with (value5171 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5171), enabled |-> ((network)[j]).enabled]];
                                        leader := leader5;
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1171 = (fd)[j]) {
                                        await yielded_fd1171;
                                        leader := leader5;
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value357).cmd) = (LogPop)) {
                                    plog := [plog62 EXCEPT ![i] = SubSeq((plog62)[i], 1, (Len((plog62)[i])) - ((value357).cnt))];
                                    log := [log81 EXCEPT ![i] = ((log81)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4172 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4172)];
                                      either {
                                        with (value5172 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5172), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1172 = (fd)[j]) {
                                          await yielded_fd1172;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log81 EXCEPT ![i] = ((log81)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4173 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4173)];
                                      either {
                                        with (value5173 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5173), enabled |-> ((network)[j]).enabled]];
                                          plog := plog62;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1173 = (fd)[j]) {
                                          await yielded_fd1173;
                                          plog := plog62;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value223).cmd) = (LogPop)) {
                                with (
                                  plog63 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value223).cnt))], 
                                  log82 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value358 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value358).cmd) = (LogConcat)) {
                                    plog := [plog63 EXCEPT ![i] = ((plog63)[i]) \o ((value358).entries)];
                                    log := [log82 EXCEPT ![i] = ((log82)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4174 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4174)];
                                      either {
                                        with (value5174 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5174), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1174 = (fd)[j]) {
                                          await yielded_fd1174;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value358).cmd) = (LogPop)) {
                                      plog := [plog63 EXCEPT ![i] = SubSeq((plog63)[i], 1, (Len((plog63)[i])) - ((value358).cnt))];
                                      log := [log82 EXCEPT ![i] = ((log82)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4175 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4175)];
                                        either {
                                          with (value5175 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5175), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1175 = (fd)[j]) {
                                            await yielded_fd1175;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log82 EXCEPT ![i] = ((log82)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4176 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4176)];
                                        either {
                                          with (value5176 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5176), enabled |-> ((network)[j]).enabled]];
                                            plog := plog63;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1176 = (fd)[j]) {
                                            await yielded_fd1176;
                                            plog := plog63;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log83 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value359 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value359).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value359).entries)];
                                    log := [log83 EXCEPT ![i] = ((log83)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4177 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4177)];
                                      either {
                                        with (value5177 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5177), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1177 = (fd)[j]) {
                                          await yielded_fd1177;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value359).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value359).cnt))];
                                      log := [log83 EXCEPT ![i] = ((log83)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4178 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4178)];
                                        either {
                                          with (value5178 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5178), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1178 = (fd)[j]) {
                                            await yielded_fd1178;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log83 EXCEPT ![i] = ((log83)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (value4179 = m) {
                                        await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                        fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4179)];
                                        either {
                                          with (value5179 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5179), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1179 = (fd)[j]) {
                                            await yielded_fd1179;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                };
              } else {
                with (
                  i = self, 
                  j = (m).msource, 
                  logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                ) {
                  assert ((m).mterm) <= ((currentTerm)[i]);
                  if (((m).mterm) = ((currentTerm)[i])) {
                    leader := [leader EXCEPT ![i] = (m).msource];
                    leaderTimeout := LeaderTimeoutReset;
                    if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                      state := [state EXCEPT ![i] = Follower];
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value130 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value130), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd020 = (fd)[j]) {
                            await yielded_fd020;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value224 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value224).cmd) = (LogConcat)) {
                            with (
                              plog64 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value224).entries)], 
                              log84 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value360 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value360).cmd) = (LogConcat)) {
                                plog := [plog64 EXCEPT ![i] = ((plog64)[i]) \o ((value360).entries)];
                                log := [log84 EXCEPT ![i] = ((log84)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (value4180 = m) {
                                  await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                  fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4180)];
                                  either {
                                    with (value5180 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5180), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1180 = (fd)[j]) {
                                      await yielded_fd1180;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value360).cmd) = (LogPop)) {
                                  plog := [plog64 EXCEPT ![i] = SubSeq((plog64)[i], 1, (Len((plog64)[i])) - ((value360).cnt))];
                                  log := [log84 EXCEPT ![i] = ((log84)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4181 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4181)];
                                    either {
                                      with (value5181 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5181), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1181 = (fd)[j]) {
                                        await yielded_fd1181;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log84 EXCEPT ![i] = ((log84)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4182 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4182)];
                                    either {
                                      with (value5182 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5182), enabled |-> ((network)[j]).enabled]];
                                        plog := plog64;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1182 = (fd)[j]) {
                                        await yielded_fd1182;
                                        plog := plog64;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value224).cmd) = (LogPop)) {
                              with (
                                plog65 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value224).cnt))], 
                                log85 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value361 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value361).cmd) = (LogConcat)) {
                                  plog := [plog65 EXCEPT ![i] = ((plog65)[i]) \o ((value361).entries)];
                                  log := [log85 EXCEPT ![i] = ((log85)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4183 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4183)];
                                    either {
                                      with (value5183 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5183), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1183 = (fd)[j]) {
                                        await yielded_fd1183;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value361).cmd) = (LogPop)) {
                                    plog := [plog65 EXCEPT ![i] = SubSeq((plog65)[i], 1, (Len((plog65)[i])) - ((value361).cnt))];
                                    log := [log85 EXCEPT ![i] = ((log85)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4184 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4184)];
                                      either {
                                        with (value5184 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5184), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1184 = (fd)[j]) {
                                          await yielded_fd1184;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log85 EXCEPT ![i] = ((log85)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4185 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4185)];
                                      either {
                                        with (value5185 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5185), enabled |-> ((network)[j]).enabled]];
                                          plog := plog65;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1185 = (fd)[j]) {
                                          await yielded_fd1185;
                                          plog := plog65;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log86 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value362 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value362).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value362).entries)];
                                  log := [log86 EXCEPT ![i] = ((log86)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4186 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4186)];
                                    either {
                                      with (value5186 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5186), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1186 = (fd)[j]) {
                                        await yielded_fd1186;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value362).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value362).cnt))];
                                    log := [log86 EXCEPT ![i] = ((log86)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4187 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4187)];
                                      either {
                                        with (value5187 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5187), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1187 = (fd)[j]) {
                                          await yielded_fd1187;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log86 EXCEPT ![i] = ((log86)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4188 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4188)];
                                      either {
                                        with (value5188 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5188), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1188 = (fd)[j]) {
                                          await yielded_fd1188;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    } else {
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value131 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value131), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd021 = (fd)[j]) {
                            await yielded_fd021;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value225 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value225).cmd) = (LogConcat)) {
                            with (
                              plog66 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value225).entries)], 
                              log87 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value363 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value363).cmd) = (LogConcat)) {
                                plog := [plog66 EXCEPT ![i] = ((plog66)[i]) \o ((value363).entries)];
                                log := [log87 EXCEPT ![i] = ((log87)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (value4189 = m) {
                                  await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                  fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4189)];
                                  either {
                                    with (value5189 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5189), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1189 = (fd)[j]) {
                                      await yielded_fd1189;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value363).cmd) = (LogPop)) {
                                  plog := [plog66 EXCEPT ![i] = SubSeq((plog66)[i], 1, (Len((plog66)[i])) - ((value363).cnt))];
                                  log := [log87 EXCEPT ![i] = ((log87)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4190 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4190)];
                                    either {
                                      with (value5190 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5190), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1190 = (fd)[j]) {
                                        await yielded_fd1190;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log87 EXCEPT ![i] = ((log87)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4191 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4191)];
                                    either {
                                      with (value5191 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5191), enabled |-> ((network)[j]).enabled]];
                                        plog := plog66;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1191 = (fd)[j]) {
                                        await yielded_fd1191;
                                        plog := plog66;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value225).cmd) = (LogPop)) {
                              with (
                                plog67 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value225).cnt))], 
                                log88 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value364 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value364).cmd) = (LogConcat)) {
                                  plog := [plog67 EXCEPT ![i] = ((plog67)[i]) \o ((value364).entries)];
                                  log := [log88 EXCEPT ![i] = ((log88)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4192 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4192)];
                                    either {
                                      with (value5192 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5192), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1192 = (fd)[j]) {
                                        await yielded_fd1192;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value364).cmd) = (LogPop)) {
                                    plog := [plog67 EXCEPT ![i] = SubSeq((plog67)[i], 1, (Len((plog67)[i])) - ((value364).cnt))];
                                    log := [log88 EXCEPT ![i] = ((log88)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4193 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4193)];
                                      either {
                                        with (value5193 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5193), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1193 = (fd)[j]) {
                                          await yielded_fd1193;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log88 EXCEPT ![i] = ((log88)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4194 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4194)];
                                      either {
                                        with (value5194 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5194), enabled |-> ((network)[j]).enabled]];
                                          plog := plog67;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1194 = (fd)[j]) {
                                          await yielded_fd1194;
                                          plog := plog67;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log89 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value365 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value365).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value365).entries)];
                                  log := [log89 EXCEPT ![i] = ((log89)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4195 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4195)];
                                    either {
                                      with (value5195 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5195), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1195 = (fd)[j]) {
                                        await yielded_fd1195;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value365).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value365).cnt))];
                                    log := [log89 EXCEPT ![i] = ((log89)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4196 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4196)];
                                      either {
                                        with (value5196 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5196), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1196 = (fd)[j]) {
                                          await yielded_fd1196;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log89 EXCEPT ![i] = ((log89)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4197 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4197)];
                                      either {
                                        with (value5197 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5197), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1197 = (fd)[j]) {
                                          await yielded_fd1197;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  } else {
                    if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                      state := [state EXCEPT ![i] = Follower];
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value132 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value132), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd022 = (fd)[j]) {
                            await yielded_fd022;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value226 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value226).cmd) = (LogConcat)) {
                            with (
                              plog68 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value226).entries)], 
                              log90 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value366 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value366).cmd) = (LogConcat)) {
                                plog := [plog68 EXCEPT ![i] = ((plog68)[i]) \o ((value366).entries)];
                                log := [log90 EXCEPT ![i] = ((log90)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (value4198 = m) {
                                  await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                  fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4198)];
                                  either {
                                    with (value5198 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5198), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1198 = (fd)[j]) {
                                      await yielded_fd1198;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value366).cmd) = (LogPop)) {
                                  plog := [plog68 EXCEPT ![i] = SubSeq((plog68)[i], 1, (Len((plog68)[i])) - ((value366).cnt))];
                                  log := [log90 EXCEPT ![i] = ((log90)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4199 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4199)];
                                    either {
                                      with (value5199 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5199), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1199 = (fd)[j]) {
                                        await yielded_fd1199;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log90 EXCEPT ![i] = ((log90)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4200 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4200)];
                                    either {
                                      with (value5200 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5200), enabled |-> ((network)[j]).enabled]];
                                        plog := plog68;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1200 = (fd)[j]) {
                                        await yielded_fd1200;
                                        plog := plog68;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value226).cmd) = (LogPop)) {
                              with (
                                plog69 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value226).cnt))], 
                                log91 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value367 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value367).cmd) = (LogConcat)) {
                                  plog := [plog69 EXCEPT ![i] = ((plog69)[i]) \o ((value367).entries)];
                                  log := [log91 EXCEPT ![i] = ((log91)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4201 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4201)];
                                    either {
                                      with (value5201 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5201), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1201 = (fd)[j]) {
                                        await yielded_fd1201;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value367).cmd) = (LogPop)) {
                                    plog := [plog69 EXCEPT ![i] = SubSeq((plog69)[i], 1, (Len((plog69)[i])) - ((value367).cnt))];
                                    log := [log91 EXCEPT ![i] = ((log91)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4202 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4202)];
                                      either {
                                        with (value5202 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5202), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1202 = (fd)[j]) {
                                          await yielded_fd1202;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log91 EXCEPT ![i] = ((log91)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4203 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4203)];
                                      either {
                                        with (value5203 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5203), enabled |-> ((network)[j]).enabled]];
                                          plog := plog69;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1203 = (fd)[j]) {
                                          await yielded_fd1203;
                                          plog := plog69;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log92 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value368 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value368).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value368).entries)];
                                  log := [log92 EXCEPT ![i] = ((log92)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4204 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4204)];
                                    either {
                                      with (value5204 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5204), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1204 = (fd)[j]) {
                                        await yielded_fd1204;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value368).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value368).cnt))];
                                    log := [log92 EXCEPT ![i] = ((log92)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4205 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4205)];
                                      either {
                                        with (value5205 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5205), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1205 = (fd)[j]) {
                                          await yielded_fd1205;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log92 EXCEPT ![i] = ((log92)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4206 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4206)];
                                      either {
                                        with (value5206 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5206), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1206 = (fd)[j]) {
                                          await yielded_fd1206;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    } else {
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value133 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value133), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd023 = (fd)[j]) {
                            await yielded_fd023;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value227 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value227).cmd) = (LogConcat)) {
                            with (
                              plog70 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value227).entries)], 
                              log93 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value369 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value369).cmd) = (LogConcat)) {
                                plog := [plog70 EXCEPT ![i] = ((plog70)[i]) \o ((value369).entries)];
                                log := [log93 EXCEPT ![i] = ((log93)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (value4207 = m) {
                                  await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                  fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4207)];
                                  either {
                                    with (value5207 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5207), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1207 = (fd)[j]) {
                                      await yielded_fd1207;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value369).cmd) = (LogPop)) {
                                  plog := [plog70 EXCEPT ![i] = SubSeq((plog70)[i], 1, (Len((plog70)[i])) - ((value369).cnt))];
                                  log := [log93 EXCEPT ![i] = ((log93)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4208 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4208)];
                                    either {
                                      with (value5208 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5208), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1208 = (fd)[j]) {
                                        await yielded_fd1208;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log93 EXCEPT ![i] = ((log93)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4209 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4209)];
                                    either {
                                      with (value5209 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5209), enabled |-> ((network)[j]).enabled]];
                                        plog := plog70;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1209 = (fd)[j]) {
                                        await yielded_fd1209;
                                        plog := plog70;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value227).cmd) = (LogPop)) {
                              with (
                                plog71 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value227).cnt))], 
                                log94 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value370 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value370).cmd) = (LogConcat)) {
                                  plog := [plog71 EXCEPT ![i] = ((plog71)[i]) \o ((value370).entries)];
                                  log := [log94 EXCEPT ![i] = ((log94)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4210 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4210)];
                                    either {
                                      with (value5210 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5210), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1210 = (fd)[j]) {
                                        await yielded_fd1210;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value370).cmd) = (LogPop)) {
                                    plog := [plog71 EXCEPT ![i] = SubSeq((plog71)[i], 1, (Len((plog71)[i])) - ((value370).cnt))];
                                    log := [log94 EXCEPT ![i] = ((log94)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4211 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4211)];
                                      either {
                                        with (value5211 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5211), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1211 = (fd)[j]) {
                                          await yielded_fd1211;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log94 EXCEPT ![i] = ((log94)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4212 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4212)];
                                      either {
                                        with (value5212 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5212), enabled |-> ((network)[j]).enabled]];
                                          plog := plog71;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1212 = (fd)[j]) {
                                          await yielded_fd1212;
                                          plog := plog71;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log95 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value371 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value371).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value371).entries)];
                                  log := [log95 EXCEPT ![i] = ((log95)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (value4213 = m) {
                                    await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                    fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4213)];
                                    either {
                                      with (value5213 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5213), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1213 = (fd)[j]) {
                                        await yielded_fd1213;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value371).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value371).cnt))];
                                    log := [log95 EXCEPT ![i] = ((log95)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4214 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4214)];
                                      either {
                                        with (value5214 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5214), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1214 = (fd)[j]) {
                                          await yielded_fd1214;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log95 EXCEPT ![i] = ((log95)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (value4215 = m) {
                                      await (Len((fAdvCommitIdxCh)[i])) < (BufferSize);
                                      fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value4215)];
                                      either {
                                        with (value5215 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5215), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1215 = (fd)[j]) {
                                          await yielded_fd1215;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                };
              };
            } else {
              if (((m).mtype) = (AppendEntriesResponse)) {
                if (((m).mterm) > ((currentTerm)[self])) {
                  currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                  state := [state EXCEPT ![self] = Follower];
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  leader := [leader EXCEPT ![self] = Nil];
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      if ((m).msuccess) {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                        matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                        goto serverLoop;
                      } else {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      if ((m).msuccess) {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                        matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                        goto serverLoop;
                      } else {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                        goto serverLoop;
                      };
                    };
                  };
                };
              } else {
                if (((m).mtype) = (ProposeMessage)) {
                  with (i = self) {
                    if (Debug) {
                      print <<"HandleProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                      if (((state)[i]) = (Leader)) {
                        with (entry = [term |-> (currentTerm)[i], cmd |-> (m).mcmd]) {
                          log := [log EXCEPT ![i] = Append((log)[i], entry)];
                          with (value610 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                            if (((value610).cmd) = (LogConcat)) {
                              plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value610).entries)];
                              with (value712 = TRUE) {
                                await (Len((appendEntriesCh)[i])) < (BufferSize);
                                appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value712)];
                                goto serverLoop;
                              };
                            } else {
                              if (((value610).cmd) = (LogPop)) {
                                plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value610).cnt))];
                                with (value713 = TRUE) {
                                  await (Len((appendEntriesCh)[i])) < (BufferSize);
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value713)];
                                  goto serverLoop;
                                };
                              } else {
                                with (value714 = TRUE) {
                                  await (Len((appendEntriesCh)[i])) < (BufferSize);
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value714)];
                                  goto serverLoop;
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if (((leader)[i]) # (Nil)) {
                          with (j = (leader)[i]) {
                            either {
                              with (value86 = [mtype |-> ProposeMessage, mcmd |-> (m).mcmd, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value86), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd210 = (fd)[j]) {
                                await yielded_fd210;
                                goto serverLoop;
                              };
                            };
                          };
                        } else {
                          goto serverLoop;
                        };
                      };
                    } else {
                      if (((state)[i]) = (Leader)) {
                        with (entry = [term |-> (currentTerm)[i], cmd |-> (m).mcmd]) {
                          log := [log EXCEPT ![i] = Append((log)[i], entry)];
                          with (value611 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                            if (((value611).cmd) = (LogConcat)) {
                              plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value611).entries)];
                              with (value715 = TRUE) {
                                await (Len((appendEntriesCh)[i])) < (BufferSize);
                                appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value715)];
                                goto serverLoop;
                              };
                            } else {
                              if (((value611).cmd) = (LogPop)) {
                                plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value611).cnt))];
                                with (value716 = TRUE) {
                                  await (Len((appendEntriesCh)[i])) < (BufferSize);
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value716)];
                                  goto serverLoop;
                                };
                              } else {
                                with (value717 = TRUE) {
                                  await (Len((appendEntriesCh)[i])) < (BufferSize);
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value717)];
                                  goto serverLoop;
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if (((leader)[i]) # (Nil)) {
                          with (j = (leader)[i]) {
                            either {
                              with (value87 = [mtype |-> ProposeMessage, mcmd |-> (m).mcmd, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value87), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd211 = (fd)[j]) {
                                await yielded_fd211;
                                goto serverLoop;
                              };
                            };
                          };
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  };
                } else {
                  goto serverLoop;
                };
              };
            };
          };
        };
      };
  }
  
  fair process (s1 \in ServerPropChListenerSet)
    variables idx0 = 1; m0; srvId0 = (propChListenerSrvId)[self];
  {
    serverLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network20 = ((network)[srvId0]).enabled) {
            if (~ (yielded_network20)) {
              await FALSE;
              with (i = srvId0) {
                await (Len((propCh)[i])) > (0);
                with (res00 = Head((propCh)[i])) {
                  propCh := [propCh EXCEPT ![i] = Tail((propCh)[i])];
                  with (yielded_propCh0 = res00) {
                    m0 := yielded_propCh0;
                    if (Debug) {
                      print <<"ReceiveProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i], m0>>;
                      goto handleMsg;
                    } else {
                      goto handleMsg;
                    };
                  };
                };
              };
            } else {
              with (i = srvId0) {
                await (Len((propCh)[i])) > (0);
                with (res01 = Head((propCh)[i])) {
                  propCh := [propCh EXCEPT ![i] = Tail((propCh)[i])];
                  with (yielded_propCh1 = res01) {
                    m0 := yielded_propCh1;
                    if (Debug) {
                      print <<"ReceiveProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i], m0>>;
                      goto handleMsg;
                    } else {
                      goto handleMsg;
                    };
                  };
                };
              };
            };
          };
        } else {
          with (i = srvId0) {
            await (Len((propCh)[i])) > (0);
            with (res02 = Head((propCh)[i])) {
              propCh := [propCh EXCEPT ![i] = Tail((propCh)[i])];
              with (yielded_propCh2 = res02) {
                m0 := yielded_propCh2;
                if (Debug) {
                  print <<"ReceiveProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i], m0>>;
                  goto handleMsg;
                } else {
                  goto handleMsg;
                };
              };
            };
          };
        };
      } else {
        goto Done;
      };
    handleMsg:
      if (ExploreFail) {
        with (yielded_network30 = ((network)[srvId0]).enabled) {
          if (~ (yielded_network30)) {
            await FALSE;
            assert ((m0).mtype) = (ProposeMessage);
            with (i = srvId0) {
              if (Debug) {
                print <<"HandleProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                if (((state)[i]) = (Leader)) {
                  with (entry = [term |-> (currentTerm)[i], cmd |-> (m0).mcmd]) {
                    log := [log EXCEPT ![i] = Append((log)[i], entry)];
                    with (value90 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                      if (((value90).cmd) = (LogConcat)) {
                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value90).entries)];
                        with (value100 = TRUE) {
                          await (Len((appendEntriesCh)[i])) < (BufferSize);
                          appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value100)];
                          goto serverLoop;
                        };
                      } else {
                        if (((value90).cmd) = (LogPop)) {
                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value90).cnt))];
                          with (value101 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value101)];
                            goto serverLoop;
                          };
                        } else {
                          with (value102 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value102)];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((leader)[i]) # (Nil)) {
                    with (j = (leader)[i]) {
                      either {
                        with (value1110 = [mtype |-> ProposeMessage, mcmd |-> (m0).mcmd, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value1110), enabled |-> ((network)[j]).enabled]];
                          goto serverLoop;
                        };
                      } or {
                        with (yielded_fd32 = (fd)[j]) {
                          await yielded_fd32;
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              } else {
                if (((state)[i]) = (Leader)) {
                  with (entry = [term |-> (currentTerm)[i], cmd |-> (m0).mcmd]) {
                    log := [log EXCEPT ![i] = Append((log)[i], entry)];
                    with (value91 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                      if (((value91).cmd) = (LogConcat)) {
                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value91).entries)];
                        with (value103 = TRUE) {
                          await (Len((appendEntriesCh)[i])) < (BufferSize);
                          appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value103)];
                          goto serverLoop;
                        };
                      } else {
                        if (((value91).cmd) = (LogPop)) {
                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value91).cnt))];
                          with (value104 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value104)];
                            goto serverLoop;
                          };
                        } else {
                          with (value105 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value105)];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((leader)[i]) # (Nil)) {
                    with (j = (leader)[i]) {
                      either {
                        with (value1111 = [mtype |-> ProposeMessage, mcmd |-> (m0).mcmd, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value1111), enabled |-> ((network)[j]).enabled]];
                          goto serverLoop;
                        };
                      } or {
                        with (yielded_fd33 = (fd)[j]) {
                          await yielded_fd33;
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              };
            };
          } else {
            assert ((m0).mtype) = (ProposeMessage);
            with (i = srvId0) {
              if (Debug) {
                print <<"HandleProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                if (((state)[i]) = (Leader)) {
                  with (entry = [term |-> (currentTerm)[i], cmd |-> (m0).mcmd]) {
                    log := [log EXCEPT ![i] = Append((log)[i], entry)];
                    with (value92 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                      if (((value92).cmd) = (LogConcat)) {
                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value92).entries)];
                        with (value106 = TRUE) {
                          await (Len((appendEntriesCh)[i])) < (BufferSize);
                          appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value106)];
                          goto serverLoop;
                        };
                      } else {
                        if (((value92).cmd) = (LogPop)) {
                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value92).cnt))];
                          with (value107 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value107)];
                            goto serverLoop;
                          };
                        } else {
                          with (value108 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value108)];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((leader)[i]) # (Nil)) {
                    with (j = (leader)[i]) {
                      either {
                        with (value1112 = [mtype |-> ProposeMessage, mcmd |-> (m0).mcmd, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value1112), enabled |-> ((network)[j]).enabled]];
                          goto serverLoop;
                        };
                      } or {
                        with (yielded_fd34 = (fd)[j]) {
                          await yielded_fd34;
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              } else {
                if (((state)[i]) = (Leader)) {
                  with (entry = [term |-> (currentTerm)[i], cmd |-> (m0).mcmd]) {
                    log := [log EXCEPT ![i] = Append((log)[i], entry)];
                    with (value93 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                      if (((value93).cmd) = (LogConcat)) {
                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value93).entries)];
                        with (value109 = TRUE) {
                          await (Len((appendEntriesCh)[i])) < (BufferSize);
                          appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value109)];
                          goto serverLoop;
                        };
                      } else {
                        if (((value93).cmd) = (LogPop)) {
                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value93).cnt))];
                          with (value1010 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1010)];
                            goto serverLoop;
                          };
                        } else {
                          with (value1011 = TRUE) {
                            await (Len((appendEntriesCh)[i])) < (BufferSize);
                            appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1011)];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((leader)[i]) # (Nil)) {
                    with (j = (leader)[i]) {
                      either {
                        with (value1113 = [mtype |-> ProposeMessage, mcmd |-> (m0).mcmd, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value1113), enabled |-> ((network)[j]).enabled]];
                          goto serverLoop;
                        };
                      } or {
                        with (yielded_fd35 = (fd)[j]) {
                          await yielded_fd35;
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              };
            };
          };
        };
      } else {
        assert ((m0).mtype) = (ProposeMessage);
        with (i = srvId0) {
          if (Debug) {
            print <<"HandleProposeMessage", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
            if (((state)[i]) = (Leader)) {
              with (entry = [term |-> (currentTerm)[i], cmd |-> (m0).mcmd]) {
                log := [log EXCEPT ![i] = Append((log)[i], entry)];
                with (value94 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                  if (((value94).cmd) = (LogConcat)) {
                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value94).entries)];
                    with (value1012 = TRUE) {
                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1012)];
                      goto serverLoop;
                    };
                  } else {
                    if (((value94).cmd) = (LogPop)) {
                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value94).cnt))];
                      with (value1013 = TRUE) {
                        await (Len((appendEntriesCh)[i])) < (BufferSize);
                        appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1013)];
                        goto serverLoop;
                      };
                    } else {
                      with (value1014 = TRUE) {
                        await (Len((appendEntriesCh)[i])) < (BufferSize);
                        appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1014)];
                        goto serverLoop;
                      };
                    };
                  };
                };
              };
            } else {
              if (((leader)[i]) # (Nil)) {
                with (j = (leader)[i]) {
                  either {
                    with (value1114 = [mtype |-> ProposeMessage, mcmd |-> (m0).mcmd, msource |-> i, mdest |-> j]) {
                      await ((network)[j]).enabled;
                      await (Len(((network)[j]).queue)) < (BufferSize);
                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value1114), enabled |-> ((network)[j]).enabled]];
                      goto serverLoop;
                    };
                  } or {
                    with (yielded_fd36 = (fd)[j]) {
                      await yielded_fd36;
                      goto serverLoop;
                    };
                  };
                };
              } else {
                goto serverLoop;
              };
            };
          } else {
            if (((state)[i]) = (Leader)) {
              with (entry = [term |-> (currentTerm)[i], cmd |-> (m0).mcmd]) {
                log := [log EXCEPT ![i] = Append((log)[i], entry)];
                with (value95 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                  if (((value95).cmd) = (LogConcat)) {
                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value95).entries)];
                    with (value1015 = TRUE) {
                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1015)];
                      goto serverLoop;
                    };
                  } else {
                    if (((value95).cmd) = (LogPop)) {
                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value95).cnt))];
                      with (value1016 = TRUE) {
                        await (Len((appendEntriesCh)[i])) < (BufferSize);
                        appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1016)];
                        goto serverLoop;
                      };
                    } else {
                      with (value1017 = TRUE) {
                        await (Len((appendEntriesCh)[i])) < (BufferSize);
                        appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value1017)];
                        goto serverLoop;
                      };
                    };
                  };
                };
              };
            } else {
              if (((leader)[i]) # (Nil)) {
                with (j = (leader)[i]) {
                  either {
                    with (value1115 = [mtype |-> ProposeMessage, mcmd |-> (m0).mcmd, msource |-> i, mdest |-> j]) {
                      await ((network)[j]).enabled;
                      await (Len(((network)[j]).queue)) < (BufferSize);
                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value1115), enabled |-> ((network)[j]).enabled]];
                      goto serverLoop;
                    };
                  } or {
                    with (yielded_fd37 = (fd)[j]) {
                      await yielded_fd37;
                      goto serverLoop;
                    };
                  };
                };
              } else {
                goto serverLoop;
              };
            };
          };
        };
      };
  }
  
  fair process (s2 \in ServerRequestVoteSet)
    variables idx1 = 1; srvId1 = (requestVoteSrvId)[self];
  {
    serverRequestVoteLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network40 = ((network)[srvId1]).enabled) {
            if (~ (yielded_network40)) {
              await FALSE;
              await leaderTimeout;
              with (yielded_network50 = Len(((network)[srvId1]).queue)) {
                await (yielded_network50) = (0);
                await ((state)[srvId1]) \in ({Follower, Candidate});
                with (i3 = srvId1) {
                  state := [state EXCEPT ![i3] = Candidate];
                  currentTerm := [currentTerm EXCEPT ![i3] = ((currentTerm)[i3]) + (1)];
                  votedFor := [votedFor EXCEPT ![i3] = i3];
                  votesResponded := [votesResponded EXCEPT ![i3] = {i3}];
                  votesGranted := [votesGranted EXCEPT ![i3] = {i3}];
                  leader := [leader EXCEPT ![i3] = Nil];
                  if (Debug) {
                    print <<"ServerTimeout", i3, (currentTerm)[i3]>>;
                    idx1 := 1;
                    goto requestVoteLoop;
                  } else {
                    idx1 := 1;
                    goto requestVoteLoop;
                  };
                };
              };
            } else {
              await leaderTimeout;
              with (yielded_network51 = Len(((network)[srvId1]).queue)) {
                await (yielded_network51) = (0);
                await ((state)[srvId1]) \in ({Follower, Candidate});
                with (i4 = srvId1) {
                  state := [state EXCEPT ![i4] = Candidate];
                  currentTerm := [currentTerm EXCEPT ![i4] = ((currentTerm)[i4]) + (1)];
                  votedFor := [votedFor EXCEPT ![i4] = i4];
                  votesResponded := [votesResponded EXCEPT ![i4] = {i4}];
                  votesGranted := [votesGranted EXCEPT ![i4] = {i4}];
                  leader := [leader EXCEPT ![i4] = Nil];
                  if (Debug) {
                    print <<"ServerTimeout", i4, (currentTerm)[i4]>>;
                    idx1 := 1;
                    goto requestVoteLoop;
                  } else {
                    idx1 := 1;
                    goto requestVoteLoop;
                  };
                };
              };
            };
          };
        } else {
          await leaderTimeout;
          with (yielded_network52 = Len(((network)[srvId1]).queue)) {
            await (yielded_network52) = (0);
            await ((state)[srvId1]) \in ({Follower, Candidate});
            with (i5 = srvId1) {
              state := [state EXCEPT ![i5] = Candidate];
              currentTerm := [currentTerm EXCEPT ![i5] = ((currentTerm)[i5]) + (1)];
              votedFor := [votedFor EXCEPT ![i5] = i5];
              votesResponded := [votesResponded EXCEPT ![i5] = {i5}];
              votesGranted := [votesGranted EXCEPT ![i5] = {i5}];
              leader := [leader EXCEPT ![i5] = Nil];
              if (Debug) {
                print <<"ServerTimeout", i5, (currentTerm)[i5]>>;
                idx1 := 1;
                goto requestVoteLoop;
              } else {
                idx1 := 1;
                goto requestVoteLoop;
              };
            };
          };
        };
      } else {
        goto Done;
      };
    requestVoteLoop:
      if ((idx1) <= (NumServers)) {
        if (ExploreFail) {
          with (yielded_network60 = ((network)[srvId1]).enabled) {
            if (~ (yielded_network60)) {
              await FALSE;
              if ((idx1) # (srvId1)) {
                either {
                  with (value1210 = [mtype |-> RequestVoteRequest, mterm |-> (currentTerm)[srvId1], mlastLogTerm |-> LastTerm((log)[srvId1]), mlastLogIndex |-> Len((log)[srvId1]), msource |-> srvId1, mdest |-> idx1]) {
                    await ((network)[idx1]).enabled;
                    await (Len(((network)[idx1]).queue)) < (BufferSize);
                    network := [network EXCEPT ![idx1] = [queue |-> Append(((network)[idx1]).queue, value1210), enabled |-> ((network)[idx1]).enabled]];
                    idx1 := (idx1) + (1);
                    goto requestVoteLoop;
                  };
                } or {
                  with (yielded_fd40 = (fd)[idx1]) {
                    await yielded_fd40;
                    idx1 := (idx1) + (1);
                    goto requestVoteLoop;
                  };
                };
              } else {
                idx1 := (idx1) + (1);
                goto requestVoteLoop;
              };
            } else {
              if ((idx1) # (srvId1)) {
                either {
                  with (value1211 = [mtype |-> RequestVoteRequest, mterm |-> (currentTerm)[srvId1], mlastLogTerm |-> LastTerm((log)[srvId1]), mlastLogIndex |-> Len((log)[srvId1]), msource |-> srvId1, mdest |-> idx1]) {
                    await ((network)[idx1]).enabled;
                    await (Len(((network)[idx1]).queue)) < (BufferSize);
                    network := [network EXCEPT ![idx1] = [queue |-> Append(((network)[idx1]).queue, value1211), enabled |-> ((network)[idx1]).enabled]];
                    idx1 := (idx1) + (1);
                    goto requestVoteLoop;
                  };
                } or {
                  with (yielded_fd41 = (fd)[idx1]) {
                    await yielded_fd41;
                    idx1 := (idx1) + (1);
                    goto requestVoteLoop;
                  };
                };
              } else {
                idx1 := (idx1) + (1);
                goto requestVoteLoop;
              };
            };
          };
        } else {
          if ((idx1) # (srvId1)) {
            either {
              with (value1212 = [mtype |-> RequestVoteRequest, mterm |-> (currentTerm)[srvId1], mlastLogTerm |-> LastTerm((log)[srvId1]), mlastLogIndex |-> Len((log)[srvId1]), msource |-> srvId1, mdest |-> idx1]) {
                await ((network)[idx1]).enabled;
                await (Len(((network)[idx1]).queue)) < (BufferSize);
                network := [network EXCEPT ![idx1] = [queue |-> Append(((network)[idx1]).queue, value1212), enabled |-> ((network)[idx1]).enabled]];
                idx1 := (idx1) + (1);
                goto requestVoteLoop;
              };
            } or {
              with (yielded_fd42 = (fd)[idx1]) {
                await yielded_fd42;
                idx1 := (idx1) + (1);
                goto requestVoteLoop;
              };
            };
          } else {
            idx1 := (idx1) + (1);
            goto requestVoteLoop;
          };
        };
      } else {
        goto serverRequestVoteLoop;
      };
  }
  
  fair process (s3 \in ServerAppendEntriesSet)
    variables idx2; srvId2 = (appendEntriesSrvId)[self];
  {
    serverAppendEntriesLoop:
      await (Len((appendEntriesCh)[srvId2])) > (0);
      with (res1 = Head((appendEntriesCh)[srvId2])) {
        appendEntriesCh := [appendEntriesCh EXCEPT ![srvId2] = Tail((appendEntriesCh)[srvId2])];
        with (yielded_appendEntriesCh = res1) {
          if (yielded_appendEntriesCh) {
            if (ExploreFail) {
              with (yielded_network70 = ((network)[srvId2]).enabled) {
                if (~ (yielded_network70)) {
                  await FALSE;
                  await ((state)[srvId2]) = (Leader);
                  idx2 := 1;
                  goto appendEntriesLoop;
                } else {
                  await ((state)[srvId2]) = (Leader);
                  idx2 := 1;
                  goto appendEntriesLoop;
                };
              };
            } else {
              await ((state)[srvId2]) = (Leader);
              idx2 := 1;
              goto appendEntriesLoop;
            };
          } else {
            goto Done;
          };
        };
      };
    appendEntriesLoop:
      if ((((state)[srvId2]) = (Leader)) /\ ((idx2) <= (NumServers))) {
        if (ExploreFail) {
          with (yielded_network80 = ((network)[srvId2]).enabled) {
            if (~ (yielded_network80)) {
              await FALSE;
              if ((idx2) # (srvId2)) {
                with (
                  prevLogIndex3 = (((nextIndex)[srvId2])[idx2]) - (1), 
                  prevLogTerm3 = IF (prevLogIndex3) > (0) THEN (((log)[srvId2])[prevLogIndex3]).term ELSE 0, 
                  entries3 = SubSeq((log)[srvId2], ((nextIndex)[srvId2])[idx2], Len((log)[srvId2]))
                ) {
                  either {
                    with (value134 = [mtype |-> AppendEntriesRequest, mterm |-> (currentTerm)[srvId2], mprevLogIndex |-> prevLogIndex3, mprevLogTerm |-> prevLogTerm3, mentries |-> entries3, mcommitIndex |-> (commitIndex)[srvId2], msource |-> srvId2, mdest |-> idx2]) {
                      await ((network)[idx2]).enabled;
                      await (Len(((network)[idx2]).queue)) < (BufferSize);
                      network := [network EXCEPT ![idx2] = [queue |-> Append(((network)[idx2]).queue, value134), enabled |-> ((network)[idx2]).enabled]];
                      idx2 := (idx2) + (1);
                      goto appendEntriesLoop;
                    };
                  } or {
                    with (yielded_fd50 = (fd)[idx2]) {
                      await yielded_fd50;
                      idx2 := (idx2) + (1);
                      goto appendEntriesLoop;
                    };
                  };
                };
              } else {
                idx2 := (idx2) + (1);
                goto appendEntriesLoop;
              };
            } else {
              if ((idx2) # (srvId2)) {
                with (
                  prevLogIndex4 = (((nextIndex)[srvId2])[idx2]) - (1), 
                  prevLogTerm4 = IF (prevLogIndex4) > (0) THEN (((log)[srvId2])[prevLogIndex4]).term ELSE 0, 
                  entries4 = SubSeq((log)[srvId2], ((nextIndex)[srvId2])[idx2], Len((log)[srvId2]))
                ) {
                  either {
                    with (value135 = [mtype |-> AppendEntriesRequest, mterm |-> (currentTerm)[srvId2], mprevLogIndex |-> prevLogIndex4, mprevLogTerm |-> prevLogTerm4, mentries |-> entries4, mcommitIndex |-> (commitIndex)[srvId2], msource |-> srvId2, mdest |-> idx2]) {
                      await ((network)[idx2]).enabled;
                      await (Len(((network)[idx2]).queue)) < (BufferSize);
                      network := [network EXCEPT ![idx2] = [queue |-> Append(((network)[idx2]).queue, value135), enabled |-> ((network)[idx2]).enabled]];
                      idx2 := (idx2) + (1);
                      goto appendEntriesLoop;
                    };
                  } or {
                    with (yielded_fd51 = (fd)[idx2]) {
                      await yielded_fd51;
                      idx2 := (idx2) + (1);
                      goto appendEntriesLoop;
                    };
                  };
                };
              } else {
                idx2 := (idx2) + (1);
                goto appendEntriesLoop;
              };
            };
          };
        } else {
          if ((idx2) # (srvId2)) {
            with (
              prevLogIndex5 = (((nextIndex)[srvId2])[idx2]) - (1), 
              prevLogTerm5 = IF (prevLogIndex5) > (0) THEN (((log)[srvId2])[prevLogIndex5]).term ELSE 0, 
              entries5 = SubSeq((log)[srvId2], ((nextIndex)[srvId2])[idx2], Len((log)[srvId2]))
            ) {
              either {
                with (value136 = [mtype |-> AppendEntriesRequest, mterm |-> (currentTerm)[srvId2], mprevLogIndex |-> prevLogIndex5, mprevLogTerm |-> prevLogTerm5, mentries |-> entries5, mcommitIndex |-> (commitIndex)[srvId2], msource |-> srvId2, mdest |-> idx2]) {
                  await ((network)[idx2]).enabled;
                  await (Len(((network)[idx2]).queue)) < (BufferSize);
                  network := [network EXCEPT ![idx2] = [queue |-> Append(((network)[idx2]).queue, value136), enabled |-> ((network)[idx2]).enabled]];
                  idx2 := (idx2) + (1);
                  goto appendEntriesLoop;
                };
              } or {
                with (yielded_fd52 = (fd)[idx2]) {
                  await yielded_fd52;
                  idx2 := (idx2) + (1);
                  goto appendEntriesLoop;
                };
              };
            };
          } else {
            idx2 := (idx2) + (1);
            goto appendEntriesLoop;
          };
        };
      } else {
        goto serverAppendEntriesLoop;
      };
  }
  
  fair process (s4 \in ServerAdvanceCommitIndexSet)
    variables newCommitIndex = 0; srvId3 = (advanceCommitIndexSrvId)[self];
  {
    serverAdvanceCommitIndexLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network90 = ((network)[srvId3]).enabled) {
            if (~ (yielded_network90)) {
              await FALSE;
              await ((state)[srvId3]) = (Leader);
              with (
                i = srvId3, 
                maxAgreeIndex = FindMaxAgreeIndex((log)[i], i, (matchIndex)[i]), 
                nCommitIndex = IF ((maxAgreeIndex) # (Nil)) /\ (((((log)[i])[maxAgreeIndex]).term) = ((currentTerm)[i])) THEN maxAgreeIndex ELSE (commitIndex)[i]
              ) {
                newCommitIndex := nCommitIndex;
                assert (newCommitIndex) >= ((commitIndex)[i]);
                goto applyLoop;
              };
            } else {
              await ((state)[srvId3]) = (Leader);
              with (
                i = srvId3, 
                maxAgreeIndex = FindMaxAgreeIndex((log)[i], i, (matchIndex)[i]), 
                nCommitIndex = IF ((maxAgreeIndex) # (Nil)) /\ (((((log)[i])[maxAgreeIndex]).term) = ((currentTerm)[i])) THEN maxAgreeIndex ELSE (commitIndex)[i]
              ) {
                newCommitIndex := nCommitIndex;
                assert (newCommitIndex) >= ((commitIndex)[i]);
                goto applyLoop;
              };
            };
          };
        } else {
          await ((state)[srvId3]) = (Leader);
          with (
            i = srvId3, 
            maxAgreeIndex = FindMaxAgreeIndex((log)[i], i, (matchIndex)[i]), 
            nCommitIndex = IF ((maxAgreeIndex) # (Nil)) /\ (((((log)[i])[maxAgreeIndex]).term) = ((currentTerm)[i])) THEN maxAgreeIndex ELSE (commitIndex)[i]
          ) {
            newCommitIndex := nCommitIndex;
            assert (newCommitIndex) >= ((commitIndex)[i]);
            goto applyLoop;
          };
        };
      } else {
        goto Done;
      };
    applyLoop:
      if (((commitIndex)[srvId3]) < (newCommitIndex)) {
        if (ExploreFail) {
          with (yielded_network100 = ((network)[srvId3]).enabled) {
            if (~ (yielded_network100)) {
              await FALSE;
              commitIndex := [commitIndex EXCEPT ![srvId3] = ((commitIndex)[srvId3]) + (1)];
              with (
                i = srvId3, 
                k = (commitIndex)[i], 
                entry = ((log)[i])[k], 
                cmd = (entry).cmd
              ) {
                if (Debug) {
                  print <<"ServerAccept", i, cmd>>;
                  with (value140 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value140)];
                    goto applyLoop;
                  };
                } else {
                  with (value141 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value141)];
                    goto applyLoop;
                  };
                };
              };
            } else {
              commitIndex := [commitIndex EXCEPT ![srvId3] = ((commitIndex)[srvId3]) + (1)];
              with (
                i = srvId3, 
                k = (commitIndex)[i], 
                entry = ((log)[i])[k], 
                cmd = (entry).cmd
              ) {
                if (Debug) {
                  print <<"ServerAccept", i, cmd>>;
                  with (value142 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value142)];
                    goto applyLoop;
                  };
                } else {
                  with (value143 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value143)];
                    goto applyLoop;
                  };
                };
              };
            };
          };
        } else {
          commitIndex := [commitIndex EXCEPT ![srvId3] = ((commitIndex)[srvId3]) + (1)];
          with (
            i = srvId3, 
            k = (commitIndex)[i], 
            entry = ((log)[i])[k], 
            cmd = (entry).cmd
          ) {
            if (Debug) {
              print <<"ServerAccept", i, cmd>>;
              with (value144 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                await (Len((acctCh)[i])) < (BufferSize);
                acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value144)];
                goto applyLoop;
              };
            } else {
              with (value145 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                await (Len((acctCh)[i])) < (BufferSize);
                acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value145)];
                goto applyLoop;
              };
            };
          };
        };
      } else {
        goto serverAdvanceCommitIndexLoop;
      };
  }
  
  fair process (s5 \in ServerBecomeLeaderSet)
    variables srvId4 = (becomeLeaderSrvId)[self];
  {
    serverBecomeLeaderLoop:
      await (Len((becomeLeaderCh)[srvId4])) > (0);
      with (res2 = Head((becomeLeaderCh)[srvId4])) {
        becomeLeaderCh := [becomeLeaderCh EXCEPT ![srvId4] = Tail((becomeLeaderCh)[srvId4])];
        with (yielded_becomeLeaderCh = res2) {
          if (yielded_becomeLeaderCh) {
            if (ExploreFail) {
              with (yielded_network110 = ((network)[srvId4]).enabled) {
                if (~ (yielded_network110)) {
                  await FALSE;
                  await ((state)[srvId4]) = (Candidate);
                  await isQuorum((votesGranted)[srvId4]);
                  with (i = srvId4) {
                    state := [state EXCEPT ![i] = Leader];
                    nextIndex := [nextIndex EXCEPT ![i] = [j \in ServerSet |-> (Len((log)[i])) + (1)]];
                    matchIndex := [matchIndex EXCEPT ![i] = [j \in ServerSet |-> 0]];
                    leader := [leader EXCEPT ![i] = i];
                    with (value150 = TRUE) {
                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value150)];
                      if (Debug) {
                        print <<"BecomeLeader", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                        goto serverBecomeLeaderLoop;
                      } else {
                        goto serverBecomeLeaderLoop;
                      };
                    };
                  };
                } else {
                  await ((state)[srvId4]) = (Candidate);
                  await isQuorum((votesGranted)[srvId4]);
                  with (i = srvId4) {
                    state := [state EXCEPT ![i] = Leader];
                    nextIndex := [nextIndex EXCEPT ![i] = [j \in ServerSet |-> (Len((log)[i])) + (1)]];
                    matchIndex := [matchIndex EXCEPT ![i] = [j \in ServerSet |-> 0]];
                    leader := [leader EXCEPT ![i] = i];
                    with (value151 = TRUE) {
                      await (Len((appendEntriesCh)[i])) < (BufferSize);
                      appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value151)];
                      if (Debug) {
                        print <<"BecomeLeader", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                        goto serverBecomeLeaderLoop;
                      } else {
                        goto serverBecomeLeaderLoop;
                      };
                    };
                  };
                };
              };
            } else {
              await ((state)[srvId4]) = (Candidate);
              await isQuorum((votesGranted)[srvId4]);
              with (i = srvId4) {
                state := [state EXCEPT ![i] = Leader];
                nextIndex := [nextIndex EXCEPT ![i] = [j \in ServerSet |-> (Len((log)[i])) + (1)]];
                matchIndex := [matchIndex EXCEPT ![i] = [j \in ServerSet |-> 0]];
                leader := [leader EXCEPT ![i] = i];
                with (value152 = TRUE) {
                  await (Len((appendEntriesCh)[i])) < (BufferSize);
                  appendEntriesCh := [appendEntriesCh EXCEPT ![i] = Append((appendEntriesCh)[i], value152)];
                  if (Debug) {
                    print <<"BecomeLeader", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                    goto serverBecomeLeaderLoop;
                  } else {
                    goto serverBecomeLeaderLoop;
                  };
                };
              };
            };
          } else {
            goto Done;
          };
        };
      };
  }
  
  fair process (s6 \in ServerFollowerAdvanceCommitIndexSet)
    variables m1; srvId5 = (fAdvCommitSrvId)[self];
  {
    serverFollowerAdvanceCommitIndexLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network120 = ((network)[srvId5]).enabled) {
            if (~ (yielded_network120)) {
              await FALSE;
              await (Len((fAdvCommitIdxCh)[srvId5])) > (0);
              with (res30 = Head((fAdvCommitIdxCh)[srvId5])) {
                fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![srvId5] = Tail((fAdvCommitIdxCh)[srvId5])];
                with (yielded_fAdvCommitIdxCh0 = res30) {
                  m1 := yielded_fAdvCommitIdxCh0;
                  goto acctLoop;
                };
              };
            } else {
              await (Len((fAdvCommitIdxCh)[srvId5])) > (0);
              with (res31 = Head((fAdvCommitIdxCh)[srvId5])) {
                fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![srvId5] = Tail((fAdvCommitIdxCh)[srvId5])];
                with (yielded_fAdvCommitIdxCh1 = res31) {
                  m1 := yielded_fAdvCommitIdxCh1;
                  goto acctLoop;
                };
              };
            };
          };
        } else {
          await (Len((fAdvCommitIdxCh)[srvId5])) > (0);
          with (res32 = Head((fAdvCommitIdxCh)[srvId5])) {
            fAdvCommitIdxCh := [fAdvCommitIdxCh EXCEPT ![srvId5] = Tail((fAdvCommitIdxCh)[srvId5])];
            with (yielded_fAdvCommitIdxCh2 = res32) {
              m1 := yielded_fAdvCommitIdxCh2;
              goto acctLoop;
            };
          };
        };
      } else {
        goto Done;
      };
    acctLoop:
      if (((commitIndex)[srvId5]) < ((m1).mcommitIndex)) {
        if (ExploreFail) {
          with (yielded_network130 = ((network)[srvId5]).enabled) {
            if (~ (yielded_network130)) {
              await FALSE;
              commitIndex := [commitIndex EXCEPT ![srvId5] = ((commitIndex)[srvId5]) + (1)];
              with (
                i = srvId5, 
                k = (commitIndex)[i], 
                entry = ((log)[i])[k], 
                cmd = (entry).cmd
              ) {
                if (Debug) {
                  print <<"ServerAccept", i, cmd>>;
                  with (value160 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value160)];
                    goto acctLoop;
                  };
                } else {
                  with (value161 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value161)];
                    goto acctLoop;
                  };
                };
              };
            } else {
              commitIndex := [commitIndex EXCEPT ![srvId5] = ((commitIndex)[srvId5]) + (1)];
              with (
                i = srvId5, 
                k = (commitIndex)[i], 
                entry = ((log)[i])[k], 
                cmd = (entry).cmd
              ) {
                if (Debug) {
                  print <<"ServerAccept", i, cmd>>;
                  with (value162 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value162)];
                    goto acctLoop;
                  };
                } else {
                  with (value163 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                    await (Len((acctCh)[i])) < (BufferSize);
                    acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value163)];
                    goto acctLoop;
                  };
                };
              };
            };
          };
        } else {
          commitIndex := [commitIndex EXCEPT ![srvId5] = ((commitIndex)[srvId5]) + (1)];
          with (
            i = srvId5, 
            k = (commitIndex)[i], 
            entry = ((log)[i])[k], 
            cmd = (entry).cmd
          ) {
            if (Debug) {
              print <<"ServerAccept", i, cmd>>;
              with (value164 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                await (Len((acctCh)[i])) < (BufferSize);
                acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value164)];
                goto acctLoop;
              };
            } else {
              with (value165 = [mtype |-> AcceptMessage, mcmd |-> cmd]) {
                await (Len((acctCh)[i])) < (BufferSize);
                acctCh := [acctCh EXCEPT ![i] = Append((acctCh)[i], value165)];
                goto acctLoop;
              };
            };
          };
        };
      } else {
        goto serverFollowerAdvanceCommitIndexLoop;
      };
  }
  
  fair process (crasher \in ServerCrasherSet)
    variables srvId6 = (crasherSrvId)[self];
  {
    serverCrash:
      with (value170 = FALSE) {
        network := [network EXCEPT ![srvId6] = [queue |-> ((network)[srvId6]).queue, enabled |-> value170]];
        goto fdUpdate;
      };
    fdUpdate:
      with (value180 = TRUE) {
        fd := [fd EXCEPT ![srvId6] = value180];
        goto Done;
      };
  }
  
  fair process (proposer = 0)
  {
    sndReq:
      with (value190 = [mtype |-> ProposeMessage, mcmd |-> [data |-> "hello"]]) {
        await (Len((propCh)[1])) < (BufferSize);
        propCh := [propCh EXCEPT ![1] = Append((propCh)[1], value190)];
        goto Done;
      };
  }
}

\* END PLUSCAL TRANSLATION

********************)
\* BEGIN TRANSLATION (chksum(pcal) = "9001dd7" /\ chksum(tla) = "4cc41aae") PCal-18049938ece8066a38eb5044080cf45c
\* Label serverLoop of process s0 at line 1004 col 7 changed to serverLoop_
\* Label handleMsg of process s0 at line 1048 col 7 changed to handleMsg_
CONSTANT defaultInitValue
VARIABLES network, fd, state, currentTerm, commitIndex, nextIndex, matchIndex, 
          log, plog, votedFor, votesResponded, votesGranted, leader, propCh, 
          acctCh, leaderTimeout, appendEntriesCh, becomeLeaderCh, 
          fAdvCommitIdxCh, netListenerSrvId, propChListenerSrvId, 
          requestVoteSrvId, appendEntriesSrvId, advanceCommitIndexSrvId, 
          becomeLeaderSrvId, fAdvCommitSrvId, crasherSrvId, pc

(* define statement *)
Follower == "follower"
Candidate == "candidate"
Leader == "leader"
RequestVoteRequest == "rvq"
RequestVoteResponse == "rvp"
AppendEntriesRequest == "apq"
AppendEntriesResponse == "app"
ProposeMessage == "prm"
AcceptMessage == "acm"
Key1 == "key1"
Key2 == "key2"
Value1 == "value1"
LastTerm(xlog) == IF (Len(xlog)) = (0) THEN 0 ELSE ((xlog)[Len(xlog)]).term
ServerNetListenerSet == (((0) * (NumServers)) + (1)) .. ((1) * (NumServers))
ServerPropChListenerSet == (((1) * (NumServers)) + (1)) .. ((2) * (NumServers))
ServerRequestVoteSet == (((2) * (NumServers)) + (1)) .. ((3) * (NumServers))
ServerAppendEntriesSet == (((3) * (NumServers)) + (1)) .. ((4) * (NumServers))
ServerAdvanceCommitIndexSet == (((4) * (NumServers)) + (1)) .. ((5) * (NumServers))
ServerBecomeLeaderSet == (((5) * (NumServers)) + (1)) .. ((6) * (NumServers))
ServerFollowerAdvanceCommitIndexSet == (((6) * (NumServers)) + (1)) .. ((7) * (NumServers))
ServerCrasherSet == IF ExploreFail THEN (((7) * (NumServers)) + (1)) .. (((7) * (NumServers)) + (MaxNodeFail)) ELSE {}
NodeSet == ServerSet
KeySet == {}

VARIABLES idx, m, srvId, idx0, m0, srvId0, idx1, srvId1, idx2, srvId2, 
          newCommitIndex, srvId3, srvId4, m1, srvId5, srvId6

vars == << network, fd, state, currentTerm, commitIndex, nextIndex, 
           matchIndex, log, plog, votedFor, votesResponded, votesGranted, 
           leader, propCh, acctCh, leaderTimeout, appendEntriesCh, 
           becomeLeaderCh, fAdvCommitIdxCh, netListenerSrvId, 
           propChListenerSrvId, requestVoteSrvId, appendEntriesSrvId, 
           advanceCommitIndexSrvId, becomeLeaderSrvId, fAdvCommitSrvId, 
           crasherSrvId, pc, idx, m, srvId, idx0, m0, srvId0, idx1, srvId1, 
           idx2, srvId2, newCommitIndex, srvId3, srvId4, m1, srvId5, srvId6
        >>

ProcSet == (ServerNetListenerSet) \cup (ServerPropChListenerSet) \cup (ServerRequestVoteSet) \cup (ServerAppendEntriesSet) \cup (ServerAdvanceCommitIndexSet) \cup (ServerBecomeLeaderSet) \cup (ServerFollowerAdvanceCommitIndexSet) \cup (ServerCrasherSet) \cup {0}

Init == (* Global variables *)
        /\ network = [i \in NodeSet |-> [queue |-> <<>>, enabled |-> TRUE]]
        /\ fd = [i \in ServerSet |-> FALSE]
        /\ state = [i \in ServerSet |-> Follower]
        /\ currentTerm = [i \in ServerSet |-> 1]
        /\ commitIndex = [i \in ServerSet |-> 0]
        /\ nextIndex = [i \in ServerSet |-> [j \in ServerSet |-> 1]]
        /\ matchIndex = [i \in ServerSet |-> [j \in ServerSet |-> 0]]
        /\ log = [i \in ServerSet |-> <<>>]
        /\ plog = [i \in ServerSet |-> <<>>]
        /\ votedFor = [i \in ServerSet |-> Nil]
        /\ votesResponded = [i \in ServerSet |-> {}]
        /\ votesGranted = [i \in ServerSet |-> {}]
        /\ leader = [i \in ServerSet |-> Nil]
        /\ propCh = [i \in ServerSet |-> <<>>]
        /\ acctCh = [i \in ServerSet |-> <<>>]
        /\ leaderTimeout = TRUE
        /\ appendEntriesCh = [i \in ServerSet |-> <<>>]
        /\ becomeLeaderCh = [i \in ServerSet |-> IF (NumServers) > (1) THEN <<>> ELSE <<TRUE>>]
        /\ fAdvCommitIdxCh = [i \in ServerSet |-> <<>>]
        /\ netListenerSrvId = [i \in ServerNetListenerSet |-> (i) - ((0) * (NumServers))]
        /\ propChListenerSrvId = [i \in ServerPropChListenerSet |-> (i) - ((1) * (NumServers))]
        /\ requestVoteSrvId = [i \in ServerRequestVoteSet |-> (i) - ((2) * (NumServers))]
        /\ appendEntriesSrvId = [i \in ServerAppendEntriesSet |-> (i) - ((3) * (NumServers))]
        /\ advanceCommitIndexSrvId = [i \in ServerAdvanceCommitIndexSet |-> (i) - ((4) * (NumServers))]
        /\ becomeLeaderSrvId = [i \in ServerBecomeLeaderSet |-> (i) - ((5) * (NumServers))]
        /\ fAdvCommitSrvId = [i \in ServerFollowerAdvanceCommitIndexSet |-> (i) - ((6) * (NumServers))]
        /\ crasherSrvId = [i \in ServerCrasherSet |-> (i) - ((7) * (NumServers))]
        (* Process s0 *)
        /\ idx = [self \in ServerNetListenerSet |-> 1]
        /\ m = [self \in ServerNetListenerSet |-> defaultInitValue]
        /\ srvId = [self \in ServerNetListenerSet |-> (netListenerSrvId)[self]]
        (* Process s1 *)
        /\ idx0 = [self \in ServerPropChListenerSet |-> 1]
        /\ m0 = [self \in ServerPropChListenerSet |-> defaultInitValue]
        /\ srvId0 = [self \in ServerPropChListenerSet |-> (propChListenerSrvId)[self]]
        (* Process s2 *)
        /\ idx1 = [self \in ServerRequestVoteSet |-> 1]
        /\ srvId1 = [self \in ServerRequestVoteSet |-> (requestVoteSrvId)[self]]
        (* Process s3 *)
        /\ idx2 = [self \in ServerAppendEntriesSet |-> defaultInitValue]
        /\ srvId2 = [self \in ServerAppendEntriesSet |-> (appendEntriesSrvId)[self]]
        (* Process s4 *)
        /\ newCommitIndex = [self \in ServerAdvanceCommitIndexSet |-> 0]
        /\ srvId3 = [self \in ServerAdvanceCommitIndexSet |-> (advanceCommitIndexSrvId)[self]]
        (* Process s5 *)
        /\ srvId4 = [self \in ServerBecomeLeaderSet |-> (becomeLeaderSrvId)[self]]
        (* Process s6 *)
        /\ m1 = [self \in ServerFollowerAdvanceCommitIndexSet |-> defaultInitValue]
        /\ srvId5 = [self \in ServerFollowerAdvanceCommitIndexSet |-> (fAdvCommitSrvId)[self]]
        (* Process crasher *)
        /\ srvId6 = [self \in ServerCrasherSet |-> (crasherSrvId)[self]]
        /\ pc = [self \in ProcSet |-> CASE self \in ServerNetListenerSet -> "serverLoop_"
                                        [] self \in ServerPropChListenerSet -> "serverLoop"
                                        [] self \in ServerRequestVoteSet -> "serverRequestVoteLoop"
                                        [] self \in ServerAppendEntriesSet -> "serverAppendEntriesLoop"
                                        [] self \in ServerAdvanceCommitIndexSet -> "serverAdvanceCommitIndexLoop"
                                        [] self \in ServerBecomeLeaderSet -> "serverBecomeLeaderLoop"
                                        [] self \in ServerFollowerAdvanceCommitIndexSet -> "serverFollowerAdvanceCommitIndexLoop"
                                        [] self \in ServerCrasherSet -> "serverCrash"
                                        [] self = 0 -> "sndReq"]

serverLoop_(self) == /\ pc[self] = "serverLoop_"
                     /\ IF TRUE
                           THEN /\ IF ExploreFail
                                      THEN /\ LET yielded_network14 == ((network)[srvId[self]]).enabled IN
                                                IF ~ (yielded_network14)
                                                   THEN /\ FALSE
                                                        /\ Assert(((network)[self]).enabled, 
                                                                  "Failure of assertion at line 1009, column 15.")
                                                        /\ (Len(((network)[self]).queue)) > (0)
                                                        /\ LET readMsg00 == Head(((network)[self]).queue) IN
                                                             /\ network' = [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]]
                                                             /\ LET yielded_network00 == readMsg00 IN
                                                                  /\ m' = [m EXCEPT ![self] = yielded_network00]
                                                                  /\ Assert(((m'[self]).mdest) = (self), 
                                                                            "Failure of assertion at line 1015, column 19.")
                                                                  /\ pc' = [pc EXCEPT ![self] = "handleMsg_"]
                                                   ELSE /\ Assert(((network)[self]).enabled, 
                                                                  "Failure of assertion at line 1020, column 15.")
                                                        /\ (Len(((network)[self]).queue)) > (0)
                                                        /\ LET readMsg01 == Head(((network)[self]).queue) IN
                                                             /\ network' = [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]]
                                                             /\ LET yielded_network01 == readMsg01 IN
                                                                  /\ m' = [m EXCEPT ![self] = yielded_network01]
                                                                  /\ Assert(((m'[self]).mdest) = (self), 
                                                                            "Failure of assertion at line 1026, column 19.")
                                                                  /\ pc' = [pc EXCEPT ![self] = "handleMsg_"]
                                      ELSE /\ Assert(((network)[self]).enabled, 
                                                     "Failure of assertion at line 1033, column 11.")
                                           /\ (Len(((network)[self]).queue)) > (0)
                                           /\ LET readMsg02 == Head(((network)[self]).queue) IN
                                                /\ network' = [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]]
                                                /\ LET yielded_network02 == readMsg02 IN
                                                     /\ m' = [m EXCEPT ![self] = yielded_network02]
                                                     /\ Assert(((m'[self]).mdest) = (self), 
                                                               "Failure of assertion at line 1039, column 15.")
                                                     /\ pc' = [pc EXCEPT ![self] = "handleMsg_"]
                           ELSE /\ pc' = [pc EXCEPT ![self] = "Done"]
                                /\ UNCHANGED << network, m >>
                     /\ UNCHANGED << fd, state, currentTerm, commitIndex, 
                                     nextIndex, matchIndex, log, plog, 
                                     votedFor, votesResponded, votesGranted, 
                                     leader, propCh, acctCh, leaderTimeout, 
                                     appendEntriesCh, becomeLeaderCh, 
                                     fAdvCommitIdxCh, netListenerSrvId, 
                                     propChListenerSrvId, requestVoteSrvId, 
                                     appendEntriesSrvId, 
                                     advanceCommitIndexSrvId, 
                                     becomeLeaderSrvId, fAdvCommitSrvId, 
                                     crasherSrvId, idx, srvId, idx0, m0, 
                                     srvId0, idx1, srvId1, idx2, srvId2, 
                                     newCommitIndex, srvId3, srvId4, m1, 
                                     srvId5, srvId6 >>

handleMsg_(self) == /\ pc[self] = "handleMsg_"
                    /\ IF ExploreFail
                          THEN /\ LET yielded_network15 == ((network)[srvId[self]]).enabled IN
                                    IF ~ (yielded_network15)
                                       THEN /\ FALSE
                                            /\ IF ((m[self]).mtype) = (RequestVoteRequest)
                                                  THEN /\ IF ((m[self]).mterm) > ((currentTerm)[self])
                                                             THEN /\ currentTerm' = [currentTerm EXCEPT ![self] = (m[self]).mterm]
                                                                  /\ state' = [state EXCEPT ![self] = Follower]
                                                                  /\ LET votedFor3 == [votedFor EXCEPT ![self] = Nil] IN
                                                                       /\ leader' = [leader EXCEPT ![self] = Nil]
                                                                       /\ LET i == self IN
                                                                            LET j == (m[self]).msource IN
                                                                              LET logOK == (((m[self]).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m[self]).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m[self]).mlastLogIndex) >= (Len((log)[i])))) IN
                                                                                LET grant == ((((m[self]).mterm) = ((currentTerm')[i])) /\ (logOK)) /\ (((votedFor3)[self]) \in ({Nil, j})) IN
                                                                                  /\ Assert(((m[self]).mterm) <= ((currentTerm')[i]), 
                                                                                            "Failure of assertion at line 1064, column 21.")
                                                                                  /\ IF grant
                                                                                        THEN /\ votedFor' = [votedFor3 EXCEPT ![i] = j]
                                                                                             /\ \/ /\ LET value20 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm')[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                        /\ ((network)[j]).enabled
                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value20), enabled |-> ((network)[j]).enabled]]
                                                                                                        /\ IF Debug
                                                                                                              THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                              ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                \/ /\ LET yielded_fd6 == (fd)[j] IN
                                                                                                        /\ yielded_fd6
                                                                                                        /\ IF Debug
                                                                                                              THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                              ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                   /\ UNCHANGED network
                                                                                        ELSE /\ \/ /\ LET value21 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm')[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                        /\ ((network)[j]).enabled
                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value21), enabled |-> ((network)[j]).enabled]]
                                                                                                        /\ IF Debug
                                                                                                              THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                   /\ votedFor' = votedFor3
                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                              ELSE /\ votedFor' = votedFor3
                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                \/ /\ LET yielded_fd7 == (fd)[j] IN
                                                                                                        /\ yielded_fd7
                                                                                                        /\ IF Debug
                                                                                                              THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                   /\ votedFor' = votedFor3
                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                              ELSE /\ votedFor' = votedFor3
                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                   /\ UNCHANGED network
                                                             ELSE /\ LET i == self IN
                                                                       LET j == (m[self]).msource IN
                                                                         LET logOK == (((m[self]).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m[self]).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m[self]).mlastLogIndex) >= (Len((log)[i])))) IN
                                                                           LET grant == ((((m[self]).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j})) IN
                                                                             /\ Assert(((m[self]).mterm) <= ((currentTerm)[i]), 
                                                                                       "Failure of assertion at line 1128, column 19.")
                                                                             /\ IF grant
                                                                                   THEN /\ votedFor' = [votedFor EXCEPT ![i] = j]
                                                                                        /\ \/ /\ LET value22 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                   /\ ((network)[j]).enabled
                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value22), enabled |-> ((network)[j]).enabled]]
                                                                                                   /\ IF Debug
                                                                                                         THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                         ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                           \/ /\ LET yielded_fd8 == (fd)[j] IN
                                                                                                   /\ yielded_fd8
                                                                                                   /\ IF Debug
                                                                                                         THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                         ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                              /\ UNCHANGED network
                                                                                   ELSE /\ \/ /\ LET value23 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                   /\ ((network)[j]).enabled
                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value23), enabled |-> ((network)[j]).enabled]]
                                                                                                   /\ IF Debug
                                                                                                         THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                         ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                           \/ /\ LET yielded_fd9 == (fd)[j] IN
                                                                                                   /\ yielded_fd9
                                                                                                   /\ IF Debug
                                                                                                         THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                         ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                              /\ UNCHANGED network
                                                                                        /\ UNCHANGED votedFor
                                                                  /\ UNCHANGED << state, 
                                                                                  currentTerm, 
                                                                                  leader >>
                                                       /\ UNCHANGED << nextIndex, 
                                                                       matchIndex, 
                                                                       log, 
                                                                       plog, 
                                                                       votesResponded, 
                                                                       votesGranted, 
                                                                       leaderTimeout, 
                                                                       appendEntriesCh, 
                                                                       becomeLeaderCh, 
                                                                       fAdvCommitIdxCh >>
                                                  ELSE /\ IF ((m[self]).mtype) = (RequestVoteResponse)
                                                             THEN /\ IF ((m[self]).mterm) > ((currentTerm)[self])
                                                                        THEN /\ currentTerm' = [currentTerm EXCEPT ![self] = (m[self]).mterm]
                                                                             /\ state' = [state EXCEPT ![self] = Follower]
                                                                             /\ votedFor' = [votedFor EXCEPT ![self] = Nil]
                                                                             /\ leader' = [leader EXCEPT ![self] = Nil]
                                                                             /\ IF ((m[self]).mterm) < ((currentTerm')[self])
                                                                                   THEN /\ TRUE
                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                        /\ UNCHANGED << votesResponded, 
                                                                                                        votesGranted, 
                                                                                                        becomeLeaderCh >>
                                                                                   ELSE /\ LET i == self IN
                                                                                             LET j == (m[self]).msource IN
                                                                                               /\ Assert(((m[self]).mterm) = ((currentTerm')[i]), 
                                                                                                         "Failure of assertion at line 1196, column 23.")
                                                                                               /\ votesResponded' = [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})]
                                                                                               /\ IF (m[self]).mvoteGranted
                                                                                                     THEN /\ votesGranted' = [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})]
                                                                                                          /\ IF (((state')[i]) = (Candidate)) /\ (isQuorum((votesGranted')[i]))
                                                                                                                THEN /\ LET value00 == TRUE IN
                                                                                                                          /\ (Len((becomeLeaderCh)[i])) < (BufferSize)
                                                                                                                          /\ becomeLeaderCh' = [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value00)]
                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                     /\ UNCHANGED becomeLeaderCh
                                                                                                     ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                          /\ UNCHANGED << votesGranted, 
                                                                                                                          becomeLeaderCh >>
                                                                        ELSE /\ IF ((m[self]).mterm) < ((currentTerm)[self])
                                                                                   THEN /\ TRUE
                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                        /\ UNCHANGED << votesResponded, 
                                                                                                        votesGranted, 
                                                                                                        becomeLeaderCh >>
                                                                                   ELSE /\ LET i == self IN
                                                                                             LET j == (m[self]).msource IN
                                                                                               /\ Assert(((m[self]).mterm) = ((currentTerm)[i]), 
                                                                                                         "Failure of assertion at line 1223, column 23.")
                                                                                               /\ votesResponded' = [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})]
                                                                                               /\ IF (m[self]).mvoteGranted
                                                                                                     THEN /\ votesGranted' = [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})]
                                                                                                          /\ IF (((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted')[i]))
                                                                                                                THEN /\ LET value01 == TRUE IN
                                                                                                                          /\ (Len((becomeLeaderCh)[i])) < (BufferSize)
                                                                                                                          /\ becomeLeaderCh' = [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value01)]
                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                     /\ UNCHANGED becomeLeaderCh
                                                                                                     ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                          /\ UNCHANGED << votesGranted, 
                                                                                                                          becomeLeaderCh >>
                                                                             /\ UNCHANGED << state, 
                                                                                             currentTerm, 
                                                                                             votedFor, 
                                                                                             leader >>
                                                                  /\ UNCHANGED << network, 
                                                                                  nextIndex, 
                                                                                  matchIndex, 
                                                                                  log, 
                                                                                  plog, 
                                                                                  leaderTimeout, 
                                                                                  appendEntriesCh, 
                                                                                  fAdvCommitIdxCh >>
                                                             ELSE /\ IF ((m[self]).mtype) = (AppendEntriesRequest)
                                                                        THEN /\ IF ((m[self]).mterm) > ((currentTerm)[self])
                                                                                   THEN /\ currentTerm' = [currentTerm EXCEPT ![self] = (m[self]).mterm]
                                                                                        /\ LET state3 == [state EXCEPT ![self] = Follower] IN
                                                                                             /\ votedFor' = [votedFor EXCEPT ![self] = Nil]
                                                                                             /\ LET leader3 == [leader EXCEPT ![self] = Nil] IN
                                                                                                  LET i == self IN
                                                                                                    LET j == (m[self]).msource IN
                                                                                                      LET logOK == (((m[self]).mprevLogIndex) = (0)) \/ (((((m[self]).mprevLogIndex) > (0)) /\ (((m[self]).mprevLogIndex) <= (Len((log)[i])))) /\ (((m[self]).mprevLogTerm) = ((((log)[i])[(m[self]).mprevLogIndex]).term))) IN
                                                                                                        /\ Assert(((m[self]).mterm) <= ((currentTerm')[i]), 
                                                                                                                  "Failure of assertion at line 1254, column 25.")
                                                                                                        /\ IF ((m[self]).mterm) = ((currentTerm')[i])
                                                                                                              THEN /\ leader' = [leader3 EXCEPT ![i] = (m[self]).msource]
                                                                                                                   /\ leaderTimeout' = LeaderTimeoutReset
                                                                                                                   /\ IF (((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Candidate))
                                                                                                                         THEN /\ state' = [state3 EXCEPT ![i] = Follower]
                                                                                                                              /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                    THEN /\ \/ /\ LET value110 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value110), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                            \/ /\ LET yielded_fd00 == (fd)[j] IN
                                                                                                                                                    /\ yielded_fd00
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                               /\ UNCHANGED network
                                                                                                                                         /\ UNCHANGED << log, 
                                                                                                                                                         plog, 
                                                                                                                                                         fAdvCommitIdxCh >>
                                                                                                                                    ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                   "Failure of assertion at line 1275, column 31.")
                                                                                                                                         /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                              LET value24 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                                IF ((value24).cmd) = (LogConcat)
                                                                                                                                                   THEN /\ LET plog24 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value24).entries)] IN
                                                                                                                                                             LET log24 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                               LET value30 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                 IF ((value30).cmd) = (LogConcat)
                                                                                                                                                                    THEN /\ plog' = [plog24 EXCEPT ![i] = ((plog24)[i]) \o ((value30).entries)]
                                                                                                                                                                         /\ log' = [log24 EXCEPT ![i] = ((log24)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                         /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                   "Failure of assertion at line 1289, column 39.")
                                                                                                                                                                         /\ LET value40 == m[self] IN
                                                                                                                                                                              /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                              /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value40)]
                                                                                                                                                                              /\ \/ /\ LET value50 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                         /\ ((network)[j]).enabled
                                                                                                                                                                                         /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                         /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value50), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                 \/ /\ LET yielded_fd10 == (fd)[j] IN
                                                                                                                                                                                         /\ yielded_fd10
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                    /\ UNCHANGED network
                                                                                                                                                                    ELSE /\ IF ((value30).cmd) = (LogPop)
                                                                                                                                                                               THEN /\ plog' = [plog24 EXCEPT ![i] = SubSeq((plog24)[i], 1, (Len((plog24)[i])) - ((value30).cnt))]
                                                                                                                                                                                    /\ log' = [log24 EXCEPT ![i] = ((log24)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1311, column 41.")
                                                                                                                                                                                    /\ LET value41 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value41)]
                                                                                                                                                                                         /\ \/ /\ LET value51 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value51), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd11 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd11
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ log' = [log24 EXCEPT ![i] = ((log24)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1331, column 41.")
                                                                                                                                                                                    /\ LET value42 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value42)]
                                                                                                                                                                                         /\ \/ /\ LET value52 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value52), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ plog' = plog24
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd12 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd12
                                                                                                                                                                                                    /\ plog' = plog24
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                   ELSE /\ IF ((value24).cmd) = (LogPop)
                                                                                                                                                              THEN /\ LET plog25 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value24).cnt))] IN
                                                                                                                                                                        LET log25 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                          LET value31 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                            IF ((value31).cmd) = (LogConcat)
                                                                                                                                                                               THEN /\ plog' = [plog25 EXCEPT ![i] = ((plog25)[i]) \o ((value31).entries)]
                                                                                                                                                                                    /\ log' = [log25 EXCEPT ![i] = ((log25)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1364, column 41.")
                                                                                                                                                                                    /\ LET value43 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value43)]
                                                                                                                                                                                         /\ \/ /\ LET value53 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value53), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd13 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd13
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ IF ((value31).cmd) = (LogPop)
                                                                                                                                                                                          THEN /\ plog' = [plog25 EXCEPT ![i] = SubSeq((plog25)[i], 1, (Len((plog25)[i])) - ((value31).cnt))]
                                                                                                                                                                                               /\ log' = [log25 EXCEPT ![i] = ((log25)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 1386, column 43.")
                                                                                                                                                                                               /\ LET value44 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value44)]
                                                                                                                                                                                                    /\ \/ /\ LET value54 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value54), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd14 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd14
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                                                          ELSE /\ log' = [log25 EXCEPT ![i] = ((log25)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 1406, column 43.")
                                                                                                                                                                                               /\ LET value45 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value45)]
                                                                                                                                                                                                    /\ \/ /\ LET value55 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value55), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ plog' = plog25
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd15 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd15
                                                                                                                                                                                                               /\ plog' = plog25
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                              ELSE /\ LET log26 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                        LET value32 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                          IF ((value32).cmd) = (LogConcat)
                                                                                                                                                                             THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value32).entries)]
                                                                                                                                                                                  /\ log' = [log26 EXCEPT ![i] = ((log26)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                            "Failure of assertion at line 1437, column 41.")
                                                                                                                                                                                  /\ LET value46 == m[self] IN
                                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value46)]
                                                                                                                                                                                       /\ \/ /\ LET value56 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value56), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                          \/ /\ LET yielded_fd16 == (fd)[j] IN
                                                                                                                                                                                                  /\ yielded_fd16
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                                             ELSE /\ IF ((value32).cmd) = (LogPop)
                                                                                                                                                                                        THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value32).cnt))]
                                                                                                                                                                                             /\ log' = [log26 EXCEPT ![i] = ((log26)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 1459, column 43.")
                                                                                                                                                                                             /\ LET value47 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value47)]
                                                                                                                                                                                                  /\ \/ /\ LET value57 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value57), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd17 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd17
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                        ELSE /\ log' = [log26 EXCEPT ![i] = ((log26)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 1479, column 43.")
                                                                                                                                                                                             /\ LET value48 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value48)]
                                                                                                                                                                                                  /\ \/ /\ LET value58 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value58), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd18 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd18
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                             /\ plog' = plog
                                                                                                                         ELSE /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                    THEN /\ \/ /\ LET value111 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value111), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                    /\ state' = state3
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                            \/ /\ LET yielded_fd01 == (fd)[j] IN
                                                                                                                                                    /\ yielded_fd01
                                                                                                                                                    /\ state' = state3
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                               /\ UNCHANGED network
                                                                                                                                         /\ UNCHANGED << log, 
                                                                                                                                                         plog, 
                                                                                                                                                         fAdvCommitIdxCh >>
                                                                                                                                    ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                   "Failure of assertion at line 1522, column 31.")
                                                                                                                                         /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                              LET value25 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                                IF ((value25).cmd) = (LogConcat)
                                                                                                                                                   THEN /\ LET plog26 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value25).entries)] IN
                                                                                                                                                             LET log27 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                               LET value33 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                 IF ((value33).cmd) = (LogConcat)
                                                                                                                                                                    THEN /\ plog' = [plog26 EXCEPT ![i] = ((plog26)[i]) \o ((value33).entries)]
                                                                                                                                                                         /\ log' = [log27 EXCEPT ![i] = ((log27)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                         /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                   "Failure of assertion at line 1536, column 39.")
                                                                                                                                                                         /\ LET value49 == m[self] IN
                                                                                                                                                                              /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                              /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value49)]
                                                                                                                                                                              /\ \/ /\ LET value59 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                         /\ ((network)[j]).enabled
                                                                                                                                                                                         /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                         /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value59), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                         /\ state' = state3
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                 \/ /\ LET yielded_fd19 == (fd)[j] IN
                                                                                                                                                                                         /\ yielded_fd19
                                                                                                                                                                                         /\ state' = state3
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                    /\ UNCHANGED network
                                                                                                                                                                    ELSE /\ IF ((value33).cmd) = (LogPop)
                                                                                                                                                                               THEN /\ plog' = [plog26 EXCEPT ![i] = SubSeq((plog26)[i], 1, (Len((plog26)[i])) - ((value33).cnt))]
                                                                                                                                                                                    /\ log' = [log27 EXCEPT ![i] = ((log27)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1560, column 41.")
                                                                                                                                                                                    /\ LET value410 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value410)]
                                                                                                                                                                                         /\ \/ /\ LET value510 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value510), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd110 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd110
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ log' = [log27 EXCEPT ![i] = ((log27)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1582, column 41.")
                                                                                                                                                                                    /\ LET value411 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value411)]
                                                                                                                                                                                         /\ \/ /\ LET value511 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value511), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ plog' = plog26
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd111 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd111
                                                                                                                                                                                                    /\ plog' = plog26
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                   ELSE /\ IF ((value25).cmd) = (LogPop)
                                                                                                                                                              THEN /\ LET plog27 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value25).cnt))] IN
                                                                                                                                                                        LET log28 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                          LET value34 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                            IF ((value34).cmd) = (LogConcat)
                                                                                                                                                                               THEN /\ plog' = [plog27 EXCEPT ![i] = ((plog27)[i]) \o ((value34).entries)]
                                                                                                                                                                                    /\ log' = [log28 EXCEPT ![i] = ((log28)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1617, column 41.")
                                                                                                                                                                                    /\ LET value412 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value412)]
                                                                                                                                                                                         /\ \/ /\ LET value512 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value512), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd112 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd112
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ IF ((value34).cmd) = (LogPop)
                                                                                                                                                                                          THEN /\ plog' = [plog27 EXCEPT ![i] = SubSeq((plog27)[i], 1, (Len((plog27)[i])) - ((value34).cnt))]
                                                                                                                                                                                               /\ log' = [log28 EXCEPT ![i] = ((log28)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 1641, column 43.")
                                                                                                                                                                                               /\ LET value413 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value413)]
                                                                                                                                                                                                    /\ \/ /\ LET value513 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value513), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd113 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd113
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                                                          ELSE /\ log' = [log28 EXCEPT ![i] = ((log28)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 1663, column 43.")
                                                                                                                                                                                               /\ LET value414 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value414)]
                                                                                                                                                                                                    /\ \/ /\ LET value514 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value514), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ plog' = plog27
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd114 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd114
                                                                                                                                                                                                               /\ plog' = plog27
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                              ELSE /\ LET log29 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                        LET value35 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                          IF ((value35).cmd) = (LogConcat)
                                                                                                                                                                             THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value35).entries)]
                                                                                                                                                                                  /\ log' = [log29 EXCEPT ![i] = ((log29)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                            "Failure of assertion at line 1696, column 41.")
                                                                                                                                                                                  /\ LET value415 == m[self] IN
                                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value415)]
                                                                                                                                                                                       /\ \/ /\ LET value515 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value515), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                  /\ state' = state3
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                          \/ /\ LET yielded_fd115 == (fd)[j] IN
                                                                                                                                                                                                  /\ yielded_fd115
                                                                                                                                                                                                  /\ state' = state3
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                                             ELSE /\ IF ((value35).cmd) = (LogPop)
                                                                                                                                                                                        THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value35).cnt))]
                                                                                                                                                                                             /\ log' = [log29 EXCEPT ![i] = ((log29)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 1720, column 43.")
                                                                                                                                                                                             /\ LET value416 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value416)]
                                                                                                                                                                                                  /\ \/ /\ LET value516 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value516), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd116 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd116
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                        ELSE /\ log' = [log29 EXCEPT ![i] = ((log29)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 1742, column 43.")
                                                                                                                                                                                             /\ LET value417 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value417)]
                                                                                                                                                                                                  /\ \/ /\ LET value517 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value517), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd117 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd117
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                             /\ plog' = plog
                                                                                                              ELSE /\ IF (((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Candidate))
                                                                                                                         THEN /\ state' = [state3 EXCEPT ![i] = Follower]
                                                                                                                              /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                    THEN /\ \/ /\ LET value112 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value112), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                            \/ /\ LET yielded_fd02 == (fd)[j] IN
                                                                                                                                                    /\ yielded_fd02
                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                               /\ UNCHANGED network
                                                                                                                                         /\ UNCHANGED << log, 
                                                                                                                                                         plog, 
                                                                                                                                                         fAdvCommitIdxCh >>
                                                                                                                                    ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                   "Failure of assertion at line 1790, column 31.")
                                                                                                                                         /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                              LET value26 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                                IF ((value26).cmd) = (LogConcat)
                                                                                                                                                   THEN /\ LET plog28 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value26).entries)] IN
                                                                                                                                                             LET log30 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                               LET value36 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                 IF ((value36).cmd) = (LogConcat)
                                                                                                                                                                    THEN /\ plog' = [plog28 EXCEPT ![i] = ((plog28)[i]) \o ((value36).entries)]
                                                                                                                                                                         /\ log' = [log30 EXCEPT ![i] = ((log30)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                         /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                   "Failure of assertion at line 1804, column 39.")
                                                                                                                                                                         /\ LET value418 == m[self] IN
                                                                                                                                                                              /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                              /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value418)]
                                                                                                                                                                              /\ \/ /\ LET value518 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                         /\ ((network)[j]).enabled
                                                                                                                                                                                         /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                         /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value518), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                         /\ leader' = leader3
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                 \/ /\ LET yielded_fd118 == (fd)[j] IN
                                                                                                                                                                                         /\ yielded_fd118
                                                                                                                                                                                         /\ leader' = leader3
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                    /\ UNCHANGED network
                                                                                                                                                                    ELSE /\ IF ((value36).cmd) = (LogPop)
                                                                                                                                                                               THEN /\ plog' = [plog28 EXCEPT ![i] = SubSeq((plog28)[i], 1, (Len((plog28)[i])) - ((value36).cnt))]
                                                                                                                                                                                    /\ log' = [log30 EXCEPT ![i] = ((log30)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1828, column 41.")
                                                                                                                                                                                    /\ LET value419 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value419)]
                                                                                                                                                                                         /\ \/ /\ LET value519 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value519), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd119 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd119
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ log' = [log30 EXCEPT ![i] = ((log30)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1850, column 41.")
                                                                                                                                                                                    /\ LET value420 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value420)]
                                                                                                                                                                                         /\ \/ /\ LET value520 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value520), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ plog' = plog28
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd120 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd120
                                                                                                                                                                                                    /\ plog' = plog28
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                   ELSE /\ IF ((value26).cmd) = (LogPop)
                                                                                                                                                              THEN /\ LET plog29 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value26).cnt))] IN
                                                                                                                                                                        LET log31 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                          LET value37 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                            IF ((value37).cmd) = (LogConcat)
                                                                                                                                                                               THEN /\ plog' = [plog29 EXCEPT ![i] = ((plog29)[i]) \o ((value37).entries)]
                                                                                                                                                                                    /\ log' = [log31 EXCEPT ![i] = ((log31)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 1885, column 41.")
                                                                                                                                                                                    /\ LET value421 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value421)]
                                                                                                                                                                                         /\ \/ /\ LET value521 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value521), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd121 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd121
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ IF ((value37).cmd) = (LogPop)
                                                                                                                                                                                          THEN /\ plog' = [plog29 EXCEPT ![i] = SubSeq((plog29)[i], 1, (Len((plog29)[i])) - ((value37).cnt))]
                                                                                                                                                                                               /\ log' = [log31 EXCEPT ![i] = ((log31)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 1909, column 43.")
                                                                                                                                                                                               /\ LET value422 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value422)]
                                                                                                                                                                                                    /\ \/ /\ LET value522 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value522), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd122 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd122
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                                                          ELSE /\ log' = [log31 EXCEPT ![i] = ((log31)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 1931, column 43.")
                                                                                                                                                                                               /\ LET value423 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value423)]
                                                                                                                                                                                                    /\ \/ /\ LET value523 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value523), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ plog' = plog29
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd123 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd123
                                                                                                                                                                                                               /\ plog' = plog29
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                              ELSE /\ LET log32 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                        LET value38 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                          IF ((value38).cmd) = (LogConcat)
                                                                                                                                                                             THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value38).entries)]
                                                                                                                                                                                  /\ log' = [log32 EXCEPT ![i] = ((log32)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                            "Failure of assertion at line 1964, column 41.")
                                                                                                                                                                                  /\ LET value424 == m[self] IN
                                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value424)]
                                                                                                                                                                                       /\ \/ /\ LET value524 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value524), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                  /\ leader' = leader3
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                          \/ /\ LET yielded_fd124 == (fd)[j] IN
                                                                                                                                                                                                  /\ yielded_fd124
                                                                                                                                                                                                  /\ leader' = leader3
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                                             ELSE /\ IF ((value38).cmd) = (LogPop)
                                                                                                                                                                                        THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value38).cnt))]
                                                                                                                                                                                             /\ log' = [log32 EXCEPT ![i] = ((log32)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 1988, column 43.")
                                                                                                                                                                                             /\ LET value425 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value425)]
                                                                                                                                                                                                  /\ \/ /\ LET value525 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value525), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd125 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd125
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                        ELSE /\ log' = [log32 EXCEPT ![i] = ((log32)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 2010, column 43.")
                                                                                                                                                                                             /\ LET value426 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value426)]
                                                                                                                                                                                                  /\ \/ /\ LET value526 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value526), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd126 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd126
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                             /\ plog' = plog
                                                                                                                         ELSE /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                    THEN /\ \/ /\ LET value113 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value113), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                    /\ state' = state3
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                            \/ /\ LET yielded_fd03 == (fd)[j] IN
                                                                                                                                                    /\ yielded_fd03
                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                    /\ state' = state3
                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                               /\ UNCHANGED network
                                                                                                                                         /\ UNCHANGED << log, 
                                                                                                                                                         plog, 
                                                                                                                                                         fAdvCommitIdxCh >>
                                                                                                                                    ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                   "Failure of assertion at line 2057, column 31.")
                                                                                                                                         /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                              LET value27 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                                IF ((value27).cmd) = (LogConcat)
                                                                                                                                                   THEN /\ LET plog30 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value27).entries)] IN
                                                                                                                                                             LET log33 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                               LET value39 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                 IF ((value39).cmd) = (LogConcat)
                                                                                                                                                                    THEN /\ plog' = [plog30 EXCEPT ![i] = ((plog30)[i]) \o ((value39).entries)]
                                                                                                                                                                         /\ log' = [log33 EXCEPT ![i] = ((log33)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                         /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                   "Failure of assertion at line 2071, column 39.")
                                                                                                                                                                         /\ LET value427 == m[self] IN
                                                                                                                                                                              /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                              /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value427)]
                                                                                                                                                                              /\ \/ /\ LET value527 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                         /\ ((network)[j]).enabled
                                                                                                                                                                                         /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                         /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value527), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                         /\ leader' = leader3
                                                                                                                                                                                         /\ state' = state3
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                 \/ /\ LET yielded_fd127 == (fd)[j] IN
                                                                                                                                                                                         /\ yielded_fd127
                                                                                                                                                                                         /\ leader' = leader3
                                                                                                                                                                                         /\ state' = state3
                                                                                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                    /\ UNCHANGED network
                                                                                                                                                                    ELSE /\ IF ((value39).cmd) = (LogPop)
                                                                                                                                                                               THEN /\ plog' = [plog30 EXCEPT ![i] = SubSeq((plog30)[i], 1, (Len((plog30)[i])) - ((value39).cnt))]
                                                                                                                                                                                    /\ log' = [log33 EXCEPT ![i] = ((log33)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 2097, column 41.")
                                                                                                                                                                                    /\ LET value428 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value428)]
                                                                                                                                                                                         /\ \/ /\ LET value528 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value528), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd128 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd128
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ log' = [log33 EXCEPT ![i] = ((log33)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 2121, column 41.")
                                                                                                                                                                                    /\ LET value429 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value429)]
                                                                                                                                                                                         /\ \/ /\ LET value529 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value529), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ plog' = plog30
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd129 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd129
                                                                                                                                                                                                    /\ plog' = plog30
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                   ELSE /\ IF ((value27).cmd) = (LogPop)
                                                                                                                                                              THEN /\ LET plog31 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value27).cnt))] IN
                                                                                                                                                                        LET log34 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                          LET value310 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                            IF ((value310).cmd) = (LogConcat)
                                                                                                                                                                               THEN /\ plog' = [plog31 EXCEPT ![i] = ((plog31)[i]) \o ((value310).entries)]
                                                                                                                                                                                    /\ log' = [log34 EXCEPT ![i] = ((log34)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                    /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                              "Failure of assertion at line 2158, column 41.")
                                                                                                                                                                                    /\ LET value430 == m[self] IN
                                                                                                                                                                                         /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                         /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value430)]
                                                                                                                                                                                         /\ \/ /\ LET value530 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                    /\ ((network)[j]).enabled
                                                                                                                                                                                                    /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                    /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value530), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                            \/ /\ LET yielded_fd130 == (fd)[j] IN
                                                                                                                                                                                                    /\ yielded_fd130
                                                                                                                                                                                                    /\ leader' = leader3
                                                                                                                                                                                                    /\ state' = state3
                                                                                                                                                                                                    /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                               /\ UNCHANGED network
                                                                                                                                                                               ELSE /\ IF ((value310).cmd) = (LogPop)
                                                                                                                                                                                          THEN /\ plog' = [plog31 EXCEPT ![i] = SubSeq((plog31)[i], 1, (Len((plog31)[i])) - ((value310).cnt))]
                                                                                                                                                                                               /\ log' = [log34 EXCEPT ![i] = ((log34)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 2184, column 43.")
                                                                                                                                                                                               /\ LET value431 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value431)]
                                                                                                                                                                                                    /\ \/ /\ LET value531 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value531), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd131 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd131
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                                                          ELSE /\ log' = [log34 EXCEPT ![i] = ((log34)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                               /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                         "Failure of assertion at line 2208, column 43.")
                                                                                                                                                                                               /\ LET value432 == m[self] IN
                                                                                                                                                                                                    /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                    /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value432)]
                                                                                                                                                                                                    /\ \/ /\ LET value532 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                               /\ ((network)[j]).enabled
                                                                                                                                                                                                               /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                               /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value532), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                               /\ plog' = plog31
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                       \/ /\ LET yielded_fd132 == (fd)[j] IN
                                                                                                                                                                                                               /\ yielded_fd132
                                                                                                                                                                                                               /\ plog' = plog31
                                                                                                                                                                                                               /\ leader' = leader3
                                                                                                                                                                                                               /\ state' = state3
                                                                                                                                                                                                               /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                          /\ UNCHANGED network
                                                                                                                                                              ELSE /\ LET log35 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                        LET value311 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                          IF ((value311).cmd) = (LogConcat)
                                                                                                                                                                             THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value311).entries)]
                                                                                                                                                                                  /\ log' = [log35 EXCEPT ![i] = ((log35)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                            "Failure of assertion at line 2243, column 41.")
                                                                                                                                                                                  /\ LET value433 == m[self] IN
                                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value433)]
                                                                                                                                                                                       /\ \/ /\ LET value533 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value533), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                  /\ leader' = leader3
                                                                                                                                                                                                  /\ state' = state3
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                          \/ /\ LET yielded_fd133 == (fd)[j] IN
                                                                                                                                                                                                  /\ yielded_fd133
                                                                                                                                                                                                  /\ leader' = leader3
                                                                                                                                                                                                  /\ state' = state3
                                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                                             ELSE /\ IF ((value311).cmd) = (LogPop)
                                                                                                                                                                                        THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value311).cnt))]
                                                                                                                                                                                             /\ log' = [log35 EXCEPT ![i] = ((log35)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 2269, column 43.")
                                                                                                                                                                                             /\ LET value434 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value434)]
                                                                                                                                                                                                  /\ \/ /\ LET value534 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value534), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd134 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd134
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                        ELSE /\ log' = [log35 EXCEPT ![i] = ((log35)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                       "Failure of assertion at line 2293, column 43.")
                                                                                                                                                                                             /\ LET value435 == m[self] IN
                                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value435)]
                                                                                                                                                                                                  /\ \/ /\ LET value535 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value535), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                     \/ /\ LET yielded_fd135 == (fd)[j] IN
                                                                                                                                                                                                             /\ yielded_fd135
                                                                                                                                                                                                             /\ leader' = leader3
                                                                                                                                                                                                             /\ state' = state3
                                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                                             /\ plog' = plog
                                                                                                                   /\ UNCHANGED leaderTimeout
                                                                                   ELSE /\ LET i == self IN
                                                                                             LET j == (m[self]).msource IN
                                                                                               LET logOK == (((m[self]).mprevLogIndex) = (0)) \/ (((((m[self]).mprevLogIndex) > (0)) /\ (((m[self]).mprevLogIndex) <= (Len((log)[i])))) /\ (((m[self]).mprevLogTerm) = ((((log)[i])[(m[self]).mprevLogIndex]).term))) IN
                                                                                                 /\ Assert(((m[self]).mterm) <= ((currentTerm)[i]), 
                                                                                                           "Failure of assertion at line 2332, column 23.")
                                                                                                 /\ IF ((m[self]).mterm) = ((currentTerm)[i])
                                                                                                       THEN /\ leader' = [leader EXCEPT ![i] = (m[self]).msource]
                                                                                                            /\ leaderTimeout' = LeaderTimeoutReset
                                                                                                            /\ IF (((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))
                                                                                                                  THEN /\ state' = [state EXCEPT ![i] = Follower]
                                                                                                                       /\ IF (((m[self]).mterm) < ((currentTerm)[i])) \/ (((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                             THEN /\ \/ /\ LET value114 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value114), enabled |-> ((network)[j]).enabled]]
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                     \/ /\ LET yielded_fd04 == (fd)[j] IN
                                                                                                                                             /\ yielded_fd04
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                        /\ UNCHANGED network
                                                                                                                                  /\ UNCHANGED << log, 
                                                                                                                                                  plog, 
                                                                                                                                                  fAdvCommitIdxCh >>
                                                                                                                             ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                            "Failure of assertion at line 2353, column 29.")
                                                                                                                                  /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                       LET value28 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                         IF ((value28).cmd) = (LogConcat)
                                                                                                                                            THEN /\ LET plog32 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value28).entries)] IN
                                                                                                                                                      LET log36 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                        LET value312 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                          IF ((value312).cmd) = (LogConcat)
                                                                                                                                                             THEN /\ plog' = [plog32 EXCEPT ![i] = ((plog32)[i]) \o ((value312).entries)]
                                                                                                                                                                  /\ log' = [log36 EXCEPT ![i] = ((log36)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                            "Failure of assertion at line 2367, column 37.")
                                                                                                                                                                  /\ LET value436 == m[self] IN
                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value436)]
                                                                                                                                                                       /\ \/ /\ LET value536 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value536), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                          \/ /\ LET yielded_fd136 == (fd)[j] IN
                                                                                                                                                                                  /\ yielded_fd136
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                             ELSE /\ IF ((value312).cmd) = (LogPop)
                                                                                                                                                                        THEN /\ plog' = [plog32 EXCEPT ![i] = SubSeq((plog32)[i], 1, (Len((plog32)[i])) - ((value312).cnt))]
                                                                                                                                                                             /\ log' = [log36 EXCEPT ![i] = ((log36)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2389, column 39.")
                                                                                                                                                                             /\ LET value437 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value437)]
                                                                                                                                                                                  /\ \/ /\ LET value537 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value537), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd137 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd137
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ log' = [log36 EXCEPT ![i] = ((log36)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2409, column 39.")
                                                                                                                                                                             /\ LET value438 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value438)]
                                                                                                                                                                                  /\ \/ /\ LET value538 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value538), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ plog' = plog32
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd138 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd138
                                                                                                                                                                                             /\ plog' = plog32
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                            ELSE /\ IF ((value28).cmd) = (LogPop)
                                                                                                                                                       THEN /\ LET plog33 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value28).cnt))] IN
                                                                                                                                                                 LET log37 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                   LET value313 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                     IF ((value313).cmd) = (LogConcat)
                                                                                                                                                                        THEN /\ plog' = [plog33 EXCEPT ![i] = ((plog33)[i]) \o ((value313).entries)]
                                                                                                                                                                             /\ log' = [log37 EXCEPT ![i] = ((log37)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2442, column 39.")
                                                                                                                                                                             /\ LET value439 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value439)]
                                                                                                                                                                                  /\ \/ /\ LET value539 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value539), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd139 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd139
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ IF ((value313).cmd) = (LogPop)
                                                                                                                                                                                   THEN /\ plog' = [plog33 EXCEPT ![i] = SubSeq((plog33)[i], 1, (Len((plog33)[i])) - ((value313).cnt))]
                                                                                                                                                                                        /\ log' = [log37 EXCEPT ![i] = ((log37)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                  "Failure of assertion at line 2464, column 41.")
                                                                                                                                                                                        /\ LET value440 == m[self] IN
                                                                                                                                                                                             /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                             /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value440)]
                                                                                                                                                                                             /\ \/ /\ LET value540 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value540), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                \/ /\ LET yielded_fd140 == (fd)[j] IN
                                                                                                                                                                                                        /\ yielded_fd140
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                                                   ELSE /\ log' = [log37 EXCEPT ![i] = ((log37)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                  "Failure of assertion at line 2484, column 41.")
                                                                                                                                                                                        /\ LET value441 == m[self] IN
                                                                                                                                                                                             /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                             /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value441)]
                                                                                                                                                                                             /\ \/ /\ LET value541 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value541), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                        /\ plog' = plog33
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                \/ /\ LET yielded_fd141 == (fd)[j] IN
                                                                                                                                                                                                        /\ yielded_fd141
                                                                                                                                                                                                        /\ plog' = plog33
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                       ELSE /\ LET log38 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                 LET value314 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                   IF ((value314).cmd) = (LogConcat)
                                                                                                                                                                      THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value314).entries)]
                                                                                                                                                                           /\ log' = [log38 EXCEPT ![i] = ((log38)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                           /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                     "Failure of assertion at line 2515, column 39.")
                                                                                                                                                                           /\ LET value442 == m[self] IN
                                                                                                                                                                                /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value442)]
                                                                                                                                                                                /\ \/ /\ LET value542 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                           /\ ((network)[j]).enabled
                                                                                                                                                                                           /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                           /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value542), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                           /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                   \/ /\ LET yielded_fd142 == (fd)[j] IN
                                                                                                                                                                                           /\ yielded_fd142
                                                                                                                                                                                           /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                      /\ UNCHANGED network
                                                                                                                                                                      ELSE /\ IF ((value314).cmd) = (LogPop)
                                                                                                                                                                                 THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value314).cnt))]
                                                                                                                                                                                      /\ log' = [log38 EXCEPT ![i] = ((log38)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                      /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                "Failure of assertion at line 2537, column 41.")
                                                                                                                                                                                      /\ LET value443 == m[self] IN
                                                                                                                                                                                           /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                           /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value443)]
                                                                                                                                                                                           /\ \/ /\ LET value543 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                      /\ ((network)[j]).enabled
                                                                                                                                                                                                      /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                      /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value543), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                              \/ /\ LET yielded_fd143 == (fd)[j] IN
                                                                                                                                                                                                      /\ yielded_fd143
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                 /\ UNCHANGED network
                                                                                                                                                                                 ELSE /\ log' = [log38 EXCEPT ![i] = ((log38)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                      /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                "Failure of assertion at line 2557, column 41.")
                                                                                                                                                                                      /\ LET value444 == m[self] IN
                                                                                                                                                                                           /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                           /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value444)]
                                                                                                                                                                                           /\ \/ /\ LET value544 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                      /\ ((network)[j]).enabled
                                                                                                                                                                                                      /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                      /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value544), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                              \/ /\ LET yielded_fd144 == (fd)[j] IN
                                                                                                                                                                                                      /\ yielded_fd144
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                 /\ UNCHANGED network
                                                                                                                                                                                      /\ plog' = plog
                                                                                                                  ELSE /\ IF (((m[self]).mterm) < ((currentTerm)[i])) \/ (((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                             THEN /\ \/ /\ LET value115 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value115), enabled |-> ((network)[j]).enabled]]
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                     \/ /\ LET yielded_fd05 == (fd)[j] IN
                                                                                                                                             /\ yielded_fd05
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                        /\ UNCHANGED network
                                                                                                                                  /\ UNCHANGED << log, 
                                                                                                                                                  plog, 
                                                                                                                                                  fAdvCommitIdxCh >>
                                                                                                                             ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK), 
                                                                                                                                            "Failure of assertion at line 2598, column 29.")
                                                                                                                                  /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                       LET value29 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                         IF ((value29).cmd) = (LogConcat)
                                                                                                                                            THEN /\ LET plog34 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value29).entries)] IN
                                                                                                                                                      LET log39 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                        LET value315 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                          IF ((value315).cmd) = (LogConcat)
                                                                                                                                                             THEN /\ plog' = [plog34 EXCEPT ![i] = ((plog34)[i]) \o ((value315).entries)]
                                                                                                                                                                  /\ log' = [log39 EXCEPT ![i] = ((log39)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                            "Failure of assertion at line 2612, column 37.")
                                                                                                                                                                  /\ LET value445 == m[self] IN
                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value445)]
                                                                                                                                                                       /\ \/ /\ LET value545 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value545), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                          \/ /\ LET yielded_fd145 == (fd)[j] IN
                                                                                                                                                                                  /\ yielded_fd145
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                             ELSE /\ IF ((value315).cmd) = (LogPop)
                                                                                                                                                                        THEN /\ plog' = [plog34 EXCEPT ![i] = SubSeq((plog34)[i], 1, (Len((plog34)[i])) - ((value315).cnt))]
                                                                                                                                                                             /\ log' = [log39 EXCEPT ![i] = ((log39)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2634, column 39.")
                                                                                                                                                                             /\ LET value446 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value446)]
                                                                                                                                                                                  /\ \/ /\ LET value546 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value546), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd146 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd146
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ log' = [log39 EXCEPT ![i] = ((log39)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2654, column 39.")
                                                                                                                                                                             /\ LET value447 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value447)]
                                                                                                                                                                                  /\ \/ /\ LET value547 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value547), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ plog' = plog34
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd147 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd147
                                                                                                                                                                                             /\ plog' = plog34
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                            ELSE /\ IF ((value29).cmd) = (LogPop)
                                                                                                                                                       THEN /\ LET plog35 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value29).cnt))] IN
                                                                                                                                                                 LET log40 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                   LET value316 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                     IF ((value316).cmd) = (LogConcat)
                                                                                                                                                                        THEN /\ plog' = [plog35 EXCEPT ![i] = ((plog35)[i]) \o ((value316).entries)]
                                                                                                                                                                             /\ log' = [log40 EXCEPT ![i] = ((log40)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2687, column 39.")
                                                                                                                                                                             /\ LET value448 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value448)]
                                                                                                                                                                                  /\ \/ /\ LET value548 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value548), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd148 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd148
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ IF ((value316).cmd) = (LogPop)
                                                                                                                                                                                   THEN /\ plog' = [plog35 EXCEPT ![i] = SubSeq((plog35)[i], 1, (Len((plog35)[i])) - ((value316).cnt))]
                                                                                                                                                                                        /\ log' = [log40 EXCEPT ![i] = ((log40)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                  "Failure of assertion at line 2709, column 41.")
                                                                                                                                                                                        /\ LET value449 == m[self] IN
                                                                                                                                                                                             /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                             /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value449)]
                                                                                                                                                                                             /\ \/ /\ LET value549 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value549), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                \/ /\ LET yielded_fd149 == (fd)[j] IN
                                                                                                                                                                                                        /\ yielded_fd149
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                                                   ELSE /\ log' = [log40 EXCEPT ![i] = ((log40)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                  "Failure of assertion at line 2729, column 41.")
                                                                                                                                                                                        /\ LET value450 == m[self] IN
                                                                                                                                                                                             /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                             /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value450)]
                                                                                                                                                                                             /\ \/ /\ LET value550 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value550), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                        /\ plog' = plog35
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                \/ /\ LET yielded_fd150 == (fd)[j] IN
                                                                                                                                                                                                        /\ yielded_fd150
                                                                                                                                                                                                        /\ plog' = plog35
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                       ELSE /\ LET log41 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                 LET value317 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                   IF ((value317).cmd) = (LogConcat)
                                                                                                                                                                      THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value317).entries)]
                                                                                                                                                                           /\ log' = [log41 EXCEPT ![i] = ((log41)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                           /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                     "Failure of assertion at line 2760, column 39.")
                                                                                                                                                                           /\ LET value451 == m[self] IN
                                                                                                                                                                                /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value451)]
                                                                                                                                                                                /\ \/ /\ LET value551 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                           /\ ((network)[j]).enabled
                                                                                                                                                                                           /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                           /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value551), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                           /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                   \/ /\ LET yielded_fd151 == (fd)[j] IN
                                                                                                                                                                                           /\ yielded_fd151
                                                                                                                                                                                           /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                      /\ UNCHANGED network
                                                                                                                                                                      ELSE /\ IF ((value317).cmd) = (LogPop)
                                                                                                                                                                                 THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value317).cnt))]
                                                                                                                                                                                      /\ log' = [log41 EXCEPT ![i] = ((log41)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                      /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                "Failure of assertion at line 2782, column 41.")
                                                                                                                                                                                      /\ LET value452 == m[self] IN
                                                                                                                                                                                           /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                           /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value452)]
                                                                                                                                                                                           /\ \/ /\ LET value552 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                      /\ ((network)[j]).enabled
                                                                                                                                                                                                      /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                      /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value552), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                              \/ /\ LET yielded_fd152 == (fd)[j] IN
                                                                                                                                                                                                      /\ yielded_fd152
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                 /\ UNCHANGED network
                                                                                                                                                                                 ELSE /\ log' = [log41 EXCEPT ![i] = ((log41)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                      /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                "Failure of assertion at line 2802, column 41.")
                                                                                                                                                                                      /\ LET value453 == m[self] IN
                                                                                                                                                                                           /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                           /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value453)]
                                                                                                                                                                                           /\ \/ /\ LET value553 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                      /\ ((network)[j]).enabled
                                                                                                                                                                                                      /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                      /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value553), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                              \/ /\ LET yielded_fd153 == (fd)[j] IN
                                                                                                                                                                                                      /\ yielded_fd153
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                 /\ UNCHANGED network
                                                                                                                                                                                      /\ plog' = plog
                                                                                                                       /\ state' = state
                                                                                                       ELSE /\ IF (((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))
                                                                                                                  THEN /\ state' = [state EXCEPT ![i] = Follower]
                                                                                                                       /\ IF (((m[self]).mterm) < ((currentTerm)[i])) \/ (((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                             THEN /\ \/ /\ LET value116 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value116), enabled |-> ((network)[j]).enabled]]
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                     \/ /\ LET yielded_fd06 == (fd)[j] IN
                                                                                                                                             /\ yielded_fd06
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                        /\ UNCHANGED network
                                                                                                                                  /\ UNCHANGED << log, 
                                                                                                                                                  plog, 
                                                                                                                                                  fAdvCommitIdxCh >>
                                                                                                                             ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                            "Failure of assertion at line 2846, column 29.")
                                                                                                                                  /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                       LET value210 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                         IF ((value210).cmd) = (LogConcat)
                                                                                                                                            THEN /\ LET plog36 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value210).entries)] IN
                                                                                                                                                      LET log42 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                        LET value318 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                          IF ((value318).cmd) = (LogConcat)
                                                                                                                                                             THEN /\ plog' = [plog36 EXCEPT ![i] = ((plog36)[i]) \o ((value318).entries)]
                                                                                                                                                                  /\ log' = [log42 EXCEPT ![i] = ((log42)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                            "Failure of assertion at line 2860, column 37.")
                                                                                                                                                                  /\ LET value454 == m[self] IN
                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value454)]
                                                                                                                                                                       /\ \/ /\ LET value554 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value554), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                          \/ /\ LET yielded_fd154 == (fd)[j] IN
                                                                                                                                                                                  /\ yielded_fd154
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                             ELSE /\ IF ((value318).cmd) = (LogPop)
                                                                                                                                                                        THEN /\ plog' = [plog36 EXCEPT ![i] = SubSeq((plog36)[i], 1, (Len((plog36)[i])) - ((value318).cnt))]
                                                                                                                                                                             /\ log' = [log42 EXCEPT ![i] = ((log42)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2882, column 39.")
                                                                                                                                                                             /\ LET value455 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value455)]
                                                                                                                                                                                  /\ \/ /\ LET value555 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value555), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd155 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd155
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ log' = [log42 EXCEPT ![i] = ((log42)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2902, column 39.")
                                                                                                                                                                             /\ LET value456 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value456)]
                                                                                                                                                                                  /\ \/ /\ LET value556 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value556), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ plog' = plog36
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd156 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd156
                                                                                                                                                                                             /\ plog' = plog36
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                            ELSE /\ IF ((value210).cmd) = (LogPop)
                                                                                                                                                       THEN /\ LET plog37 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value210).cnt))] IN
                                                                                                                                                                 LET log43 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                   LET value319 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                     IF ((value319).cmd) = (LogConcat)
                                                                                                                                                                        THEN /\ plog' = [plog37 EXCEPT ![i] = ((plog37)[i]) \o ((value319).entries)]
                                                                                                                                                                             /\ log' = [log43 EXCEPT ![i] = ((log43)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 2935, column 39.")
                                                                                                                                                                             /\ LET value457 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value457)]
                                                                                                                                                                                  /\ \/ /\ LET value557 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value557), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd157 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd157
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ IF ((value319).cmd) = (LogPop)
                                                                                                                                                                                   THEN /\ plog' = [plog37 EXCEPT ![i] = SubSeq((plog37)[i], 1, (Len((plog37)[i])) - ((value319).cnt))]
                                                                                                                                                                                        /\ log' = [log43 EXCEPT ![i] = ((log43)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                  "Failure of assertion at line 2957, column 41.")
                                                                                                                                                                                        /\ LET value458 == m[self] IN
                                                                                                                                                                                             /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                             /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value458)]
                                                                                                                                                                                             /\ \/ /\ LET value558 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value558), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                \/ /\ LET yielded_fd158 == (fd)[j] IN
                                                                                                                                                                                                        /\ yielded_fd158
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                                                   ELSE /\ log' = [log43 EXCEPT ![i] = ((log43)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                  "Failure of assertion at line 2977, column 41.")
                                                                                                                                                                                        /\ LET value459 == m[self] IN
                                                                                                                                                                                             /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                             /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value459)]
                                                                                                                                                                                             /\ \/ /\ LET value559 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value559), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                        /\ plog' = plog37
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                \/ /\ LET yielded_fd159 == (fd)[j] IN
                                                                                                                                                                                                        /\ yielded_fd159
                                                                                                                                                                                                        /\ plog' = plog37
                                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                       ELSE /\ LET log44 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                 LET value320 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                   IF ((value320).cmd) = (LogConcat)
                                                                                                                                                                      THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value320).entries)]
                                                                                                                                                                           /\ log' = [log44 EXCEPT ![i] = ((log44)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                           /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                     "Failure of assertion at line 3008, column 39.")
                                                                                                                                                                           /\ LET value460 == m[self] IN
                                                                                                                                                                                /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value460)]
                                                                                                                                                                                /\ \/ /\ LET value560 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                           /\ ((network)[j]).enabled
                                                                                                                                                                                           /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                           /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value560), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                           /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                   \/ /\ LET yielded_fd160 == (fd)[j] IN
                                                                                                                                                                                           /\ yielded_fd160
                                                                                                                                                                                           /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                      /\ UNCHANGED network
                                                                                                                                                                      ELSE /\ IF ((value320).cmd) = (LogPop)
                                                                                                                                                                                 THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value320).cnt))]
                                                                                                                                                                                      /\ log' = [log44 EXCEPT ![i] = ((log44)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                      /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                "Failure of assertion at line 3030, column 41.")
                                                                                                                                                                                      /\ LET value461 == m[self] IN
                                                                                                                                                                                           /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                           /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value461)]
                                                                                                                                                                                           /\ \/ /\ LET value561 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                      /\ ((network)[j]).enabled
                                                                                                                                                                                                      /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                      /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value561), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                              \/ /\ LET yielded_fd161 == (fd)[j] IN
                                                                                                                                                                                                      /\ yielded_fd161
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                 /\ UNCHANGED network
                                                                                                                                                                                 ELSE /\ log' = [log44 EXCEPT ![i] = ((log44)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                      /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                "Failure of assertion at line 3050, column 41.")
                                                                                                                                                                                      /\ LET value462 == m[self] IN
                                                                                                                                                                                           /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                           /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value462)]
                                                                                                                                                                                           /\ \/ /\ LET value562 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                      /\ ((network)[j]).enabled
                                                                                                                                                                                                      /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                      /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value562), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                              \/ /\ LET yielded_fd162 == (fd)[j] IN
                                                                                                                                                                                                      /\ yielded_fd162
                                                                                                                                                                                                      /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                                 /\ UNCHANGED network
                                                                                                                                                                                      /\ plog' = plog
                                                                                                                  ELSE /\ IF (((m[self]).mterm) < ((currentTerm)[i])) \/ (((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                             THEN /\ \/ /\ LET value117 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value117), enabled |-> ((network)[j]).enabled]]
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                     \/ /\ LET yielded_fd07 == (fd)[j] IN
                                                                                                                                             /\ yielded_fd07
                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                        /\ UNCHANGED network
                                                                                                                                  /\ UNCHANGED << log, 
                                                                                                                                                  plog, 
                                                                                                                                                  fAdvCommitIdxCh >>
                                                                                                                             ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK), 
                                                                                                                                            "Failure of assertion at line 3091, column 29.")
                                                                                                                                  /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                       LET value211 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                         IF ((value211).cmd) = (LogConcat)
                                                                                                                                            THEN /\ LET plog38 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value211).entries)] IN
                                                                                                                                                      LET log45 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                        LET value321 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                          IF ((value321).cmd) = (LogConcat)
                                                                                                                                                             THEN /\ plog' = [plog38 EXCEPT ![i] = ((plog38)[i]) \o ((value321).entries)]
                                                                                                                                                                  /\ log' = [log45 EXCEPT ![i] = ((log45)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                  /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                            "Failure of assertion at line 3105, column 37.")
                                                                                                                                                                  /\ LET value463 == m[self] IN
                                                                                                                                                                       /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                       /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value463)]
                                                                                                                                                                       /\ \/ /\ LET value563 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                  /\ ((network)[j]).enabled
                                                                                                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value563), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                          \/ /\ LET yielded_fd163 == (fd)[j] IN
                                                                                                                                                                                  /\ yielded_fd163
                                                                                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                             /\ UNCHANGED network
                                                                                                                                                             ELSE /\ IF ((value321).cmd) = (LogPop)
                                                                                                                                                                        THEN /\ plog' = [plog38 EXCEPT ![i] = SubSeq((plog38)[i], 1, (Len((plog38)[i])) - ((value321).cnt))]
                                                                                                                                                                             /\ log' = [log45 EXCEPT ![i] = ((log45)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 3127, column 39.")
                                                                                                                                                                             /\ LET value464 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value464)]
                                                                                                                                                                                  /\ \/ /\ LET value564 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value564), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd164 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd164
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ log' = [log45 EXCEPT ![i] = ((log45)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 3147, column 39.")
                                                                                                                                                                             /\ LET value465 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value465)]
                                                                                                                                                                                  /\ \/ /\ LET value565 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value565), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ plog' = plog38
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd165 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd165
                                                                                                                                                                                             /\ plog' = plog38
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                            ELSE /\ IF ((value211).cmd) = (LogPop)
                                                                                                                                                       THEN /\ LET plog39 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value211).cnt))] IN
                                                                                                                                                                 LET log46 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                   LET value322 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                     IF ((value322).cmd) = (LogConcat)
                                                                                                                                                                        THEN /\ plog' = [plog39 EXCEPT ![i] = ((plog39)[i]) \o ((value322).entries)]
                                                                                                                                                                             /\ log' = [log46 EXCEPT ![i] = ((log46)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                             /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                       "Failure of assertion at line 3180, column 39.")
                                                                                                                                                                             /\ LET value466 == m[self] IN
                                                                                                                                                                                  /\ (Len((fAdvCommitIdxCh)[i])) < (BufferSize)
                                                                                                                                                                                  /\ fAdvCommitIdxCh' = [fAdvCommitIdxCh EXCEPT ![i] = Append((fAdvCommitIdxCh)[i], value466)]
                                                                                                                                                                                  /\ \/ /\ LET value566 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                             /\ ((network)[j]).enabled
                                                                                                                                                                                             /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                             /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value566), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                     \/ /\ LET yielded_fd166 == (fd)[j] IN
                                                                                                                                                                                             /\ yielded_fd166
                                                                                                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop_"]
                                                                                                                                                                                        /\ UNCHANGED network
                                                                                                                                                                        ELSE /\ IF ((value322).cmd) = (LogPop)
                                                                                                                                                                                   THEN /\ plog' = [plog39 EXCEPT ![i] = SubSeq((plog39)[i], 1, (Len((plog39)[i])) - ((value322).cnt))]
                                                                                                                                                                                        /\ log' = [log46 EXCEPT ![i] = ((log46)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                    