-------------------------------- MODULE raftkvs --------------------------------

\* Raft consensus protocol specification based on the TLA+ spec from
\* https://github.com/ongardie/raft.tla.
\*
\* We used limited buffer sized network channels to improve model checking 
\* perofmance. Also leader election part of Raft is not live by design.
\* These two can cause deadlock, so don't check deadlocks in model checking.
\*
\* Properties are defined based on the presented properties in the original 
\* Raft paper. See figure 3 in https://raft.github.io/raft.pdf.

EXTENDS Naturals, Sequences, TLC, FiniteSets

CONSTANT ExploreFail
CONSTANT Debug

CONSTANT NumServers
CONSTANT NumClients

CONSTANT BufferSize

CONSTANT MaxTerm
CONSTANT MaxCommitIndex

CONSTANT MaxNodeFail

CONSTANT LogConcat
CONSTANT LogPop

CONSTANT LeaderTimeoutReset

\* This is only used for model checking / simulation
CONSTANT NumRequests

Nil == 0

RECURSIVE MinAcc(_, _), MaxAcc(_, _)

Min(s) ==
    LET e == CHOOSE e \in s : TRUE
    IN MinAcc(s \ { e }, e)

MinAcc(s, e1) ==
    IF s = {} THEN e1
    ELSE
        LET e2 == CHOOSE e2 \in s : TRUE
        IN MinAcc(s \ { e2 }, IF e2 < e1 THEN e2 ELSE e1)

Max(s) ==
    LET e == CHOOSE e \in s : TRUE
    IN MaxAcc(s \ { e }, e)

MaxAcc(s, e1) ==
    IF s = {} THEN e1
    ELSE
        LET e2 == CHOOSE e2 \in s : TRUE
        IN MaxAcc(s \ { e2 }, IF e2 > e1 THEN e2 ELSE e1)

RECURSIVE FindMaxAgreeIndexRec(_, _, _, _)

isQuorum(s) == Cardinality(s) * 2 > NumServers
ServerSet   == 1..NumServers

FindMaxAgreeIndex(logLocal, i, matchIndex) ==
    FindMaxAgreeIndexRec(logLocal, i, matchIndex, Len(logLocal))

FindMaxAgreeIndexRec(logLocal, i, matchIndex, index) ==
    IF index = 0 THEN Nil
    ELSE
        IF isQuorum({i} \cup {k \in ServerSet : matchIndex[k] >= index})
        THEN index
        ELSE FindMaxAgreeIndexRec(logLocal, i, matchIndex, index - 1)

RECURSIVE ApplyLog(_, _, _, _, _)

Put == "put"
Get == "get"

ApplyLogEntry(xentry, xsm, xsmDomain) ==
    LET cmd == xentry.cmd
    IN IF cmd.type = Put
       THEN <<xsm @@ (cmd.key :> cmd.value), xsmDomain \cup {cmd.key}>>
       ELSE <<xsm, xsmDomain>>

\* applies range [start, end] from the log to the state machine
ApplyLog(xlog, start, end, xsm, xsmDomain) == 
    IF start > end
    THEN <<xsm, xsmDomain>>
    ELSE
        LET result == ApplyLogEntry(xlog[start], xsm, xsmDomain)
        IN ApplyLog(xlog, start+1, end, result[1], result[2])

(********************

--mpcal raftkvs {
    define {
        Follower  == "follower"
        Candidate == "candidate"
        Leader    == "leader"

        RequestVoteRequest    == "rvq"
        RequestVoteResponse   == "rvp"
        AppendEntriesRequest  == "apq"
        AppendEntriesResponse == "app"
        ClientPutRequest      == "cpq"
        ClientPutResponse     == "cpp"
        ClientGetRequest      == "cgq"
        ClientGetResponse     == "cgp"

        Key1   == "key1"
        Key2   == "key2"
        Value1 == "value1"

        LastTerm(xlog) == IF Len(xlog) = 0 THEN 0 ELSE xlog[Len(xlog)].term

        ServerRequestVoteSet        == (1*NumServers+1)..(2*NumServers)
        ServerAppendEntriesSet      == (2*NumServers+1)..(3*NumServers)
        ServerAdvanceCommitIndexSet == (3*NumServers+1)..(4*NumServers)
        ServerBecomeLeaderSet       == (4*NumServers+1)..(5*NumServers)

        ServerCrasherSet == IF ExploreFail 
                            THEN (5*NumServers+1)..(5*NumServers+MaxNodeFail)
                            ELSE {}

        ClientSet == (6*NumServers+1)..(6*NumServers+NumClients)

        NodeSet   == ServerSet \cup ClientSet

        KeySet == {}
    }

    macro checkFail(selfId, netEnabled) {
        if (ExploreFail) {
            if (\lnot netEnabled[selfId]) {
                await FALSE;
                \* goto Done;
            };
        };
    }

    macro debug(toPrint) {
        if (Debug) {
            print toPrint;
        };
    }

    macro Send(net, dest, fd, m) {
        either {
            net[dest] := m;
        } or {
            await fd[dest];
        };
    }

    macro UpdateTerm(i, m, currentTerm, state, votedFor, leader) {
        if (m.mterm > currentTerm[i]) {
            currentTerm[i] := m.mterm;
            state[i]       := Follower;
            votedFor[i]    := Nil;
            leader[i]      := Nil;
        };
    }

    mapping macro ReliableFIFOLink {
        read {
            assert $variable.enabled;
            await Len($variable.queue) > 0;
            with (readMsg = Head($variable.queue)) {
                $variable := [queue |-> Tail($variable.queue), enabled |-> $variable.enabled];
                yield readMsg;
            };
        }

        write {
            await $variable.enabled;
            await Len($variable.queue) < BufferSize;
            yield [queue |-> Append($variable.queue, $value), enabled |-> $variable.enabled];
        }
    }

    mapping macro NetworkToggle {
        read { yield $variable.enabled; }

        write {
            yield [queue |-> $variable.queue, enabled |-> $value];
        }
    }

    mapping macro PerfectFD {
        read {
            yield $variable;
        }

        write { yield $value; }
    }

    mapping macro NetworkBufferLength {
        read {
            yield Len($variable.queue);
        }

        write {
            assert FALSE;
            yield $value;
        }
    }

    mapping macro Channel {
        read {
            await Len($variable) > 0;
            with (res = Head($variable)) {
                $variable := Tail($variable);
                yield res;
            };
        }

        write {
            yield Append($variable, $value);
        }
    }

    mapping macro PersistentLog {
        read {
            yield $variable;
        }

        write {
            if ($value.cmd = LogConcat) {
                yield $variable \o $value.entries;
            } else if ($value.cmd = LogPop) {
                yield SubSeq($variable, 1, Len($variable) - $value.cnt);
            };
        }
    }

    archetype AServer(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
    variables
        idx = 1,
        m;
    {
    serverLoop:
        while (TRUE) {
            checkFail(self, netEnabled);

            m := net[self];
            assert m.mdest = self;

        handleMsg:
            checkFail(self, netEnabled);

            if (m.mtype = RequestVoteRequest) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* HandleRequestVoteRequest
                with (
                    i = self, j = m.msource,
                    logOK = \/ m.mlastLogTerm > LastTerm(log[i])
                            \/ /\ m.mlastLogTerm = LastTerm(log[i])
                               /\ m.mlastLogIndex >= Len(log[i]),
                    grant = /\ m.mterm = currentTerm[i]
                            /\ logOK
                            /\ votedFor[self] \in {Nil, j}
                ) {
                    assert m.mterm <= currentTerm[i];
                    if (grant) {
                        votedFor[i] := j;
                    };
                    Send(net, j, fd, [
                        mtype        |-> RequestVoteResponse,
                        mterm        |-> currentTerm[i],
                        mvoteGranted |-> grant,
                        msource      |-> i,
                        mdest        |-> j
                    ]);

                    debug(<<"HandleRequestVoteRequest", i, j, currentTerm[i], grant>>);
                };
            } else if (m.mtype = RequestVoteResponse) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* DropStaleResponse
                if (m.mterm < currentTerm[self]) {
                    \* goto serverLoop;
                    skip;
                } else {
                    \* HandleRequestVoteResponse
                    with (i = self, j = m.msource) {
                        assert m.mterm = currentTerm[i];
                        votesResponded[i] := votesResponded[i] \cup {j};
                        if (m.mvoteGranted) {
                            votesGranted[i] := votesGranted[i] \cup {j};
                            if (
                                /\ state[i] = Candidate
                                /\ isQuorum(votesGranted[i])
                            ) {
                                becomeLeaderCh[i] := TRUE;
                            };
                        }; 
                    };
                };
            } else if (m.mtype = AppendEntriesRequest) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* HandleAppendEntriesRequest
                with (
                    i = self, j = m.msource,
                    logOK = \/ m.mprevLogIndex = 0
                            \/ /\ m.mprevLogIndex > 0
                               /\ m.mprevLogIndex <= Len(log[i])
                               /\ m.mprevLogTerm = log[i][m.mprevLogIndex].term
                ) {
                    assert m.mterm <= currentTerm[i];

                    if (m.mterm = currentTerm[i]) {
                        leader[i] := m.msource;
                        leaderTimeout := LeaderTimeoutReset;
                    };

                    \* return to follower state
                    if (
                        /\ m.mterm = currentTerm[i]
                        /\ state[i] = Candidate
                    ) {
                        state[i] := Follower;
                    };

                    \* reject request
                    if (
                        \/ m.mterm < currentTerm[i]
                        \/ /\ m.mterm = currentTerm[i]
                           /\ state[i] = Follower
                           /\ \lnot logOK
                    ) {
                        Send(net, j, fd, [
                            mtype       |-> AppendEntriesResponse,
                            mterm       |-> currentTerm[i],
                            msuccess    |-> FALSE,
                            mmatchIndex |-> 0,
                            msource     |-> i,
                            mdest       |-> j
                        ]);
                    }
                    \* accept request
                    else {
                        assert (
                            /\ m.mterm = currentTerm[i]
                            /\ state[i] = Follower
                            /\ logOK
                        );
                        with (index = m.mprevLogIndex + 1) {
                            plog[i] := [
                                cmd |-> LogPop, 
                                cnt |-> Len(log[i]) - m.mprevLogIndex
                            ];
                            log[i]  := SubSeq(log[i], 1, m.mprevLogIndex);

                            plog[i] := [
                                cmd     |-> LogConcat, 
                                entries |-> m.mentries
                            ];
                            log[i]  := log[i] \o m.mentries;

                            assert m.mcommitIndex <= Len(log[i]);
                            with (result = ApplyLog(log[i], commitIndex[i]+1, m.mcommitIndex, sm[i], smDomain[i])) {
                                sm[i]       := result[1];
                                smDomain[i] := result[2];
                            };
                            commitIndex[i] := Max({commitIndex[i], m.mcommitIndex});
                            Send(net, j, fd, [
                                mtype       |-> AppendEntriesResponse,
                                mterm       |-> currentTerm[i],
                                msuccess    |-> TRUE,
                                mmatchIndex |-> m.mprevLogIndex + Len(m.mentries),
                                msource     |-> i,
                                mdest       |-> j
                            ]);
                       };
                    };
                };
            } else if (m.mtype = AppendEntriesResponse) {
                UpdateTerm(self, m, currentTerm, state, votedFor, leader);

                \* DropStaleResponse
                if (m.mterm < currentTerm[self]) {
                    \* goto serverLoop;
                    skip;
                } else {
                    \* HandleAppendEntriesResponse
                    with (i = self, j = m.msource) {
                        assert m.mterm = currentTerm[i];
                        if (m.msuccess) {
                            \* nextIndex[j]  := m.mmatchIndex + 1;
                            nextIndex[i] := [nextIndex[i] EXCEPT ![j] = m.mmatchIndex + 1];
                            \* matchIndex[j] := m.mmatchIndex;
                            matchIndex[i] := [matchIndex[i] EXCEPT ![j] = m.mmatchIndex];
                        } else {
                            nextIndex[i] := [nextIndex[i] EXCEPT ![j] = Max({nextIndex[i][j]-1, 1})];
                            \* nextIndex[j]  := Max({nextIndex[j]-1, 1});
                        };
                    };
                };
            } else if (
                \/ m.mtype = ClientPutRequest
                \/ m.mtype = ClientGetRequest
            ) {
                \* HandleClientRequest

                \* debug(<<"HandleClientRequest", self, m.msource, currentTerm[self], state[self]>>);

                if (state[self] = Leader) {
                    with (entry = [term   |-> currentTerm[self],
                                   cmd    |-> m.mcmd,
                                   client |-> m.msource]
                    ) {
                        log[self]  := Append(log[self], entry);
                        plog[self] := [cmd |-> LogConcat, entries |-> <<entry>>];

                        \* replace appendEntriesCh with a place holder resource on the Go side
                        \* to get better performance
                        appendEntriesCh[self] := TRUE;
                    };
                } else {
                    with (
                        i = self, j = m.msource,
                        respType = IF m.mcmd.type = Put
                                   THEN ClientPutResponse
                                   ELSE ClientGetResponse
                    ) {
                        net[j] := [
                            mtype       |-> respType,
                            msuccess    |-> FALSE,
                            mresponse   |-> [
                                idx |-> m.mcmd.idx,
                                key |-> m.mcmd.key
                            ],
                            mleaderHint |-> leader[i],
                            msource     |-> i,
                            mdest       |-> j
                        ];
                    };
                };
            };
        };
    }

    archetype AServerRequestVote(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
    variable idx = 1;
    {
    serverRequestVoteLoop:
        while (TRUE) {
            checkFail(srvId, netEnabled);

            \* Server times out and starts a new election.
            await leaderTimeout;
            await netLen[srvId] = 0;

            await state[srvId] \in {Follower, Candidate};

            with (i = srvId) {
                state[i]          := Candidate;
                currentTerm[i]    := currentTerm[i] + 1;
                votedFor[i]       := i;
                votesResponded[i] := {i};
                votesGranted[i]   := {i};
                leader[i]         := Nil;

                debug(<<"ServerTimeout", i, currentTerm[i]>>);
            };

            idx := 1;
        requestVoteLoop:
            while (idx <= NumServers) {
                checkFail(srvId, netEnabled);

                \* RequestVote
                if (idx /= srvId) {
                    Send(net, idx, fd, [
                        mtype         |-> RequestVoteRequest,
                        mterm         |-> currentTerm[srvId],
                        mlastLogTerm  |-> LastTerm(log[srvId]),
                        mlastLogIndex |-> Len(log[srvId]),
                        msource       |-> srvId,
                        mdest         |-> idx
                    ]);
                };
                idx := idx + 1;
            };
        }
    }

    archetype AServerAppendEntries(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
    variable idx;
    {
    serverAppendEntriesLoop:
        while (appendEntriesCh[srvId]) {
            checkFail(srvId, netEnabled);

            await state[srvId] = Leader;
            idx := 1;

        appendEntriesLoop:
            \* AppendEntries
            while (
                /\ state[srvId] = Leader
                /\ idx <= NumServers
            ) {
                checkFail(srvId, netEnabled);

                if (idx /= srvId) {
                    with (
                        prevLogIndex = nextIndex[srvId][idx] - 1,
                        prevLogTerm  = IF prevLogIndex > 0
                                       THEN log[srvId][prevLogIndex].term
                                       ELSE 0,
                        \* lastEntry    = Min({Len(log[srvId]), nextIndex[srvId][idx]}),
                        entries      = SubSeq(log[srvId], nextIndex[srvId][idx], Len(log[srvId]))
                    ) {
                        Send(net, idx, fd, [
                            mtype         |-> AppendEntriesRequest,
                            mterm         |-> currentTerm[srvId],
                            mprevLogIndex |-> prevLogIndex,
                            mprevLogTerm  |-> prevLogTerm,
                            mentries      |-> entries,
                            mcommitIndex  |-> commitIndex[srvId],
                            msource       |-> srvId,
                            mdest         |-> idx
                        ]);
                    };
                };
                idx := idx + 1;
            };
        };
    }

    archetype AServerAdvanceCommitIndex(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
    variable newCommitIndex = 0,
    {
    serverAdvanceCommitIndexLoop:
        while (TRUE) {
            checkFail(srvId, netEnabled);

            await state[srvId] = Leader;

            \* AdvanceCommitIndex
            with (
                i = srvId,

                \* explores a much smaller space than the previous version,
                \* proportional to disagreement not the log size
                maxAgreeIndex = FindMaxAgreeIndex(log[i], i, matchIndex[i]),

                nCommitIndex =
                    IF /\ maxAgreeIndex /= Nil
                       /\ log[i][maxAgreeIndex].term = currentTerm[i]
                    THEN maxAgreeIndex
                    ELSE commitIndex[i]
            ) {
                newCommitIndex := nCommitIndex;
                assert newCommitIndex >= commitIndex[i];
            };

        applyLoop:
            while (commitIndex[srvId] < newCommitIndex) {
                checkFail(srvId, netEnabled);

                commitIndex[srvId] := commitIndex[srvId] + 1;
                with (
                    i = srvId,
                    k = commitIndex[i],
                    entry = log[i][k],
                    cmd   = entry.cmd,
                    respType = IF cmd.type = Put
                               THEN ClientPutResponse
                               ELSE ClientGetResponse
                ) {
                    if (cmd.type = Put) {
                        sm[i]       := sm[i] @@ (cmd.key :> cmd.value); \* allows sm[i] to grow
                        smDomain[i] := smDomain[i] \cup {cmd.key};
                    };
                    with (reqOk = cmd.key \in smDomain[i]) {
                        net[entry.client] := [
                            mtype       |-> respType,
                            msuccess    |-> TRUE,
                            mresponse   |-> [
                                idx   |-> cmd.idx,
                                key   |-> cmd.key,
                                value |-> IF reqOk THEN sm[i][cmd.key] ELSE Nil,
                                ok    |-> reqOk
                            ],
                            mleaderHint |-> i,
                            msource     |-> i,
                            mdest       |-> entry.client
                        ];
                    };
                };
            };
        };
    }

    archetype AServerBecomeLeader(
        srvId,
        ref net[_], ref netLen[_], ref netEnabled[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        \* added by Shayan
        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
    {
    serverBecomeLeaderLoop:
        while (becomeLeaderCh[srvId]) {
            checkFail(srvId, netEnabled);

            \* BecomeLeader
            await state[srvId] = Candidate;
            await isQuorum(votesGranted[srvId]);

            with (i = srvId) {
                state[i]      := Leader;
                nextIndex[i]  := [j \in ServerSet |-> Len(log[i]) + 1];
                matchIndex[i] := [j \in ServerSet |-> 0];
                leader[i]     := i;

                appendEntriesCh[srvId] := TRUE;
                debug(<<"BecomeLeader", i, currentTerm[i], state[i], leader[i]>>);
            };
        };
    }

    archetype AClient(ref net[_], ref netLen[_], ref fd[_], ref reqCh, ref respCh, ref timeout)
    variables leader = Nil, req, resp, reqIdx = 0;
    {
    clientLoop:
        while (TRUE) {
            req := reqCh;
            reqIdx := reqIdx + 1;

        sndReq:
            if (leader = Nil) {
                with (srv \in ServerSet) {
                    leader := srv;
                };
            };
            debug(<<"ClientSndReq", self, leader, reqIdx, req>>);
            if (req.type = Put) {
                Send(net, leader, fd, [
                    mtype   |-> ClientPutRequest,
                    mcmd    |-> [
                        idx   |-> reqIdx,
                        type  |-> Put,
                        key   |-> req.key,
                        value |-> req.value
                    ],
                    msource |-> self,
                    mdest   |-> leader
                ]);
            } else if (req.type = Get) {
                Send(net, leader, fd, [
                    mtype   |-> ClientGetRequest,
                    mcmd    |-> [
                        idx  |-> reqIdx,
                        type |-> Get,
                        key  |-> req.key
                    ],
                    msource |-> self,
                    mdest   |-> leader
                ]);
            };

        rcvResp:
            either {
                resp := net[self];
                debug(<<"ClientRcvResp", self, leader, reqIdx, resp>>);
                assert resp.mdest = self;

                \* it should be /very likely/ that indexed requests will help us
                \* throw out duplicate server responses.
                \* one edge case I can think of: start, do one req, immediately
                \* stop + restart, immediately get stale response to last req.
                if (resp.mresponse.idx /= reqIdx) {
                    goto rcvResp;
                } else {
                    leader := resp.mleaderHint;
                    assert /\ req.type = Get => resp.mtype = ClientGetResponse
                           /\ req.type = Put => resp.mtype = ClientPutResponse;
                    if (\lnot resp.msuccess) {
                        goto sndReq;
                    } else {
                        assert /\ resp.mresponse.idx = reqIdx
                               /\ resp.mresponse.key = req.key;
                        respCh := resp;
                        debug(<<"ClientRcvChDone", self, leader, reqIdx, resp>>);
                    };
                };
            } or {
                await \/ /\ fd[leader] 
                         /\ netLen[self] = 0
                      \/ timeout;
                debug(<<"ClientTimeout", self, leader, reqIdx>>);
                leader := Nil;
                goto sndReq;
            };
        };
    }

    archetype AServerCrasher(srvId, ref netEnabled[_], ref fd[_]) {
    serverCrash:
        netEnabled[srvId] := FALSE;
    fdUpdate:
        fd[srvId] := TRUE;
    }

    variables
        network = [i \in NodeSet |-> [queue |-> << >>, enabled |-> TRUE]];
        fd      = [i \in ServerSet |-> FALSE];

        state       = [i \in ServerSet |-> Follower];
        currentTerm = [i \in ServerSet |-> 1];

        commitIndex = [i \in ServerSet |-> 0];
        nextIndex   = [i \in ServerSet |-> [j \in ServerSet |-> 1]];
        matchIndex  = [i \in ServerSet |-> [j \in ServerSet |-> 0]];

        log  = [i \in ServerSet |-> <<>>];
        plog = [i \in ServerSet |-> <<>>];

        votedFor       = [i \in ServerSet |-> Nil];
        votesResponded = [i \in ServerSet |-> {}];
        votesGranted   = [i \in ServerSet |-> {}];

        leader = [i \in ServerSet |-> Nil];

        sm       = [i \in ServerSet |-> [k \in KeySet |-> Nil]];
        smDomain = [i \in ServerSet |-> KeySet];

        leaderTimeout   = TRUE;
        appendEntriesCh = [i \in ServerSet |-> <<>>];
        becomeLeaderCh  = [i \in ServerSet |-> 
            IF NumServers > 1 
            THEN <<>>
            ELSE <<TRUE>>
        ];

        allReqs = <<
            [type |-> Put, key |-> Key1, value |-> Value1],
            [type |-> Get, key |-> Key2],
            [type |-> Get, key |-> Key1]
        >>;

        reqCh = SubSeq(allReqs, 1, NumRequests);
        respCh;

        requestVoteSrvId        = [i \in ServerRequestVoteSet        |-> i - 1*NumServers];
        appendEntriesSrvId      = [i \in ServerAppendEntriesSet      |-> i - 2*NumServers];
        advanceCommitIndexSrvId = [i \in ServerAdvanceCommitIndexSet |-> i - 3*NumServers];
        becomeLeaderSrvId       = [i \in ServerBecomeLeaderSet       |-> i - 4*NumServers];
        crasherSrvId            = [i \in ServerCrasherSet            |-> i - 5*NumServers];

    fair process (s0 \in ServerSet) == instance AServer(
        s0,
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
        mapping @2[_] via ReliableFIFOLink
        mapping @3[_] via NetworkBufferLength
        mapping @4[_] via NetworkToggle
        mapping @5[_] via PerfectFD
        mapping @9[_] via PersistentLog
        mapping @20[_] via Channel
        mapping @21[_] via Channel;

    fair process (s1 \in ServerRequestVoteSet) == instance AServerRequestVote(
        requestVoteSrvId[s1],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
        mapping @2[_] via ReliableFIFOLink
        mapping @3[_] via NetworkBufferLength
        mapping @4[_] via NetworkToggle
        mapping @5[_] via PerfectFD
        mapping @9[_] via PersistentLog
        mapping @20[_] via Channel
        mapping @21[_] via Channel;

    fair process (s2 \in ServerAppendEntriesSet) == instance AServerAppendEntries(
        appendEntriesSrvId[s2],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
        mapping @2[_] via ReliableFIFOLink
        mapping @3[_] via NetworkBufferLength
        mapping @4[_] via NetworkToggle
        mapping @5[_] via PerfectFD
        mapping @9[_] via PersistentLog
        mapping @20[_] via Channel
        mapping @21[_] via Channel;

    fair process (s3 \in ServerAdvanceCommitIndexSet) == instance AServerAdvanceCommitIndex(
        advanceCommitIndexSrvId[s3],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
        mapping @2[_] via ReliableFIFOLink
        mapping @3[_] via NetworkBufferLength
        mapping @4[_] via NetworkToggle
        mapping @5[_] via PerfectFD
        mapping @9[_] via PersistentLog
        mapping @20[_] via Channel
        mapping @21[_] via Channel;

    fair process (s4 \in ServerBecomeLeaderSet) == instance AServerBecomeLeader(
        becomeLeaderSrvId[s4],
        ref network[_], ref network[_], ref network[_], ref fd[_],
        ref state[_], ref currentTerm[_],
        ref log[_], ref plog[_],
        ref commitIndex[_], ref nextIndex[_], ref matchIndex[_],
        ref votedFor[_], ref votesResponded[_], ref votesGranted[_],

        ref leader[_],
        ref sm[_], ref smDomain[_],
        ref leaderTimeout,
        ref appendEntriesCh[_], ref becomeLeaderCh[_]
    )
        mapping @2[_] via ReliableFIFOLink
        mapping @3[_] via NetworkBufferLength
        mapping @4[_] via NetworkToggle
        mapping @5[_] via PerfectFD
        mapping @9[_] via PersistentLog
        mapping @20[_] via Channel
        mapping @21[_] via Channel;

    fair process (client \in ClientSet) == instance AClient(
        ref network[_], ref network[_], ref fd[_],
        ref reqCh, ref respCh, FALSE
    )
        mapping @1[_] via ReliableFIFOLink
        mapping @2[_] via NetworkBufferLength
        mapping @3[_] via PerfectFD
        mapping @4    via Channel;
    
    fair process (crasher \in ServerCrasherSet) == instance AServerCrasher(
        crasherSrvId[crasher],
        ref network[_], ref fd[_]
    )
        mapping @2[_] via NetworkToggle
        mapping @3[_] via PerfectFD;
}

\* BEGIN PLUSCAL TRANSLATION
--algorithm raftkvs {
  variables network = [i \in NodeSet |-> [queue |-> <<>>, enabled |-> TRUE]]; fd = [i \in ServerSet |-> FALSE]; state = [i \in ServerSet |-> Follower]; currentTerm = [i \in ServerSet |-> 1]; commitIndex = [i \in ServerSet |-> 0]; nextIndex = [i \in ServerSet |-> [j \in ServerSet |-> 1]]; matchIndex = [i \in ServerSet |-> [j \in ServerSet |-> 0]]; log = [i \in ServerSet |-> <<>>]; plog = [i \in ServerSet |-> <<>>]; votedFor = [i \in ServerSet |-> Nil]; votesResponded = [i \in ServerSet |-> {}]; votesGranted = [i \in ServerSet |-> {}]; leader = [i \in ServerSet |-> Nil]; sm = [i \in ServerSet |-> [k \in KeySet |-> Nil]]; smDomain = [i \in ServerSet |-> KeySet]; leaderTimeout = TRUE; appendEntriesCh = [i \in ServerSet |-> <<>>]; becomeLeaderCh = [i \in ServerSet |-> IF (NumServers) > (1) THEN <<>> ELSE <<TRUE>>]; allReqs = <<[type |-> Put, key |-> Key1, value |-> Value1], [type |-> Get, key |-> Key2], [type |-> Get, key |-> Key1]>>; reqCh = SubSeq(allReqs, 1, NumRequests); respCh; requestVoteSrvId = [i \in ServerRequestVoteSet |-> (i) - ((1) * (NumServers))]; appendEntriesSrvId = [i \in ServerAppendEntriesSet |-> (i) - ((2) * (NumServers))]; advanceCommitIndexSrvId = [i \in ServerAdvanceCommitIndexSet |-> (i) - ((3) * (NumServers))]; becomeLeaderSrvId = [i \in ServerBecomeLeaderSet |-> (i) - ((4) * (NumServers))]; crasherSrvId = [i \in ServerCrasherSet |-> (i) - ((5) * (NumServers))];
  define{
    Follower == "follower"
    Candidate == "candidate"
    Leader == "leader"
    RequestVoteRequest == "rvq"
    RequestVoteResponse == "rvp"
    AppendEntriesRequest == "apq"
    AppendEntriesResponse == "app"
    ClientPutRequest == "cpq"
    ClientPutResponse == "cpp"
    ClientGetRequest == "cgq"
    ClientGetResponse == "cgp"
    Key1 == "key1"
    Key2 == "key2"
    Value1 == "value1"
    LastTerm(xlog) == IF (Len(xlog)) = (0) THEN 0 ELSE ((xlog)[Len(xlog)]).term
    ServerRequestVoteSet == (((1) * (NumServers)) + (1)) .. ((2) * (NumServers))
    ServerAppendEntriesSet == (((2) * (NumServers)) + (1)) .. ((3) * (NumServers))
    ServerAdvanceCommitIndexSet == (((3) * (NumServers)) + (1)) .. ((4) * (NumServers))
    ServerBecomeLeaderSet == (((4) * (NumServers)) + (1)) .. ((5) * (NumServers))
    ServerCrasherSet == IF ExploreFail THEN (((5) * (NumServers)) + (1)) .. (((5) * (NumServers)) + (MaxNodeFail)) ELSE {}
    ClientSet == (((6) * (NumServers)) + (1)) .. (((6) * (NumServers)) + (NumClients))
    NodeSet == (ServerSet) \union (ClientSet)
    KeySet == {}
  }
  
  fair process (s0 \in ServerSet)
    variables idx = 1; m; srvId = self;
  {
    serverLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network12 = ((network)[self]).enabled) {
            if (~ (yielded_network12)) {
              await FALSE;
              assert ((network)[self]).enabled;
              await (Len(((network)[self]).queue)) > (0);
              with (readMsg00 = Head(((network)[self]).queue)) {
                network := [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]];
                with (yielded_network00 = readMsg00) {
                  m := yielded_network00;
                  assert ((m).mdest) = (self);
                  goto handleMsg;
                };
              };
            } else {
              assert ((network)[self]).enabled;
              await (Len(((network)[self]).queue)) > (0);
              with (readMsg01 = Head(((network)[self]).queue)) {
                network := [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]];
                with (yielded_network01 = readMsg01) {
                  m := yielded_network01;
                  assert ((m).mdest) = (self);
                  goto handleMsg;
                };
              };
            };
          };
        } else {
          assert ((network)[self]).enabled;
          await (Len(((network)[self]).queue)) > (0);
          with (readMsg02 = Head(((network)[self]).queue)) {
            network := [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]];
            with (yielded_network02 = readMsg02) {
              m := yielded_network02;
              assert ((m).mdest) = (self);
              goto handleMsg;
            };
          };
        };
      } else {
        goto Done;
      };
    handleMsg:
      if (ExploreFail) {
        with (yielded_network13 = ((network)[self]).enabled) {
          if (~ (yielded_network13)) {
            await FALSE;
            if (((m).mtype) = (RequestVoteRequest)) {
              if (((m).mterm) > ((currentTerm)[self])) {
                currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                state := [state EXCEPT ![self] = Follower];
                with (votedFor3 = [votedFor EXCEPT ![self] = Nil]) {
                  leader := [leader EXCEPT ![self] = Nil];
                  with (
                    i = self, 
                    j = (m).msource, 
                    logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                    grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor3)[self]) \in ({Nil, j}))
                  ) {
                    assert ((m).mterm) <= ((currentTerm)[i]);
                    if (grant) {
                      votedFor := [votedFor3 EXCEPT ![i] = j];
                      either {
                        with (value00 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value00), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd7 = (fd)[j]) {
                          await yielded_fd7;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      either {
                        with (value01 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value01), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor3;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor3;
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd8 = (fd)[j]) {
                          await yielded_fd8;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor3;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor3;
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                };
              } else {
                with (
                  i = self, 
                  j = (m).msource, 
                  logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                  grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j}))
                ) {
                  assert ((m).mterm) <= ((currentTerm)[i]);
                  if (grant) {
                    votedFor := [votedFor EXCEPT ![i] = j];
                    either {
                      with (value02 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value02), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd9 = (fd)[j]) {
                        await yielded_fd9;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    either {
                      with (value03 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value03), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd10 = (fd)[j]) {
                        await yielded_fd10;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  };
                };
              };
            } else {
              if (((m).mtype) = (RequestVoteResponse)) {
                if (((m).mterm) > ((currentTerm)[self])) {
                  currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                  state := [state EXCEPT ![self] = Follower];
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  leader := [leader EXCEPT ![self] = Nil];
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value17 = TRUE) {
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value17)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value18 = TRUE) {
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value18)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                };
              } else {
                if (((m).mtype) = (AppendEntriesRequest)) {
                  if (((m).mterm) > ((currentTerm)[self])) {
                    currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                    with (state3 = [state EXCEPT ![self] = Follower]) {
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      with (
                        leader3 = [leader EXCEPT ![self] = Nil], 
                        i = self, 
                        j = (m).msource, 
                        logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                      ) {
                        assert ((m).mterm) <= ((currentTerm)[i]);
                        if (((m).mterm) = ((currentTerm)[i])) {
                          leader := [leader3 EXCEPT ![i] = (m).msource];
                          leaderTimeout := LeaderTimeoutReset;
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Candidate))) {
                            state := [state3 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value20 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value20), enabled |-> ((network)[j]).enabled]];
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd00 = (fd)[j]) {
                                  await yielded_fd00;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value30 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value30).cmd) = (LogConcat)) {
                                  with (
                                    plog24 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value30).entries)], 
                                    log24 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value40 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value40).cmd) = (LogConcat)) {
                                      plog := [plog24 EXCEPT ![i] = ((plog24)[i]) \o ((value40).entries)];
                                      log := [log24 EXCEPT ![i] = ((log24)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result24 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result24)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result24)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value50 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value50), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd11 = (fd)[j]) {
                                            await yielded_fd11;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value40).cmd) = (LogPop)) {
                                        plog := [plog24 EXCEPT ![i] = SubSeq((plog24)[i], 1, (Len((plog24)[i])) - ((value40).cnt))];
                                        log := [log24 EXCEPT ![i] = ((log24)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result25 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result25)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result25)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value51 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value51), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd12 = (fd)[j]) {
                                              await yielded_fd12;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log24 EXCEPT ![i] = ((log24)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result26 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result26)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result26)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value52 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value52), enabled |-> ((network)[j]).enabled]];
                                              plog := plog24;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd13 = (fd)[j]) {
                                              await yielded_fd13;
                                              plog := plog24;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value30).cmd) = (LogPop)) {
                                    with (
                                      plog25 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value30).cnt))], 
                                      log25 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value41 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value41).cmd) = (LogConcat)) {
                                        plog := [plog25 EXCEPT ![i] = ((plog25)[i]) \o ((value41).entries)];
                                        log := [log25 EXCEPT ![i] = ((log25)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result27 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result27)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result27)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value53 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value53), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd14 = (fd)[j]) {
                                              await yielded_fd14;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value41).cmd) = (LogPop)) {
                                          plog := [plog25 EXCEPT ![i] = SubSeq((plog25)[i], 1, (Len((plog25)[i])) - ((value41).cnt))];
                                          log := [log25 EXCEPT ![i] = ((log25)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result28 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result28)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result28)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value54 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value54), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd15 = (fd)[j]) {
                                                await yielded_fd15;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log25 EXCEPT ![i] = ((log25)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result29 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result29)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result29)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value55 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value55), enabled |-> ((network)[j]).enabled]];
                                                plog := plog25;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd16 = (fd)[j]) {
                                                await yielded_fd16;
                                                plog := plog25;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log26 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value42 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value42).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value42).entries)];
                                        log := [log26 EXCEPT ![i] = ((log26)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result30 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result30)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result30)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value56 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value56), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd17 = (fd)[j]) {
                                              await yielded_fd17;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value42).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value42).cnt))];
                                          log := [log26 EXCEPT ![i] = ((log26)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result31 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result31)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result31)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value57 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value57), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd18 = (fd)[j]) {
                                                await yielded_fd18;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log26 EXCEPT ![i] = ((log26)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result32 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result32)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result32)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value58 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value58), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd19 = (fd)[j]) {
                                                await yielded_fd19;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value21 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value21), enabled |-> ((network)[j]).enabled]];
                                  state := state3;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd01 = (fd)[j]) {
                                  await yielded_fd01;
                                  state := state3;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value31 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value31).cmd) = (LogConcat)) {
                                  with (
                                    plog26 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value31).entries)], 
                                    log27 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value43 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value43).cmd) = (LogConcat)) {
                                      plog := [plog26 EXCEPT ![i] = ((plog26)[i]) \o ((value43).entries)];
                                      log := [log27 EXCEPT ![i] = ((log27)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result33 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result33)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result33)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value59 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value59), enabled |-> ((network)[j]).enabled]];
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd110 = (fd)[j]) {
                                            await yielded_fd110;
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value43).cmd) = (LogPop)) {
                                        plog := [plog26 EXCEPT ![i] = SubSeq((plog26)[i], 1, (Len((plog26)[i])) - ((value43).cnt))];
                                        log := [log27 EXCEPT ![i] = ((log27)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result34 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result34)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result34)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value510 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value510), enabled |-> ((network)[j]).enabled]];
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd111 = (fd)[j]) {
                                              await yielded_fd111;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log27 EXCEPT ![i] = ((log27)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result35 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result35)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result35)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value511 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value511), enabled |-> ((network)[j]).enabled]];
                                              plog := plog26;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd112 = (fd)[j]) {
                                              await yielded_fd112;
                                              plog := plog26;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value31).cmd) = (LogPop)) {
                                    with (
                                      plog27 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value31).cnt))], 
                                      log28 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value44 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value44).cmd) = (LogConcat)) {
                                        plog := [plog27 EXCEPT ![i] = ((plog27)[i]) \o ((value44).entries)];
                                        log := [log28 EXCEPT ![i] = ((log28)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result36 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result36)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result36)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value512 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value512), enabled |-> ((network)[j]).enabled]];
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd113 = (fd)[j]) {
                                              await yielded_fd113;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value44).cmd) = (LogPop)) {
                                          plog := [plog27 EXCEPT ![i] = SubSeq((plog27)[i], 1, (Len((plog27)[i])) - ((value44).cnt))];
                                          log := [log28 EXCEPT ![i] = ((log28)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result37 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result37)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result37)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value513 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value513), enabled |-> ((network)[j]).enabled]];
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd114 = (fd)[j]) {
                                                await yielded_fd114;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log28 EXCEPT ![i] = ((log28)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result38 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result38)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result38)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value514 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value514), enabled |-> ((network)[j]).enabled]];
                                                plog := plog27;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd115 = (fd)[j]) {
                                                await yielded_fd115;
                                                plog := plog27;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log29 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value45 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value45).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value45).entries)];
                                        log := [log29 EXCEPT ![i] = ((log29)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result39 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result39)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result39)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value515 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value515), enabled |-> ((network)[j]).enabled]];
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd116 = (fd)[j]) {
                                              await yielded_fd116;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value45).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value45).cnt))];
                                          log := [log29 EXCEPT ![i] = ((log29)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result40 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result40)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result40)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value516 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value516), enabled |-> ((network)[j]).enabled]];
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd117 = (fd)[j]) {
                                                await yielded_fd117;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log29 EXCEPT ![i] = ((log29)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result41 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result41)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result41)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value517 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value517), enabled |-> ((network)[j]).enabled]];
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd118 = (fd)[j]) {
                                                await yielded_fd118;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Candidate))) {
                            state := [state3 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value22 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value22), enabled |-> ((network)[j]).enabled]];
                                  leader := leader3;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd02 = (fd)[j]) {
                                  await yielded_fd02;
                                  leader := leader3;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value32 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value32).cmd) = (LogConcat)) {
                                  with (
                                    plog28 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value32).entries)], 
                                    log30 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value46 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value46).cmd) = (LogConcat)) {
                                      plog := [plog28 EXCEPT ![i] = ((plog28)[i]) \o ((value46).entries)];
                                      log := [log30 EXCEPT ![i] = ((log30)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result42 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result42)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result42)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value518 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value518), enabled |-> ((network)[j]).enabled]];
                                            leader := leader3;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd119 = (fd)[j]) {
                                            await yielded_fd119;
                                            leader := leader3;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value46).cmd) = (LogPop)) {
                                        plog := [plog28 EXCEPT ![i] = SubSeq((plog28)[i], 1, (Len((plog28)[i])) - ((value46).cnt))];
                                        log := [log30 EXCEPT ![i] = ((log30)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result43 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result43)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result43)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value519 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value519), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd120 = (fd)[j]) {
                                              await yielded_fd120;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log30 EXCEPT ![i] = ((log30)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result44 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result44)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result44)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value520 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value520), enabled |-> ((network)[j]).enabled]];
                                              plog := plog28;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd121 = (fd)[j]) {
                                              await yielded_fd121;
                                              plog := plog28;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value32).cmd) = (LogPop)) {
                                    with (
                                      plog29 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value32).cnt))], 
                                      log31 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value47 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value47).cmd) = (LogConcat)) {
                                        plog := [plog29 EXCEPT ![i] = ((plog29)[i]) \o ((value47).entries)];
                                        log := [log31 EXCEPT ![i] = ((log31)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result45 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result45)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result45)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value521 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value521), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd122 = (fd)[j]) {
                                              await yielded_fd122;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value47).cmd) = (LogPop)) {
                                          plog := [plog29 EXCEPT ![i] = SubSeq((plog29)[i], 1, (Len((plog29)[i])) - ((value47).cnt))];
                                          log := [log31 EXCEPT ![i] = ((log31)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result46 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result46)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result46)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value522 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value522), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd123 = (fd)[j]) {
                                                await yielded_fd123;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log31 EXCEPT ![i] = ((log31)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result47 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result47)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result47)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value523 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value523), enabled |-> ((network)[j]).enabled]];
                                                plog := plog29;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd124 = (fd)[j]) {
                                                await yielded_fd124;
                                                plog := plog29;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log32 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value48 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value48).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value48).entries)];
                                        log := [log32 EXCEPT ![i] = ((log32)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result48 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result48)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result48)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value524 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value524), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd125 = (fd)[j]) {
                                              await yielded_fd125;
                                              leader := leader3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value48).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value48).cnt))];
                                          log := [log32 EXCEPT ![i] = ((log32)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result49 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result49)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result49)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value525 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value525), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd126 = (fd)[j]) {
                                                await yielded_fd126;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log32 EXCEPT ![i] = ((log32)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result50 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result50)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result50)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value526 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value526), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd127 = (fd)[j]) {
                                                await yielded_fd127;
                                                leader := leader3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value23 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value23), enabled |-> ((network)[j]).enabled]];
                                  leader := leader3;
                                  state := state3;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd03 = (fd)[j]) {
                                  await yielded_fd03;
                                  leader := leader3;
                                  state := state3;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value33 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value33).cmd) = (LogConcat)) {
                                  with (
                                    plog30 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value33).entries)], 
                                    log33 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value49 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value49).cmd) = (LogConcat)) {
                                      plog := [plog30 EXCEPT ![i] = ((plog30)[i]) \o ((value49).entries)];
                                      log := [log33 EXCEPT ![i] = ((log33)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result51 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result51)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result51)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value527 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value527), enabled |-> ((network)[j]).enabled]];
                                            leader := leader3;
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd128 = (fd)[j]) {
                                            await yielded_fd128;
                                            leader := leader3;
                                            state := state3;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value49).cmd) = (LogPop)) {
                                        plog := [plog30 EXCEPT ![i] = SubSeq((plog30)[i], 1, (Len((plog30)[i])) - ((value49).cnt))];
                                        log := [log33 EXCEPT ![i] = ((log33)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result52 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result52)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result52)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value528 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value528), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd129 = (fd)[j]) {
                                              await yielded_fd129;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log33 EXCEPT ![i] = ((log33)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result53 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result53)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result53)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value529 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value529), enabled |-> ((network)[j]).enabled]];
                                              plog := plog30;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd130 = (fd)[j]) {
                                              await yielded_fd130;
                                              plog := plog30;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value33).cmd) = (LogPop)) {
                                    with (
                                      plog31 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value33).cnt))], 
                                      log34 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value410 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value410).cmd) = (LogConcat)) {
                                        plog := [plog31 EXCEPT ![i] = ((plog31)[i]) \o ((value410).entries)];
                                        log := [log34 EXCEPT ![i] = ((log34)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result54 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result54)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result54)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value530 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value530), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd131 = (fd)[j]) {
                                              await yielded_fd131;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value410).cmd) = (LogPop)) {
                                          plog := [plog31 EXCEPT ![i] = SubSeq((plog31)[i], 1, (Len((plog31)[i])) - ((value410).cnt))];
                                          log := [log34 EXCEPT ![i] = ((log34)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result55 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result55)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result55)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value531 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value531), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd132 = (fd)[j]) {
                                                await yielded_fd132;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log34 EXCEPT ![i] = ((log34)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result56 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result56)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result56)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value532 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value532), enabled |-> ((network)[j]).enabled]];
                                                plog := plog31;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd133 = (fd)[j]) {
                                                await yielded_fd133;
                                                plog := plog31;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log35 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value411 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value411).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value411).entries)];
                                        log := [log35 EXCEPT ![i] = ((log35)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result57 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result57)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result57)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value533 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value533), enabled |-> ((network)[j]).enabled]];
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd134 = (fd)[j]) {
                                              await yielded_fd134;
                                              leader := leader3;
                                              state := state3;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value411).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value411).cnt))];
                                          log := [log35 EXCEPT ![i] = ((log35)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result58 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result58)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result58)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value534 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value534), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd135 = (fd)[j]) {
                                                await yielded_fd135;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log35 EXCEPT ![i] = ((log35)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result59 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result59)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result59)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value535 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value535), enabled |-> ((network)[j]).enabled]];
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd136 = (fd)[j]) {
                                                await yielded_fd136;
                                                leader := leader3;
                                                state := state3;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  } else {
                    with (
                      i = self, 
                      j = (m).msource, 
                      logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                    ) {
                      assert ((m).mterm) <= ((currentTerm)[i]);
                      if (((m).mterm) = ((currentTerm)[i])) {
                        leader := [leader EXCEPT ![i] = (m).msource];
                        leaderTimeout := LeaderTimeoutReset;
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value24 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value24), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd04 = (fd)[j]) {
                                await yielded_fd04;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value34 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value34).cmd) = (LogConcat)) {
                                with (
                                  plog32 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value34).entries)], 
                                  log36 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value412 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value412).cmd) = (LogConcat)) {
                                    plog := [plog32 EXCEPT ![i] = ((plog32)[i]) \o ((value412).entries)];
                                    log := [log36 EXCEPT ![i] = ((log36)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result60 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result60)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result60)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value536 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value536), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd137 = (fd)[j]) {
                                          await yielded_fd137;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value412).cmd) = (LogPop)) {
                                      plog := [plog32 EXCEPT ![i] = SubSeq((plog32)[i], 1, (Len((plog32)[i])) - ((value412).cnt))];
                                      log := [log36 EXCEPT ![i] = ((log36)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result61 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result61)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result61)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value537 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value537), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd138 = (fd)[j]) {
                                            await yielded_fd138;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log36 EXCEPT ![i] = ((log36)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result62 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result62)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result62)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value538 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value538), enabled |-> ((network)[j]).enabled]];
                                            plog := plog32;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd139 = (fd)[j]) {
                                            await yielded_fd139;
                                            plog := plog32;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value34).cmd) = (LogPop)) {
                                  with (
                                    plog33 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value34).cnt))], 
                                    log37 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value413 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value413).cmd) = (LogConcat)) {
                                      plog := [plog33 EXCEPT ![i] = ((plog33)[i]) \o ((value413).entries)];
                                      log := [log37 EXCEPT ![i] = ((log37)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result63 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result63)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result63)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value539 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value539), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd140 = (fd)[j]) {
                                            await yielded_fd140;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value413).cmd) = (LogPop)) {
                                        plog := [plog33 EXCEPT ![i] = SubSeq((plog33)[i], 1, (Len((plog33)[i])) - ((value413).cnt))];
                                        log := [log37 EXCEPT ![i] = ((log37)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result64 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result64)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result64)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value540 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value540), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd141 = (fd)[j]) {
                                              await yielded_fd141;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log37 EXCEPT ![i] = ((log37)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result65 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result65)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result65)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value541 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value541), enabled |-> ((network)[j]).enabled]];
                                              plog := plog33;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd142 = (fd)[j]) {
                                              await yielded_fd142;
                                              plog := plog33;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log38 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value414 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value414).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value414).entries)];
                                      log := [log38 EXCEPT ![i] = ((log38)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result66 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result66)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result66)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value542 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value542), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd143 = (fd)[j]) {
                                            await yielded_fd143;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value414).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value414).cnt))];
                                        log := [log38 EXCEPT ![i] = ((log38)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result67 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result67)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result67)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value543 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value543), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd144 = (fd)[j]) {
                                              await yielded_fd144;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log38 EXCEPT ![i] = ((log38)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result68 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result68)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result68)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value544 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value544), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd145 = (fd)[j]) {
                                              await yielded_fd145;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value25 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value25), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd05 = (fd)[j]) {
                                await yielded_fd05;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value35 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value35).cmd) = (LogConcat)) {
                                with (
                                  plog34 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value35).entries)], 
                                  log39 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value415 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value415).cmd) = (LogConcat)) {
                                    plog := [plog34 EXCEPT ![i] = ((plog34)[i]) \o ((value415).entries)];
                                    log := [log39 EXCEPT ![i] = ((log39)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result69 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result69)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result69)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value545 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value545), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd146 = (fd)[j]) {
                                          await yielded_fd146;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value415).cmd) = (LogPop)) {
                                      plog := [plog34 EXCEPT ![i] = SubSeq((plog34)[i], 1, (Len((plog34)[i])) - ((value415).cnt))];
                                      log := [log39 EXCEPT ![i] = ((log39)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result70 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result70)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result70)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value546 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value546), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd147 = (fd)[j]) {
                                            await yielded_fd147;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log39 EXCEPT ![i] = ((log39)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result71 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result71)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result71)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value547 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value547), enabled |-> ((network)[j]).enabled]];
                                            plog := plog34;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd148 = (fd)[j]) {
                                            await yielded_fd148;
                                            plog := plog34;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value35).cmd) = (LogPop)) {
                                  with (
                                    plog35 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value35).cnt))], 
                                    log40 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value416 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value416).cmd) = (LogConcat)) {
                                      plog := [plog35 EXCEPT ![i] = ((plog35)[i]) \o ((value416).entries)];
                                      log := [log40 EXCEPT ![i] = ((log40)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result72 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result72)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result72)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value548 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value548), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd149 = (fd)[j]) {
                                            await yielded_fd149;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value416).cmd) = (LogPop)) {
                                        plog := [plog35 EXCEPT ![i] = SubSeq((plog35)[i], 1, (Len((plog35)[i])) - ((value416).cnt))];
                                        log := [log40 EXCEPT ![i] = ((log40)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result73 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result73)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result73)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value549 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value549), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd150 = (fd)[j]) {
                                              await yielded_fd150;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log40 EXCEPT ![i] = ((log40)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result74 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result74)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result74)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value550 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value550), enabled |-> ((network)[j]).enabled]];
                                              plog := plog35;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd151 = (fd)[j]) {
                                              await yielded_fd151;
                                              plog := plog35;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log41 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value417 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value417).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value417).entries)];
                                      log := [log41 EXCEPT ![i] = ((log41)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result75 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result75)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result75)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value551 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value551), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd152 = (fd)[j]) {
                                            await yielded_fd152;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value417).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value417).cnt))];
                                        log := [log41 EXCEPT ![i] = ((log41)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result76 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result76)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result76)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value552 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value552), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd153 = (fd)[j]) {
                                              await yielded_fd153;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log41 EXCEPT ![i] = ((log41)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result77 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result77)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result77)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value553 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value553), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd154 = (fd)[j]) {
                                              await yielded_fd154;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value26 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value26), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd06 = (fd)[j]) {
                                await yielded_fd06;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value36 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value36).cmd) = (LogConcat)) {
                                with (
                                  plog36 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value36).entries)], 
                                  log42 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value418 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value418).cmd) = (LogConcat)) {
                                    plog := [plog36 EXCEPT ![i] = ((plog36)[i]) \o ((value418).entries)];
                                    log := [log42 EXCEPT ![i] = ((log42)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result78 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result78)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result78)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value554 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value554), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd155 = (fd)[j]) {
                                          await yielded_fd155;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value418).cmd) = (LogPop)) {
                                      plog := [plog36 EXCEPT ![i] = SubSeq((plog36)[i], 1, (Len((plog36)[i])) - ((value418).cnt))];
                                      log := [log42 EXCEPT ![i] = ((log42)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result79 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result79)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result79)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value555 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value555), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd156 = (fd)[j]) {
                                            await yielded_fd156;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log42 EXCEPT ![i] = ((log42)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result80 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result80)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result80)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value556 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value556), enabled |-> ((network)[j]).enabled]];
                                            plog := plog36;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd157 = (fd)[j]) {
                                            await yielded_fd157;
                                            plog := plog36;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value36).cmd) = (LogPop)) {
                                  with (
                                    plog37 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value36).cnt))], 
                                    log43 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value419 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value419).cmd) = (LogConcat)) {
                                      plog := [plog37 EXCEPT ![i] = ((plog37)[i]) \o ((value419).entries)];
                                      log := [log43 EXCEPT ![i] = ((log43)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result81 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result81)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result81)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value557 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value557), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd158 = (fd)[j]) {
                                            await yielded_fd158;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value419).cmd) = (LogPop)) {
                                        plog := [plog37 EXCEPT ![i] = SubSeq((plog37)[i], 1, (Len((plog37)[i])) - ((value419).cnt))];
                                        log := [log43 EXCEPT ![i] = ((log43)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result82 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result82)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result82)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value558 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value558), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd159 = (fd)[j]) {
                                              await yielded_fd159;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log43 EXCEPT ![i] = ((log43)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result83 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result83)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result83)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value559 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value559), enabled |-> ((network)[j]).enabled]];
                                              plog := plog37;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd160 = (fd)[j]) {
                                              await yielded_fd160;
                                              plog := plog37;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log44 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value420 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value420).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value420).entries)];
                                      log := [log44 EXCEPT ![i] = ((log44)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result84 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result84)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result84)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value560 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value560), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd161 = (fd)[j]) {
                                            await yielded_fd161;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value420).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value420).cnt))];
                                        log := [log44 EXCEPT ![i] = ((log44)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result85 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result85)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result85)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value561 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value561), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd162 = (fd)[j]) {
                                              await yielded_fd162;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log44 EXCEPT ![i] = ((log44)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result86 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result86)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result86)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value562 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value562), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd163 = (fd)[j]) {
                                              await yielded_fd163;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value27 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value27), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd07 = (fd)[j]) {
                                await yielded_fd07;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value37 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value37).cmd) = (LogConcat)) {
                                with (
                                  plog38 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value37).entries)], 
                                  log45 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value421 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value421).cmd) = (LogConcat)) {
                                    plog := [plog38 EXCEPT ![i] = ((plog38)[i]) \o ((value421).entries)];
                                    log := [log45 EXCEPT ![i] = ((log45)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result87 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result87)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result87)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value563 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value563), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd164 = (fd)[j]) {
                                          await yielded_fd164;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value421).cmd) = (LogPop)) {
                                      plog := [plog38 EXCEPT ![i] = SubSeq((plog38)[i], 1, (Len((plog38)[i])) - ((value421).cnt))];
                                      log := [log45 EXCEPT ![i] = ((log45)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result88 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result88)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result88)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value564 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value564), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd165 = (fd)[j]) {
                                            await yielded_fd165;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log45 EXCEPT ![i] = ((log45)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result89 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result89)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result89)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value565 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value565), enabled |-> ((network)[j]).enabled]];
                                            plog := plog38;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd166 = (fd)[j]) {
                                            await yielded_fd166;
                                            plog := plog38;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value37).cmd) = (LogPop)) {
                                  with (
                                    plog39 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value37).cnt))], 
                                    log46 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value422 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value422).cmd) = (LogConcat)) {
                                      plog := [plog39 EXCEPT ![i] = ((plog39)[i]) \o ((value422).entries)];
                                      log := [log46 EXCEPT ![i] = ((log46)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result90 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result90)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result90)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value566 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value566), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd167 = (fd)[j]) {
                                            await yielded_fd167;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value422).cmd) = (LogPop)) {
                                        plog := [plog39 EXCEPT ![i] = SubSeq((plog39)[i], 1, (Len((plog39)[i])) - ((value422).cnt))];
                                        log := [log46 EXCEPT ![i] = ((log46)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result91 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result91)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result91)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value567 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value567), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd168 = (fd)[j]) {
                                              await yielded_fd168;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log46 EXCEPT ![i] = ((log46)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result92 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result92)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result92)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value568 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value568), enabled |-> ((network)[j]).enabled]];
                                              plog := plog39;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd169 = (fd)[j]) {
                                              await yielded_fd169;
                                              plog := plog39;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log47 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value423 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value423).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value423).entries)];
                                      log := [log47 EXCEPT ![i] = ((log47)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result93 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result93)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result93)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value569 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value569), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd170 = (fd)[j]) {
                                            await yielded_fd170;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value423).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value423).cnt))];
                                        log := [log47 EXCEPT ![i] = ((log47)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result94 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result94)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result94)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value570 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value570), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd171 = (fd)[j]) {
                                              await yielded_fd171;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log47 EXCEPT ![i] = ((log47)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result95 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result95)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result95)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value571 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value571), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd172 = (fd)[j]) {
                                              await yielded_fd172;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((m).mtype) = (AppendEntriesResponse)) {
                    if (((m).mterm) > ((currentTerm)[self])) {
                      currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                      state := [state EXCEPT ![self] = Follower];
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      leader := [leader EXCEPT ![self] = Nil];
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  } else {
                    if ((((m).mtype) = (ClientPutRequest)) \/ (((m).mtype) = (ClientGetRequest))) {
                      if (((state)[self]) = (Leader)) {
                        with (entry = [term |-> (currentTerm)[self], cmd |-> (m).mcmd, client |-> (m).msource]) {
                          log := [log EXCEPT ![self] = Append((log)[self], entry)];
                          with (value60 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                            if (((value60).cmd) = (LogConcat)) {
                              plog := [plog EXCEPT ![self] = ((plog)[self]) \o ((value60).entries)];
                              with (value70 = TRUE) {
                                appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value70)];
                                goto serverLoop;
                              };
                            } else {
                              if (((value60).cmd) = (LogPop)) {
                                plog := [plog EXCEPT ![self] = SubSeq((plog)[self], 1, (Len((plog)[self])) - ((value60).cnt))];
                                with (value71 = TRUE) {
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value71)];
                                  goto serverLoop;
                                };
                              } else {
                                with (value72 = TRUE) {
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value72)];
                                  goto serverLoop;
                                };
                              };
                            };
                          };
                        };
                      } else {
                        with (
                          i = self, 
                          j = (m).msource, 
                          respType = IF (((m).mcmd).type) = (Put) THEN ClientPutResponse ELSE ClientGetResponse, 
                          value80 = [mtype |-> respType, msuccess |-> FALSE, mresponse |-> [idx |-> ((m).mcmd).idx, key |-> ((m).mcmd).key], mleaderHint |-> (leader)[i], msource |-> i, mdest |-> j]
                        ) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value80), enabled |-> ((network)[j]).enabled]];
                          goto serverLoop;
                        };
                      };
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              };
            };
          } else {
            if (((m).mtype) = (RequestVoteRequest)) {
              if (((m).mterm) > ((currentTerm)[self])) {
                currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                state := [state EXCEPT ![self] = Follower];
                with (votedFor4 = [votedFor EXCEPT ![self] = Nil]) {
                  leader := [leader EXCEPT ![self] = Nil];
                  with (
                    i = self, 
                    j = (m).msource, 
                    logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                    grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor4)[self]) \in ({Nil, j}))
                  ) {
                    assert ((m).mterm) <= ((currentTerm)[i]);
                    if (grant) {
                      votedFor := [votedFor4 EXCEPT ![i] = j];
                      either {
                        with (value04 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value04), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd20 = (fd)[j]) {
                          await yielded_fd20;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            goto serverLoop;
                          } else {
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      either {
                        with (value05 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value05), enabled |-> ((network)[j]).enabled]];
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor4;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor4;
                            goto serverLoop;
                          };
                        };
                      } or {
                        with (yielded_fd21 = (fd)[j]) {
                          await yielded_fd21;
                          if (Debug) {
                            print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                            votedFor := votedFor4;
                            goto serverLoop;
                          } else {
                            votedFor := votedFor4;
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  };
                };
              } else {
                with (
                  i = self, 
                  j = (m).msource, 
                  logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                  grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j}))
                ) {
                  assert ((m).mterm) <= ((currentTerm)[i]);
                  if (grant) {
                    votedFor := [votedFor EXCEPT ![i] = j];
                    either {
                      with (value06 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value06), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd22 = (fd)[j]) {
                        await yielded_fd22;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  } else {
                    either {
                      with (value07 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                        await ((network)[j]).enabled;
                        await (Len(((network)[j]).queue)) < (BufferSize);
                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value07), enabled |-> ((network)[j]).enabled]];
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    } or {
                      with (yielded_fd23 = (fd)[j]) {
                        await yielded_fd23;
                        if (Debug) {
                          print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                          goto serverLoop;
                        } else {
                          goto serverLoop;
                        };
                      };
                    };
                  };
                };
              };
            } else {
              if (((m).mtype) = (RequestVoteResponse)) {
                if (((m).mterm) > ((currentTerm)[self])) {
                  currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                  state := [state EXCEPT ![self] = Follower];
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  leader := [leader EXCEPT ![self] = Nil];
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value19 = TRUE) {
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value19)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                      if ((m).mvoteGranted) {
                        votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                        if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                          with (value110 = TRUE) {
                            becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value110)];
                            goto serverLoop;
                          };
                        } else {
                          goto serverLoop;
                        };
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                };
              } else {
                if (((m).mtype) = (AppendEntriesRequest)) {
                  if (((m).mterm) > ((currentTerm)[self])) {
                    currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                    with (state4 = [state EXCEPT ![self] = Follower]) {
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      with (
                        leader4 = [leader EXCEPT ![self] = Nil], 
                        i = self, 
                        j = (m).msource, 
                        logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                      ) {
                        assert ((m).mterm) <= ((currentTerm)[i]);
                        if (((m).mterm) = ((currentTerm)[i])) {
                          leader := [leader4 EXCEPT ![i] = (m).msource];
                          leaderTimeout := LeaderTimeoutReset;
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Candidate))) {
                            state := [state4 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value28 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value28), enabled |-> ((network)[j]).enabled]];
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd08 = (fd)[j]) {
                                  await yielded_fd08;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value38 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value38).cmd) = (LogConcat)) {
                                  with (
                                    plog40 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value38).entries)], 
                                    log48 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value424 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value424).cmd) = (LogConcat)) {
                                      plog := [plog40 EXCEPT ![i] = ((plog40)[i]) \o ((value424).entries)];
                                      log := [log48 EXCEPT ![i] = ((log48)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result96 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result96)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result96)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value572 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value572), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd173 = (fd)[j]) {
                                            await yielded_fd173;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value424).cmd) = (LogPop)) {
                                        plog := [plog40 EXCEPT ![i] = SubSeq((plog40)[i], 1, (Len((plog40)[i])) - ((value424).cnt))];
                                        log := [log48 EXCEPT ![i] = ((log48)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result97 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result97)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result97)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value573 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value573), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd174 = (fd)[j]) {
                                              await yielded_fd174;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log48 EXCEPT ![i] = ((log48)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result98 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result98)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result98)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value574 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value574), enabled |-> ((network)[j]).enabled]];
                                              plog := plog40;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd175 = (fd)[j]) {
                                              await yielded_fd175;
                                              plog := plog40;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value38).cmd) = (LogPop)) {
                                    with (
                                      plog41 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value38).cnt))], 
                                      log49 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value425 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value425).cmd) = (LogConcat)) {
                                        plog := [plog41 EXCEPT ![i] = ((plog41)[i]) \o ((value425).entries)];
                                        log := [log49 EXCEPT ![i] = ((log49)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result99 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result99)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result99)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value575 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value575), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd176 = (fd)[j]) {
                                              await yielded_fd176;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value425).cmd) = (LogPop)) {
                                          plog := [plog41 EXCEPT ![i] = SubSeq((plog41)[i], 1, (Len((plog41)[i])) - ((value425).cnt))];
                                          log := [log49 EXCEPT ![i] = ((log49)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result100 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result100)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result100)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value576 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value576), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd177 = (fd)[j]) {
                                                await yielded_fd177;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log49 EXCEPT ![i] = ((log49)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result101 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result101)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result101)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value577 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value577), enabled |-> ((network)[j]).enabled]];
                                                plog := plog41;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd178 = (fd)[j]) {
                                                await yielded_fd178;
                                                plog := plog41;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log50 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value426 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value426).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value426).entries)];
                                        log := [log50 EXCEPT ![i] = ((log50)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result102 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result102)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result102)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value578 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value578), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd179 = (fd)[j]) {
                                              await yielded_fd179;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value426).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value426).cnt))];
                                          log := [log50 EXCEPT ![i] = ((log50)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result103 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result103)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result103)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value579 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value579), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd180 = (fd)[j]) {
                                                await yielded_fd180;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log50 EXCEPT ![i] = ((log50)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result104 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result104)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result104)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value580 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value580), enabled |-> ((network)[j]).enabled]];
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd181 = (fd)[j]) {
                                                await yielded_fd181;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value29 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value29), enabled |-> ((network)[j]).enabled]];
                                  state := state4;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd09 = (fd)[j]) {
                                  await yielded_fd09;
                                  state := state4;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value39 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value39).cmd) = (LogConcat)) {
                                  with (
                                    plog42 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value39).entries)], 
                                    log51 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value427 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value427).cmd) = (LogConcat)) {
                                      plog := [plog42 EXCEPT ![i] = ((plog42)[i]) \o ((value427).entries)];
                                      log := [log51 EXCEPT ![i] = ((log51)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result105 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result105)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result105)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value581 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value581), enabled |-> ((network)[j]).enabled]];
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd182 = (fd)[j]) {
                                            await yielded_fd182;
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value427).cmd) = (LogPop)) {
                                        plog := [plog42 EXCEPT ![i] = SubSeq((plog42)[i], 1, (Len((plog42)[i])) - ((value427).cnt))];
                                        log := [log51 EXCEPT ![i] = ((log51)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result106 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result106)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result106)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value582 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value582), enabled |-> ((network)[j]).enabled]];
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd183 = (fd)[j]) {
                                              await yielded_fd183;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log51 EXCEPT ![i] = ((log51)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result107 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result107)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result107)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value583 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value583), enabled |-> ((network)[j]).enabled]];
                                              plog := plog42;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd184 = (fd)[j]) {
                                              await yielded_fd184;
                                              plog := plog42;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value39).cmd) = (LogPop)) {
                                    with (
                                      plog43 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value39).cnt))], 
                                      log52 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value428 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value428).cmd) = (LogConcat)) {
                                        plog := [plog43 EXCEPT ![i] = ((plog43)[i]) \o ((value428).entries)];
                                        log := [log52 EXCEPT ![i] = ((log52)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result108 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result108)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result108)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value584 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value584), enabled |-> ((network)[j]).enabled]];
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd185 = (fd)[j]) {
                                              await yielded_fd185;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value428).cmd) = (LogPop)) {
                                          plog := [plog43 EXCEPT ![i] = SubSeq((plog43)[i], 1, (Len((plog43)[i])) - ((value428).cnt))];
                                          log := [log52 EXCEPT ![i] = ((log52)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result109 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result109)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result109)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value585 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value585), enabled |-> ((network)[j]).enabled]];
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd186 = (fd)[j]) {
                                                await yielded_fd186;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log52 EXCEPT ![i] = ((log52)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result110 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result110)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result110)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value586 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value586), enabled |-> ((network)[j]).enabled]];
                                                plog := plog43;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd187 = (fd)[j]) {
                                                await yielded_fd187;
                                                plog := plog43;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log53 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value429 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value429).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value429).entries)];
                                        log := [log53 EXCEPT ![i] = ((log53)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result111 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result111)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result111)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value587 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value587), enabled |-> ((network)[j]).enabled]];
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd188 = (fd)[j]) {
                                              await yielded_fd188;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value429).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value429).cnt))];
                                          log := [log53 EXCEPT ![i] = ((log53)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result112 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result112)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result112)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value588 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value588), enabled |-> ((network)[j]).enabled]];
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd189 = (fd)[j]) {
                                                await yielded_fd189;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log53 EXCEPT ![i] = ((log53)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result113 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result113)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result113)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value589 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value589), enabled |-> ((network)[j]).enabled]];
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd190 = (fd)[j]) {
                                                await yielded_fd190;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Candidate))) {
                            state := [state4 EXCEPT ![i] = Follower];
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value210 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value210), enabled |-> ((network)[j]).enabled]];
                                  leader := leader4;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd010 = (fd)[j]) {
                                  await yielded_fd010;
                                  leader := leader4;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value310 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value310).cmd) = (LogConcat)) {
                                  with (
                                    plog44 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value310).entries)], 
                                    log54 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value430 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value430).cmd) = (LogConcat)) {
                                      plog := [plog44 EXCEPT ![i] = ((plog44)[i]) \o ((value430).entries)];
                                      log := [log54 EXCEPT ![i] = ((log54)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result114 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result114)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result114)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value590 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value590), enabled |-> ((network)[j]).enabled]];
                                            leader := leader4;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd191 = (fd)[j]) {
                                            await yielded_fd191;
                                            leader := leader4;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value430).cmd) = (LogPop)) {
                                        plog := [plog44 EXCEPT ![i] = SubSeq((plog44)[i], 1, (Len((plog44)[i])) - ((value430).cnt))];
                                        log := [log54 EXCEPT ![i] = ((log54)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result115 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result115)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result115)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value591 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value591), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd192 = (fd)[j]) {
                                              await yielded_fd192;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log54 EXCEPT ![i] = ((log54)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result116 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result116)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result116)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value592 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value592), enabled |-> ((network)[j]).enabled]];
                                              plog := plog44;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd193 = (fd)[j]) {
                                              await yielded_fd193;
                                              plog := plog44;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value310).cmd) = (LogPop)) {
                                    with (
                                      plog45 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value310).cnt))], 
                                      log55 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value431 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value431).cmd) = (LogConcat)) {
                                        plog := [plog45 EXCEPT ![i] = ((plog45)[i]) \o ((value431).entries)];
                                        log := [log55 EXCEPT ![i] = ((log55)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result117 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result117)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result117)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value593 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value593), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd194 = (fd)[j]) {
                                              await yielded_fd194;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value431).cmd) = (LogPop)) {
                                          plog := [plog45 EXCEPT ![i] = SubSeq((plog45)[i], 1, (Len((plog45)[i])) - ((value431).cnt))];
                                          log := [log55 EXCEPT ![i] = ((log55)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result118 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result118)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result118)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value594 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value594), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd195 = (fd)[j]) {
                                                await yielded_fd195;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log55 EXCEPT ![i] = ((log55)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result119 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result119)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result119)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value595 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value595), enabled |-> ((network)[j]).enabled]];
                                                plog := plog45;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd196 = (fd)[j]) {
                                                await yielded_fd196;
                                                plog := plog45;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log56 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value432 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value432).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value432).entries)];
                                        log := [log56 EXCEPT ![i] = ((log56)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result120 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result120)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result120)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value596 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value596), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd197 = (fd)[j]) {
                                              await yielded_fd197;
                                              leader := leader4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value432).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value432).cnt))];
                                          log := [log56 EXCEPT ![i] = ((log56)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result121 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result121)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result121)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value597 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value597), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd198 = (fd)[j]) {
                                                await yielded_fd198;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log56 EXCEPT ![i] = ((log56)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result122 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result122)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result122)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value598 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value598), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd199 = (fd)[j]) {
                                                await yielded_fd199;
                                                leader := leader4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (~ (logOK)))) {
                              either {
                                with (value211 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                  await ((network)[j]).enabled;
                                  await (Len(((network)[j]).queue)) < (BufferSize);
                                  network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value211), enabled |-> ((network)[j]).enabled]];
                                  leader := leader4;
                                  state := state4;
                                  goto serverLoop;
                                };
                              } or {
                                with (yielded_fd011 = (fd)[j]) {
                                  await yielded_fd011;
                                  leader := leader4;
                                  state := state4;
                                  goto serverLoop;
                                };
                              };
                            } else {
                              assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state4)[i]) = (Follower))) /\ (logOK);
                              with (
                                index = ((m).mprevLogIndex) + (1), 
                                value311 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                              ) {
                                if (((value311).cmd) = (LogConcat)) {
                                  with (
                                    plog46 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value311).entries)], 
                                    log57 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value433 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value433).cmd) = (LogConcat)) {
                                      plog := [plog46 EXCEPT ![i] = ((plog46)[i]) \o ((value433).entries)];
                                      log := [log57 EXCEPT ![i] = ((log57)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result123 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result123)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result123)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value599 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value599), enabled |-> ((network)[j]).enabled]];
                                            leader := leader4;
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1100 = (fd)[j]) {
                                            await yielded_fd1100;
                                            leader := leader4;
                                            state := state4;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value433).cmd) = (LogPop)) {
                                        plog := [plog46 EXCEPT ![i] = SubSeq((plog46)[i], 1, (Len((plog46)[i])) - ((value433).cnt))];
                                        log := [log57 EXCEPT ![i] = ((log57)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result124 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result124)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result124)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5100 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5100), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1101 = (fd)[j]) {
                                              await yielded_fd1101;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log57 EXCEPT ![i] = ((log57)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result125 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result125)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result125)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5101 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5101), enabled |-> ((network)[j]).enabled]];
                                              plog := plog46;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1102 = (fd)[j]) {
                                              await yielded_fd1102;
                                              plog := plog46;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  if (((value311).cmd) = (LogPop)) {
                                    with (
                                      plog47 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value311).cnt))], 
                                      log58 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value434 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value434).cmd) = (LogConcat)) {
                                        plog := [plog47 EXCEPT ![i] = ((plog47)[i]) \o ((value434).entries)];
                                        log := [log58 EXCEPT ![i] = ((log58)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result126 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result126)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result126)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5102 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5102), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1103 = (fd)[j]) {
                                              await yielded_fd1103;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value434).cmd) = (LogPop)) {
                                          plog := [plog47 EXCEPT ![i] = SubSeq((plog47)[i], 1, (Len((plog47)[i])) - ((value434).cnt))];
                                          log := [log58 EXCEPT ![i] = ((log58)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result127 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result127)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result127)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value5103 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5103), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1104 = (fd)[j]) {
                                                await yielded_fd1104;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log58 EXCEPT ![i] = ((log58)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result128 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result128)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result128)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value5104 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5104), enabled |-> ((network)[j]).enabled]];
                                                plog := plog47;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1105 = (fd)[j]) {
                                                await yielded_fd1105;
                                                plog := plog47;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  } else {
                                    with (
                                      log59 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                      value435 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                    ) {
                                      if (((value435).cmd) = (LogConcat)) {
                                        plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value435).entries)];
                                        log := [log59 EXCEPT ![i] = ((log59)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result129 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result129)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result129)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5105 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5105), enabled |-> ((network)[j]).enabled]];
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1106 = (fd)[j]) {
                                              await yielded_fd1106;
                                              leader := leader4;
                                              state := state4;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        if (((value435).cmd) = (LogPop)) {
                                          plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value435).cnt))];
                                          log := [log59 EXCEPT ![i] = ((log59)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result130 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result130)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result130)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value5106 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5106), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1107 = (fd)[j]) {
                                                await yielded_fd1107;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        } else {
                                          log := [log59 EXCEPT ![i] = ((log59)[i]) \o ((m).mentries)];
                                          assert ((m).mcommitIndex) <= (Len((log)[i]));
                                          with (result131 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                            sm := [sm EXCEPT ![i] = (result131)[1]];
                                            smDomain := [smDomain EXCEPT ![i] = (result131)[2]];
                                            commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                            either {
                                              with (value5107 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                                await ((network)[j]).enabled;
                                                await (Len(((network)[j]).queue)) < (BufferSize);
                                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5107), enabled |-> ((network)[j]).enabled]];
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            } or {
                                              with (yielded_fd1108 = (fd)[j]) {
                                                await yielded_fd1108;
                                                leader := leader4;
                                                state := state4;
                                                goto serverLoop;
                                              };
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  } else {
                    with (
                      i = self, 
                      j = (m).msource, 
                      logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                    ) {
                      assert ((m).mterm) <= ((currentTerm)[i]);
                      if (((m).mterm) = ((currentTerm)[i])) {
                        leader := [leader EXCEPT ![i] = (m).msource];
                        leaderTimeout := LeaderTimeoutReset;
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value212 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value212), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd012 = (fd)[j]) {
                                await yielded_fd012;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value312 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value312).cmd) = (LogConcat)) {
                                with (
                                  plog48 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value312).entries)], 
                                  log60 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value436 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value436).cmd) = (LogConcat)) {
                                    plog := [plog48 EXCEPT ![i] = ((plog48)[i]) \o ((value436).entries)];
                                    log := [log60 EXCEPT ![i] = ((log60)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result132 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result132)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result132)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5108 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5108), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1109 = (fd)[j]) {
                                          await yielded_fd1109;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value436).cmd) = (LogPop)) {
                                      plog := [plog48 EXCEPT ![i] = SubSeq((plog48)[i], 1, (Len((plog48)[i])) - ((value436).cnt))];
                                      log := [log60 EXCEPT ![i] = ((log60)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result133 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result133)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result133)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5109 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5109), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1110 = (fd)[j]) {
                                            await yielded_fd1110;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log60 EXCEPT ![i] = ((log60)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result134 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result134)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result134)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5110 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5110), enabled |-> ((network)[j]).enabled]];
                                            plog := plog48;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1111 = (fd)[j]) {
                                            await yielded_fd1111;
                                            plog := plog48;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value312).cmd) = (LogPop)) {
                                  with (
                                    plog49 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value312).cnt))], 
                                    log61 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value437 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value437).cmd) = (LogConcat)) {
                                      plog := [plog49 EXCEPT ![i] = ((plog49)[i]) \o ((value437).entries)];
                                      log := [log61 EXCEPT ![i] = ((log61)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result135 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result135)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result135)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5111 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5111), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1112 = (fd)[j]) {
                                            await yielded_fd1112;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value437).cmd) = (LogPop)) {
                                        plog := [plog49 EXCEPT ![i] = SubSeq((plog49)[i], 1, (Len((plog49)[i])) - ((value437).cnt))];
                                        log := [log61 EXCEPT ![i] = ((log61)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result136 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result136)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result136)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5112 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5112), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1113 = (fd)[j]) {
                                              await yielded_fd1113;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log61 EXCEPT ![i] = ((log61)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result137 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result137)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result137)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5113 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5113), enabled |-> ((network)[j]).enabled]];
                                              plog := plog49;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1114 = (fd)[j]) {
                                              await yielded_fd1114;
                                              plog := plog49;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log62 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value438 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value438).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value438).entries)];
                                      log := [log62 EXCEPT ![i] = ((log62)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result138 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result138)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result138)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5114 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5114), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1115 = (fd)[j]) {
                                            await yielded_fd1115;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value438).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value438).cnt))];
                                        log := [log62 EXCEPT ![i] = ((log62)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result139 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result139)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result139)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5115 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5115), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1116 = (fd)[j]) {
                                              await yielded_fd1116;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log62 EXCEPT ![i] = ((log62)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result140 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result140)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result140)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5116 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5116), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1117 = (fd)[j]) {
                                              await yielded_fd1117;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value213 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value213), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd013 = (fd)[j]) {
                                await yielded_fd013;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value313 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value313).cmd) = (LogConcat)) {
                                with (
                                  plog50 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value313).entries)], 
                                  log63 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value439 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value439).cmd) = (LogConcat)) {
                                    plog := [plog50 EXCEPT ![i] = ((plog50)[i]) \o ((value439).entries)];
                                    log := [log63 EXCEPT ![i] = ((log63)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result141 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result141)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result141)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5117 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5117), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1118 = (fd)[j]) {
                                          await yielded_fd1118;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value439).cmd) = (LogPop)) {
                                      plog := [plog50 EXCEPT ![i] = SubSeq((plog50)[i], 1, (Len((plog50)[i])) - ((value439).cnt))];
                                      log := [log63 EXCEPT ![i] = ((log63)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result142 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result142)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result142)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5118 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5118), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1119 = (fd)[j]) {
                                            await yielded_fd1119;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log63 EXCEPT ![i] = ((log63)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result143 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result143)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result143)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5119 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5119), enabled |-> ((network)[j]).enabled]];
                                            plog := plog50;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1120 = (fd)[j]) {
                                            await yielded_fd1120;
                                            plog := plog50;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value313).cmd) = (LogPop)) {
                                  with (
                                    plog51 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value313).cnt))], 
                                    log64 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value440 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value440).cmd) = (LogConcat)) {
                                      plog := [plog51 EXCEPT ![i] = ((plog51)[i]) \o ((value440).entries)];
                                      log := [log64 EXCEPT ![i] = ((log64)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result144 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result144)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result144)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5120 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5120), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1121 = (fd)[j]) {
                                            await yielded_fd1121;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value440).cmd) = (LogPop)) {
                                        plog := [plog51 EXCEPT ![i] = SubSeq((plog51)[i], 1, (Len((plog51)[i])) - ((value440).cnt))];
                                        log := [log64 EXCEPT ![i] = ((log64)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result145 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result145)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result145)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5121 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5121), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1122 = (fd)[j]) {
                                              await yielded_fd1122;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log64 EXCEPT ![i] = ((log64)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result146 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result146)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result146)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5122 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5122), enabled |-> ((network)[j]).enabled]];
                                              plog := plog51;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1123 = (fd)[j]) {
                                              await yielded_fd1123;
                                              plog := plog51;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log65 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value441 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value441).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value441).entries)];
                                      log := [log65 EXCEPT ![i] = ((log65)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result147 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result147)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result147)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5123 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5123), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1124 = (fd)[j]) {
                                            await yielded_fd1124;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value441).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value441).cnt))];
                                        log := [log65 EXCEPT ![i] = ((log65)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result148 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result148)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result148)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5124 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5124), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1125 = (fd)[j]) {
                                              await yielded_fd1125;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log65 EXCEPT ![i] = ((log65)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result149 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result149)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result149)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5125 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5125), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1126 = (fd)[j]) {
                                              await yielded_fd1126;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                          state := [state EXCEPT ![i] = Follower];
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value214 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value214), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd014 = (fd)[j]) {
                                await yielded_fd014;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value314 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value314).cmd) = (LogConcat)) {
                                with (
                                  plog52 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value314).entries)], 
                                  log66 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value442 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value442).cmd) = (LogConcat)) {
                                    plog := [plog52 EXCEPT ![i] = ((plog52)[i]) \o ((value442).entries)];
                                    log := [log66 EXCEPT ![i] = ((log66)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result150 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result150)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result150)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5126 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5126), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1127 = (fd)[j]) {
                                          await yielded_fd1127;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value442).cmd) = (LogPop)) {
                                      plog := [plog52 EXCEPT ![i] = SubSeq((plog52)[i], 1, (Len((plog52)[i])) - ((value442).cnt))];
                                      log := [log66 EXCEPT ![i] = ((log66)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result151 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result151)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result151)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5127 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5127), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1128 = (fd)[j]) {
                                            await yielded_fd1128;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log66 EXCEPT ![i] = ((log66)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result152 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result152)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result152)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5128 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5128), enabled |-> ((network)[j]).enabled]];
                                            plog := plog52;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1129 = (fd)[j]) {
                                            await yielded_fd1129;
                                            plog := plog52;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value314).cmd) = (LogPop)) {
                                  with (
                                    plog53 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value314).cnt))], 
                                    log67 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value443 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value443).cmd) = (LogConcat)) {
                                      plog := [plog53 EXCEPT ![i] = ((plog53)[i]) \o ((value443).entries)];
                                      log := [log67 EXCEPT ![i] = ((log67)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result153 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result153)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result153)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5129 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5129), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1130 = (fd)[j]) {
                                            await yielded_fd1130;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value443).cmd) = (LogPop)) {
                                        plog := [plog53 EXCEPT ![i] = SubSeq((plog53)[i], 1, (Len((plog53)[i])) - ((value443).cnt))];
                                        log := [log67 EXCEPT ![i] = ((log67)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result154 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result154)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result154)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5130 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5130), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1131 = (fd)[j]) {
                                              await yielded_fd1131;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log67 EXCEPT ![i] = ((log67)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result155 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result155)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result155)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5131 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5131), enabled |-> ((network)[j]).enabled]];
                                              plog := plog53;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1132 = (fd)[j]) {
                                              await yielded_fd1132;
                                              plog := plog53;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log68 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value444 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value444).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value444).entries)];
                                      log := [log68 EXCEPT ![i] = ((log68)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result156 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result156)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result156)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5132 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5132), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1133 = (fd)[j]) {
                                            await yielded_fd1133;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value444).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value444).cnt))];
                                        log := [log68 EXCEPT ![i] = ((log68)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result157 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result157)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result157)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5133 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5133), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1134 = (fd)[j]) {
                                              await yielded_fd1134;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log68 EXCEPT ![i] = ((log68)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result158 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result158)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result158)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5134 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5134), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1135 = (fd)[j]) {
                                              await yielded_fd1135;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        } else {
                          if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                            either {
                              with (value215 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                                await ((network)[j]).enabled;
                                await (Len(((network)[j]).queue)) < (BufferSize);
                                network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value215), enabled |-> ((network)[j]).enabled]];
                                goto serverLoop;
                              };
                            } or {
                              with (yielded_fd015 = (fd)[j]) {
                                await yielded_fd015;
                                goto serverLoop;
                              };
                            };
                          } else {
                            assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                            with (
                              index = ((m).mprevLogIndex) + (1), 
                              value315 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                            ) {
                              if (((value315).cmd) = (LogConcat)) {
                                with (
                                  plog54 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value315).entries)], 
                                  log69 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value445 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value445).cmd) = (LogConcat)) {
                                    plog := [plog54 EXCEPT ![i] = ((plog54)[i]) \o ((value445).entries)];
                                    log := [log69 EXCEPT ![i] = ((log69)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result159 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result159)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result159)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5135 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5135), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1136 = (fd)[j]) {
                                          await yielded_fd1136;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value445).cmd) = (LogPop)) {
                                      plog := [plog54 EXCEPT ![i] = SubSeq((plog54)[i], 1, (Len((plog54)[i])) - ((value445).cnt))];
                                      log := [log69 EXCEPT ![i] = ((log69)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result160 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result160)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result160)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5136 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5136), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1137 = (fd)[j]) {
                                            await yielded_fd1137;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log69 EXCEPT ![i] = ((log69)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result161 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result161)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result161)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5137 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5137), enabled |-> ((network)[j]).enabled]];
                                            plog := plog54;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1138 = (fd)[j]) {
                                            await yielded_fd1138;
                                            plog := plog54;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                if (((value315).cmd) = (LogPop)) {
                                  with (
                                    plog55 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value315).cnt))], 
                                    log70 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value446 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value446).cmd) = (LogConcat)) {
                                      plog := [plog55 EXCEPT ![i] = ((plog55)[i]) \o ((value446).entries)];
                                      log := [log70 EXCEPT ![i] = ((log70)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result162 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result162)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result162)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5138 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5138), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1139 = (fd)[j]) {
                                            await yielded_fd1139;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value446).cmd) = (LogPop)) {
                                        plog := [plog55 EXCEPT ![i] = SubSeq((plog55)[i], 1, (Len((plog55)[i])) - ((value446).cnt))];
                                        log := [log70 EXCEPT ![i] = ((log70)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result163 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result163)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result163)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5139 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5139), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1140 = (fd)[j]) {
                                              await yielded_fd1140;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log70 EXCEPT ![i] = ((log70)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result164 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result164)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result164)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5140 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5140), enabled |-> ((network)[j]).enabled]];
                                              plog := plog55;
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1141 = (fd)[j]) {
                                              await yielded_fd1141;
                                              plog := plog55;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                } else {
                                  with (
                                    log71 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                    value447 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                  ) {
                                    if (((value447).cmd) = (LogConcat)) {
                                      plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value447).entries)];
                                      log := [log71 EXCEPT ![i] = ((log71)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result165 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result165)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result165)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5141 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5141), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1142 = (fd)[j]) {
                                            await yielded_fd1142;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      if (((value447).cmd) = (LogPop)) {
                                        plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value447).cnt))];
                                        log := [log71 EXCEPT ![i] = ((log71)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result166 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result166)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result166)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5142 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5142), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1143 = (fd)[j]) {
                                              await yielded_fd1143;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      } else {
                                        log := [log71 EXCEPT ![i] = ((log71)[i]) \o ((m).mentries)];
                                        assert ((m).mcommitIndex) <= (Len((log)[i]));
                                        with (result167 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                          sm := [sm EXCEPT ![i] = (result167)[1]];
                                          smDomain := [smDomain EXCEPT ![i] = (result167)[2]];
                                          commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                          either {
                                            with (value5143 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                              await ((network)[j]).enabled;
                                              await (Len(((network)[j]).queue)) < (BufferSize);
                                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5143), enabled |-> ((network)[j]).enabled]];
                                              goto serverLoop;
                                            };
                                          } or {
                                            with (yielded_fd1144 = (fd)[j]) {
                                              await yielded_fd1144;
                                              goto serverLoop;
                                            };
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                } else {
                  if (((m).mtype) = (AppendEntriesResponse)) {
                    if (((m).mterm) > ((currentTerm)[self])) {
                      currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                      state := [state EXCEPT ![self] = Follower];
                      votedFor := [votedFor EXCEPT ![self] = Nil];
                      leader := [leader EXCEPT ![self] = Nil];
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    } else {
                      if (((m).mterm) < ((currentTerm)[self])) {
                        skip;
                        goto serverLoop;
                      } else {
                        with (
                          i = self, 
                          j = (m).msource
                        ) {
                          assert ((m).mterm) = ((currentTerm)[i]);
                          if ((m).msuccess) {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                            matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                            goto serverLoop;
                          } else {
                            nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                            goto serverLoop;
                          };
                        };
                      };
                    };
                  } else {
                    if ((((m).mtype) = (ClientPutRequest)) \/ (((m).mtype) = (ClientGetRequest))) {
                      if (((state)[self]) = (Leader)) {
                        with (entry = [term |-> (currentTerm)[self], cmd |-> (m).mcmd, client |-> (m).msource]) {
                          log := [log EXCEPT ![self] = Append((log)[self], entry)];
                          with (value61 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                            if (((value61).cmd) = (LogConcat)) {
                              plog := [plog EXCEPT ![self] = ((plog)[self]) \o ((value61).entries)];
                              with (value73 = TRUE) {
                                appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value73)];
                                goto serverLoop;
                              };
                            } else {
                              if (((value61).cmd) = (LogPop)) {
                                plog := [plog EXCEPT ![self] = SubSeq((plog)[self], 1, (Len((plog)[self])) - ((value61).cnt))];
                                with (value74 = TRUE) {
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value74)];
                                  goto serverLoop;
                                };
                              } else {
                                with (value75 = TRUE) {
                                  appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value75)];
                                  goto serverLoop;
                                };
                              };
                            };
                          };
                        };
                      } else {
                        with (
                          i = self, 
                          j = (m).msource, 
                          respType = IF (((m).mcmd).type) = (Put) THEN ClientPutResponse ELSE ClientGetResponse, 
                          value81 = [mtype |-> respType, msuccess |-> FALSE, mresponse |-> [idx |-> ((m).mcmd).idx, key |-> ((m).mcmd).key], mleaderHint |-> (leader)[i], msource |-> i, mdest |-> j]
                        ) {
                          await ((network)[j]).enabled;
                          await (Len(((network)[j]).queue)) < (BufferSize);
                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value81), enabled |-> ((network)[j]).enabled]];
                          goto serverLoop;
                        };
                      };
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              };
            };
          };
        };
      } else {
        if (((m).mtype) = (RequestVoteRequest)) {
          if (((m).mterm) > ((currentTerm)[self])) {
            currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
            state := [state EXCEPT ![self] = Follower];
            with (votedFor5 = [votedFor EXCEPT ![self] = Nil]) {
              leader := [leader EXCEPT ![self] = Nil];
              with (
                i = self, 
                j = (m).msource, 
                logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
                grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor5)[self]) \in ({Nil, j}))
              ) {
                assert ((m).mterm) <= ((currentTerm)[i]);
                if (grant) {
                  votedFor := [votedFor5 EXCEPT ![i] = j];
                  either {
                    with (value08 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                      await ((network)[j]).enabled;
                      await (Len(((network)[j]).queue)) < (BufferSize);
                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value08), enabled |-> ((network)[j]).enabled]];
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        goto serverLoop;
                      } else {
                        goto serverLoop;
                      };
                    };
                  } or {
                    with (yielded_fd24 = (fd)[j]) {
                      await yielded_fd24;
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        goto serverLoop;
                      } else {
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  either {
                    with (value09 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                      await ((network)[j]).enabled;
                      await (Len(((network)[j]).queue)) < (BufferSize);
                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value09), enabled |-> ((network)[j]).enabled]];
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        votedFor := votedFor5;
                        goto serverLoop;
                      } else {
                        votedFor := votedFor5;
                        goto serverLoop;
                      };
                    };
                  } or {
                    with (yielded_fd25 = (fd)[j]) {
                      await yielded_fd25;
                      if (Debug) {
                        print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                        votedFor := votedFor5;
                        goto serverLoop;
                      } else {
                        votedFor := votedFor5;
                        goto serverLoop;
                      };
                    };
                  };
                };
              };
            };
          } else {
            with (
              i = self, 
              j = (m).msource, 
              logOK = (((m).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m).mlastLogIndex) >= (Len((log)[i])))), 
              grant = ((((m).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j}))
            ) {
              assert ((m).mterm) <= ((currentTerm)[i]);
              if (grant) {
                votedFor := [votedFor EXCEPT ![i] = j];
                either {
                  with (value010 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                    await ((network)[j]).enabled;
                    await (Len(((network)[j]).queue)) < (BufferSize);
                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value010), enabled |-> ((network)[j]).enabled]];
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                } or {
                  with (yielded_fd26 = (fd)[j]) {
                    await yielded_fd26;
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              } else {
                either {
                  with (value011 = [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j]) {
                    await ((network)[j]).enabled;
                    await (Len(((network)[j]).queue)) < (BufferSize);
                    network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value011), enabled |-> ((network)[j]).enabled]];
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                } or {
                  with (yielded_fd27 = (fd)[j]) {
                    await yielded_fd27;
                    if (Debug) {
                      print <<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>;
                      goto serverLoop;
                    } else {
                      goto serverLoop;
                    };
                  };
                };
              };
            };
          };
        } else {
          if (((m).mtype) = (RequestVoteResponse)) {
            if (((m).mterm) > ((currentTerm)[self])) {
              currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
              state := [state EXCEPT ![self] = Follower];
              votedFor := [votedFor EXCEPT ![self] = Nil];
              leader := [leader EXCEPT ![self] = Nil];
              if (((m).mterm) < ((currentTerm)[self])) {
                skip;
                goto serverLoop;
              } else {
                with (
                  i = self, 
                  j = (m).msource
                ) {
                  assert ((m).mterm) = ((currentTerm)[i]);
                  votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                  if ((m).mvoteGranted) {
                    votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                    if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                      with (value111 = TRUE) {
                        becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value111)];
                        goto serverLoop;
                      };
                    } else {
                      goto serverLoop;
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              };
            } else {
              if (((m).mterm) < ((currentTerm)[self])) {
                skip;
                goto serverLoop;
              } else {
                with (
                  i = self, 
                  j = (m).msource
                ) {
                  assert ((m).mterm) = ((currentTerm)[i]);
                  votesResponded := [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})];
                  if ((m).mvoteGranted) {
                    votesGranted := [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})];
                    if ((((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted)[i]))) {
                      with (value112 = TRUE) {
                        becomeLeaderCh := [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value112)];
                        goto serverLoop;
                      };
                    } else {
                      goto serverLoop;
                    };
                  } else {
                    goto serverLoop;
                  };
                };
              };
            };
          } else {
            if (((m).mtype) = (AppendEntriesRequest)) {
              if (((m).mterm) > ((currentTerm)[self])) {
                currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                with (state5 = [state EXCEPT ![self] = Follower]) {
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  with (
                    leader5 = [leader EXCEPT ![self] = Nil], 
                    i = self, 
                    j = (m).msource, 
                    logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                  ) {
                    assert ((m).mterm) <= ((currentTerm)[i]);
                    if (((m).mterm) = ((currentTerm)[i])) {
                      leader := [leader5 EXCEPT ![i] = (m).msource];
                      leaderTimeout := LeaderTimeoutReset;
                      if ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Candidate))) {
                        state := [state5 EXCEPT ![i] = Follower];
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value216 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value216), enabled |-> ((network)[j]).enabled]];
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd016 = (fd)[j]) {
                              await yielded_fd016;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value316 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value316).cmd) = (LogConcat)) {
                              with (
                                plog56 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value316).entries)], 
                                log72 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value448 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value448).cmd) = (LogConcat)) {
                                  plog := [plog56 EXCEPT ![i] = ((plog56)[i]) \o ((value448).entries)];
                                  log := [log72 EXCEPT ![i] = ((log72)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result168 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result168)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result168)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5144 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5144), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1145 = (fd)[j]) {
                                        await yielded_fd1145;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value448).cmd) = (LogPop)) {
                                    plog := [plog56 EXCEPT ![i] = SubSeq((plog56)[i], 1, (Len((plog56)[i])) - ((value448).cnt))];
                                    log := [log72 EXCEPT ![i] = ((log72)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result169 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result169)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result169)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5145 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5145), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1146 = (fd)[j]) {
                                          await yielded_fd1146;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log72 EXCEPT ![i] = ((log72)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result170 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result170)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result170)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5146 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5146), enabled |-> ((network)[j]).enabled]];
                                          plog := plog56;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1147 = (fd)[j]) {
                                          await yielded_fd1147;
                                          plog := plog56;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value316).cmd) = (LogPop)) {
                                with (
                                  plog57 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value316).cnt))], 
                                  log73 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value449 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value449).cmd) = (LogConcat)) {
                                    plog := [plog57 EXCEPT ![i] = ((plog57)[i]) \o ((value449).entries)];
                                    log := [log73 EXCEPT ![i] = ((log73)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result171 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result171)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result171)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5147 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5147), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1148 = (fd)[j]) {
                                          await yielded_fd1148;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value449).cmd) = (LogPop)) {
                                      plog := [plog57 EXCEPT ![i] = SubSeq((plog57)[i], 1, (Len((plog57)[i])) - ((value449).cnt))];
                                      log := [log73 EXCEPT ![i] = ((log73)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result172 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result172)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result172)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5148 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5148), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1149 = (fd)[j]) {
                                            await yielded_fd1149;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log73 EXCEPT ![i] = ((log73)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result173 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result173)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result173)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5149 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5149), enabled |-> ((network)[j]).enabled]];
                                            plog := plog57;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1150 = (fd)[j]) {
                                            await yielded_fd1150;
                                            plog := plog57;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log74 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value450 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value450).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value450).entries)];
                                    log := [log74 EXCEPT ![i] = ((log74)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result174 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result174)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result174)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5150 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5150), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1151 = (fd)[j]) {
                                          await yielded_fd1151;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value450).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value450).cnt))];
                                      log := [log74 EXCEPT ![i] = ((log74)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result175 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result175)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result175)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5151 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5151), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1152 = (fd)[j]) {
                                            await yielded_fd1152;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log74 EXCEPT ![i] = ((log74)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result176 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result176)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result176)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5152 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5152), enabled |-> ((network)[j]).enabled]];
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1153 = (fd)[j]) {
                                            await yielded_fd1153;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value217 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value217), enabled |-> ((network)[j]).enabled]];
                              state := state5;
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd017 = (fd)[j]) {
                              await yielded_fd017;
                              state := state5;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value317 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value317).cmd) = (LogConcat)) {
                              with (
                                plog58 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value317).entries)], 
                                log75 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value451 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value451).cmd) = (LogConcat)) {
                                  plog := [plog58 EXCEPT ![i] = ((plog58)[i]) \o ((value451).entries)];
                                  log := [log75 EXCEPT ![i] = ((log75)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result177 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result177)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result177)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5153 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5153), enabled |-> ((network)[j]).enabled]];
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1154 = (fd)[j]) {
                                        await yielded_fd1154;
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value451).cmd) = (LogPop)) {
                                    plog := [plog58 EXCEPT ![i] = SubSeq((plog58)[i], 1, (Len((plog58)[i])) - ((value451).cnt))];
                                    log := [log75 EXCEPT ![i] = ((log75)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result178 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result178)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result178)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5154 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5154), enabled |-> ((network)[j]).enabled]];
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1155 = (fd)[j]) {
                                          await yielded_fd1155;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log75 EXCEPT ![i] = ((log75)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result179 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result179)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result179)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5155 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5155), enabled |-> ((network)[j]).enabled]];
                                          plog := plog58;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1156 = (fd)[j]) {
                                          await yielded_fd1156;
                                          plog := plog58;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value317).cmd) = (LogPop)) {
                                with (
                                  plog59 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value317).cnt))], 
                                  log76 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value452 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value452).cmd) = (LogConcat)) {
                                    plog := [plog59 EXCEPT ![i] = ((plog59)[i]) \o ((value452).entries)];
                                    log := [log76 EXCEPT ![i] = ((log76)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result180 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result180)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result180)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5156 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5156), enabled |-> ((network)[j]).enabled]];
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1157 = (fd)[j]) {
                                          await yielded_fd1157;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value452).cmd) = (LogPop)) {
                                      plog := [plog59 EXCEPT ![i] = SubSeq((plog59)[i], 1, (Len((plog59)[i])) - ((value452).cnt))];
                                      log := [log76 EXCEPT ![i] = ((log76)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result181 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result181)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result181)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5157 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5157), enabled |-> ((network)[j]).enabled]];
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1158 = (fd)[j]) {
                                            await yielded_fd1158;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log76 EXCEPT ![i] = ((log76)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result182 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result182)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result182)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5158 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5158), enabled |-> ((network)[j]).enabled]];
                                            plog := plog59;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1159 = (fd)[j]) {
                                            await yielded_fd1159;
                                            plog := plog59;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log77 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value453 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value453).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value453).entries)];
                                    log := [log77 EXCEPT ![i] = ((log77)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result183 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result183)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result183)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5159 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5159), enabled |-> ((network)[j]).enabled]];
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1160 = (fd)[j]) {
                                          await yielded_fd1160;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value453).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value453).cnt))];
                                      log := [log77 EXCEPT ![i] = ((log77)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result184 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result184)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result184)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5160 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5160), enabled |-> ((network)[j]).enabled]];
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1161 = (fd)[j]) {
                                            await yielded_fd1161;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log77 EXCEPT ![i] = ((log77)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result185 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result185)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result185)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5161 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5161), enabled |-> ((network)[j]).enabled]];
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1162 = (fd)[j]) {
                                            await yielded_fd1162;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    } else {
                      if ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Candidate))) {
                        state := [state5 EXCEPT ![i] = Follower];
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value218 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value218), enabled |-> ((network)[j]).enabled]];
                              leader := leader5;
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd018 = (fd)[j]) {
                              await yielded_fd018;
                              leader := leader5;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value318 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value318).cmd) = (LogConcat)) {
                              with (
                                plog60 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value318).entries)], 
                                log78 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value454 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value454).cmd) = (LogConcat)) {
                                  plog := [plog60 EXCEPT ![i] = ((plog60)[i]) \o ((value454).entries)];
                                  log := [log78 EXCEPT ![i] = ((log78)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result186 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result186)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result186)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5162 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5162), enabled |-> ((network)[j]).enabled]];
                                        leader := leader5;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1163 = (fd)[j]) {
                                        await yielded_fd1163;
                                        leader := leader5;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value454).cmd) = (LogPop)) {
                                    plog := [plog60 EXCEPT ![i] = SubSeq((plog60)[i], 1, (Len((plog60)[i])) - ((value454).cnt))];
                                    log := [log78 EXCEPT ![i] = ((log78)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result187 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result187)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result187)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5163 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5163), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1164 = (fd)[j]) {
                                          await yielded_fd1164;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log78 EXCEPT ![i] = ((log78)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result188 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result188)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result188)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5164 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5164), enabled |-> ((network)[j]).enabled]];
                                          plog := plog60;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1165 = (fd)[j]) {
                                          await yielded_fd1165;
                                          plog := plog60;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value318).cmd) = (LogPop)) {
                                with (
                                  plog61 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value318).cnt))], 
                                  log79 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value455 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value455).cmd) = (LogConcat)) {
                                    plog := [plog61 EXCEPT ![i] = ((plog61)[i]) \o ((value455).entries)];
                                    log := [log79 EXCEPT ![i] = ((log79)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result189 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result189)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result189)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5165 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5165), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1166 = (fd)[j]) {
                                          await yielded_fd1166;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value455).cmd) = (LogPop)) {
                                      plog := [plog61 EXCEPT ![i] = SubSeq((plog61)[i], 1, (Len((plog61)[i])) - ((value455).cnt))];
                                      log := [log79 EXCEPT ![i] = ((log79)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result190 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result190)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result190)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5166 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5166), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1167 = (fd)[j]) {
                                            await yielded_fd1167;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log79 EXCEPT ![i] = ((log79)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result191 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result191)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result191)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5167 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5167), enabled |-> ((network)[j]).enabled]];
                                            plog := plog61;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1168 = (fd)[j]) {
                                            await yielded_fd1168;
                                            plog := plog61;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log80 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value456 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value456).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value456).entries)];
                                    log := [log80 EXCEPT ![i] = ((log80)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result192 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result192)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result192)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5168 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5168), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1169 = (fd)[j]) {
                                          await yielded_fd1169;
                                          leader := leader5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value456).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value456).cnt))];
                                      log := [log80 EXCEPT ![i] = ((log80)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result193 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result193)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result193)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5169 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5169), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1170 = (fd)[j]) {
                                            await yielded_fd1170;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log80 EXCEPT ![i] = ((log80)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result194 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result194)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result194)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5170 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5170), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1171 = (fd)[j]) {
                                            await yielded_fd1171;
                                            leader := leader5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      } else {
                        if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (~ (logOK)))) {
                          either {
                            with (value219 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                              await ((network)[j]).enabled;
                              await (Len(((network)[j]).queue)) < (BufferSize);
                              network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value219), enabled |-> ((network)[j]).enabled]];
                              leader := leader5;
                              state := state5;
                              goto serverLoop;
                            };
                          } or {
                            with (yielded_fd019 = (fd)[j]) {
                              await yielded_fd019;
                              leader := leader5;
                              state := state5;
                              goto serverLoop;
                            };
                          };
                        } else {
                          assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state5)[i]) = (Follower))) /\ (logOK);
                          with (
                            index = ((m).mprevLogIndex) + (1), 
                            value319 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                          ) {
                            if (((value319).cmd) = (LogConcat)) {
                              with (
                                plog62 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value319).entries)], 
                                log81 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value457 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value457).cmd) = (LogConcat)) {
                                  plog := [plog62 EXCEPT ![i] = ((plog62)[i]) \o ((value457).entries)];
                                  log := [log81 EXCEPT ![i] = ((log81)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result195 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result195)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result195)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5171 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5171), enabled |-> ((network)[j]).enabled]];
                                        leader := leader5;
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1172 = (fd)[j]) {
                                        await yielded_fd1172;
                                        leader := leader5;
                                        state := state5;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value457).cmd) = (LogPop)) {
                                    plog := [plog62 EXCEPT ![i] = SubSeq((plog62)[i], 1, (Len((plog62)[i])) - ((value457).cnt))];
                                    log := [log81 EXCEPT ![i] = ((log81)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result196 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result196)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result196)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5172 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5172), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1173 = (fd)[j]) {
                                          await yielded_fd1173;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log81 EXCEPT ![i] = ((log81)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result197 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result197)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result197)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5173 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5173), enabled |-> ((network)[j]).enabled]];
                                          plog := plog62;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1174 = (fd)[j]) {
                                          await yielded_fd1174;
                                          plog := plog62;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              if (((value319).cmd) = (LogPop)) {
                                with (
                                  plog63 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value319).cnt))], 
                                  log82 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value458 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value458).cmd) = (LogConcat)) {
                                    plog := [plog63 EXCEPT ![i] = ((plog63)[i]) \o ((value458).entries)];
                                    log := [log82 EXCEPT ![i] = ((log82)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result198 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result198)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result198)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5174 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5174), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1175 = (fd)[j]) {
                                          await yielded_fd1175;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value458).cmd) = (LogPop)) {
                                      plog := [plog63 EXCEPT ![i] = SubSeq((plog63)[i], 1, (Len((plog63)[i])) - ((value458).cnt))];
                                      log := [log82 EXCEPT ![i] = ((log82)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result199 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result199)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result199)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5175 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5175), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1176 = (fd)[j]) {
                                            await yielded_fd1176;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log82 EXCEPT ![i] = ((log82)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result200 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result200)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result200)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5176 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5176), enabled |-> ((network)[j]).enabled]];
                                            plog := plog63;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1177 = (fd)[j]) {
                                            await yielded_fd1177;
                                            plog := plog63;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              } else {
                                with (
                                  log83 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                  value459 = [cmd |-> LogConcat, entries |-> (m).mentries]
                                ) {
                                  if (((value459).cmd) = (LogConcat)) {
                                    plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value459).entries)];
                                    log := [log83 EXCEPT ![i] = ((log83)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result201 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result201)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result201)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5177 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5177), enabled |-> ((network)[j]).enabled]];
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1178 = (fd)[j]) {
                                          await yielded_fd1178;
                                          leader := leader5;
                                          state := state5;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    if (((value459).cmd) = (LogPop)) {
                                      plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value459).cnt))];
                                      log := [log83 EXCEPT ![i] = ((log83)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result202 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result202)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result202)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5178 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5178), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1179 = (fd)[j]) {
                                            await yielded_fd1179;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    } else {
                                      log := [log83 EXCEPT ![i] = ((log83)[i]) \o ((m).mentries)];
                                      assert ((m).mcommitIndex) <= (Len((log)[i]));
                                      with (result203 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                        sm := [sm EXCEPT ![i] = (result203)[1]];
                                        smDomain := [smDomain EXCEPT ![i] = (result203)[2]];
                                        commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                        either {
                                          with (value5179 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                            await ((network)[j]).enabled;
                                            await (Len(((network)[j]).queue)) < (BufferSize);
                                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5179), enabled |-> ((network)[j]).enabled]];
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        } or {
                                          with (yielded_fd1180 = (fd)[j]) {
                                            await yielded_fd1180;
                                            leader := leader5;
                                            state := state5;
                                            goto serverLoop;
                                          };
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                };
              } else {
                with (
                  i = self, 
                  j = (m).msource, 
                  logOK = (((m).mprevLogIndex) = (0)) \/ (((((m).mprevLogIndex) > (0)) /\ (((m).mprevLogIndex) <= (Len((log)[i])))) /\ (((m).mprevLogTerm) = ((((log)[i])[(m).mprevLogIndex]).term)))
                ) {
                  assert ((m).mterm) <= ((currentTerm)[i]);
                  if (((m).mterm) = ((currentTerm)[i])) {
                    leader := [leader EXCEPT ![i] = (m).msource];
                    leaderTimeout := LeaderTimeoutReset;
                    if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                      state := [state EXCEPT ![i] = Follower];
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value220 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value220), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd020 = (fd)[j]) {
                            await yielded_fd020;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value320 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value320).cmd) = (LogConcat)) {
                            with (
                              plog64 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value320).entries)], 
                              log84 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value460 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value460).cmd) = (LogConcat)) {
                                plog := [plog64 EXCEPT ![i] = ((plog64)[i]) \o ((value460).entries)];
                                log := [log84 EXCEPT ![i] = ((log84)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (result204 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                  sm := [sm EXCEPT ![i] = (result204)[1]];
                                  smDomain := [smDomain EXCEPT ![i] = (result204)[2]];
                                  commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                  either {
                                    with (value5180 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5180), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1181 = (fd)[j]) {
                                      await yielded_fd1181;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value460).cmd) = (LogPop)) {
                                  plog := [plog64 EXCEPT ![i] = SubSeq((plog64)[i], 1, (Len((plog64)[i])) - ((value460).cnt))];
                                  log := [log84 EXCEPT ![i] = ((log84)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result205 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result205)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result205)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5181 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5181), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1182 = (fd)[j]) {
                                        await yielded_fd1182;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log84 EXCEPT ![i] = ((log84)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result206 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result206)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result206)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5182 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5182), enabled |-> ((network)[j]).enabled]];
                                        plog := plog64;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1183 = (fd)[j]) {
                                        await yielded_fd1183;
                                        plog := plog64;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value320).cmd) = (LogPop)) {
                              with (
                                plog65 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value320).cnt))], 
                                log85 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value461 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value461).cmd) = (LogConcat)) {
                                  plog := [plog65 EXCEPT ![i] = ((plog65)[i]) \o ((value461).entries)];
                                  log := [log85 EXCEPT ![i] = ((log85)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result207 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result207)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result207)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5183 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5183), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1184 = (fd)[j]) {
                                        await yielded_fd1184;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value461).cmd) = (LogPop)) {
                                    plog := [plog65 EXCEPT ![i] = SubSeq((plog65)[i], 1, (Len((plog65)[i])) - ((value461).cnt))];
                                    log := [log85 EXCEPT ![i] = ((log85)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result208 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result208)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result208)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5184 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5184), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1185 = (fd)[j]) {
                                          await yielded_fd1185;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log85 EXCEPT ![i] = ((log85)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result209 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result209)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result209)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5185 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5185), enabled |-> ((network)[j]).enabled]];
                                          plog := plog65;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1186 = (fd)[j]) {
                                          await yielded_fd1186;
                                          plog := plog65;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log86 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value462 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value462).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value462).entries)];
                                  log := [log86 EXCEPT ![i] = ((log86)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result210 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result210)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result210)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5186 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5186), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1187 = (fd)[j]) {
                                        await yielded_fd1187;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value462).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value462).cnt))];
                                    log := [log86 EXCEPT ![i] = ((log86)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result211 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result211)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result211)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5187 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5187), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1188 = (fd)[j]) {
                                          await yielded_fd1188;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log86 EXCEPT ![i] = ((log86)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result212 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result212)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result212)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5188 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5188), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1189 = (fd)[j]) {
                                          await yielded_fd1189;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    } else {
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value221 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value221), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd021 = (fd)[j]) {
                            await yielded_fd021;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value321 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value321).cmd) = (LogConcat)) {
                            with (
                              plog66 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value321).entries)], 
                              log87 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value463 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value463).cmd) = (LogConcat)) {
                                plog := [plog66 EXCEPT ![i] = ((plog66)[i]) \o ((value463).entries)];
                                log := [log87 EXCEPT ![i] = ((log87)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (result213 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                  sm := [sm EXCEPT ![i] = (result213)[1]];
                                  smDomain := [smDomain EXCEPT ![i] = (result213)[2]];
                                  commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                  either {
                                    with (value5189 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5189), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1190 = (fd)[j]) {
                                      await yielded_fd1190;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value463).cmd) = (LogPop)) {
                                  plog := [plog66 EXCEPT ![i] = SubSeq((plog66)[i], 1, (Len((plog66)[i])) - ((value463).cnt))];
                                  log := [log87 EXCEPT ![i] = ((log87)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result214 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result214)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result214)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5190 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5190), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1191 = (fd)[j]) {
                                        await yielded_fd1191;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log87 EXCEPT ![i] = ((log87)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result215 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result215)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result215)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5191 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5191), enabled |-> ((network)[j]).enabled]];
                                        plog := plog66;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1192 = (fd)[j]) {
                                        await yielded_fd1192;
                                        plog := plog66;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value321).cmd) = (LogPop)) {
                              with (
                                plog67 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value321).cnt))], 
                                log88 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value464 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value464).cmd) = (LogConcat)) {
                                  plog := [plog67 EXCEPT ![i] = ((plog67)[i]) \o ((value464).entries)];
                                  log := [log88 EXCEPT ![i] = ((log88)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result216 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result216)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result216)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5192 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5192), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1193 = (fd)[j]) {
                                        await yielded_fd1193;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value464).cmd) = (LogPop)) {
                                    plog := [plog67 EXCEPT ![i] = SubSeq((plog67)[i], 1, (Len((plog67)[i])) - ((value464).cnt))];
                                    log := [log88 EXCEPT ![i] = ((log88)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result217 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result217)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result217)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5193 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5193), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1194 = (fd)[j]) {
                                          await yielded_fd1194;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log88 EXCEPT ![i] = ((log88)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result218 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result218)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result218)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5194 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5194), enabled |-> ((network)[j]).enabled]];
                                          plog := plog67;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1195 = (fd)[j]) {
                                          await yielded_fd1195;
                                          plog := plog67;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log89 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value465 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value465).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value465).entries)];
                                  log := [log89 EXCEPT ![i] = ((log89)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result219 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result219)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result219)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5195 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5195), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1196 = (fd)[j]) {
                                        await yielded_fd1196;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value465).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value465).cnt))];
                                    log := [log89 EXCEPT ![i] = ((log89)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result220 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result220)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result220)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5196 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5196), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1197 = (fd)[j]) {
                                          await yielded_fd1197;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log89 EXCEPT ![i] = ((log89)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result221 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result221)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result221)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5197 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5197), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1198 = (fd)[j]) {
                                          await yielded_fd1198;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  } else {
                    if ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))) {
                      state := [state EXCEPT ![i] = Follower];
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value222 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value222), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd022 = (fd)[j]) {
                            await yielded_fd022;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value322 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value322).cmd) = (LogConcat)) {
                            with (
                              plog68 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value322).entries)], 
                              log90 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value466 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value466).cmd) = (LogConcat)) {
                                plog := [plog68 EXCEPT ![i] = ((plog68)[i]) \o ((value466).entries)];
                                log := [log90 EXCEPT ![i] = ((log90)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (result222 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                  sm := [sm EXCEPT ![i] = (result222)[1]];
                                  smDomain := [smDomain EXCEPT ![i] = (result222)[2]];
                                  commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                  either {
                                    with (value5198 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5198), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1199 = (fd)[j]) {
                                      await yielded_fd1199;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value466).cmd) = (LogPop)) {
                                  plog := [plog68 EXCEPT ![i] = SubSeq((plog68)[i], 1, (Len((plog68)[i])) - ((value466).cnt))];
                                  log := [log90 EXCEPT ![i] = ((log90)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result223 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result223)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result223)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5199 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5199), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1200 = (fd)[j]) {
                                        await yielded_fd1200;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log90 EXCEPT ![i] = ((log90)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result224 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result224)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result224)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5200 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5200), enabled |-> ((network)[j]).enabled]];
                                        plog := plog68;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1201 = (fd)[j]) {
                                        await yielded_fd1201;
                                        plog := plog68;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value322).cmd) = (LogPop)) {
                              with (
                                plog69 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value322).cnt))], 
                                log91 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value467 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value467).cmd) = (LogConcat)) {
                                  plog := [plog69 EXCEPT ![i] = ((plog69)[i]) \o ((value467).entries)];
                                  log := [log91 EXCEPT ![i] = ((log91)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result225 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result225)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result225)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5201 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5201), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1202 = (fd)[j]) {
                                        await yielded_fd1202;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value467).cmd) = (LogPop)) {
                                    plog := [plog69 EXCEPT ![i] = SubSeq((plog69)[i], 1, (Len((plog69)[i])) - ((value467).cnt))];
                                    log := [log91 EXCEPT ![i] = ((log91)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result226 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result226)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result226)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5202 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5202), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1203 = (fd)[j]) {
                                          await yielded_fd1203;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log91 EXCEPT ![i] = ((log91)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result227 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result227)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result227)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5203 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5203), enabled |-> ((network)[j]).enabled]];
                                          plog := plog69;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1204 = (fd)[j]) {
                                          await yielded_fd1204;
                                          plog := plog69;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log92 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value468 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value468).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value468).entries)];
                                  log := [log92 EXCEPT ![i] = ((log92)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result228 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result228)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result228)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5204 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5204), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1205 = (fd)[j]) {
                                        await yielded_fd1205;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value468).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value468).cnt))];
                                    log := [log92 EXCEPT ![i] = ((log92)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result229 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result229)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result229)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5205 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5205), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1206 = (fd)[j]) {
                                          await yielded_fd1206;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log92 EXCEPT ![i] = ((log92)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result230 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result230)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result230)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5206 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5206), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1207 = (fd)[j]) {
                                          await yielded_fd1207;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    } else {
                      if ((((m).mterm) < ((currentTerm)[i])) \/ (((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))) {
                        either {
                          with (value223 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j]) {
                            await ((network)[j]).enabled;
                            await (Len(((network)[j]).queue)) < (BufferSize);
                            network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value223), enabled |-> ((network)[j]).enabled]];
                            goto serverLoop;
                          };
                        } or {
                          with (yielded_fd023 = (fd)[j]) {
                            await yielded_fd023;
                            goto serverLoop;
                          };
                        };
                      } else {
                        assert ((((m).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK);
                        with (
                          index = ((m).mprevLogIndex) + (1), 
                          value323 = [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m).mprevLogIndex)]
                        ) {
                          if (((value323).cmd) = (LogConcat)) {
                            with (
                              plog70 = [plog EXCEPT ![i] = ((plog)[i]) \o ((value323).entries)], 
                              log93 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                              value469 = [cmd |-> LogConcat, entries |-> (m).mentries]
                            ) {
                              if (((value469).cmd) = (LogConcat)) {
                                plog := [plog70 EXCEPT ![i] = ((plog70)[i]) \o ((value469).entries)];
                                log := [log93 EXCEPT ![i] = ((log93)[i]) \o ((m).mentries)];
                                assert ((m).mcommitIndex) <= (Len((log)[i]));
                                with (result231 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                  sm := [sm EXCEPT ![i] = (result231)[1]];
                                  smDomain := [smDomain EXCEPT ![i] = (result231)[2]];
                                  commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                  either {
                                    with (value5207 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                      await ((network)[j]).enabled;
                                      await (Len(((network)[j]).queue)) < (BufferSize);
                                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5207), enabled |-> ((network)[j]).enabled]];
                                      goto serverLoop;
                                    };
                                  } or {
                                    with (yielded_fd1208 = (fd)[j]) {
                                      await yielded_fd1208;
                                      goto serverLoop;
                                    };
                                  };
                                };
                              } else {
                                if (((value469).cmd) = (LogPop)) {
                                  plog := [plog70 EXCEPT ![i] = SubSeq((plog70)[i], 1, (Len((plog70)[i])) - ((value469).cnt))];
                                  log := [log93 EXCEPT ![i] = ((log93)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result232 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result232)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result232)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5208 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5208), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1209 = (fd)[j]) {
                                        await yielded_fd1209;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  log := [log93 EXCEPT ![i] = ((log93)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result233 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result233)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result233)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5209 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5209), enabled |-> ((network)[j]).enabled]];
                                        plog := plog70;
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1210 = (fd)[j]) {
                                        await yielded_fd1210;
                                        plog := plog70;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          } else {
                            if (((value323).cmd) = (LogPop)) {
                              with (
                                plog71 = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value323).cnt))], 
                                log94 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value470 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value470).cmd) = (LogConcat)) {
                                  plog := [plog71 EXCEPT ![i] = ((plog71)[i]) \o ((value470).entries)];
                                  log := [log94 EXCEPT ![i] = ((log94)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result234 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result234)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result234)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5210 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5210), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1211 = (fd)[j]) {
                                        await yielded_fd1211;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value470).cmd) = (LogPop)) {
                                    plog := [plog71 EXCEPT ![i] = SubSeq((plog71)[i], 1, (Len((plog71)[i])) - ((value470).cnt))];
                                    log := [log94 EXCEPT ![i] = ((log94)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result235 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result235)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result235)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5211 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5211), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1212 = (fd)[j]) {
                                          await yielded_fd1212;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log94 EXCEPT ![i] = ((log94)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result236 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result236)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result236)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5212 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5212), enabled |-> ((network)[j]).enabled]];
                                          plog := plog71;
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1213 = (fd)[j]) {
                                          await yielded_fd1213;
                                          plog := plog71;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            } else {
                              with (
                                log95 = [log EXCEPT ![i] = SubSeq((log)[i], 1, (m).mprevLogIndex)], 
                                value471 = [cmd |-> LogConcat, entries |-> (m).mentries]
                              ) {
                                if (((value471).cmd) = (LogConcat)) {
                                  plog := [plog EXCEPT ![i] = ((plog)[i]) \o ((value471).entries)];
                                  log := [log95 EXCEPT ![i] = ((log95)[i]) \o ((m).mentries)];
                                  assert ((m).mcommitIndex) <= (Len((log)[i]));
                                  with (result237 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                    sm := [sm EXCEPT ![i] = (result237)[1]];
                                    smDomain := [smDomain EXCEPT ![i] = (result237)[2]];
                                    commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                    either {
                                      with (value5213 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                        await ((network)[j]).enabled;
                                        await (Len(((network)[j]).queue)) < (BufferSize);
                                        network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5213), enabled |-> ((network)[j]).enabled]];
                                        goto serverLoop;
                                      };
                                    } or {
                                      with (yielded_fd1214 = (fd)[j]) {
                                        await yielded_fd1214;
                                        goto serverLoop;
                                      };
                                    };
                                  };
                                } else {
                                  if (((value471).cmd) = (LogPop)) {
                                    plog := [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value471).cnt))];
                                    log := [log95 EXCEPT ![i] = ((log95)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result238 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result238)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result238)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5214 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5214), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1215 = (fd)[j]) {
                                          await yielded_fd1215;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  } else {
                                    log := [log95 EXCEPT ![i] = ((log95)[i]) \o ((m).mentries)];
                                    assert ((m).mcommitIndex) <= (Len((log)[i]));
                                    with (result239 = ApplyLog((log)[i], ((commitIndex)[i]) + (1), (m).mcommitIndex, (sm)[i], (smDomain)[i])) {
                                      sm := [sm EXCEPT ![i] = (result239)[1]];
                                      smDomain := [smDomain EXCEPT ![i] = (result239)[2]];
                                      commitIndex := [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m).mcommitIndex})];
                                      either {
                                        with (value5215 = [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m).mprevLogIndex) + (Len((m).mentries)), msource |-> i, mdest |-> j]) {
                                          await ((network)[j]).enabled;
                                          await (Len(((network)[j]).queue)) < (BufferSize);
                                          network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value5215), enabled |-> ((network)[j]).enabled]];
                                          goto serverLoop;
                                        };
                                      } or {
                                        with (yielded_fd1216 = (fd)[j]) {
                                          await yielded_fd1216;
                                          goto serverLoop;
                                        };
                                      };
                                    };
                                  };
                                };
                              };
                            };
                          };
                        };
                      };
                    };
                  };
                };
              };
            } else {
              if (((m).mtype) = (AppendEntriesResponse)) {
                if (((m).mterm) > ((currentTerm)[self])) {
                  currentTerm := [currentTerm EXCEPT ![self] = (m).mterm];
                  state := [state EXCEPT ![self] = Follower];
                  votedFor := [votedFor EXCEPT ![self] = Nil];
                  leader := [leader EXCEPT ![self] = Nil];
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      if ((m).msuccess) {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                        matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                        goto serverLoop;
                      } else {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                        goto serverLoop;
                      };
                    };
                  };
                } else {
                  if (((m).mterm) < ((currentTerm)[self])) {
                    skip;
                    goto serverLoop;
                  } else {
                    with (
                      i = self, 
                      j = (m).msource
                    ) {
                      assert ((m).mterm) = ((currentTerm)[i]);
                      if ((m).msuccess) {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = ((m).mmatchIndex) + (1)]];
                        matchIndex := [matchIndex EXCEPT ![i] = [(matchIndex)[i] EXCEPT ![j] = (m).mmatchIndex]];
                        goto serverLoop;
                      } else {
                        nextIndex := [nextIndex EXCEPT ![i] = [(nextIndex)[i] EXCEPT ![j] = Max({(((nextIndex)[i])[j]) - (1), 1})]];
                        goto serverLoop;
                      };
                    };
                  };
                };
              } else {
                if ((((m).mtype) = (ClientPutRequest)) \/ (((m).mtype) = (ClientGetRequest))) {
                  if (((state)[self]) = (Leader)) {
                    with (entry = [term |-> (currentTerm)[self], cmd |-> (m).mcmd, client |-> (m).msource]) {
                      log := [log EXCEPT ![self] = Append((log)[self], entry)];
                      with (value62 = [cmd |-> LogConcat, entries |-> <<entry>>]) {
                        if (((value62).cmd) = (LogConcat)) {
                          plog := [plog EXCEPT ![self] = ((plog)[self]) \o ((value62).entries)];
                          with (value76 = TRUE) {
                            appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value76)];
                            goto serverLoop;
                          };
                        } else {
                          if (((value62).cmd) = (LogPop)) {
                            plog := [plog EXCEPT ![self] = SubSeq((plog)[self], 1, (Len((plog)[self])) - ((value62).cnt))];
                            with (value77 = TRUE) {
                              appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value77)];
                              goto serverLoop;
                            };
                          } else {
                            with (value78 = TRUE) {
                              appendEntriesCh := [appendEntriesCh EXCEPT ![self] = Append((appendEntriesCh)[self], value78)];
                              goto serverLoop;
                            };
                          };
                        };
                      };
                    };
                  } else {
                    with (
                      i = self, 
                      j = (m).msource, 
                      respType = IF (((m).mcmd).type) = (Put) THEN ClientPutResponse ELSE ClientGetResponse, 
                      value82 = [mtype |-> respType, msuccess |-> FALSE, mresponse |-> [idx |-> ((m).mcmd).idx, key |-> ((m).mcmd).key], mleaderHint |-> (leader)[i], msource |-> i, mdest |-> j]
                    ) {
                      await ((network)[j]).enabled;
                      await (Len(((network)[j]).queue)) < (BufferSize);
                      network := [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value82), enabled |-> ((network)[j]).enabled]];
                      goto serverLoop;
                    };
                  };
                } else {
                  goto serverLoop;
                };
              };
            };
          };
        };
      };
  }
  
  fair process (s1 \in ServerRequestVoteSet)
    variables idx0 = 1; srvId0 = (requestVoteSrvId)[self];
  {
    serverRequestVoteLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network20 = ((network)[srvId0]).enabled) {
            if (~ (yielded_network20)) {
              await FALSE;
              await leaderTimeout;
              with (yielded_network30 = Len(((network)[srvId0]).queue)) {
                await (yielded_network30) = (0);
                await ((state)[srvId0]) \in ({Follower, Candidate});
                with (i3 = srvId0) {
                  state := [state EXCEPT ![i3] = Candidate];
                  currentTerm := [currentTerm EXCEPT ![i3] = ((currentTerm)[i3]) + (1)];
                  votedFor := [votedFor EXCEPT ![i3] = i3];
                  votesResponded := [votesResponded EXCEPT ![i3] = {i3}];
                  votesGranted := [votesGranted EXCEPT ![i3] = {i3}];
                  leader := [leader EXCEPT ![i3] = Nil];
                  if (Debug) {
                    print <<"ServerTimeout", i3, (currentTerm)[i3]>>;
                    idx0 := 1;
                    goto requestVoteLoop;
                  } else {
                    idx0 := 1;
                    goto requestVoteLoop;
                  };
                };
              };
            } else {
              await leaderTimeout;
              with (yielded_network31 = Len(((network)[srvId0]).queue)) {
                await (yielded_network31) = (0);
                await ((state)[srvId0]) \in ({Follower, Candidate});
                with (i4 = srvId0) {
                  state := [state EXCEPT ![i4] = Candidate];
                  currentTerm := [currentTerm EXCEPT ![i4] = ((currentTerm)[i4]) + (1)];
                  votedFor := [votedFor EXCEPT ![i4] = i4];
                  votesResponded := [votesResponded EXCEPT ![i4] = {i4}];
                  votesGranted := [votesGranted EXCEPT ![i4] = {i4}];
                  leader := [leader EXCEPT ![i4] = Nil];
                  if (Debug) {
                    print <<"ServerTimeout", i4, (currentTerm)[i4]>>;
                    idx0 := 1;
                    goto requestVoteLoop;
                  } else {
                    idx0 := 1;
                    goto requestVoteLoop;
                  };
                };
              };
            };
          };
        } else {
          await leaderTimeout;
          with (yielded_network32 = Len(((network)[srvId0]).queue)) {
            await (yielded_network32) = (0);
            await ((state)[srvId0]) \in ({Follower, Candidate});
            with (i5 = srvId0) {
              state := [state EXCEPT ![i5] = Candidate];
              currentTerm := [currentTerm EXCEPT ![i5] = ((currentTerm)[i5]) + (1)];
              votedFor := [votedFor EXCEPT ![i5] = i5];
              votesResponded := [votesResponded EXCEPT ![i5] = {i5}];
              votesGranted := [votesGranted EXCEPT ![i5] = {i5}];
              leader := [leader EXCEPT ![i5] = Nil];
              if (Debug) {
                print <<"ServerTimeout", i5, (currentTerm)[i5]>>;
                idx0 := 1;
                goto requestVoteLoop;
              } else {
                idx0 := 1;
                goto requestVoteLoop;
              };
            };
          };
        };
      } else {
        goto Done;
      };
    requestVoteLoop:
      if ((idx0) <= (NumServers)) {
        if (ExploreFail) {
          with (yielded_network40 = ((network)[srvId0]).enabled) {
            if (~ (yielded_network40)) {
              await FALSE;
              if ((idx0) # (srvId0)) {
                either {
                  with (value90 = [mtype |-> RequestVoteRequest, mterm |-> (currentTerm)[srvId0], mlastLogTerm |-> LastTerm((log)[srvId0]), mlastLogIndex |-> Len((log)[srvId0]), msource |-> srvId0, mdest |-> idx0]) {
                    await ((network)[idx0]).enabled;
                    await (Len(((network)[idx0]).queue)) < (BufferSize);
                    network := [network EXCEPT ![idx0] = [queue |-> Append(((network)[idx0]).queue, value90), enabled |-> ((network)[idx0]).enabled]];
                    idx0 := (idx0) + (1);
                    goto requestVoteLoop;
                  };
                } or {
                  with (yielded_fd28 = (fd)[idx0]) {
                    await yielded_fd28;
                    idx0 := (idx0) + (1);
                    goto requestVoteLoop;
                  };
                };
              } else {
                idx0 := (idx0) + (1);
                goto requestVoteLoop;
              };
            } else {
              if ((idx0) # (srvId0)) {
                either {
                  with (value91 = [mtype |-> RequestVoteRequest, mterm |-> (currentTerm)[srvId0], mlastLogTerm |-> LastTerm((log)[srvId0]), mlastLogIndex |-> Len((log)[srvId0]), msource |-> srvId0, mdest |-> idx0]) {
                    await ((network)[idx0]).enabled;
                    await (Len(((network)[idx0]).queue)) < (BufferSize);
                    network := [network EXCEPT ![idx0] = [queue |-> Append(((network)[idx0]).queue, value91), enabled |-> ((network)[idx0]).enabled]];
                    idx0 := (idx0) + (1);
                    goto requestVoteLoop;
                  };
                } or {
                  with (yielded_fd29 = (fd)[idx0]) {
                    await yielded_fd29;
                    idx0 := (idx0) + (1);
                    goto requestVoteLoop;
                  };
                };
              } else {
                idx0 := (idx0) + (1);
                goto requestVoteLoop;
              };
            };
          };
        } else {
          if ((idx0) # (srvId0)) {
            either {
              with (value92 = [mtype |-> RequestVoteRequest, mterm |-> (currentTerm)[srvId0], mlastLogTerm |-> LastTerm((log)[srvId0]), mlastLogIndex |-> Len((log)[srvId0]), msource |-> srvId0, mdest |-> idx0]) {
                await ((network)[idx0]).enabled;
                await (Len(((network)[idx0]).queue)) < (BufferSize);
                network := [network EXCEPT ![idx0] = [queue |-> Append(((network)[idx0]).queue, value92), enabled |-> ((network)[idx0]).enabled]];
                idx0 := (idx0) + (1);
                goto requestVoteLoop;
              };
            } or {
              with (yielded_fd210 = (fd)[idx0]) {
                await yielded_fd210;
                idx0 := (idx0) + (1);
                goto requestVoteLoop;
              };
            };
          } else {
            idx0 := (idx0) + (1);
            goto requestVoteLoop;
          };
        };
      } else {
        goto serverRequestVoteLoop;
      };
  }
  
  fair process (s2 \in ServerAppendEntriesSet)
    variables idx1; srvId1 = (appendEntriesSrvId)[self];
  {
    serverAppendEntriesLoop:
      await (Len((appendEntriesCh)[srvId1])) > (0);
      with (res0 = Head((appendEntriesCh)[srvId1])) {
        appendEntriesCh := [appendEntriesCh EXCEPT ![srvId1] = Tail((appendEntriesCh)[srvId1])];
        with (yielded_appendEntriesCh = res0) {
          if (yielded_appendEntriesCh) {
            if (ExploreFail) {
              with (yielded_network50 = ((network)[srvId1]).enabled) {
                if (~ (yielded_network50)) {
                  await FALSE;
                  await ((state)[srvId1]) = (Leader);
                  idx1 := 1;
                  goto appendEntriesLoop;
                } else {
                  await ((state)[srvId1]) = (Leader);
                  idx1 := 1;
                  goto appendEntriesLoop;
                };
              };
            } else {
              await ((state)[srvId1]) = (Leader);
              idx1 := 1;
              goto appendEntriesLoop;
            };
          } else {
            goto Done;
          };
        };
      };
    appendEntriesLoop:
      if ((((state)[srvId1]) = (Leader)) /\ ((idx1) <= (NumServers))) {
        if (ExploreFail) {
          with (yielded_network60 = ((network)[srvId1]).enabled) {
            if (~ (yielded_network60)) {
              await FALSE;
              if ((idx1) # (srvId1)) {
                with (
                  prevLogIndex3 = (((nextIndex)[srvId1])[idx1]) - (1), 
                  prevLogTerm3 = IF (prevLogIndex3) > (0) THEN (((log)[srvId1])[prevLogIndex3]).term ELSE 0, 
                  entries3 = SubSeq((log)[srvId1], ((nextIndex)[srvId1])[idx1], Len((log)[srvId1]))
                ) {
                  either {
                    with (value100 = [mtype |-> AppendEntriesRequest, mterm |-> (currentTerm)[srvId1], mprevLogIndex |-> prevLogIndex3, mprevLogTerm |-> prevLogTerm3, mentries |-> entries3, mcommitIndex |-> (commitIndex)[srvId1], msource |-> srvId1, mdest |-> idx1]) {
                      await ((network)[idx1]).enabled;
                      await (Len(((network)[idx1]).queue)) < (BufferSize);
                      network := [network EXCEPT ![idx1] = [queue |-> Append(((network)[idx1]).queue, value100), enabled |-> ((network)[idx1]).enabled]];
                      idx1 := (idx1) + (1);
                      goto appendEntriesLoop;
                    };
                  } or {
                    with (yielded_fd30 = (fd)[idx1]) {
                      await yielded_fd30;
                      idx1 := (idx1) + (1);
                      goto appendEntriesLoop;
                    };
                  };
                };
              } else {
                idx1 := (idx1) + (1);
                goto appendEntriesLoop;
              };
            } else {
              if ((idx1) # (srvId1)) {
                with (
                  prevLogIndex4 = (((nextIndex)[srvId1])[idx1]) - (1), 
                  prevLogTerm4 = IF (prevLogIndex4) > (0) THEN (((log)[srvId1])[prevLogIndex4]).term ELSE 0, 
                  entries4 = SubSeq((log)[srvId1], ((nextIndex)[srvId1])[idx1], Len((log)[srvId1]))
                ) {
                  either {
                    with (value101 = [mtype |-> AppendEntriesRequest, mterm |-> (currentTerm)[srvId1], mprevLogIndex |-> prevLogIndex4, mprevLogTerm |-> prevLogTerm4, mentries |-> entries4, mcommitIndex |-> (commitIndex)[srvId1], msource |-> srvId1, mdest |-> idx1]) {
                      await ((network)[idx1]).enabled;
                      await (Len(((network)[idx1]).queue)) < (BufferSize);
                      network := [network EXCEPT ![idx1] = [queue |-> Append(((network)[idx1]).queue, value101), enabled |-> ((network)[idx1]).enabled]];
                      idx1 := (idx1) + (1);
                      goto appendEntriesLoop;
                    };
                  } or {
                    with (yielded_fd31 = (fd)[idx1]) {
                      await yielded_fd31;
                      idx1 := (idx1) + (1);
                      goto appendEntriesLoop;
                    };
                  };
                };
              } else {
                idx1 := (idx1) + (1);
                goto appendEntriesLoop;
              };
            };
          };
        } else {
          if ((idx1) # (srvId1)) {
            with (
              prevLogIndex5 = (((nextIndex)[srvId1])[idx1]) - (1), 
              prevLogTerm5 = IF (prevLogIndex5) > (0) THEN (((log)[srvId1])[prevLogIndex5]).term ELSE 0, 
              entries5 = SubSeq((log)[srvId1], ((nextIndex)[srvId1])[idx1], Len((log)[srvId1]))
            ) {
              either {
                with (value102 = [mtype |-> AppendEntriesRequest, mterm |-> (currentTerm)[srvId1], mprevLogIndex |-> prevLogIndex5, mprevLogTerm |-> prevLogTerm5, mentries |-> entries5, mcommitIndex |-> (commitIndex)[srvId1], msource |-> srvId1, mdest |-> idx1]) {
                  await ((network)[idx1]).enabled;
                  await (Len(((network)[idx1]).queue)) < (BufferSize);
                  network := [network EXCEPT ![idx1] = [queue |-> Append(((network)[idx1]).queue, value102), enabled |-> ((network)[idx1]).enabled]];
                  idx1 := (idx1) + (1);
                  goto appendEntriesLoop;
                };
              } or {
                with (yielded_fd32 = (fd)[idx1]) {
                  await yielded_fd32;
                  idx1 := (idx1) + (1);
                  goto appendEntriesLoop;
                };
              };
            };
          } else {
            idx1 := (idx1) + (1);
            goto appendEntriesLoop;
          };
        };
      } else {
        goto serverAppendEntriesLoop;
      };
  }
  
  fair process (s3 \in ServerAdvanceCommitIndexSet)
    variables newCommitIndex = 0; srvId2 = (advanceCommitIndexSrvId)[self];
  {
    serverAdvanceCommitIndexLoop:
      if (TRUE) {
        if (ExploreFail) {
          with (yielded_network70 = ((network)[srvId2]).enabled) {
            if (~ (yielded_network70)) {
              await FALSE;
              await ((state)[srvId2]) = (Leader);
              with (
                i = srvId2, 
                maxAgreeIndex = FindMaxAgreeIndex((log)[i], i, (matchIndex)[i]), 
                nCommitIndex = IF ((maxAgreeIndex) # (Nil)) /\ (((((log)[i])[maxAgreeIndex]).term) = ((currentTerm)[i])) THEN maxAgreeIndex ELSE (commitIndex)[i]
              ) {
                newCommitIndex := nCommitIndex;
                assert (newCommitIndex) >= ((commitIndex)[i]);
                goto applyLoop;
              };
            } else {
              await ((state)[srvId2]) = (Leader);
              with (
                i = srvId2, 
                maxAgreeIndex = FindMaxAgreeIndex((log)[i], i, (matchIndex)[i]), 
                nCommitIndex = IF ((maxAgreeIndex) # (Nil)) /\ (((((log)[i])[maxAgreeIndex]).term) = ((currentTerm)[i])) THEN maxAgreeIndex ELSE (commitIndex)[i]
              ) {
                newCommitIndex := nCommitIndex;
                assert (newCommitIndex) >= ((commitIndex)[i]);
                goto applyLoop;
              };
            };
          };
        } else {
          await ((state)[srvId2]) = (Leader);
          with (
            i = srvId2, 
            maxAgreeIndex = FindMaxAgreeIndex((log)[i], i, (matchIndex)[i]), 
            nCommitIndex = IF ((maxAgreeIndex) # (Nil)) /\ (((((log)[i])[maxAgreeIndex]).term) = ((currentTerm)[i])) THEN maxAgreeIndex ELSE (commitIndex)[i]
          ) {
            newCommitIndex := nCommitIndex;
            assert (newCommitIndex) >= ((commitIndex)[i]);
            goto applyLoop;
          };
        };
      } else {
        goto Done;
      };
    applyLoop:
      if (((commitIndex)[srvId2]) < (newCommitIndex)) {
        if (ExploreFail) {
          with (yielded_network80 = ((network)[srvId2]).enabled) {
            if (~ (yielded_network80)) {
              await FALSE;
              commitIndex := [commitIndex EXCEPT ![srvId2] = ((commitIndex)[srvId2]) + (1)];
              with (
                i = srvId2, 
                k = (commitIndex)[i], 
                entry = ((log)[i])[k], 
                cmd = (entry).cmd, 
                respType = IF ((cmd).type) = (Put) THEN ClientPutResponse ELSE ClientGetResponse
              ) {
                if (((cmd).type) = (Put)) {
                  sm := [sm EXCEPT ![i] = ((sm)[i]) @@ (((cmd).key) :> ((cmd).value))];
                  smDomain := [smDomain EXCEPT ![i] = ((smDomain)[i]) \union ({(cmd).key})];
                  with (
                    reqOk = ((cmd).key) \in ((smDomain)[i]), 
                    value113 = [mtype |-> respType, msuccess |-> TRUE, mresponse |-> [idx |-> (cmd).idx, key |-> (cmd).key, value |-> IF reqOk THEN ((sm)[i])[(cmd).key] ELSE Nil, ok |-> reqOk], mleaderHint |-> i, msource |-> i, mdest |-> (entry).client]
                  ) {
                    await ((network)[(entry).client]).enabled;
                    await (Len(((network)[(entry).client]).queue)) < (BufferSize);
                    network := [network EXCEPT ![(entry).client] = [queue |-> Append(((network)[(entry).client]).queue, value113), enabled |-> ((network)[(entry).client]).enabled]];
                    goto applyLoop;
                  };
                } else {
                  with (
                    reqOk = ((cmd).key) \in ((smDomain)[i]), 
                    value114 = [mtype |-> respType, msuccess |-> TRUE, mresponse |-> [idx |-> (cmd).idx, key |-> (cmd).key, value |-> IF reqOk THEN ((sm)[i])[(cmd).key] ELSE Nil, ok |-> reqOk], mleaderHint |-> i, msource |-> i, mdest |-> (entry).client]
                  ) {
                    await ((network)[(entry).client]).enabled;
                    await (Len(((network)[(entry).client]).queue)) < (BufferSize);
                    network := [network EXCEPT ![(entry).client] = [queue |-> Append(((network)[(entry).client]).queue, value114), enabled |-> ((network)[(entry).client]).enabled]];
                    goto applyLoop;
                  };
                };
              };
            } else {
              commitIndex := [commitIndex EXCEPT ![srvId2] = ((commitIndex)[srvId2]) + (1)];
              with (
                i = srvId2, 
                k = (commitIndex)[i], 
                entry = ((log)[i])[k], 
                cmd = (entry).cmd, 
                respType = IF ((cmd).type) = (Put) THEN ClientPutResponse ELSE ClientGetResponse
              ) {
                if (((cmd).type) = (Put)) {
                  sm := [sm EXCEPT ![i] = ((sm)[i]) @@ (((cmd).key) :> ((cmd).value))];
                  smDomain := [smDomain EXCEPT ![i] = ((smDomain)[i]) \union ({(cmd).key})];
                  with (
                    reqOk = ((cmd).key) \in ((smDomain)[i]), 
                    value115 = [mtype |-> respType, msuccess |-> TRUE, mresponse |-> [idx |-> (cmd).idx, key |-> (cmd).key, value |-> IF reqOk THEN ((sm)[i])[(cmd).key] ELSE Nil, ok |-> reqOk], mleaderHint |-> i, msource |-> i, mdest |-> (entry).client]
                  ) {
                    await ((network)[(entry).client]).enabled;
                    await (Len(((network)[(entry).client]).queue)) < (BufferSize);
                    network := [network EXCEPT ![(entry).client] = [queue |-> Append(((network)[(entry).client]).queue, value115), enabled |-> ((network)[(entry).client]).enabled]];
                    goto applyLoop;
                  };
                } else {
                  with (
                    reqOk = ((cmd).key) \in ((smDomain)[i]), 
                    value116 = [mtype |-> respType, msuccess |-> TRUE, mresponse |-> [idx |-> (cmd).idx, key |-> (cmd).key, value |-> IF reqOk THEN ((sm)[i])[(cmd).key] ELSE Nil, ok |-> reqOk], mleaderHint |-> i, msource |-> i, mdest |-> (entry).client]
                  ) {
                    await ((network)[(entry).client]).enabled;
                    await (Len(((network)[(entry).client]).queue)) < (BufferSize);
                    network := [network EXCEPT ![(entry).client] = [queue |-> Append(((network)[(entry).client]).queue, value116), enabled |-> ((network)[(entry).client]).enabled]];
                    goto applyLoop;
                  };
                };
              };
            };
          };
        } else {
          commitIndex := [commitIndex EXCEPT ![srvId2] = ((commitIndex)[srvId2]) + (1)];
          with (
            i = srvId2, 
            k = (commitIndex)[i], 
            entry = ((log)[i])[k], 
            cmd = (entry).cmd, 
            respType = IF ((cmd).type) = (Put) THEN ClientPutResponse ELSE ClientGetResponse
          ) {
            if (((cmd).type) = (Put)) {
              sm := [sm EXCEPT ![i] = ((sm)[i]) @@ (((cmd).key) :> ((cmd).value))];
              smDomain := [smDomain EXCEPT ![i] = ((smDomain)[i]) \union ({(cmd).key})];
              with (
                reqOk = ((cmd).key) \in ((smDomain)[i]), 
                value117 = [mtype |-> respType, msuccess |-> TRUE, mresponse |-> [idx |-> (cmd).idx, key |-> (cmd).key, value |-> IF reqOk THEN ((sm)[i])[(cmd).key] ELSE Nil, ok |-> reqOk], mleaderHint |-> i, msource |-> i, mdest |-> (entry).client]
              ) {
                await ((network)[(entry).client]).enabled;
                await (Len(((network)[(entry).client]).queue)) < (BufferSize);
                network := [network EXCEPT ![(entry).client] = [queue |-> Append(((network)[(entry).client]).queue, value117), enabled |-> ((network)[(entry).client]).enabled]];
                goto applyLoop;
              };
            } else {
              with (
                reqOk = ((cmd).key) \in ((smDomain)[i]), 
                value118 = [mtype |-> respType, msuccess |-> TRUE, mresponse |-> [idx |-> (cmd).idx, key |-> (cmd).key, value |-> IF reqOk THEN ((sm)[i])[(cmd).key] ELSE Nil, ok |-> reqOk], mleaderHint |-> i, msource |-> i, mdest |-> (entry).client]
              ) {
                await ((network)[(entry).client]).enabled;
                await (Len(((network)[(entry).client]).queue)) < (BufferSize);
                network := [network EXCEPT ![(entry).client] = [queue |-> Append(((network)[(entry).client]).queue, value118), enabled |-> ((network)[(entry).client]).enabled]];
                goto applyLoop;
              };
            };
          };
        };
      } else {
        goto serverAdvanceCommitIndexLoop;
      };
  }
  
  fair process (s4 \in ServerBecomeLeaderSet)
    variables srvId3 = (becomeLeaderSrvId)[self];
  {
    serverBecomeLeaderLoop:
      await (Len((becomeLeaderCh)[srvId3])) > (0);
      with (res1 = Head((becomeLeaderCh)[srvId3])) {
        becomeLeaderCh := [becomeLeaderCh EXCEPT ![srvId3] = Tail((becomeLeaderCh)[srvId3])];
        with (yielded_becomeLeaderCh = res1) {
          if (yielded_becomeLeaderCh) {
            if (ExploreFail) {
              with (yielded_network90 = ((network)[srvId3]).enabled) {
                if (~ (yielded_network90)) {
                  await FALSE;
                  await ((state)[srvId3]) = (Candidate);
                  await isQuorum((votesGranted)[srvId3]);
                  with (i = srvId3) {
                    state := [state EXCEPT ![i] = Leader];
                    nextIndex := [nextIndex EXCEPT ![i] = [j \in ServerSet |-> (Len((log)[i])) + (1)]];
                    matchIndex := [matchIndex EXCEPT ![i] = [j \in ServerSet |-> 0]];
                    leader := [leader EXCEPT ![i] = i];
                    with (value120 = TRUE) {
                      appendEntriesCh := [appendEntriesCh EXCEPT ![srvId3] = Append((appendEntriesCh)[srvId3], value120)];
                      if (Debug) {
                        print <<"BecomeLeader", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                        goto serverBecomeLeaderLoop;
                      } else {
                        goto serverBecomeLeaderLoop;
                      };
                    };
                  };
                } else {
                  await ((state)[srvId3]) = (Candidate);
                  await isQuorum((votesGranted)[srvId3]);
                  with (i = srvId3) {
                    state := [state EXCEPT ![i] = Leader];
                    nextIndex := [nextIndex EXCEPT ![i] = [j \in ServerSet |-> (Len((log)[i])) + (1)]];
                    matchIndex := [matchIndex EXCEPT ![i] = [j \in ServerSet |-> 0]];
                    leader := [leader EXCEPT ![i] = i];
                    with (value121 = TRUE) {
                      appendEntriesCh := [appendEntriesCh EXCEPT ![srvId3] = Append((appendEntriesCh)[srvId3], value121)];
                      if (Debug) {
                        print <<"BecomeLeader", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                        goto serverBecomeLeaderLoop;
                      } else {
                        goto serverBecomeLeaderLoop;
                      };
                    };
                  };
                };
              };
            } else {
              await ((state)[srvId3]) = (Candidate);
              await isQuorum((votesGranted)[srvId3]);
              with (i = srvId3) {
                state := [state EXCEPT ![i] = Leader];
                nextIndex := [nextIndex EXCEPT ![i] = [j \in ServerSet |-> (Len((log)[i])) + (1)]];
                matchIndex := [matchIndex EXCEPT ![i] = [j \in ServerSet |-> 0]];
                leader := [leader EXCEPT ![i] = i];
                with (value122 = TRUE) {
                  appendEntriesCh := [appendEntriesCh EXCEPT ![srvId3] = Append((appendEntriesCh)[srvId3], value122)];
                  if (Debug) {
                    print <<"BecomeLeader", i, (currentTerm)[i], (state)[i], (leader)[i]>>;
                    goto serverBecomeLeaderLoop;
                  } else {
                    goto serverBecomeLeaderLoop;
                  };
                };
              };
            };
          } else {
            goto Done;
          };
        };
      };
  }
  
  fair process (client \in ClientSet)
    variables leader0 = Nil; req; resp; reqIdx = 0; timeout = FALSE;
  {
    clientLoop:
      if (TRUE) {
        await (Len(reqCh)) > (0);
        with (res20 = Head(reqCh)) {
          reqCh := Tail(reqCh);
          with (yielded_reqCh0 = res20) {
            req := yielded_reqCh0;
            reqIdx := (reqIdx) + (1);
            goto sndReq;
          };
        };
      } else {
        goto Done;
      };
    sndReq:
      if ((leader0) = (Nil)) {
        with (srv1 \in ServerSet) {
          leader0 := srv1;
          if (Debug) {
            print <<"ClientSndReq", self, leader0, reqIdx, req>>;
            if (((req).type) = (Put)) {
              either {
                with (value130 = [mtype |-> ClientPutRequest, mcmd |-> [idx |-> reqIdx, type |-> Put, key |-> (req).key, value |-> (req).value], msource |-> self, mdest |-> leader0]) {
                  await ((network)[leader0]).enabled;
                  await (Len(((network)[leader0]).queue)) < (BufferSize);
                  network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value130), enabled |-> ((network)[leader0]).enabled]];
                  goto rcvResp;
                };
              } or {
                with (yielded_fd40 = (fd)[leader0]) {
                  await yielded_fd40;
                  goto rcvResp;
                };
              };
            } else {
              if (((req).type) = (Get)) {
                either {
                  with (value140 = [mtype |-> ClientGetRequest, mcmd |-> [idx |-> reqIdx, type |-> Get, key |-> (req).key], msource |-> self, mdest |-> leader0]) {
                    await ((network)[leader0]).enabled;
                    await (Len(((network)[leader0]).queue)) < (BufferSize);
                    network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value140), enabled |-> ((network)[leader0]).enabled]];
                    goto rcvResp;
                  };
                } or {
                  with (yielded_fd50 = (fd)[leader0]) {
                    await yielded_fd50;
                    goto rcvResp;
                  };
                };
              } else {
                goto rcvResp;
              };
            };
          } else {
            if (((req).type) = (Put)) {
              either {
                with (value131 = [mtype |-> ClientPutRequest, mcmd |-> [idx |-> reqIdx, type |-> Put, key |-> (req).key, value |-> (req).value], msource |-> self, mdest |-> leader0]) {
                  await ((network)[leader0]).enabled;
                  await (Len(((network)[leader0]).queue)) < (BufferSize);
                  network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value131), enabled |-> ((network)[leader0]).enabled]];
                  goto rcvResp;
                };
              } or {
                with (yielded_fd41 = (fd)[leader0]) {
                  await yielded_fd41;
                  goto rcvResp;
                };
              };
            } else {
              if (((req).type) = (Get)) {
                either {
                  with (value141 = [mtype |-> ClientGetRequest, mcmd |-> [idx |-> reqIdx, type |-> Get, key |-> (req).key], msource |-> self, mdest |-> leader0]) {
                    await ((network)[leader0]).enabled;
                    await (Len(((network)[leader0]).queue)) < (BufferSize);
                    network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value141), enabled |-> ((network)[leader0]).enabled]];
                    goto rcvResp;
                  };
                } or {
                  with (yielded_fd51 = (fd)[leader0]) {
                    await yielded_fd51;
                    goto rcvResp;
                  };
                };
              } else {
                goto rcvResp;
              };
            };
          };
        };
      } else {
        if (Debug) {
          print <<"ClientSndReq", self, leader0, reqIdx, req>>;
          if (((req).type) = (Put)) {
            either {
              with (value132 = [mtype |-> ClientPutRequest, mcmd |-> [idx |-> reqIdx, type |-> Put, key |-> (req).key, value |-> (req).value], msource |-> self, mdest |-> leader0]) {
                await ((network)[leader0]).enabled;
                await (Len(((network)[leader0]).queue)) < (BufferSize);
                network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value132), enabled |-> ((network)[leader0]).enabled]];
                goto rcvResp;
              };
            } or {
              with (yielded_fd42 = (fd)[leader0]) {
                await yielded_fd42;
                goto rcvResp;
              };
            };
          } else {
            if (((req).type) = (Get)) {
              either {
                with (value142 = [mtype |-> ClientGetRequest, mcmd |-> [idx |-> reqIdx, type |-> Get, key |-> (req).key], msource |-> self, mdest |-> leader0]) {
                  await ((network)[leader0]).enabled;
                  await (Len(((network)[leader0]).queue)) < (BufferSize);
                  network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value142), enabled |-> ((network)[leader0]).enabled]];
                  goto rcvResp;
                };
              } or {
                with (yielded_fd52 = (fd)[leader0]) {
                  await yielded_fd52;
                  goto rcvResp;
                };
              };
            } else {
              goto rcvResp;
            };
          };
        } else {
          if (((req).type) = (Put)) {
            either {
              with (value133 = [mtype |-> ClientPutRequest, mcmd |-> [idx |-> reqIdx, type |-> Put, key |-> (req).key, value |-> (req).value], msource |-> self, mdest |-> leader0]) {
                await ((network)[leader0]).enabled;
                await (Len(((network)[leader0]).queue)) < (BufferSize);
                network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value133), enabled |-> ((network)[leader0]).enabled]];
                goto rcvResp;
              };
            } or {
              with (yielded_fd43 = (fd)[leader0]) {
                await yielded_fd43;
                goto rcvResp;
              };
            };
          } else {
            if (((req).type) = (Get)) {
              either {
                with (value143 = [mtype |-> ClientGetRequest, mcmd |-> [idx |-> reqIdx, type |-> Get, key |-> (req).key], msource |-> self, mdest |-> leader0]) {
                  await ((network)[leader0]).enabled;
                  await (Len(((network)[leader0]).queue)) < (BufferSize);
                  network := [network EXCEPT ![leader0] = [queue |-> Append(((network)[leader0]).queue, value143), enabled |-> ((network)[leader0]).enabled]];
                  goto rcvResp;
                };
              } or {
                with (yielded_fd53 = (fd)[leader0]) {
                  await yielded_fd53;
                  goto rcvResp;
                };
              };
            } else {
              goto rcvResp;
            };
          };
        };
      };
    rcvResp:
      either {
        assert ((network)[self]).enabled;
        await (Len(((network)[self]).queue)) > (0);
        with (readMsg10 = Head(((network)[self]).queue)) {
          network := [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]];
          with (yielded_network100 = readMsg10) {
            resp := yielded_network100;
            if (Debug) {
              print <<"ClientRcvResp", self, leader0, reqIdx, resp>>;
              assert ((resp).mdest) = (self);
              if ((((resp).mresponse).idx) # (reqIdx)) {
                goto rcvResp;
              } else {
                leader0 := (resp).mleaderHint;
                assert ((((req).type) = (Get)) => (((resp).mtype) = (ClientGetResponse))) /\ ((((req).type) = (Put)) => (((resp).mtype) = (ClientPutResponse)));
                if (~ ((resp).msuccess)) {
                  goto sndReq;
                } else {
                  assert ((((resp).mresponse).idx) = (reqIdx)) /\ ((((resp).mresponse).key) = ((req).key));
                  respCh := resp;
                  if (Debug) {
                    print <<"ClientRcvChDone", self, leader0, reqIdx, resp>>;
                    goto clientLoop;
                  } else {
                    goto clientLoop;
                  };
                };
              };
            } else {
              assert ((resp).mdest) = (self);
              if ((((resp).mresponse).idx) # (reqIdx)) {
                goto rcvResp;
              } else {
                leader0 := (resp).mleaderHint;
                assert ((((req).type) = (Get)) => (((resp).mtype) = (ClientGetResponse))) /\ ((((req).type) = (Put)) => (((resp).mtype) = (ClientPutResponse)));
                if (~ ((resp).msuccess)) {
                  goto sndReq;
                } else {
                  assert ((((resp).mresponse).idx) = (reqIdx)) /\ ((((resp).mresponse).key) = ((req).key));
                  respCh := resp;
                  if (Debug) {
                    print <<"ClientRcvChDone", self, leader0, reqIdx, resp>>;
                    goto clientLoop;
                  } else {
                    goto clientLoop;
                  };
                };
              };
            };
          };
        };
      } or {
        with (
          yielded_fd60 = (fd)[leader0], 
          yielded_network110 = Len(((network)[self]).queue)
        ) {
          await ((yielded_fd60) /\ ((yielded_network110) = (0))) \/ (timeout);
          if (Debug) {
            print <<"ClientTimeout", self, leader0, reqIdx>>;
            leader0 := Nil;
            goto sndReq;
          } else {
            leader0 := Nil;
            goto sndReq;
          };
        };
      };
  }
  
  fair process (crasher \in ServerCrasherSet)
    variables srvId4 = (crasherSrvId)[self];
  {
    serverCrash:
      with (value150 = FALSE) {
        network := [network EXCEPT ![srvId4] = [queue |-> ((network)[srvId4]).queue, enabled |-> value150]];
        goto fdUpdate;
      };
    fdUpdate:
      with (value160 = TRUE) {
        fd := [fd EXCEPT ![srvId4] = value160];
        goto Done;
      };
  }
}

\* END PLUSCAL TRANSLATION

********************)
\* BEGIN TRANSLATION (chksum(pcal) = "542a6c39" /\ chksum(tla) = "817685fe") PCal-18049938ece8066a38eb5044080cf45c
CONSTANT defaultInitValue
VARIABLES network, fd, state, currentTerm, commitIndex, nextIndex, matchIndex, 
          log, plog, votedFor, votesResponded, votesGranted, leader, sm, 
          smDomain, leaderTimeout, appendEntriesCh, becomeLeaderCh, allReqs, 
          reqCh, respCh, requestVoteSrvId, appendEntriesSrvId, 
          advanceCommitIndexSrvId, becomeLeaderSrvId, crasherSrvId, pc

(* define statement *)
Follower == "follower"
Candidate == "candidate"
Leader == "leader"
RequestVoteRequest == "rvq"
RequestVoteResponse == "rvp"
AppendEntriesRequest == "apq"
AppendEntriesResponse == "app"
ClientPutRequest == "cpq"
ClientPutResponse == "cpp"
ClientGetRequest == "cgq"
ClientGetResponse == "cgp"
Key1 == "key1"
Key2 == "key2"
Value1 == "value1"
LastTerm(xlog) == IF (Len(xlog)) = (0) THEN 0 ELSE ((xlog)[Len(xlog)]).term
ServerRequestVoteSet == (((1) * (NumServers)) + (1)) .. ((2) * (NumServers))
ServerAppendEntriesSet == (((2) * (NumServers)) + (1)) .. ((3) * (NumServers))
ServerAdvanceCommitIndexSet == (((3) * (NumServers)) + (1)) .. ((4) * (NumServers))
ServerBecomeLeaderSet == (((4) * (NumServers)) + (1)) .. ((5) * (NumServers))
ServerCrasherSet == IF ExploreFail THEN (((5) * (NumServers)) + (1)) .. (((5) * (NumServers)) + (MaxNodeFail)) ELSE {}
ClientSet == (((6) * (NumServers)) + (1)) .. (((6) * (NumServers)) + (NumClients))
NodeSet == (ServerSet) \union (ClientSet)
KeySet == {}

VARIABLES idx, m, srvId, idx0, srvId0, idx1, srvId1, newCommitIndex, srvId2, 
          srvId3, leader0, req, resp, reqIdx, timeout, srvId4

vars == << network, fd, state, currentTerm, commitIndex, nextIndex, 
           matchIndex, log, plog, votedFor, votesResponded, votesGranted, 
           leader, sm, smDomain, leaderTimeout, appendEntriesCh, 
           becomeLeaderCh, allReqs, reqCh, respCh, requestVoteSrvId, 
           appendEntriesSrvId, advanceCommitIndexSrvId, becomeLeaderSrvId, 
           crasherSrvId, pc, idx, m, srvId, idx0, srvId0, idx1, srvId1, 
           newCommitIndex, srvId2, srvId3, leader0, req, resp, reqIdx, 
           timeout, srvId4 >>

ProcSet == (ServerSet) \cup (ServerRequestVoteSet) \cup (ServerAppendEntriesSet) \cup (ServerAdvanceCommitIndexSet) \cup (ServerBecomeLeaderSet) \cup (ClientSet) \cup (ServerCrasherSet)

Init == (* Global variables *)
        /\ network = [i \in NodeSet |-> [queue |-> <<>>, enabled |-> TRUE]]
        /\ fd = [i \in ServerSet |-> FALSE]
        /\ state = [i \in ServerSet |-> Follower]
        /\ currentTerm = [i \in ServerSet |-> 1]
        /\ commitIndex = [i \in ServerSet |-> 0]
        /\ nextIndex = [i \in ServerSet |-> [j \in ServerSet |-> 1]]
        /\ matchIndex = [i \in ServerSet |-> [j \in ServerSet |-> 0]]
        /\ log = [i \in ServerSet |-> <<>>]
        /\ plog = [i \in ServerSet |-> <<>>]
        /\ votedFor = [i \in ServerSet |-> Nil]
        /\ votesResponded = [i \in ServerSet |-> {}]
        /\ votesGranted = [i \in ServerSet |-> {}]
        /\ leader = [i \in ServerSet |-> Nil]
        /\ sm = [i \in ServerSet |-> [k \in KeySet |-> Nil]]
        /\ smDomain = [i \in ServerSet |-> KeySet]
        /\ leaderTimeout = TRUE
        /\ appendEntriesCh = [i \in ServerSet |-> <<>>]
        /\ becomeLeaderCh = [i \in ServerSet |-> IF (NumServers) > (1) THEN <<>> ELSE <<TRUE>>]
        /\ allReqs = <<[type |-> Put, key |-> Key1, value |-> Value1], [type |-> Get, key |-> Key2], [type |-> Get, key |-> Key1]>>
        /\ reqCh = SubSeq(allReqs, 1, NumRequests)
        /\ respCh = defaultInitValue
        /\ requestVoteSrvId = [i \in ServerRequestVoteSet |-> (i) - ((1) * (NumServers))]
        /\ appendEntriesSrvId = [i \in ServerAppendEntriesSet |-> (i) - ((2) * (NumServers))]
        /\ advanceCommitIndexSrvId = [i \in ServerAdvanceCommitIndexSet |-> (i) - ((3) * (NumServers))]
        /\ becomeLeaderSrvId = [i \in ServerBecomeLeaderSet |-> (i) - ((4) * (NumServers))]
        /\ crasherSrvId = [i \in ServerCrasherSet |-> (i) - ((5) * (NumServers))]
        (* Process s0 *)
        /\ idx = [self \in ServerSet |-> 1]
        /\ m = [self \in ServerSet |-> defaultInitValue]
        /\ srvId = [self \in ServerSet |-> self]
        (* Process s1 *)
        /\ idx0 = [self \in ServerRequestVoteSet |-> 1]
        /\ srvId0 = [self \in ServerRequestVoteSet |-> (requestVoteSrvId)[self]]
        (* Process s2 *)
        /\ idx1 = [self \in ServerAppendEntriesSet |-> defaultInitValue]
        /\ srvId1 = [self \in ServerAppendEntriesSet |-> (appendEntriesSrvId)[self]]
        (* Process s3 *)
        /\ newCommitIndex = [self \in ServerAdvanceCommitIndexSet |-> 0]
        /\ srvId2 = [self \in ServerAdvanceCommitIndexSet |-> (advanceCommitIndexSrvId)[self]]
        (* Process s4 *)
        /\ srvId3 = [self \in ServerBecomeLeaderSet |-> (becomeLeaderSrvId)[self]]
        (* Process client *)
        /\ leader0 = [self \in ClientSet |-> Nil]
        /\ req = [self \in ClientSet |-> defaultInitValue]
        /\ resp = [self \in ClientSet |-> defaultInitValue]
        /\ reqIdx = [self \in ClientSet |-> 0]
        /\ timeout = [self \in ClientSet |-> FALSE]
        (* Process crasher *)
        /\ srvId4 = [self \in ServerCrasherSet |-> (crasherSrvId)[self]]
        /\ pc = [self \in ProcSet |-> CASE self \in ServerSet -> "serverLoop"
                                        [] self \in ServerRequestVoteSet -> "serverRequestVoteLoop"
                                        [] self \in ServerAppendEntriesSet -> "serverAppendEntriesLoop"
                                        [] self \in ServerAdvanceCommitIndexSet -> "serverAdvanceCommitIndexLoop"
                                        [] self \in ServerBecomeLeaderSet -> "serverBecomeLeaderLoop"
                                        [] self \in ClientSet -> "clientLoop"
                                        [] self \in ServerCrasherSet -> "serverCrash"]

serverLoop(self) == /\ pc[self] = "serverLoop"
                    /\ IF TRUE
                          THEN /\ IF ExploreFail
                                     THEN /\ LET yielded_network12 == ((network)[self]).enabled IN
                                               IF ~ (yielded_network12)
                                                  THEN /\ FALSE
                                                       /\ Assert(((network)[self]).enabled, 
                                                                 "Failure of assertion at line 977, column 15.")
                                                       /\ (Len(((network)[self]).queue)) > (0)
                                                       /\ LET readMsg00 == Head(((network)[self]).queue) IN
                                                            /\ network' = [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]]
                                                            /\ LET yielded_network00 == readMsg00 IN
                                                                 /\ m' = [m EXCEPT ![self] = yielded_network00]
                                                                 /\ Assert(((m'[self]).mdest) = (self), 
                                                                           "Failure of assertion at line 983, column 19.")
                                                                 /\ pc' = [pc EXCEPT ![self] = "handleMsg"]
                                                  ELSE /\ Assert(((network)[self]).enabled, 
                                                                 "Failure of assertion at line 988, column 15.")
                                                       /\ (Len(((network)[self]).queue)) > (0)
                                                       /\ LET readMsg01 == Head(((network)[self]).queue) IN
                                                            /\ network' = [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]]
                                                            /\ LET yielded_network01 == readMsg01 IN
                                                                 /\ m' = [m EXCEPT ![self] = yielded_network01]
                                                                 /\ Assert(((m'[self]).mdest) = (self), 
                                                                           "Failure of assertion at line 994, column 19.")
                                                                 /\ pc' = [pc EXCEPT ![self] = "handleMsg"]
                                     ELSE /\ Assert(((network)[self]).enabled, 
                                                    "Failure of assertion at line 1001, column 11.")
                                          /\ (Len(((network)[self]).queue)) > (0)
                                          /\ LET readMsg02 == Head(((network)[self]).queue) IN
                                               /\ network' = [network EXCEPT ![self] = [queue |-> Tail(((network)[self]).queue), enabled |-> ((network)[self]).enabled]]
                                               /\ LET yielded_network02 == readMsg02 IN
                                                    /\ m' = [m EXCEPT ![self] = yielded_network02]
                                                    /\ Assert(((m'[self]).mdest) = (self), 
                                                              "Failure of assertion at line 1007, column 15.")
                                                    /\ pc' = [pc EXCEPT ![self] = "handleMsg"]
                          ELSE /\ pc' = [pc EXCEPT ![self] = "Done"]
                               /\ UNCHANGED << network, m >>
                    /\ UNCHANGED << fd, state, currentTerm, commitIndex, 
                                    nextIndex, matchIndex, log, plog, votedFor, 
                                    votesResponded, votesGranted, leader, sm, 
                                    smDomain, leaderTimeout, appendEntriesCh, 
                                    becomeLeaderCh, allReqs, reqCh, respCh, 
                                    requestVoteSrvId, appendEntriesSrvId, 
                                    advanceCommitIndexSrvId, becomeLeaderSrvId, 
                                    crasherSrvId, idx, srvId, idx0, srvId0, 
                                    idx1, srvId1, newCommitIndex, srvId2, 
                                    srvId3, leader0, req, resp, reqIdx, 
                                    timeout, srvId4 >>

handleMsg(self) == /\ pc[self] = "handleMsg"
                   /\ IF ExploreFail
                         THEN /\ LET yielded_network13 == ((network)[self]).enabled IN
                                   IF ~ (yielded_network13)
                                      THEN /\ FALSE
                                           /\ IF ((m[self]).mtype) = (RequestVoteRequest)
                                                 THEN /\ IF ((m[self]).mterm) > ((currentTerm)[self])
                                                            THEN /\ currentTerm' = [currentTerm EXCEPT ![self] = (m[self]).mterm]
                                                                 /\ state' = [state EXCEPT ![self] = Follower]
                                                                 /\ LET votedFor3 == [votedFor EXCEPT ![self] = Nil] IN
                                                                      /\ leader' = [leader EXCEPT ![self] = Nil]
                                                                      /\ LET i == self IN
                                                                           LET j == (m[self]).msource IN
                                                                             LET logOK == (((m[self]).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m[self]).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m[self]).mlastLogIndex) >= (Len((log)[i])))) IN
                                                                               LET grant == ((((m[self]).mterm) = ((currentTerm')[i])) /\ (logOK)) /\ (((votedFor3)[self]) \in ({Nil, j})) IN
                                                                                 /\ Assert(((m[self]).mterm) <= ((currentTerm')[i]), 
                                                                                           "Failure of assertion at line 1032, column 21.")
                                                                                 /\ IF grant
                                                                                       THEN /\ votedFor' = [votedFor3 EXCEPT ![i] = j]
                                                                                            /\ \/ /\ LET value00 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm')[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                       /\ ((network)[j]).enabled
                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value00), enabled |-> ((network)[j]).enabled]]
                                                                                                       /\ IF Debug
                                                                                                             THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                             ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                               \/ /\ LET yielded_fd7 == (fd)[j] IN
                                                                                                       /\ yielded_fd7
                                                                                                       /\ IF Debug
                                                                                                             THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                             ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                  /\ UNCHANGED network
                                                                                       ELSE /\ \/ /\ LET value01 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm')[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                       /\ ((network)[j]).enabled
                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value01), enabled |-> ((network)[j]).enabled]]
                                                                                                       /\ IF Debug
                                                                                                             THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                  /\ votedFor' = votedFor3
                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                             ELSE /\ votedFor' = votedFor3
                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                               \/ /\ LET yielded_fd8 == (fd)[j] IN
                                                                                                       /\ yielded_fd8
                                                                                                       /\ IF Debug
                                                                                                             THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm')[i], grant>>)
                                                                                                                  /\ votedFor' = votedFor3
                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                             ELSE /\ votedFor' = votedFor3
                                                                                                                  /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                  /\ UNCHANGED network
                                                            ELSE /\ LET i == self IN
                                                                      LET j == (m[self]).msource IN
                                                                        LET logOK == (((m[self]).mlastLogTerm) > (LastTerm((log)[i]))) \/ ((((m[self]).mlastLogTerm) = (LastTerm((log)[i]))) /\ (((m[self]).mlastLogIndex) >= (Len((log)[i])))) IN
                                                                          LET grant == ((((m[self]).mterm) = ((currentTerm)[i])) /\ (logOK)) /\ (((votedFor)[self]) \in ({Nil, j})) IN
                                                                            /\ Assert(((m[self]).mterm) <= ((currentTerm)[i]), 
                                                                                      "Failure of assertion at line 1096, column 19.")
                                                                            /\ IF grant
                                                                                  THEN /\ votedFor' = [votedFor EXCEPT ![i] = j]
                                                                                       /\ \/ /\ LET value02 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                  /\ ((network)[j]).enabled
                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value02), enabled |-> ((network)[j]).enabled]]
                                                                                                  /\ IF Debug
                                                                                                        THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                        ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                          \/ /\ LET yielded_fd9 == (fd)[j] IN
                                                                                                  /\ yielded_fd9
                                                                                                  /\ IF Debug
                                                                                                        THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                        ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                             /\ UNCHANGED network
                                                                                  ELSE /\ \/ /\ LET value03 == [mtype |-> RequestVoteResponse, mterm |-> (currentTerm)[i], mvoteGranted |-> grant, msource |-> i, mdest |-> j] IN
                                                                                                  /\ ((network)[j]).enabled
                                                                                                  /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                  /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value03), enabled |-> ((network)[j]).enabled]]
                                                                                                  /\ IF Debug
                                                                                                        THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                        ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                          \/ /\ LET yielded_fd10 == (fd)[j] IN
                                                                                                  /\ yielded_fd10
                                                                                                  /\ IF Debug
                                                                                                        THEN /\ PrintT(<<"HandleRequestVoteRequest", i, j, (currentTerm)[i], grant>>)
                                                                                                             /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                        ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                             /\ UNCHANGED network
                                                                                       /\ UNCHANGED votedFor
                                                                 /\ UNCHANGED << state, 
                                                                                 currentTerm, 
                                                                                 leader >>
                                                      /\ UNCHANGED << commitIndex, 
                                                                      nextIndex, 
                                                                      matchIndex, 
                                                                      log, 
                                                                      plog, 
                                                                      votesResponded, 
                                                                      votesGranted, 
                                                                      sm, 
                                                                      smDomain, 
                                                                      leaderTimeout, 
                                                                      appendEntriesCh, 
                                                                      becomeLeaderCh >>
                                                 ELSE /\ IF ((m[self]).mtype) = (RequestVoteResponse)
                                                            THEN /\ IF ((m[self]).mterm) > ((currentTerm)[self])
                                                                       THEN /\ currentTerm' = [currentTerm EXCEPT ![self] = (m[self]).mterm]
                                                                            /\ state' = [state EXCEPT ![self] = Follower]
                                                                            /\ votedFor' = [votedFor EXCEPT ![self] = Nil]
                                                                            /\ leader' = [leader EXCEPT ![self] = Nil]
                                                                            /\ IF ((m[self]).mterm) < ((currentTerm')[self])
                                                                                  THEN /\ TRUE
                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                       /\ UNCHANGED << votesResponded, 
                                                                                                       votesGranted, 
                                                                                                       becomeLeaderCh >>
                                                                                  ELSE /\ LET i == self IN
                                                                                            LET j == (m[self]).msource IN
                                                                                              /\ Assert(((m[self]).mterm) = ((currentTerm')[i]), 
                                                                                                        "Failure of assertion at line 1164, column 23.")
                                                                                              /\ votesResponded' = [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})]
                                                                                              /\ IF (m[self]).mvoteGranted
                                                                                                    THEN /\ votesGranted' = [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})]
                                                                                                         /\ IF (((state')[i]) = (Candidate)) /\ (isQuorum((votesGranted')[i]))
                                                                                                               THEN /\ LET value17 == TRUE IN
                                                                                                                         /\ becomeLeaderCh' = [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value17)]
                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                               ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                    /\ UNCHANGED becomeLeaderCh
                                                                                                    ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                         /\ UNCHANGED << votesGranted, 
                                                                                                                         becomeLeaderCh >>
                                                                       ELSE /\ IF ((m[self]).mterm) < ((currentTerm)[self])
                                                                                  THEN /\ TRUE
                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                       /\ UNCHANGED << votesResponded, 
                                                                                                       votesGranted, 
                                                                                                       becomeLeaderCh >>
                                                                                  ELSE /\ LET i == self IN
                                                                                            LET j == (m[self]).msource IN
                                                                                              /\ Assert(((m[self]).mterm) = ((currentTerm)[i]), 
                                                                                                        "Failure of assertion at line 1190, column 23.")
                                                                                              /\ votesResponded' = [votesResponded EXCEPT ![i] = ((votesResponded)[i]) \union ({j})]
                                                                                              /\ IF (m[self]).mvoteGranted
                                                                                                    THEN /\ votesGranted' = [votesGranted EXCEPT ![i] = ((votesGranted)[i]) \union ({j})]
                                                                                                         /\ IF (((state)[i]) = (Candidate)) /\ (isQuorum((votesGranted')[i]))
                                                                                                               THEN /\ LET value18 == TRUE IN
                                                                                                                         /\ becomeLeaderCh' = [becomeLeaderCh EXCEPT ![i] = Append((becomeLeaderCh)[i], value18)]
                                                                                                                         /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                               ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                    /\ UNCHANGED becomeLeaderCh
                                                                                                    ELSE /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                         /\ UNCHANGED << votesGranted, 
                                                                                                                         becomeLeaderCh >>
                                                                            /\ UNCHANGED << state, 
                                                                                            currentTerm, 
                                                                                            votedFor, 
                                                                                            leader >>
                                                                 /\ UNCHANGED << network, 
                                                                                 commitIndex, 
                                                                                 nextIndex, 
                                                                                 matchIndex, 
                                                                                 log, 
                                                                                 plog, 
                                                                                 sm, 
                                                                                 smDomain, 
                                                                                 leaderTimeout, 
                                                                                 appendEntriesCh >>
                                                            ELSE /\ IF ((m[self]).mtype) = (AppendEntriesRequest)
                                                                       THEN /\ IF ((m[self]).mterm) > ((currentTerm)[self])
                                                                                  THEN /\ currentTerm' = [currentTerm EXCEPT ![self] = (m[self]).mterm]
                                                                                       /\ LET state3 == [state EXCEPT ![self] = Follower] IN
                                                                                            /\ votedFor' = [votedFor EXCEPT ![self] = Nil]
                                                                                            /\ LET leader3 == [leader EXCEPT ![self] = Nil] IN
                                                                                                 LET i == self IN
                                                                                                   LET j == (m[self]).msource IN
                                                                                                     LET logOK == (((m[self]).mprevLogIndex) = (0)) \/ (((((m[self]).mprevLogIndex) > (0)) /\ (((m[self]).mprevLogIndex) <= (Len((log)[i])))) /\ (((m[self]).mprevLogTerm) = ((((log)[i])[(m[self]).mprevLogIndex]).term))) IN
                                                                                                       /\ Assert(((m[self]).mterm) <= ((currentTerm')[i]), 
                                                                                                                 "Failure of assertion at line 1220, column 25.")
                                                                                                       /\ IF ((m[self]).mterm) = ((currentTerm')[i])
                                                                                                             THEN /\ leader' = [leader3 EXCEPT ![i] = (m[self]).msource]
                                                                                                                  /\ leaderTimeout' = LeaderTimeoutReset
                                                                                                                  /\ IF (((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Candidate))
                                                                                                                        THEN /\ state' = [state3 EXCEPT ![i] = Follower]
                                                                                                                             /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                   THEN /\ \/ /\ LET value20 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value20), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                           \/ /\ LET yielded_fd00 == (fd)[j] IN
                                                                                                                                                   /\ yielded_fd00
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                              /\ UNCHANGED network
                                                                                                                                        /\ UNCHANGED << commitIndex, 
                                                                                                                                                        log, 
                                                                                                                                                        plog, 
                                                                                                                                                        sm, 
                                                                                                                                                        smDomain >>
                                                                                                                                   ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                  "Failure of assertion at line 1241, column 31.")
                                                                                                                                        /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                             LET value30 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                               IF ((value30).cmd) = (LogConcat)
                                                                                                                                                  THEN /\ LET plog24 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value30).entries)] IN
                                                                                                                                                            LET log24 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                              LET value40 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                IF ((value40).cmd) = (LogConcat)
                                                                                                                                                                   THEN /\ plog' = [plog24 EXCEPT ![i] = ((plog24)[i]) \o ((value40).entries)]
                                                                                                                                                                        /\ log' = [log24 EXCEPT ![i] = ((log24)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                  "Failure of assertion at line 1255, column 39.")
                                                                                                                                                                        /\ LET result24 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                             /\ sm' = [sm EXCEPT ![i] = (result24)[1]]
                                                                                                                                                                             /\ smDomain' = [smDomain EXCEPT ![i] = (result24)[2]]
                                                                                                                                                                             /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                             /\ \/ /\ LET value50 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value50), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                \/ /\ LET yielded_fd11 == (fd)[j] IN
                                                                                                                                                                                        /\ yielded_fd11
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                                   ELSE /\ IF ((value40).cmd) = (LogPop)
                                                                                                                                                                              THEN /\ plog' = [plog24 EXCEPT ![i] = SubSeq((plog24)[i], 1, (Len((plog24)[i])) - ((value40).cnt))]
                                                                                                                                                                                   /\ log' = [log24 EXCEPT ![i] = ((log24)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1278, column 41.")
                                                                                                                                                                                   /\ LET result25 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result25)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result25)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value51 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value51), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd12 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd12
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ log' = [log24 EXCEPT ![i] = ((log24)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1299, column 41.")
                                                                                                                                                                                   /\ LET result26 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result26)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result26)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value52 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value52), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ plog' = plog24
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd13 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd13
                                                                                                                                                                                                   /\ plog' = plog24
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                  ELSE /\ IF ((value30).cmd) = (LogPop)
                                                                                                                                                             THEN /\ LET plog25 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value30).cnt))] IN
                                                                                                                                                                       LET log25 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                         LET value41 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                           IF ((value41).cmd) = (LogConcat)
                                                                                                                                                                              THEN /\ plog' = [plog25 EXCEPT ![i] = ((plog25)[i]) \o ((value41).entries)]
                                                                                                                                                                                   /\ log' = [log25 EXCEPT ![i] = ((log25)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1333, column 41.")
                                                                                                                                                                                   /\ LET result27 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result27)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result27)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value53 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value53), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd14 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd14
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ IF ((value41).cmd) = (LogPop)
                                                                                                                                                                                         THEN /\ plog' = [plog25 EXCEPT ![i] = SubSeq((plog25)[i], 1, (Len((plog25)[i])) - ((value41).cnt))]
                                                                                                                                                                                              /\ log' = [log25 EXCEPT ![i] = ((log25)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 1356, column 43.")
                                                                                                                                                                                              /\ LET result28 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result28)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result28)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value54 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value54), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd15 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd15
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                                                         ELSE /\ log' = [log25 EXCEPT ![i] = ((log25)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 1377, column 43.")
                                                                                                                                                                                              /\ LET result29 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result29)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result29)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value55 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value55), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ plog' = plog25
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd16 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd16
                                                                                                                                                                                                              /\ plog' = plog25
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                             ELSE /\ LET log26 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                       LET value42 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                         IF ((value42).cmd) = (LogConcat)
                                                                                                                                                                            THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value42).entries)]
                                                                                                                                                                                 /\ log' = [log26 EXCEPT ![i] = ((log26)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                 /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                           "Failure of assertion at line 1409, column 41.")
                                                                                                                                                                                 /\ LET result30 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                      /\ sm' = [sm EXCEPT ![i] = (result30)[1]]
                                                                                                                                                                                      /\ smDomain' = [smDomain EXCEPT ![i] = (result30)[2]]
                                                                                                                                                                                      /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                      /\ \/ /\ LET value56 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                 /\ ((network)[j]).enabled
                                                                                                                                                                                                 /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                 /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value56), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                         \/ /\ LET yielded_fd17 == (fd)[j] IN
                                                                                                                                                                                                 /\ yielded_fd17
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                            /\ UNCHANGED network
                                                                                                                                                                            ELSE /\ IF ((value42).cmd) = (LogPop)
                                                                                                                                                                                       THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value42).cnt))]
                                                                                                                                                                                            /\ log' = [log26 EXCEPT ![i] = ((log26)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 1432, column 43.")
                                                                                                                                                                                            /\ LET result31 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result31)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result31)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value57 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value57), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd18 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd18
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                       ELSE /\ log' = [log26 EXCEPT ![i] = ((log26)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 1453, column 43.")
                                                                                                                                                                                            /\ LET result32 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result32)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result32)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value58 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value58), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd19 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd19
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                            /\ plog' = plog
                                                                                                                        ELSE /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                   THEN /\ \/ /\ LET value21 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value21), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                   /\ state' = state3
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                           \/ /\ LET yielded_fd01 == (fd)[j] IN
                                                                                                                                                   /\ yielded_fd01
                                                                                                                                                   /\ state' = state3
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                              /\ UNCHANGED network
                                                                                                                                        /\ UNCHANGED << commitIndex, 
                                                                                                                                                        log, 
                                                                                                                                                        plog, 
                                                                                                                                                        sm, 
                                                                                                                                                        smDomain >>
                                                                                                                                   ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                  "Failure of assertion at line 1497, column 31.")
                                                                                                                                        /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                             LET value31 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                               IF ((value31).cmd) = (LogConcat)
                                                                                                                                                  THEN /\ LET plog26 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value31).entries)] IN
                                                                                                                                                            LET log27 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                              LET value43 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                IF ((value43).cmd) = (LogConcat)
                                                                                                                                                                   THEN /\ plog' = [plog26 EXCEPT ![i] = ((plog26)[i]) \o ((value43).entries)]
                                                                                                                                                                        /\ log' = [log27 EXCEPT ![i] = ((log27)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                  "Failure of assertion at line 1511, column 39.")
                                                                                                                                                                        /\ LET result33 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                             /\ sm' = [sm EXCEPT ![i] = (result33)[1]]
                                                                                                                                                                             /\ smDomain' = [smDomain EXCEPT ![i] = (result33)[2]]
                                                                                                                                                                             /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                             /\ \/ /\ LET value59 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value59), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                        /\ state' = state3
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                \/ /\ LET yielded_fd110 == (fd)[j] IN
                                                                                                                                                                                        /\ yielded_fd110
                                                                                                                                                                                        /\ state' = state3
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                                   ELSE /\ IF ((value43).cmd) = (LogPop)
                                                                                                                                                                              THEN /\ plog' = [plog26 EXCEPT ![i] = SubSeq((plog26)[i], 1, (Len((plog26)[i])) - ((value43).cnt))]
                                                                                                                                                                                   /\ log' = [log27 EXCEPT ![i] = ((log27)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1536, column 41.")
                                                                                                                                                                                   /\ LET result34 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result34)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result34)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value510 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value510), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd111 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd111
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ log' = [log27 EXCEPT ![i] = ((log27)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1559, column 41.")
                                                                                                                                                                                   /\ LET result35 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result35)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result35)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value511 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value511), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ plog' = plog26
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd112 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd112
                                                                                                                                                                                                   /\ plog' = plog26
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                  ELSE /\ IF ((value31).cmd) = (LogPop)
                                                                                                                                                             THEN /\ LET plog27 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value31).cnt))] IN
                                                                                                                                                                       LET log28 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                         LET value44 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                           IF ((value44).cmd) = (LogConcat)
                                                                                                                                                                              THEN /\ plog' = [plog27 EXCEPT ![i] = ((plog27)[i]) \o ((value44).entries)]
                                                                                                                                                                                   /\ log' = [log28 EXCEPT ![i] = ((log28)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1595, column 41.")
                                                                                                                                                                                   /\ LET result36 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result36)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result36)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value512 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value512), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd113 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd113
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ IF ((value44).cmd) = (LogPop)
                                                                                                                                                                                         THEN /\ plog' = [plog27 EXCEPT ![i] = SubSeq((plog27)[i], 1, (Len((plog27)[i])) - ((value44).cnt))]
                                                                                                                                                                                              /\ log' = [log28 EXCEPT ![i] = ((log28)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 1620, column 43.")
                                                                                                                                                                                              /\ LET result37 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result37)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result37)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value513 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value513), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd114 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd114
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                                                         ELSE /\ log' = [log28 EXCEPT ![i] = ((log28)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 1643, column 43.")
                                                                                                                                                                                              /\ LET result38 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result38)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result38)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value514 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value514), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ plog' = plog27
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd115 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd115
                                                                                                                                                                                                              /\ plog' = plog27
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                             ELSE /\ LET log29 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                       LET value45 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                         IF ((value45).cmd) = (LogConcat)
                                                                                                                                                                            THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value45).entries)]
                                                                                                                                                                                 /\ log' = [log29 EXCEPT ![i] = ((log29)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                 /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                           "Failure of assertion at line 1677, column 41.")
                                                                                                                                                                                 /\ LET result39 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                      /\ sm' = [sm EXCEPT ![i] = (result39)[1]]
                                                                                                                                                                                      /\ smDomain' = [smDomain EXCEPT ![i] = (result39)[2]]
                                                                                                                                                                                      /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                      /\ \/ /\ LET value515 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                 /\ ((network)[j]).enabled
                                                                                                                                                                                                 /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                 /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value515), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                 /\ state' = state3
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                         \/ /\ LET yielded_fd116 == (fd)[j] IN
                                                                                                                                                                                                 /\ yielded_fd116
                                                                                                                                                                                                 /\ state' = state3
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                            /\ UNCHANGED network
                                                                                                                                                                            ELSE /\ IF ((value45).cmd) = (LogPop)
                                                                                                                                                                                       THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value45).cnt))]
                                                                                                                                                                                            /\ log' = [log29 EXCEPT ![i] = ((log29)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 1702, column 43.")
                                                                                                                                                                                            /\ LET result40 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result40)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result40)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value516 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value516), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd117 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd117
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                       ELSE /\ log' = [log29 EXCEPT ![i] = ((log29)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 1725, column 43.")
                                                                                                                                                                                            /\ LET result41 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result41)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result41)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value517 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value517), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd118 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd118
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                            /\ plog' = plog
                                                                                                             ELSE /\ IF (((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Candidate))
                                                                                                                        THEN /\ state' = [state3 EXCEPT ![i] = Follower]
                                                                                                                             /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                   THEN /\ \/ /\ LET value22 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value22), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                           \/ /\ LET yielded_fd02 == (fd)[j] IN
                                                                                                                                                   /\ yielded_fd02
                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                              /\ UNCHANGED network
                                                                                                                                        /\ UNCHANGED << commitIndex, 
                                                                                                                                                        log, 
                                                                                                                                                        plog, 
                                                                                                                                                        sm, 
                                                                                                                                                        smDomain >>
                                                                                                                                   ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                  "Failure of assertion at line 1774, column 31.")
                                                                                                                                        /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                             LET value32 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                               IF ((value32).cmd) = (LogConcat)
                                                                                                                                                  THEN /\ LET plog28 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value32).entries)] IN
                                                                                                                                                            LET log30 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                              LET value46 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                IF ((value46).cmd) = (LogConcat)
                                                                                                                                                                   THEN /\ plog' = [plog28 EXCEPT ![i] = ((plog28)[i]) \o ((value46).entries)]
                                                                                                                                                                        /\ log' = [log30 EXCEPT ![i] = ((log30)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                  "Failure of assertion at line 1788, column 39.")
                                                                                                                                                                        /\ LET result42 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                             /\ sm' = [sm EXCEPT ![i] = (result42)[1]]
                                                                                                                                                                             /\ smDomain' = [smDomain EXCEPT ![i] = (result42)[2]]
                                                                                                                                                                             /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                             /\ \/ /\ LET value518 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value518), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                        /\ leader' = leader3
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                \/ /\ LET yielded_fd119 == (fd)[j] IN
                                                                                                                                                                                        /\ yielded_fd119
                                                                                                                                                                                        /\ leader' = leader3
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                                   ELSE /\ IF ((value46).cmd) = (LogPop)
                                                                                                                                                                              THEN /\ plog' = [plog28 EXCEPT ![i] = SubSeq((plog28)[i], 1, (Len((plog28)[i])) - ((value46).cnt))]
                                                                                                                                                                                   /\ log' = [log30 EXCEPT ![i] = ((log30)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1813, column 41.")
                                                                                                                                                                                   /\ LET result43 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result43)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result43)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value519 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value519), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd120 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd120
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ log' = [log30 EXCEPT ![i] = ((log30)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1836, column 41.")
                                                                                                                                                                                   /\ LET result44 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result44)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result44)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value520 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value520), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ plog' = plog28
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd121 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd121
                                                                                                                                                                                                   /\ plog' = plog28
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                  ELSE /\ IF ((value32).cmd) = (LogPop)
                                                                                                                                                             THEN /\ LET plog29 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value32).cnt))] IN
                                                                                                                                                                       LET log31 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                         LET value47 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                           IF ((value47).cmd) = (LogConcat)
                                                                                                                                                                              THEN /\ plog' = [plog29 EXCEPT ![i] = ((plog29)[i]) \o ((value47).entries)]
                                                                                                                                                                                   /\ log' = [log31 EXCEPT ![i] = ((log31)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 1872, column 41.")
                                                                                                                                                                                   /\ LET result45 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result45)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result45)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value521 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value521), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd122 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd122
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ IF ((value47).cmd) = (LogPop)
                                                                                                                                                                                         THEN /\ plog' = [plog29 EXCEPT ![i] = SubSeq((plog29)[i], 1, (Len((plog29)[i])) - ((value47).cnt))]
                                                                                                                                                                                              /\ log' = [log31 EXCEPT ![i] = ((log31)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 1897, column 43.")
                                                                                                                                                                                              /\ LET result46 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result46)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result46)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value522 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value522), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd123 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd123
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                                                         ELSE /\ log' = [log31 EXCEPT ![i] = ((log31)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 1920, column 43.")
                                                                                                                                                                                              /\ LET result47 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result47)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result47)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value523 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value523), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ plog' = plog29
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd124 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd124
                                                                                                                                                                                                              /\ plog' = plog29
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                             ELSE /\ LET log32 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                       LET value48 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                         IF ((value48).cmd) = (LogConcat)
                                                                                                                                                                            THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value48).entries)]
                                                                                                                                                                                 /\ log' = [log32 EXCEPT ![i] = ((log32)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                 /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                           "Failure of assertion at line 1954, column 41.")
                                                                                                                                                                                 /\ LET result48 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                      /\ sm' = [sm EXCEPT ![i] = (result48)[1]]
                                                                                                                                                                                      /\ smDomain' = [smDomain EXCEPT ![i] = (result48)[2]]
                                                                                                                                                                                      /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                      /\ \/ /\ LET value524 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                 /\ ((network)[j]).enabled
                                                                                                                                                                                                 /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                 /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value524), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                 /\ leader' = leader3
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                         \/ /\ LET yielded_fd125 == (fd)[j] IN
                                                                                                                                                                                                 /\ yielded_fd125
                                                                                                                                                                                                 /\ leader' = leader3
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                            /\ UNCHANGED network
                                                                                                                                                                            ELSE /\ IF ((value48).cmd) = (LogPop)
                                                                                                                                                                                       THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value48).cnt))]
                                                                                                                                                                                            /\ log' = [log32 EXCEPT ![i] = ((log32)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 1979, column 43.")
                                                                                                                                                                                            /\ LET result49 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result49)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result49)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value525 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value525), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd126 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd126
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                       ELSE /\ log' = [log32 EXCEPT ![i] = ((log32)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 2002, column 43.")
                                                                                                                                                                                            /\ LET result50 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result50)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result50)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value526 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value526), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd127 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd127
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                            /\ plog' = plog
                                                                                                                        ELSE /\ IF (((m[self]).mterm) < ((currentTerm')[i])) \/ (((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                                   THEN /\ \/ /\ LET value23 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value23), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                   /\ state' = state3
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                           \/ /\ LET yielded_fd03 == (fd)[j] IN
                                                                                                                                                   /\ yielded_fd03
                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                   /\ state' = state3
                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                              /\ UNCHANGED network
                                                                                                                                        /\ UNCHANGED << commitIndex, 
                                                                                                                                                        log, 
                                                                                                                                                        plog, 
                                                                                                                                                        sm, 
                                                                                                                                                        smDomain >>
                                                                                                                                   ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm')[i])) /\ (((state3)[i]) = (Follower))) /\ (logOK), 
                                                                                                                                                  "Failure of assertion at line 2050, column 31.")
                                                                                                                                        /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                             LET value33 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                               IF ((value33).cmd) = (LogConcat)
                                                                                                                                                  THEN /\ LET plog30 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value33).entries)] IN
                                                                                                                                                            LET log33 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                              LET value49 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                IF ((value49).cmd) = (LogConcat)
                                                                                                                                                                   THEN /\ plog' = [plog30 EXCEPT ![i] = ((plog30)[i]) \o ((value49).entries)]
                                                                                                                                                                        /\ log' = [log33 EXCEPT ![i] = ((log33)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                        /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                  "Failure of assertion at line 2064, column 39.")
                                                                                                                                                                        /\ LET result51 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                             /\ sm' = [sm EXCEPT ![i] = (result51)[1]]
                                                                                                                                                                             /\ smDomain' = [smDomain EXCEPT ![i] = (result51)[2]]
                                                                                                                                                                             /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                             /\ \/ /\ LET value527 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                        /\ ((network)[j]).enabled
                                                                                                                                                                                        /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                        /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value527), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                        /\ leader' = leader3
                                                                                                                                                                                        /\ state' = state3
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                \/ /\ LET yielded_fd128 == (fd)[j] IN
                                                                                                                                                                                        /\ yielded_fd128
                                                                                                                                                                                        /\ leader' = leader3
                                                                                                                                                                                        /\ state' = state3
                                                                                                                                                                                        /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                   /\ UNCHANGED network
                                                                                                                                                                   ELSE /\ IF ((value49).cmd) = (LogPop)
                                                                                                                                                                              THEN /\ plog' = [plog30 EXCEPT ![i] = SubSeq((plog30)[i], 1, (Len((plog30)[i])) - ((value49).cnt))]
                                                                                                                                                                                   /\ log' = [log33 EXCEPT ![i] = ((log33)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 2091, column 41.")
                                                                                                                                                                                   /\ LET result52 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result52)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result52)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value528 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value528), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd129 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd129
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ log' = [log33 EXCEPT ![i] = ((log33)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 2116, column 41.")
                                                                                                                                                                                   /\ LET result53 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result53)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result53)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value529 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value529), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ plog' = plog30
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd130 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd130
                                                                                                                                                                                                   /\ plog' = plog30
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                  ELSE /\ IF ((value33).cmd) = (LogPop)
                                                                                                                                                             THEN /\ LET plog31 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value33).cnt))] IN
                                                                                                                                                                       LET log34 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                         LET value410 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                           IF ((value410).cmd) = (LogConcat)
                                                                                                                                                                              THEN /\ plog' = [plog31 EXCEPT ![i] = ((plog31)[i]) \o ((value410).entries)]
                                                                                                                                                                                   /\ log' = [log34 EXCEPT ![i] = ((log34)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                   /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                             "Failure of assertion at line 2154, column 41.")
                                                                                                                                                                                   /\ LET result54 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                        /\ sm' = [sm EXCEPT ![i] = (result54)[1]]
                                                                                                                                                                                        /\ smDomain' = [smDomain EXCEPT ![i] = (result54)[2]]
                                                                                                                                                                                        /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                        /\ \/ /\ LET value530 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                   /\ ((network)[j]).enabled
                                                                                                                                                                                                   /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                   /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value530), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                           \/ /\ LET yielded_fd131 == (fd)[j] IN
                                                                                                                                                                                                   /\ yielded_fd131
                                                                                                                                                                                                   /\ leader' = leader3
                                                                                                                                                                                                   /\ state' = state3
                                                                                                                                                                                                   /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                              /\ UNCHANGED network
                                                                                                                                                                              ELSE /\ IF ((value410).cmd) = (LogPop)
                                                                                                                                                                                         THEN /\ plog' = [plog31 EXCEPT ![i] = SubSeq((plog31)[i], 1, (Len((plog31)[i])) - ((value410).cnt))]
                                                                                                                                                                                              /\ log' = [log34 EXCEPT ![i] = ((log34)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 2181, column 43.")
                                                                                                                                                                                              /\ LET result55 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result55)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result55)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value531 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value531), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd132 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd132
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                                                         ELSE /\ log' = [log34 EXCEPT ![i] = ((log34)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                              /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                        "Failure of assertion at line 2206, column 43.")
                                                                                                                                                                                              /\ LET result56 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                   /\ sm' = [sm EXCEPT ![i] = (result56)[1]]
                                                                                                                                                                                                   /\ smDomain' = [smDomain EXCEPT ![i] = (result56)[2]]
                                                                                                                                                                                                   /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                   /\ \/ /\ LET value532 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                              /\ ((network)[j]).enabled
                                                                                                                                                                                                              /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                              /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value532), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                              /\ plog' = plog31
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                      \/ /\ LET yielded_fd133 == (fd)[j] IN
                                                                                                                                                                                                              /\ yielded_fd133
                                                                                                                                                                                                              /\ plog' = plog31
                                                                                                                                                                                                              /\ leader' = leader3
                                                                                                                                                                                                              /\ state' = state3
                                                                                                                                                                                                              /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                         /\ UNCHANGED network
                                                                                                                                                             ELSE /\ LET log35 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                       LET value411 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                         IF ((value411).cmd) = (LogConcat)
                                                                                                                                                                            THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value411).entries)]
                                                                                                                                                                                 /\ log' = [log35 EXCEPT ![i] = ((log35)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                 /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                           "Failure of assertion at line 2242, column 41.")
                                                                                                                                                                                 /\ LET result57 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                      /\ sm' = [sm EXCEPT ![i] = (result57)[1]]
                                                                                                                                                                                      /\ smDomain' = [smDomain EXCEPT ![i] = (result57)[2]]
                                                                                                                                                                                      /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                      /\ \/ /\ LET value533 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                 /\ ((network)[j]).enabled
                                                                                                                                                                                                 /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                 /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value533), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                 /\ leader' = leader3
                                                                                                                                                                                                 /\ state' = state3
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                         \/ /\ LET yielded_fd134 == (fd)[j] IN
                                                                                                                                                                                                 /\ yielded_fd134
                                                                                                                                                                                                 /\ leader' = leader3
                                                                                                                                                                                                 /\ state' = state3
                                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                            /\ UNCHANGED network
                                                                                                                                                                            ELSE /\ IF ((value411).cmd) = (LogPop)
                                                                                                                                                                                       THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value411).cnt))]
                                                                                                                                                                                            /\ log' = [log35 EXCEPT ![i] = ((log35)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 2269, column 43.")
                                                                                                                                                                                            /\ LET result58 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result58)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result58)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value534 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value534), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd135 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd135
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                       ELSE /\ log' = [log35 EXCEPT ![i] = ((log35)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                      "Failure of assertion at line 2294, column 43.")
                                                                                                                                                                                            /\ LET result59 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result59)[1]]
                                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result59)[2]]
                                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                                 /\ \/ /\ LET value535 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm')[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value535), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                    \/ /\ LET yielded_fd136 == (fd)[j] IN
                                                                                                                                                                                                            /\ yielded_fd136
                                                                                                                                                                                                            /\ leader' = leader3
                                                                                                                                                                                                            /\ state' = state3
                                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                                            /\ plog' = plog
                                                                                                                  /\ UNCHANGED leaderTimeout
                                                                                  ELSE /\ LET i == self IN
                                                                                            LET j == (m[self]).msource IN
                                                                                              LET logOK == (((m[self]).mprevLogIndex) = (0)) \/ (((((m[self]).mprevLogIndex) > (0)) /\ (((m[self]).mprevLogIndex) <= (Len((log)[i])))) /\ (((m[self]).mprevLogTerm) = ((((log)[i])[(m[self]).mprevLogIndex]).term))) IN
                                                                                                /\ Assert(((m[self]).mterm) <= ((currentTerm)[i]), 
                                                                                                          "Failure of assertion at line 2334, column 23.")
                                                                                                /\ IF ((m[self]).mterm) = ((currentTerm)[i])
                                                                                                      THEN /\ leader' = [leader EXCEPT ![i] = (m[self]).msource]
                                                                                                           /\ leaderTimeout' = LeaderTimeoutReset
                                                                                                           /\ IF (((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))
                                                                                                                 THEN /\ state' = [state EXCEPT ![i] = Follower]
                                                                                                                      /\ IF (((m[self]).mterm) < ((currentTerm)[i])) \/ (((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                            THEN /\ \/ /\ LET value24 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value24), enabled |-> ((network)[j]).enabled]]
                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                    \/ /\ LET yielded_fd04 == (fd)[j] IN
                                                                                                                                            /\ yielded_fd04
                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                       /\ UNCHANGED network
                                                                                                                                 /\ UNCHANGED << commitIndex, 
                                                                                                                                                 log, 
                                                                                                                                                 plog, 
                                                                                                                                                 sm, 
                                                                                                                                                 smDomain >>
                                                                                                                            ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                           "Failure of assertion at line 2355, column 29.")
                                                                                                                                 /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                      LET value34 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                        IF ((value34).cmd) = (LogConcat)
                                                                                                                                           THEN /\ LET plog32 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value34).entries)] IN
                                                                                                                                                     LET log36 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                       LET value412 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                         IF ((value412).cmd) = (LogConcat)
                                                                                                                                                            THEN /\ plog' = [plog32 EXCEPT ![i] = ((plog32)[i]) \o ((value412).entries)]
                                                                                                                                                                 /\ log' = [log36 EXCEPT ![i] = ((log36)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                 /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                           "Failure of assertion at line 2369, column 37.")
                                                                                                                                                                 /\ LET result60 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                      /\ sm' = [sm EXCEPT ![i] = (result60)[1]]
                                                                                                                                                                      /\ smDomain' = [smDomain EXCEPT ![i] = (result60)[2]]
                                                                                                                                                                      /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                      /\ \/ /\ LET value536 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                 /\ ((network)[j]).enabled
                                                                                                                                                                                 /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                 /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value536), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                         \/ /\ LET yielded_fd137 == (fd)[j] IN
                                                                                                                                                                                 /\ yielded_fd137
                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                            /\ UNCHANGED network
                                                                                                                                                            ELSE /\ IF ((value412).cmd) = (LogPop)
                                                                                                                                                                       THEN /\ plog' = [plog32 EXCEPT ![i] = SubSeq((plog32)[i], 1, (Len((plog32)[i])) - ((value412).cnt))]
                                                                                                                                                                            /\ log' = [log36 EXCEPT ![i] = ((log36)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2392, column 39.")
                                                                                                                                                                            /\ LET result61 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result61)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result61)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value537 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value537), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd138 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd138
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                       ELSE /\ log' = [log36 EXCEPT ![i] = ((log36)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2413, column 39.")
                                                                                                                                                                            /\ LET result62 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result62)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result62)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value538 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value538), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ plog' = plog32
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd139 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd139
                                                                                                                                                                                            /\ plog' = plog32
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                           ELSE /\ IF ((value34).cmd) = (LogPop)
                                                                                                                                                      THEN /\ LET plog33 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value34).cnt))] IN
                                                                                                                                                                LET log37 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                  LET value413 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                    IF ((value413).cmd) = (LogConcat)
                                                                                                                                                                       THEN /\ plog' = [plog33 EXCEPT ![i] = ((plog33)[i]) \o ((value413).entries)]
                                                                                                                                                                            /\ log' = [log37 EXCEPT ![i] = ((log37)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2447, column 39.")
                                                                                                                                                                            /\ LET result63 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result63)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result63)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value539 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value539), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd140 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd140
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                       ELSE /\ IF ((value413).cmd) = (LogPop)
                                                                                                                                                                                  THEN /\ plog' = [plog33 EXCEPT ![i] = SubSeq((plog33)[i], 1, (Len((plog33)[i])) - ((value413).cnt))]
                                                                                                                                                                                       /\ log' = [log37 EXCEPT ![i] = ((log37)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                       /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                 "Failure of assertion at line 2470, column 41.")
                                                                                                                                                                                       /\ LET result64 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                            /\ sm' = [sm EXCEPT ![i] = (result64)[1]]
                                                                                                                                                                                            /\ smDomain' = [smDomain EXCEPT ![i] = (result64)[2]]
                                                                                                                                                                                            /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                            /\ \/ /\ LET value540 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                       /\ ((network)[j]).enabled
                                                                                                                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value540), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                               \/ /\ LET yielded_fd141 == (fd)[j] IN
                                                                                                                                                                                                       /\ yielded_fd141
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                  /\ UNCHANGED network
                                                                                                                                                                                  ELSE /\ log' = [log37 EXCEPT ![i] = ((log37)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                       /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                 "Failure of assertion at line 2491, column 41.")
                                                                                                                                                                                       /\ LET result65 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                            /\ sm' = [sm EXCEPT ![i] = (result65)[1]]
                                                                                                                                                                                            /\ smDomain' = [smDomain EXCEPT ![i] = (result65)[2]]
                                                                                                                                                                                            /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                            /\ \/ /\ LET value541 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                       /\ ((network)[j]).enabled
                                                                                                                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value541), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                       /\ plog' = plog33
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                               \/ /\ LET yielded_fd142 == (fd)[j] IN
                                                                                                                                                                                                       /\ yielded_fd142
                                                                                                                                                                                                       /\ plog' = plog33
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                  /\ UNCHANGED network
                                                                                                                                                      ELSE /\ LET log38 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                LET value414 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                  IF ((value414).cmd) = (LogConcat)
                                                                                                                                                                     THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value414).entries)]
                                                                                                                                                                          /\ log' = [log38 EXCEPT ![i] = ((log38)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                          /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                    "Failure of assertion at line 2523, column 39.")
                                                                                                                                                                          /\ LET result66 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                               /\ sm' = [sm EXCEPT ![i] = (result66)[1]]
                                                                                                                                                                               /\ smDomain' = [smDomain EXCEPT ![i] = (result66)[2]]
                                                                                                                                                                               /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                               /\ \/ /\ LET value542 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                          /\ ((network)[j]).enabled
                                                                                                                                                                                          /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                          /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value542), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                  \/ /\ LET yielded_fd143 == (fd)[j] IN
                                                                                                                                                                                          /\ yielded_fd143
                                                                                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                     /\ UNCHANGED network
                                                                                                                                                                     ELSE /\ IF ((value414).cmd) = (LogPop)
                                                                                                                                                                                THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value414).cnt))]
                                                                                                                                                                                     /\ log' = [log38 EXCEPT ![i] = ((log38)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                     /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                               "Failure of assertion at line 2546, column 41.")
                                                                                                                                                                                     /\ LET result67 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                          /\ sm' = [sm EXCEPT ![i] = (result67)[1]]
                                                                                                                                                                                          /\ smDomain' = [smDomain EXCEPT ![i] = (result67)[2]]
                                                                                                                                                                                          /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                          /\ \/ /\ LET value543 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                     /\ ((network)[j]).enabled
                                                                                                                                                                                                     /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                     /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value543), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                             \/ /\ LET yielded_fd144 == (fd)[j] IN
                                                                                                                                                                                                     /\ yielded_fd144
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                /\ UNCHANGED network
                                                                                                                                                                                ELSE /\ log' = [log38 EXCEPT ![i] = ((log38)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                     /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                               "Failure of assertion at line 2567, column 41.")
                                                                                                                                                                                     /\ LET result68 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                          /\ sm' = [sm EXCEPT ![i] = (result68)[1]]
                                                                                                                                                                                          /\ smDomain' = [smDomain EXCEPT ![i] = (result68)[2]]
                                                                                                                                                                                          /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                          /\ \/ /\ LET value544 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                     /\ ((network)[j]).enabled
                                                                                                                                                                                                     /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                     /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value544), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                             \/ /\ LET yielded_fd145 == (fd)[j] IN
                                                                                                                                                                                                     /\ yielded_fd145
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                /\ UNCHANGED network
                                                                                                                                                                                     /\ plog' = plog
                                                                                                                 ELSE /\ IF (((m[self]).mterm) < ((currentTerm)[i])) \/ (((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                            THEN /\ \/ /\ LET value25 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value25), enabled |-> ((network)[j]).enabled]]
                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                    \/ /\ LET yielded_fd05 == (fd)[j] IN
                                                                                                                                            /\ yielded_fd05
                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                       /\ UNCHANGED network
                                                                                                                                 /\ UNCHANGED << commitIndex, 
                                                                                                                                                 log, 
                                                                                                                                                 plog, 
                                                                                                                                                 sm, 
                                                                                                                                                 smDomain >>
                                                                                                                            ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Follower))) /\ (logOK), 
                                                                                                                                           "Failure of assertion at line 2609, column 29.")
                                                                                                                                 /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                      LET value35 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                        IF ((value35).cmd) = (LogConcat)
                                                                                                                                           THEN /\ LET plog34 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value35).entries)] IN
                                                                                                                                                     LET log39 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                       LET value415 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                         IF ((value415).cmd) = (LogConcat)
                                                                                                                                                            THEN /\ plog' = [plog34 EXCEPT ![i] = ((plog34)[i]) \o ((value415).entries)]
                                                                                                                                                                 /\ log' = [log39 EXCEPT ![i] = ((log39)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                 /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                           "Failure of assertion at line 2623, column 37.")
                                                                                                                                                                 /\ LET result69 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                      /\ sm' = [sm EXCEPT ![i] = (result69)[1]]
                                                                                                                                                                      /\ smDomain' = [smDomain EXCEPT ![i] = (result69)[2]]
                                                                                                                                                                      /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                      /\ \/ /\ LET value545 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                 /\ ((network)[j]).enabled
                                                                                                                                                                                 /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                 /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value545), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                         \/ /\ LET yielded_fd146 == (fd)[j] IN
                                                                                                                                                                                 /\ yielded_fd146
                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                            /\ UNCHANGED network
                                                                                                                                                            ELSE /\ IF ((value415).cmd) = (LogPop)
                                                                                                                                                                       THEN /\ plog' = [plog34 EXCEPT ![i] = SubSeq((plog34)[i], 1, (Len((plog34)[i])) - ((value415).cnt))]
                                                                                                                                                                            /\ log' = [log39 EXCEPT ![i] = ((log39)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2646, column 39.")
                                                                                                                                                                            /\ LET result70 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result70)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result70)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value546 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value546), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd147 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd147
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                       ELSE /\ log' = [log39 EXCEPT ![i] = ((log39)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2667, column 39.")
                                                                                                                                                                            /\ LET result71 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result71)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result71)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value547 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value547), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ plog' = plog34
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd148 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd148
                                                                                                                                                                                            /\ plog' = plog34
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                           ELSE /\ IF ((value35).cmd) = (LogPop)
                                                                                                                                                      THEN /\ LET plog35 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value35).cnt))] IN
                                                                                                                                                                LET log40 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                  LET value416 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                    IF ((value416).cmd) = (LogConcat)
                                                                                                                                                                       THEN /\ plog' = [plog35 EXCEPT ![i] = ((plog35)[i]) \o ((value416).entries)]
                                                                                                                                                                            /\ log' = [log40 EXCEPT ![i] = ((log40)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2701, column 39.")
                                                                                                                                                                            /\ LET result72 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result72)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result72)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value548 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value548), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd149 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd149
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                       ELSE /\ IF ((value416).cmd) = (LogPop)
                                                                                                                                                                                  THEN /\ plog' = [plog35 EXCEPT ![i] = SubSeq((plog35)[i], 1, (Len((plog35)[i])) - ((value416).cnt))]
                                                                                                                                                                                       /\ log' = [log40 EXCEPT ![i] = ((log40)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                       /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                 "Failure of assertion at line 2724, column 41.")
                                                                                                                                                                                       /\ LET result73 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                            /\ sm' = [sm EXCEPT ![i] = (result73)[1]]
                                                                                                                                                                                            /\ smDomain' = [smDomain EXCEPT ![i] = (result73)[2]]
                                                                                                                                                                                            /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                            /\ \/ /\ LET value549 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                       /\ ((network)[j]).enabled
                                                                                                                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value549), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                               \/ /\ LET yielded_fd150 == (fd)[j] IN
                                                                                                                                                                                                       /\ yielded_fd150
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                  /\ UNCHANGED network
                                                                                                                                                                                  ELSE /\ log' = [log40 EXCEPT ![i] = ((log40)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                       /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                 "Failure of assertion at line 2745, column 41.")
                                                                                                                                                                                       /\ LET result74 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                            /\ sm' = [sm EXCEPT ![i] = (result74)[1]]
                                                                                                                                                                                            /\ smDomain' = [smDomain EXCEPT ![i] = (result74)[2]]
                                                                                                                                                                                            /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                            /\ \/ /\ LET value550 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                       /\ ((network)[j]).enabled
                                                                                                                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value550), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                       /\ plog' = plog35
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                               \/ /\ LET yielded_fd151 == (fd)[j] IN
                                                                                                                                                                                                       /\ yielded_fd151
                                                                                                                                                                                                       /\ plog' = plog35
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                  /\ UNCHANGED network
                                                                                                                                                      ELSE /\ LET log41 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                LET value417 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                  IF ((value417).cmd) = (LogConcat)
                                                                                                                                                                     THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value417).entries)]
                                                                                                                                                                          /\ log' = [log41 EXCEPT ![i] = ((log41)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                          /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                    "Failure of assertion at line 2777, column 39.")
                                                                                                                                                                          /\ LET result75 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                               /\ sm' = [sm EXCEPT ![i] = (result75)[1]]
                                                                                                                                                                               /\ smDomain' = [smDomain EXCEPT ![i] = (result75)[2]]
                                                                                                                                                                               /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                               /\ \/ /\ LET value551 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                          /\ ((network)[j]).enabled
                                                                                                                                                                                          /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                          /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value551), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                  \/ /\ LET yielded_fd152 == (fd)[j] IN
                                                                                                                                                                                          /\ yielded_fd152
                                                                                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                     /\ UNCHANGED network
                                                                                                                                                                     ELSE /\ IF ((value417).cmd) = (LogPop)
                                                                                                                                                                                THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value417).cnt))]
                                                                                                                                                                                     /\ log' = [log41 EXCEPT ![i] = ((log41)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                     /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                               "Failure of assertion at line 2800, column 41.")
                                                                                                                                                                                     /\ LET result76 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                          /\ sm' = [sm EXCEPT ![i] = (result76)[1]]
                                                                                                                                                                                          /\ smDomain' = [smDomain EXCEPT ![i] = (result76)[2]]
                                                                                                                                                                                          /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                          /\ \/ /\ LET value552 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                     /\ ((network)[j]).enabled
                                                                                                                                                                                                     /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                     /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value552), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                             \/ /\ LET yielded_fd153 == (fd)[j] IN
                                                                                                                                                                                                     /\ yielded_fd153
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                /\ UNCHANGED network
                                                                                                                                                                                ELSE /\ log' = [log41 EXCEPT ![i] = ((log41)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                     /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                               "Failure of assertion at line 2821, column 41.")
                                                                                                                                                                                     /\ LET result77 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                          /\ sm' = [sm EXCEPT ![i] = (result77)[1]]
                                                                                                                                                                                          /\ smDomain' = [smDomain EXCEPT ![i] = (result77)[2]]
                                                                                                                                                                                          /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                          /\ \/ /\ LET value553 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                     /\ ((network)[j]).enabled
                                                                                                                                                                                                     /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                     /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value553), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                             \/ /\ LET yielded_fd154 == (fd)[j] IN
                                                                                                                                                                                                     /\ yielded_fd154
                                                                                                                                                                                                     /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                /\ UNCHANGED network
                                                                                                                                                                                     /\ plog' = plog
                                                                                                                      /\ state' = state
                                                                                                      ELSE /\ IF (((m[self]).mterm) = ((currentTerm)[i])) /\ (((state)[i]) = (Candidate))
                                                                                                                 THEN /\ state' = [state EXCEPT ![i] = Follower]
                                                                                                                      /\ IF (((m[self]).mterm) < ((currentTerm)[i])) \/ (((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (~ (logOK)))
                                                                                                                            THEN /\ \/ /\ LET value26 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> FALSE, mmatchIndex |-> 0, msource |-> i, mdest |-> j] IN
                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value26), enabled |-> ((network)[j]).enabled]]
                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                    \/ /\ LET yielded_fd06 == (fd)[j] IN
                                                                                                                                            /\ yielded_fd06
                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                       /\ UNCHANGED network
                                                                                                                                 /\ UNCHANGED << commitIndex, 
                                                                                                                                                 log, 
                                                                                                                                                 plog, 
                                                                                                                                                 sm, 
                                                                                                                                                 smDomain >>
                                                                                                                            ELSE /\ Assert(((((m[self]).mterm) = ((currentTerm)[i])) /\ (((state')[i]) = (Follower))) /\ (logOK), 
                                                                                                                                           "Failure of assertion at line 2866, column 29.")
                                                                                                                                 /\ LET index == ((m[self]).mprevLogIndex) + (1) IN
                                                                                                                                      LET value36 == [cmd |-> LogPop, cnt |-> (Len((log)[i])) - ((m[self]).mprevLogIndex)] IN
                                                                                                                                        IF ((value36).cmd) = (LogConcat)
                                                                                                                                           THEN /\ LET plog36 == [plog EXCEPT ![i] = ((plog)[i]) \o ((value36).entries)] IN
                                                                                                                                                     LET log42 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                       LET value418 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                         IF ((value418).cmd) = (LogConcat)
                                                                                                                                                            THEN /\ plog' = [plog36 EXCEPT ![i] = ((plog36)[i]) \o ((value418).entries)]
                                                                                                                                                                 /\ log' = [log42 EXCEPT ![i] = ((log42)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                 /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                           "Failure of assertion at line 2880, column 37.")
                                                                                                                                                                 /\ LET result78 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                      /\ sm' = [sm EXCEPT ![i] = (result78)[1]]
                                                                                                                                                                      /\ smDomain' = [smDomain EXCEPT ![i] = (result78)[2]]
                                                                                                                                                                      /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                      /\ \/ /\ LET value554 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                 /\ ((network)[j]).enabled
                                                                                                                                                                                 /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                 /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value554), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                         \/ /\ LET yielded_fd155 == (fd)[j] IN
                                                                                                                                                                                 /\ yielded_fd155
                                                                                                                                                                                 /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                            /\ UNCHANGED network
                                                                                                                                                            ELSE /\ IF ((value418).cmd) = (LogPop)
                                                                                                                                                                       THEN /\ plog' = [plog36 EXCEPT ![i] = SubSeq((plog36)[i], 1, (Len((plog36)[i])) - ((value418).cnt))]
                                                                                                                                                                            /\ log' = [log42 EXCEPT ![i] = ((log42)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2903, column 39.")
                                                                                                                                                                            /\ LET result79 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result79)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result79)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value555 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value555), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd156 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd156
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                       ELSE /\ log' = [log42 EXCEPT ![i] = ((log42)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2924, column 39.")
                                                                                                                                                                            /\ LET result80 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result80)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result80)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value556 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value556), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ plog' = plog36
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd157 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd157
                                                                                                                                                                                            /\ plog' = plog36
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                           ELSE /\ IF ((value36).cmd) = (LogPop)
                                                                                                                                                      THEN /\ LET plog37 == [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value36).cnt))] IN
                                                                                                                                                                LET log43 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                  LET value419 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                    IF ((value419).cmd) = (LogConcat)
                                                                                                                                                                       THEN /\ plog' = [plog37 EXCEPT ![i] = ((plog37)[i]) \o ((value419).entries)]
                                                                                                                                                                            /\ log' = [log43 EXCEPT ![i] = ((log43)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                            /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                      "Failure of assertion at line 2958, column 39.")
                                                                                                                                                                            /\ LET result81 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                 /\ sm' = [sm EXCEPT ![i] = (result81)[1]]
                                                                                                                                                                                 /\ smDomain' = [smDomain EXCEPT ![i] = (result81)[2]]
                                                                                                                                                                                 /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                 /\ \/ /\ LET value557 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                            /\ ((network)[j]).enabled
                                                                                                                                                                                            /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                            /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value557), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                    \/ /\ LET yielded_fd158 == (fd)[j] IN
                                                                                                                                                                                            /\ yielded_fd158
                                                                                                                                                                                            /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                       /\ UNCHANGED network
                                                                                                                                                                       ELSE /\ IF ((value419).cmd) = (LogPop)
                                                                                                                                                                                  THEN /\ plog' = [plog37 EXCEPT ![i] = SubSeq((plog37)[i], 1, (Len((plog37)[i])) - ((value419).cnt))]
                                                                                                                                                                                       /\ log' = [log43 EXCEPT ![i] = ((log43)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                       /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                 "Failure of assertion at line 2981, column 41.")
                                                                                                                                                                                       /\ LET result82 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                            /\ sm' = [sm EXCEPT ![i] = (result82)[1]]
                                                                                                                                                                                            /\ smDomain' = [smDomain EXCEPT ![i] = (result82)[2]]
                                                                                                                                                                                            /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                            /\ \/ /\ LET value558 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                       /\ ((network)[j]).enabled
                                                                                                                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value558), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                               \/ /\ LET yielded_fd159 == (fd)[j] IN
                                                                                                                                                                                                       /\ yielded_fd159
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                  /\ UNCHANGED network
                                                                                                                                                                                  ELSE /\ log' = [log43 EXCEPT ![i] = ((log43)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                       /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                                 "Failure of assertion at line 3002, column 41.")
                                                                                                                                                                                       /\ LET result83 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                                            /\ sm' = [sm EXCEPT ![i] = (result83)[1]]
                                                                                                                                                                                            /\ smDomain' = [smDomain EXCEPT ![i] = (result83)[2]]
                                                                                                                                                                                            /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                                            /\ \/ /\ LET value559 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                                       /\ ((network)[j]).enabled
                                                                                                                                                                                                       /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                                       /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value559), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                                       /\ plog' = plog37
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                               \/ /\ LET yielded_fd160 == (fd)[j] IN
                                                                                                                                                                                                       /\ yielded_fd160
                                                                                                                                                                                                       /\ plog' = plog37
                                                                                                                                                                                                       /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                                  /\ UNCHANGED network
                                                                                                                                                      ELSE /\ LET log44 == [log EXCEPT ![i] = SubSeq((log)[i], 1, (m[self]).mprevLogIndex)] IN
                                                                                                                                                                LET value420 == [cmd |-> LogConcat, entries |-> (m[self]).mentries] IN
                                                                                                                                                                  IF ((value420).cmd) = (LogConcat)
                                                                                                                                                                     THEN /\ plog' = [plog EXCEPT ![i] = ((plog)[i]) \o ((value420).entries)]
                                                                                                                                                                          /\ log' = [log44 EXCEPT ![i] = ((log44)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                          /\ Assert(((m[self]).mcommitIndex) <= (Len((log')[i])), 
                                                                                                                                                                                    "Failure of assertion at line 3034, column 39.")
                                                                                                                                                                          /\ LET result84 == ApplyLog((log')[i], ((commitIndex)[i]) + (1), (m[self]).mcommitIndex, (sm)[i], (smDomain)[i]) IN
                                                                                                                                                                               /\ sm' = [sm EXCEPT ![i] = (result84)[1]]
                                                                                                                                                                               /\ smDomain' = [smDomain EXCEPT ![i] = (result84)[2]]
                                                                                                                                                                               /\ commitIndex' = [commitIndex EXCEPT ![i] = Max({(commitIndex)[i], (m[self]).mcommitIndex})]
                                                                                                                                                                               /\ \/ /\ LET value560 == [mtype |-> AppendEntriesResponse, mterm |-> (currentTerm)[i], msuccess |-> TRUE, mmatchIndex |-> ((m[self]).mprevLogIndex) + (Len((m[self]).mentries)), msource |-> i, mdest |-> j] IN
                                                                                                                                                                                          /\ ((network)[j]).enabled
                                                                                                                                                                                          /\ (Len(((network)[j]).queue)) < (BufferSize)
                                                                                                                                                                                          /\ network' = [network EXCEPT ![j] = [queue |-> Append(((network)[j]).queue, value560), enabled |-> ((network)[j]).enabled]]
                                                                                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                  \/ /\ LET yielded_fd161 == (fd)[j] IN
                                                                                                                                                                                          /\ yielded_fd161
                                                                                                                                                                                          /\ pc' = [pc EXCEPT ![self] = "serverLoop"]
                                                                                                                                                                                     /\ UNCHANGED network
                                                                                                                                                                     ELSE /\ IF ((value420).cmd) = (LogPop)
                                                                                                                                                                                THEN /\ plog' = [plog EXCEPT ![i] = SubSeq((plog)[i], 1, (Len((plog)[i])) - ((value420).cnt))]
                                                                                                                                                                                     /\ log' = [log44 EXCEPT ![i] = ((log44)[i]) \o ((m[self]).mentries)]
                                                                                                                                                                                     /\ Assert(((m[self]).mcommitInd