# Data References

There are two kinds of identifiers that can be used to reference pieces of static data: id and GUID.

A data id is a 64-bit hash derived from the file path. Since it changes when a file is moved or renamed, the same data can have different ids in different versions of the game. The client encapsulates data ids in DataRef structs, and in some places id and data ref are used interchangeably.

GUID is also a 64-bit value, but unlike a data id, it never changes for a piece of data. Ids are generally used for client <-> server communication instead of full file paths, while GUIDs are used for storing persistent data, such as player mission progress. Resource prototypes do not have a GUID, and every time a "resource GUID" is mentioned in the code it actually refers to its name.

To refer to a piece of data by its file path it first needs to be converted to a data reference, which is done by various instances of the `DataRefManager` class.

[Calligraphy](./Calligraphy.md) data references are pre-generated and stored in `.directory` files, while [resource](./Resources.md) references are generated during `GameDatabase` initialization.

## Reference Generation

Curve, asset type, blueprint, and prototype ids are generated by hashing the file path with a combination of [Adler-32](https://en.wikipedia.org/wiki/Adler-32) and [CRC-32](https://en.wikipedia.org/wiki/Cyclic_redundancy_check) algorithms:

```csharp
string path;
path = path.ToLower();
ulong dataId = (Adler32(path) | (Crc32(path) << 32)) - 1;
```

Before hashing, Calligraphy and resource paths undergo different additional formatting.

For Calligraphy data (curves, asset types, blueprints, and Calligraphy prototypes):

1. Take the relative path (without `Calligraphy/`).

2. Replace `.` characters with `?`.

3. Replace `/` characters with `.`.

Example: `Powers/EnemyPowers/MobPowers/Cows/CowKingSpearLunge.prototype` should be formatted as `Powers.EnemyPowers.MobPowers.Cows.CowKingSpearLunge?prototype`. Similarly formatted Calligraphy paths are also used in the replacement directory that you can read more about [here](./Calligraphy.md).

**Important Note:** `.defaults` prototype files use their paired blueprint hashes as their ids. GUIDs for blueprints and their default are different.

For resource prototypes:

1. Take the full path.

2. Insert `&` as the first character.

Example: `Resource/Encounters/CH09Norway/OM_NordicRuins_CowBosses.encounter` should be formatted as `&Resource/Encounters/CH09Norway/OM_NordicRuins_CowBosses.encounter`.

[Pak file](./PakFile.md) entry hashes are generated the same way, but without any additional formatting.

GUID generation algorithms are currently unknown.
