# Pull requests

There are two fundamental components of the pull request process: one concrete
and technical, and one more process oriented. The concrete and technical
component involves the specific details of setting up your local environment
so that you can make the actual changes. This is where we will start.

* [Dependencies](#dependencies)
* [Setting up your local environment](#setting-up-your-local-environment)
  * [Step 1: Fork](#step-1-fork)
  * [Step 2: Branch](#step-2-branch)
* [The Process of Making Changes](#the-process-of-making-changes)
  * [Step 3: Code](#step-3-code)
  * [Step 4: Commit](#step-4-commit)
    * [Developer Certificate of Origin (DCO) configuration](#developer-certificate-of-origin-dco-configuration)
    * [Commit message guidelines](#commit-message-guidelines)
  * [Step 5: Rebase](#step-5-rebase)
  * [Step 6: Test](#step-6-test)
  * [Step 7: Push](#step-7-push)
  * [Step 8: Opening the pull request](#step-8-opening-the-pull-request)
  * [Step 9: Discuss and update](#step-9-discuss-and-update)
    * [Approval and request changes workflow](#approval-and-request-changes-workflow)
  * [Step 10: Landing](#step-10-landing)
* [Reviewing pull requests](#reviewing-pull-requests)
  * [Review a bit at a time](#review-a-bit-at-a-time)
  * [Be aware of the person behind the code](#be-aware-of-the-person-behind-the-code)
  * [Respect the minimum wait time for comments](#respect-the-minimum-wait-time-for-comments)
  * [Abandoned or stalled pull requests](#abandoned-or-stalled-pull-requests)
  * [Approving a change](#approving-a-change)
  * [Accept that there are different opinions about what belongs in ClusterDuck-Protocol](#accept-that-there-are-different-opinions-about-what-belongs-in-clusterduck-protocol)
  * [Performance is not everything](#performance-is-not-everything)
  * [Continuous integration testing](#continuous-integration-testing)
* [Notes](#notes)
  * [Commit squashing](#commit-squashing)
  * [Getting approvals for your pull request](#getting-approvals-for-your-pull-request)
  * [CI testing](#ci-testing)
  * [Waiting until the pull request gets landed](#waiting-until-the-pull-request-gets-landed)

## Dependencies

More details to come on ClusterDuck-Protocol dependencies. In case of doubt, open an issue in the
[issue tracker](https://github.com/Call-for-Code/ClusterDuck-Protocol/issues/) or engage on the
[ClusterDuck-Protocol Slack workspace][].

## Setting up your local environment

To get started, you will need to have `git` installed locally. Depending on
your operating system, there are also a number of other dependencies required.

Once you have `git` and are sure you have all of the necessary dependencies,
it's time to create a fork.

### Step 1: Fork

Fork the project [on GitHub](https://github.com/Call-for-Code/ClusterDuck-Protocol) and clone your fork
locally.

You will need to have [Git Large File Storage](https://help.github.com/en/github/managing-large-files/installing-git-large-file-storage) installed locally. Check out this [StackOverflow question if you run into a common error message](https://stackoverflow.com/questions/67395259/git-clone-git-lfs-filter-process-git-lfs-command-not-found).

```sh
git clone git@github.com:username/ClusterDuck-Protocol.git
cd ClusterDuck-Protocol
git remote add upstream https://github.com/Call-for-Code/ClusterDuck-Protocol.git
git fetch upstream
```

It is recommended to configure `git` so that it knows who you are:

```sh
git config user.name "J. Random User"
git config user.email "j.random.user@example.com"
```

You can use any name/email address you prefer here. We only use the
metadata generated by `git` using this configuration for properly attributing
your changes to you in the `AUTHORS` file and the changelog.

### Step 2: Branch

As a best practice to keep your development environment as organized as
possible, create local branches to work within. These should also be created
directly off of the `master` branch.

```sh
git checkout -b my-branch -t upstream/master
```

## The process of making changes

### Step 3: Code

ClusterDuck-Protocol does not yet have a style guide. If you're interested in creating one, you're on the right page 😃

#### Static analysis

[Cppcheck](http://cppcheck.sourceforge.net/) is used within the ClusterDuck's Travis CI to detect some of the more obvious C++ problems. If cppcheck finds a problem the build will fail, so you probably want to run cppcheck locally before committing and pushing changes. See [.travis.yml](https://github.com/Call-for-Code/ClusterDuck-Protocol/blob/master/.travis.yml) for the current `cppcheck` command being used by CI.

### Step 4: Commit

#### Developer Certificate of Origin (DCO) configuration

> All pull requests will require you to ensure the change is certified via the [Developer Certificate of Origin (DCO)](https://github.com/apps/dco/). The DCO is a lightweight way for contributors to validate that they wrote or otherwise have the right to submit the code they are contributing to the project. See more details on how to [configure your `git` environment for the DCO](https://github.com/Call-for-Code/ClusterDuck-Protocol/blob/master/DEVELOPER_CERTIFICATE_OF_ORIGIN.md).

It is a recommended best practice to keep your changes as logically grouped
as possible within individual commits. There is no limit to the number of
commits any single pull request may have, and many contributors find it easier
to review changes that are split across multiple commits.

```sh
git add my/changed/files
git commit
```

Note that multiple commits often get squashed when they are landed (see the
notes about [commit squashing](#commit-squashing)).

#### Commit message guidelines

A good commit message should describe what changed and why. One line commits are good, but if you want to go into more detail:

1. The first line should:
   * contain a short description of the change (preferably 50 characters or
     less, and no more than 72 characters)
   * be entirely in lowercase with the exception of proper nouns, acronyms, and
   the words that refer to code, like function/variable names
   * be prefixed with the name of the changed subsystem and start with an
   imperative verb. Check the output of `git log --oneline files/you/changed` to
   find out what subsystems your changes touch.

   Examples:
   * `net: add localAddress and localPort to Socket`
   * `src: fix typos in async_wrap.h`

2. Keep the second line blank.
3. Wrap all other lines at 72 columns (except for long URLs).

4. If your patch fixes an open issue, you can add a reference to it at the end
of the log. Use the `Fixes:` prefix and the full issue URL. For other references
use `Refs:`.

Sample complete commit message:

```txt
subsystem: explain the commit in one line

Body of commit message is a few lines of text, explaining things
in more detail, possibly giving some background about the issue
being fixed, etc.

The body of the commit message can be several paragraphs, and
please do proper word-wrap and keep columns shorter than about
72 characters or so. That way, `git log` will show things
nicely even when it is indented.

Fixes: https://github.com/Call-for-Code/ClusterDuck-Protocol/issues/1337
Refs: http://eslint.org/docs/rules/space-in-parens.html
```

If you are new to contributing to ClusterDuck-Protocol, please try to do your best at
conforming to these guidelines, but do not worry if you get something wrong.
One of the existing contributors will help get things situated and the
contributor landing the pull request will ensure that everything follows
the project guidelines.

### Step 5: Rebase

As a best practice, once you have committed your changes, it is a good idea
to use `git rebase` (not `git merge`) to synchronize your work with the main
repository.

```sh
git fetch upstream
git rebase upstream/master
```

This ensures that your working branch has the latest changes from
master.

### Step 6: Test

Instructions should also be provided so that the core team can validate the change.

### Step 7: Push

Once you are sure your commits are ready to go, with passing tests and linting,
begin the process of opening a pull request by pushing your working branch to
your fork on GitHub.

```sh
git push origin my-branch
```

### Step 8: Opening the pull request

From within GitHub, opening a new pull request will present you with a few
form fields.

Please try to do your best at filling out the details, but feel free to skip
parts if you're not sure what to put.

Once opened, pull requests are usually reviewed within a few days.

### Step 9: Discuss and update

You will probably get feedback or requests for changes to your pull request.
This is a big part of the submission process so don't be discouraged! Some
contributors may sign off on the pull request right away, others may have
more detailed comments or feedback. This is a necessary part of the process
in order to evaluate whether the changes are correct and necessary.

To make changes to an existing pull request, make the changes to your local
branch, add a new commit with those changes, and push those to your fork.
GitHub will automatically update the pull request.

```sh
git add my/changed/files
git commit
git push origin my-branch
```

It is also frequently necessary to synchronize your pull request with other
changes that have landed in `master` by using `git rebase`:

```sh
git fetch --all
git rebase origin/master
git push --force-with-lease origin my-branch
```

**Important:** The `git push --force-with-lease` command is one of the few ways
to delete history in `git`. Before you use it, make sure you understand the
risks. If in doubt, you can always ask for guidance in the pull request or on the
[ClusterDuck-Protocol Slack workspace].

If you happen to make a mistake in any of your commits, do not worry. You can
amend the last commit (for example if you want to change the commit log).

```sh
git add any/changed/files
git commit --amend
git push --force-with-lease origin my-branch
```

There are a number of more advanced mechanisms for managing commits using
`git rebase` that can be used, but are beyond the scope of this guide.

Feel free to post a comment in the pull request to ping reviewers if you are
awaiting an answer on something. If you encounter words or acronyms that
seem unfamiliar, refer to this
[glossary](https://sites.google.com/a/chromium.org/dev/glossary).

#### Approval and request changes workflow

All pull requests require "sign off" in order to land. Whenever a contributor
reviews a pull request they may find specific details that they would like to
see changed or fixed. These may be as simple as fixing a typo, or may involve
substantive changes to the code you have written. While such requests are
intended to be helpful, they may come across as abrupt or unhelpful, especially
requests to change things that do not include concrete suggestions on *how* to
change them.

Try not to be discouraged. If you feel that a particular review is unfair,
say so, or contact one of the other contributors in the project and seek their
input. Often such comments are the result of the reviewer having only taken a
short amount of time to review and are not ill-intended. Such issues can often
be resolved with a bit of patience. That said, reviewers should be expected to
be helpful in their feedback, and feedback that is simply vague, dismissive and
unhelpful is likely safe to ignore.

### Step 10: Landing

In order to land, a pull request needs to be reviewed and [approved][] by
at least one ClusterDuck-Protocol Collaborators. After that, as long as there are no
objections from other contributors, the pull request can be merged. If you find
your pull request waiting longer than you expect, see the
[notes about the waiting time](#waiting-until-the-pull-request-gets-landed).

When a collaborator lands your pull request, they will post
a comment to the pull request page mentioning the commit(s) it
landed as. GitHub often shows the pull request as `Closed` at this
point, but don't worry. If you look at the branch you raised your
pull request against (probably `master`), you should see a commit with
your name on it. Congratulations and thanks for your contribution!

## Reviewing pull requests

All ClusterDuck-Protocol contributors who choose to review and provide feedback on pull
requests have a responsibility to both the project and the individual making the
contribution. Reviews and feedback must be helpful, insightful, and geared
towards improving the contribution as opposed to simply blocking it. If there
are reasons why you feel the PR should not land, explain what those are. Do not
expect to be able to block a pull request from advancing simply because you say
"No" without giving an explanation. Be open to having your mind changed. Be open
to working with the contributor to make the pull request better.

Reviews that are dismissive or disrespectful of the contributor or any other
reviewers are strictly counter to the [Code of Conduct][].

When reviewing a pull request, the primary goals are for the codebase to improve
and for the person submitting the request to succeed. Even if a pull request
does not land, the submitters should come away from the experience feeling like
their effort was not wasted or unappreciated. Every pull request from a new
contributor is an opportunity to grow the community.

### Review a bit at a time

Do not overwhelm new contributors.

It is tempting to micro-optimize and make everything about relative performance,
perfect grammar, or exact style matches. Do not succumb to that temptation.

Focus first on the most significant aspects of the change:

1. Does this change make sense for ClusterDuck-Protocol?
2. Does this change make ClusterDuck-Protocol better, even if only incrementally?
3. Are there clear bugs or larger scale issues that need attending to?
4. Is the commit message readable and correct? If it contains a breaking change
   is it clear enough?

When changes are necessary, *request* them, do not *demand* them, and do not
assume that the submitter already knows how to add a test or run a benchmark.

Specific performance optimization techniques, coding styles and conventions
change over time. The first impression you give to a new contributor never does.

Nits (requests for small changes that are not essential) are fine, but try to
avoid stalling the pull request. Most nits can typically be fixed by the
ClusterDuck-Protocol Collaborator landing the pull request but they can also be an
opportunity for the contributor to learn a bit more about the project.

It is always good to clearly indicate nits when you comment: e.g.
`Nit: change foo() to bar(). But this is not blocking.`

If your comments were addressed but were not folded automatically after new
commits or if they proved to be mistaken, please, [hide them][hiding-a-comment]
with the appropriate reason to keep the conversation flow concise and relevant.

### Be aware of the person behind the code

Be aware that *how* you communicate requests and reviews in your feedback can
have a significant impact on the success of the pull request. Yes, we may land
a particular change that makes ClusterDuck-Protocol better, but the individual might just
not want to have anything to do with ClusterDuck-Protocol ever again. The goal is not just
having good code.

### Respect the minimum wait time for comments

There is a minimum waiting time which we try to respect for non-trivial
changes, so that people who may have important input in such a distributed
project are able to respond.

For non-trivial changes, pull requests must be left open for at least 48 hours.
In most cases, when the PR is relatively small and focused on a narrow set of
changes, that will provide more than enough time to adequately review. Sometimes
changes take far longer to review, or need more specialized review from subject
matter experts. When in doubt, do not rush.

Trivial changes, typically limited to small formatting changes or fixes to
documentation, may be landed within the minimum 48 hour window.

### Abandoned or stalled pull requests

If a pull request appears to be abandoned or stalled, it is polite to first
check with the contributor to see if they intend to continue the work before
checking if they would mind if you took it over (especially if it just has
nits left). When doing so, it is courteous to give the original contributor
credit for the work they started (either by preserving their name and email
address in the commit log, or by using an `Author:` meta-data tag in the
commit.

### Approving a change

Any ClusterDuck-Protocol core Collaborator (any GitHub user with commit rights in the
`Call-for-Code/ClusterDuck-Protocol` repository) is authorized to approve any other contributor's
work. Collaborators are not permitted to approve their own pull requests.

Collaborators indicate that they have reviewed and approve of the changes in
a pull request either by using GitHub's Approval Workflow, which is preferred,
or by leaving an `LGTM` ("Looks Good To Me") comment.

When explicitly using the "Changes requested" component of the GitHub Approval
Workflow, show empathy. That is, do not be rude or abrupt with your feedback
and offer concrete suggestions for improvement, if possible. If you're not
sure *how* a particular change can be improved, say so.

Most importantly, after leaving such requests, it is courteous to make yourself
available later to check whether your comments have been addressed.

If you see that requested changes have been made, you can clear another
collaborator's `Changes requested` review.

Change requests that are vague, dismissive, or unconstructive may also be
dismissed if requests for greater clarification go unanswered within a
reasonable period of time.

If you do not believe that the pull request should land at all, use
`Changes requested` to indicate that you are considering some of your comments
to block the PR from landing. When doing so, explain *why* you believe the
pull request should not land along with an explanation of what may be an
acceptable alternative course, if any.

### Accept that there are different opinions about what belongs in ClusterDuck-Protocol

Opinions on this vary, even among the members of the Technical Steering
Committee.

One general rule of thumb is that if ClusterDuck-Protocol itself needs it (due to historic
or functional reasons), then it belongs in ClusterDuck-Protocol.

Also, functionality that either cannot be implemented outside of core in any
reasonable way, or only with significant pain.

It is not uncommon for contributors to suggest new features they feel would
make ClusterDuck-Protocol better. These may or may not make sense to add, but as with all
changes, be courteous in how you communicate your stance on these. Comments
that make the contributor feel like they should have "known better" or
ridiculed for even trying run counter to the [Code of Conduct][].

### Performance is not everything

ClusterDuck-Protocol has always optimized for speed of execution. If a particular change
can be shown to make some part of ClusterDuck-Protocol faster, it's quite likely to be
accepted. Claims that a particular pull request will make things faster will
almost always be met by requests for performance benchmark results that
demonstrate the improvement.

That said, performance is not the only factor to consider. ClusterDuck-Protocol also
optimizes in favor of not breaking existing code in the ecosystem, and not
changing working functional code just for the sake of changing.

If a particular pull request introduces a performance or functional
regression, rather than simply rejecting the pull request, take the time to
work *with* the contributor on improving the change. Offer feedback and
advice on what would make the pull request acceptable, and do not assume that
the contributor should already know how to do that. Be explicit in your
feedback.

### Continuous integration testing

Every pull request needs to be tested
to make sure that it works on the platforms that ClusterDuck-Protocol
supports. This is currently done manually, but automated testing
may come soon.

## Notes

### Commit squashing

In most cases, do not squash commits that you add to your pull request during
the review process. When the commits in your pull request land, they may be
squashed into one commit per logical change. Metadata will be added to the
commit message (including links to the pull request, links to relevant issues,
and the names of the reviewers). The commit history of your pull request,
however, will stay intact on the pull request page.

For the size of "one logical change",
[0b5191f](https://github.com/nodejs/node/commit/0b5191f15d0f311c804d542b67e2e922d98834f8)
can be a good example. It touches the implementation, the documentation,
and the tests, but is still one logical change. All tests should always pass
when each individual commit lands on the master branch.

### Getting approvals for your pull request

A pull request is approved either by saying LGTM, which stands for
"Looks Good To Me", or by using GitHub's Approve button.
GitHub's pull request review feature can be used during the process.
For more information, check out
[the video tutorial](https://www.youtube.com/watch?v=HW0RPaJqm4g)
or [the official documentation](https://help.github.com/articles/reviewing-changes-in-pull-requests/).

After you push new changes to your branch, you need to get
approval for these new changes again, even if GitHub shows "Approved"
because the reviewers have hit the buttons before.

### CI testing

Every pull request needs to be tested
to make sure that it works on the platforms that ClusterDuck-Protocol
supports. This is currently done manually, but automated testing
may come soon.

### Waiting until the pull request gets landed

A pull request needs to stay open for at least 48 hours from when it is
submitted, even after it gets approved and passes the CI. This is to make sure
that everyone has a chance to weigh in. If the changes are trivial,
collaborators may decide it doesn't need to wait. A pull request may well take
longer to be merged in.

[Code of Conduct]: https://github.com/Call-for-Code/ClusterDuck-Protocol/blob/master/CONTRIBUTING.md#code-of-conduct
[ClusterDuck-Protocol Slack workspace]: https://www.project-owl.com/slack
