"""
Modified from the official implementation of PVT.
https://github.com/whai362/PVT/blob/v2/classification/pvt.py
"""

import os
import torch
import torch.nn as nn
import torch.nn.functional as F
from timm.models.layers import DropPath, to_2tuple, trunc_normal_

from genformer.models.utils import ckpt_path
from genformer.engine.utils import load_checkpoint
from genformer.models.common import layernorm
from genformer.models.base import BaseTransformerModel


class Mlp(nn.Module):

    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


class Attention(nn.Module):

    def __init__(self, dim, num_heads=8, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0., sr_ratio=1, eps=1e-6):
        super().__init__()
        assert dim % num_heads == 0, f"dim {dim} should be divided by num_heads {num_heads}."

        self.dim = dim
        self.num_heads = num_heads
        head_dim = dim // num_heads
        self.scale = qk_scale or head_dim ** -0.5

        self.q = nn.Linear(dim, dim, bias=qkv_bias)
        self.kv = nn.Linear(dim, dim * 2, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

        self.sr_ratio = sr_ratio
        if sr_ratio > 1:
            self.sr = nn.Conv2d(dim, dim, kernel_size=sr_ratio, stride=sr_ratio)
            self.norm = layernorm(dim, eps)

    def forward(self, x, H, W):
        B, N, C = x.shape
        q = self.q(x).reshape(B, N, self.num_heads, C // self.num_heads).permute(0, 2, 1, 3)

        if self.sr_ratio > 1:
            x_ = x.permute(0, 2, 1).reshape(B, C, H, W)
            x_ = self.sr(x_).reshape(B, C, -1).permute(0, 2, 1)
            x_ = self.norm(x_)
            kv = self.kv(x_).reshape(B, -1, 2, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        else:
            kv = self.kv(x).reshape(B, -1, 2, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        k, v = kv[0], kv[1]

        attn = (q @ k.transpose(-2, -1)) * self.scale
        attn = attn.softmax(dim=-1)
        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)

        return x


class Block(nn.Module):

    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=nn.GELU, norm_layer=nn.LayerNorm, sr_ratio=1, eps=1e-6):
        super().__init__()
        self.norm1 = norm_layer(dim, eps)
        self.attn = Attention(
            dim,
            num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale,
            attn_drop=attn_drop, proj_drop=drop, sr_ratio=sr_ratio, eps=eps)
        self.drop_path_rate = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim, eps)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

    def forward(self, x, H, W):
        x = x + self.drop_path_rate(self.attn(self.norm1(x), H, W))
        x = x + self.drop_path_rate(self.mlp(self.norm2(x)))

        return x


class PatchEmbed(nn.Module):

    def __init__(self, img_size=224, patch_size=16, in_chans=3, embed_dim=768, eps=1e-6):
        super().__init__()
        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)

        self.img_size = img_size
        self.patch_size = patch_size
        self.H, self.W = img_size[0] // patch_size[0], img_size[1] // patch_size[1]
        self.num_patches = self.H * self.W
        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size)
        self.norm = layernorm(embed_dim, eps)

    def forward(self, x):
        B, C, H, W = x.shape

        x = self.proj(x).flatten(2).transpose(1, 2)
        x = self.norm(x)
        H, W = H // self.patch_size[0], W // self.patch_size[1]

        return x, (H, W)


class PVT(BaseTransformerModel):

    def __init__(self, in_channels:int=None, num_classes:int=None, cfg:dict=None, **kwargs):
        super().__init__(cfg=cfg)
        self.in_channels = in_channels
        self.num_classes = num_classes
        
        self.sr_ratio = cfg.MODEL.PVT.SR_RATIO
        self.num_stages = len(self.hidden_dim)

        dpr = [x.item() for x in torch.linspace(0, self.drop_path_rate, sum(self.depth))]  # stochastic depth decay rule
        cur = 0

        for i in range(self.num_stages):
            patch_embed = PatchEmbed(img_size=self.img_size if i == 0 else self.img_size // (2 ** (i + 1)),
                                     patch_size=self.patch_size[i],
                                     in_chans=self.in_channels if i == 0 else self.hidden_dim[i - 1],
                                     embed_dim=self.hidden_dim[i],
                                     eps=self.ln_eps)
            num_patches = patch_embed.num_patches if i != self.num_stages - 1 else patch_embed.num_patches + 1
            pos_embed = nn.Parameter(torch.zeros(1, num_patches, self.hidden_dim[i]))
            pos_drop = nn.Dropout(p=self.drop_rate)

            block = nn.ModuleList([Block(
                dim=self.hidden_dim[i], num_heads=self.num_heads[i], mlp_ratio=self.mlp_ratio[i], qkv_bias=True,
                qk_scale=None, drop=self.drop_rate, attn_drop=self.attn_drop_rate, drop_path=dpr[cur + j],
                norm_layer=layernorm, sr_ratio=self.sr_ratio[i], eps=self.ln_eps)
                for j in range(self.depth[i])])
            cur += self.depth[i]

            setattr(self, f"patch_embed{i + 1}", patch_embed)
            setattr(self, f"pos_embed{i + 1}", pos_embed)
            setattr(self, f"pos_drop{i + 1}", pos_drop)
            setattr(self, f"block{i + 1}", block)

        layers = [[m for m in getattr(self, f'block{i + 1}')] for i in range(self.num_stages)]
        layers = sum(layers, [])

        self.norm = layernorm(self.hidden_dim[-1], self.ln_eps)

        # cls_token
        self.cls_token = nn.Parameter(torch.zeros(1, 1, self.hidden_dim[-1]))

        # classification head
        self.head = nn.Linear(self.hidden_dim[3], self.num_classes)

        # init weights
        for i in range(self.num_stages):
            pos_embed = getattr(self, f"pos_embed{i + 1}")
            trunc_normal_(pos_embed, std=.02)
        trunc_normal_(self.cls_token, std=.02)
        self.apply(self._init_weights)

    def _feature_hook(self, module, inputs, outputs):
        _, H, W = inputs
        if outputs.size(1) == H * W:
            x = outputs.view(outputs.size(0), H, W, outputs.size(-1))
        else:
            x = outputs[:, 1:].view(outputs.size(0), H, W, outputs.size(-1))
        x = x.permute(0, 3, 1, 2).contiguous()
        self.features.append(x)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def _get_pos_embed(self, pos_embed, patch_embed, H, W):
        if H * W == self.patch_embed1.num_patches:
            return pos_embed
        else:
            return F.interpolate(
                pos_embed.reshape(1, patch_embed.H, patch_embed.W, -1).permute(0, 3, 1, 2),
                size=(H, W), mode="bilinear", align_corners=False).reshape(1, -1, H * W).permute(0, 2, 1)

    def forward_features(self, x):
        B = x.shape[0]

        for i in range(self.num_stages):
            patch_embed = getattr(self, f"patch_embed{i + 1}")
            pos_embed = getattr(self, f"pos_embed{i + 1}")
            pos_drop = getattr(self, f"pos_drop{i + 1}")
            block = getattr(self, f"block{i + 1}")
            x, (H, W) = patch_embed(x)

            if i == self.num_stages - 1:
                cls_tokens = self.cls_token.expand(B, -1, -1)
                x = torch.cat((cls_tokens, x), dim=1)
                pos_embed_ = self._get_pos_embed(pos_embed[:, 1:], patch_embed, H, W)
                pos_embed = torch.cat((pos_embed[:, 0:1], pos_embed_), dim=1)
            else:
                pos_embed = self._get_pos_embed(pos_embed, patch_embed, H, W)

            x = pos_drop(x + pos_embed)
            for blk in block:
                x = blk(x, H, W)
            if i != self.num_stages - 1:
                x = x.reshape(B, H, W, -1).permute(0, 3, 1, 2).contiguous()

        x = self.norm(x)

        return x[:, 0]

    def forward(self, x, tsne=False):
        x = self.forward_features(x)
        x = self.head(x)
        feats = {}
        if tsne:
            return x, feats
        else:
            return x
    

def pvt(pretrained=False, **kwargs):
    model = PVT(**kwargs)
    cfg = kwargs["cfg"]
    if pretrained:
        try:
            model.load_state_dict(load_checkpoint(os.path.join(
                ckpt_path(), f"cifar{kwargs['num_classes']}_models", "pvt", "student_best"))["model"])
        except:
            raise Exception(f"No suitable pretrained model for: {model.__class__.__name__}")
    return model


if __name__ == "__main__":
    from genformer.engine.cfg import CFG as cfg
    
    img_size = 32
    in_channels = 3
    num_classes = 100
    
    cfg.DATASET.IMG_SIZE = img_size
    
    cfg.MODEL.TRANSFORMER.PATCH_SIZE = [4, 2, 2, 2]
    cfg.MODEL.TRANSFORMER.HIDDEN_DIM = [64, 128, 320, 512]
    cfg.MODEL.TRANSFORMER.DEPTH = [2, 2, 2, 2]
    cfg.MODEL.TRANSFORMER.NUM_HEADS = [1, 2, 5, 8]
    cfg.MODEL.TRANSFORMER.MLP_RATIO = [8, 8, 4, 4]
    cfg.MODEL.TRANSFORMER.LN_EPS = 1e-6
    cfg.MODEL.TRANSFORMER.DROP_RATE = 0.0
    cfg.MODEL.TRANSFORMER.DROP_PATH_RATE = 0.1
    cfg.MODEL.TRANSFORMER.ATTENTION_DROP_RATE = 0.0
    
    cfg.MODEL.PVT.SR_RATIO = [8, 4, 2, 1]
    
    x = torch.randn(2, in_channels, img_size, img_size).cuda()
    net = pvt(in_channels=in_channels, num_classes=num_classes, cfg=cfg).cuda()
    import time

    a = time.time()
    logit, feats = net(x)
    b = time.time()
    print(b - a)
    print(logit.shape)
    print(net.complexity())
