import {
  DataResult,
  DataSource,
  ExecuteOptions,
  ExportOptions,
  InternalError,
  RequestParameter,
  VulcanExtensionId,
} from '@vulcan-sql/core';
import * as snowflake from 'snowflake-sdk';
import { Options as PoolOptions, Pool } from 'generic-pool';
import { buildSQL } from './sqlBuilder';
import { mapFromSnowflakeColumnType } from './typeMapper';
import * as fs from 'fs';

export interface SnowflakeOptions
  extends snowflake.ConnectionOptions,
    PoolOptions {}

@VulcanExtensionId('snowflake')
export class SnowflakeDataSource extends DataSource<any, SnowflakeOptions> {
  private logger = this.getLogger();
  private poolMapping = new Map<
    string,
    { pool: Pool<snowflake.Connection>; options?: SnowflakeOptions }
  >();

  public override async onActivate() {
    const profiles = this.getProfiles().values();
    for (const profile of profiles) {
      this.logger.debug(
        `Initializing profile: ${profile.name} using snowflake driver`
      );
      const options: SnowflakeOptions = {
        ...{ application: 'VulcanSQL' },
        ...profile.connection!,
      };
      const pool = snowflake.createPool(
        options, // optionals for Snowflake SDK
        options // optionals for connection pool
      );

      this.poolMapping.set(profile.name, {
        pool,
        options: profile.connection,
      });

      await pool.use((connection) =>
        this.getResultFromSQL(connection, 'SELECT 1;')
      );
      this.logger.debug(`Profile ${profile.name} initialized`);
    }
  }

  public async execute({
    statement: sql,
    bindParams,
    profileName,
    operations,
  }: ExecuteOptions): Promise<DataResult> {
    const { connection, pool } = await this.getConnection(profileName);
    try {
      const builtSQL = buildSQL(sql, operations);
      const statement = await this.getStatementFromSQL(
        connection,
        builtSQL,
        Array.from(bindParams.values())
      );
      return {
        getColumns: () =>
          statement.getColumns().map((column) => ({
            name: column.getName(),
            type: mapFromSnowflakeColumnType(column),
          })),
        // The chunk size of stream is decided by Snowflake server, it used exponential chunk size, that is, the chunk size will be 0.1Mb, 0.2Mb, 0.4Mb ... 16Mb.
        // It sometimes occupies too mush memory with large results.
        // See https://github.com/snowflakedb/snowflake-connector-nodejs/issues/43
        getData: () => statement.streamRows(),
      };
    } finally {
      this.logger.debug(`Release connection to ${profileName}`);
      // We release the client right after the query is done without waiting the data to be fetched.
      // Because Snowflake server executes queries asynchronously, it saves the query result to storage providers like s3, so we don't need a database connection while fetching results.
      pool.release(connection);
    }
  }

  public async prepare({ parameterIndex }: RequestParameter) {
    return `:${parameterIndex}`;
  }

  public async destroy() {
    for (const { pool } of this.poolMapping.values()) {
      await pool.drain();
      await pool.clear();
    }
  }

  public override async export(exportOptions: ExportOptions): Promise<void> {
    const { directory } = exportOptions;
    // throw error if filepath dir not exist
    if (!fs.existsSync(directory)) {
      throw new InternalError(`Directory ${directory} not exist`);
    }
    const { profileName, sql } = exportOptions;
    const { connection, pool } = await this.getConnection(profileName);

    /**
     * Because the "copy into" command in Snowflake does not support downloading files to the local file system.
     * Data is temporarily stored in a stage and then deleted after downloading.
     */
    try {
      const stage = '@~/vulcan_cache_stage';
      const stageFilePath = `${stage}/`;
      const builtSQL = this.getCopyToStageSQL(sql, stageFilePath);
      // copy data to a named stage
      const copyStatement = await this.getStatementFromSQL(
        connection,
        builtSQL
      );
      await this.waitStatement(copyStatement);
      this.logger.debug('Copied data to internal stage: @~/vulcan_cache_stage');

      // get parquet file from a named stage
      const statementId = copyStatement.getStatementId();
      const pattern = `.*${statementId}.*`; // UUID generated by snowflake
      const getStatement = await this.getStatementFromSQL(
        connection,
        `GET ${stage} file://${directory} pattern='${pattern}'`
      );
      await this.waitStatement(getStatement);
      this.logger.debug(`Exported parquet files to ${directory}`);

      // remove parquet file from stage
      await this.getStatementFromSQL(
        connection,
        `REMOVE ${stage} pattern='${pattern}'`
      );
      this.logger.debug('Removed parquet files from stage');
    } catch (err) {
      this.logger.error(`Error when exporting parquet "${directory}": `, err);
      throw err;
    } finally {
      this.logger.debug(`Release connection to ${profileName}`);
      pool.release(connection);
    }
  }

  // use protected for testing purpose
  // the MAX_FILE_SIZE is 5GB, see https://docs.snowflake.com/en/sql-reference/sql/copy-into-location#copy-options-copyoptions
  protected getCopyToStageSQL(sql: string, stageFilePath: string) {
    return `COPY INTO ${stageFilePath} FROM (${sql}) FILE_FORMAT = (TYPE = 'parquet') HEADER=true INCLUDE_QUERY_ID=true MAX_FILE_SIZE=5368709120;`;
  }

  private async waitStatement(statement: snowflake.Statement): Promise<void> {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    return new Promise((resolve, reject) => {
      const status = statement.getStatus();
      while (status !== 'complete');
      resolve();
    });
  }

  private async getStatementFromSQL(
    connection: snowflake.Connection,
    sql: string,
    binding: any[] = []
  ): Promise<snowflake.Statement> {
    return new Promise<snowflake.Statement>((resolve, reject) => {
      connection.execute({
        sqlText: sql,
        streamResult: true,
        binds: binding,
        complete: (err, stmt) => {
          if (err) reject(err);
          resolve(stmt);
        },
      });
    });
  }

  private async getResultFromSQL(
    connection: snowflake.Connection,
    sql: string
  ): Promise<any[]> {
    return new Promise<any[]>((resolve, reject) => {
      connection.execute({
        sqlText: sql,
        complete: (err, _stmt, rows) => {
          if (err) return reject(err);
          resolve(rows || []);
        },
      });
    });
  }

  private async getConnection(profileName: string) {
    if (!this.poolMapping.has(profileName)) {
      throw new InternalError(`Profile instance ${profileName} not found`);
    }
    const { pool } = this.poolMapping.get(profileName)!;
    this.logger.debug(`Acquiring connection from ${profileName}`);
    const connection = await pool.acquire();
    this.logger.debug(`Acquired connection from ${profileName}`);
    return { connection, pool };
  }
}
