import { expect, it } from 'vitest'
import type * as AST from '../ast'
import parse from '../parse'
import gen from '../gen'

type Tests = {
  [key: string]: AST.Regex
}
const tests: Tests = {
  '/foo/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'foo',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/foo|bar/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'choice',
        branches: [
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: 'foo',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: 'bar',
              quantifier: null,
            },
          ],
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/||||/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'choice',
        branches: [
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: '',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: '',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: '',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: '',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: '',
              quantifier: null,
            },
          ],
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^|$|\\b|\\B/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'choice',
        branches: [
          [{ id: '', type: 'boundaryAssertion', kind: 'beginning' }],
          [{ id: '', type: 'boundaryAssertion', kind: 'end' }],
          [{ id: '', type: 'boundaryAssertion', kind: 'word', negate: false }],
          [{ id: '', type: 'boundaryAssertion', kind: 'word', negate: true }],
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(?=)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'lookAroundAssertion',
        kind: 'lookahead',
        negate: false,
        children: [],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(?=foo)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'lookAroundAssertion',
        kind: 'lookahead',
        negate: false,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'foo',
            quantifier: null,
          },
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(?!)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'lookAroundAssertion',
        kind: 'lookahead',
        negate: true,
        children: [],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(?!foo)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'lookAroundAssertion',
        kind: 'lookahead',
        negate: true,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'foo',
            quantifier: null,
          },
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a*/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a+/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{}',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{a}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{a}',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: 'custom', min: 1, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{1',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: 'custom', min: 1, max: Infinity, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{1,',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,2}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: 'custom', min: 1, max: 2, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,2/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{1,2',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{2,1/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{2,1',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a*?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: false },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a+?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: false },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a??/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: '?', min: 0, max: 1, greedy: false },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '{',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{}?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '}',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{a}?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{a',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '}',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1}?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: 'custom', min: 1, max: 1, greedy: false },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '1',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,}?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: 'custom', min: 1, max: Infinity, greedy: false },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{1',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: ',',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,2}?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: { kind: 'custom', min: 1, max: 2, greedy: false },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{1,2?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{1,',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '2',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a{2,1?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a{2,',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '1',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/👍🚀❇️/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '👍🚀❇️',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^/': {
    id: '',
    type: 'regex',
    body: [{ id: '', type: 'boundaryAssertion', kind: 'beginning' }],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/$/': {
    id: '',
    type: 'regex',
    body: [{ id: '', type: 'boundaryAssertion', kind: 'end' }],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/./': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '.',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: ']',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/{/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '{',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '}',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/|/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'choice',
        branches: [
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: '',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: '',
              quantifier: null,
            },
          ],
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/${1,2/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'end' },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '{1,2',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\1/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: '1',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(a)\\1/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'a',
            quantifier: null,
          },
        ],
        quantifier: null,
      },
      {
        id: '',
        type: 'backReference',
        ref: '1',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\1(a)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: '1',
        quantifier: null,
      },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'a',
            quantifier: null,
          },
        ],
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(?:a)\\1/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'group',
        kind: 'nonCapturing',
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'a',
            quantifier: null,
          },
        ],
        quantifier: null,
      },
      {
        id: '',
        type: 'backReference',
        ref: '1',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(?:a)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'group',
        kind: 'nonCapturing',
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'a',
            quantifier: null,
          },
        ],
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\d/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\d',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\D/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\D',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\s/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\s',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\S/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\S',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\w/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\w',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\W/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\W',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\f/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\f',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\n/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\n',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\r/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\r',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\t/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\t',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\v/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\v',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\cA/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\cA',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\cz/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\cz',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\c1/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'c1',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\c/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'c',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\0/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\0',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u1/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u1',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u12/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u12',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u123/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u123',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u1234/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\u1234',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u12345/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\u1234',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '5',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u{/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u{',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u{z/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u{z',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u{a}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u{a}',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u{20/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u{20',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\u{20}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'u',
        quantifier: { kind: 'custom', min: 20, max: 20, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\377/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: '377',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\400/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: '400',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\^/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '^',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\$/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '$',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\./': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '.',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\+/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '+',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '?',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\(/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '(',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: ')',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\[/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '[',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: ']',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\{/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '{',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '}',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\|/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '|',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\//': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '/',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\a/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'a',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[^-a-b-]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '-', to: '-', id: '' },
          { from: 'a', to: 'b', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: true,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[-]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '-', to: '-', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[a]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: 'a', to: 'a', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[--]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '-', to: '-', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[-a]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '-', to: '-', id: '' },
          { from: 'a', to: 'a', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[-a-]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '-', to: '-', id: '' },
          { from: 'a', to: 'a', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[a-]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'a', to: 'a', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[a-b]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: 'a', to: 'b', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[-a-b-]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '-', to: '-', id: '' },
          { from: 'a', to: 'b', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[---]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '-', to: '-', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[a-b--/]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'a', to: 'b', id: '' },
          { from: '-', to: '/', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\b-\\n]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\b', to: '\\n', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[b\\-a]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'b', to: 'b', id: '' },
          { from: '-', to: '-', id: '' },
          { from: 'a', to: 'a', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\d]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\d', to: '\\d', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\D]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\D', to: '\\D', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\s]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\s', to: '\\s', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\S]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\S', to: '\\S', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\w]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\w', to: '\\w', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\W]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\W', to: '\\W', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\f]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\f', to: '\\f', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\n]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\n', to: '\\n', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\r]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\r', to: '\\r', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\t]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\t', to: '\\t', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\v]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\v', to: '\\v', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\cA]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\cA', to: '\\cA', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\cz]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\cz', to: '\\cz', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\c1]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'c', to: 'c', id: '' },
          { from: '1', to: '1', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\c]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: 'c', to: 'c', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\0]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\0', to: '\\0', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\x]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: 'x', to: 'x', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\xz]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'x', to: 'x', id: '' },
          { from: 'z', to: 'z', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\x1]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'x', to: 'x', id: '' },
          { from: '1', to: '1', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\x12]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\x12', to: '\\x12', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\x123]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '\\x12', to: '\\x12', id: '' },
          { from: '3', to: '3', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: 'u', to: 'u', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u1]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '1', to: '1', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u12]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '1', to: '1', id: '' },
          { from: '2', to: '2', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u123]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '1', to: '1', id: '' },
          { from: '2', to: '2', id: '' },
          { from: '3', to: '3', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u1234]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u1234', to: '\\u1234', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u12345]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '\\u1234', to: '\\u1234', id: '' },
          { from: '5', to: '5', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u{]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '{', to: '{', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u{z]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '{', to: '{', id: '' },
          { from: 'z', to: 'z', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u{a}]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '{', to: '{', id: '' },
          { from: 'a', to: 'a', id: '' },
          { from: '}', to: '}', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u{20]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '{', to: '{', id: '' },
          { from: '2', to: '2', id: '' },
          { from: '0', to: '0', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u{20}]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '{', to: '{', id: '' },
          { from: '2', to: '2', id: '' },
          { from: '0', to: '0', id: '' },
          { from: '}', to: '}', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\77]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '7', to: '7', id: '' },
          { from: '7', to: '7', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\377]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '3', to: '3', id: '' },
          { from: '7', to: '7', id: '' },
          { from: '7', to: '7', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\400]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '4', to: '4', id: '' },
          { from: '0', to: '0', id: '' },
          { from: '0', to: '0', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\^]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '^', to: '^', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\$]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '$', to: '$', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\.]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '.', to: '.', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\+]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '+', to: '+', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\?]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '?', to: '?', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\(]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '(', to: '(', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\)]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: ')', to: ')', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\[]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '[', to: '[', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\]]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: ']', to: ']', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\{]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '{', to: '{', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\}]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '}', to: '}', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\|]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '|', to: '|', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\/]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '/', to: '/', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\a]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: 'a', to: 'a', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\d-\\uFFFF]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\d', to: '\\uFFFF', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\D-\\uFFFF]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\D', to: '\\uFFFF', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\s-\\uFFFF]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\s', to: '\\uFFFF', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\S-\\uFFFF]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\S', to: '\\uFFFF', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\w-\\uFFFF]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\w', to: '\\uFFFF', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\W-\\uFFFF]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\W', to: '\\uFFFF', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u0000-\\d]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u0000', to: '\\d', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u0000-\\D]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u0000', to: '\\D', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u0000-\\s]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u0000', to: '\\s', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u0000-\\S]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u0000', to: '\\S', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u0000-\\w]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u0000', to: '\\w', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u0000-\\W]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u0000', to: '\\W', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u0000-\\u0001]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\u0000', to: '\\u0001', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\u{2-\\u{1}]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'u', to: 'u', id: '' },
          { from: '{', to: '{', id: '' },
          { from: '2', to: 'u', id: '' },
          { from: '{', to: '{', id: '' },
          { from: '1', to: '1', id: '' },
          { from: '}', to: '}', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\a-\\z]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: 'a', to: 'z', id: '' }],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[0-9--/]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '0', to: '9', id: '' },
          { from: '-', to: '/', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\c0-\\u001f]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'c', to: 'c', id: '' },
          { from: '0', to: '\\u001f', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/[\\c_]/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'c', to: 'c', id: '' },
          { from: '_', to: '_', id: '' },
        ],
        negate: false,
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[0-9]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '0', to: '9', id: '' }],
        negate: false,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[0-9]+$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '0', to: '9', id: '' }],
        negate: false,
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[a-zA-Z]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'a', to: 'z', id: '' },
          { from: 'A', to: 'Z', id: '' },
        ],
        negate: false,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[a-zA-Z]+$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'a', to: 'z', id: '' },
          { from: 'A', to: 'Z', id: '' },
        ],
        negate: false,
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[0-9a-zA-Z]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: '0', to: '9', id: '' },
          { from: 'a', to: 'z', id: '' },
          { from: 'A', to: 'Z', id: '' },
        ],
        negate: false,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[a-zA-Z0-9!-/:-@\\[-`{-~]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'a', to: 'z', id: '' },
          { from: 'A', to: 'Z', id: '' },
          { from: '0', to: '9', id: '' },
          { from: '!', to: '/', id: '' },
          { from: ':', to: '@', id: '' },
          { from: '[', to: '`', id: '' },
          { from: '{', to: '~', id: '' },
        ],
        negate: false,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^([a-zA-Z0-9]{8,})$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'ranges',
            ranges: [
              { from: 'a', to: 'z', id: '' },
              { from: 'A', to: 'Z', id: '' },
              { from: '0', to: '9', id: '' },
            ],
            negate: false,
            quantifier: { kind: 'custom', min: 8, max: Infinity, greedy: true },
          },
        ],
        quantifier: null,
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^([a-zA-Z0-9]{6,8})$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'ranges',
            ranges: [
              { from: 'a', to: 'z', id: '' },
              { from: 'A', to: 'Z', id: '' },
              { from: '0', to: '9', id: '' },
            ],
            negate: false,
            quantifier: { kind: 'custom', min: 6, max: 8, greedy: true },
          },
        ],
        quantifier: null,
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^([0-9]{0,8})$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'ranges',
            ranges: [{ from: '0', to: '9', id: '' }],
            negate: false,
            quantifier: { kind: 'custom', min: 0, max: 8, greedy: true },
          },
        ],
        quantifier: null,
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[0-9]{8}$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '0', to: '9', id: '' }],
        negate: false,
        quantifier: { kind: 'custom', min: 8, max: 8, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^https?:\\/\\//': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'http',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 's',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '://',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^\\d{3}-\\d{4}$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\d',
        quantifier: { kind: 'custom', min: 3, max: 3, greedy: true },
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '-',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\d',
        quantifier: { kind: 'custom', min: 4, max: 4, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^\\d{1,3}(.\\d{1,3}){3}$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'class',
        value: '\\d',
        quantifier: { kind: 'custom', min: 1, max: 3, greedy: true },
      },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'class',
            value: '.',
            quantifier: null,
          },
          {
            id: '',
            type: 'character',
            kind: 'class',
            value: '\\d',
            quantifier: { kind: 'custom', min: 1, max: 3, greedy: true },
          },
        ],
        quantifier: { kind: 'custom', min: 3, max: 3, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^([1-9][0-9]*|0)(\\.[0-9]+)?$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'choice',
            branches: [
              [
                {
                  id: '',
                  type: 'character',
                  kind: 'ranges',
                  ranges: [{ from: '1', to: '9', id: '' }],
                  negate: false,
                  quantifier: null,
                },
                {
                  id: '',
                  type: 'character',
                  kind: 'ranges',
                  ranges: [{ from: '0', to: '9', id: '' }],
                  negate: false,
                  quantifier: {
                    kind: '*',
                    min: 0,
                    max: Infinity,
                    greedy: true,
                  },
                },
              ],
              [
                {
                  id: '',
                  type: 'character',
                  kind: 'string',
                  value: '0',
                  quantifier: null,
                },
              ],
            ],
          },
        ],
        quantifier: null,
      },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '2',
        index: 2,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: '.',
            quantifier: null,
          },
          {
            id: '',
            type: 'character',
            kind: 'ranges',
            ranges: [{ from: '0', to: '9', id: '' }],
            negate: false,
            quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
          },
        ],
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^-?([1-9][0-9]*|0)(\\.[0-9]+)?$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '-',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'choice',
            branches: [
              [
                {
                  id: '',
                  type: 'character',
                  kind: 'ranges',
                  ranges: [{ from: '1', to: '9', id: '' }],
                  negate: false,
                  quantifier: null,
                },
                {
                  id: '',
                  type: 'character',
                  kind: 'ranges',
                  ranges: [{ from: '0', to: '9', id: '' }],
                  negate: false,
                  quantifier: {
                    kind: '*',
                    min: 0,
                    max: Infinity,
                    greedy: true,
                  },
                },
              ],
              [
                {
                  id: '',
                  type: 'character',
                  kind: 'string',
                  value: '0',
                  quantifier: null,
                },
              ],
            ],
          },
        ],
        quantifier: null,
      },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '2',
        index: 2,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: '.',
            quantifier: null,
          },
          {
            id: '',
            type: 'character',
            kind: 'ranges',
            ranges: [{ from: '0', to: '9', id: '' }],
            negate: false,
            quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
          },
        ],
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[ぁ-んー]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'ぁ', to: 'ん', id: '' },
          { from: 'ー', to: 'ー', id: '' },
        ],
        negate: false,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[ァ-ンヴー]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'ァ', to: 'ン', id: '' },
          { from: 'ヴ', to: 'ヴ', id: '' },
          { from: 'ー', to: 'ー', id: '' },
        ],
        negate: false,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[ｧ-ﾝﾞﾟ\\-]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'ｧ', to: 'ﾝ', id: '' },
          { from: 'ﾞ', to: 'ﾞ', id: '' },
          { from: 'ﾟ', to: 'ﾟ', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: false,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[^\\x20-\\x7e]*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [{ from: '\\x20', to: '\\x7e', id: '' }],
        negate: true,
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^[a-zA-Z0-9.!#$%&\'*+/=?^_`{|}~-]+@[a-zA-Z0-9-]+(?:\\.[a-zA-Z0-9-]+)*$/': {
    id: '',
    type: 'regex',
    body: [
      { id: '', type: 'boundaryAssertion', kind: 'beginning' },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'a', to: 'z', id: '' },
          { from: 'A', to: 'Z', id: '' },
          { from: '0', to: '9', id: '' },
          { from: '.', to: '.', id: '' },
          { from: '!', to: '!', id: '' },
          { from: '#', to: '#', id: '' },
          { from: '$', to: '$', id: '' },
          { from: '%', to: '%', id: '' },
          { from: '&', to: '&', id: '' },
          { from: '\'', to: '\'', id: '' },
          { from: '*', to: '*', id: '' },
          { from: '+', to: '+', id: '' },
          { from: '/', to: '/', id: '' },
          { from: '=', to: '=', id: '' },
          { from: '?', to: '?', id: '' },
          { from: '^', to: '^', id: '' },
          { from: '_', to: '_', id: '' },
          { from: '`', to: '`', id: '' },
          { from: '{', to: '{', id: '' },
          { from: '|', to: '|', id: '' },
          { from: '}', to: '}', id: '' },
          { from: '~', to: '~', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: false,
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: '@',
        quantifier: null,
      },
      {
        id: '',
        type: 'character',
        kind: 'ranges',
        ranges: [
          { from: 'a', to: 'z', id: '' },
          { from: 'A', to: 'Z', id: '' },
          { from: '0', to: '9', id: '' },
          { from: '-', to: '-', id: '' },
        ],
        negate: false,
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
      },
      {
        id: '',
        type: 'group',
        kind: 'nonCapturing',
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: '.',
            quantifier: null,
          },
          {
            id: '',
            type: 'character',
            kind: 'ranges',
            ranges: [
              { from: 'a', to: 'z', id: '' },
              { from: 'A', to: 'Z', id: '' },
              { from: '0', to: '9', id: '' },
              { from: '-', to: '-', id: '' },
            ],
            negate: false,
            quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
          },
        ],
        quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
      },
      { id: '', type: 'boundaryAssertion', kind: 'end' },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/^((4\\d{3})|(5[1-5]\\d{2})|(6011))([- ])?\\d{4}([- ])?\\d{4}([- ])?\\d{4}|3[4,7]\\d{13}$/':
    {
      id: '',
      type: 'regex',
      body: [
        {
          id: '',
          type: 'choice',
          branches: [
            [
              { id: '', type: 'boundaryAssertion', kind: 'beginning' },
              {
                id: '',
                type: 'group',
                kind: 'capturing',
                name: '1',
                index: 1,
                children: [
                  {
                    id: '',
                    type: 'choice',
                    branches: [
                      [
                        {
                          id: '',
                          type: 'group',
                          kind: 'capturing',
                          name: '2',
                          index: 2,
                          children: [
                            {
                              id: '',
                              type: 'character',
                              kind: 'string',
                              value: '4',
                              quantifier: null,
                            },
                            {
                              id: '',
                              type: 'character',
                              kind: 'class',
                              value: '\\d',
                              quantifier: {
                                kind: 'custom',
                                min: 3,
                                max: 3,
                                greedy: true,
                              },
                            },
                          ],
                          quantifier: null,
                        },
                      ],
                      [
                        {
                          id: '',
                          type: 'group',
                          kind: 'capturing',
                          name: '3',
                          index: 3,
                          children: [
                            {
                              id: '',
                              type: 'character',
                              kind: 'string',
                              value: '5',
                              quantifier: null,
                            },
                            {
                              id: '',
                              type: 'character',
                              kind: 'ranges',
                              negate: false,
                              ranges: [{ from: '1', to: '5', id: '' }],
                              quantifier: null,
                            },
                            {
                              id: '',
                              type: 'character',
                              kind: 'class',
                              value: '\\d',
                              quantifier: {
                                kind: 'custom',
                                min: 2,
                                max: 2,
                                greedy: true,
                              },
                            },
                          ],
                          quantifier: null,
                        },
                      ],
                      [
                        {
                          id: '',
                          type: 'group',
                          kind: 'capturing',
                          name: '4',
                          index: 4,
                          children: [
                            {
                              id: '',
                              type: 'character',
                              kind: 'string',
                              value: '6011',
                              quantifier: null,
                            },
                          ],
                          quantifier: null,
                        },
                      ],
                    ],
                  },
                ],
                quantifier: null,
              },
              {
                id: '',
                type: 'group',
                kind: 'capturing',
                name: '5',
                index: 5,
                children: [
                  {
                    id: '',
                    type: 'character',
                    kind: 'ranges',
                    negate: false,
                    ranges: [
                      { from: '-', to: '-', id: '' },
                      { from: ' ', to: ' ', id: '' },
                    ],
                    quantifier: null,
                  },
                ],
                quantifier: { kind: '?', min: 0, max: 1, greedy: true },
              },
              {
                id: '',
                type: 'character',
                kind: 'class',
                value: '\\d',
                quantifier: { kind: 'custom', min: 4, max: 4, greedy: true },
              },
              {
                id: '',
                type: 'group',
                kind: 'capturing',
                name: '6',
                index: 6,
                children: [
                  {
                    id: '',
                    type: 'character',
                    kind: 'ranges',
                    negate: false,
                    ranges: [
                      { from: '-', to: '-', id: '' },
                      { from: ' ', to: ' ', id: '' },
                    ],
                    quantifier: null,
                  },
                ],
                quantifier: { kind: '?', min: 0, max: 1, greedy: true },
              },
              {
                id: '',
                type: 'character',
                kind: 'class',
                value: '\\d',
                quantifier: { kind: 'custom', min: 4, max: 4, greedy: true },
              },
              {
                id: '',
                type: 'group',
                kind: 'capturing',
                name: '7',
                index: 7,
                children: [
                  {
                    id: '',
                    type: 'character',
                    kind: 'ranges',
                    negate: false,
                    ranges: [
                      { from: '-', to: '-', id: '' },
                      { from: ' ', to: ' ', id: '' },
                    ],
                    quantifier: null,
                  },
                ],
                quantifier: { kind: '?', min: 0, max: 1, greedy: true },
              },
              {
                id: '',
                type: 'character',
                kind: 'class',
                value: '\\d',
                quantifier: { kind: 'custom', min: 4, max: 4, greedy: true },
              },
            ],
            [
              {
                id: '',
                type: 'character',
                kind: 'string',
                value: '3',
                quantifier: null,
              },
              {
                id: '',
                type: 'character',
                kind: 'ranges',
                negate: false,
                ranges: [
                  { from: '4', to: '4', id: '' },
                  { from: ',', to: ',', id: '' },
                  { from: '7', to: '7', id: '' },
                ],
                quantifier: null,
              },
              {
                id: '',
                type: 'character',
                kind: 'class',
                value: '\\d',
                quantifier: { kind: 'custom', min: 13, max: 13, greedy: true },
              },
              { id: '', type: 'boundaryAssertion', kind: 'end' },
            ],
          ],
        },
      ],
      flags: [],
      literal: true,
      escapeBackslash: false,
    },
  '/^\\s*|\\s*$/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'choice',
        branches: [
          [
            { id: '', type: 'boundaryAssertion', kind: 'beginning' },
            {
              id: '',
              type: 'character',
              kind: 'class',
              value: '\\s',
              quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'class',
              value: '\\s',
              quantifier: { kind: '*', min: 0, max: Infinity, greedy: true },
            },
            { id: '', type: 'boundaryAssertion', kind: 'end' },
          ],
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/(a)(?<n>b)(c)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '1',
        index: 1,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'a',
            quantifier: null,
          },
        ],
        quantifier: null,
      },
      {
        id: '',
        type: 'group',
        kind: 'namedCapturing',
        name: 'n',
        index: 2,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'b',
            quantifier: null,
          },
        ],
        quantifier: null,
      },
      {
        id: '',
        type: 'group',
        kind: 'capturing',
        name: '3',
        index: 3,
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'c',
            quantifier: null,
          },
        ],
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/x(?=y)/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'x',
        quantifier: null,
      },
      {
        id: '',
        type: 'lookAroundAssertion',
        kind: 'lookahead',
        children: [
          {
            id: '',
            type: 'character',
            kind: 'string',
            value: 'y',
            quantifier: null,
          },
        ],
        negate: false,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/x?y/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'x',
        quantifier: { kind: '?', min: 0, max: 1, greedy: true },
      },
      {
        id: '',
        type: 'character',
        kind: 'string',
        value: 'y',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\k<Name>/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: 'Name',
        quantifier: null,
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/a|b|c/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'choice',
        branches: [
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: 'a',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: 'b',
              quantifier: null,
            },
          ],
          [
            {
              id: '',
              type: 'character',
              kind: 'string',
              value: 'c',
              quantifier: null,
            },
          ],
        ],
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\1+/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: '1',
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\1+?/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: '1',
        quantifier: { kind: '+', min: 1, max: Infinity, greedy: false },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\k<name>{1,10}/': {
    id: '',
    type: 'regex',
    body: [
      {
        id: '',
        type: 'backReference',
        ref: 'name',
        quantifier: { kind: 'custom', min: 1, max: 10, greedy: true },
      },
    ],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
  '/\\aaa?/': {
    id: '',
    type: 'regex',
    body: [{
      id: '',
      type: 'character',
      kind: 'string',
      value: 'aa',
      quantifier: null,
    }, {
      id: '',
      type: 'character',
      kind: 'string',
      value: 'a',
      quantifier: { kind: '?', min: 0, max: 1, greedy: true },
    }],
    flags: [],
    literal: true,
    escapeBackslash: false,
  },
}

const withoutGenRegex = [
  '/a{/',
  '/a{}/',
  '/a{a}/',
  '/a{1/',
  '/a{1,/',
  '/a{1,2/',
  '/a{2,1/',
  '/a{?/',
  '/a{}?/',
  '/a{a}?/',
  '/a{1?/',
  '/a{1,?/',
  '/a{1,2?/',
  '/a{2,1?/',
  '/]/',
  '/{/',
  '/}/',
  '/${1,2/',
  '/\\c1/',
  '/\\c/',
  '/\\u/',
  '/\\u1/',
  '/\\u12/',
  '/\\u123/',
  '/\\u{/',
  '/\\u{z/',
  '/\\u{a}/',
  '/\\u{20/',
  '/\\u{20}/',
  '/\\a/',
  '/[---]/',
  '/[\\c1]/',
  '/[\\c]/',
  '/[\\x]/',
  '/[\\xz]/',
  '/[\\x1]/',
  '/[\\u]/',
  '/[\\u1]/',
  '/[\\u12]/',
  '/[\\u123]/',
  '/[\\u{]/',
  '/[\\u{z]/',
  '/[\\u{a}]/',
  '/[\\u{20]/',
  '/[\\u{20}]/',
  '/[\\77]/',
  '/[\\377]/',
  '/[\\400]/',
  '/[\\^]/',
  '/[\\$]/',
  '/[\\.]/',
  '/[\\+]/',
  '/[\\?]/',
  '/[\\(]/',
  '/[\\)]/',
  '/[\\[]/',
  '/[\\{]/',
  '/[\\}]/',
  '/[\\|]/',
  '/[\\/]/',
  '/[\\a]/',
  '/[\\u{2-\\u{1}]/',
  '/[\\a-\\z]/',
  '/[\\c0-\\u001f]/',
  '/[\\c_]/',
  '/^[a-zA-Z0-9!-/:-@\\[-`{-~]*$/',
  '/^[ｧ-ﾝﾞﾟ\\-]*$/',
  '/\\aaa?/',
]

it('parse es2015 regex', () => {
  for (const [regex, ast] of Object.entries(tests)) {
    expect(parse(regex, { idGenerator: () => '' })).toEqual(ast)
  }
})

it('gen es2015 regex by ast', () => {
  const genTests = { ...tests }
  withoutGenRegex.forEach((regex) => {
    delete genTests[regex]
  })
  for (const [regex, ast] of Object.entries(genTests)) {
    expect(gen(ast)).toEqual(regex)
  }
})
