#include "settings/filename-window.h"
#include <QDesktopServices>
#include <QMessageBox>
#include <QRegularExpression>
#include <ui_filename-window.h>
#include "filename/filename.h"
#include "models/image.h"
#include "models/profile.h"
#include "models/site.h"


FilenameWindow::FilenameWindow(Profile *profile, QString value, QWidget *parent)
	: QDialog(parent), ui(new Ui::FilenameWindow), m_profile(profile)
{
	ui->setupUi(this);

	#if defined(USE_QSCINTILLA)
		m_scintilla = new QsciScintilla(this);
		QsciLexerJavaScript *lexer = new QsciLexerJavaScript(this);
		m_scintilla->setLexer(lexer);
	#else
		m_scintilla = new QTextEdit(this);
	#endif

	connect(ui->radioJavascript, &QRadioButton::toggled, m_scintilla, &QWidget::setEnabled);
	ui->verticalLayout->insertWidget(ui->verticalLayout->count() - 1, m_scintilla);

	if (value.startsWith("javascript:")) {
		value = value.right(value.length() - 11);
		m_scintilla->setText(value);
		ui->lineClassic->setEnabled(false);
		ui->radioJavascript->toggle();
	} else {
		ui->lineClassic->setText(value);
		m_scintilla->setEnabled(false);
		ui->radioClassic->toggle();
	}

	connect(this, &QDialog::accepted, this, &FilenameWindow::send);
}

FilenameWindow::~FilenameWindow()
{
	delete ui;
}

void FilenameWindow::validateClassicFilename(QString text)
{
	QString message;
	Filename fn(text);
	fn.isValid(m_profile, &message);
	ui->labelValidator->setText(message);

	text = text.replace("\\", "\\\\").replace("'", "\\'");

	static const QRegularExpression date("%date:format=([^%]+)%");
	int pos = 0;
	QRegularExpressionMatch match;
	while ((pos = text.indexOf(date, pos, &match)) != -1) {
		QString cap = match.captured(1);
		QString format;
		for (const QChar &c : cap) {
			if (c == 'Y') {
				format += "' + date.getFullYear() + '";
			} else if (c == 'M') {
				format += "' + date.getMonth() + '";
			} else if (c == 'd') {
				format += "' + date.getDate() + '";
			} else if (c == 'h') {
				format += "' + date.getHours() + '";
			} else if (c == 'm') {
				format += "' + date.getMinutes() + '";
			} else if (c == 's') {
				format += "' + date.getSeconds() + '";
			} else {
				format += c;
			}
		}

		text = text.left(pos) + format + text.mid(pos + match.capturedLength());
		pos += match.capturedLength();
	}

	static const QRegularExpression rxVariable("%([^%]+)%");
	QString value = "'" + text.replace(rxVariable, "' + \\1 + '").remove(" + '' + ").trimmed() + "'";
	if (value.startsWith("' + ")) {
		value = value.right(value.length() - 4);
	}
	if (value.startsWith("'' + ")) {
		value = value.right(value.length() - 5);
	}
	if (value.endsWith(" + '")) {
		value = value.left(value.length() - 4);
	}
	if (value.endsWith(" + ''")) {
		value = value.left(value.length() - 5);
	}

	m_scintilla->setText(value);
}

void FilenameWindow::helpClassic()
{
	QDesktopServices::openUrl(QUrl(QString(PROJECT_WEBSITE_URL) + "/docs/filename.html"));
}
void FilenameWindow::helpJavascript()
{
	QDesktopServices::openUrl(QUrl(QString(PROJECT_WEBSITE_URL) + "/docs/filename.html#javascript"));
}

QString FilenameWindow::format() const
{
	if (ui->radioJavascript->isChecked()) {
		#if defined(USE_QSCINTILLA)
			return "javascript:" + m_scintilla->text();
		#else
			return "javascript:" + m_scintilla->toPlainText();
		#endif
	}

	return ui->lineClassic->text();
}

void FilenameWindow::done(int r)
{
	QMap<QString, Site*> sites = m_profile->getSites();

	if (QDialog::Accepted == r && ui->radioJavascript->isChecked() && !sites.isEmpty()) {
		Site *site = sites.first();

		QMap<QString, QString> info;
		info.insert("tags_general", "general_1 general_2");
		info.insert("tags_artist", "artist_1 artist_2");
		info.insert("tags_model", "model_1 model_2");
		info.insert("tags_character", "character_1 character_2");
		info.insert("tags_copyright", "copyright_1 copyright_2");

		Image image(site, info, m_profile);
		QStringList det = image.paths(format(), QString(), 0);

		if (det.isEmpty()) {
			const int reply = QMessageBox::question(this, tr("Warning"), tr("You script contains error, are you sure you want to save it?"), QMessageBox::Yes | QMessageBox::Cancel);
			if (reply == QMessageBox::Cancel) {
				return;
			}
		}
	}

	QDialog::done(r);
}

void FilenameWindow::send()
{
	emit validated(format());
}
