#include <bdn/ui/UIApplicationController.h>
#include <bdn/ui/View.h>
#include <bdn/ui/ViewCoreFactory.h>

#include <utility>

namespace bdn::ui
{
    bool &View::debugViewEnabled()
    {
        static bool s_debugViewEnabled = false;
        return s_debugViewEnabled;
    }

    bool &View::debugViewBaselineEnabled()
    {
        static bool s_debugViewBaselineEnabled = false;
        return s_debugViewBaselineEnabled;
    }

    View::View(std::shared_ptr<ViewCoreFactory> viewCoreFactory)
        : _viewCoreFactory(viewCoreFactory ? std::move(viewCoreFactory)
                                           : UIApplicationController::ViewCoreFactoryStack::top())
    {
        if (!_viewCoreFactory) {
            throw std::runtime_error("Couldn't get ViewCore Factory!");
        }

        stylesheet.onChange() += [=](auto &property) {
            updateFromStylesheet();
            if (auto layout = _layout.get()) {
                layout->updateStylesheet(this);
            }
        };

        isLayoutRoot.onChange() += [=](auto) { updateLayout(_layout.get(), _layout.get()); };

        visible.onChange() += [=](auto) {
            if (auto layout = _layout.get()) {
                layout->updateStylesheet(this);
            }
        };

        registerCoreCreatingProperties(this, &visible, &geometry, &stylesheet);
    }

    View::~View()
    {
        if (auto layout = _layout.get()) {
            layout->unregisterView(this);
        }
    }

    void View::setLayout(std::shared_ptr<Layout> layout)
    {
        auto oldLayout = _layout.set(std::move(layout));
        updateLayout(oldLayout, _layout.get());
    }

    void View::setFallbackLayout(std::shared_ptr<Layout> layout)
    {
        auto oldLayout = _layout.setFallback(std::move(layout));

        updateLayout(oldLayout, _layout.get());
    }

    void View::updateLayout(const std::shared_ptr<Layout> &oldLayout, const std::shared_ptr<Layout> &newLayout)
    {
        if (oldLayout == newLayout) {
            return;
        }

        if (oldLayout) {
            oldLayout->unregisterView(this);
        }

        if (newLayout) {
            newLayout->registerView(this);
        }

        if (oldLayout != newLayout) {
            viewCore()->setLayout(newLayout);
        }

        for (const auto &child : childViews()) {
            child->setFallbackLayout(newLayout);
        }
    }

    std::shared_ptr<Layout> View::getLayout() { return _layout.get(); }

    void View::updateFromStylesheet()
    {
        if (auto core = viewCore()) {
            if (stylesheet->count("background-color")) {
                core->backgroundColor = stylesheet->at("background-color");
            } else {
                core->backgroundColor = std::nullopt;
            }

            core->updateFromStylesheet(stylesheet.get());
        }
    }

    void View::scheduleLayout()
    {
        if (auto core = viewCore()) {
            core->scheduleLayout();
        } else {
            _hasLayoutSchedulePending = true;
        }
    }

    const std::type_info &View::typeInfoForCoreCreation() const { return typeid(*this); }

    void View::lazyInitCore() const
    {
        if (_core) {
            return;
        }

        // NOLINTNEXTLINE(cppcoreguidelines-pro-type-const-cast)
        auto un_const_this = const_cast<View *>(this);

        _core = _viewCoreFactory->createViewCore(typeInfoForCoreCreation());
        un_const_this->bindViewCore();
    }

    std::shared_ptr<View::Core> View::viewCore()
    {
        lazyInitCore();
        return _core;
    }

    std::shared_ptr<View::Core> View::viewCore() const
    {
        // NOLINTNEXTLINE(cppcoreguidelines-pro-type-const-cast)
        return const_cast<View *>(this)->viewCore();
    }

    void View::bindViewCore()
    {
        viewCore()->visible.bind(visible);
        viewCore()->geometry.bind(geometry);

        _layoutCallbackReceiver = viewCore()->_layoutCallback.set([=]() { onCoreLayout(); });
        _dirtyCallbackReceiver = viewCore()->_dirtyCallback.set([=]() { onCoreDirty(); });
    }

    void View::setParentViewOfView(const std::shared_ptr<View> &view, const std::shared_ptr<View> &parentView)
    {
        assert(view->parentView.get().lock() == nullptr || parentView == nullptr);

        view->internalParentView = parentView;

        if (parentView) {
            view->setFallbackLayout(parentView->getLayout());
        } else {
            view->setFallbackLayout(nullptr);
        }
    }

    void View::onCoreLayout()
    {
        if (auto layout = getLayout()) {
            layout->layout(this);
        }
    }

    void View::onCoreDirty()
    {
        if (auto layout = getLayout()) {
            layout->markDirty(this);
        }
    }

    Size View::sizeForSpace(Size availableSpace) const { return viewCore()->sizeForSpace(availableSpace); }

    float View::baseline(Size forSize) const { return viewCore()->baseline(forSize); }

    float View::pointScaleFactor() const { return viewCore()->pointScaleFactor(); }

    View::Core::Core(std::shared_ptr<ViewCoreFactory> viewCoreFactory) : _viewCoreFactory(std::move(viewCoreFactory)) {}
}
