﻿using System;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Identity;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using RolebaseAuthorization.Data;

namespace RolebaseAuthorization {
    public class Startup {
        public Startup (IConfiguration configuration) {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices (IServiceCollection services) {
            services.Configure<CookiePolicyOptions> (options => {
                // This lambda determines whether user consent for non-essential cookies is needed for a given request.
                options.CheckConsentNeeded = context => true;
                options.MinimumSameSitePolicy = SameSiteMode.None;
            });

            services.AddDbContext<ApplicationDbContext> (options =>
                options.UseSqlServer (
                    Configuration.GetConnectionString ("DefaultConnection")));

            services.AddIdentity<IdentityUser, IdentityRole> ()
                .AddEntityFrameworkStores<ApplicationDbContext> ()
                .AddDefaultTokenProviders ();

            services.AddMvc ().AddRazorPagesOptions (options => {
                options.Conventions.AuthorizePage ("/test1", "OnlyAdminAccess");

            }).SetCompatibilityVersion (CompatibilityVersion.Version_2_1);

            services.AddAuthorization (options => {
                options.AddPolicy ("OnlyAdminAccess", policy => policy.RequireRole ("Admin"));
            });
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure (IApplicationBuilder app, IHostingEnvironment env, IServiceProvider serviceProvider) {
            if (env.IsDevelopment ()) {
                app.UseDeveloperExceptionPage ();
                app.UseDatabaseErrorPage ();
            } else {
                app.UseExceptionHandler ("/Home/Error");
                app.UseHsts ();
            }

            app.UseHttpsRedirection ();
            app.UseStaticFiles ();
            app.UseCookiePolicy ();

            app.UseAuthentication ();
            app.UseMvc (routes => {
                routes.MapRoute (
                    name: "default",
                    template: "{controller=Home}/{action=Index}/{id?}");
            });

            CreateRoles (serviceProvider).Wait ();
        }

        private async Task CreateRoles (IServiceProvider serviceProvider) {
            //initializing custom roles 
            var RoleManager = serviceProvider.GetRequiredService<RoleManager<IdentityRole>> ();
            var UserManager = serviceProvider.GetRequiredService<UserManager<IdentityUser>> ();
            string[] roleNames = { "Admin", "User", "HR" };
            IdentityResult roleResult;

            foreach (var roleName in roleNames) {
                var roleExist = await RoleManager.RoleExistsAsync (roleName);
                if (!roleExist) {
                    //create the roles and seed them to the database: Question 1
                    roleResult = await RoleManager.CreateAsync (new IdentityRole (roleName));
                }
            }

            IdentityUser user = await UserManager.FindByEmailAsync ("Amine@Amine.com.com");

            if (user == null) {
                user = new IdentityUser () {
                UserName = "Amine@Amine.com.com",
                Email = "Amine@Amine.com.com",
                };
                await UserManager.CreateAsync (user, "Test@123");
            }
            await UserManager.AddToRoleAsync (user, "Admin");

            IdentityUser user1 = await UserManager.FindByEmailAsync ("User@Amine.com");

            if (user1 == null) {
                user1 = new IdentityUser () {
                UserName = "User@Amine.com",
                Email = "User@Amine.com",
                };
                await UserManager.CreateAsync (user1, "Test@123");
            }
            await UserManager.AddToRoleAsync (user1, "User");

            IdentityUser user2 = await UserManager.FindByEmailAsync ("HR@Amine.com");

            if (user2 == null) {
                user2 = new IdentityUser () {
                UserName = "HR@Amine.com",
                Email = "HR@Amine.com",
                };
                await UserManager.CreateAsync (user2, "Test@123");
            }
            await UserManager.AddToRoleAsync (user2, "HR");

        }
    }
}