﻿using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Diagnostics;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.ApiExplorer;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Microsoft.AspNetCore.Mvc.Routing;
using Microsoft.AspNetCore.Mvc.Versioning;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;
using MeklaAPI.Entities;
using MeklaAPI.MeklaCRUD;
using MeklaAPI.Repositories;
using MeklaAPI.Services;
using Swashbuckle.AspNetCore.Swagger;

namespace MeklaAPI.Swagger {
    public class Startup {
        public Startup (IConfiguration configuration) {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices (IServiceCollection services) {
            services.AddOptions ();
            services.AddDbContext<MeklaDbContext> (opt => opt.UseInMemoryDatabase ("MeklaDatabase"));
            services.AddCors (options => {
                options.AddPolicy ("AllowAllOrigins",
                    builder => {
                        builder
                            .AllowAnyOrigin ()
                            .AllowAnyHeader ()
                            .AllowAnyMethod ();
                    });
            });

            services.AddSingleton<ISeedDataService, SeedDataService> ();
            services.AddScoped<IMeklaRepository, EfMeklaRepository> ();
            services.AddRouting (options => options.LowercaseUrls = true);
            services.AddSingleton<IActionContextAccessor, ActionContextAccessor> ();
            services.AddScoped<IUrlHelper> (implementationFactory => {
                var actionContext = implementationFactory.GetService<IActionContextAccessor> ().ActionContext;
                return new UrlHelper (actionContext);
            });

            services.AddMvcCore ().AddVersionedApiExplorer (o => o.GroupNameFormat = "'v'VVV");
            services.AddMvc ().SetCompatibilityVersion (CompatibilityVersion.Version_2_1);
            services.AddApiVersioning (config => {
                config.ReportApiVersions = true;
                config.AssumeDefaultVersionWhenUnspecified = true;
                config.DefaultApiVersion = new ApiVersion (1, 0);
                config.ApiVersionReader = new HeaderApiVersionReader ("api-version");
            });
            services.AddSwaggerGen (
                options => {
                    var provider = services.BuildServiceProvider ()
                        .GetRequiredService<IApiVersionDescriptionProvider> ();

                    foreach (var description in provider.ApiVersionDescriptions) {
                        options.SwaggerDoc (
                            description.GroupName,
                            new Info () {
                                Title = $"Sample API {description.ApiVersion}",
                                    Version = description.ApiVersion.ToString ()
                            });
                    }
                });
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure (IApplicationBuilder app, ILoggerFactory loggerFactory,
            IHostingEnvironment env, IApiVersionDescriptionProvider provider) {
            loggerFactory.AddConsole ();

            if (env.IsDevelopment ()) {
                app.UseDeveloperExceptionPage ();
            } else {
                app.UseHsts ();
                app.UseExceptionHandler (errorApp => {
                    errorApp.Run (async context => {
                        context.Response.StatusCode = 500;
                        context.Response.ContentType = "text/plain";
                        var errorFeature = context.Features.Get<IExceptionHandlerFeature> ();
                        if (errorFeature != null) {
                            var logger = loggerFactory.CreateLogger ("Global exception logger");
                            logger.LogError (500, errorFeature.Error, errorFeature.Error.Message);
                        }

                        await context.Response.WriteAsync ("There was an error");
                    });
                });
            }

            //app.AddSeedData();

            app.UseHttpsRedirection ();

            app.UseSwagger ();
            app.UseSwaggerUI (
                options => {
                    foreach (var description in provider.ApiVersionDescriptions) {
                        options.SwaggerEndpoint (
                            $"/swagger/{description.GroupName}/swagger.json",
                            description.GroupName.ToUpperInvariant ());
                    }
                });

            app.UseCors ("AllowAllOrigins");
            AutoMapper.Mapper.Initialize (mapper => {
                mapper.CreateMap<Mekla, Mekla> ().ReverseMap ();
                mapper.CreateMap<Mekla, MeklaUpdate> ().ReverseMap ();
                mapper.CreateMap<Mekla, MeklaCreate> ().ReverseMap ();
            });

            app.UseMvc ();
        }
    }
}