unit Neslib.LibYaml;
{ Header translation for LibYaml (https://github.com/yaml/libyaml).
  This unit is partially generated by Chet (https://github.com/neslib/Chet). }

{$MINENUMSIZE 4}
{$IF Defined(MACOS) or Defined(ANDROID)}
  {$DEFINE USE_LIB}
{$ENDIF}

interface

{$IF Defined(WIN32)}
const
  _PU = '_';

{$LINK 'Lib\Windows32\dumper.obj'}
{$LINK 'Lib\Windows32\loader.obj'}
{$LINK 'Lib\Windows32\emitter.obj'}
{$LINK 'Lib\Windows32\writer.obj'}
{$LINK 'Lib\Windows32\parser.obj'}
{$LINK 'Lib\Windows32\scanner.obj'}
{$LINK 'Lib\Windows32\reader.obj'}
{$LINK 'Lib\Windows32\api.obj'}

{$ELSEIF Defined(WIN64)}
const
  _PU = '';

{$LINK 'Lib\Windows64\dumper.obj'}
{$LINK 'Lib\Windows64\loader.obj'}
{$LINK 'Lib\Windows64\emitter.obj'}
{$LINK 'Lib\Windows64\writer.obj'}
{$LINK 'Lib\Windows64\parser.obj'}
{$LINK 'Lib\Windows64\scanner.obj'}
{$LINK 'Lib\Windows64\reader.obj'}
{$LINK 'Lib\Windows64\api.obj'}

{$ELSEIF Defined(ANDROID32)}
const
  LIB_YAML = 'libyaml_android32.a';
  _PU = '';
{$ELSEIF Defined(ANDROID64)}
const
  LIB_YAML = 'libyaml_android64.a';
  _PU = '';
{$ELSEIF Defined(IOS)}
const
  LIB_YAML = 'libyaml_ios.a';
  _PU = '';
{$ELSEIF Defined(MACOS32)}
const
  LIB_YAML = 'libyaml_mac32.dylib';
  _PU = '_';
{$ELSEIF Defined(MACOS64)}
const
  LIB_YAML = 'libyaml_mac64.a';
  _PU = '';
{$ELSE}
  {$MESSAGE Error 'Unsupported platform'}
{$ENDIF}

(**
 * @defgroup version Version Information
 * @{
 *)

(**
 * Get the library version as a string.
 *
 * @returns The function returns the pointer to a static string of the form
 * @c "X.Y.Z", where @c X is the major version number, @c Y is a minor version
 * number, and @c Z is the patch version number.
 *)

function yaml_get_version_string(): PUTF8Char; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_get_version_string';

(**
 * Get the library version numbers.
 *
 * @param[out]      major   Major version number.
 * @param[out]      minor   Minor version number.
 * @param[out]      patch   Patch version number.
 *)
procedure yaml_get_version(out major, minor, patch: Integer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_get_version';

(**
 * @defgroup basic Basic Types
 * @{
 *)

(** The character type (UTF-8 octet). *)
type
  yaml_char_t = UTF8Char;
  Pyaml_char_t = PUTF8Char;

(** The version directive data. *)
type
  yaml_version_directive_t = record
    (** The major version number. *)
    major: Integer;
    (** The minor version number. *)
    minor: Integer;
  end;
  Pyaml_version_directive_t = ^yaml_version_directive_t;

(** The tag directive data. *)
type
  yaml_tag_directive_t = record
    (** The tag handle. *)
    handle: Pyaml_char_t;
    (** The tag prefix. *)
    prefix: Pyaml_char_t;
  end;
  Pyaml_tag_directive_t = ^yaml_tag_directive_t;

(** The stream encoding. *)
type
  yaml_encoding_t = (
    (** Let the parser choose the encoding. *)
    YAML_ANY_ENCODING,
    (** The default UTF-8 encoding. *)
    YAML_UTF8_ENCODING,
    (** The UTF-16-LE encoding with BOM. *)
    YAML_UTF16LE_ENCODING,
    (** The UTF-16-BE encoding with BOM. *)
    YAML_UTF16BE_ENCODING);

(** Line break types. *)
type
  yaml_break_t = (
    (** Let the parser choose the break type. *)
    YAML_ANY_BREAK,
    (** Use CR for line breaks (Mac style). *)
    YAML_CR_BREAK,
    (** Use LN for line breaks (Unix style). *)
    YAML_LN_BREAK,
    (** Use CR LN for line breaks (DOS style). *)
    YAML_CRLN_BREAK);

(** Many bad things could happen with the parser and emitter. *)
type
  yaml_error_type_t = (
    (** No error is produced. *)
    YAML_NO_ERROR,

    (** Cannot allocate or reallocate a block of memory. *)
    YAML_MEMORY_ERROR,

    (** Cannot read or decode the input stream. *)
    YAML_READER_ERROR,
    (** Cannot scan the input stream. *)
    YAML_SCANNER_ERROR,
    (** Cannot parse the input stream. *)
    YAML_PARSER_ERROR,
    (** Cannot compose a YAML document. *)
    YAML_COMPOSER_ERROR,

    (** Cannot write to the output stream. *)
    YAML_WRITER_ERROR,
    (** Cannot emit a YAML stream. *)
    YAML_EMITTER_ERROR);

(** The pointer position. *)
type
  yaml_mark_t = record
    (** The position index. *)
    index: NativeInt;

    (** The position line. *)
    line: NativeInt;

    (** The position column. *)
    column: NativeInt;
  end;
  Pyaml_mark_t = ^yaml_mark_t;

(**
 * @defgroup styles Node Styles
 * @{
 *)

(** Scalar styles. *)
type
  yaml_scalar_style_t = (
    (** Let the emitter choose the style. *)
    YAML_ANY_SCALAR_STYLE,

    (** The plain scalar style. *)
    YAML_PLAIN_SCALAR_STYLE,

    (** The single-quoted scalar style. *)
    YAML_SINGLE_QUOTED_SCALAR_STYLE,
    (** The double-quoted scalar style. *)
    YAML_DOUBLE_QUOTED_SCALAR_STYLE,

    (** The literal scalar style. *)
    YAML_LITERAL_SCALAR_STYLE,
    (** The folded scalar style. *)
    YAML_FOLDED_SCALAR_STYLE);

(** Sequence styles. *)
type
  yaml_sequence_style_t = (
    (** Let the emitter choose the style. *)
    YAML_ANY_SEQUENCE_STYLE,

    (** The block sequence style. *)
    YAML_BLOCK_SEQUENCE_STYLE,
    (** The flow sequence style. *)
    YAML_FLOW_SEQUENCE_STYLE);

(** Mapping styles. *)
type
  yaml_mapping_style_t = (
    (** Let the emitter choose the style. *)
    YAML_ANY_MAPPING_STYLE,

    (** The block mapping style. *)
    YAML_BLOCK_MAPPING_STYLE,
    (** The flow mapping style. *)
    YAML_FLOW_MAPPING_STYLE);

(**
 * @defgroup tokens Tokens
 * @{
 *)

(** Token types. *)
type
  yaml_token_type_t = (
    (** An empty token. *)
    YAML_NO_TOKEN,

    (** A STREAM-START token. *)
    YAML_STREAM_START_TOKEN,
    (** A STREAM-END token. *)
    YAML_STREAM_END_TOKEN,

    (** A VERSION-DIRECTIVE token. *)
    YAML_VERSION_DIRECTIVE_TOKEN,
    (** A TAG-DIRECTIVE token. *)
    YAML_TAG_DIRECTIVE_TOKEN,
    (** A DOCUMENT-START token. *)
    YAML_DOCUMENT_START_TOKEN,
    (** A DOCUMENT-END token. *)
    YAML_DOCUMENT_END_TOKEN,

    (** A BLOCK-SEQUENCE-START token. *)
    YAML_BLOCK_SEQUENCE_START_TOKEN,
    (** A BLOCK-MAPPING-START token. *)
    YAML_BLOCK_MAPPING_START_TOKEN,
    (** A BLOCK-END token. *)
    YAML_BLOCK_END_TOKEN,

    (** A FLOW-SEQUENCE-START token. *)
    YAML_FLOW_SEQUENCE_START_TOKEN,
    (** A FLOW-SEQUENCE-END token. *)
    YAML_FLOW_SEQUENCE_END_TOKEN,
    (** A FLOW-MAPPING-START token. *)
    YAML_FLOW_MAPPING_START_TOKEN,
    (** A FLOW-MAPPING-END token. *)
    YAML_FLOW_MAPPING_END_TOKEN,

    (** A BLOCK-ENTRY token. *)
    YAML_BLOCK_ENTRY_TOKEN,
    (** A FLOW-ENTRY token. *)
    YAML_FLOW_ENTRY_TOKEN,
    (** A KEY token. *)
    YAML_KEY_TOKEN,
    (** A VALUE token. *)
    YAML_VALUE_TOKEN,

    (** An ALIAS token. *)
    YAML_ALIAS_TOKEN,
    (** An ANCHOR token. *)
    YAML_ANCHOR_TOKEN,
    (** A TAG token. *)
    YAML_TAG_TOKEN,
    (** A SCALAR token. *)
    YAML_SCALAR_TOKEN);

(** The token structure. *)
type
  yaml_token_t = record
    (** The token type. *)
    _type: yaml_token_type_t;

    (** The token data. *)
    data: record
      case Byte of
        (** The stream start (for @c YAML_STREAM_START_TOKEN). *)
        0: (stream_start: record
              (** The stream encoding. *)
              encoding: yaml_encoding_t;
            end);

        (** The alias (for @c YAML_ALIAS_TOKEN). *)
        1: (alias: record
              (** The alias value. *)
              value: Pyaml_char_t;
            end);

        (** The anchor (for @c YAML_ANCHOR_TOKEN). *)
        2: (anchor: record
              (** The anchor value. *)
              value: Pyaml_char_t;
            end);

        (** The tag (for @c YAML_TAG_TOKEN). *)
        3: (tag: record
              (** The tag handle. *)
              handle: Pyaml_char_t;
              (** The tag suffix. *)
              suffix: Pyaml_char_t;
            end);

        (** The scalar value (for @c YAML_SCALAR_TOKEN). *)
        4: (scalar: record
              (** The scalar value. *)
              value: Pyaml_char_t;
              (** The length of the scalar value. *)
              length: NativeInt;
              (** The scalar style. *)
              style: yaml_scalar_style_t;
            end);

        (** The version directive (for @c YAML_VERSION_DIRECTIVE_TOKEN). *)
        5: (version_directive: record
              (** The major version number. *)
              major: Integer;
              (** The minor version number. *)
              minor: Integer;
            end);

        (** The tag directive (for @c YAML_TAG_DIRECTIVE_TOKEN). *)
        6: (tag_directive: record
              (** The tag handle. *)
              handle: Pyaml_char_t;
              (** The tag prefix. *)
              prefix: Pyaml_char_t;
            end);
    end;

    (** The beginning of the token. *)
    start_mark: yaml_mark_t;
    (** The end of the token. *)
    end_mark: yaml_mark_t;
  end;
  Pyaml_token_t = ^yaml_token_t;

(**
 * Free any memory allocated for a token object.
 *
 * @param[in,out]   token   A token object.
 *)
procedure yaml_token_delete(token: Pyaml_token_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_token_delete';

(**
 * @defgroup events Events
 * @{
 *)

(** Event types. *)
type
  yaml_event_type_t = (
    (** An empty event. *)
    YAML_NO_EVENT,

    (** A STREAM-START event. *)
    YAML_STREAM_START_EVENT,
    (** A STREAM-END event. *)
    YAML_STREAM_END_EVENT,

    (** A DOCUMENT-START event. *)
    YAML_DOCUMENT_START_EVENT,
    (** A DOCUMENT-END event. *)
    YAML_DOCUMENT_END_EVENT,

    (** An ALIAS event. *)
    YAML_ALIAS_EVENT,
    (** A SCALAR event. *)
    YAML_SCALAR_EVENT,

    (** A SEQUENCE-START event. *)
    YAML_SEQUENCE_START_EVENT,
    (** A SEQUENCE-END event. *)
    YAML_SEQUENCE_END_EVENT,

    (** A MAPPING-START event. *)
    YAML_MAPPING_START_EVENT,
    (** A MAPPING-END event. *)
    YAML_MAPPING_END_EVENT);

type
  yaml_document_start_event_t = record
    (** The version directive. *)
    version_directive: Pyaml_version_directive_t;

    (** The list of tag directives. *)
    tag_directives: record
      (** The beginning of the tag directives list. *)
      start: Pyaml_tag_directive_t;
      (** The end of the tag directives list. *)
      _end: Pyaml_tag_directive_t;
    end;

    (** Is the document indicator implicit? *)
    implicit: Integer;
  end;

type
  yaml_scalar_event_t = record
    (** The anchor. *)
    anchor: Pyaml_char_t;
    (** The tag. *)
    tag: Pyaml_char_t;
    (** The scalar value. *)
    value: Pyaml_char_t;
    (** The length of the scalar value. *)
    length: NativeInt;
    (** Is the tag optional for the plain style? *)
    plain_implicit: Integer;
    (** Is the tag optional for any non-plain style? *)
    quoted_implicit: Integer;
    (** The scalar style. *)
    style: yaml_scalar_style_t;
  end;

type
  yaml_mapping_start_event_t = record
    (** The anchor. *)
    anchor: Pyaml_char_t;
    (** The tag. *)
    tag: Pyaml_char_t;
    (** Is the tag optional? *)
    implicit: Integer;
    (** The mapping style. *)
    style: yaml_mapping_style_t;
  end;

type
  yaml_sequence_start_event_t = record
    (** The anchor. *)
    anchor: Pyaml_char_t;
    (** The tag. *)
    tag: Pyaml_char_t;
    (** Is the tag optional? *)
    implicit: Integer;
    (** The sequence style. *)
    style: yaml_sequence_style_t;
  end;

(** The event structure. *)
type
  yaml_event_t = record
    (** The event type. *)
    _type: yaml_event_type_t;

    (** The event data. *)
    data: record
      case Byte of
        (** The stream parameters (for @c YAML_STREAM_START_EVENT). *)
        0: (stream_start: record
              (** The document encoding. *)
              encoding: yaml_encoding_t;
            end);

        (** The document parameters (for @c YAML_DOCUMENT_START_EVENT). *)
        1: (document_start: yaml_document_start_event_t);

        (** The document end parameters (for @c YAML_DOCUMENT_END_EVENT). *)
        3: (document_end: record
              (** Is the document end indicator implicit? *)
              implicit: Integer;
            end);

        (** The alias parameters (for @c YAML_ALIAS_EVENT). *)
        4: (alias: record
              (** The anchor. *)
              anchor: Pyaml_char_t;
            end);

        (** The scalar parameters (for @c YAML_SCALAR_EVENT). *)
        5: (scalar: yaml_scalar_event_t);

        (** The sequence parameters (for @c YAML_SEQUENCE_START_EVENT). *)
        6: (sequence_start: yaml_sequence_start_event_t);

        (** The mapping parameters (for @c YAML_MAPPING_START_EVENT). *)
        7: (mapping_start: yaml_mapping_start_event_t);
      end;

    (** The beginning of the event. *)
    start_mark: yaml_mark_t;
    (** The end of the event. *)
    end_mark: yaml_mark_t;
  end;
  Pyaml_event_t = ^yaml_event_t;

(**
 * Create the STREAM-START event.
 *
 * @param[out]      event       An empty event object.
 * @param[in]       encoding    The stream encoding.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_stream_start_event_initialize(event: Pyaml_event_t; encoding: yaml_encoding_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_stream_start_event_initialize';

(**
 * Create the STREAM-END event.
 *
 * @param[out]      event       An empty event object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_stream_end_event_initialize(event: Pyaml_event_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_stream_end_event_initialize';

(**
 * Create the DOCUMENT-START event.
 *
 * The @a implicit argument is considered as a stylistic parameter and may be
 * ignored by the emitter.
 *
 * @param[out]      event                   An empty event object.
 * @param[in]       version_directive       The %YAML directive value or
 *                                          @c NULL.
 * @param[in]       tag_directives_start    The beginning of the %TAG
 *                                          directives list.
 * @param[in]       tag_directives_end      The end of the %TAG directives
 *                                          list.
 * @param[in]       implicit                If the document start indicator is
 *                                          implicit.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_document_start_event_initialize(event: Pyaml_event_t; version_directive: Pyaml_version_directive_t; tag_directives_start: Pyaml_tag_directive_t; tag_directives_end: Pyaml_tag_directive_t; implicit: Integer): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_start_event_initialize';

(**
 * Create the DOCUMENT-END event.
 *
 * The @a implicit argument is considered as a stylistic parameter and may be
 * ignored by the emitter.
 *
 * @param[out]      event       An empty event object.
 * @param[in]       implicit    If the document end indicator is implicit.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_document_end_event_initialize(event: Pyaml_event_t; implicit: Integer): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_end_event_initialize';

(**
 * Create an ALIAS event.
 *
 * @param[out]      event       An empty event object.
 * @param[in]       anchor      The anchor value.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_alias_event_initialize(event: Pyaml_event_t; anchor: Pyaml_char_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_alias_event_initialize';

(**
 * Create a SCALAR event.
 *
 * The @a style argument may be ignored by the emitter.
 *
 * Either the @a tag attribute or one of the @a plain_implicit and
 * @a quoted_implicit flags must be set.
 *
 * @param[out]      event           An empty event object.
 * @param[in]       anchor          The scalar anchor or @c NULL.
 * @param[in]       tag             The scalar tag or @c NULL.
 * @param[in]       value           The scalar value.
 * @param[in]       length          The length of the scalar value.
 * @param[in]       plain_implicit  If the tag may be omitted for the plain
 *                                  style.
 * @param[in]       quoted_implicit If the tag may be omitted for any
 *                                  non-plain style.
 * @param[in]       style           The scalar style.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_scalar_event_initialize(event: Pyaml_event_t; anchor: Pyaml_char_t; tag: Pyaml_char_t; value: Pyaml_char_t; length: Integer; plain_implicit: Integer; quoted_implicit: Integer; style: yaml_scalar_style_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_scalar_event_initialize';

(**
 * Create a SEQUENCE-START event.
 *
 * The @a style argument may be ignored by the emitter.
 *
 * Either the @a tag attribute or the @a implicit flag must be set.
 *
 * @param[out]      event       An empty event object.
 * @param[in]       anchor      The sequence anchor or @c NULL.
 * @param[in]       tag         The sequence tag or @c NULL.
 * @param[in]       implicit    If the tag may be omitted.
 * @param[in]       style       The sequence style.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_sequence_start_event_initialize(event: Pyaml_event_t; anchor: Pyaml_char_t; tag: Pyaml_char_t; implicit: Integer; style: yaml_sequence_style_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_sequence_start_event_initialize';

(**
 * Create a SEQUENCE-END event.
 *
 * @param[out]      event       An empty event object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_sequence_end_event_initialize(event: Pyaml_event_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_sequence_end_event_initialize';

(**
 * Create a MAPPING-START event.
 *
 * The @a style argument may be ignored by the emitter.
 *
 * Either the @a tag attribute or the @a implicit flag must be set.
 *
 * @param[out]      event       An empty event object.
 * @param[in]       anchor      The mapping anchor or @c NULL.
 * @param[in]       tag         The mapping tag or @c NULL.
 * @param[in]       implicit    If the tag may be omitted.
 * @param[in]       style       The mapping style.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_mapping_start_event_initialize(event: Pyaml_event_t; anchor: Pyaml_char_t; tag: Pyaml_char_t; implicit: Integer; style: yaml_mapping_style_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_mapping_start_event_initialize';

(**
 * Create a MAPPING-END event.
 *
 * @param[out]      event       An empty event object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_mapping_end_event_initialize(event: Pyaml_event_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_mapping_end_event_initialize';

(**
 * Free any memory allocated for an event object.
 *
 * @param[in,out]   event   An event object.
 *)
procedure yaml_event_delete(event: Pyaml_event_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_event_delete';

(**
 * @defgroup nodes Nodes
 * @{
 *)

const
  (** The tag @c !!null with the only possible value: @c null. *)
  YAML_NULL_TAG       = 'tag:yaml.org,2002:null';
  (** The tag @c !!bool with the values: @c true and @c false. *)
  YAML_BOOL_TAG       = 'tag:yaml.org,2002:bool';
  (** The tag @c !!str for string values. *)
  YAML_STR_TAG        = 'tag:yaml.org,2002:str';
  (** The tag @c !!int for integer values. *)
  YAML_INT_TAG        = 'tag:yaml.org,2002:int';
  (** The tag @c !!float for float values. *)
  YAML_FLOAT_TAG      = 'tag:yaml.org,2002:float';
  (** The tag @c !!timestamp for date and time values. *)
  YAML_TIMESTAMP_TAG  = 'tag:yaml.org,2002:timestamp';

  (** The tag @c !!seq is used to denote sequences. *)
  YAML_SEQ_TAG        = 'tag:yaml.org,2002:seq';
  (** The tag @c !!map is used to denote mapping. *)
  YAML_MAP_TAG        = 'tag:yaml.org,2002:map';

  (** The default scalar tag is @c !!str. *)
  YAML_DEFAULT_SCALAR_TAG     = YAML_STR_TAG;
  (** The default sequence tag is @c !!seq. *)
  YAML_DEFAULT_SEQUENCE_TAG   = YAML_SEQ_TAG;
  (** The default mapping tag is @c !!map. *)
  YAML_DEFAULT_MAPPING_TAG    = YAML_MAP_TAG;

(** Node types. *)
type
  yaml_node_type_t = (
    (** An empty node. *)
    YAML_NO_NODE,

    (** A scalar node. *)
    YAML_SCALAR_NODE,
    (** A sequence node. *)
    YAML_SEQUENCE_NODE,
    (** A mapping node. *)
    YAML_MAPPING_NODE);

(** An element of a sequence node. *)
type
  yaml_node_item_t = Integer;
  Pyaml_node_item_t = PInteger;

(** An element of a mapping node. *)
type
  yaml_node_pair_t = record
    (** The key of the element. *)
    key: Integer;
    (** The value of the element. *)
    value: Integer;
  end;
  Pyaml_node_pair_t = ^yaml_node_pair_t;

(** The node structure. *)
type
  yaml_node_t = record
    (** The node type. *)
    _type: yaml_node_type_t;

    (** The node tag. *)
    tag: Pyaml_char_t;

    (** The node data. *)
    data: record
      case Byte of
        (** The scalar parameters (for @c YAML_SCALAR_NODE). *)
        0: (scalar: record
              (** The scalar value. *)
              value: Pyaml_char_t;
              (** The length of the scalar value. *)
              length: NativeInt;
              (** The scalar style. *)
              style: yaml_scalar_style_t;
            end);

        (** The sequence parameters (for @c YAML_SEQUENCE_NODE). *)
        1: (sequence: record
              (** The stack of sequence items. *)
              items: record
                (** The beginning of the stack. *)
                start: Pyaml_node_item_t;
                (** The end of the stack. *)
                _end: Pyaml_node_item_t;
                (** The top of the stack. *)
                top: Pyaml_node_item_t;
              end;
              (** The sequence style. *)
              style: yaml_sequence_style_t;
            end);

        (** The mapping parameters (for @c YAML_MAPPING_NODE). *)
        2: (mapping: record
              (** The stack of mapping pairs (key, value). *)
              pairs: record
                (** The beginning of the stack. *)
                start: Pyaml_node_pair_t;
                (** The end of the stack. *)
                _end: Pyaml_node_pair_t;
                (** The top of the stack. *)
                top: Pyaml_node_pair_t;
              end;
              (** The mapping style. *)
              style: yaml_mapping_style_t;
            end);
    end;

    (** The beginning of the node. *)
    start_marp: yaml_mark_t;
    (** The end of the node. *)
    end_mark: yaml_mark_t;
  end;
  Pyaml_node_t = ^yaml_node_t;

(** The document structure. *)
type
  yaml_document_t = record
    (** The document nodes. *)
    node: record
      (** The beginning of the stack. *)
      start: Pyaml_node_t;
      (** The end of the stack. *)
      _end: Pyaml_node_t;
      (** The top of the stack. *)
      top: Pyaml_node_t;
    end;

    (** The version directive. *)
    version_directive: Pyaml_version_directive_t;

    (** The list of tag directives. *)
    tag_directives: record
      (** The beginning of the tag directives list. *)
      start: Pyaml_tag_directive_t;
      (** The end of the tag directives list. *)
      _end: Pyaml_tag_directive_t;
    end;

    (** Is the document start indicator implicit? *)
    start_implicit: Integer;
    (** Is the document end indicator implicit? *)
    end_implicit: Integer;

    (** The beginning of the document. *)
    start_mark: yaml_mark_t;
    (** The end of the document. *)
    end_mark: yaml_mark_t;
  end;
  Pyaml_document_t = ^yaml_document_t;

(**
 * Create a YAML document.
 *
 * @param[out]      document                An empty document object.
 * @param[in]       version_directive       The %YAML directive value or
 *                                          @c NULL.
 * @param[in]       tag_directives_start    The beginning of the %TAG
 *                                          directives list.
 * @param[in]       tag_directives_end      The end of the %TAG directives
 *                                          list.
 * @param[in]       start_implicit          If the document start indicator is
 *                                          implicit.
 * @param[in]       end_implicit            If the document end indicator is
 *                                          implicit.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_document_initialize(document: Pyaml_document_t; version_directive: Pyaml_version_directive_t; tag_directives_start: Pyaml_tag_directive_t; tag_directives_end: Pyaml_tag_directive_t; start_implicit: Integer; end_implicit: Integer): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_initialize';

(**
 * Delete a YAML document and all its nodes.
 *
 * @param[in,out]   document        A document object.
 *)
procedure yaml_document_delete(document: Pyaml_document_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_delete';

(**
 * Get a node of a YAML document.
 *
 * The pointer returned by this function is valid until any of the functions
 * modifying the documents are called.
 *
 * @param[in]       document        A document object.
 * @param[in]       index           The node id.
 *
 * @returns the node objct or @c NULL if @c node_id is out of range.
 *)
function yaml_document_get_node(document: Pyaml_document_t; index: Integer): Pyaml_node_t; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_get_node';

(**
 * Get the root of a YAML document node.
 *
 * The root object is the first object added to the document.
 *
 * The pointer returned by this function is valid until any of the functions
 * modifying the documents are called.
 *
 * An empty document produced by the parser signifies the end of a YAML
 * stream.
 *
 * @param[in]       document        A document object.
 *
 * @returns the node object or @c NULL if the document is empty.
 *)
function yaml_document_get_root_node(document: Pyaml_document_t): Pyaml_node_t; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_get_root_node';

(**
 * Create a SCALAR node and attach it to the document.
 *
 * The @a style argument may be ignored by the emitter.
 *
 * @param[in,out]   document        A document object.
 * @param[in]       tag             The scalar tag.
 * @param[in]       value           The scalar value.
 * @param[in]       length          The length of the scalar value.
 * @param[in]       style           The scalar style.
 *
 * @returns the node id or @c 0 on error.
 *)
function yaml_document_add_scalar(document: Pyaml_document_t; tag: Pyaml_char_t; value: Pyaml_char_t; length: Integer; style: yaml_scalar_style_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_add_scalar';

(**
 * Create a SEQUENCE node and attach it to the document.
 *
 * The @a style argument may be ignored by the emitter.
 *
 * @param[in,out]   document    A document object.
 * @param[in]       tag         The sequence tag.
 * @param[in]       style       The sequence style.
 *
 * @returns the node id or @c 0 on error.
 *)
function yaml_document_add_sequence(document: Pyaml_document_t; tag: Pyaml_char_t; style: yaml_sequence_style_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_add_sequence';

(**
 * Create a MAPPING node and attach it to the document.
 *
 * The @a style argument may be ignored by the emitter.
 *
 * @param[in,out]   document    A document object.
 * @param[in]       tag         The sequence tag.
 * @param[in]       style       The sequence style.
 *
 * @returns the node id or @c 0 on error.
 *)
function yaml_document_add_mapping(document: Pyaml_document_t; tag: Pyaml_char_t; style: yaml_mapping_style_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_add_mapping';

(**
 * Add an item to a SEQUENCE node.
 *
 * @param[in,out]   document    A document object.
 * @param[in]       sequence    The sequence node id.
 * @param[in]       item        The item node id.
*
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_document_append_sequence_item(document: Pyaml_document_t; sequence: Integer; item: Integer): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_append_sequence_item';

(**
 * Add a pair of a key and a value to a MAPPING node.
 *
 * @param[in,out]   document    A document object.
 * @param[in]       mapping     The mapping node id.
 * @param[in]       key         The key node id.
 * @param[in]       value       The value node id.
*
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_document_append_mapping_pair(document: Pyaml_document_t; mapping: Integer; key: Integer; value: Integer): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_document_append_mapping_pair';

(**
 * @defgroup parser Parser Definitions
 * @{
 *)

(**
 * The prototype of a read handler.
 *
 * The read handler is called when the parser needs to read more bytes from the
 * source.  The handler should write not more than @a size bytes to the @a
 * buffer.  The number of written bytes should be set to the @a length variable.
 *
 * @param[in,out]   data        A pointer to an application data specified by
 *                              yaml_parser_set_input().
 * @param[out]      buffer      The buffer to write the data from the source.
 * @param[in]       size        The size of the buffer.
 * @param[out]      size_read   The actual number of bytes read from the source.
 *
 * @returns On success, the handler should return @c 1.  If the handler failed,
 * the returned value should be @c 0.  On EOF, the handler should set the
 * @a size_read to @c 0 and return @c 1.
 *)
type
  yaml_read_handler_t = function(data, buffer: Pointer; size: NativeUInt; size_read: PNativeUInt): Integer; cdecl;

(**
 * This structure holds information about a potential simple key.
 *)
type
  yaml_simple_key_t = record
    (** Is a simple key possible? *)
    possible: Integer;

    (** Is a simple key required? *)
    required: Integer;

    (** The number of the token. *)
    token_number: NativeInt;

    (** The position mark. *)
    mark: yaml_mark_t;
  end;
  Pyaml_simple_key_t = ^yaml_simple_key_t;

(**
 * The states of the parser.
 *)
type
  yaml_parser_state_t = (
    (** Expect STREAM-START. *)
    YAML_PARSE_STREAM_START_STATE,
    (** Expect the beginning of an implicit document. *)
    YAML_PARSE_IMPLICIT_DOCUMENT_START_STATE,
    (** Expect DOCUMENT-START. *)
    YAML_PARSE_DOCUMENT_START_STATE,
    (** Expect the content of a document. *)
    YAML_PARSE_DOCUMENT_CONTENT_STATE,
    (** Expect DOCUMENT-END. *)
    YAML_PARSE_DOCUMENT_END_STATE,
    (** Expect a block node. *)
    YAML_PARSE_BLOCK_NODE_STATE,
    (** Expect a block node or indentless sequence. *)
    YAML_PARSE_BLOCK_NODE_OR_INDENTLESS_SEQUENCE_STATE,
    (** Expect a flow node. *)
    YAML_PARSE_FLOW_NODE_STATE,
    (** Expect the first entry of a block sequence. *)
    YAML_PARSE_BLOCK_SEQUENCE_FIRST_ENTRY_STATE,
    (** Expect an entry of a block sequence. *)
    YAML_PARSE_BLOCK_SEQUENCE_ENTRY_STATE,
    (** Expect an entry of an indentless sequence. *)
    YAML_PARSE_INDENTLESS_SEQUENCE_ENTRY_STATE,
    (** Expect the first key of a block mapping. *)
    YAML_PARSE_BLOCK_MAPPING_FIRST_KEY_STATE,
    (** Expect a block mapping key. *)
    YAML_PARSE_BLOCK_MAPPING_KEY_STATE,
    (** Expect a block mapping value. *)
    YAML_PARSE_BLOCK_MAPPING_VALUE_STATE,
    (** Expect the first entry of a flow sequence. *)
    YAML_PARSE_FLOW_SEQUENCE_FIRST_ENTRY_STATE,
    (** Expect an entry of a flow sequence. *)
    YAML_PARSE_FLOW_SEQUENCE_ENTRY_STATE,
    (** Expect a key of an ordered mapping. *)
    YAML_PARSE_FLOW_SEQUENCE_ENTRY_MAPPING_KEY_STATE,
    (** Expect a value of an ordered mapping. *)
    YAML_PARSE_FLOW_SEQUENCE_ENTRY_MAPPING_VALUE_STATE,
    (** Expect the and of an ordered mapping entry. *)
    YAML_PARSE_FLOW_SEQUENCE_ENTRY_MAPPING_END_STATE,
    (** Expect the first key of a flow mapping. *)
    YAML_PARSE_FLOW_MAPPING_FIRST_KEY_STATE,
    (** Expect a key of a flow mapping. *)
    YAML_PARSE_FLOW_MAPPING_KEY_STATE,
    (** Expect a value of a flow mapping. *)
    YAML_PARSE_FLOW_MAPPING_VALUE_STATE,
    (** Expect an empty value of a flow mapping. *)
    YAML_PARSE_FLOW_MAPPING_EMPTY_VALUE_STATE,
    (** Expect nothing. *)
    YAML_PARSE_END_STATE);
  Pyaml_parser_state_t = ^yaml_parser_state_t;

(**
 * This structure holds aliases data.
 *)
type
  yaml_alias_data_t = record
    (** The anchor. *)
    anchor: Pyaml_char_t;
    (** The node id. *)
    index: Integer;
    (** The anchor mark. *)
    mark: yaml_mark_t;
  end;
  Pyaml_alias_data_t = ^yaml_alias_data_t;

(**
 * The parser structure.
 *
 * All members are internal.  Manage the structure using the @c yaml_parser_
 * family of functions.
 *)
type
  yaml_parser_t = record
    (**
     * @name Error handling
     * @{
     *)

    (** Error type. *)
    error: yaml_error_type_t;
    (** Error description. *)
    problem: PUTF8Char;
    (** The byte about which the problem occured. *)
    problem_offset: NativeInt;
    (** The problematic value (@c -1 is none). *)
    problem_value: Integer;
    (** The problem position. *)
    problem_mark: yaml_mark_t;
    (** The error context. *)
    context: PUTF8Char;
    (** The context position. *)
    context_mark: yaml_mark_t;

    (**
     * @name Reader stuff
     * @{
     *)

    (** Read handler. *)
    read_handler: yaml_read_handler_t;

    (** A pointer for passing to the read handler. *)
    read_handler_data: Pointer;

    (** Standard (string or file) input data. *)
    input: record
      (** String input data. *)
      _string: record
        (** The string start pointer. *)
        start: PUTF8Char;
        (** The string end pointer. *)
        _end: PUTF8Char;
        (** The string current position. *)
        current: PUTF8Char;
      end;

      (** File input data. *)
      _file: Pointer;
    end;

    (** EOF flag *)
    eof: Integer;

    (** The working buffer. *)
    buffer: record
      (** The beginning of the buffer. *)
      start: Pyaml_char_t;
      (** The end of the buffer. *)
      _end: Pyaml_char_t;
      (** The current position of the buffer. *)
      _pointer: Pyaml_char_t;
      (** The last filled position of the buffer. *)
      last: Pyaml_char_t;
    end;

    (* The number of unread characters in the buffer. *)
    unread: NativeInt;

    (** The raw buffer. *)
    raw_buffer: record
      (** The beginning of the buffer. *)
      start: PUTF8Char;
      (** The end of the buffer. *)
      _end: PUTF8Char;
      (** The current position of the buffer. *)
      _pointer: PUTF8Char;
      (** The last filled position of the buffer. *)
      last: PUTF8Char;
    end;

    (** The input encoding. *)
    encoding: yaml_encoding_t;

    (** The offset of the current position (in bytes). *)
    offset: NativeInt;

    (** The mark of the current position. *)
    mark: yaml_mark_t;

    (**
     * @name Scanner stuff
     * @{
     *)

    (** Have we started to scan the input stream? *)
    stream_start_produced: Integer;

    (** Have we reached the end of the input stream? *)
    stream_end_produced: Integer;

    (** The number of unclosed '[' and '{' indicators. *)
    flow_level: Integer;

    (** The tokens queue. *)
    tokens: record
      (** The beginning of the tokens queue. *)
      start: Pyaml_token_t;
      (** The end of the tokens queue. *)
      _end: Pyaml_token_t;
      (** The head of the tokens queue. *)
      head: Pyaml_token_t;
      (** The tail of the tokens queue. *)
      tail: Pyaml_token_t;
    end;

    (** The number of tokens fetched from the queue. *)
    tokens_parsed: NativeInt;

    (* Does the tokens queue contain a token ready for dequeueing. *)
    token_available: Integer;

    (** The indentation levels stack. *)
    indents: record
      (** The beginning of the stack. *)
      start: PInteger;
      (** The end of the stack. *)
      _end: PInteger;
      (** The top of the stack. *)
      top: PInteger;
    end;

    (** The current indentation level. *)
    indent: Integer;

    (** May a simple key occur at the current position? *)
    simple_key_allowed: Integer;

    (** The stack of simple keys. *)
    simple_keys: record
      (** The beginning of the stack. *)
      start: Pyaml_simple_key_t;
      (** The end of the stack. *)
      _end: Pyaml_simple_key_t;
      (** The top of the stack. *)
      top: Pyaml_simple_key_t;
    end;

    (**
     * @name Parser stuff
     * @{
     *)

    (** The parser states stack. *)
    states: record
      (** The beginning of the stack. *)
      start: Pyaml_parser_state_t;
      (** The end of the stack. *)
      _end: Pyaml_parser_state_t;
      (** The top of the stack. *)
      top: Pyaml_parser_state_t;
    end;

    (** The current parser state. *)
    state: yaml_parser_state_t;

    (** The stack of marks. *)
    marks: record
      (** The beginning of the stack. *)
      start: Pyaml_mark_t;
      (** The end of the stack. *)
      _end: Pyaml_mark_t;
      (** The top of the stack. *)
      top: Pyaml_mark_t;
    end;

    (** The list of TAG directives. *)
    tag_directives: record
      (** The beginning of the list. *)
      start: Pyaml_tag_directive_t;
      (** The end of the list. *)
      _end: Pyaml_tag_directive_t;
      (** The top of the list. *)
      top: Pyaml_tag_directive_t;
    end;

    (**
     * @name Dumper stuff
     * @{
     *)

    (** The alias data. *)
    aliases: record
      (** The beginning of the list. *)
      start: Pyaml_alias_data_t;
      (** The end of the list. *)
      _end: Pyaml_alias_data_t;
      (** The top of the list. *)
      top: Pyaml_alias_data_t;
    end;

    (** The currently parsed document. *)
    document: Pyaml_document_t;
  end;
  Pyaml_parser_t = ^yaml_parser_t;

(**
 * Initialize a parser.
 *
 * This function creates a new parser object.  An application is responsible
 * for destroying the object using the yaml_parser_delete() function.
 *
 * @param[out]      parser  An empty parser object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_parser_initialize(parser: Pyaml_parser_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_initialize';

(**
 * Destroy a parser.
 *
 * @param[in,out]   parser  A parser object.
 *)
procedure yaml_parser_delete(parser: Pyaml_parser_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_delete';

(**
 * Set a string input.
 *
 * Note that the @a input pointer must be valid while the @a parser object
 * exists.  The application is responsible for destroing @a input after
 * destroying the @a parser.
 *
 * @param[in,out]   parser  A parser object.
 * @param[in]       input   A source data.
 * @param[in]       size    The length of the source data in bytes.
 *)
procedure yaml_parser_set_input_string(parser: Pyaml_parser_t; input: Pointer; size: NativeUInt); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_set_input_string';

(**
 * Set a file input.
 *
 * @a file should be a file object open for reading.  The application is
 * responsible for closing the @a file.
 *
 * @param[in,out]   parser  A parser object.
 * @param[in]       file    An open file.
 *)
procedure yaml_parser_set_input_file(parser: Pyaml_parser_t; _file: PPointer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_set_input_file';

(**
 * Set a generic input handler.
 *
 * @param[in,out]   parser  A parser object.
 * @param[in]       handler A read handler.
 * @param[in]       data    Any application data for passing to the read
 *                          handler.
 *)
procedure yaml_parser_set_input(parser: Pyaml_parser_t; handler: yaml_read_handler_t; data: Pointer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_set_input';

(**
 * Set the source encoding.
 *
 * @param[in,out]   parser      A parser object.
 * @param[in]       encoding    The source encoding.
 *)
procedure yaml_parser_set_encoding(parser: Pyaml_parser_t; encoding: yaml_encoding_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_set_encoding';

(**
 * Scan the input stream and produce the next token.
 *
 * Call the function subsequently to produce a sequence of tokens corresponding
 * to the input stream.  The initial token has the type
 * @c YAML_STREAM_START_TOKEN while the ending token has the type
 * @c YAML_STREAM_END_TOKEN.
 *
 * An application is responsible for freeing any buffers associated with the
 * produced token object using the @c yaml_token_delete function.
 *
 * An application must not alternate the calls of yaml_parser_scan() with the
 * calls of yaml_parser_parse() or yaml_parser_load(). Doing this will break
 * the parser.
 *
 * @param[in,out]   parser      A parser object.
 * @param[out]      token       An empty token object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_parser_scan(parser: Pyaml_parser_t; token: Pyaml_token_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_scan';

(**
 * Parse the input stream and produce the next parsing event.
 *
 * Call the function subsequently to produce a sequence of events corresponding
 * to the input stream.  The initial event has the type
 * @c YAML_STREAM_START_EVENT while the ending event has the type
 * @c YAML_STREAM_END_EVENT.
 *
 * An application is responsible for freeing any buffers associated with the
 * produced event object using the yaml_event_delete() function.
 *
 * An application must not alternate the calls of yaml_parser_parse() with the
 * calls of yaml_parser_scan() or yaml_parser_load(). Doing this will break the
 * parser.
 *
 * @param[in,out]   parser      A parser object.
 * @param[out]      event       An empty event object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_parser_parse(parser: Pyaml_parser_t; event: Pyaml_event_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_parse';

(**
 * Parse the input stream and produce the next YAML document.
 *
 * Call this function subsequently to produce a sequence of documents
 * constituting the input stream.
 *
 * If the produced document has no root node, it means that the document
 * end has been reached.
 *
 * An application is responsible for freeing any data associated with the
 * produced document object using the yaml_document_delete() function.
 *
 * An application must not alternate the calls of yaml_parser_load() with the
 * calls of yaml_parser_scan() or yaml_parser_parse(). Doing this will break
 * the parser.
 *
 * @param[in,out]   parser      A parser object.
 * @param[out]      document    An empty document object.
 *
 * @return @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_parser_load(parser: Pyaml_parser_t; document: Pyaml_document_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_parser_load';

(**
 * @defgroup emitter Emitter Definitions
 * @{
 *)

(**
 * The prototype of a write handler.
 *
 * The write handler is called when the emitter needs to flush the accumulated
 * characters to the output.  The handler should write @a size bytes of the
 * @a buffer to the output.
 *
 * @param[in,out]   data        A pointer to an application data specified by
 *                              yaml_emitter_set_output().
 * @param[in]       buffer      The buffer with bytes to be written.
 * @param[in]       size        The size of the buffer.
 *
 * @returns On success, the handler should return @c 1.  If the handler failed,
 * the returned value should be @c 0.
 *)
type
  yaml_write_handler_t = function(data, buffer: Pointer; size: NativeInt): Integer; cdecl;

(** The emitter states. *)
type
  yaml_emitter_state_t = (
    (** Expect STREAM-START. *)
    YAML_EMIT_STREAM_START_STATE,
    (** Expect the first DOCUMENT-START or STREAM-END. *)
    YAML_EMIT_FIRST_DOCUMENT_START_STATE,
    (** Expect DOCUMENT-START or STREAM-END. *)
    YAML_EMIT_DOCUMENT_START_STATE,
    (** Expect the content of a document. *)
    YAML_EMIT_DOCUMENT_CONTENT_STATE,
    (** Expect DOCUMENT-END. *)
    YAML_EMIT_DOCUMENT_END_STATE,
    (** Expect the first item of a flow sequence. *)
    YAML_EMIT_FLOW_SEQUENCE_FIRST_ITEM_STATE,
    (** Expect an item of a flow sequence. *)
    YAML_EMIT_FLOW_SEQUENCE_ITEM_STATE,
    (** Expect the first key of a flow mapping. *)
    YAML_EMIT_FLOW_MAPPING_FIRST_KEY_STATE,
    (** Expect a key of a flow mapping. *)
    YAML_EMIT_FLOW_MAPPING_KEY_STATE,
    (** Expect a value for a simple key of a flow mapping. *)
    YAML_EMIT_FLOW_MAPPING_SIMPLE_VALUE_STATE,
    (** Expect a value of a flow mapping. *)
    YAML_EMIT_FLOW_MAPPING_VALUE_STATE,
    (** Expect the first item of a block sequence. *)
    YAML_EMIT_BLOCK_SEQUENCE_FIRST_ITEM_STATE,
    (** Expect an item of a block sequence. *)
    YAML_EMIT_BLOCK_SEQUENCE_ITEM_STATE,
    (** Expect the first key of a block mapping. *)
    YAML_EMIT_BLOCK_MAPPING_FIRST_KEY_STATE,
    (** Expect the key of a block mapping. *)
    YAML_EMIT_BLOCK_MAPPING_KEY_STATE,
    (** Expect a value for a simple key of a block mapping. *)
    YAML_EMIT_BLOCK_MAPPING_SIMPLE_VALUE_STATE,
    (** Expect a value of a block mapping. *)
    YAML_EMIT_BLOCK_MAPPING_VALUE_STATE,
    (** Expect nothing. *)
    YAML_EMIT_END_STATE);
  Pyaml_emitter_state_t = ^yaml_emitter_state_t;

(* This is needed for C++ *)
type
  yaml_anchors_t = record
    (** The number of references. *)
    references: Integer;
    (** The anchor id. *)
    anchor: Integer;
    (** If the node has been emitted? *)
    serialized: Integer;
  end;
  Pyaml_anchors_t = ^yaml_anchors_t;

(**
 * The emitter structure.
 *
 * All members are internal.  Manage the structure using the @c yaml_emitter_
 * family of functions.
 *)
type
  yaml_emitter_t = record
    (**
     * @name Error handling
     * @{
     *)

    (** Error type. *)
    error: yaml_error_type_t;
    (** Error description. *)
    problem: PUTF8Char;

    (**
     * @name Writer stuff
     * @{
     *)

    (** Write handler. *)
    write_handler: yaml_write_handler_t;

    (** A pointer for passing to the white handler. *)
    write_handler_data: Pointer;

    (** Standard (string or file) output data. *)
    output: record
      (** String output data. *)
      _string: record
        (** The buffer pointer. *)
        buffer: Pointer;
        (** The buffer size. *)
        size: NativeInt;
        (** The number of written bytes. *)
        size_written: PNativeInt;
      end;

      (** File output data. *)
      _file: Pointer;
    end;

    (** The working buffer. *)
    buffer: record
      (** The beginning of the buffer. *)
      start: Pyaml_char_t;
      (** The end of the buffer. *)
      _end: Pyaml_char_t;
      (** The current position of the buffer. *)
      _pointer: Pyaml_char_t;
      (** The last filled position of the buffer. *)
      last: Pyaml_char_t;
    end;

    (** The raw buffer. *)
    raw_buffer: record
      (** The beginning of the buffer. *)
      start: Pointer;
      (** The end of the buffer. *)
      _end: Pointer;
      (** The current position of the buffer. *)
      _pointer: Pointer;
      (** The last filled position of the buffer. *)
      last: Pointer;
    end;

    (** The stream encoding. *)
    encoding: yaml_encoding_t;

    (**
     * @name Emitter stuff
     * @{
     *)

    (** If the output is in the canonical style? *)
    canonical: Integer;
    (** The number of indentation spaces. *)
    best_indent: Integer;
    (** The preferred width of the output lines. *)
    best_width: Integer;
    (** Allow unescaped non-ASCII characters? *)
    unicode: Integer;
    (** The preferred line break. *)
    line_break: yaml_break_t;

    (** The stack of states. *)
    states: record
      (** The beginning of the stack. *)
      start: Pyaml_emitter_state_t;
      (** The end of the stack. *)
      _end: Pyaml_emitter_state_t;
      (** The top of the stack. *)
      top: Pyaml_emitter_state_t;
    end;

    (** The current emitter state. *)
    state: yaml_emitter_state_t;

    (** The event queue. *)
    events: record
      (** The beginning of the event queue. *)
      start: Pyaml_event_t;
      (** The end of the event queue. *)
      _end: Pyaml_event_t;
      (** The head of the event queue. *)
      head: Pyaml_event_t;
      (** The tail of the event queue. *)
      tail: Pyaml_event_t;
    end;

    (** The stack of indentation levels. *)
    indents: record
      (** The beginning of the stack. *)
      start: PInteger;
      (** The end of the stack. *)
      _end: PInteger;
      (** The top of the stack. *)
      top: PInteger;
    end;

    (** The list of tag directives. *)
    tag_directives: record
      (** The beginning of the list. *)
      start: Pyaml_tag_directive_t;
      (** The end of the list. *)
      _end: Pyaml_tag_directive_t;
      (** The top of the list. *)
      top: Pyaml_tag_directive_t;
    end;

    (** The current indentation level. *)
    indent: Integer;

    (** The current flow level. *)
    flow_level: Integer;

    (** Is it the document root context? *)
    root_context: Integer;
    (** Is it a sequence context? *)
    sequence_context: Integer;
    (** Is it a mapping context? *)
    mapping_context: Integer;
    (** Is it a simple mapping key context? *)
    simple_key_context: Integer;

    (** The current line. *)
    line: Integer;
    (** The current column. *)
    column: Integer;
    (** If the last character was a whitespace? *)
    whitespace: Integer;
    (** If the last character was an indentation character (' ', '-', '?', ':')? *)
    indention: Integer;
    (** If an explicit document end is required? *)
    open_ended: Integer;

    (** Anchor analysis. *)
    anchor_data: record
      (** The anchor value. *)
      anchor: Pyaml_char_t;
      (** The anchor length. *)
      anchor_length: NativeInt;
      (** Is it an alias? *)
      alias: Integer;
    end;

    (** Tag analysis. *)
    tag_data: record
      (** The tag handle. *)
      handle: Pyaml_char_t;
      (** The tag handle length. *)
      handle_length: NativeInt;
      (** The tag suffix. *)
      suffix: Pyaml_char_t;
      (** The tag suffix length. *)
      suffix_length: NativeInt;
    end;

    (** Scalar analysis. *)
    scalar_data: record
      (** The scalar value. *)
      value: Pyaml_char_t;
      (** The scalar length. *)
      length: NativeInt;
      (** Does the scalar contain line breaks? *)
      multiline: Integer;
      (** Can the scalar be expessed in the flow plain style? *)
      flow_plain_allowed: Integer;
      (** Can the scalar be expressed in the block plain style? *)
      block_plain_allowed: Integer;
      (** Can the scalar be expressed in the single quoted style? *)
      single_quoted_allowed: Integer;
      (** Can the scalar be expressed in the literal or folded styles? *)
      block_allowed: Integer;
      (** The output style. *)
      style: yaml_scalar_style_t;
    end;

    (**
     * @name Dumper stuff
     * @{
     *)

    (** If the stream was already opened? *)
    opened: Integer;
    (** If the stream was already closed? *)
    closed: Integer;

    (** The information associated with the document nodes. *)
    anchors: Pyaml_anchors_t;

    (** The last assigned anchor id. *)
    last_anchor_id: Integer;

    (** The currently emitted document. *)
    document: Pyaml_document_t;
  end;
  Pyaml_emitter_t = ^yaml_emitter_t;

(**
 * Initialize an emitter.
 *
 * This function creates a new emitter object.  An application is responsible
 * for destroying the object using the yaml_emitter_delete() function.
 *
 * @param[out]      emitter     An empty parser object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_emitter_initialize(emitter: Pyaml_emitter_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_initialize';

(**
 * Destroy an emitter.
 *
 * @param[in,out]   emitter     An emitter object.
 *)
procedure yaml_emitter_delete(emitter: Pyaml_emitter_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_delete';

(**
 * Set a string output.
 *
 * The emitter will write the output characters to the @a output buffer of the
 * size @a size.  The emitter will set @a size_written to the number of written
 * bytes.  If the buffer is smaller than required, the emitter produces the
 * YAML_WRITE_ERROR error.
 *
 * @param[in,out]   emitter         An emitter object.
 * @param[in]       output          An output buffer.
 * @param[in]       size            The buffer size.
 * @param[in]       size_written    The pointer to save the number of written
 *                                  bytes.
 *)
procedure yaml_emitter_set_output_string(emitter: Pyaml_emitter_t; output: Pointer; size: NativeUInt; size_written: PNativeUInt); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_output_string';

(**
 * Set a file output.
 *
 * @a file should be a file object open for writing.  The application is
 * responsible for closing the @a file.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       file        An open file.
 *)
procedure yaml_emitter_set_output_file(emitter: Pyaml_emitter_t; _file: PPointer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_output_file';

(**
 * Set a generic output handler.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       handler     A write handler.
 * @param[in]       data        Any application data for passing to the write
 *                              handler.
 *)
procedure yaml_emitter_set_output(emitter: Pyaml_emitter_t; handler: yaml_write_handler_t; data: Pointer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_output';

(**
 * Set the output encoding.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       encoding    The output encoding.
 *)
procedure yaml_emitter_set_encoding(emitter: Pyaml_emitter_t; encoding: yaml_encoding_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_encoding';

(**
 * Set if the output should be in the "canonical" format as in the YAML
 * specification.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       canonical   If the output is canonical.
 *)
procedure yaml_emitter_set_canonical(emitter: Pyaml_emitter_t; canonical: Integer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_canonical';

(**
 * Set the intendation increment.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       indent      The indentation increment (1 < . < 10).
 *)
procedure yaml_emitter_set_indent(emitter: Pyaml_emitter_t; indent: Integer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_indent';

(**
 * Set the preferred line width. @c -1 means unlimited.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       width       The preferred line width.
 *)
procedure yaml_emitter_set_width(emitter: Pyaml_emitter_t; width: Integer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_width';

(**
 * Set if unescaped non-ASCII characters are allowed.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       unicode     If unescaped Unicode characters are allowed.
 *)
procedure yaml_emitter_set_unicode(emitter: Pyaml_emitter_t; unicode: Integer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_unicode';

(**
 * Set the preferred line break.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in]       line_break  The preferred line break.
 *)
procedure yaml_emitter_set_break(emitter: Pyaml_emitter_t; line_break: yaml_break_t); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_set_break';

(**
 * Emit an event.
 *
 * The event object may be generated using the yaml_parser_parse() function.
 * The emitter takes the responsibility for the event object and destroys its
 * content after it is emitted. The event object is destroyed even if the
 * function fails.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in,out]   event       An event object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_emitter_emit(emitter: Pyaml_emitter_t; event: Pyaml_event_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_emit';

(**
 * Start a YAML stream.
 *
 * This function should be used before yaml_emitter_dump() is called.
 *
 * @param[in,out]   emitter     An emitter object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_emitter_open(emitter: Pyaml_emitter_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_open';

(**
 * Finish a YAML stream.
 *
 * This function should be used after yaml_emitter_dump() is called.
 *
 * @param[in,out]   emitter     An emitter object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_emitter_close(emitter: Pyaml_emitter_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_close';

(**
 * Emit a YAML document.
 *
 * The documen object may be generated using the yaml_parser_load() function
 * or the yaml_document_initialize() function.  The emitter takes the
 * responsibility for the document object and destroys its content after
 * it is emitted. The document object is destroyed even if the function fails.
 *
 * @param[in,out]   emitter     An emitter object.
 * @param[in,out]   document    A document object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_emitter_dump(emitter: Pyaml_emitter_t; document: Pyaml_document_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_dump';

(**
 * Flush the accumulated characters to the output.
 *
 * @param[in,out]   emitter     An emitter object.
 *
 * @returns @c 1 if the function succeeded, @c 0 on error.
 *)
function yaml_emitter_flush(emitter: Pyaml_emitter_t): Integer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_emitter_flush';

{ Private }

function yaml_malloc(size: NativeInt): Pointer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_malloc';

function yaml_realloc(ptr: Pointer; size: NativeInt): Pointer; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_realloc';

procedure yaml_free(ptr: Pointer); cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_free';

function yaml_strdup(str: Pyaml_char_t): Pyaml_char_t; cdecl;
  external {$IFDEF USE_LIB}LIB_YAML{$ENDIF} name _PU + 'yaml_strdup';

implementation

{$IF Defined(WIN32)}
const
  MSVCRT = 'msvcrt.dll';

procedure _sprintf; cdecl; external MSVCRT name 'sprintf';
procedure _free; cdecl; external MSVCRT name 'free';
procedure _malloc; cdecl; external MSVCRT name 'malloc';
procedure _realloc; cdecl; external MSVCRT name 'realloc';
procedure _memset; cdecl; external MSVCRT name 'memset';
procedure _memcpy; cdecl; external MSVCRT name 'memcpy';
procedure _memmove; cdecl; external MSVCRT name 'memmove';
procedure _strdup; cdecl; external MSVCRT name '_strdup';
procedure _strcmp; cdecl; external MSVCRT name 'strcmp';
procedure _strncmp; cdecl; external MSVCRT name 'strncmp';
procedure _ferror; cdecl; external MSVCRT name 'ferror';
procedure _fread; cdecl; external MSVCRT name 'fread';
procedure _fwrite; cdecl; external MSVCRT name 'fwrite';
{$ELSEIF Defined(WIN64)}
const
  MSVCRT = 'msvcrt.dll';

procedure sprintf; cdecl; external MSVCRT name 'sprintf';
procedure free; cdecl; external MSVCRT name 'free';
procedure malloc; cdecl; external MSVCRT name 'malloc';
procedure realloc; cdecl; external MSVCRT name 'realloc';
procedure memset; cdecl; external MSVCRT name 'memset';
procedure memcpy; cdecl; external MSVCRT name 'memcpy';
procedure memmove; cdecl; external MSVCRT name 'memmove';
procedure strdup; cdecl; external MSVCRT name '_strdup';
procedure strcmp; cdecl; external MSVCRT name 'strcmp';
procedure strncmp; cdecl; external MSVCRT name 'strncmp';
procedure ferror; cdecl; external MSVCRT name 'ferror';
procedure fread; cdecl; external MSVCRT name 'fread';
procedure fwrite; cdecl; external MSVCRT name 'fwrite';
{$ENDIF}

end.
