import agenta as ag
from pydantic import BaseModel, Field
from agenta.sdk.assets import supported_llm_models
from typing import Annotated
from opentelemetry.instrumentation.openai import OpenAIInstrumentor
from openai import OpenAI

ag.init()


openai = OpenAI()


OpenAIInstrumentor().instrument()


class MyConfig(BaseModel):
    temperature: float = Field(default=0.2, le=1, ge=0)
    model: Annotated[str, ag.MultipleChoice(choices=supported_llm_models)] = Field(
        default="gpt-3.5-turbo"
    )
    max_tokens: int = Field(default=-1, ge=-1, le=4000)
    prompt_system: str = Field(default="system prompt")
    multiselect: Annotated[str, ag.MultipleChoice(choices=["a", "b", "c"])] = Field(
        default="a"
    )


base64 = "data:image/jpeg;base64,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"


@ag.entrypoint
async def image(none: str):
    response = openai.chat.completions.create(
        model="gpt-4o-mini",
        messages=[
            {
                "role": "user",
                "content": [
                    {"type": "text", "text": "What’s in this image?"},
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": "https://upload.wikimedia.org/wikipedia/commons/thumb/d/dd/Gfp-wisconsin-madison-the-nature-boardwalk.jpg/2560px-Gfp-wisconsin-madison-the-nature-boardwalk.jpg",
                        },
                    },
                ],
            }
        ],
    )

    response = openai.chat.completions.create(
        model="gpt-4o-mini",
        messages=[
            {
                "role": "user",
                "content": [
                    {"type": "text", "text": "What’s in this image?"},
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": base64,
                        },
                    },
                ],
            }
        ],
    )

    return response.choices[0].message.content
