# Routes & Resources Review
## Learning Goals
- Revisit RESTful routing in Rails
- Learn about using the Rails `resources` route helpers
- Know how to override the root path

## Override Root You Say?

Whenever we have been booting up rails and going to `localhost:3000` we've been seeing that cute, but not terribly helpful "Welcome to Rails" splash page. What if we want to change where that root route goes?

It's pretty easy actually, and it looks a lot like what we've done for other routes!

```ruby
root to: "controller#action"
```

So, for example, in our Ada Books project, we might do something like this:

```ruby
root to: "books#index"
```

This also means that we can choose to redirect to the root, rather than having to pick a specific controller action. Sometimes, it just makes sense to send the user home!

A good example is found in the `books#create` controller action:

```ruby
def create
  @book = Book.new(author: params[:book][:author], title: params[:book][:title], params[:description]) #instantiate a new book
  if @book.save # save returns true if the database insert succeeds
    redirect_to root_path # go to the index so we can see the book in the list
    return
  else # save failed :(
    render :new # show the new book form view again
    return
  end
end
```

## Quick Review
Rails routes are matched in the order they are specified. This can present a problem when a wildcard is used.

```ruby
get "/books/:id", to: "books#show"
get "/books/new", to: "books#new"
```

When Rails receives a request to `books/new` in this example, it will match the first route. Both of the routes above match `/books/new`! We fix it by doing this instead:

```ruby
get "/books/new", to: "books#new"
get "/books/:id", to: "books#show"
```

That general rule is that routes go from most specific at the top, to least specific at the bottom for any given resource.

## REST
Representational state transfer (REST) is an architectural style. This is a recommendation for the structure and style in which a resource (Books, or Authors or Users etc.) is managed by a web server.

By name, they are called index, create, new, show, update, edit and destroy.

## Resources
Rails has several _route helpers_ available. The most commons is the `resources` method which generates the default RESTful routes for the names resource:

```ruby
Rails.application.routes.draw do
  resources :books
end
```

This is equivalent to:

```ruby
Rails.application.routes.draw do
  get    "/books"          , to: "books#index",   as: :book
  post   "/books"          , to: "books#create"
  get    "/books/new"      , to: "books#new",     as: :new_book
  get    "/books/:id"      , to: "books#show",    as: :book
  patch  "/books/:id"      , to: "books#update"
  put    "/books/:id"      , to: "books#update"
  delete "/books/:id"      , to: "books#destroy"
  get    "/books/:id/edit" , to: "books#edit",    as: :edit_book
end
```

### Don't Get Lazy With `Resources`
Only ever use `resources` if you intend on implementing the full suite of RESTful routes for the resource. It's much more typical to only use a few of the routes generated by `resources`. 

For example, if you were creating a blog site for yourself, you probably wouldn't want people to be able to change the blogs besides you. In this case, you would probably only need to have `index` and `show`! 

Leaving in the other routes is not only sloppy, but could cause you and your users a bunch of headaches.

To account for this, the `resources` route helper accepts options to limit the routes generated:

```ruby
Rails.application.routes.draw do
  # only generate the specified routes
  resources :books, only: [:index, :show]

  # generates every RESTful route _except_ for POST (create) and DELETE (destroy)
  resources :authors, except: [:create, :destroy]
end
```

Finally, remember that sometimes you will want a combination of some routes from `resources` and some that you write custom for your site. That's fine! REST is a framework to help you remember what's needed, and for you to communicate with other programmers or your future self, but by definition it can't catch all cases.

## Exercise
Let's try out using `resources` in our TaskList project to see what will happen.

1. Run `rails routes` in your terminal. Make notes of what you see - either copy/paste the result or jot it down. Did your custom routes follow the standards listed in the table above?
1. Comment out all of your routes in the `routes.rb` file.
1. Add one line using `resources` and the name of your resource. (Most folks probably used "tasks", but we know not everyone did).
1. Run `rails routes` in your terminal again. Make notes of what is different from this version to the version that was created by your custom routes.
1. See if you can run your `rails server` and navigate to some of the pages you have set up. This will probably not work depending on how different your custom routes are from the standard set.

## Final Notes
- From now on, you are encouraged to use `resources` rather than defining each route "manually"
- When you look up Rails documentation on routing, most reference material uses `resources`
- Use `except` and `only` regularly to ensure you aren't creating extraneous routes that you will not use in your application
- Now it is even more important to be running `rails routes` regularly. Since the routes created are behind this new layer of abstraction, `rails routes` should help you clarify how your resources are set up.

## Resources
- [Official Rails Guide on Resources](http://guides.rubyonrails.org/routing.html#singular-resources)
