# Predefined Annotation Types

## Annotation Types Used by the Java Language

The predefined annotation types defined in ```java.lang``` are ```@Deprecated```, ```@Override```, and ```@SuppressWarnings```.

These annotations are explained in detail here : 

1.  ```java
    @Deprecated
    ```

    This annotation indicates that the marked element is **deprecated** and should no longer be used. The compiler generates a warning whenever a program uses a method, class, or field with the ```@Deprecated``` annotation. When an element is deprecated, it should also be documented using the Javadoc ```@deprecated``` tag

    :warning: Also please note that the Javadoc tag starts with a lowercase d and the annotation starts with an uppercase D.
    
1.  ```java
    @Override
    ```

    This annotation informs the compiler that the element is meant to override an element declared in a superclass. Usage of this annotation is optional, however using it prevents errors. 

    :warning: If a method marked with @Override fails to correctly override a method in one of its superclasses, the compiler generates an error.

1.  ```java
    @SuppressWarnings(value="someValue")
    ```
    This annotation tells the compiler to suppress specific warnings that it would otherwise generate. 

    :exclamation:   There are three categories of warnings **deprecation**, **unchecked** and **removal**.

    :warning:   To suppress multiple categories of warnings, use the following syntax
    
    ```java
                @SuppressWarnings({"unchecked", "deprecation"})
    ```

1.  ```java
    @SafeVarargs
    ```
    This annotation, when applied to a method or constructor, asserts that the code does not perform potentially unsafe operations on its varargs parameter.

1.  ```java
    @FunctionalInterface
    ```
    This annotation indicates that the type declaration is intended to be a functional interface, as defined by the Java Language Specification.

## Meta-annotations

Annotations that apply to other annotations are called **meta-annotations**. There are several meta-annotation types defined in ```java.lang.annotation``` package.

### Retention and Target

Creating an annotation requires two pieces of information: (1) a **retention policy** and (2) a **target**. 

1.  **Retention**

    A retention policy specifies how long, in terms of the program lifecycle, the annotation should be retained for.

    As of ```Java 9```, there are three standard retention policies which are shown below -  

    | Policy  | Description                                                                                                                                                                         |
    |---------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
    | Source  | Annotations are discarded by the compiler                                                                                                                                           |
    | Class   | Annotations are recorded in the class file generated by the compiler but are not required to be retained by the Java Virtual Machine (JVM) that processes the class file at runtime |
    | Runtime | Annotations are recorded in the class file by the compiler and retained at runtime by the JVM                                                                                       |
    
    :warning: Note that an annotation has **exactly one** associated retention policy.

1.  **Target**
    
    The target of an annotation specifies which Java constructs an annotation can be applied to.

    As of ```Java 9```, there are eleven standard annotation targets which are mentioned below : 

    | TARGET          |                                      DESCRIPTION                                      |
    |-----------------|:-------------------------------------------------------------------------------------:|
    | Annotation Type |                              Annotates another annotation                             |
    | Constructor     |                                Annotates a constructor                                |
    | Field           |     Annotates a field, such as an instance variable of a class or an enum constant    |
    | Local variable  | Annotates a local variable                                                            |
    | Method          | Annotates a method of a class                                                         |
    | Module          | Annotates a module (new in Java 9)                                                    |
    | Package         | Annotates a package                                                                   |
    | Parameter       | Annotates a parameter to a method or constructor                                      |
    | Type            | Annotates a type, such as a class, interfaces, annotation types, or enum declarations |

    :warning: It is important to note that **one or more** targets may be associated with an annotation.

### Other Annotations

1.  ```java
    @Documented
    ```
    This annotation indicates that whenever the specified annotation is used those elements should be documented using the ```Javadoc``` tool.

1.  ```java
    @Inherited
    ```
    This annotation indicates that the annotation type can be inherited from the super class. **(This is not true by default.)**
    When the user queries the annotation type and the class has no annotation for this type, the class' superclass is queried for the annotation type.

1.  ```java
    @Repeatable
    ```
    This annotation indicates that the marked annotation can be applied **more than once** to the same declaration or type use.

## Miscellaneous 

This [StackOverflow post](https://stackoverflow.com/questions/35892063/which-nonnull-java-annotation-to-use) explains why the ```@NonNull``` annotation is not present as a standard annotation in Java SE.

Also, the [Checker Framework](https://checkerframework.org/) provides a good set of annotations that can be used in Java Projects.

## References 

1. [DZone Article](https://dzone.com/articles/5-annotations-every-java-developer-should-know)
